/* 
 * $Id: ctkscrolledwindow.c,v 1.21 2000/07/26 22:31:16 terpstra Exp $
 *
 * CTK - Console Toolkit
 *
 * Copyright (C) 1998-2000 Stormix Technologies Inc.
 *
 * License: LGPL
 *
 * Authors: Kevin Lindsay, Wesley Terpstra
 *  
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2 of the License, or (at your option) any later version.
 *    
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *    
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include <stdio.h>

#include "ctk.h"
#include "ctkcolor.h"

gpointer ctk_scrolled_window_destroy(CtkObject *object);

gboolean ctk_scrolled_window_press   (CtkScrolledWindow* sw, CdkEventButton* event, gpointer data);
gboolean ctk_scrolled_window_release (CtkScrolledWindow* sw, CdkEventButton* event, gpointer data);
gboolean ctk_scrolled_window_drag    (CtkScrolledWindow* sw, CdkEventButton* event, gpointer data);
gboolean ctk_scrolled_window_keypress(CtkScrolledWindow* sw, CdkEventKey*    event, gpointer data);

void ctk_scrolled_window_init(CtkScrolledWindow* scrolledwindow)
{		
	ctk_bin_init(&scrolledwindow->bin);	
	CTK_OBJECT(scrolledwindow)->type = CtkTypeScrolledWindow;

	scrolledwindow->hadjustment = NULL;
	scrolledwindow->vadjustment = NULL;

	scrolledwindow->hscrollbar_policy = CTK_POLICY_AUTOMATIC;
	scrolledwindow->vscrollbar_policy = CTK_POLICY_AUTOMATIC;
	scrolledwindow->hscrollbar_shown = 0;
	scrolledwindow->vscrollbar_shown = 0;
	
	scrolledwindow->locked_vertical = FALSE;
	scrolledwindow->locked_horizontal = FALSE;

	((CtkWidget *)scrolledwindow)->sensitive   = TRUE;
	((CtkWidget *)scrolledwindow)->width       = 4;
	((CtkWidget *)scrolledwindow)->height      = 4;
	((CtkWidget *)scrolledwindow)->orig_width  = 4;
	((CtkWidget *)scrolledwindow)->orig_height = 4;
	((CtkWidget *)scrolledwindow)->main_col = ctk_calculate_palette(CTK_COLOR_WHITE,CTK_COLOR_BLUE);
	((CtkWidget *)scrolledwindow)->inverse_col = ctk_calculate_palette(CTK_COLOR_WHITE,CTK_COLOR_RED);
	((CtkObject *)scrolledwindow)->destroy_func = ctk_scrolled_window_destroy;
	
	((CtkWidget*)scrolledwindow)->set_min_size  = &ctk_scrolled_window_min_size;
	((CtkWidget*)scrolledwindow)->set_real_size = &ctk_scrolled_window_real_size;

	ctk_signal_connect(CTK_OBJECT(scrolledwindow), "key_press_event",
			   CTK_SIGNAL_FUNC(&ctk_scrolled_window_keypress), NULL);	
	ctk_signal_connect(CTK_OBJECT(scrolledwindow), "ctk_drag",
			   CTK_SIGNAL_FUNC(&ctk_scrolled_window_drag), NULL);
	ctk_signal_connect(CTK_OBJECT(scrolledwindow), "button_press_event",
			   CTK_SIGNAL_FUNC(&ctk_scrolled_window_press), NULL);	
	ctk_signal_connect(CTK_OBJECT(scrolledwindow), "button_release_event",
			   CTK_SIGNAL_FUNC(&ctk_scrolled_window_release), NULL);
}

CtkWidget* ctk_scrolled_window_new(CtkAdjustment *hadjustment,
			   CtkAdjustment *vadjustment)
{
	CtkScrolledWindow *scrolledwindow;
	
	scrolledwindow = g_malloc(sizeof(CtkScrolledWindow));
	
	ctk_scrolled_window_init(scrolledwindow);
	
	scrolledwindow->hadjustment = hadjustment;
	scrolledwindow->vadjustment = vadjustment;

	return ((CtkWidget *)scrolledwindow);
}

/* Scrolled Window scrolling function */
void ctk_scrolled_window_scroll(CtkWidget* widget,
			      CtkTypeScroll scroll_type)
{
	CtkScrolledWindow* scrolledwindow;
	scrolledwindow = ((CtkScrolledWindow *)widget);
	
	if (!scrolledwindow->hadjustment || !scrolledwindow->vadjustment)
	    return;

	if (scroll_type == CTK_SCROLL_LEFT) {
		scrolledwindow->hadjustment->value--;
		ctk_draw_mark_changed(widget);
	}
	else if (scroll_type == CTK_SCROLL_RIGHT) {
		scrolledwindow->hadjustment->value++;
		ctk_draw_mark_changed(widget);
	}
	else if (scroll_type == CTK_SCROLL_UP) {
		scrolledwindow->vadjustment->value--;
		ctk_draw_mark_changed(widget);
	}
	else if (scroll_type == CTK_SCROLL_DOWN) {
		scrolledwindow->vadjustment->value++;
		ctk_draw_mark_changed(widget);
	}
	else if (scroll_type == CTK_PAGE_RIGHT) {
		scrolledwindow->hadjustment->value +=
			scrolledwindow->hadjustment->page_increment;
		ctk_draw_mark_changed(widget);
	}
	else if (scroll_type == CTK_PAGE_LEFT) {
		scrolledwindow->hadjustment->value -=
		    scrolledwindow->hadjustment->page_increment;
		ctk_draw_mark_changed(widget);
	}
	else if (scroll_type == CTK_PAGE_DOWN) {
		scrolledwindow->vadjustment->value +=
		    scrolledwindow->vadjustment->page_increment;
		ctk_draw_mark_changed(widget);
	}
	else if (scroll_type == CTK_PAGE_UP) {
		scrolledwindow->vadjustment->value -=
		    scrolledwindow->vadjustment->page_increment;
		ctk_draw_mark_changed(widget);
	}
}

gboolean ctk_scrolled_window_press(CtkScrolledWindow* sw, CdkEventButton* event, gpointer data)
{
      CtkWidget* widget = CTK_WIDGET(sw);
      gint       x      = event->x;
      gint       y      = event->y;

      sw->locked_vertical   = FALSE;
      sw->locked_horizontal = FALSE;

      /* Check Horiz scroll */
      if (x > widget->col && 
	  x < widget->col+(widget->width-1-sw->vscrollbar_shown) &&
	  y == widget->row+(widget->height-1) &&
	  sw->hadjustment->slider_len < widget->width-3) 
      {
	    /* Right Page */
	    if (x > sw->hadjustment->slider_col +
		sw->hadjustment->slider_len)
		  ctk_scrolled_window_scroll(widget, CTK_PAGE_RIGHT);
	    /* Left Page */
	    else if (x < sw->hadjustment->slider_col)
		  ctk_scrolled_window_scroll(widget, CTK_PAGE_LEFT);
		  
            sw->locked_horizontal = TRUE;
      }
      /* Check Vertical Scroll */
      else if (y > widget->row && 
	       y < widget->row+(widget->height-1-sw->hscrollbar_shown) &&
	       x == widget->col+(widget->width-1) &&
	       sw->vadjustment->slider_len < widget->height-3) 
      {
	    /* Down Page */
	    if (y > sw->vadjustment->slider_row+
		sw->vadjustment->slider_len)
		  ctk_scrolled_window_scroll(widget, CTK_PAGE_DOWN);
	    /* Up Page */
	    else if (y < sw->vadjustment->slider_row)
		  ctk_scrolled_window_scroll(widget, CTK_PAGE_UP);
	    
            sw->locked_vertical = TRUE;
      }	

      return TRUE;
}

gboolean ctk_scrolled_window_release(CtkScrolledWindow* sw, CdkEventButton* event, gpointer data)
{
      CtkWidget* widget = CTK_WIDGET(sw);
      gint       x      = event->x;
      gint       y      = event->y;

      sw->locked_vertical   = FALSE;
      sw->locked_horizontal = FALSE;

      /* Move left */
      if (x == widget->col && 
	  y == widget->row+widget->height-1)
      {
	    ctk_scrolled_window_scroll(widget, CTK_SCROLL_LEFT);
      }
      /* Move right */
      else if (x == widget->col + widget->width-1-sw->vscrollbar_shown &&
	       y == widget->row+widget->height - 1)
      {
	    ctk_scrolled_window_scroll(widget,CTK_SCROLL_RIGHT);
      }
      /* Move Up */
      else if (x == widget->col+widget->width-1 && 
	       y == widget->row) 
      {
	    ctk_scrolled_window_scroll(widget, CTK_SCROLL_UP);
      }
      /* Move Down */
      else if (x == widget->col+widget->width-1 &&
	       y == widget->row+widget->height-1-sw->hscrollbar_shown) 
      {
	    ctk_scrolled_window_scroll(widget, CTK_SCROLL_DOWN);
      }

      return TRUE;
}

gboolean ctk_scrolled_window_drag(CtkScrolledWindow* sw, CdkEventButton* event, gpointer data)
{
      gint x = event->dx;
      gint y = event->dy;

      while (x < 0)
      {
	    if (!sw->locked_vertical)
	          ctk_scrolled_window_scroll(CTK_WIDGET(sw), CTK_SCROLL_LEFT);
	    x++;
      }
      while (y < 0)
      {
            if (!sw->locked_horizontal)
	          ctk_scrolled_window_scroll(CTK_WIDGET(sw), CTK_SCROLL_UP);
	    y++;
      }      
      while (x > 0)
      {
            if (!sw->locked_vertical)
	          ctk_scrolled_window_scroll(CTK_WIDGET(sw), CTK_SCROLL_RIGHT);
	    x--;
      }        
      while (y > 0)
      {
            if (!sw->locked_horizontal)
	         ctk_scrolled_window_scroll(CTK_WIDGET(sw), CTK_SCROLL_DOWN);
	    y--;
      }    

      return TRUE;
}

gboolean ctk_scrolled_window_keypress(CtkScrolledWindow* sw, CdkEventKey* event, gpointer data)
{
	switch (event->keyval)
	{
	case AK_ARROW_UP:
		ctk_scrolled_window_scroll(CTK_WIDGET(sw), CTK_SCROLL_UP);
		return TRUE;
	case AK_ARROW_DOWN:
		ctk_scrolled_window_scroll(CTK_WIDGET(sw), CTK_SCROLL_DOWN);
		return TRUE;
	case AK_ARROW_LEFT:
		ctk_scrolled_window_scroll(CTK_WIDGET(sw), CTK_SCROLL_LEFT);
		return TRUE;
	case AK_ARROW_RIGHT:
		ctk_scrolled_window_scroll(CTK_WIDGET(sw), CTK_SCROLL_RIGHT);
		return TRUE;
	}
	
	return FALSE;
}

/* Set Policy */
void ctk_scrolled_window_set_policy(CtkScrolledWindow* scrolled_window, 
				   CtkPolicyType hscrollbar_policy,
				   CtkPolicyType vscrollbar_policy)
{
	if (!scrolled_window)
	    return;
	
	scrolled_window->hscrollbar_policy = hscrollbar_policy;
	scrolled_window->vscrollbar_policy = vscrollbar_policy;
	
	ctk_size_mark_changed(CTK_WIDGET(scrolled_window));
}

/* Set Scrolled Window hadjustment */
void ctk_scrolled_window_set_hadjustment(CtkScrolledWindow* scrolled_window,
					CtkAdjustment* hadjustment)
{
	if (!scrolled_window)
	    return;
	
	if (scrolled_window->hadjustment)
	    g_free(scrolled_window->hadjustment);
	
	scrolled_window->hadjustment = hadjustment;

	ctk_draw_mark_changed(CTK_WIDGET(scrolled_window));
}

/* Set Scrolled Window vadjustment */
void ctk_scrolled_window_set_vadjustment(CtkScrolledWindow* scrolled_window,
					CtkAdjustment* vadjustment)
{
	if (!scrolled_window)
	    return;
	
	if (scrolled_window->vadjustment)
	    g_free(scrolled_window->vadjustment);
	
	scrolled_window->vadjustment = vadjustment;
	
	ctk_draw_mark_changed(CTK_WIDGET(scrolled_window));
}

/* Add scrollbar with view port */
void ctk_scrolled_window_add_with_viewport(CtkScrolledWindow *scrolled_window,
					  CtkWidget *child)
{
	CtkWidget *viewport;
	CtkAdjustment *hadjustment = NULL;
	CtkAdjustment *vadjustment = NULL;

	if (!scrolled_window)
	    return;

	viewport = ctk_viewport_new(CTK_ADJUSTMENT(hadjustment),
				    CTK_ADJUSTMENT(vadjustment));
	ctk_widget_show(viewport);

	ctk_container_add(CTK_CONTAINER(scrolled_window),viewport);
	ctk_container_add(CTK_CONTAINER(viewport),child);
	
	ctk_size_mark_changed(CTK_WIDGET(scrolled_window));
}

/* Destroy Data */
gpointer ctk_scrolled_window_destroy(CtkObject* object)
{
	CtkScrolledWindow* scrolledwindow;
	
	scrolledwindow = CTK_SCROLLED_WINDOW(object);
	
	if (scrolledwindow->hadjustment)
	    g_free(scrolledwindow->hadjustment);
	if (scrolledwindow->vadjustment)
	    g_free(scrolledwindow->vadjustment);

	return NULL;
}

void ctk_scrolled_window_min_size(CtkWidget* widget)
{
	/* Need space for the scrollbars */
	widget->min_height = 3;
	widget->min_width  = 3;
}

void ctk_scrolled_window_real_size(CtkWidget* widget)
{
	/* We leave ourselves in place and move the slave subject to the adjustment */
	CtkWidget*         child;
	CtkScrolledWindow* self = CTK_SCROLLED_WINDOW(widget);
	
	if (!widget || !widget->node || !widget->node->children ||
		!widget->node->children->data)
		return;
	
	child = CTK_WIDGET(widget->node->children->data);

	child->width  = child->min_width;
	child->height = child->min_height;
	
	/* We need to decide when to place scrollbars. Note that they are not
	 * independant; adding a scrollbar can force the other to be necessary.
	 * Hence the many cases below. :-(
	 */
	// FIXME: there has to be a more elegant way than this...
	if (self->hscrollbar_policy != CTK_POLICY_NEVER &&
		(child->width > widget->width || 
		 self->hscrollbar_policy == CTK_POLICY_ALWAYS))
	{
		self->hscrollbar_shown = 1;
		if (self->vscrollbar_policy != CTK_POLICY_NEVER &&
			(child->height > widget->height - 1 || 
			 self->vscrollbar_policy == CTK_POLICY_ALWAYS))
		{
			self->vscrollbar_shown = 1;
		}
		else
		{
			self->vscrollbar_shown = 0;
		}
	}
	else
	{
		if (self->vscrollbar_policy != CTK_POLICY_NEVER &&
			(child->height > widget->height ||
			 self->vscrollbar_policy == CTK_POLICY_ALWAYS))
		{
			self->vscrollbar_shown = 1;
			if (self->hscrollbar_policy != CTK_POLICY_NEVER &&
				(child->width > widget->width - 1 ||
				 self->hscrollbar_policy == CTK_POLICY_ALWAYS))
			{
				self->hscrollbar_shown = 1;
			}
			else
			{
				self->hscrollbar_shown = 0;
			}
		}
		else
		{
			self->vscrollbar_shown = 0;
			self->hscrollbar_shown = 0;
		}
	}
	
	/* Now, if we have more space than we need, grow */
	if (child->width < widget->width - self->vscrollbar_shown)
		child->width = widget->width - self->vscrollbar_shown;
	
	if (child->height < widget->height - self->hscrollbar_shown)
		child->height = widget->height - self->hscrollbar_shown;
	
	/* Now, lets sanitize the adjustments */
	if (ctk_label_wrap_hack_report_vertible)
	{
		/* Hack hack hack: We need to avoid this until real sizes avialable */
	
		if (child->width - self->hadjustment->value <
			widget->width - self->vscrollbar_shown)
		{
			self->hadjustment->value = child->width -
				 (widget->width - self->vscrollbar_shown);
		}
	
		if (child->height - self->vadjustment->value <
			widget->height - self->hscrollbar_shown)
		{
			self->vadjustment->value = child->height -
				(widget->height - self->hscrollbar_shown);
		}
		
		if (self->hadjustment->value < 0)
		{
			self->hadjustment->value = 0;
		}
		
		if (self->vadjustment->value < 0)
		{
			self->vadjustment->value = 0;
		}
	}
		
	/* Now set the offset */
	child->col = widget->col - self->hadjustment->value;
	child->row = widget->row - self->vadjustment->value;
}
	