/* -*- indent-tabs-mode: t; tab-width: 8; c-basic-offset: 8; -*- */

#include "util.h"
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <ctype.h>
#include <stdio.h>

static int valid_a_char(char c);
static int valid_d_char(char c);

int valid_a_char(char c)
{
	return ((c >= ' ' && c <= '"') ||
		(c >= '%' && c <= '?') ||
		(c >= 'A' && c <= 'Z') || (c == '_'));
}

int valid_d_char(char c)
{
	return ((c >= '0' && c <= '9') ||
		(c >= 'A' && c <= 'Z') || (c == '_'));
}

void iso_a_strcpy(unsigned char *dest, int size, const char *src)
{
	int i = 0, d = 0;

	if (src)
		for (; i < size && *src; ++i, ++src) {
			char s = toupper(*src);

			if (valid_a_char(s))
				dest[d++] = s;
			else
				dest[d++] = '_';
		}
	for (; i < size; ++i) {
		/* pad with spaces */
		dest[d++] = ' ';
	}
}

void iso_d_strcpy(unsigned char *dest, int size, const char *src)
{
	int i = 0, d = 0;

	if (src)
		for (; i < size && *src; ++i, ++src) {
			char s = toupper(*src);

			if (valid_d_char(s))
				dest[d++] = s;
			else
				dest[d++] = '_';
		}
	for (; i < size; ++i) {
		/* pad with spaces */
		dest[d++] = ' ';
	}
}

char *iso_a_strndup(const char *src, int size)
{
	int i, d;
	char *out;

	out = malloc(size + 1);
	for (i = d = 0; i < size && *src; ++i, ++src) {
		char s = toupper(*src);

		if (valid_a_char(s))
			out[d++] = s;
		else
			out[d++] = '_';
	}
	out[d++] = '\0';
	out = realloc(out, d);	/* shrink the buffer to what we used */
	return out;
}

char *iso_d_strndup(const char *src, int size)
{
	int i, d;
	char *out;

	out = malloc(size + 1);
	for (i = d = 0; i < size && *src; ++i, ++src) {
		char s = toupper(*src);

		if (valid_d_char(s))
			out[d++] = s;
		else
			out[d++] = '_';
	}
	out[d++] = '\0';
	out = realloc(out, d);	/* shrink the buffer to what we used */
	return out;
}

char *iso_strconcat(char sep, const char *a, const char *b)
{
	char *out;
	int la, lb;

	la = strlen(a);
	lb = strlen(b);
	out = malloc(la + lb + 1 + (sep ? 1 : 0));
	memcpy(out, a, la);
	memcpy(out + la + (sep ? 1 : 0), b, lb);
	if (sep)
		out[la] = sep;
	out[la + lb + (sep ? 1 : 0)] = '\0';
	return out;
}

char *iso_strdup(const char *src)
{
	return src ? strdup(src) : NULL;
}

void iso_lsb(unsigned char *buf, int num, int bytes)
{
	int i;

	assert(bytes <= sizeof(int));

	for (i = 0; i < bytes; ++i)
		buf[i] = (num >> (8 * i)) & 0xff;
}

void iso_msb(unsigned char *buf, int num, int bytes)
{
	int i;

	assert(bytes <= sizeof(int));

	for (i = 0; i < bytes; ++i)
		buf[bytes - 1 - i] = (num >> (8 * i)) & 0xff;
}

void iso_bb(unsigned char *buf, int num, int bytes)
{
	int i;

	assert(bytes <= sizeof(int));

	for (i = 0; i < bytes; ++i)
		buf[i] = buf[2 * bytes - 1 - i] = (num >> (8 * i)) & 0xff;
}

void iso_datetime_7(unsigned char *buf, time_t t)
{
	static int tzsetup = 0;
	static int tzoffset;
	struct tm tm;

	if (!tzsetup) {
		tzset();

		tzoffset = -timezone / 60 / 15;
		if (tzoffset < -48)
			tzoffset += 101;

		tzsetup = 1;
	}

	localtime_r(&t, &tm);

	buf[0] = tm.tm_year;
	buf[1] = tm.tm_mon + 1;
	buf[2] = tm.tm_mday;
	buf[3] = tm.tm_hour;
	buf[4] = tm.tm_min;
	buf[5] = tm.tm_sec;
	buf[6] = tzoffset;
}

void iso_datetime_17(unsigned char *buf, time_t t)
{
	static int tzsetup = 0;
	static int tzoffset;
	struct tm tm;
	char c[5];

	if (t == (time_t) - 1) {
		/* unspecified time */
		memset(buf, '0', 16);
		buf[16] = 0;
	} else {
		if (!tzsetup) {
			tzset();

			tzoffset = -timezone / 60 / 15;
			if (tzoffset < -48)
				tzoffset += 101;

			tzsetup = 1;
		}

		localtime_r(&t, &tm);

		/* year */
		sprintf(c, "%04d", tm.tm_year + 1900);
		memcpy(&buf[0], c, 4);
		/* month */
		sprintf(c, "%02d", tm.tm_mon + 1);
		memcpy(&buf[4], c, 2);
		/* day */
		sprintf(c, "%02d", tm.tm_mday);
		memcpy(&buf[6], c, 2);
		/* hour */
		sprintf(c, "%02d", tm.tm_hour);
		memcpy(&buf[8], c, 2);
		/* minute */
		sprintf(c, "%02d", tm.tm_min);
		memcpy(&buf[10], c, 2);
		/* second */
		sprintf(c, "%02d", MIN(59, tm.tm_sec));
		memcpy(&buf[12], c, 2);
		/* hundreths */
		memcpy(&buf[14], "00", 2);
		/* timezone */
		buf[16] = tzoffset;
	}
}

void iso_split_filename(char *name, char **ext)
{
	char *r;

	r = strrchr(name, '.');
	if (r) {
		*r = '\0';
		*ext = r + 1;
	} else
		*ext = "";
}

void iso_filecpy(unsigned char *buf, int size, const char *name, int version)
{
	char v[6];
	int nl, vl;

	assert(version >= 0);
	assert(version < 0x8000);

	nl = strlen(name);

	memcpy(buf, name, nl);

	if (!version)
		assert(size >= strlen(name));
	else {
		sprintf(v, "%d", version);
		vl = strlen(v);
		assert(size >= nl + vl + 1);

		buf[nl] = ';';
		memcpy(&buf[nl + 1], v, vl);

		nl += vl + 1;
	}
	/* pad with spaces */
	if (nl < size)
		memset(&buf[nl], ' ', size - nl);
}
