/****************************************************************************
** genparse.cc
**
** Mike Borella <mike@borella.net>
**
** A command-line parser generator
**
** $Id: genparse.cc,v 1.4 2000/10/25 22:52:05 mborella Exp $
**
****************************************************************************/

#include <iostream.h>
#include <cstdio>
#include <string>
#include "genparse.h"
#include "logfile.h"
#include "clparam_list.h"
#include "parse_cl.h"
#include "unique_int.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

// Characters are 0-255, so this the first safe value
const int FIRST_LONGOPT_VALUE = 256;

char *yyfilename;
extern FILE *yyin;
extern int yyparse();
extern int yydebug;

// This is the cat's pajamas - the main object
Clparam_list my_params;

// Declare the logfile object globally so that we can access if from 
// everywhere
Logfile mylog;

// The object that will give us longopt numbers
Unique_Int longopt_value(FIRST_LONGOPT_VALUE);

/*---------------------------------------------------------------------------
 * 
 * add_extra_params()
 *
 * Add -h, -v
 *-------------------------------------------------------------------------*/

void add_extra_params(void)
{
  Clparam *p;

  // if any of the extra parameters are not already set, add them now
  // start with help
  mylog.write("Adding h parameter");
  if (!my_params.contains('h'))
    {
      p = new Clparam();
      p->short_rep('h');
      p->long_rep("help");
      p->internal_name("h___help");
      p->external_name("h");
      p->type(flag_t);
      p->add_description("Display help information.");
      p->long_only(false);
      my_params.add(*p);
      delete p;
    }

  // version
  mylog.write("Adding v parameter");
  if (!my_params.contains('v'))
    {
      p = new Clparam();
      p->short_rep('v');
      p->long_rep("version");
      p->internal_name("v___version");
      p->external_name("v");
      p->type(flag_t);
      p->add_description("Output version.");
      p->long_only(false);
      my_params.add(*p);
      delete p;
    }
}

/*---------------------------------------------------------------------------
 * 
 * main()
 *
 *-------------------------------------------------------------------------*/

int main(int argc, char *argv[])
{
  // Parse my command line parameters
  // Here's the lowdown:
  // d = debug (logging) flag
  // f = filename of log file
  // o = filename of output file
  Cmdline cl(argc, argv);
  
  // if the user just wants help or the version, do that now then quit
  if (cl.v())
    {
      cout << PACKAGE << " v" << VERSION << endl;
      exit(0);
    }
  if (cl.h())
    {
      cl.usage();
      exit(0);
    }
  
  // set quiet mode if specified
  if (cl.q())
    my_params.quiet();

  // make sure that we've got a file to parse
  if (cl.next_param() >= argc)
    {
      EH e("Must include a genparse file on the command line.");
      throw (e);
    }
  
  // Start the log file if debugging mode is turned on
  if (cl.d())
    {
      mylog.open(cl.f());
      yydebug= 1;
    }

  // Open the .gp file for parsing in C style so that the yacc code works
  FILE *fp = fopen(argv[cl.next_param()], "r");
  if (fp == NULL) 
    {
      EH e("can't open input genparse file: " 
	   + (string) argv[cl.next_param()]);
      throw e;
    }
  yyin = fp;
  mylog.write("opened input file: " + (string) argv[cl.next_param()]);

  // set the filename and run the parser on the .gp file
  yyfilename = argv[cl.next_param()];
  if (yyparse())
    exit(1);

  // if we got here, it parsed successfully, now we clean up a little
  fclose(fp);
  mylog.write("done reading input file");
  mylog.write_separator();

  // add -h, -q, and -v params if not already defined
  add_extra_params();
  
  // for debugging purposes, it is useful to traverse the list and make sure
  // it really has everything that we gave it
  my_params.logdump(mylog);
  
  // set the parsing function / class
  my_params.set_parsefunc(cl.p());

  // set the c++ extension
  my_params.set_cppext(cl.c());

  // Now we have to decide which language to output the files in
  if (cl.l() == "c" || cl.l() == "C")
    my_params.output_c(cl.o());
  if (cl.l() == "c++" || cl.l() == "cpp" || cl.l() == "cc" 
      || cl.l() == "cxx")
    my_params.output_cpp(cl.o());

  // Close the log file before we leave
  mylog.close();

  // quit, and do it nicely!
  exit(0);
  
}
