#
#  This file contains code which makes fv more scriptable either from
#  TCL or via XPA entry points
#


namespace eval fvCmds {
   variable currObj ""
   variable currHDU 1

   proc open { args } {
      global listObjs
      variable currObj
      variable currHDU

      if { [llength $args]==0 } {
	 # Return list of opened files
	 set files ""
	 foreach file $listObjs {
	    append files "[$file getOrigName]\n"
	 }
	 return $files
      }

      foreach file $args {
         set currObj [openFitsFile $file]
      }
      set currHDU 1
   }


   proc close { args } {
      variable currObj

      checkForCurrentFile
      set argc [llength $args]
      if { $argc==0 } {
         $currObj closeCmd
      } else {
         foreach ext $args {
            if { $ext == "current" || $ext == "." } {
               set ext [expr $currHDU - 1]
            }
            $currObj closeExtension [expr $ext+1]
         }
      }
   }

   proc display { args } {
      global listObjs
      variable currObj
      variable currHDU

      checkForCurrentFile
      if { [llength $args]<2 } {
         error "Usage: display header|table|image|curve extNum ?opts?"
      }

      set ext [lindex $args 1]
      if { $ext!="current" && $ext!="." } {
         set currHDU [expr $ext+1]
      }
      switch -- [lindex $args 0] {
         "header" {
            $currObj openHeader $currHDU
         }
         "table" {
            $currObj openTable  $currHDU [lrange $args 2 end] 1
         }
         "image" {
            $currObj plotData   $currHDU [lrange $args 2 end]
         }
         "curve" {
            $currObj plotData   $currHDU [lrange $args 2 end]
         }
         default {
            error "Unrecognized display object: [lindex $args 0]"
         }
      }
   }

   proc pow { args } {
      if { [llength $args]==0 } {
         error "Usage: pow powCmd ?args ...?"
      }
      return [namespace eval ::powCmds $args]
   }

   proc quit {} {
      exitCmd
   }


   proc save { args } {
      variable currObj
      
      checkForCurrentFile
      set argc [llength $args]
      if { $argc == 0 } {
         $currObj save
      } else {
         $currObj saveAs [lindex $args 0]
      }
   }

   proc select { args } {
      global listObjs
      variable currObj
      variable currHDU

      if { [llength $args]==0 } {
         checkForCurrentFile
         set idx [lsearch -exact $listObjs $currObj]
         set file [lindex $listObjs $idx]
         return [$file getOrigName]
      } else {
         if { [llength $args]>1 } {
            set currHDU [expr [lindex $args 1]+1]
         } else {
            set currHDU 1
         }
         set theIdx [lindex $args 0]
         if { [regexp {^[0-9]*$} $theIdx] \
               || $theIdx=="first" || $theIdx=="end" } {
            set currObj [lindex $listObjs $theIdx]
         } elseif { $currObj != ""  && $theIdx != "current" && $theIdx!="."} {
            foreach file $listObjs {
               set f [$file getOrigName]
               if { $theIdx==$f || $theIdx==[urlTail $f] } {
                  set currObj $file
                  return
               }
            }
            error "No file found matching $theIdx"
         }
      }

   }

   proc sort { args } {
      variable currObj
      variable currHDU

      checkForCurrentFile
      set ext [lindex $args 0]
      if { $ext!="current" && $ext!="." } {
         set currHDU [expr $ext+1]
      }
      set fT [$currObj openTable $currHDU - 0]

      set keys    {}
      set dirs    {}
      set unique  0
      set currDir 1
      foreach arg [lrange $args 1 end] {
         switch -glob -- $arg {
            "-asc*" {
               set currDir 1
            }
            "-des*" {
               set currDir 0
            }
            "-un*" {
               set unique 1
            }
            default {
               lappend keys $arg
               lappend dirs $currDir
            }
         }
      }

      $fT doSort $keys $dirs $unique
   }

   proc version {} {
      global fvVersion
      return $fvVersion
   }

   ########################
   # Utility routines, not public command
   ########################

   proc checkForCurrentFile {} {
      global listObjs
      variable currObj
      variable currHDU

      if { $currObj=="" || [lsearch -exact $listObjs $currObj]==-1 } {
         # currObj is invalid.  Can we select a new default?
         if { [llength $listObjs]==0 } {
            error "No files available"
         }
         set currObj [lindex $listObjs end]
         set currHDU 1
      }

   }

}


####################   XPA  Entry Points   ####################

namespace eval fvXPA {
   variable xpaPt

   proc init {} {
      variable xpaPt

      if { [catch {load libtclxpa.so}] && [catch {load "" xpa}] } return
      set xpaPt [xpacmdnew "" fv]

      foreach {cmd snd rcv hlp} [list                                    \
            close      0 1 "Close a file, or extension windows"          \
            display    0 1 "Display contents of an extension"            \
            open       1 1 "Open a FITS file"                            \
            pow        1 1 "Execute a pow command"                       \
            quit       0 1 "Quit fv"                                     \
            save       0 1 "Save file"                                   \
            select     1 1 "Select an opened file for manipulation"      \
            sort       0 1 "Sort a table"                                \
            tcl        0 1 "Execute tcl code"                            \
            version    1 0 "Return fv version number"                    \
            ] {
         register $cmd $snd $rcv $hlp
      }

   }

   proc register { cmd snd rcv hlp } {
      variable xpaPt
      
      if { $snd } {
         set sndParam [list fvXPA::send $cmd ""]
      } else {
         set sndParam [list "" "" ""]
      }
      if { $rcv } {
         set rcvParam [list fvXPA::recv $cmd "fillbuf=false"]
      } else {
         set rcvParam [list "" "" ""]
      }
      eval xpacmdadd $xpaPt $cmd {$hlp} $sndParam $rcvParam
   }


   proc send { xpa client_data paramlist } {
      switch -exact $client_data {
         default {
            if { [namespace eval ::fvCmds info procs $client_data]!="" } {
               xpasetbuf $xpa [eval ::fvCmds::$client_data $paramlist]
            } else {
               error "$client_data is an invalid command"
            }
         }
      }
   }


   proc recv { xpa client_data paramlist buf len } {
      switch -exact $client_data {
         pow {
            if { [llength $paramlist]==0 } {
               error "pow powCmd ?args ...?"
            }
            ::powXPA::recv $xpa [lindex $paramlist 0] \
                  [lrange $paramlist 1 end] $buf $len
         }
         tcl {
            set dchan [xparec $xpa datachan]
            set scrpt [read $dchan]
            close $dchan
            namespace eval ::fvCmds $scrpt
         }
         default {
            if { [namespace eval ::fvCmds info procs $client_data]!="" } {
               eval ::fvCmds::$client_data $paramlist
            } else {
               error "$client_data is an invalid command"
            }
         }
      }
   }

}
