
/**************************************************************************
 *                                                                        *
 *  BTools - Miscellaneous Java utility classes                           *
 *                                                                        *
 *  Copyright (c) 1998-2001, Ben Burton                                   *
 *  For further details contact Ben Burton (benb@acm.org).                *
 *                                                                        *
 *  This program is free software; you can redistribute it and/or         *
 *  modify it under the terms of the GNU General Public License as        *
 *  published by the Free Software Foundation; either version 2 of the    *
 *  License, or (at your option) any later version.                       *
 *                                                                        *
 *  This program is distributed in the hope that it will be useful, but   *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *  General Public License for more details.                              *
 *                                                                        *
 *  You should have received a copy of the GNU General Public             *
 *  License along with this program; if not, write to the Free            *
 *  Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,        *
 *  MA 02111-1307, USA.                                                   *
 *                                                                        *
 **************************************************************************/

/* end stub */

package org.gjt.btools.gui.lookandfeel;

import java.awt.event.*;
import javax.swing.*;

/**
 * Produces a menu through which the user can set the swing look and
 * feel.  The menu will contain a menu item for each look and
 * feel.  Selecting a menu item will instigate a change to the
 * corresponding look and feel.
 * <p>
 * The menu will automatically fill itself with a list of available 
 * look and feels.  The currently selected look and feel will have a
 * check mark beside it.  The checked/unchecked status of menu items
 * will be automatically maintained.
 * <p>
 * If the swing look and feel is changed from outside this menu,
 * <tt>refreshStates()</tt> should be called to ensure the menu is displaying
 * the correct look and feel as being selected.
 *
 * @see #refreshStates
 */
public class LookAndFeelMenu extends JMenu {
    /**
     * The object used to set the look and feel once a menu item has
     * been selected.
     */
    private LookAndFeelSetter setter;

    /**
     * The look and feel menu item currently selected.
     */
    private JCheckBoxMenuItem selected;

    /**
     * Creates a new look and feel menu.  This menu will automatically
     * fill itself with a list of available look and feels.
     *
     * @param setter the object used to set the look and feel when a
     * menu item is selected.
     */
    public LookAndFeelMenu(LookAndFeelSetter setter) {
        super("Look and Feel");
        this.setter = setter;
        this.selected = null;
        setMnemonic(KeyEvent.VK_L);

        try {
            LookAndFeelItem[] all = LookAndFeelItem.getInstalledLookAndFeels();
            LookAndFeelMenuItem item;
            String currentLFName = UIManager.getLookAndFeel().getName();
            for (int i=0; i<all.length; i++) {
                item = new LookAndFeelMenuItem(all[i]);
                add(item);
                if (all[i].toString().equals(currentLFName)) {
                    item.setSelected(true);
                    selected = item;
                }
            }
        } catch (Throwable th) {
            add(new JMenuItem("Cannot list installed look and feels."));
        }
    }

    /**
     * Determines the look and feel currently in use and updates the menu
     * to display it as being selected.
     * This should be called whenever the swing look and feel
     * is changed from outside this menu.
     */
    public void refreshStates() {
        LookAndFeelMenuItem shouldSelect = null;
        try {
            String currentLFName = UIManager.getLookAndFeel().getName();

            JMenuItem item;
            int n = getItemCount();
            for (int i = 0; i < n; i++) {
                item = getItem(i);
                if (item instanceof LookAndFeelMenuItem)
                    if (item.getText().equals(currentLFName)) {
                        shouldSelect = (LookAndFeelMenuItem)item;
                        break;
                    }
            }
        } catch (Throwable th) {}

        if (selected != shouldSelect) {
            if (selected != null)
                selected.setSelected(false);
            if (shouldSelect != null)
                shouldSelect.setSelected(true);
            selected = shouldSelect;
        }
    }

    /**
     * A menu item for a specific look and feel.
     */
    private class LookAndFeelMenuItem extends JCheckBoxMenuItem
            implements ActionListener {
        /**
         * The specific look and feel that this item refers to.
         */
        private LookAndFeelItem item;

        /**
         * Creates a new look and feel menu item based upon the given
         * look and feel.  An action listener is automatically
         * registered.
         *
         * @param item the given look and feel.
         */
        public LookAndFeelMenuItem(LookAndFeelItem item) {
            super(item.toString());
            this.item = item;
            addActionListener(this);
        }

        /**
         * Returns the look and feel associated with this menu item.
         *
         * @return the associated look and feel.
         */
        public LookAndFeelItem getLookAndFeelItem() {
            return item;
        }

        /**
         * Called when this menu item is selected.
         *
         * @param e the selection event.
         */
        public void actionPerformed(ActionEvent e) {
            if (setter.setLookAndFeel(item)) {
                if (selected != null)
                    selected.setSelected(false);
                setSelected(true);
                selected = this;
            } else
                setSelected(false);
        }
    }
}

