/*********************************************************************/
/*  bibView: Administration of BibTeX-Databases                      */
/*           (Verwaltung von BibTeX-Literaturdatenbanken)            */
/*                                                                   */
/*  Module:  ctl_open.c                                              */
/*                                                                   */
/*             Open Control                                          */
/*             - Menu function New                                   */
/*             - Menu function Open                                  */
/*                                                                   */
/*  Author:  Holger Martin,  martinh@informatik.tu-muenchen.de       */
/*           Peter M. Urban, urban@informatik.tu-muenchen.de         */
/*                                                                   */
/*  History:                                                         */
/*    12.05.91  PMU  created                                         */
/*    05.26.92       Version 1.0 released                            */
/*                                                                   */
/*  Copyright 1992 TU MUENCHEN                                       */
/*    See ./Copyright for complete rights and liability information. */
/*                                                                   */
/*********************************************************************/

#include <stdio.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xaw/Cardinals.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Dialog.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/Box.h>
#include "FileNom.h"
#include "bibview.h"


/* macros and definitions */
/* ---------------------- */
#define NN_FILENAME	"noname"
#define NN_FILEEXT	".bib"
#define NN_FILENAMELEN  14


/* imported global variables */
/* ------------------------- */
extern XtAppContext app_context;
extern Widget topLevel, mainMenu, desktop;
extern Pixmap questPixmap;

extern int sortedby;

extern char *actual_path;


/* exported global variables */
/* ------------------------- */


/* local function prototypes */
/* ------------------------- */
static void LoadBibFile (Widget w, XtPointer clientData, XtPointer callData);
static void CancelLoad (Widget w, XtPointer clientData, XtPointer callData);
static void confirmLoad (BibPtr bp);
static void cancelLoadCmd (Widget w, XtPointer clientData, XtPointer callData);
static void loadCmdOk (Widget w, XtPointer clientData, XtPointer callData);


/* local global variables */
/* ---------------------- */
static int nnCnt = 0;
static BibPtr gbp = NULL;


/*********************************************************************/
/* copNewCmd:                                                        */
/*    Callback function for option new in main window file menu      */
/*********************************************************************/
void
copNewCmd (Widget w, XtPointer clientData, XtPointer callData)
{
BibPtr bp;
FILE *tempFile;
int status;

   /* make new global data struct for bib file */
   if ((status = glbNewBibListEl(&bp)) != OK) {
      guwError(status);
      return;
   }

   /* build new noname filename */
   sprintf(bp->filename, "%s%d%s", NN_FILENAME, nnCnt+1, NN_FILEEXT);
   /*
#ifdef SYSV
   (void) getcwd(bp->filepath, MAX_FILEPATHLEN);
#else 
   (void) getwd(bp->filepath, MAX_FILEPATHLEN);
#endif
   */
   strcpy(bp->filepath, actual_path);
   strcat(bp->filepath, "/");
   strcat(bp->filepath, bp->filename);

   bp->ew = NULL;
   bp->mw = NULL;
   bp->lw = NULL;
   bp->sortedby = sortedby;

   /* create temp file */
#ifdef NO_TEMPNAM
   bp->tempfile = (char *)tmpnam(NULL);
#else
   bp->tempfile = (char *)tempnam(NULL, NULL);
#endif
   tempFile = fopen(bp->tempfile, "w" );
   fclose(tempFile);

   /* display window for bib */
   if ((status = gubOpenBibWin(bp)) != OK) {
      guwError(status); 
      return;
   }
   nnCnt++;
}


/*********************************************************************/
/* copOpenCmd:                                                       */
/*    Callback function for option open in main window file menu     */
/*********************************************************************/
void
copOpenCmd (Widget w, XtPointer clientData, XtPointer callData)
{
static Widget fsbShell, fsbBox, fsbDialog;
Position dx, dy, x, y;

   XtVaGetValues(desktop,
                 XtNx, &dx,
                 XtNy, &dy, NULL);
   XtTranslateCoords(desktop,
                     (Position)dx + SUBWIN_MARGIN,
                     (Position)dy + SUBWIN_MARGIN,
                     &x, &y);
   fsbShell  = XtVaCreatePopupShell("fileSelectBoxShell",
	         topLevelShellWidgetClass, desktop, 
	 	 XtNx, x, XtNy, y, NULL);
   fsbBox    = XtVaCreateManagedWidget("fileSelectBox",
                 boxWidgetClass, fsbShell, NULL);
   fsbDialog = XtVaCreateManagedWidget("loadFileBoxShell",
                fileNominatorWidgetClass, fsbBox,
                XtNborderWidth, 0, NULL);

   XtAddCallback(fsbDialog, XtNcancelCallback, CancelLoad, fsbShell);
   XtAddCallback(fsbDialog, XtNselectCallback, LoadBibFile, fsbDialog);

   XtSetSensitive(mainMenu, FALSE);
   gubSetSensitive(NULL, FALSE);
   XtPopup(fsbShell, XtGrabNonexclusive);
}



/*********************************************************************/
/* LOCAL FUNCTIONS                                                   */
/*********************************************************************/

/*********************************************************************/
/* CancelLoad:                                                       */
/*    Callback function for CANCEL button in file select box         */
/*********************************************************************/
static void
CancelLoad (Widget w, XtPointer clientData, XtPointer callData)
{
Widget dialog = (Widget)clientData;

   XtSetSensitive(mainMenu, TRUE);
   gubSetSensitive(NULL, TRUE);
   XtPopdown(dialog);

}


/*********************************************************************/
/* LoadBibFile:                                                      */
/*    Callback function for OK button in file select box             */
/*********************************************************************/
static void
LoadBibFile (Widget w, XtPointer clientData, XtPointer callData)
{
Widget dialog = (Widget)clientData;
String filepath;
BibPtr p, bp;
Errcode status;

   /* get and keep filename */
   filepath = (String)FileNominatorGetFullFileName(dialog);
   if (filepath == NULL){
      XtPopdown(XtParent(XtParent(dialog)));
      guwError(NO_VALID_FILENAME);
      return;
      }

   /* check if opened already */
   p = glbFirstBibListEl();
   while (p != NULL) {
      if (strcmp(p->filepath, filepath) == 0) {
	 XtSetSensitive(mainMenu, TRUE);
	 gubSetSensitive(NULL, TRUE);
	 XtPopdown(XtParent(XtParent(dialog)));
	 confirmLoad(p);
	 return;
      }
      p = glbNextBibListEl(p);
   }

   /* make new global data struct for bib file */
   if ((status = glbNewBibListEl(&bp)) != OK) {
      guwError(status);
      XtSetSensitive(mainMenu, TRUE);
      gubSetSensitive(NULL, TRUE);
      XtPopdown(XtParent(XtParent(dialog)));
      return;
   }
   strcpy(bp->filepath, filepath);
   strcpy(bp->filename, (String)FileNominatorGetFileName(dialog));
   bp->mw=NULL;
   bp->ew=NULL;
   bp->lw=NULL;

   bp->sortedby = sortedby;

   /* remove file select box */
   XtSetSensitive(mainMenu, TRUE);
   gubSetSensitive(NULL, TRUE);
   XtPopdown(XtParent(XtParent(dialog))); 

   glbReadFileOpenBib(bp);

}


/*********************************************************************/
/* confirmLoad:                                                      */
/*    Opens dialogbox for user to confirm loading from disk          */
/*********************************************************************/
static void
confirmLoad (BibPtr bp)
{
static Widget conShell, conDialog, conOKButton, conQuitButton;
Position dx, dy, x, y;

   XtVaGetValues(desktop,
                 XtNx, &dx,
                 XtNy, &dy, NULL);
   XtTranslateCoords(desktop,
                     (Position)dx + SUBWIN_MARGIN,
                     (Position)dy + SUBWIN_MARGIN,
                     &x, &y);
   conShell = XtVaCreatePopupShell("confirmLoadShell",
	        transientShellWidgetClass, topLevel, 
		XtNx, x, XtNy, y, NULL);
   conDialog = XtVaCreateManagedWidget("confirmLoadBox",
	         dialogWidgetClass, conShell, 
		 XtNicon, questPixmap, NULL);
   conOKButton = XtVaCreateManagedWidget("ok",
	           commandWidgetClass, conDialog, NULL);
   conQuitButton = XtVaCreateManagedWidget("cancel",
	           commandWidgetClass, conDialog, NULL);
   XtAddCallback(conQuitButton, XtNcallback, cancelLoadCmd, conShell);
   XtAddCallback(conOKButton, XtNcallback, loadCmdOk, conShell);

   XtSetSensitive(mainMenu, FALSE);
   gubSetSensitive(NULL, FALSE);
   gbp = bp;
   XtPopup(conShell, XtGrabNonexclusive);
}


/*********************************************************************/
/* cancelLoadCmd:                                                    */
/*    Callback function for CANCEL button in confirm box             */
/*********************************************************************/
static void
cancelLoadCmd (Widget w, XtPointer clientData, XtPointer callData)
{
Widget shell = (Widget)clientData;

   XtSetSensitive(mainMenu, TRUE);
   gubSetSensitive(NULL, TRUE);
   XtPopdown(shell);
   gbp = NULL;
}


/*********************************************************************/
/* loadCmdOK:                                                        */
/*    Callback function for OK button in confirm box                 */
/*********************************************************************/
static void
loadCmdOk (Widget w, XtPointer clientData, XtPointer callData)
{
Widget shell = (Widget) clientData;
BibPtr bp;
Errcode status;

   /* make new global data struct for bib file */
   if ((status = glbNewBibListEl(&bp)) != OK) {
      guwError(status);
      return;
   }
   strcpy(bp->filepath, gbp->filepath);
   strcpy(bp->filename, gbp->filename);
   
   bp->lw = NULL;
   bp->ew = NULL;
   bp->mw = NULL;
   bp->sortedby = sortedby;

   /* delete file in memory */
   if ((status = dbtDeleteTree(gbp->treeIdx)) != DBT_OK) {
      guwError(status);
   }

   if ((status = gueCloseBibErrWin(gbp)) != OK) {
      guwError(status);
   }

   if ((status = gubCloseBibWin(gbp)) != OK) {
      guwError(status);
   }
   gbp = NULL;

   /* remove confirm box */
   XtPopdown(shell);
   XtSetSensitive(mainMenu, TRUE);
   gubSetSensitive(NULL, TRUE);
   glbReadFileOpenBib(bp);
}


