/******************************************************************************
 * program:     wp2latex                                                      *
 * function:    convert WordPerfect 5.0 or 5.1 files into LaTeX               *
 * modul:       wp2lfir.c                                                     *
 * description: This modul contains functions for first pass. In the first    *
 *              pass information of the WP binary file will splitted in two   *
 *              parts:                                                        *
 *              1) A text file containing the whole text. The special WP      *
 *                 characters are translated.                                 *
 *              2) A binary file which contains information about             *
 *                 environments, tabbings, line endings                       *
 ******************************************************************************/
#include <io.h>
#include <math.h>
#include <stdio.h>
#include "wp2latex.h"

static void Make_tabelentry_attr(struct LOC_Convert_first_pass *LINK)
{
  uchar num_of_attr;
  short j;

  putc(num_of_attr = LINK->openptr[LINK->depth], tabel);

  for (j = 1; j <= num_of_attr; j++)
     putc(LINK->attr_rij[LINK->depth][j], tabel);
}

static void Make_tabelentry_tabset(struct LOC_Convert_first_pass *LINK)
{
  short j, FORLIM;

  putc('S', tabel);
  putc(LINK->num_of_tabs, tabel);

  FORLIM = LINK->num_of_tabs;
  for (j = 0; j < FORLIM; j++)
    Wr_word(tabel, LINK->tabpos[j]);
}

static void Make_tabelentry_rightjustification(struct LOC_Convert_first_pass *LINK)
{
  putc('U', tabel);
  putc( (LINK->by == 0x81) ? 1 : 0, tabel);  /* regels NIET uitvullen */
                                             /* 1 -> regels WEL uitvullen */
}

static void Make_tabelentry_envir_extra_end(struct LOC_Convert_first_pass *LINK)
{
  switch (LINK->envir) {
  case 'C': putc('C', tabel);
            break;
  case 'T': putc('T', tabel);
            break;
  case 'I': putc('I', tabel);
            Wr_word(tabel, LINK->ind_leftmargin);
            Wr_word(tabel, LINK->ind_rightmargin);
            putc( (LINK->ind_text2) ? 1 : 0, tabel);
            break;
  }/*Case*/
  putc(LINK->line_term, tabel);
  putc(0xff, tabel);
}

static void Reset_attr_rij(short d, struct LOC_Convert_first_pass *LINK)
{
  short j;

  for (j = 0; j <= 16; j++)
    LINK->attr_rij[d][j] = 0;
  LINK->leegptr[d] = 1;
  LINK->openptr[d] = 0;
}

static void Open_all_attr(struct LOC_Convert_first_pass *LINK)
{
  /* -- Open alle commando's door de Attributen-rij af te lopen -- */
  short j, FORLIM;

  FORLIM = LINK->leegptr[LINK->depth];
  for (j = LINK->openptr[LINK->depth] + 1; j < FORLIM; j++) {
    fputs(open_com[LINK->attr_rij[LINK->depth][j]], strip);
    LINK->openptr[LINK->depth]++;
  }

  LINK->open_attr_rij[LINK->depth] = false;
      /* Alle attributen staan weer goed */
}

static void Close_all_attr(struct LOC_Convert_first_pass *LINK)
{
  /* -- Sluit alle commando's door de Attributen-rij af te lopen -- */
  short j;

  for (j = LINK->openptr[LINK->depth]; j >= 1; j--) {
    fputs(close_com[LINK->attr_rij[LINK->depth][j]], strip);
    LINK->openptr[LINK->depth]--;
  }
  LINK->open_attr_rij[LINK->depth] = true;
}

static void Attr_ON(struct LOC_Convert_first_pass *LINK)
{
  /* Deze procedure plaatst een attribuut (lettertype) in de attribuut-rij */
                                                     /* read attribut-code */
  LINK->attr_rij[LINK->depth][LINK->leegptr[LINK->depth]] = getc(wpd);
      /* attribuut in attr-rij */
  LINK->leegptr[LINK->depth]++;   /* plaats 1 verder. */
  LINK->open_attr_rij[LINK->depth] = true;   /* openstaande attr-rij */

  getc(wpd);       /* read over absulte code */
}

int Attr_OFF(struct LOC_Convert_first_pass *LINK)
/* --- return: ---
 * int Attr_OFF() : 0 if OK, else error code
 */
{
  /* Deze procedure haalt een uit een attribuut (lettertype) uit de */
  /* attribuut-rij door middel van een stack principe omdat binnen  */
  /* LaTeX de later geopende kommando's eerst afgesloten te worden  */
  uchar b;
  boolean found;
  short j, codeptr, FORLIM;

  b = getc(wpd);   /* read attribut-code */

  j = LINK->leegptr[LINK->depth];   /* zoek vanaf top attr-rij */
  found = false;   /* nog niet gevonden */

  while (j > 1 && !found) {   /* zoek attr-code in attr-rij */
    j--;
    found = (LINK->attr_rij[LINK->depth][j] == b);
  }

  if (j <= 0) {   /* Moet nooit kunnen voorkomen */
    fprintf(stderr, "Program error.\n");
    return 0x100;
  }
  codeptr = j;   /* plaats van attr-code in rij */

  /* Sluit alle commando's t/m de desbetreffende code als deze nog niet */
  /* gesloten zijn.                                                     */

  if (codeptr <= LINK->openptr[LINK->depth]) {
    for (j = LINK->openptr[LINK->depth]; j >= codeptr; j--) {
      fputs(close_com[LINK->attr_rij[LINK->depth][j]], strip);
      LINK->openptr[LINK->depth]--;
    }
  }

  FORLIM = LINK->leegptr[LINK->depth];
  /* Haal de desbetreffende attribuut uit de rij en werk pointers bij */

  for (j = codeptr; j < FORLIM; j++)
    LINK->attr_rij[LINK->depth][j] = LINK->attr_rij[LINK->depth][j + 1];
  LINK->leegptr[LINK->depth]--;

  LINK->open_attr_rij[LINK->depth] = true;   /* openstaande attr-rij */

  getc(wpd);   /* read over absolute code */
  return 0;
}

static void Center(struct LOC_Convert_first_pass *LINK)
{
  /* Deze procedure zorgt voor center environment zolang er nog geen */
  /* andere environment is begonnen.                                 */
  if (LINK->envir == ' ')   /* environment = center */
    LINK->envir = 'C';

  fseek(wpd, 7L, SEEK_CUR);   /* rest van code overslaan */
}

static void End_Align(struct LOC_Convert_first_pass *LINK)
{
  if (LINK->align_tab) {
    Close_all_attr(LINK);
    fprintf(strip, "\\'");
    LINK->align_tab = false;
    Open_all_attr(LINK);
  }

  if (LINK->right_tab) {
    Close_all_attr(LINK);
    fprintf(strip, "\\'");
    LINK->right_tab = false;
    Open_all_attr(LINK);
  }

  if (!LINK->center_tab)
    return;
  Close_all_attr(LINK);
  putc('}', strip);
  LINK->center_tab = false;
  Open_all_attr(LINK);
}

static void Tab(struct LOC_Convert_first_pass *LINK)
{
  short j;
  unsigned short wpu;
  short tabnum, new_tabs, FORLIM;

  if (LINK->envir == 'I' || LINK->nomore_valid_tabs)
  {   /* Noggeen indent --> normaal tab */
    fseek(wpd, 7L, SEEK_CUR);
    return;
  }

  if (LINK->by == 0x48)
    LINK->right_tab = true;

  if (LINK->by == 0x40)
    LINK->align_tab = true;

  if (LINK->by == 0xc8)
    LINK->center_tab = true;

  fseek(wpd, 2L, SEEK_CUR);

  Rd_word(wpd, &wpu);   /* Lees abs.-indent [wpu] */
  wpu -= LINK->WP_sidemargin;   /* Correctie ivm WP kantlijn */

  tabnum = 0;
  FORLIM = LINK->num_of_tabs;
  for (j = 1; j <= FORLIM; j++) {   /* Bepaal welke tabpos */
    if (wpu >= LINK->tabpos[j - 1])
      tabnum = j;
  }

  new_tabs = tabnum - LINK->latex_tabpos;

  if (new_tabs > 0) {
    Close_all_attr(LINK);

    for (j = 1; j <= new_tabs; j++)
      fprintf(strip, "\\>");

    if (LINK->center_tab)
      fprintf(strip, "\\ctab{");

    Open_all_attr(LINK);
  }

  LINK->latex_tabpos = tabnum;

  fseek(wpd, 3L, SEEK_CUR);

  LINK->envir = 'T';   /* Er zit een tab in deze regel */
}

static void Flush_right_tab(struct LOC_Convert_first_pass *LINK)
{
  if (LINK->envir != 'I') {
    Close_all_attr(LINK);
    fprintf(strip, "\\`");
    Open_all_attr(LINK);

    LINK->nomore_valid_tabs = true;

    LINK->envir = 'T';
  }

  fseek(wpd, 7L, SEEK_CUR);
}

static void Indent(struct LOC_Convert_first_pass *LINK)
{
  unsigned short dif, abs;
  uchar b;

  if (LINK->envir == 'T') {
        /*Al een tabcommando gezet dus er mag geen insp */
          fseek(wpd, 10L, SEEK_CUR);
    return;
  }
  LINK->envir = 'I';
  LINK->indenting = true;

  if (LINK->ind_text2) {
    fseek(wpd, 10L, SEEK_CUR);
    return;
  }
  b = getc(wpd) & 0x1;

  Rd_word(wpd, &dif);
  Rd_word(wpd, &abs);   /* Eigenlijk Old current column */
  Rd_word(wpd, &abs);

  LINK->ind_leftmargin = abs - LINK->WP_sidemargin;

  if (b == 1)
    LINK->ind_rightmargin += dif;
  /*Margins bepaald lees voorby rest van functie-codes */
  fseek(wpd, 3L, SEEK_CUR);

  if (LINK->ind_text1)
    return;
  if (LINK->char_on_line) {
    putc('}', strip);
    LINK->ind_text1 = true;
  }
}

static void End_of_indent(struct LOC_Convert_first_pass *LINK)
{
  LINK->indent_end = true;
  fseek(wpd, 5, SEEK_CUR);
}

static void Tabset(struct LOC_Convert_first_pass *LINK)
{
  /* Updated by M. Covington 1993.
     The WP 5.1 tabset code is 4 bytes longer than the WP 5.0 one.
     I have no idea why, but by reading the length fields, we
     can skip the right distance. */

  short    j;
  unsigned short w, length;

  printf("  (Found a [TabSet]; conversion may be unsatisfactory.)\r");
  printf("First pass  :     "); /* Kluged way to display message! MC */

  Rd_word(wpd, &length);  /* Get the length given by Word Perfect */
                           /* Not the same in WP 5.1 and 5.0. */
                           /* Added by M. Covington 1993 */

  fseek(wpd, 100, SEEK_CUR);   /* Ga naar TAB-info */

  LINK->num_of_tabs = 0;

  for (j = 1; j <= 40; j++) {
    Rd_word(wpd, &w);
    if ( ((short)w > LINK->WP_sidemargin) &&
         (w != 0xffffL)            ) {
      LINK->num_of_tabs++;
      LINK->tabpos[LINK->num_of_tabs - 1] = w - LINK->WP_sidemargin;
    }
  }

  fseek(wpd, 2+length-182, SEEK_CUR);

  Make_tabelentry_tabset(LINK);
}

static void Page_number_position(struct LOC_Convert_first_pass *LINK)
{
  uchar position_code;

  fseek(wpd, 5, SEEK_CUR);   /*Skip length of code; always 10*/
  /* + old information */
  position_code = getc(wpd);

  fprintf(strip, "\\pagenumpos");
  switch (position_code) {

  case 0x1:
    fprintf(strip, "{\\pntl}");
    break;

  case 0x2:
    fprintf(strip, "{\\pntc}");
    break;

  case 0x3:
    fprintf(strip, "{\\pntr}");
    break;

  case 0x5:
    fprintf(strip, "{\\pnbl}");
    break;

  case 0x6:
    fprintf(strip, "{\\pnbc}");
    break;

  case 0x7:
    fprintf(strip, "{\\pnbr}");
    break;

  default:
    fprintf(strip, "{\\pnno}");
    break;
  }

  fseek(wpd, 6, SEEK_CUR);
}

static void Character(LINK)
struct LOC_Convert_first_pass *LINK;
{
  char ch[256];

  short j;
  uchar chr_code, chr_set;
  boolean found;

#ifdef DEBUG
        fprintf(stderr, "Called Character with LINK->by = 0x%x\n",LINK->by);
#endif

  if (LINK->open_attr_rij[LINK->depth])
    Open_all_attr(LINK);

  switch (LINK->by) {

  case 0xc4:
#ifdef DEBUG
        fprintf(stderr, "Character got 0x%x\n", LINK->by);
#endif
        fprintf(stderr, "detected special attr\n");
        break;

  case 0xa9:   /* Special_char -- '-' */
  case 0xaa:   /* There are two different representations of '-'      *
                * 0xa9 for '-' and 0xaa if the '-' is situated at the *
                * end of the line. The later were deleted from former *
                * Wp2LaTeX-Releases.                                  */
#ifdef DEBUG
             fprintf(stderr, "Character got 0x%x\n", LINK->by);
#endif
             strcpy(ch, "-");
             break;

  case 0xc0:   /* Extended_char */
#ifdef DEBUG
        fprintf(stderr, "Character got 0x%x\n", LINK->by);
#endif
    j = 127;
    found = false;

    chr_code = getc(wpd);
    chr_set  = getc(wpd);

    while (j < 511 && !found) {
      j++;
      if (chr_code == LINK->char_code[j - 0x80] &&
          chr_set == LINK->char_set[j - 0x80])
        found = true;
    }

    if (found)
      strcpy(ch, LINK->ext_lat[j - 0x80]);
    else
      strcpy(ch, "\\message{Unknown character} ");
		 /* warn the user of unrecognized character.! */
    getc(wpd);
    break;

  default:
    if (LINK->by >= 0x20 && LINK->by <= 0x7f) {
      /* Normal_char  */
      strcpy(ch, LINK->lat[LINK->by - 0x20]);
    }

    break;
  }

  fputs(ch, strip);

}

void Return_Page(struct LOC_Convert_first_pass *LINK)
{
  switch (LINK->by) {

  case 0x0a:
  case 0x8c:   /* Hard return */
    LINK->line_term = 'R';
    break;

  case 0x0d:   /* Soft return */
    LINK->line_term = 'r';
    break;

  case 0xc:   /* Hard page */
    LINK->line_term = 'P';
    break;

  case 0xb:   /* Soft page */
    LINK->line_term = 'p';
    break;
  }

  putc('\n', strip);

  Make_tabelentry_envir_extra_end(LINK);

  if (LINK->indent_end) {
    LINK->envir = ' ';
    LINK->indenting = false;
    LINK->ind_text1 = false;
    LINK->ind_text2 = false;
    LINK->ind_leftmargin = 0;
    LINK->ind_rightmargin = 0;

    LINK->indent_end = false;
  } else if (LINK->envir != 'I')
    LINK->envir = ' ';


  LINK->char_on_line = false;
  LINK->nomore_valid_tabs = false;

  LINK->regelnum++;

  Make_tabelentry_attr(LINK);

  LINK->latex_tabpos = 0;
}

static void Nop80(LINK)
struct LOC_Convert_first_pass *LINK;
{
  /* Om dat het een 1-byte funktie is hoeft er niks overgeslagen */
  /* te worden.                                                  */
}

static void NopC0(LINK)
struct LOC_Convert_first_pass *LINK;
{
  unsigned char s[7], *h, *fip;
#ifdef DEBUG
        fprintf(stderr, "NopC0 called with LINK->by = 0x%x\n", LINK->by);
#endif
  if (LINK->by == 0xc0)
    fseek(wpd, 3, SEEK_CUR);
  if (LINK->by == 0xc1)
    fseek(wpd, 8, SEEK_CUR);
  if (LINK->by == 0xc2)
    fseek(wpd, 10, SEEK_CUR);
  if (LINK->by == 0xc3)
    fseek(wpd, 2, SEEK_CUR);
  if (LINK->by == 0xc4)
    fseek(wpd, 2, SEEK_CUR);
  if (LINK->by == 0xc5)
    fseek(wpd, 4, SEEK_CUR);
  if (LINK->by == 0xc6)
    fseek(wpd, 5, SEEK_CUR);
  if (LINK->by == 0xc7) {
    for ( fip = (h = s) + 6; fip > h; h++ )
       if ( (*h = fgetc(wpd)) == (unsigned char)EOF ) return;
       /* the best way would be to write                        *
        *   fputs("\\discretionary{k-}{k}{ck}"                  *
        * after checking, whether a kind of '-' and 'k' follows */
    *h = 0;
    if ( !memcmp("\x02\x63\x00\x6B\x00\xc7", s, 6) )
       fputc('c', strip);   /* sorry for quick and dirty and not using  *
                             * Character as output routine              */
       /* the best way would be to write
        *   fputs("\\discretionary{k-}{k}{ck}" 
        * after checking, whether a kind of '-' and 'k' follows */
  }
}

static void NopD0(already_read_subfunc_code, LINK)
boolean already_read_subfunc_code;
struct LOC_Convert_first_pass *LINK;
{
  uchar b;
  unsigned short w;

  if (!already_read_subfunc_code)   /* read subfunctioncode */
     b = getc(wpd);

  Rd_word(wpd, &w);   /* Lees lengte 'die nog volgt ' */
  fseek(wpd, w, SEEK_CUR);
}

static void Overstrike(LINK)
struct LOC_Convert_first_pass *LINK;
{
  boolean  first_char_os;
  unsigned short char_width_os, len_of_code;
  long     end_of_code;

#ifdef DEBUG
        fprintf(stderr, "Called Overstrike\n");
#endif

  Rd_word(wpd, &len_of_code);   /* Lees lengte */
  end_of_code = ftell(wpd) / sizeof(uchar) + len_of_code - 4;

  Rd_word(wpd, &char_width_os);

  first_char_os = true;

  while (ftell(wpd) / sizeof(uchar) < end_of_code) {
     LINK->by = getc(wpd);

    if ( IsLegalCharacter(LINK->by) ) {
#ifdef DEBUG
        fprintf(stderr, "Overstrike got 0x%x\n", LINK->by);
#endif
      if (first_char_os) {
        Character(LINK);
        first_char_os = false;
      } else {
        fprintf(strip, "\\llap{");
        Character(LINK);
        putc('}', strip);
      }
      continue;
    }

    if (LINK->by <= 0xbf)
      Nop80(LINK);
    else if (LINK->by >= 0xc0 && LINK->by <= 0xcf)
      NopC0(LINK);
    else if (LINK->by >= 0xd0 && LINK->by <= 0xfe)
      NopD0(false, LINK);
  }

  fseek(wpd, 4, SEEK_CUR);

}

static void Datecode(LINK)
struct LOC_Convert_first_pass *LINK;
{
  /* MC   Found Word Perfect date code.  Just emit "\today". */
  fprintf(strip,"\\today ");
  NopD0(true,LINK);
}

static void Footnote(LINK)
struct LOC_Convert_first_pass *LINK;
{
  uchar    flags;
  unsigned short fn_num, len_of_code;
  long end_of_code;

  Rd_word(wpd, &len_of_code);   /* Lees lengte */
  end_of_code = ftell(wpd) / sizeof(uchar) + len_of_code - 4;

  flags = getc(wpd);

  Rd_word(wpd, &fn_num);

  /* Skip all the garbage */

  fseek(wpd, (getc(wpd) + 1) * 2 + 9, SEEK_CUR);

  Close_all_attr(LINK);

  LINK->depth = 1;
  Reset_attr_rij(LINK->depth, LINK);

  fprintf(strip, "\\footnote[%u]{", fn_num);

  while (ftell(wpd) / sizeof(uchar) < end_of_code) {
     LINK->by = getc(wpd);

    switch (LINK->by) {

    case 0xa:
    case 0xc:
      fprintf(strip, "\\\\ ");
      break;

    case 0xb:
    case 0xd:
      putc(' ', strip);
      break;

    case 0xc3:
      Attr_ON(LINK);
      break;

    case 0xc4:
      if ( Attr_OFF(LINK) ) exit(0x100);
      break;

    default:
#ifdef DEBUG
        fprintf(stderr, "defaulting\n");
#endif

      if ( IsLegalCharacter(LINK->by) ) 
         Character(LINK);
      else if (LINK->by <= 0xbf)
         Nop80(LINK);
      else if (LINK->by >= 0xc0 && LINK->by <= 0xcf)
         NopC0(LINK);
      else if (LINK->by >= 0xd0 && LINK->by <= 0xfe)
         NopD0(false, LINK);

      break;
    }

  }

  Close_all_attr(LINK);   /* Echt nodig ? */
  putc('}', strip);

  fseek(wpd, 4, SEEK_CUR);

  LINK->depth = 0;
  Open_all_attr(LINK);

}

static void Header_Footer(LINK)
struct LOC_Convert_first_pass *LINK;
{
  uchar    subfunc, occurance;
  unsigned short len_of_code;
  long     end_of_code;
  boolean  hf_left, hf_center, hf_right;
  short    j;

  subfunc = getc(wpd);
  Rd_word(wpd, &len_of_code);

  if (len_of_code <= 22) {
    fseek(wpd, len_of_code, SEEK_CUR);
    return;
  }

  end_of_code = ftell(wpd) / sizeof(uchar) + len_of_code - 4;

  fseek(wpd, 7, SEEK_CUR);

  occurance = getc(wpd);

  fseek(wpd, 10, SEEK_CUR);

  Close_all_attr(LINK);
  LINK->depth = 1;

  /* Geen schone attr._lei; Kopieer attributen uit Niveau 0;  Fout in WP 5.0 ? */

  for (j = 0; j <= 15; j++)
    LINK->attr_rij[1][j] = LINK->attr_rij[0][j];

  LINK->leegptr[1] = LINK->leegptr[0];
  LINK->openptr[1] = LINK->openptr[0];

  switch (subfunc) {

  case 0:
  case 1:
    fprintf(strip, "\\headtext");
    break;

  case 2:
  case 3:
    fprintf(strip, "\\foottext");
    break;
  }

  switch (occurance) {

  case 0:
    fprintf(strip, "{\\neverpages}{");
    break;

  case 1:
    fprintf(strip, "{\\allpages}{");
    break;

  case 2:
    fprintf(strip, "{\\oddpages}{");
    break;

  case 3:
    fprintf(strip, "{\\evenpages}{");
    break;
  }

  Open_all_attr(LINK);
  hf_left = true;   /* Beginnen met de linkerkant */
  hf_center = false;
  hf_right = false;

  while (ftell(wpd) / sizeof(uchar) < end_of_code) {
     LINK->by = getc(wpd);

    switch (LINK->by) {

    case 0xc1:
      LINK->by  = getc(wpd) & 0xe0;
      fseek(wpd, 7, SEEK_CUR);

      if (LINK->by == 0xe0) {
        if (hf_left) {
          Close_all_attr(LINK);
          fprintf(strip, "}{");
          Open_all_attr(LINK);

          hf_left = false;
          hf_center = true;
        }
      }

      if (LINK->by == 0x60) {
        if (hf_left) {
          Close_all_attr(LINK);
          fprintf(strip, "}{}{");
          Open_all_attr(LINK);

          hf_left = false;
          hf_right = true;
        }

        if (hf_center) {
          Close_all_attr(LINK);
          fprintf(strip, "}{");
          Open_all_attr(LINK);

          hf_center = false;
          hf_right = true;
        }
      }
      break;

    case 0xc3:
      Attr_ON(LINK);
      break;

#ifndef DEBUG
    case 0xc4:
      if ( Attr_OFF(LINK) ) exit(0x100);
      break;
#endif

    default:
      if ( IsLegalCharacter(LINK->by) )
         Character(LINK);
      else if (LINK->by <= 0xbf)
         Nop80(LINK);
      else if (LINK->by >= 0xc0 && LINK->by <= 0xcf)
         NopC0(LINK);
      else if (LINK->by >= 0xd0 && LINK->by <= 0xfe)
         NopD0(false, LINK);

      break;
    }
  }

  Close_all_attr(LINK);   /* Echt nodig ? */

  fseek(wpd, 4, SEEK_CUR);

  if (hf_left)
    fprintf(strip, "}{}{}");
  if (hf_center)
    fprintf(strip, "}{}");
  if (hf_right)
    putc('}', strip);

  LINK->depth = 0;
  Open_all_attr(LINK);



}

/*---SLAG1----*/

void Convert_first_pass(void)
{
  /* Dit is de komplete procedure van de eerste slag met zijn eigen proc.'s. */
  struct LOC_Convert_first_pass V;

  short convperc;
  long  srtdocpos, fsize;

  Table_Init(&V);
  Ext_chr_init(&V);

  Reset_attr_rij(0, &V);
  Reset_attr_rij(1, &V);
  V.depth = 0;

  WP_Default(&V);

  V.latex_tabpos = 0;
  V.right_tab = false;
  V.align_tab = false;
  V.center_tab = false;

  V.indenting = false;
  V.indent_end = false;
  V.ind_text1 = false;
  V.ind_text2 = false;
  V.ind_leftmargin = 0;
  V.ind_rightmargin = 0;

  V.envir = ' ';

  V.nomore_valid_tabs = false;

  printf("First pass  :     ");

  srtdocpos = ftell(wpd) / sizeof(uchar);
  fsize     = filelength(fileno(wpd)) + 1;

  V.regelnum = 0;

  Make_tabelentry_attr(&V);     /* attribuut instelling */
  Make_tabelentry_tabset(&V);   /* Geef de defaulttabinstelling door */
  /* aan de 2e slag */

  /* MAIN LOOP  */

  while (ftell(wpd) / sizeof(uchar) < fsize-1) {
                           /* MC: was fsize-2 above */
#ifndef sun
    convperc = floor ((double) (ftell(wpd) / sizeof(uchar) - srtdocpos) /
                     (fsize - srtdocpos) * 100 + 1   /* MC */
                     ) ;
    printf("\b\b\b\b%3d%%", convperc);
#endif

    V.by = getc(wpd);

    switch (V.by) {

    case 0xa:
    case 0xd:
    case 0xb:
    case 0xc:
    case 0x8c:
      Return_Page(&V);
      break;

    case 0xc1:
      V.by = getc(wpd) & 0xe8;

      switch (V.by) {

      case 0:
      case 0xc8:
      case 0x48:
      case 0x40:
        Tab(&V);
        break;

      case 0x60:
        Flush_right_tab(&V);
        break;

      case 0xe0:
        Center(&V);
        break;

      default:
        fseek(wpd, 7, SEEK_CUR);
        break;
      }
      break;

    case 0x81:
    case 0x82:
      Make_tabelentry_rightjustification(&V);
      break;

    case 0x83:
      End_Align(&V);
      break;

    case 0xc3:
      Attr_ON(&V);
      break;

    case 0xc4:
      if ( Attr_OFF(&V) ) exit(0x100);
      break;

    case 0xc2:
      Indent(&V);
      break;

    case 0xc6:
      End_of_indent(&V);
      break;

    case 0xc5:
    case 0xc7:
      NopC0(&V);
      break;

    case 0xd0:
      V.by = getc(wpd);
      switch (V.by) {

      case 0x4:
        Tabset(&V);
        break;

      case 0x8:
        Page_number_position(&V);
        break;

      default:
        NopD0(true, &V);
        break;
      }
      break;

#ifdef _IncludeFormulaParsing
		/* formula parsing added by Dirk Lellinger */
    case 0xda: fread(&V.by, sizeof(uchar), 1, wpd);
			#ifdef _DebugFormulaParsing
               printf("\nCode DA und SubCode %d, an Stelle %lx\n",V.by,ftell(wpd));
			#endif
               if ( V.by <= 4 ) ExtractMathematicalFormula(&V);
               else             NopD0(true, &V);
               break;
/* end of add case formula parsing */
#endif

    case 0xd5:
      Header_Footer(&V);
      break;

    case 0xd6:
      V.by = getc(wpd);
      switch (V.by) {

      case 0:
        Footnote(&V);
        break;

      default:
        NopD0(true, &V);
        break;
      }
      break;

    case 0xd8:
      V.by = getc(wpd);
      switch (V.by) {

      case 0:            /* Added by M. Covington. */
        Datecode(&V);
        break;

      case 0x2:
        Overstrike(&V);
        break;

      default:
        NopD0(true, &V);
        break;
      }
      break;

    default:
      if (  
           IsLegalCharacter(V.by) ) {
        V.char_on_line = true;   /*Er (al) is een karakter op deze regel */
        if (V.indenting) {   /*Als er is ingeprongen er na een stuk */
          if (V.ind_text1)   /*tekst is weer ingesprongen (ind_text1) */
            V.ind_text2 = true;
        }
        /*dan hoort dit char bij het ind_txt-blok */

        Character(&V);
      } else if (V.by <= 0x1f || (V.by >= 0x80 && V.by <= 0xbf))
        Nop80(&V);
      else if (V.by >= 0xd0 && V.by <= 0xff)
        NopD0(false, &V);

      break;
    }
  }

  putchar('\n');
  Make_tabelentry_envir_extra_end(&V);

  num_of_lines_stripfile = V.regelnum;
}

