#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include "wp2latex.h"

/* #define _DebugFormulaParsing */
int FormulaNumber = 0;

struct FmlTranslationArray
    {
    char *szWP;               /* original WordPerfect term */
    char *szTeX;              /* translated LATEX term     */
    };

/* special meaning of digits in the LATEX term:
** 0: previous argument,
** 1: next argument,
** 2: next next argument,
** 5: same word as WordPerfect, with "\" before, lower case,
** 6: same word as WordPerfect, with "\" before,right case
** 7: count Ampersands and write count times the following char
** 8: adding \displaystyle if in first level
**
** lower case chars in the WP-term match only lower case chars in
** the WordPerfect formula,
** upper case chars match both cases
**
** note: the order of words is sometimes essential !
** first the more specialized and longer words
** then the more common and shorter words.
*/

struct FmlTranslationArray FmlTransTable[] =
{
/* 11 char words */
{"SMALLCOPROD","\\coprod"}, /* bad translation */
/* 10 char words */
{"STACKALIGN","\\begin{array}{r@{}l}81\\end{array}"},
{"varepsilon","5"},
/* 9 char words */
{"BIGOMINUS","\\ominus"}, /* bad translation */
{"BIGOTIMES","5"},
{"IDENTICAL","?"}, /* no translation known */
{"LINESPACE","\\vspace{1}"}, /* very bad translation */
{"MINUSPLUS","\\mp"},
{"PLUSMINUS","\\pm"},
{"SMALLOINT","\\oint"}, /* bad translation */
{"SMALLPROD","\\prod"}, /* bad translation */
{"THEREFORE","?"}, /* no translation known */
{"UNDERLINE","5{1}"},
/* 8 char words */
{"ANGSTROM","\\AA"},
{"BIGOPLUS","5"},
{"BIGSQCAP","\\sqcap"}, /* bad translation */
{"BIGSQCUP","5"},
{"BIGUPLUS","5"},
{"BIGWEDGE","5"},
{"DOTSAXIS","\\cdots"},
{"DOTSDIAG","\\ddots"},
{"DOTSVERT","\\vdots"},
{"EMPTYSET","5"},
{"INFINITY","\\infty"},
{"LONGDIVS","\\div"}, /* very bad translation */
{"OVERLINE","5{1}"},
{"PARALLEL","5"},
{"SETMINUS","\\\\"},
{"SMALLINT","\\int"}, /* bad translation */
{"SMALLSUM","\\sum"}, /* bad translation */
{"SQSUBSET","5"},
{"SQSUPSET","5"},
{"varsigma","5"},
{"vartheta","5"},
/* 7 char words */
{"BECAUSE","?"}, /* no translation known */
{"BETWEEN","?"}, /* no translation known */
{"BIGODIV","\\div"}, /* very bad translation */
{"BIGODOT","5"},
{"BINOMSM","{\\small {1 \\choose 2}}"},
{"DOTSLOW","\\ldots"},
{"ePSILON","5"},
{"EPSILON","E"},
{"LDBRACK","["}, /* very bad translation */
{"LONGDIV","\\div"}, /* very bad translation */
{"oMIKRON","o"}, /* very bad translation */
{"OMIKRON","O"}, /* very bad translation */
{"MASSERT","\\dashv"},
{"MATFORM",""},   /* no translation kwown */
{"MSANGLE","\\angle"}, /* bad translation */
{"NASYMEQ","\\not\\asymp"},
{"PARTIAL","5"},
{"PHANTOM",""},   /* no translation known */
{"RDBRACK","]"}, /* very bad translation */
{"RTANGLE","?"}, /* no translation known */
{"yPSILON","\\upsilon"},
{"YPSILON","\\Upsilon"},
/* 6 char words */
{"ALIGNC",""},    /* no translation known */
{"ALIGNL",""},    /* no translation kwown */
{"ALINGR",""},    /* no translation known */
{"APPROX","5"},
{"arccos","5"},
{"arcsin","5"},
{"arctan","5"},
{"ASSERT","\\vdash"},
{"ASYMEQ","\\asymp"},
{"BIGCAP","5"},
{"BIGCUP","5"},
{"BIGVEE","5"},
{"CIRCLE","\\dot{0}"}, /* no translation known */
{"COPROD","5"},
{"EXISTS","5"},
{"FORALL","5"},
{"LAMBDA","6"},
{"LANGLE","5"},
{"LBRACE","\\{"},
{"LFLOOR","5"},
{"liminf","5"},
{"limsup","5"},
{"MATRIX","\\begin{array}{7c}1\\end{array}"},
{"MODELS","5"},
{"NEQUIV","\\not\\equiv"},
{"OMINUS","5"},
{"OTIMES","5"},
{"OVERSM","{\\small \\frac{0}{1}}"},
{"PRECEQ","5"},
{"PROPTO","5"},
{"QEQUAL","?"}, /* no translation known */
{"RANGLE","5"},
{"RBRACE","\\}"},
{"RFLOOR","5"},
{"RIMAGE","?"}, /* no translation known */
{"SANGLE","\\angle"}, /* bad translation */
{"SUBSET","5"},
{"SUCCEQ","5"},
{"SUPSET","5"},
{"varphi","5"},
{"varrho","5"},
{"WREATH","\\wr"},
/* 5 char words */
{"ACUTE","5{0}"},
{"aLPHA","5"},
{"ALPHA","A"},
{"ANGLE","5"},
{"BINOM","{1 \\choose 2}"},
{"BREVE","5{0}"},
{"cosec","\\csc"},
{"DDDOT","\\ddot{0}"}, /* no tranlation known */
{"DELTA","6"},
{"DLINE","\\|"},
{"DOTEQ","5"},
{"EQUIV","5"},
{"FROWN","5"},
{"GAMMA","6"},
{"GRAVE","5{0}"},
{"IMAGE","?"}, /* no translation known */
{"kAPPA","5"},
{"KAPPA","K"},
{"LCEIL","5"},
{"NOTIN","5"},
{"NROOT","\\sqrt[1]{2}"},
{"OMEGA","6"},
{"OPLUS","5"},
{"RCEIL","5"},
{"RIGHT","5"},
{"SIGMA","6"},
{"SIMEQ","5"},
{"SMILE","5"},
{"SQCAP","5"},
{"SQCUP","5"},
{"STACK","\\begin{array}{c}1\\end{array}"},
{"THETA","6"},
{"TILDE","5{0}"},
{"TIMES","5"},
{"UPLUS","5"},
{"VARPI","6"},
/* Words with 4 chars */
{"bETA","5"},
{"BETA","B"},
{"BOLD","{\\bf 1}"},
{"CDOT","5"},
{"CONG","5"},
{"cosh","5"},
{"coth","5"},
{"DDOT","5{0}"},
{"DSUM","?"}, /* no translation known */
{"DYAD","\\bar{0}"}, /* no translation known */
{"FROM","_{1}"},
{"FUNC","{\\rm 1}"},
{"GRAD","\\nabla"},
{"HORZ","\\hspace{1}"}, /* very bad translation */
{"IMAG","\\Im"},
{"iOTA","5"},
{"IOTA","I"},
{"ITAL","{\\it 1}"},
{"LEFT","5"},
{"LINE","|"},
{"NISO","?"}, /* no translation known */
{"ODIV","\\div"}, /* very bad translation */
{"ODOT","5"},
{"OINT","5"},
{"OVER","\\frac{0}{1}"},
{"OWNS","\\ni"},
{"PERP","5"},
{"PREC","5"},
{"PROD","5"},
{"REAL","\\Re"},
{"sinh","5"},
{"SQRT","5{1}"},
{"SUCC","5"},
{"tanh","5"},
{"VERT","\\vspace{1}"}, /* very bad translation */
{"zETA","5"},
{"ZETA","Z"},
/* Words with 3 chars */
{"'''","{0'''}"},
{"AND","\\wedge"},
{"arc","{\\rm arc}"},
{"BAR","5{0}"},
{"CAP","5"},
{"cHI","5"},
{"CHI","X"},
{"cos","5"},
{"cot","5"},
{"CUP","5"},
{"DEG","\\degree"},
{"det","5"},
{"DIV","5"},
{"DOT","5{0}"},
{"eTA","5"},
{"ETA","H"},
{"exp","5"},
{"gcd","5"},
{"GGG","\\gg"}, /* no tranlation known */
{"HAT","5{0}"},
{"INF","\\infty"},
{"INT","5"},
{"ISO","?"}, /* no translation known */
{"lim","5"},
{"LLL","\\ll"}, /* no translation known */
{"log","{\\rm log}"},
{"max","5"},
{"MHO","5"},
{"min","5"},
{"mod","{\\rm mod}"},
{"NOT","\\neg"},
{"PHI","6"},
{"PSI","6"},
{"rHO","5"},
{"RHO","P"},
{"sec","5"},
{"SIM","5"},
{"sin","5"},
{"SUB","_{1}"},
{"SUM","5"},
{"SUP","^{1}"},
{"tan","5"},
{"tAU","5"},
{"TAU","T"},
{"TOP","5"},
{"VEC","5{0}"},
{"XOR","?"}, /* no translation known */

/* words with 2 chars */
{"!=","\\neq"},
{"==","\\equiv"},
{"+-","\\pm"},
{"-+","\\mp"},
{"<=","\\le"},
{"<<","\\ll"},
{">=","\\ge"},
{">>","\\gg"},
{"''","{0''}"},
{"IN","5"},
{"ln","5"},
{"mY","\\mu"},
{"MY","M"},
{"nY","\\nu"},
{"NY","N"},
{"OR","\\vee"},
{"PI","6"},
{"TO","^{1}"},
{"XI","6"},
/* 1 char words */
{"^","^{1}"},
{"_","_{1}"},
{"'","{0'}"},
{"`","\\,"},
{"#"," \\\\\n8"},
{"&","&8"},
{"\"","\\dq"}
};
int nFmlTransWords = sizeof(FmlTransTable)/sizeof(FmlTransTable[0]);

/* my own compare function, o= Word from Table, a=formula string*/
/* advantage: compare string a has not to be ended with 0x0 */
int mystrcmp(const char *o, const char *a)
{
    /* if first char of compare word o is lower case, then *
    ** compare cases , else ignore cases                   */

   for ( ; '\0' != *o; o++, a++) {
      if ( islower(*o) ) {
         if (*o != *a) return 1;
      } else {
         if( *o != toupper(*a)) return 1;
      }
   }
   return 0;
}

int SearchTableEntry(char *word)
/* Search an entry in the formula translation table
 * This search was neccessary three times in the original source.
 * To simplify debugging I put it here (AT)
 * --- parameters: ---
 * char *word
 * --- return:     ---
 * int  SearchTableEntry(): number of table entry of -1 if not found
 */
{
   register int i;

   for (i=0; i < nFmlTransWords; i++) {
      if ( 0 == mystrcmp(FmlTransTable[i].szWP, word) ) {
          /* if the first char of word that match is alpha char, then the *
           * char after the word must not be alphanumeric                 */
         if ( isalpha(*word) &&
              isalnum(*(word+strlen(FmlTransTable[i].szWP))))
            continue;
        return i;
      }
   }
   return -1;
}

/* nocasestrongstrcmp gives only 0 if string b is like
** string a, also the same length, but case independent */
int nocasestrongstrcmp(const char *a, const char *b)
{
   for ( ; ((*a)!='\0') && ((*b) != 0) ; a++, b++)
      if ( *a != toupper(*b) )
         return 1;
   return (*a) != (*b);
}

/* because I didn't known how to use the extended char sets in the
** original version , I create my own one */

/* the next array is the entire charset 6 of WordPerfect */
/* unknown translations are translated by "?" */
char *szLEXcharset6[]=
{
"-",							/* 0 */
"\\pm",
"\\le",
"\\ge",
"\\propto",
"/",
"/",
"\\backslash",
"\\div",          /* 8 */
"\\mid",
"\\langle",
"\\rangle",
"\\sim",
"\\approx",
"\\equiv",
"\\in",
"\\cap",          /* 16 */
"\\parallel",
"\\sum",
"\\infty",
"\\neg",
"\\rightarrow",
"\\leftarrow",
"\\uparrow",
"\\downarrow",    /* 24 */
"\\leftrightarrow",
"\\updownarrow",
"?",
"?",
"?",
"?",
"\\cdot",
"\\cdot",         /* 32 */
"\\circ",
"\\circ",
"\\AA",
"\\degrees",
"\\mu",
"-",
"\\times",
"\\int",          /* 40 */
"\\prod",
"\\mp",
"\\nabla",
"\\partial",
"\\prime",
"\\dq",
"\\vec{}",
"{\\cal e}",      /* 48 */
"\\ell",
"\\hbar",
"\\Im",
"\\Re",
"\\wp",
"\\leftrightarrow",
"\\leftrightarrow",
"\\Rightarrow",   /* 56 */
"\\Leftarrow",
"\\Uparrow",
"\\Downarrow",
"\\Leftrightarrow",
"\\Updownarrow",
"\\nearrow",
"\\searrow",
"\\nwarrow",      /* 64 */
"\\swarrow",
"\\cup",
"\\subset",
"\\supset",
"\\subseteq",
"\\supseteq",
"\\ni",
"\\emptyset",     /* 72 */
"\\lceil",
"\\rceil",
"\\lfloor",
"\\rfloor",
"\\ll",
"\\gg",
"\\angle",
"\\otimes",       /* 80 */
"\\oplus",
"\\ominus",
"\\div",
"\\odot",
"\\wedge",
"\\vee",
"?",
"\\top",          /* 88 */
"\\bot",
"\\t{}",
"\\vdash",
"\\dashv",
"\\Box",
"\\Box",
"\\diamondsuit",
"\\diamondsuit",  /* 96 */
"[",
"]",
"\\neq",
"\\not\\equiv",
"?",
"?",
"?",
"\\oint",         /* 104 */
"?",
"?",
"?",
"\\wp",
"\\circ",
"?",
"\\diamond",
"\\star",         /* 112 */
"{'''}",
"\\coprod",
"\\simeq",
"\\cong",
"\\prec",
"\\preceq",
"\\succ",
"\\succeq",       /* 120 */
"\\exists",
"\\forall",
"\\ll",
"\\gg",
"\\uplus",
"?",
"?",
"\\sqcap",        /* 128 */
"\\sqcup",
"\\sqsubset",
"\\sqsubseteq",
"?",
"\\sqsupset",
"\\sqsupseteq",
"?",
"\\bigtriangleup",/* 136 */
"\\bigtriangledown",
"\\triangleleft",
"\\triangleright",
"\\bowtie",
"\\smile",
"\\frown",
"\\bigcirc",
"\\leadsto",      /* 144 */
"\\hookleftarrow",
"\\hookrightarrow",
"\\mapsto",
"\\leftharpoonup",
"\\leftharpoondown",
"\\rightharpoonup",
"\\rightharpoondown",
"\\rightleftharpoons",/* 152 */
"\\rightleftharpoons",
"\\uparrow",
"\\uparrow",
"\\downarrow",
"\\downarrow",
"?",
"?",
"?",              /* 160 */
"?",
"?",
"?",
"?",
"?",
"\\ominus",
"\\mho",
"\\angle",        /* 168 */
"\\angle",
"\\triangleleft",
"\\triangleright",
"\\triangle",
"\\bigtriangledown",
"\\stackrel{.}{+}",
"\\stackrel{.}{+}",
"?",              /* 176 */
"?",
"?",
"\\asymp",
"\\models",
"?",
"?",
"\\wr",
"\\star",         /* 184 */
"\\not<",
"\\not\\le",
"\\not>",
"\\not\\ge",
"\\not\\sim",
"\\not\\simeq",
"\\not\\cong",
"\\not\\approx",  /* 192 */
"\\not\\prec",
"\\not\\preceq",
"\\not\\succ",
"\\not\\succeq",
"\\not\\subset",
"\\not\\supset",
"\\not\\subseteq",
"\\not\\supseteq",/* 200 */
"?",
"?",
"\\not\\sqsubseteq",
"\\not\\sqsupseteq",
"?",
"?",
"\\not\\asymp",
"?",              /* 208 */
"?",
"?",
"?",
"{\\cal F}",
"{\\rm C}",
"{\\rm I}",
"{\\rm N}",
"{\\rm R}",       /* 216 */
"?",
"?",
"\\ni",
"\\cdots",
"\\ldots",
"\\vdots",
"\\ddots",
"?",              /* 224 */
"?",
"+",
"-",
"=",
"\\ast",
"/",
"?",
"?",              /* 232 */
"{\\cal H}",
"{\\cal P}"       /* 234 */
};
int nLEXcharset6 = sizeof(szLEXcharset6)/sizeof(szLEXcharset6[0]);

/* because I found no better way, I must use a trick to get some
** line feed in long lines, I copy in LINK->by a 0x0d (soft return), and
** call the function Return_Page
** if there is a better way, please correct me
*/
struct LOC_Convert_first_pass *privFormulaLINK;

unsigned char *szExplodedFml;  /* string, stores exploded words in sequential order (0x0 between every word) */

unsigned char *szTranslatedFml; /* string, stores the translated formula string */
int  nTranslatedFmlPtr;        /* current position in translated formula string */
int  nTranslatedFmlNewLinePos; /* last position of a newline in the string */
int  nTranslatedFmlLength;     /* allocated length of translated formula string */
int  bEquationEnvironment;     /* true if no text style formula, but an equation */
int  bIsEquationArray;         /* TRUE if the formula is most likely an equation array */

struct tagLevelWordPos{
	int nStrgPos; /* position in exploded formula string */
	int nLevel;   /* level, i.e. number of open curly braces */
	};
struct tagLevelWordPos *ExplodedWords;  /* array, stores position and level of every exploded Words */
int nFmlWords; /* Number of Exploded Words */


/* This routine splits a formula string in many small words
** according to the syntax of WordPerfect */
void ExplodeFormula(unsigned char *szFml,int nFmlLength)
{
   int i, j, c, nWordLength, nExplodeWord=0, nCurrSourceStrgPos=0,
       nCurrStrgPos = 0, nCurrLevel = 0;

    /* read one char and grab special chars */

    while( (nCurrSourceStrgPos < nFmlLength) && 
           (0 != (c=szFml[nCurrSourceStrgPos++])) ) {
       switch(c) {
          case ' ' :
          case 0x08: break;
          case 0x0a:
          case 0x0d: break;
          case '{' : ExplodedWords[nExplodeWord].nLevel     = nCurrLevel;
                     ExplodedWords[nExplodeWord++].nStrgPos = nCurrStrgPos;
                     szExplodedFml[nCurrStrgPos++]          = '\0';
                     szExplodedFml[nCurrStrgPos++]          = '\0';
                     nCurrLevel++;
                     break;
          case '}' : nCurrLevel--;
                     ExplodedWords[nExplodeWord].nLevel     = nCurrLevel;
                     ExplodedWords[nExplodeWord++].nStrgPos = nCurrStrgPos;
                     szExplodedFml[nCurrStrgPos++]          = '\0';
                     szExplodedFml[nCurrStrgPos++]          = '\0';
                     break;
          case '\\': if ( szFml[nCurrSourceStrgPos] != '\0' ) {
                        ExplodedWords[nExplodeWord].nLevel     = nCurrLevel;
                        ExplodedWords[nExplodeWord++].nStrgPos = nCurrStrgPos;
                        szExplodedFml[nCurrStrgPos++]          = szFml[nCurrSourceStrgPos++];
                        szExplodedFml[nCurrStrgPos++]          = '\0';
                     }
                     break;
          case 0xc0: /* special char, next byte is charnumber, next byte charsetnumber */
                     ExplodedWords[nExplodeWord].nLevel     = nCurrLevel;
                     ExplodedWords[nExplodeWord++].nStrgPos = nCurrStrgPos;
                     szExplodedFml[nCurrStrgPos++]          = 0xc0;
                     szExplodedFml[nCurrStrgPos++]          = szFml[nCurrSourceStrgPos++]; /* char position number */
                     szExplodedFml[nCurrStrgPos++]          = szFml[nCurrSourceStrgPos++]; /* char set number */
                     szExplodedFml[nCurrStrgPos++]          = szFml[nCurrSourceStrgPos++]; /* yet another 0xc0 */
                     szExplodedFml[nCurrStrgPos++]          = '\0';
                     break;
          case 0xa9: /* special minus char of WordPerfect */
          case 0xaa: /* further special minus char at line end (AT) */
                     ExplodedWords[nExplodeWord].nLevel     = nCurrLevel;
                     ExplodedWords[nExplodeWord++].nStrgPos = nCurrStrgPos;
                     szExplodedFml[nCurrStrgPos++]          = '-';
                     szExplodedFml[nCurrStrgPos++]          = '\0';
                     break;
          default:   /* here is the place to search for keywords of WordPerfect */
                     nCurrSourceStrgPos--;
                     if ( (i = SearchTableEntry(szFml+nCurrSourceStrgPos)) > -1 ) {
                        /* found word in table */
                        ExplodedWords[nExplodeWord].nLevel     = nCurrLevel;
                        ExplodedWords[nExplodeWord++].nStrgPos = nCurrStrgPos;
                        nWordLength = strlen(FmlTransTable[i].szWP);
                        for (j = 0; j < nWordLength; j++)
                           szExplodedFml[nCurrStrgPos++] = szFml[nCurrSourceStrgPos++];
                        szExplodedFml[nCurrStrgPos++]          = '\0';
                        break;
                     } 
                     /* if word not found in table */
                     ExplodedWords[nExplodeWord].nLevel     = nCurrLevel;
                     ExplodedWords[nExplodeWord++].nStrgPos = nCurrStrgPos;

                     /* if first char is letter, then copy all letters and digits */
                     if ( isalpha(szFml[nCurrSourceStrgPos]) ) {
                        while ( (szFml[nCurrSourceStrgPos] != '\0') &&
                                (isalnum(szFml[nCurrSourceStrgPos])) )
                           szExplodedFml[nCurrStrgPos++] = szFml[nCurrSourceStrgPos++];
                        szExplodedFml[nCurrStrgPos++] = '\0';
                     } /* end if isalpha(c) */
                     /* if first char is a digit, copy only all digits */
                     else if ( isdigit(szFml[nCurrSourceStrgPos]) ) {
                        while ( (szFml[nCurrSourceStrgPos]!='\0') &&
                               (isdigit(szFml[nCurrSourceStrgPos])) )
                           szExplodedFml[nCurrStrgPos++] = szFml[nCurrSourceStrgPos++];
                        szExplodedFml[nCurrStrgPos++] = '\0';
                     } else {
                        szExplodedFml[nCurrStrgPos++] = szFml[nCurrSourceStrgPos++];
			szExplodedFml[nCurrStrgPos++] = '\0';
                     }
       } /* End of switch statement */
    } /* End of While */
    nFmlWords = nExplodeWord;
} /* End of Function ExplodeFormula */


/* FindNextArgument gives the Position of the next term at the same level */
int FindNextArgument(int *isNewLevel, int nCurrLevel, int nCurrPos,int nStep)
{
#ifdef _DebugFormulaParsing
#define FNA_error() {\
                       fprintf(stderr, "Formula %i: Parsing error due to lazy bracketing (line %i)!\n", \
                               FormulaNumber, __LINE__); \
                       return -1; \
                    }
#else
#define FNA_error() {\
                       fprintf(stderr, "Formula %i: Parsing error due to lazy bracketing - see file readme.\n", \
                               FormulaNumber); \
                       return -1; \
                    }
#endif

    int i, nRetVal = nCurrPos;

	#ifdef _DebugFormulaParsing
	printf("FindNextArg: Level=%d, CurPos=%d, Step=%d",nCurrLevel,nCurrPos,nStep);
	#endif
	if(nStep==0)
		{
		/* overread spaces at the same level */
		while(   (szExplodedFml[ExplodedWords[nRetVal].nStrgPos]=='\0')
					&& (ExplodedWords[nRetVal].nLevel==nCurrLevel))
			{
			nRetVal++;
            if(nRetVal>=nFmlWords) FNA_error();
			}
        if(ExplodedWords[nRetVal].nLevel<nCurrLevel) FNA_error();
		*isNewLevel = ( ExplodedWords[nRetVal].nLevel>nCurrLevel );
		}

	else if(nStep>0)
		{
		for(i=0;i<nStep;i++)
			{
			if(ExplodedWords[nRetVal].nLevel>nCurrLevel)
				{
				/* read until nCurrLevel==Level */
				do nRetVal++; while(ExplodedWords[nRetVal].nLevel>nCurrLevel);
				}
			else /* skip only one Word */
				{
				nRetVal++;
                if(nRetVal >= nFmlWords) FNA_error();
				}
            /* overread spaces at the same level */
			while(   (szExplodedFml[ExplodedWords[nRetVal].nStrgPos]=='\0')
						&& (ExplodedWords[nRetVal].nLevel==nCurrLevel))
			{
			nRetVal++;
            if(nRetVal>=nFmlWords) FNA_error();
			}
            if(ExplodedWords[nRetVal].nLevel<nCurrLevel) FNA_error();
            *isNewLevel = ( ExplodedWords[nRetVal].nLevel>nCurrLevel );
			} /*for*/
		} /* if nStep >=0 */

	else /* get previous argument */
		{
		/* skip whitespaces at the same level */
		while(   (szExplodedFml[ExplodedWords[nRetVal-1].nStrgPos]=='\0')
					&& (ExplodedWords[nRetVal-1].nLevel==nCurrLevel))
		{
		nRetVal--;
        if(nRetVal<0) FNA_error();
		}
		/* if previous argument isnt space, take it */
		if(  (szExplodedFml[ExplodedWords[nRetVal-1].nStrgPos]!='\0')
			 && (ExplodedWords[nRetVal-1].nLevel==nCurrLevel))
			{
			nRetVal--;
            if(nRetVal<0) FNA_error();
			*isNewLevel=0;
			}
		else
			{
			*isNewLevel=1;
			/* search backward until there is a transition down to the current level */
			/* or nRetVal=0 */
			for(nRetVal--;;nRetVal--)
				{
                if(nRetVal<0) FNA_error();
				if(nRetVal==0)
					break;
				if((ExplodedWords[nRetVal].nLevel>nCurrLevel) && (ExplodedWords[nRetVal-1].nLevel==nCurrLevel))
					break;
				} /* forall */
			} /* else (no simple argument) */
		} /* else (nStep is negative) */
	#ifdef _DebugFormulaParsing
	printf(" |-> RetVal=%d, IsNewLev=%d\n",nRetVal,*isNewLevel);
	#endif
    return nRetVal;
#undef FNA_error()
}

void EmitFmlChar(char o);

int bBeginFmlWord=1;

void BeginFmlWord(void)
	{
	bBeginFmlWord=1;
	if((nTranslatedFmlPtr-nTranslatedFmlNewLinePos)>60)
		EmitFmlChar('\n');
	}

void EmitFmlChar(char o)
	{
	char p;
	int nNewAlloc;
	char *szNewBlock;

	if((nTranslatedFmlPtr+8)>=nTranslatedFmlLength)
		{
		nNewAlloc = 1024 + nTranslatedFmlLength;
		szNewBlock = realloc(szTranslatedFml,nNewAlloc);
		if(szNewBlock==NULL)
			return;
		szTranslatedFml = szNewBlock;
		nTranslatedFmlLength = nNewAlloc;
		}


	if('\n'==o)
		nTranslatedFmlNewLinePos = nTranslatedFmlPtr;

	if(bBeginFmlWord && (0!=nTranslatedFmlPtr))
		{
		/* get previous char */
		p = szTranslatedFml[nTranslatedFmlPtr-1];
		if(isalnum(p) && isalnum(o))
			szTranslatedFml[nTranslatedFmlPtr++] = ' ';
		}
	szTranslatedFml[nTranslatedFmlPtr++] = o;
	bBeginFmlWord=0;

	return;
	}


void EmitFmlString(char *szT)
	{
	for(;'\0'!=*szT;szT++)
		EmitFmlChar(*szT);
	}

void EmitFmlWord(int nArg)
	{
	BeginFmlWord();
	EmitFmlString(szExplodedFml + ExplodedWords[nArg].nStrgPos);
	}


void EmitFmlLower(char *szT)
	{
	for(;'\0'!=*szT;szT++)
		EmitFmlChar(tolower(*szT));
	}

void EmitFmlCase(char *szT,char o)
	{
	if(islower(o))
		EmitFmlChar(tolower(*szT));
	else
		EmitFmlChar(toupper(*szT));

	EmitFmlLower(szT+1);

	}

int TranslateFormula( int nCurrWord, int bTermInCurlyBraces);


int IsSpecialWordWithPreviousArg(int nCurrWord)
{
    int  i;
	char *szTl;

	int RetVal=0; /* default no such word */

/* To prevent a crash caused by SIGSEGV I insert a very very primitive *
 * test. (AT)                                                          *
 * The sense of this function is to look, whether the searched string  *
 * contains '0'. But if there isn't any string, which is the case when *
 * nCurrWord == -1 (That means ANYWHERE in the computer memory!!!!)    *
 * it makes no sense to look for. So avoid testing AND CHRASHING!      *
 */
    if ( nCurrWord < 0 ) return 0;

    #ifdef _DebugFormulaParsing
    printf("IsPrevArgWord, nCurrWord=%d (%s)",nCurrWord,szExplodedFml+ExplodedWords[nCurrWord].nStrgPos);
	#endif
	/* first looking for WordPerfect Key-Words */
    if ( (i = SearchTableEntry(szExplodedFml+ExplodedWords[nCurrWord].nStrgPos))
         > 1 ) {
          /* there are special meanings of translated Word */
          /* 0 means Keyword with prepending \ */
          szTl = FmlTransTable[i].szTeX;
          /* look for a '0' in the string */
          for ( ; '\0' != *szTl; szTl++)
             if ('0' == *szTl) return 1;
    }
	#ifdef _DebugFormulaParsing
	printf("|-> %d\n",RetVal);
	#endif
    return RetVal; /* not found */
}

int CountMaxColumns(int nCurrLevel, int nCurrWord, int *nCols)
	{
	int MaxAmpersands, CurrAmpersands, i, nNextLevel;
	nNextLevel = nCurrLevel + 1;

	#ifdef _DebugFormulaParsing
	printf("CountMaxColumns: nLevel=%d, nWord=%d ",nCurrLevel,nCurrWord);
	#endif

	/* makes only sense if word after current word is a level
	** higher, i.e. a curly bracket, ergo check this first */
	if(nNextLevel != ExplodedWords[nCurrWord+2].nLevel)
		{
		*nCols=0;
		#ifdef _DebugFormulaParsing
		printf(" ->Error (Level)\n");
		#endif

		return 1; /* error */
		}
	/* strategie: look on all words which are on the same level
	** and count the ampersands (&), reset counter if find an
	** "#", store the max number of ampersands */
	MaxAmpersands = 0;
	CurrAmpersands = 0;
	for(i=nCurrWord+2;i<nFmlWords;i++)
		{
		if(ExplodedWords[i].nLevel < nNextLevel)
			break; /* end of search if lower the current level+1 */

		if(ExplodedWords[i].nLevel > nNextLevel)
			continue; /* ignore Words, which lies on an higher level */

		if(szExplodedFml[ExplodedWords[i].nStrgPos]=='&')
			{
			/* if Ampersand found, increment CurrAmpersand Counter */
			CurrAmpersands++;
			continue;
			}
		if(szExplodedFml[ExplodedWords[i].nStrgPos]=='#')
			{
			/* if # found (new line) reset current ampersand counter,
			** update max ampersands */
			if(MaxAmpersands<CurrAmpersands) MaxAmpersands=CurrAmpersands;
			CurrAmpersands=0;
			}
		} /* for all words */
	/* update max ampersands */
	if(MaxAmpersands<CurrAmpersands) MaxAmpersands=CurrAmpersands;
	*nCols = MaxAmpersands+1;
	#ifdef _DebugFormulaParsing
	printf(" #Columns=%d\n",*nCols);
	#endif

	return 0; /* no errors */
	}




void TranslateFmlWord(int nArg)
	{
	int nCharPos;
	int nCharSet;
	unsigned char cBeg;
	BeginFmlWord();
	/* is special char from charset */
	cBeg = szExplodedFml[ExplodedWords[nArg].nStrgPos];

	if(0xc0==cBeg)
		{
		nCharPos = szExplodedFml[ExplodedWords[nArg].nStrgPos+1];
		nCharSet = szExplodedFml[ExplodedWords[nArg].nStrgPos+2];
		if(6==nCharSet)
			{
			if(nCharPos<nLEXcharset6)
				EmitFmlString(szLEXcharset6[nCharPos]);
			else
				EmitFmlString("?");
			}
		else /* other charsets but 6 not yet implemented */
			{
			EmitFmlString("?");
			}
		}
	else if(('%'==cBeg)||('$'==cBeg)) /* these chars have special meaning in Latex, but not in WP */
		{
		EmitFmlChar('\\');
		EmitFmlChar(cBeg);
		}
	else /* no special character */
		EmitFmlString(szExplodedFml + ExplodedWords[nArg].nStrgPos);
	}





/* IsSpecialWordWritten translate one Word and it arguments *
** nCurrLevel : current Level (of curly braces)
** nCurrWord:   Number of first word to parse
** nJumpOver: gives number of words one to have jump over, because
** the current word needs argument(s)
** ReturnValue: !=0 if there exists a translation of current word, 0 else
*/
int IsSpecialWordWritten(int nCurrLevel, int nCurrWord, int *nJumpOver)
	{
	int i,j,col,nColumns;
	char *szTl,*szCurrWord;
	int nNextArg,isNewLevel;
	int nDummyOver;

	*nJumpOver = 0; /* default no arguments */

	#ifdef _DebugFormulaParsing
	printf("Eintritt IsSpecialWordWritten, nCurrWord=%d ",nCurrWord);
	#endif
	/* first looking for WordPerfect Key-Words */
	szCurrWord = szExplodedFml + ExplodedWords[nCurrWord].nStrgPos;

    if ( (i = SearchTableEntry(szCurrWord)) > 1 ) {
       BeginFmlWord();
       /* there are special meanings of translated Word */
       /* 0 means Keyword with prepending \ */
       szTl = FmlTransTable[i].szTeX;
       for(j=0;szTl[j]!='\0';j++)
           {
           if('5'==szTl[j])
               {
               EmitFmlChar('\\'); EmitFmlLower(FmlTransTable[i].szWP);
               continue;
               }
           if('6'==szTl[j])
               {
               EmitFmlChar('\\'); EmitFmlCase(FmlTransTable[i].szWP,szCurrWord[0]);
               continue;
               }
           if('7'==szTl[j])
               {
               CountMaxColumns(nCurrLevel,nCurrWord,&nColumns);
               /* emit nColumn times the char after '7' to format
               ** the columns */
               for(col=1;col<nColumns;col++)
                   EmitFmlChar(szTl[j+1]);
               continue;
               }
           if('8'==szTl[j])
               /* to support equation arrays, I use the array environment in
               connection with \displaystyle for every term in the table */
               {
               if(bIsEquationArray && (nCurrLevel<=1))
                   EmitFmlString("\\displaystyle ");
               continue;
               }
           if('0'==szTl[j])
               {
               nNextArg = FindNextArgument(&isNewLevel,nCurrLevel,nCurrWord,-1);
               if(nNextArg<0)
                   return -1;
               if(isNewLevel)
                   TranslateFormula(nNextArg,0);
               else
                   IsSpecialWordWritten(nCurrLevel,nNextArg,&nDummyOver);
               continue; /* next char */
               }
           if('1'==szTl[j])
               {
               nDummyOver=0;
               nNextArg = FindNextArgument(&isNewLevel,nCurrLevel,nCurrWord,1);
               if(nNextArg<0)
                   return -1;
               if(isNewLevel)
                   TranslateFormula(nNextArg,0);
               else
                   IsSpecialWordWritten(nCurrLevel,nNextArg,&nDummyOver);
               if(*nJumpOver<(1+nDummyOver)) *nJumpOver = 1+nDummyOver;
               continue;
               }
           if('2'==szTl[j])
               {
               nDummyOver=0;
               nNextArg = FindNextArgument(&isNewLevel,nCurrLevel,nCurrWord,2);
               if(nNextArg<0)
                   return -1;
               if(isNewLevel)
                   TranslateFormula(nNextArg,0);
               else
                   IsSpecialWordWritten(nCurrLevel,nNextArg,&nDummyOver);
               if(*nJumpOver<(2+nDummyOver)) *nJumpOver = 2+nDummyOver;
               continue;
               }
           /* if char has no special meaning, just emit */
           EmitFmlChar(szTl[j]);
           } /* for all chars in translated word */
       #ifdef _DebugFormulaParsing
       printf("| 1 JO=%d\n",*nJumpOver);
       #endif
       return 1;
    } /* if word found in Translation Table */
	#ifdef _DebugFormulaParsing
	printf("| 0 JO=%d\n",*nJumpOver);
	#endif

	/* if no special meaning found, translate at least special chars */
	TranslateFmlWord(nCurrWord);
	return 0; /* not found */
	}


/* CheckForEquationArray determines, wether or not the current
** formula seems to be an equation array
** privately I commonly use stackalign to align the formulas
** I hope there is no other way to do this */

void CheckForEquationArray(void)
	{
	int isCompound;
	bIsEquationArray=0; /* per default no equation array */
	/* nesseccary for equation array is: i) first word is stackalign
	** ii) third word is non existent (no words after the end
	** of stackalign */
	if(!bEquationEnvironment) /* preliminarity is equation environment */
		return;

	if(1==nocasestrongstrcmp("STACKALIGN",szExplodedFml))
		return;

	if(-1!=FindNextArgument(&isCompound,0,0,2))
		return;

	bIsEquationArray = 1;
	}

/* the next function is (indirectly) recursively defined */
int TranslateFormula(int nCurrWord, int bTermInCurlyBraces)
{
    int err,isCurrWordCompound,isNextWordCompound,nNextWord;
	int nJumpOver = 0;
	int nCurrLevel = ExplodedWords[nCurrWord].nLevel;

    if(nCurrWord==0)
		CheckForEquationArray();

	#ifdef _DebugFormulaParsing
	printf("Eintritt TranslateFormula, nCurrWord=%d\n",nCurrWord);
	#endif
    if(bTermInCurlyBraces)
		EmitFmlChar('{');

	/* first skip white spaces at the same level */
	nCurrWord = FindNextArgument(&isCurrWordCompound,nCurrLevel,nCurrWord,0);
    for ( ; (nCurrWord < nFmlWords) && (-1 != nCurrWord);
          nCurrWord = FindNextArgument(&isCurrWordCompound,nCurrLevel,nCurrWord,1+nJumpOver)
        )
		{
		#ifdef _DebugFormulaParsing
		printf("Translate w=%d, isNewLevel=%d\n",nCurrWord,isCurrWordCompound);
		#endif

		nJumpOver=0; /* default no other arguments */
		nNextWord = FindNextArgument(&isNextWordCompound,nCurrLevel,nCurrWord,1);
        if(!isNextWordCompound && IsSpecialWordWithPreviousArg(nNextWord))
			{
            err = IsSpecialWordWritten(nCurrLevel,nNextWord, &nJumpOver);
			if(err<=0)
				goto TF_Error;
			nJumpOver++; /* because it is not the current word, but the next */
			continue;
			}
        if(isCurrWordCompound)
			{
            TranslateFormula(nCurrWord,1);
			}
		else
			{
            err = IsSpecialWordWritten(nCurrLevel,nCurrWord,&nJumpOver);
			if(err<0)
				goto TF_Error;
			}
		} /* for-loop */

    if(bTermInCurlyBraces)
		EmitFmlChar('}');

    #ifdef _DebugFormulaParsing
	printf("Ende TranslateFormula *** alles ok ***\n");
	#endif
	return 0;

	TF_Error:
	#ifdef _DebugFormulaParsing
	printf("Ende TranslateFormula !!! Fehler !!! \n");
	#endif
	return -1;
}




int ReadTranslateFmlFromFile(FILE *wpd, int nFmlLength,struct LOC_Convert_first_pass *LINK)
{
    int           i, z, err;
	unsigned char *szOriginalFml;
    extern   FILE *strip;

    privFormulaLINK = LINK; /* is global for this translation, see definition for more info */
	privFormulaLINK->by = 0x0d; /* soft return */
	/* first step: reserve Buffers */
    szOriginalFml = malloc(sizeof(char)*(nFmlLength+1));
	if(NULL==szOriginalFml) return 1;

    ExplodedWords = malloc(sizeof(struct tagLevelWordPos)*(nFmlLength+1));
	if(NULL==ExplodedWords) return 1;

    szExplodedFml = malloc(sizeof(char)*(2*nFmlLength+1));
	if(NULL==szExplodedFml) return 1;

    szTranslatedFml = malloc(1024);
	nTranslatedFmlLength = 1024;
	if(NULL==szTranslatedFml) return 1;

    /* init variables */
	ExplodedWords[0].nLevel=0;
	ExplodedWords[0].nStrgPos=0;
	nTranslatedFmlPtr=0;
	nTranslatedFmlNewLinePos = 0;
	nFmlWords=0;

    /* read Formula in */
	for(i = 0;i<nFmlLength;i++)
		szOriginalFml[i] = fgetc(wpd);
	szOriginalFml[i] = '\0';

    /* TranslateFormula */
	ExplodeFormula(szOriginalFml,nFmlLength);
    err = TranslateFormula(0,0);
    szTranslatedFml[nTranslatedFmlPtr++]='\0';

    /* write Formula to output */
    for (i = 0, z = 0; i < nTranslatedFmlPtr; i++) {
       if ( szTranslatedFml[i] == '\n' ) {
          szTranslatedFml[i] = '\0';
          fputs(szTranslatedFml+z, strip);
          #ifdef _DebugFormulaParsing
          printf(szTranslatedFml+z);
          #endif
          Return_Page(privFormulaLINK);
          z = i + 1;
       }
    }
    fputs(szTranslatedFml+z, strip);
    #ifdef _DebugFormulaParsing
    printf(szTranslatedFml+z);
    printf("\n");
    #endif
     /* free reserved memory blocks */
    free(szTranslatedFml);
    free(szExplodedFml);
    free(ExplodedWords);
    free(szOriginalFml);

    return err;
}

void ExtractMathematicalFormula(struct LOC_Convert_first_pass *LINK)
{
    unsigned long  PositionAfterFormula, CurrentFilePosition, NextPos;
    unsigned short w, wFmlOffset, wFmlLength;
    unsigned char  TypeOfBox, BoxPositioning, lineterm;
    long     TableCurrPos;
    int      bBoxIsInLine;

    FormulaNumber++;
    Rd_word(wpd, &w);   /* read total length of box */
    #ifdef _DebugFormulaParsing
    printf("TotLength of Fml: %x\n",w);
    #endif

    CurrentFilePosition  = (ftell(wpd) / sizeof(uchar)) * sizeof(uchar);
    PositionAfterFormula = (ftell(wpd) / sizeof(uchar) + w) * sizeof(uchar);
    #ifdef _DebugFormulaParsing
    printf("Pos. nach Formel: %lx\n",PositionAfterFormula);
    #endif
    NextPos = CurrentFilePosition + 2;
    if ( (NextPos+1) > PositionAfterFormula ) goto skip_over_Box;
    fseek(wpd, NextPos, SEEK_SET);
    BoxPositioning = getc(wpd);
    bBoxIsInLine = ( (BoxPositioning & 0x03)==0x02);

    #ifdef _DebugFormulaParsing
    printf("BoxPosByte: %x\n",BoxPositioning&0x03);
    #endif
	/* seek that byte, which tells about the kind of contents of box
	** 0x08 means formula content */
    NextPos = CurrentFilePosition + 0x30;
    if ( (NextPos+1) > PositionAfterFormula ) goto skip_over_Box;
    fseek(wpd,NextPos, SEEK_SET);
    TypeOfBox = getc(wpd);
    #ifdef _DebugFormulaParsing
    printf("Type of Box: %x\n",TypeOfBox);
    #endif
    if( TypeOfBox != 0x8 ) goto skip_over_Box; /* Box is not of the type formula */

	/* seek that Word , which tells about the position of beginning of formula text
	** (more accurately: the position of word, that gives the length of formula text;
	** formula text itself follows thereafter) */
    NextPos = CurrentFilePosition + 0x73;
    if ( (NextPos+2) > PositionAfterFormula ) goto skip_over_Box;
    fseek(wpd,NextPos,SEEK_SET);
    Rd_word(wpd, &wFmlOffset);
    #ifdef _DebugFormulaParsing
    printf("FmlOffset: %x\n",wFmlOffset);
    #endif

    /* seek word, which gives length of formula text */
    NextPos = CurrentFilePosition + 0x7fL + wFmlOffset;
    if ( (NextPos+2) > PositionAfterFormula ) goto skip_over_Box;

    fseek(wpd,NextPos,SEEK_SET);

    Rd_word(wpd, &wFmlLength);
    #ifdef _DebugFormulaParsing
    printf("FmlLength: %x\n",wFmlLength);
    #endif

	/* check wether or not formula extends box length */
    if ( (NextPos+2+wFmlLength) > PositionAfterFormula ) goto skip_over_Box;

	/* choose wether formula should be in a equation environment
	** or simply a text style formula */
    bEquationEnvironment = 1; /* per default equation environment */
    if ( bBoxIsInLine )	{
       if ( LINK->char_on_line ) bEquationEnvironment = 0;
       else {  /* check if previous lineskip was a soft newline *
                * if true, it is a text style formula           */
          TableCurrPos = ftell(tabel);
          if ( TableCurrPos >= 2 )
          fseek(tabel, TableCurrPos-2, SEEK_SET);
          lineterm = getc(tabel);
          fseek(tabel, TableCurrPos, SEEK_SET);
          if ( lineterm == 'r' ) bEquationEnvironment = 0;
       }
    }

     /* the following chars belongs to the formula text */
    if ( bEquationEnvironment ) {
       LINK->by = 0x0d;
       fputs("\\begin{equation}", strip);
       Return_Page(LINK);
    } else 
       putc('$', strip);

    ReadTranslateFmlFromFile(wpd,wFmlLength,LINK);

    if ( bEquationEnvironment ) {
       LINK->by = 0x0d;
       Return_Page(LINK);
       fputs("\\end{equation}", strip);
    } else 
       putc('$', strip);

    skip_over_Box:
    fseek(wpd, PositionAfterFormula , 0);
}

