/***************************************************************************
** Although considerable effort has been expended to make this software   **
** correct and reliable, no warranty is implied; the author disclaims any **
** obligation or liability for damages, including but not limited to      **
** special, indirect, or consequential damages arising out of or in       **
** connection with the use or performance of this software.               **
***************************************************************************/

/*
 *	This is the Tektronix 4014 low-level driver.
 */

#include "debug.h"
#include "types.h"
#include "font.h"
#include "drivutil.h"
#include "tk.h"

#include "font.p"
#include "graph.p"
#include "raster.p"

/*
 *	The Device control identifier and device name. The device
 *	name is declared this way in order for it to be changed by
 *	patching the image:
 */

char *TK_Device;
char TK_Device_Name[32] = "DVI_TEK4014";

/*
 *	Current beam position and the 5 bytes defining the current
 *	position:
 */

struct Tek_Position {
	unsigned short CurX;
	unsigned short CurY;
	unsigned char HiY;
	unsigned char ExB;
	unsigned char LoY;
	unsigned char HiX;
	unsigned char LoX;
} TK_Current_Position, TK_Pending_Position;

unsigned char TK_Mode;		/* Mode of the terminal */
#define GRAPH_MODE GS
#define ALPHA_MODE US
#define POINT_MODE FS
#define INCR_MODE  RS

unsigned char TK_Pen_Mode;	/* Incremental pen mode (up or down) */

struct Char_Entry {
	struct Char_Entry *Link;	/* Pointer to next one */
	unsigned short Type;		/* 0: Horizontal  != 0: Vertical */
	unsigned short N_Segments;	/* Number of line segments */
	unsigned short Segment[][3];	/* Array of line segment descriptors */
} **TK_Font_Directory;			/* Pointer to font directory */
unsigned long TK_N_Fonts;		/* Copy of number of fonts */
unsigned short TK_Copies;		/* Number of copies to produce */
unsigned short TK_Max_Dev_X, TK_Max_Dev_Y; /* Page bounds, pixels */
unsigned long TK_Max_X, TK_Max_Y;	/* Page bounds, 1/1024 pixels */
struct { short a, b, c, d, e, f; } TK_Xform;/* Coordinate transformation */

#define Round_Pixel(size) (((size) + ((size) >= 0 ? 512 : -512)) >> 10)
#define Trunc_Pixel(size) ((size) >> 10)
#define Ceil_Pixel(size) (((size) + ((size) >= 0 ? 1023 : -1023)) >> 10)
#define MAX_WIDTH 3
#define MAX_TEMP_FONTS 25

int TK_Set_Layout (Widest_Width, Tallest_Height, Layout_Mode, Two_Sided,
		   N_Fonts, Max_Size, Max_Primitives, Memory_Size, Max_Fonts,
		   Resolution, Page_Size, Copies, Input_Name, Output_Name)
unsigned long Widest_Width, Tallest_Height, N_Fonts, Max_Size,
	      Max_Primitives, Memory_Size, Max_Fonts, Layout_Mode,
	      Two_Sided;
struct Ratio *Resolution, Page_Size[2];
unsigned short Copies;
char *Input_Name, *Output_Name;
{
	auto   unsigned int Index;
	static unsigned long Status;
	extern char *Connect_Device();
	extern int Is_a_Terminal();
	extern char *Mem_Alloc();
/*
 *	Connect the terminal, check if it is a terminal:
 */
	if ((TK_Device = Connect_Device ((Output_Name == 0) ? TK_Device_Name : Output_Name,
					   1000, 2, &Status)) == 0)
		return (0);
/*	if (Is_a_Terminal (TK_Device) == 0) {
		Disconnect_Device (TK_Device);
		return (0);
	}	*/
/*
 *	Allocate storage for font directory:
 */
	TK_N_Fonts = N_Fonts + MAX_TEMP_FONTS;
	TK_Font_Directory = (struct Char_Entry **) Mem_Alloc (TK_N_Fonts * sizeof (struct Char_Entry *));
	for (Index = 0; Index < TK_N_Fonts; Index++)
		TK_Font_Directory[Index] = 0;
/*
 *	Set up transformation matrix to convert DVI device coordinates into
 *	actual Tektronix coordinates. DVIOUT knows the resolution of the
 *	device, but does not know the layout of the screen area. Additionally,
 *	landscape mode turns things sideways, requiring a different
 *	transformation in that mode.
 *
 *		   Normal			 Landscape
 *
 *	    |   0    1    0   |              |   1   0   0   |
 *	T = |   1    0    0   |          T = |   0  -1   0   |
 *	    |   0    0    1   |              |   0  3120 1   |
 */
	if (Layout_Mode == 0) {	/* Normal */
		TK_Xform.a = 0; TK_Xform.b = 1; TK_Xform.c = 1; TK_Xform.d = 0;
		TK_Xform.e = 0; TK_Xform.f = 0;
		TK_Max_Dev_X = 3120; TK_Max_Dev_Y = 4096;
	} else {		/* Landscape */
		TK_Xform.a = 1; TK_Xform.b = 0; TK_Xform.c = 0; TK_Xform.d = -1;
		TK_Xform.e = 0; TK_Xform.f = 3120;
		TK_Max_Dev_X = 4096; TK_Max_Dev_Y = 3120;
	}
	TK_Max_X = TK_Max_Dev_X << 10;
	TK_Max_Y = TK_Max_Dev_Y << 10;
/*
 *	Set number of copies, current graphics state:
 */
	TK_Copies = Copies;
	Init_GState (&Current_State, 1);
	return (1);
}

TK_Terminate ()
{
	auto   struct Char_Entry *Char_Ptr;
	auto   unsigned int Index;
	extern unsigned long Flush_Device_IO();
/*
 *	Release all memory:
 */
	for (Index = 0; Index < TK_N_Fonts; Index++)
	while ((Char_Ptr = TK_Font_Directory[Index]) != 0) {
		TK_Font_Directory[Index] = Char_Ptr->Link;
		Mem_Free (Char_Ptr);
	}
	Mem_Free (TK_Font_Directory);
	TK_Font_Directory = 0;
/*
 *	Disconnect device:
 */
	Flush_Device_IO (TK_Device);
	Wait_for_Device_IO (TK_Device);
	Disconnect_Device (TK_Device);
}

TK_Download_Font (Font_Ptr, Char_Vector, N_Chars)
struct Font_Definition *Font_Ptr;
struct Char_Definition **Char_Vector;
unsigned int N_Chars;
{
	auto   struct Char_Entry *Tek_Char_Ptr;
	auto   unsigned short (*Scratch_H_Lines)[3], (*Scratch_V_Lines)[3];
	auto   unsigned char *Scratch_Raster;
	auto   unsigned int Index;
	extern struct Char_Entry *TK_Download_Char();
	extern char *Mem_Alloc();
/*
 *	Check maximum font index:
 */
	if (Font_Ptr->Font_Index >= TK_N_Fonts) {
		for (Index = 0; Index < N_Chars; Index++)
			Char_Vector[Index]->Driver_Id = 0;
		return;
	}
/*
 *	Release previous font, if any, using this slot:
 */
	while ((Tek_Char_Ptr = TK_Font_Directory[Font_Ptr->Font_Index]) != 0) {
		TK_Font_Directory[Index] = Tek_Char_Ptr->Link;
		Mem_Free (Tek_Char_Ptr);
	}
/*
 *	Allocate scratch storage for raster manipulation:
 */
	Scratch_Raster = (unsigned char *) Mem_Alloc (Font_Ptr->Max_Unpacked);
	Scratch_H_Lines = (short (*)[3]) Mem_Alloc (((Font_Ptr->Max_Unpacked + 1) >> 1) * 3 * sizeof (short));
	Scratch_V_Lines = (short (*)[3]) Mem_Alloc (((Font_Ptr->Max_Unpacked + 1) >> 1) * 3 * sizeof (short));
/*
 *	Process the individual characters:
 */
	for (Index = 0; Index < N_Chars; Index++)
		Tek_Char_Ptr = TK_Download_Char (Char_Vector[Index], Tek_Char_Ptr,
						 Scratch_Raster, Scratch_H_Lines, Scratch_V_Lines);
	TK_Font_Directory[Font_Ptr->Font_Index] = Tek_Char_Ptr;
/*
 *	Free temporary storage:
 */
	Mem_Free (Scratch_Raster);
	Mem_Free (Scratch_H_Lines);
	Mem_Free (Scratch_V_Lines);
}

struct Char_Entry *TK_Download_Char (Char_Ptr, Prev_Entry_Ptr, Scratch_Raster, Scratch_H_Lines,
                                     Scratch_V_Lines)
struct Char_Definition *Char_Ptr;
struct Char_Entry *Prev_Entry_Ptr;
unsigned char *Scratch_Raster;
unsigned short (*Scratch_H_Lines)[3], (*Scratch_V_Lines)[3];
{
	auto   struct Char_Entry *Tek_Char_Ptr;
	auto   unsigned int N_Vert, N_Hor, Index;
	auto   unsigned short (*Lines_Ptr)[3], Line_Type;
	extern char *Mem_Alloc();
/*
 *	Unpack the pixel data; choose between horizontal or vertical
 *	lines to draw the character:
 */
	Unpack_Raster_Data_M (Char_Ptr->Pixel_Width, Char_Ptr->Pixel_Height, Char_Ptr->Pixel_Array,
			      Scratch_Raster);
	if ((N_Hor = Scan_Raster_Horizontally_M (Char_Ptr->Pixel_Width, Char_Ptr->Pixel_Height,
					         Scratch_Raster, Scratch_H_Lines)) <=
	    (N_Vert = Scan_Raster_Vertically_M (Char_Ptr->Pixel_Width, Char_Ptr->Pixel_Height,
					        Scratch_Raster, Scratch_V_Lines))) {
		Line_Type = 0;
		Lines_Ptr = Scratch_H_Lines;
	} else {
		Line_Type = 1;
		Lines_Ptr = Scratch_V_Lines;
		N_Hor = N_Vert;
	}
/*
 *	Transfer the values to permanent area:
 */
	Tek_Char_Ptr = (struct Char_Entry *)
		Mem_Alloc (sizeof (struct Char_Entry) + N_Hor * 3 * sizeof (short));
	Tek_Char_Ptr->Link = Prev_Entry_Ptr;
	Tek_Char_Ptr->Type = Line_Type;
	Tek_Char_Ptr->N_Segments = N_Hor;
	for (Index = 0; Index < N_Hor; Index++) {
		Tek_Char_Ptr->Segment[Index][0] = Lines_Ptr[Index][0];
		Tek_Char_Ptr->Segment[Index][1] = Lines_Ptr[Index][1];
		Tek_Char_Ptr->Segment[Index][2] = Lines_Ptr[Index][2];
	}
	Char_Ptr->Driver_Id = (unsigned long) Tek_Char_Ptr;
	return (Tek_Char_Ptr);
}

TK_Setup_New_Page (Page_Number)
long Page_Number[10];
{
	static unsigned char Init_String[] = {
		CR, GS, 0x20, 0x60, 0x60, 0x20, 0x40,	/* Move to (0,0) */
		ESC, 0x60				/* Solid lines */
	}, Clear_String[] = {
		ESC, FF					/* Clears screen */
	};
	extern unsigned long Write_Device(), Flush_Device_IO();
/*
 *	Clear the screen. This requires a wait of about 1 second:
 */
	Write_Device (Clear_String, sizeof (Clear_String), TK_Device);
	Flush_Device_IO (TK_Device);
	Wait_for_Device_IO (TK_Device);
	Delay (1000);
/*
 *	Establish a known state of the terminal:
 */
	Write_Device (Init_String, sizeof (Init_String), TK_Device);
	TK_Current_Position.CurX = 0;
	TK_Current_Position.CurY = 0;
	TK_Current_Position.HiY = 0x20;
	TK_Current_Position.ExB = 0x60;
	TK_Current_Position.LoY = 0x60;
	TK_Current_Position.HiX = 0x20;
	TK_Current_Position.LoX = 0x40;
	TK_Mode = GRAPH_MODE;
	TK_Pen_Mode = 0;
}

TK_Eject_Page ()
{
	auto   unsigned short Count;
	static unsigned char Term_String[] = {
		CR, GS, 0x20, 0x60, 0x60, 0x20, 0x40,	/* Move to (0,0) */
		ESC, 0x60,				/* Solid lines */
		ESC, 0x38,				/* Extralarge characters */
		US					/* Alpha mode */
	}, Copy_String[] = {
		ESC, ETB
	};
	extern unsigned long Write_Device(), Flush_Device_IO();
/*
 *	Move to (0,0) and put the terminal into alpha mode:
 */
	TK_Emit_Beam (0, 0, 0);
	Write_Device (Term_String, sizeof (Term_String), TK_Device);
	TK_Mode = ALPHA_MODE;
	Flush_Device_IO (TK_Device);
	Wait_for_Device_IO (TK_Device);
/*
 *	Make the specified number of copies. Each requires a wait of
 *	about 10 seconds.
 */
	for (Count = TK_Copies; Count > 0; Count--) {
		Write_Device (Copy_String, sizeof (Copy_String), TK_Device);
		Flush_Device_IO (TK_Device);
		Wait_for_Device_IO (TK_Device);
		Delay (10000);
	}
}

TK_Typeset_String (X, Y, Font_Ptr, Char_Vector, N_Chars)
long X, Y;
struct Font_Definition *Font_Ptr;
struct Char_Definition **Char_Vector;
unsigned int N_Chars;
{
	auto   struct Char_Definition *Char_Ptr;
	auto   unsigned long Cur_X;
	auto   unsigned int Index;
	auto   unsigned short X_Pos, Y_Pos;

	Cur_X = X;
	Y_Pos = Round_Pixel (Y);
	for (Index = 0; Index < N_Chars; Index++) {
		Char_Ptr = Char_Vector[Index];
		X_Pos = Round_Pixel (Cur_X);
		if (Clip_Character_M (X_Pos, Y_Pos, Char_Ptr, 0, TK_Max_Dev_X, 0, TK_Max_Dev_Y) != 0)
			TK_Typeset_Character (X_Pos - Char_Ptr->X_Origin, Y_Pos - Char_Ptr->Y_Origin,
					      (struct Char_Entry *) Char_Ptr->Driver_Id);
		Cur_X += Char_Ptr->H_Escapement;
	}
}

TK_Typeset_Character (X, Y, Entry_Ptr)
unsigned short X, Y;
struct Char_Entry *Entry_Ptr;
{
	auto   unsigned short (*Seg_Ptr)[3], Count, X0, Y0, Coord;

	X0 = X;
	Y0 = Y;
	Seg_Ptr = Entry_Ptr->Segment[0];
	if (Entry_Ptr->Type != 0) {
		for (Count = Entry_Ptr->N_Segments; Count > 0; Count--) {
			Coord = (*Seg_Ptr)[0] + X0;
			TK_Emit_Beam (Coord, (*Seg_Ptr)[1] + Y0, 0);
			TK_Emit_Beam (Coord, (*Seg_Ptr)[2] + Y0, 1);
			Seg_Ptr++;
		}
	} else {
		for (Count = Entry_Ptr->N_Segments; Count > 0; Count--) {
			Coord = (*Seg_Ptr)[0] + Y0;
			TK_Emit_Beam ((*Seg_Ptr)[1] + X0, Coord, 0);
			TK_Emit_Beam ((*Seg_Ptr)[2] + X0, Coord, 1);
			Seg_Ptr++;
		}
	}
}

TK_Typeset_Pixel (X, Y, Width, Height, Pixel_Array)
long X, Y;
unsigned short Width, Height;
unsigned char *Pixel_Array;
{
	auto   unsigned short (*Seg_Ptr)[3], (*Seg_Base)[3];
	auto   unsigned char *Raster;
	auto   unsigned int Size;
	auto   unsigned short Coord, X0, Y0;
	extern char *Mem_Alloc();

	if (Width == 0 || Height == 0)
		return;
	X0 = Round_Pixel (X);
	Y0 = Round_Pixel (Y);
/*
 *	Allocate temporary storage:
 */
	Size = Width * Height;
	Raster = (unsigned char *) Mem_Alloc (Size);
	Seg_Ptr = Seg_Base = (short (*)[3]) Mem_Alloc (((Size + 1) >> 1) * 3 * sizeof (short));
/*
 *	Unpack the raster data; break into horizontal line segments,
 *	the draw the entity:
 */
	Unpack_Raster_Data_M (Width, Height, Pixel_Array, Raster);
	Size = Scan_Raster_Horizontally_M (Width, Height, Raster, Seg_Base);
	for (; Size > 0; Size--) {
		Coord = (*Seg_Ptr)[0] + Y0;
		TK_Emit_Beam ((*Seg_Ptr)[1] + X0, Coord, 0);
		TK_Emit_Beam ((*Seg_Ptr)[2] + X0, Coord, 1);
		Seg_Ptr++;
	}
	Mem_Free (Seg_Base);
	Mem_Free (Raster);
}

/*
 *	Routine TK_Typeset_Rule outputs a solid rule at the specified
 *	position. The rule must first be clipped, then drawn using the
 *	3-pixel wide beam of the Tektronix. The rule is drawn so that
 *	the beam movement is parallel to the longest side of the rule.
 *	Also, the beam is output in a 'zig-zag' motion, to minimize beam
 *	movement and the number of characters needed to specify the beam
 *	location.
 */

TK_Typeset_Rule (X, Y, Width, Height)
long X, Y;
unsigned long Width, Height;
{
	auto   unsigned short Dev_X, Dev_Y, Dev_Width, Dev_Height;
	auto   short Offset, Height_Remaining;
	static unsigned long wh[2];
	static long xy[2];

	xy[0] = Round_Pixel (X);
	xy[1] = Round_Pixel (Y);
	wh[0] = Ceil_Pixel (Width);
	wh[1] = Ceil_Pixel (Height);
	if (Clip_Rule_M (xy, wh, 0, TK_Max_Dev_X, 0, TK_Max_Dev_Y) != 0) {
		Dev_X = xy[0];
		Dev_Y = xy[1];
		if ((Dev_Width = wh[0]) > 0)
			Dev_Width--;
		if ((Dev_Height = wh[1]) > 0)
			Dev_Height--;
		if (Dev_Height >= Dev_Width) {
			if (Dev_Width >= MAX_WIDTH) {
				Dev_X += MAX_WIDTH >> 1;
				Dev_Width -= MAX_WIDTH >> 1;
			}
			Offset = Dev_Height;
			Height_Remaining = Dev_Width - (MAX_WIDTH >> 1);
			do {
				TK_Emit_Beam (Dev_X, Dev_Y, 0);
				Dev_Y -= Offset;
				TK_Emit_Beam (Dev_X, Dev_Y, 1);
				Dev_X += MAX_WIDTH;
				Offset = -Offset;
			} while ((Height_Remaining -= MAX_WIDTH) > 0);
		} else {
			if (Dev_Height >= MAX_WIDTH) {
				Dev_Y -= MAX_WIDTH >> 1;
				Dev_Height -= MAX_WIDTH >> 1;
			}
			Offset = Dev_Width;
			Height_Remaining = Dev_Height - (MAX_WIDTH >> 1);
			do {
				TK_Emit_Beam (Dev_X, Dev_Y, 0);
				Dev_X += Offset;
				TK_Emit_Beam (Dev_X, Dev_Y, 1);
				Dev_Y -= MAX_WIDTH;
				Offset = -Offset;
			} while ((Height_Remaining -= MAX_WIDTH) > 0);
		}
	}
}

TK_Typeset_Filled ()
{

}

TK_Typeset_Line (X1, Y1, X2, Y2)
long X1, Y1, X2, Y2;
{
	auto   unsigned long Phase;
	extern TK_Draw_Solid_Line();

	if (Current_State.Dashline_Len == 0)	/* Solid line */
		TK_Draw_Solid_Line (X1, Y1, X2, Y2, Current_State.Line_Width);
	else {
		Phase = ((Current_State.Flags & XY_DEFINED) != 0 && Current_State.Graphics_X == X1 &&
			 Current_State.Graphics_Y == Y1) ? Current_State.Scaled_Phase : 0;
		Current_State.Scaled_Phase = Segment_Line_M (X1, Y1, X2, Y2, Current_State.Dashline_Len,
							     Current_State.Scaled_Spec, Phase,
							     Current_State.Line_Width,
							     &TK_Draw_Solid_Line);
	}
	Current_State.Graphics_X = X2;
	Current_State.Graphics_Y = Y2;
	Current_State.Flags |= XY_DEFINED;
}

TK_Draw_Solid_Line (X1, Y1, X2, Y2, Width)
long X1, Y1, X2, Y2;
unsigned long Width;
{
	extern TK_Draw_Line();

	if (Width <= MAX_WIDTH << 10)
		TK_Draw_Line (X1, Y1, X2, Y2, Width);
	else {
		Build_Line_M (X1, Y1, X2, Y2, Width, MAX_WIDTH << 10, &TK_Draw_Line);
		TK_Typeset_Point (X1, Y1, Width);
		TK_Typeset_Point (X2, Y2, Width);
	}
}

TK_Typeset_Point (X, Y, Diameter)
long X, Y;
unsigned long Diameter;
{
	auto   long Radius;
	extern TK_Draw_Line();
	extern unsigned long Write_Device();

	Radius = (Diameter >> 1) & ~0x3FF;
	if (X >= Radius && X+Radius < TK_Max_X && Y >= Radius && Y+Radius < TK_Max_Y) {
		if (Radius << 1 <= MAX_WIDTH << 10) {
			TK_Emit_Beam (Round_Pixel (X), Round_Pixel (Y), 0);
			Write_Device (&TK_Current_Position.LoX, 1, TK_Device);
		} else
			Fill_Circle (X, Y, Radius, MAX_WIDTH << 10, &TK_Draw_Line);
	}
}

TK_Typeset_Arc (X, Y, Radius, Start_Ang, Stop_Ang)
long X, Y, Start_Ang, Stop_Ang;
unsigned long Radius;
{
	auto   unsigned long Phase;
	auto   long X0, Y0;
	extern TK_Draw_Solid_Arc();

	if (Current_State.Dashline_Len == 0)	/* Solid line */
		TK_Draw_Solid_Arc (X, Y, Radius, Start_Ang, Stop_Ang, Current_State.Line_Width);
	else {
		Compute_Arc_XY_M (X, Y, Radius, Start_Ang, &X0, &Y0);
		Phase = ((Current_State.Flags & XY_DEFINED) != 0 && Current_State.Graphics_X == X0 &&
			 Current_State.Graphics_Y == Y0) ? Current_State.Scaled_Phase : 0;
		Current_State.Scaled_Phase = Segment_Arc_M (X, Y, Radius, Start_Ang, Stop_Ang,
							    Current_State.Dashline_Len,
							    Current_State.Scaled_Spec, Phase,
							    Current_State.Line_Width, &TK_Draw_Solid_Arc);
	}
	Compute_Arc_XY_M (X, Y, Radius, Stop_Ang, &Current_State.Graphics_X, &Current_State.Graphics_Y);
	Current_State.Flags |= XY_DEFINED;
}

TK_Draw_Solid_Arc (X, Y, Radius, Start_Ang, Stop_Ang, Width)
long X, Y, Start_Ang, Stop_Ang;
unsigned long Radius, Width;
{
	auto   long X0, Y0;
	extern TK_Draw_Arc();

	if (Width <= MAX_WIDTH << 10)
		TK_Draw_Arc (X, Y, Radius, Start_Ang, Stop_Ang, Width);
	else {
		Build_Arc_M (X, Y, Radius, Start_Ang, Stop_Ang, Width, MAX_WIDTH << 10, &TK_Draw_Arc);
		Compute_Arc_XY_M (X, Y, Radius, Start_Ang, &X0, &Y0);
		TK_Typeset_Point (X0, Y0, Width);
		Compute_Arc_XY_M (X, Y, Radius, Stop_Ang, &X0, &Y0);
		TK_Typeset_Point (X0, Y0, Width);
	}
}

TK_Draw_Arc (X, Y, Radius, Start_Ang, Stop_Ang, Width)
long X, Y, Start_Ang, Stop_Ang;
unsigned long Radius, Width;
{
	auto   long X0, Y0, X1, Y1, S;
	extern TK_Draw_Line();

	S = Compute_Arc_Length_M (Radius, Stop_Ang - Start_Ang);
	if (S < 3 << 10) {
		Compute_Arc_XY_M (X, Y, Radius, Start_Ang, &X0, &Y0);
		Compute_Arc_XY_M (X, Y, Radius, Stop_Ang, &X1, &Y1);
		TK_Draw_Line (X0, Y0, X1, Y1, Width);
	} else
		Flatten_Arc_M (X, Y, Radius, Start_Ang, Stop_Ang, Width, 3 << 10, &TK_Draw_Line);
}

TK_Draw_Line (X1, Y1, X2, Y2, Width)
long X1, Y1, X2, Y2;
unsigned long Width;
{
	static long p1[2], p2[2];

	p1[0] = X1; p1[1] = Y1;
	p2[0] = X2; p2[1] = Y2;
	if (Clip_Line_M (p1, p2, 0, TK_Max_X - 1024, 0, TK_Max_Y - 1024) != 0) {
		TK_Emit_Beam (Round_Pixel (p1[0]), Round_Pixel (p1[1]), 0);
		TK_Emit_Beam (Round_Pixel (p2[0]), Round_Pixel (p2[1]), 1);
	}
}

TK_Set_Linewidth (Width)
unsigned long Width;
{
	auto   struct Graphics_State *State_Ptr;
	extern unsigned long Scale_Linestyle();

	State_Ptr = &Current_State;
	if (State_Ptr->Line_Width != Width) {
		State_Ptr->Line_Width = Width;
		State_Ptr->Scaled_Phase = Scale_Linestyle (State_Ptr->Dashline_Len, State_Ptr->Dashline_Spec,
							   State_Ptr->Line_Width, State_Ptr->Dashline_Phase,
							   State_Ptr->Scaled_Spec);
	}
}

TK_Set_Linestyle (Segment_Len, Segment_Desc, Start_Phase)
unsigned int Segment_Len, Segment_Desc[], Start_Phase;
{
	auto   struct Graphics_State *State_Ptr;
	auto   unsigned int Index;
	extern unsigned long Scale_Linestyle();
	extern int Compare_Linestyle();

	State_Ptr = &Current_State;
	if (Compare_Linestyle (Segment_Len, Segment_Desc, State_Ptr->Dashline_Len,
			       State_Ptr->Dashline_Spec) != 0) {
		State_Ptr->Dashline_Len = Segment_Len;
		for (Index = 0; Index < Segment_Len; Index++)
			State_Ptr->Dashline_Spec[Index] = Segment_Desc[Index];
		State_Ptr->Dashline_Phase = Start_Phase;
		State_Ptr->Scaled_Phase = Scale_Linestyle (State_Ptr->Dashline_Len, State_Ptr->Dashline_Spec,
							   State_Ptr->Line_Width, State_Ptr->Dashline_Phase,
							   State_Ptr->Scaled_Spec);
	}
}

TK_Set_Color (Red, Green, Blue)
unsigned int Red, Green, Blue;
{

}

TK_Save_Graphics (Identifier)
pointer Identifier;
{
	Save_GState (&Current_State, Identifier);
}

int TK_Restore_Graphics (Identifier)
pointer Identifier;
{
	extern int Restore_GState();

	return (Restore_GState (&Current_State, Identifier));
}

/*
 *	Routine TK_Emit_Beam moves the beam to a specific (X,Y)
 *	position:
 */

TK_Emit_Beam (X, Y, Pen_Mode)
unsigned short X, Y;
unsigned char Pen_Mode;
{
	auto   unsigned char *Ptr;
	auto   unsigned int Count;
	static unsigned char Addr_Array[20];
	extern unsigned long Write_Device();
	extern unsigned char *TK_Get_XY();

	Ptr = Addr_Array;
	if (TK_Mode != GRAPH_MODE) {
		*Ptr++ = GS;
		if (Pen_Mode != 0)
			*Ptr++ = TK_Current_Position.LoX;
	} else if (Pen_Mode == 0)
		*Ptr++ = GS;
	Count = TK_Get_XY (X * TK_Xform.a + Y * TK_Xform.c + TK_Xform.e,
			   X * TK_Xform.b + Y * TK_Xform.d + TK_Xform.f, Ptr) - Addr_Array;
	TK_Current_Position = TK_Pending_Position;
	TK_Mode = GRAPH_MODE;
	Write_Device (Addr_Array, Count, TK_Device);
}

/*
 *	Routine TK_Get_XY determines the characters needed to move
 *	to the specified location, and returns a pointer to the next
 *	character in the output array:
 */

unsigned char *TK_Get_XY (X, Y, Buffer_Ptr)
struct {
	unsigned Extra_2 : 2;
	unsigned Low_5   : 5;
	unsigned High_5  : 5;
	unsigned         : 4;
} X, Y;
char *Buffer_Ptr;
{
	auto   unsigned char *Ptr, *Out_Pos_Ptr, *In_Pos_Ptr;
	auto   unsigned char New_HiY, New_LoY, New_ExB, New_HiX;

	Ptr = Buffer_Ptr;
	In_Pos_Ptr = &TK_Current_Position.HiY;
	Out_Pos_Ptr = &TK_Pending_Position.HiY;
/*
 *	Place only the necessary character(s) into the output array:
 */
	if ((New_HiY = Y.High_5 | 0x20) != *In_Pos_Ptr++)
		*Ptr++ = New_HiY;
	*Out_Pos_Ptr++ = New_HiY;
	New_LoY = Y.Low_5 | 0x60;
	if ((New_ExB = (Y.Extra_2 << 2) | X.Extra_2 | 0x60) != *In_Pos_Ptr++) {
		*Ptr++ = New_ExB;
		*Ptr++ = New_LoY;
		*Out_Pos_Ptr++ = New_ExB;
		*Out_Pos_Ptr++ = New_LoY;
		In_Pos_Ptr++;
		if ((New_HiX = X.High_5 | 0x20) != *In_Pos_Ptr)
			*Ptr++ = New_HiX;
		*Out_Pos_Ptr++ = New_HiX;
	} else {
		*Out_Pos_Ptr++ = New_ExB;
		if ((New_HiX = X.High_5 | 0x20) != *(In_Pos_Ptr+1)) {
			*Ptr++ = New_LoY;
			*Ptr++ = New_HiX;
		} else if (New_LoY != *In_Pos_Ptr)
			*Ptr++ = New_LoY;
		*Out_Pos_Ptr++ = New_LoY;
		*Out_Pos_Ptr++ = New_HiX;
	}
	*Ptr++ = *Out_Pos_Ptr = X.Low_5 | 0x40;
/*
 *	Set new current position:
 */
	TK_Pending_Position.CurX = *(unsigned short *) &X;
	TK_Pending_Position.CurY = *(unsigned short *) &Y;
	return (Ptr);
}

#ifdef DEBUG_TK
TK_Dump_Buffer (Buffer, Count)
unsigned char *Buffer;
unsigned int Count;
{
	auto   unsigned char *Ptr, c;
	auto   unsigned int Cnt;
	static char *Ascii_Control[] = {
		"NUL", "SOH", "STX", "ETX", "EOT", "ENQ", "ACK", "BEL",
		"BS",  "HT",  "LF",  "VT",  "FF",  "CR",  "SO",  "SI",
		"DLE", "DC1", "DC2", "DC3", "DC4", "NAK", "SYN", "ETB",
		"CAN", "EM",  "SUB", "ESC", "FS",  "GS",  "RS",  "US"
	};

	Ptr = Buffer;
	for (Cnt = Count; Cnt > 0; Cnt--)
	if ((c = *Ptr++ & 0x7F) < 0x20)
		printf ("<%s>", Ascii_Control[c]);
	else if (c == 0x7F)
		printf ("<DEL>");
	else
		printf ("%c", c);
	printf ("\n");
}
#endif
