/***************************************************************************
** Although considerable effort has been expended to make this software   **
** correct and reliable, no warranty is implied; the author disclaims any **
** obligation or liability for damages, including but not limited to      **
** special, indirect, or consequential damages arising out of or in       **
** connection with the use or performance of this software.               **
***************************************************************************/

/*
 *	This file contains routines for the loading of packed (PK)
 *	files.
 */

#include "types.h"
#include "font.h"
#include "pk.h"
#include "tracedef.h"

#include "arith.p"
#include "fileio.p"
#include "fio.p"
#include "font.p"

#define case_1(k)   case k:
#define case_4(k)   case k: case k+1: case k+2: case k+3:

int Load_PK (Font_Dir, Resolution, Font_Ptr)
char *Font_Dir;
struct Ratio *Resolution;
struct Font_Definition *Font_Ptr;
{
	auto   pointer PK_File;
	auto   int Index;
	auto   unsigned char Command;
	extern pointer Open_PK_File();
	extern int Process_PK_Char(), Process_PK_Preamble();
	msgcode DVIOUT_BADPKFILE, DVIOUT_NOPKFILE, DVIOUT_UNKPKFORMAT;
/*
 *	Construct default file name; open the PK file; initialize array
 *	indicating which characters have been defined:
 */
	if ((PK_File = Open_PK_File (Font_Dir, Font_Ptr->Name_Ptr, Resolution, Font_Ptr->Magnification)) == 0) {
		Message (DVIOUT_NOPKFILE, Font_Ptr->Name_Ptr, 1, Font_Ptr->Magnification, 0);
		return (0);
	}
	for (Index = 0; Index < CHARS_PER_FONT; Index++)
		Char_Mask[Index] = 0;
/*
 *	Set the coding scheme, family name and face type to null:
 */
	Font_Ptr->Font_Coding[0] = '\0';
	Font_Ptr->Font_Family[0] = '\0';
	Font_Ptr->Font_Face = 0;
/*
 *	Process each command found in the file:
 */
	if ((Command = Read_Character_M (PK_File)) != PRE || File_At_EOF_M (PK_File) != 0)
		goto Bad_File;
	for (; Command != POST; Command = Read_Character_M (PK_File)) {
		if (trace_pixel)
			printf ("  Command byte: %02X\n", Command);
		if (File_At_EOF_M (PK_File) != 0)
			goto Bad_File;
		else if (Command < 240) {
			if (Process_PK_Char (Command, Font_Ptr, Char_Mask, PK_File) == 0)
				goto Bad_File;
		} else switch (Command) {

		case_4 (XXX1)
			Skip (Read_Unsigned_Int_M (Command-XXX1+1, PK_File), PK_File);
			break;

		case_1 (YYY)
			Read_Unsigned_Int_M (4, PK_File);
			break;

		case_1 (NOP)
			break;

		case_1 (PRE)
			if (Process_PK_Preamble (Font_Ptr, PK_File) == 0)
				goto Bad_File;
			if ((Font_Ptr->Pixel_Id & ID_M_FORMAT) != PK_FORMAT_A) {
				Message (DVIOUT_UNKPKFORMAT, Font_Ptr->Name_Ptr, 1, Font_Ptr->Magnification, 0);
				goto Bad_File_1;
			}
			break;

		default:
			goto Bad_File;
		}
	}
/*
 *	Check for undefined characters:
 */
	for (Index = 0; Index < CHARS_PER_FONT; Index++)
	if (Font_Ptr->Font_Directory[Index] != 0 && Char_Mask[Index] == 0) {
		Font_Ptr->Font_Directory[Index] = 0;
		Undef_Char_Count++;
	}
	Undef_Char_Count += Check_Undefined_Char_M (Font_Ptr, CHARS_PER_FONT, Font_Ptr->Char_Dir_Count);
	Close_File_M (PK_File);
	return (1);
Bad_File:
	Message (DVIOUT_BADPKFILE, Font_Ptr->Name_Ptr, 1, Font_Ptr->Magnification, 0);
Bad_File_1:
	Close_File_M (PK_File);
	return (0);
}

pointer Open_PK_File (Font_Dir, Font_Name, Resolution, Magnification)
char *Font_Dir, *Font_Name;
struct Ratio *Resolution;
unsigned long Magnification;
{
	static char Default_File[80], *Template = "%s.%dpk";

	sprintf (Default_File, Template, Font_Name,
		 Get_Font_Mag_M (1, Resolution, XN_Div_D_R_M (Magnification, Resolution->Numerator,
			         1000 * Resolution->Denominator)));
	if (trace_pixel)
		printf ("Name of the PK file is \"%s\"\n", Default_File);
	return (Open_File_M (Default_File, Font_Dir, "r", 0));
}

int Process_PK_Preamble (Font_Ptr, PK_File)
struct Font_Definition *Font_Ptr;
pointer PK_File;
{
	Font_Ptr->Pixel_Id = (ID_PK << ID_V_TYPE) | (Read_Unsigned_Int_M (1, PK_File) & ID_M_FORMAT);
	Skip (Read_Unsigned_Int_M (1, PK_File), PK_File);
	Font_Ptr->Font_Design_Size = Read_Unsigned_Int_M (4, PK_File);
	Font_Ptr->Checksum = Read_Unsigned_Int_M (4, PK_File);
	Read_Unsigned_Int_M (4, PK_File);
	Read_Unsigned_Int_M (4, PK_File);
	if (trace_pixel) {
		printf ("  PK file ID is %lu\n", Font_Ptr->Pixel_Id & ID_M_FORMAT);
		printf ("  PK file Design Size is %lu\n", Font_Ptr->Font_Design_Size);
		printf ("  PK file Checksum is %lu\n", Font_Ptr->Checksum);
	}
	return ((File_At_EOF_M (PK_File) == 0) ? 1 : 0);
}

int Process_PK_Char (Flag_Byte, Font_Ptr, Char_Mask, PK_File)
unsigned char Flag_Byte;
struct Font_Definition *Font_Ptr;
unsigned char *Char_Mask;
pointer PK_File;
{
	auto   struct Char_Definition *Char_Ptr;
	auto   unsigned char *Ptr;
	auto   unsigned long Char_Code;
	auto   unsigned int Pixel_Size, Packet_Length, Dyn_F;
	auto   int Status;
	static long Preamble_Array[PREAMBLE_SIZE];
	extern char *Mem_Alloc();
	extern unsigned int Read_PK_Preamble();
	extern int Read_PK_by_Bitmap(), Read_PK_by_Run_Count();
	msgcode DVIOUT_DUPLCHAR;

	Packet_Length = Read_PK_Preamble (Flag_Byte, Preamble_Array, PK_File);
	Char_Code = (unsigned long) Preamble_Array[CC];
	if (trace_pixel)
		printf ("    Packet length remaining: %u words\n", Packet_Length);
	if (Char_Code >= CHARS_PER_FONT || (unsigned long) Preamble_Array[W] > 65535 ||
	    (unsigned long) Preamble_Array[H] > 65535 ||
	    Preamble_Array[HOFF] < -65536 || Preamble_Array[HOFF] > 65535 ||
	    Preamble_Array[VOFF] < -65536 || Preamble_Array[VOFF] > 65535)
		Status = 0;
/*
 *	Allocate the character descriptor; fill it in:
 */
	else if (Char_Mask[Char_Code] != 0) {	/* already defined */
		Message (DVIOUT_DUPLCHAR, Char_Code, 0, Font_Ptr->Name_Ptr, 1);
		Skip (Packet_Length, PK_File);
		Status = 1;
	} else if (Font_Ptr->Font_Directory[Char_Code] != 0) {
		Pixel_Size = ((Preamble_Array[W] + 7) >> 3) * Preamble_Array[H];
		Char_Ptr = (struct Char_Definition *) Mem_Alloc (sizeof (struct Char_Definition) +
								 Pixel_Size);
		Ptr = (unsigned char *) Char_Ptr;
		Char_Ptr->Character_Code = Char_Code;
		Char_Ptr->DVI_Width = Preamble_Array[TFM];
		Char_Ptr->H_Escapement = (Preamble_Array[DX] >= 0) ? (Preamble_Array[DX] + 32) >> 6 :
								     (Preamble_Array[DX] - 32) >> 6;
		Char_Ptr->V_Escapement = (Preamble_Array[DY] >= 0) ? (Preamble_Array[DY] + 32) >> 6 :
								     (Preamble_Array[DY] - 32) >> 6;
		Char_Ptr->Pixel_Width = Preamble_Array[W];
		Char_Ptr->Pixel_Height = Preamble_Array[H];
		Char_Ptr->X_Origin = Preamble_Array[HOFF];
		Char_Ptr->Y_Origin = Preamble_Array[VOFF];
		Char_Ptr->Pixel_Array = &Ptr[sizeof(struct Char_Definition)];
		Char_Ptr->Driver_Id = (unsigned long) Font_Ptr->Font_Directory[Char_Code];
		Font_Ptr->Font_Directory[Char_Code] = Char_Ptr;
		Char_Mask[Char_Code] = 1;
		for (Ptr = Char_Ptr->Pixel_Array; Ptr < &Char_Ptr->Pixel_Array[Pixel_Size]; )
			*Ptr++ = 0;
		if (trace_pixel) {
			printf ("    Width in fraction of design size is %lu\n", Char_Ptr->DVI_Width);
			printf ("    Width and Height in pixels is %u by %u\n", Char_Ptr->Pixel_Width,
				Char_Ptr->Pixel_Height);
			printf ("    Origin is at (%d,%d)\n", Char_Ptr->X_Origin, Char_Ptr->Y_Origin);
		}
		if ((Dyn_F = Flag_Byte >> 4) == 14)
			Status = Read_PK_by_Bitmap (Char_Ptr->Pixel_Width, Char_Ptr->Pixel_Height,
						    Char_Ptr->Pixel_Array, PK_File);
		else
			Status = Read_PK_by_Run_Count (Dyn_F, (Flag_Byte >> 3) & 0x01, Char_Ptr->Pixel_Width,
						       Char_Ptr->Pixel_Height, Char_Ptr->Pixel_Array, PK_File);
	} else {
		if (trace_pixel)
			printf ("    (Not referenced for this font)\n");
		Skip (Packet_Length, PK_File);
		Status = 1;
	}
	return (Status);
}

/*
 *	Routine Read_PK_Preamble reads in the preamble for a character.
 */

unsigned int Read_PK_Preamble (Flag_Byte, Preamble_Array, PK_File)
unsigned char Flag_Byte;
long Preamble_Array[];
pointer PK_File;
{
	auto   unsigned long High_Bits;
	auto   unsigned int *Size_Ptr, Index, Preamble_Size;
	auto   unsigned char Type;
	static unsigned int Short_Sizes[PREAMBLE_SIZE] = {
		1, 1, 3, 1, 0, 1, 1, 1, 1
	}, Extended_Short_Sizes[PREAMBLE_SIZE] = {
		2, 1, 3, 2, 0, 2, 2, 2, 2
	}, Long_Sizes[PREAMBLE_SIZE] = {
		4, 4, 4, 4, 4, 4, 4, 4, 4
	}, If_Signed[PREAMBLE_SIZE] = {
		0, 0, 1, 1, 1, 0, 0, 1, 1
	};
/*
 *	Determine which preamble format to use:
 */
	High_Bits = Flag_Byte & 0x03;
	if ((Type = Flag_Byte & 0x07) == 7) {
		Size_Ptr = Long_Sizes;
		High_Bits = 0;
	} else if (Type > 3)
		Size_Ptr = Extended_Short_Sizes;
	else
		Size_Ptr = Short_Sizes;
/*
 *	Read in the preamble:
 */
	Preamble_Size = 0;
	for (Index = 0; Index < PREAMBLE_SIZE; Index++) {
		if (If_Signed[Index] != 0)
			Preamble_Array[Index] = Read_Signed_Int_M (Size_Ptr[Index], PK_File);
		else
			Preamble_Array[Index] = (long) Read_Unsigned_Int_M (Size_Ptr[Index], PK_File);
		Preamble_Size += Size_Ptr[Index];
	}
/*
 *	Append the most significant bits to the packet length; return
 *	the size of the remaining packet:
 */
	Preamble_Array[PL] |= High_Bits << (Size_Ptr[PL] * 8);
	if (trace_pixel) {
		printf ("  Character code: %lu\n", Preamble_Array[CC]);
		printf ("    Preamble array:\n");
		printf ("      Packet length: %lu, TFM width: %ld, Width: %lu, Height: %lu\n",
			Preamble_Array[PL], Preamble_Array[TFM], Preamble_Array[W], Preamble_Array[H]);
		printf ("      Hor. escapement: %ld, Vert. escapement: %ld, X Offset: %ld, Y Offset: %ld\n",
			Preamble_Array[DX], Preamble_Array[DY], Preamble_Array[HOFF], Preamble_Array[VOFF]);
	}
	return (Preamble_Array[PL] - (Preamble_Size - Size_Ptr[0] - Size_Ptr[1]));
}

int Read_PK_by_Bitmap (Width, Height, Out, PK_File)
unsigned short Width, Height;
unsigned char *Out;
pointer PK_File;
{
	auto   unsigned char *Out_Ptr;
	auto   unsigned int Mask;
	auto   unsigned short Wid, Hgt;
	static unsigned long Context;

	Out_Ptr = Out; Context = 0; Mask = 0x80;
	for (Hgt = Height; Hgt > 0; Hgt--) {
		for (Wid = Width; Wid > 0; Wid--) {
			if (Read_Bit_M (&Context, PK_File) != 0)
				*Out_Ptr |= Mask;
			if ((Mask >>= 1) == 0) {
				Mask = 0x80;
				Out_Ptr++;
			}
		}
		if (Mask != 0x80) {
			Mask = 0x80;
			Out_Ptr++;
		}
	}
	return (1);
}

int Read_PK_by_Run_Count (Dyn_F, Start_Color, Width, Height, Out, PK_File)
unsigned int Dyn_F, Start_Color;
unsigned short Width, Height;
unsigned char *Out;
pointer PK_File;
{
	auto   unsigned char *Out_Ptr, *Row_Ptr, *Ptr;
	auto   unsigned long Count, Count_Remaining;
	auto   unsigned int Mask, Color;
	auto   int Byte_Width;
	auto   unsigned short Column;
	static unsigned long Context, Repeat_Count;
	extern unsigned long Read_Run_Count();

	if (trace_pixel)
		printf ("    Dynamic packing factor is %u\n    ", Dyn_F);
	Column = 0; Mask = 0x80; Out_Ptr = Out;
	Context = 0; Color = Start_Color; Byte_Width = (Width + 7) >> 3;
	for (Count_Remaining = Width * Height; Count_Remaining > 0; ) {
		Count = Read_Run_Count (Dyn_F, &Context, &Repeat_Count, 0, PK_File); 
		if (Count == 0 || Count > Count_Remaining) /* Error */
			goto Bad_File;
		if (trace_pixel) {
			if (Color == 0)
				printf ("(%lu)", Count);
			else
				printf ("%lu", Count);
		}
		Count_Remaining -= Count;
		for (; Count > 0; Count--) {
			if (Color != 0)
				*Out_Ptr |= Mask;
			if ((Mask >>= 1) == 0) {	/* End of byte */
				Mask = 0x80;
				Out_Ptr++;
			}
			if (++Column == Width) {	/* End of a row */
				if (Mask != 0x80) {
					Mask = 0x80;
					Out_Ptr++;
				}
				if (Repeat_Count != 0) {
					if (trace_pixel)
						printf ("[%lu]", Repeat_Count);
					Row_Ptr = &Out_Ptr[-Byte_Width];
					for (; Repeat_Count > 0; Repeat_Count--) {
						if (Width > Count_Remaining)
							goto Bad_File;
						for (Ptr = Row_Ptr; Ptr < &Row_Ptr[Byte_Width]; )
							*Out_Ptr++ = *Ptr++;
						Count_Remaining -= Width;
					}
				}
				Column = 0;
			}
		}
		Color ^= 0x01;
	}
	if (trace_pixel) printf ("\n");
	return (1);
Bad_File:
	if (trace_pixel) printf ("\n");
	return (0);
}

unsigned long Read_Run_Count (Dyn_F, Context_Ptr, Rpt_Ptr, Level, PK_File)
unsigned int Dyn_F, Level;
unsigned long *Context_Ptr, *Rpt_Ptr;
pointer PK_File;
{
	auto   unsigned long Value;
	static unsigned long Rpt;
	extern unsigned long Read_Packed_Int();

	if ((Value = Read_Packed_Int (Context_Ptr, PK_File)) > 15)
		Value = Value - 15 + Dyn_F + ((13 - Dyn_F) << 4);
	else if (Value <= Dyn_F)
		;
	else if (Value < 14)
		Value = ((Value - Dyn_F - 1) << 4) +
			Read_Nybble_M (Context_Ptr, PK_File) + Dyn_F + 1;
	else if (Level == 0) {	/* This check necessary to prevent run-away */
		if (Value == 14)
			*Rpt_Ptr = Read_Run_Count (Dyn_F, Context_Ptr, &Rpt, Level+1, PK_File);
		else		/* Must be 15 */
			*Rpt_Ptr = 1;
		Value = Read_Run_Count (Dyn_F, Context_Ptr, &Rpt, Level+1, PK_File);
	}
	return (Value);
}

unsigned long Read_Packed_Int (Context_Ptr, File)
unsigned long *Context_Ptr;
char *File;
{
	auto   long Value;
	auto   unsigned int Count;

	Count = 0;
	while ((Value = Read_Nybble_M (Context_Ptr, File)) == 0 && File_At_EOF_M (File) == 0)
		Count++;
	while (Count > 0) {
		Value = (Value << 4) + Read_Nybble_M (Context_Ptr, File);
		Count--;
	}
	return (Value);
}
