/*
** Datei: DVIGR.C
** Autor: Ingo Eichenseher
*/

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <setjmp.h>
#include <math.h>
#include <stdarg.h>
#include <stdlib.h>
#include "dvi.h"
#include "dvisplin.h"

#define DEF_LINEWIDTH       0.4
#define DEF_ARROWWIDTH      1.2
#define DEF_ARROWLENGTH     6.4
#define DEF_ARROWLINE       0.4
#define DEF_LINECAP         1
#define DEF_ARROW_T         0.0
#define DEF_ARROW_L         0.0
#define DEF_DOTSIZE         2.4

#define MAX_PATTERN           8
#define MAX_DASH             32
#define MAX_GRAD             10
#define MAX_TOKLEN          128
#define MAX_ENV              72
#define MAX_NAME             16
#define MAX_SYMBOLS          64
#define MAX_EXPRESSIONS     512
#define MAX_ARGS             10
#define MAX_LOOPS            10
#define MAX_MATRIX	      4
#define MAX_REPPOINTS	    100

#define P_REAL                0
#define P_DIMEN               1

#define REAL                256
#define INTEGER             257
#define IDENTIFIER          258
#define FACTOR              259
#define UNIT                260
#define TRUE_SYM            261
#define ARC_SYM             262
#define POLYGON_SYM         263
#define POLYNOM_SYM         264
#define SETPOINT_SYM        265
#define SETUNITLENGTH_SYM   266
#define SETLINEWIDTH_SYM    267
#define SETDASH_SYM         268
#define SETARROWSIZE_SYM    269
#define SETARROWSHAPE_SYM   270
#define SETARROWLINE_SYM    271
#define SETLINECAP_SYM      272
#define DEFCLIP_SYM         273
#define CLIP_SYM            274
#define WHITECLIP_SYM       275
#define ENDCLIP_SYM         276
#define PLAIN_SYM           277
#define ROUND_SYM           278
#define ARROW_SYM           279
#define TRIANGLE_SYM        280
#define WEDGE_SYM           281
#define SETDOTSIZE_SYM      282
#define SETDEFAULTS_SYM     283
#define DEF_SYM             284
#define FUNCTION            285
#define FORMAL              286
#define LOOP                287
#define INTRINSIC           288
#define DOTS                289
#define FOR_SYM             290
#define INPUT_SYM           291
#define DEFPAT_SYM          292
#define STRINGSY            293
#define PATCLIP_SYM         294
#define UNDEF_SYM           295
#define INITMAT_SYM	    296
#define XROT_SYM	    297
#define YROT_SYM	    298
#define ZROT_SYM	    299
#define TRANS_SYM	    300
#define SCALE_SYM	    301
#define PUSH_SYM	    302
#define POP_SYM		    303
#define FIRST_SYM	    304
#define REP_SYM		    305

#define DRAW_SYM            306

static struct { char *keyword; int key; } keytable[] =
{
    { "true",           TRUE_SYM },
    { "arc",            ARC_SYM },
    { "polygon",        POLYGON_SYM },
    { "polynom",        POLYNOM_SYM },
    { "setpoint",       SETPOINT_SYM },
    { "setunitlength",  SETUNITLENGTH_SYM },
    { "setlinewidth",   SETLINEWIDTH_SYM },
    { "setdash",        SETDASH_SYM },
    { "setarrowsize",   SETARROWSIZE_SYM },
    { "setarrowshape",  SETARROWSHAPE_SYM },
    { "setarrowline",   SETARROWLINE_SYM },
    { "setlinecap",     SETLINECAP_SYM },
    { "setdotsize",     SETDOTSIZE_SYM },
    { "setdefaults",    SETDEFAULTS_SYM },
    { "defclip",        DEFCLIP_SYM },
    { "clip",           CLIP_SYM },
    { "whiteclip",      WHITECLIP_SYM },
    { "endclip",        ENDCLIP_SYM },
    { "arrow",          ARROW_SYM },
    { "round",          ROUND_SYM },
    { "plain",          PLAIN_SYM },
    { "triangle",       TRIANGLE_SYM },
    { "wedge",          WEDGE_SYM },
    { "def",            DEF_SYM },
    { "for",            FOR_SYM },
    { "input",          INPUT_SYM },
    { "defpattern",     DEFPAT_SYM },
    { "patclip",        PATCLIP_SYM },
    { "undef",          UNDEF_SYM},
    { "initmatrix",	INITMAT_SYM}, 
    { "xrot",		XROT_SYM}, 
    { "yrot",		YROT_SYM}, 
    { "zrot",		ZROT_SYM}, 
    { "translate",	TRANS_SYM}, 
    { "scale",		SCALE_SYM}, 
    { "pushmatrix",	PUSH_SYM}, 
    { "popmatrix",	POP_SYM}, 
    { "first",		FIRST_SYM}, 
    { "repeated",	REP_SYM}, 

    { "closedspline",       DRAW_SYM + 0 },
    { "spline",             DRAW_SYM + 1 },
    { "hermitespline",      DRAW_SYM + 2 },
    { "lefthermitespline",  DRAW_SYM + 3 },
    { "righthermitespline", DRAW_SYM + 4 },
    { "poly",               DRAW_SYM + 5 },
    { "closedpoly",         DRAW_SYM + 6 },
    { "dot",                DRAW_SYM + 7 },

    { NULL, -1}
};

static double m_pi(void){ return 3.14159265358979323846; }
static double m_e(void){ return 2.71828182845904523536; }
static double jump(double x, double y, double t) { return t<0 ? x:y; }

static struct 
{ 
    char *name; 
    int arguments;
    double (*func)();
} intrinsic[] =
{
    "pi",   0,m_pi,
    "e",    0,m_e,
    "sin",  1,sin,
    "cos",  1,cos,
    "tan",  1,tan,
    "atan", 1,atan,
    "asin", 1,asin,
    "acos", 1,acos,
    "sinh", 1,sinh,
    "cosh", 1,cosh,
    "tanh", 1,tanh,
    "exp",  1,exp,
    "abs",  1,fabs,
    "ceil", 1,ceil,
    "floor",1,floor,
    "ln",   1,log,
    "log",  1,log10,
    "sqrt", 1,sqrt,
    "pow",  2,pow,
    "jump", 3,jump,
    NULL,   1,NULL
};


typedef enum e_type 
{ 
    unused, invalid, constant, function, negate, parameter, variable, 
    loop_variable, product, quotient, sum, difference, intr_call
} e_type;

typedef struct expression_t
{
    e_type  type;
    union
    {
	real    c;
	real    *v;
	struct
	{
	    int n;
	    struct expression_t *e;
	    struct expression_t *a[MAX_ARGS];
	} f;
	struct
	{
	    int n;
	    double (*f)();
	    struct expression_t *a[MAX_ARGS];
	} i;
	struct
	{
	    struct expression_t *l, *r;
	} b;
	struct expression_t *u;
	struct 
	{
	    real d;
	    struct expression_t *e;
	} d;
	int n;
    } value;
} expression_t;

typedef struct symbol_t
{
    char name[MAX_NAME+1];
    int  arguments;
    union
    {
	expression_t *e;
	real v;
    } value;
} symbol_t;

typedef struct loop_t
{
    char name[MAX_NAME+1];
    real start, step, value;
    int n;
} loop_t;

typedef real mat_t[4][3];

typedef char formal_t[MAX_NAME+1];

static interval_t interval[MAX_INTERVAL];
static int n_curve, n_interval, spline_style;

static expression_t *expr_pool;
static symbol_t symbol_pool[MAX_SYMBOLS];
static formal_t formal[MAX_ARGS];
static loop_t loop[MAX_LOOPS];
static form_ptr, nloops;
static expression_t *loop_ex, *loop_ey, *loop_ez, *loop_rot;
static real loop_rx, loop_ry, loop_fx, loop_fy, loop_fz;
static real loop_x, loop_y;
static int loop_pnum;

static mat_t matrix[MAX_MATRIX];
static int matstack = 0;

static coord *point, *curve;
static char token[MAX_TOKLEN+1];
static real dash[MAX_DASH];
static unsigned short reppoint[MAX_REPPOINTS];
static unsigned char gr_patmask[MAX_PATTERN];
static unsigned char gr_whitemask[MAX_PATTERN];
static real hresconv, vresconv, graph_unit;
static int  gr_mulleft, gr_lastpoint = -1;
static real measure;
static int  sy;
static symbol_t *sy_function;
static symbol_t *gr_tx, *gr_ty;
static sy_index;
static char gr_intstr[] = "<integer>";
static jmp_buf gr_abort;
static long gr_orig_mag;


int gr_install(void)
{
    point = malloc(MAX_VERTICES*sizeof(coord));
    curve = malloc(MAX_VERTICES*sizeof(coord));
    expr_pool = malloc(MAX_EXPRESSIONS*sizeof(expression_t));
    if (point==NULL || curve==NULL || expr_pool==NULL)
    {
	fprintf(stderr,"Insufficient memory\n");
	return(0);
    }
    return(1);
}

void gr_destall(void)
{
    if (expr_pool) free(expr_pool);
    if (curve) free(curve);
    if (point) free(point);
}

static real expr(void);

static int grsy(void)
{
    int i = 0;
    int oldsy = sy;

    while(1)
    {
	while(isspace(spec_char) || spec_char=='\n') spec_getc();
	if (spec_char=='%')
	    while(spec_char!=EOF && spec_char!='\n') spec_getc();
	else break;
    }
    if (isdigit(spec_char) || spec_char=='.')
    {
	sy = spec_char=='.' ? DOTS:INTEGER;

	for(;isdigit(spec_char); spec_getc())
	    token[i++] = spec_char;
	if (spec_char=='.')
	{
	    token[i++] = spec_char; spec_getc();
	    if (spec_char=='.')
	    {
		if (sy==INTEGER) 
		{
		    spec_ungetc('.');
		    i--;
		}
		else
		{
		    token[i++] = spec_char; 
		    spec_getc();
		}
	    }
	    else 
	    {
		sy = REAL;
		for(;isdigit(spec_char); spec_getc())
		    token[i++] = spec_char;
	    }
	}
	if (sy!=DOTS && (spec_char=='e' || spec_char=='E'))
	{
	    sy = REAL;
	    token[i++] = spec_char; spec_getc();
	    if (spec_char=='+' || spec_char=='-')
	    {
		token[i++] = spec_char;
		spec_getc();
	    }
	    for(;isdigit(spec_char); spec_getc())
		token[i++] = spec_char;
	}
	token[i] = '\0';
    }
    else if (spec_char=='"' || spec_char=='\'')
    {
	int end_char = spec_char;
	for (spec_getc(); spec_char!=end_char && spec_char!=EOF &&
	    spec_char !=' '; spec_getc()) token[i++] = spec_char;
	token[i] = '\0';
	spec_getc();
	sy = STRINGSY;
    }
    else if (isalpha(spec_char))
    {
	sy = UNIT;
	for(; isalnum(spec_char); spec_getc())
	    token[i++] = spec_char;
	token[i] = '\0';
	if (!strcmp(token,"sp"))      measure = 1.0/65536.0;
	else if (!strcmp(token,"pc")) measure = 12.0;
	else if (!strcmp(token,"in")) measure = 72.27;
	else if (!strcmp(token,"bp")) measure = 72.27/72.0;
	else if (!strcmp(token,"cm")) measure = 72.27/2.54;
	else if (!strcmp(token,"mm")) measure = 72.27/25.4;
	else if (!strcmp(token,"dd")) measure = 1238.0/1157.0;
	else if (!strcmp(token,"cc")) measure = 1238.0*12.0/1157.0;
	else if (!strcmp(token,"pt")) measure = 1.0;
	else
	{
	    sy = IDENTIFIER;
	    for (i=0; keytable[i].keyword!=NULL && sy==IDENTIFIER; i++)
		if (!strcmp(keytable[i].keyword,token))
		    sy = keytable[i].key;
	    for (i=0; intrinsic[i].name!=NULL && sy==IDENTIFIER; i++)
		if (!strcmp(intrinsic[i].name,token))
		{
		    sy = INTRINSIC;
		    sy_index = i;
		}
	    for (i=0; i<nloops && sy==IDENTIFIER; i++)
		if (!strncmp(loop[i].name,token,MAX_NAME))
		{
		    sy = LOOP;
		    sy_index = i;
		}
	    for (i=0; i<form_ptr && sy==IDENTIFIER; i++)
		if (!strncmp(formal[i],token,MAX_NAME))
		{
		    sy = FORMAL;
		    sy_index = i;
		}
	    for (i=0; i<MAX_SYMBOLS && sy==IDENTIFIER; i++)
		if (!strncmp(symbol_pool[i].name,token,MAX_NAME))
		{
		    sy = FUNCTION;
		    sy_function = symbol_pool+i;
		}
	}
    }
    else if (spec_char==EOF)
    {
	strcpy(token,"<end-of-special>");
	sy = spec_char;
    }
    else
    {
	token[0] = sy = spec_char;
	token[1] = '\0';
	spec_getc();
    }
    return oldsy;
}

void gr_error(char *format, ...)
{
    va_list l;
    va_start(l,format);
    print("\"%s\"",spec_string(MAX_ENV));
    vprint(format,l);
    va_end(l);
    longjmp(gr_abort,1);
}

static void gr_expect(int symbol, char *message)
{
    if (sy!=symbol)
    {
	if (message!=NULL)
	    gr_error("Syntax error: Expected %s found '%s'",message,token);
	else
	    gr_error("Syntax error: Expected '%c' found '%s'",symbol,token);
    }
    (void)grsy();
}

#ifdef IBMPC

int matherr(struct exception *e)
{
    print("\"%s\"",spec_string(MAX_ENV));
    print("Floating Point Error evaluating %s(%g,%g)",e->name,e->arg1,e->arg2);
    e->retval = 0.0;
    return 1;
}

#endif

static void gr_check(int point_number)
{
    if (point_number<0 || point_number>=MAX_VERTICES)
	gr_error("Error: Point number %d out of range",point_number);
}

static expression_t *exp_alloc(void)
{
    int i;
    for (i=0; i<MAX_EXPRESSIONS && expr_pool[i].type!=unused; i++)
    if (i>=MAX_EXPRESSIONS)
	halt("Out of expression memory");
    return expr_pool + i;
}

static void exp_free(expression_t *e)
{
    e->type = unused;
}

static void exp_remove(expression_t *e)
{
    int i;

    if (e==NULL) return;

    switch(e->type)
    {
	case variable:
	case parameter: 
	case constant:
	case loop_variable:
	    break;

	case negate: 
	    exp_remove(e->value.u);
	    break;

	case product: 
	case quotient:
	case sum:
	case difference:
	    exp_remove(e->value.b.l);
	    exp_remove(e->value.b.r);
	    break;

	case function:
	    for (i=0; i<e->value.f.n; i++)
		exp_remove(e->value.f.a[i]);
	    break;

	case intr_call:
	    for (i=0; i<e->value.i.n; i++)
		exp_remove(e->value.i.a[i]);
	    break;

	default:
	    halt("Internal Error: unknown expression type");
    }
    exp_free(e);
}

static void exp_invalid(expression_t *e, expression_t *n)
{
    register int i;
    for (i=0; i<MAX_EXPRESSIONS; i++)
	if (expr_pool[i].type==function && expr_pool[i].value.f.e==e)
	    if (n==NULL) expr_pool[i].type = invalid;
	    else expr_pool[i].value.f.e = n;
}

static void exp_init(void)
{
    int i;
    for (i=0; i<MAX_EXPRESSIONS; i++)
	expr_pool[i].type = unused;
}

static symbol_t *sym_alloc(void)
{
    int i;
    for (i=0; i<MAX_SYMBOLS && symbol_pool[i].name[0]; i++);
    if (i>=MAX_SYMBOLS)
	halt("Out of symbol memory");
    return symbol_pool + i;
}

static void sym_free(symbol_t *s)
{
    s->name[0] = '\0';
}

static void sym_init(void)
{
    int i;
    for (i=0; i<MAX_SYMBOLS; i++)
	symbol_pool[i].name[0] = '\0';
}

static real evaluate(expression_t *e, real *actual)
{
    switch(e->type)
    {
	case constant: return e->value.c;
	case negate: return -evaluate(e->value.u,actual);
	case parameter: return actual[e->value.n];
	case variable: return *e->value.v;
	case loop_variable: return loop[e->value.n].value;
	case product: 
	    return evaluate(e->value.b.l,actual)*
		   evaluate(e->value.b.r,actual);
	case quotient:
	    return evaluate(e->value.b.l,actual)/
		   evaluate(e->value.b.r,actual);
	case sum:
	    return evaluate(e->value.b.l,actual)+
		   evaluate(e->value.b.r,actual);
	case difference:
	    return evaluate(e->value.b.l,actual)-
		   evaluate(e->value.b.r,actual);
	case function:
	    if (e->value.f.n)
	    {
		real newact[MAX_ARGS]; 
		int i;
		for (i=0; i<e->value.f.n; i++)
		    newact[i] = evaluate(e->value.f.a[i],actual);
		return evaluate(e->value.f.e,newact);
	    }
	    return evaluate(e->value.f.e,(real*)NULL);
	case intr_call:
	    if (e->value.i.n)
	    {
		real newact[MAX_ARGS]; 
		int i;
		for (i=0; i<e->value.i.n; i++)
		    newact[i] = evaluate(e->value.i.a[i],actual);
		switch(e->value.i.n)
		{
		    case 1: 
			return (real)(*(double(*)(double))
			    e->value.i.f)(newact[0]);
		    case 2:
			return (real)(*(double(*)(double,double))
			    e->value.i.f)(newact[0],newact[1]);
		    case 3:
			return (real)(*(double(*)(double,double,double))
			    e->value.i.f)(newact[0],newact[1],newact[2]);
		    default:
			halt("Internal Error: Illegal evaluation of intrinsic");
		}
	    }
	    return (real) (*(double(*)(void))e->value.i.f)();

	case invalid:
	    gr_error("Error: Invalid function definition");
	    break;

	default:
	    halt("Internal Error: unknown expression type");
    }
    return 0.0;
}

static expression_t *co_expr(void);

static expression_t *co_factor(void)
{
    expression_t *e;
    symbol_t *f;
    switch(sy)
    {
	case REAL:
	    e = exp_alloc();
	    e->type = constant;
	    e->value.c = atof(token);
	    (void)grsy();
	    break;
	case INTEGER:
	    e = exp_alloc();
	    e->type = constant;
	    e->value.c = (real)atoi(token);
	    (void)grsy();
	    break;
	case '(':
	    (void)grsy();
	    e = co_expr();
	    gr_expect(')',(char*)NULL);
	    break;
	case '-':
	    (void)grsy();
	    e = exp_alloc();
	    e->type = negate;
	    e->value.u = co_factor();
	    break;
	case FUNCTION:
	    e = exp_alloc();
	    f = sy_function;
	    (void)grsy();
	    if (f->arguments)
	    {
		int i;
		e->type = function;
		gr_expect('(',(char*)NULL);
		for (i=0; i<f->arguments; i++)
		{
		    e->value.f.a[i] = co_expr();
		    if (i<f->arguments-1) gr_expect(',',(char*)NULL);
		}
		gr_expect(')',(char*)NULL);
		e->value.f.e = f->value.e;
		e->value.f.n = f->arguments;
	    }
	    else
	    {
		e->type = variable;
		e->value.v = &f->value.v;
	    }
	    break;
	case INTRINSIC:
	    e = exp_alloc();
	    e->type = intr_call;
	    e->value.i.n = intrinsic[sy_index].arguments;
	    e->value.i.f = intrinsic[sy_index].func;
	    (void)grsy();
	    if (e->value.i.n)
	    {
		int i;
		gr_expect('(',(char*)NULL);
		for (i=0; i<e->value.i.n; i++)
		{
		    e->value.i.a[i] = co_expr();
		    if (i<e->value.i.n-1) gr_expect(',',(char*)NULL);
		}
		gr_expect(')',(char*)NULL);
	    }
	    break;
	case FORMAL:
	    e = exp_alloc();
	    e->type = parameter;
	    e->value.n = sy_index;
	    (void)grsy();
	    break;
	case LOOP:
	    e = exp_alloc();
	    e->type = loop_variable;
	    e->value.n = sy_index;
	    (void)grsy();
	    break;
	default: 
	    gr_expect(FACTOR,"<factor>"); 
	    break;
    }
    return e;
}

static expression_t *co_term(void)
{
    expression_t *e = co_factor();

    while(sy=='*' || sy=='/')
    {
	expression_t *new = exp_alloc();
	new->type = grsy()=='*' ? product:quotient;
	new->value.b.l = e;
	new->value.b.r = co_factor();
	e = new;
    }
    return e;
}

static expression_t *co_expr(void)
{
    expression_t *e = co_term();

    while(sy=='+' || sy=='-')
    {
	expression_t *new = exp_alloc();
	new->type = grsy()=='+' ? sum:difference;
	new->value.b.l = e;
	new->value.b.r = co_term();
	e = new;
    }
    return e;
}

static expression_t *co_dimen(real *f)
{
    expression_t *d;

    d = co_expr();

    if (sy==TRUE_SYM)
    {
	(void)grsy();
	if (sy==UNIT)
	{
	    *f = measure*1000.0/gr_orig_mag;
	    grsy();
	}
	else gr_expect(UNIT,"<unit-of-measure>");
    }
    else if (sy==UNIT)
    {
	*f = measure;
	(void)grsy();
    }
    else *f = graph_unit;
    return d;
}

static void gr_undef(void)
{

    if (sy==FUNCTION) 
    {
	symbol_t *f = sy_function;
	if (f->arguments) 
	{
	    exp_invalid(f->value.e,(expression_t*)NULL);
	    exp_remove(f->value.e);
	}
	sym_free(f);
	(void)grsy();
    }
    else gr_expect(FUNCTION,"<defined-identifier>");
}

static void gr_def(void)
{
    if (sy==IDENTIFIER || sy==FUNCTION)
    {
	symbol_t s, **s1 = NULL, *s2=NULL;
	strncpy(s.name,token,MAX_NAME);
	if (!strcmp(token,"Tx")) s1 = &gr_tx;
	else if (!strcmp(token,"Ty")) s1 = &gr_ty;

	if (sy==FUNCTION) s2 = sy_function;

	s.name[MAX_NAME] = '\0';
	(void)grsy();
	s.arguments = 0;
	form_ptr = nloops = 0;
	if (sy=='(')
	{
	    int n;
	    (void)grsy();
	    for(n=0; sy==IDENTIFIER || sy==FUNCTION && n<MAX_ARGS; n++)
	    {
		strncpy(formal[n],token,MAX_NAME);
		formal[form_ptr++][MAX_NAME]='\0';
		(void)grsy();
		if (sy!=',') break;
		(void)grsy();
	    }
	    gr_expect(')',(char*)NULL);
	}
	if (s1!=NULL && form_ptr!=2)
	    gr_error("Error: %s must have two arguments",s.name);
	gr_expect('=', (char*)NULL);

	if ( (s.arguments = form_ptr)>0 ) s.value.e = co_expr();
	else s.value.v = expr();

	form_ptr = 0;
	if (s2!=NULL)
	{
	    if (s2->arguments!=s.arguments) 
		exp_invalid(s2->value.e,(expression_t*)NULL);
	    else
		exp_invalid(s2->value.e,s.value.e);
	    if (s2->arguments) exp_remove(s2->value.e);
	}
	else s2 = sym_alloc();
	*s2 = s;
	if (s1!=NULL) *s1 = s2;
    }
    else gr_expect(IDENTIFIER,"<identifier>");
}

static real factor(void)
{
    real value;
    symbol_t *f;

    switch(sy)
    {
	case REAL : 
	    value = atof(token); 
	    (void)grsy(); 
	    break;
	case INTEGER : 
	    value = (real)atoi(token); 
	    (void)grsy(); 
	    break;
	case '(' : 
	    (void)grsy(); 
	    value = expr(); 
	    gr_expect(')',(char*)NULL); 
	    break;
	case '-' : 
	    (void)grsy(); 
	    value = -factor(); 
	    break;
	case FUNCTION:
	    f = sy_function;
	    (void)grsy();
	    if (f->arguments)
	    {
		int i;
		real actual[MAX_ARGS];
		gr_expect('(',(char*)NULL);
		for (i=0; i<f->arguments; i++)
		{
		    actual[i] = expr();
		    if (i<f->arguments-1) gr_expect(',',(char*)NULL);
		}
		gr_expect(')',(char*)NULL);
		value = evaluate(f->value.e,actual);
	    }
	    else value = f->value.v;
	    break;
	case INTRINSIC:
	    if (intrinsic[sy_index].arguments)
	    {
		real actual[MAX_ARGS];
		double (*f)() = intrinsic[sy_index].func;
		int i, n = intrinsic[sy_index].arguments;

		(void)grsy();
		gr_expect('(',(char*)NULL);
		for (i=0; i<n; i++)
		{
		    actual[i] = expr();
		    if (i<n-1) gr_expect(',',(char*)NULL);
		}
		gr_expect(')',(char*)NULL);
		switch(n)
		{
		    case 1: 
			value = (real)(*(double(*)(double))f)
			    (actual[0]);
			break;
		    case 2:
			value = (real)(*(double(*)(double,double))f)
			    (actual[0],actual[1]);
			break;
		    case 3:
			value = (real)(*(double(*)(double,double,double))f)
			    (actual[0],actual[1],actual[2]);
			break;
		    default:
			halt("Internal Error calling intrinsic");
		}
	    }
	    else
	    {
		double (*f)(void) = (double (*)(void))intrinsic[sy_index].func;
		value = (real)(*f)();
		(void)grsy();
	    }
	    break;
	default: 
	    gr_expect(FACTOR,"<factor>"); 
	    break;
    }
    return value;
}

static real term(void)
{
    real value = factor();

    while(sy=='*' || sy=='/')
	if (grsy()=='*') value *= factor();
	else value /= factor();
    return value;
}

static real expr(void)
{
    real value = term();

    while(sy=='+' || sy=='-')
	if (grsy()=='+') value += term();
	else value -= term();
    return value;
}

static real dimen(real *factor)
{
    real d = expr();

    if (sy==TRUE_SYM)
    {
	(void)grsy();
	if (sy==UNIT)
	{
	    *factor = measure*1000.0/gr_orig_mag;
	    grsy();
	}
	else gr_expect(UNIT,"<unit-of-measure>");
    }
    else if (sy==UNIT)
    {
	*factor = measure;
	(void)grsy();
    }
    else *factor = graph_unit;
    return d;
}

static void gr_initmat(mat_t a)
{
    register int i, j;
    for (i=0; i<4; i++)
    for (j=0; j<3; j++)
	a[i][j] = (i==j) ? 1.0:0.0;
}

static void gr_mulmat(mat_t t, mat_t a)
{
    register int i, j;

    if (gr_mulleft)
    {
	for (j=0; j<3; j++)
	{
	    double c[4];
	    for (i=0; i<4; i++)
		c[i] = a[i][0]*t[0][j] + a[i][1]*t[1][j] + a[i][2]*t[2][j];
	    c[3] += t[3][j];
	    for (i=0; i<4; i++) t[i][j] = c[i];
	}
    }
    else
    {
	for (i=0; i<4; i++)
	{
	    double c[3];
	    for (j=0; j<3; j++)
	    {
		c[j] = t[i][0]*a[0][j] + t[i][1]*a[1][j] + t[i][2]*a[2][j];
		if (i==3) c[j] += a[3][j];
	    }
	    for (j=0; j<3; j++) t[i][j] = c[j];
	}
    }
}

static void gr_testleft(void)
{
    if (sy==FIRST_SYM)
    {
	gr_mulleft = 1;
	(void) grsy();
    }
    else gr_mulleft = 0;
}

static void gr_rotate(mat_t t, int d, real omega)
{
    register int i, j;
    double s = sin(omega);
    double c = cos(omega);
    mat_t a;
    gr_initmat(a);
    switch(d)
    {
	case 0: i=1; j=2; break;
	case 1: i=0; j=2; break;
	case 2: i=0; j=1; break;
	default: gr_error("Internal error: gr_rotate()");
    }
    a[i][i] =  c; a[i][j] = s;
    a[j][i] = -s; a[j][j] = c;
    gr_mulmat(t, a);
}

static void gr_translate(mat_t t, real x, real y, real z)
{
    mat_t a;
    gr_initmat(a);
    a[3][0] = x; a[3][1] = y; a[3][2] = z;
    gr_mulmat(t, a);
}

static void gr_scalemat(mat_t t, real x, real y, real z)
{
    mat_t a;
    gr_initmat(a);
    a[0][0] = x; a[1][1] = y; a[2][2] = z;
    gr_mulmat(t, a);
}

static void gr_project(real *x, real *y, mat_t t, real *p)
{
    *x = p[0]*t[0][0] + p[1]*t[1][0] + p[2]*t[2][0] + t[3][0];
    *y = p[0]*t[0][2] + p[1]*t[1][2] + p[2]*t[2][2] + t[3][2]; 
}

void gr_defaults(void)
{
    setdash((real*)NULL,0);
    setarrowsize(DEF_ARROWWIDTH*hresconv,DEF_ARROWLENGTH*hresconv);
    setarrowshape(DEF_ARROW_T,DEF_ARROW_L);
    setarrowline(DEF_ARROW_L*hresconv);
    setdotsize(DEF_DOTSIZE*hresconv);
    setlinewidth(DEF_LINEWIDTH*hresconv);
    setlinecap(DEF_LINECAP,DEF_LINECAP);
    clip_end();
    graph_unit = 1000.0/(real)gr_orig_mag;
    gr_lastpoint = -1;
    gr_tx = gr_ty = (symbol_t*)NULL;
    sym_init();
    exp_init();
    gr_initmat(matrix[matstack=0]);
}

void gr_init(long mag, long orig_mag)
{
    gr_orig_mag = orig_mag;
    hresconv = (real)op.hres*(real)mag/1000.0/72.27;
    vresconv = (real)op.hres*(real)mag/1000.0/72.27;
    gr_defaults();
}

static void gr_scale(real x,real y,real xo,real yo,coord *new)
{
    if (dvilw)
    {
	xo *= hresconv;
	yo *= vresconv;
	new->x = x + xo;
	new->y = y + yo;
    }
    else
    {
	if (op.landscape)
	{
	    xo *= vresconv;
	    yo *= hresconv;
	    new->x = x + yo;
	    new->y = y + xo;
	}
	else
	{
	    xo *= hresconv;
	    yo *= vresconv;
	    new->x = x + xo;
	    new->y = y - yo;
	}
    }
}


static int gr_parameters(int numpar, ...)
{
    va_list l;
    int i;

    va_start(l,numpar);

    if (numpar<0)
    {
	int type = va_arg(l,int);
	real f, *value = va_arg(l,real *);
	for (i=0; i<-numpar && sy!=';' && sy!=EOF; i++)
	{
	    if (type)
	    {
		value[i] = dimen(&f);
		value[i] *= f;
	    }
	    else value[i] = expr();
	    if (sy==',') grsy();
	}
    }
    else
	for (i=0; i<numpar && sy!=';' && sy!=EOF; i++)
	{
	    int type = va_arg(l,int);
	    real f, *value = va_arg(l,real*);
	    if (type)
	    {
		*value = dimen(&f);
		*value *= f;
	    }
	    else *value = expr();
	    if (sy==',') grsy();
	}

    va_end(l);
    return i;
}

static int gr_linecap(void)
{
    int linecap;
    switch(grsy())
    {
	case PLAIN_SYM : linecap = 0; break;
	case ROUND_SYM : linecap = 1; break;
	case ARROW_SYM : linecap = 2; break;
	case TRIANGLE_SYM : linecap = 3; break;
	case WEDGE_SYM : linecap = 4; break;
	default : gr_expect(PLAIN_SYM,"<linecap>"); break;
    }
    return linecap;
}

static void gr_point(real lx, real ly, real *x, real *y);

static void gr_component(real lx, real ly, real *x, real *y)
{
    if (sy==INTEGER)
    {
	int l = (int)atoi(token);
	gr_check(l);
	*x = point[l].x;
	*y = point[l].y;
	grsy();
    }
    else if (sy=='(' || sy=='[')
    {
	gr_point(lx,ly,x,y);
    }
    else if (sy=='@')
    {
	*x = lx;
	*y = ly;
	grsy();
    }
    else gr_expect(INTEGER,gr_intstr);
}

static void gr_point(real lx, real ly, real *x, real *y)
{
    real x1, y1, x2, y2;
    int endsy = grsy()=='(' ? ')' : ']';

    gr_component(lx,ly,&x1,&y1);
    gr_expect(',',(char*)NULL);
    gr_component(lx,ly,&x2,&y2);
    gr_expect(endsy,(char*)NULL);

    if (endsy==')')
    {
	*x = x1;
	*y = y2;
    }
    else
    {
	real lambda = expr();
	*x = x2*lambda + x1*(1.0-lambda);
	*y = y2*lambda + y1*(1.0-lambda);
    }
}

static void gr_lset(real xoffset, real yoffset)
{
    gr_check(loop_pnum);
    gr_scale(loop_x,loop_y,xoffset,yoffset,point+loop_pnum);
    loop_pnum++;
}

static void gr_loop(loop_t *l, int n, void (*f)(real,real))
{
    if (n>0)
    {
	int i;
	for (i=l->n, l->value=l->start; i--; l->value+=l->step)
	    gr_loop(l+1,n-1,f);
    }
    else
    {
	real off[3], x, y;

	off[0] = evaluate(loop_ex,(real*)NULL);
	off[1] = evaluate(loop_ey,(real*)NULL);
    if (loop_ez)
    {
	off[0] *= loop_fx;
	off[1] *= loop_fy;
	off[2]  = loop_fz * evaluate(loop_ez,(real*)NULL);
	gr_project(&x, &y, matrix[matstack], off);
    }
    else
    {
	if (gr_tx==0) x = off[0]*loop_fx;
	else x = evaluate(gr_tx->value.e,off)*loop_fx;
	if (gr_ty==0) y = off[1]*loop_fy;
	else y = evaluate(gr_ty->value.e,off)*loop_fy;

	if (loop_rot!=NULL)
	{
	real phi = evaluate(loop_rot,(real*)NULL);
	real c = cos(phi), s = sin(phi);
	real x0 = x-loop_rx, y0 = y-loop_ry;
	x = x0*c - y0*s + loop_rx;
	y = y0*c + x0*s + loop_ry;
	}
    }
	(*f)(x,y);
    }
}

static void gr_endloops(void)
{
    nloops = 0;
    exp_remove(loop_ex);
    exp_remove(loop_ey);
    exp_remove(loop_ez);
    exp_remove(loop_rot);
}

static void gr_defloops(void)
{
    int n = nloops = 0;

    if (sy!=IDENTIFIER) gr_expect(IDENTIFIER,"<identifier>");
    while(sy==IDENTIFIER && n<MAX_LOOPS)
    {
	real fin, rn;
	strncpy(loop[n].name,token,MAX_NAME);
	(void)grsy(); gr_expect('=',(char*)NULL);
	loop[n].start = expr();
	gr_expect(DOTS,"..");
	fin = expr();
	gr_expect(':',(char*)NULL);
	rn = expr();
	loop[n].n = iround(rn);
	if (loop[n].n>1) 
	    loop[n].step = (fin-loop[n].start)/(loop[n].n-1);
	else 
	    loop[n].step = 0;
	n++;
    }
    nloops = n;
    gr_expect('[',(char*)NULL);
    loop_ex = co_dimen(&loop_fx);
    gr_expect(',',(char*)NULL);
    loop_ey = co_dimen(&loop_fy);
    loop_ez = loop_rot = NULL;
    if (sy==':')
    {
	(void)grsy();
	loop_rot = co_expr();
	if (sy=='[')
	{
	    real f;
	    (void)grsy();
	    loop_rx = dimen(&f);
	    loop_rx *= f;
	    gr_expect(',',(char*)NULL);
	    loop_ry = dimen(&f);
	    loop_ry *= f;
	    gr_expect(']',(char*)NULL);
	}
	else loop_rx = loop_ry = 0.0;
    }
    else if (sy==',')
    {
    (void)grsy();
    loop_ez = co_dimen(&loop_fz);
    }
    gr_expect(']',(char*)NULL);
}

static void gr_setpoint(real x, real y)
{
    int np = 0, point_number;
    while(sy==INTEGER || sy=='[' || sy==IDENTIFIER)
    {
	np++;
	if (sy==INTEGER)
	{
	    gr_lastpoint = point_number = (int)atoi(token);
	    (void)grsy();
	}
	else point_number = ++gr_lastpoint;
	gr_check(point_number);
	point[point_number].x = x;
	point[point_number].y = y;
	if (sy=='[')
	{
	    real xoffset, yoffset, off[3], f[3];
	    (void)grsy();
	    off[0] = dimen(&f[0]);
	    gr_expect(',',(char*)NULL);
	    off[1] = dimen(&f[1]);
	    if (sy==',')
	    {
		int i;
		(void)grsy();
		off[2] = dimen(&f[2]);
		gr_expect(']',(char*)NULL);
		for (i=0; i<3; i++) off[i] *= f[i];
		gr_project(&xoffset, &yoffset, matrix[matstack], off);
		gr_scale(x, y, xoffset, yoffset, point+point_number);
	    }
	    else
	    {
		gr_expect(']',(char*)NULL);
		if (gr_tx==NULL) xoffset = off[0];
		else xoffset = evaluate(gr_tx->value.e,off);
		if (gr_ty==NULL) yoffset = off[1];
		else yoffset = evaluate(gr_ty->value.e,off);
		gr_scale(x,y,xoffset*f[0],yoffset*f[1],point+point_number);
	    }
	}
	else if (sy==IDENTIFIER || sy==FOR_SYM)
	{
	    if (sy==FOR_SYM) (void)grsy();
	    gr_defloops();
	    loop_x = x;
	    loop_y = y;
	    loop_pnum = point_number;
	    gr_loop(loop,nloops,gr_lset);
	    gr_endloops();
	    gr_lastpoint = loop_pnum-1;
	}
	if (sy==',') grsy();
    }
    if (np==0) gr_lastpoint = -1;
}

static void gr_clip(unsigned char *patmask)
{
    int cerror = clip_exec(patmask);
    if (cerror)
    {
	print("\"%s\"",spec_string(MAX_ENV));
	switch(cerror)
	{
	    case 0  : break;    /* Fillpath OK */
	    case 1  : print("Fillpath not connected"); break;
	    case 2  : print("Fillpath too long"); break;
	    case 3  : print("Internal error in fillpath"); break;
	    default : print("Illegal fillpath"); break;
	}
	clip_end();
    }
}

static void gr_lspline(real xoffset, real yoffset)
{
    coord o;
    int i;

    gr_scale((real)0,(real)0,xoffset,yoffset,&o);
    for(i=0; i<n_curve; i++)
	curve[i].x += o.x, curve[i].y += o.y;
    spline(curve,n_curve,interval,n_interval,spline_style);
    for(i=0; i<n_curve; i++)
	curve[i].x -= o.x, curve[i].y -= o.y;
}

static void gr_range(int *from,  int *to,  int *step)
{
    *step = 1;
    *from = (int)atoi(token); grsy();
    if (sy=='-')
    {
	grsy();
	if (sy!=INTEGER) gr_expect(INTEGER,gr_intstr);
	*to = (int)atoi(token); grsy();
	if (sy=='/')
	{
	    grsy();
	    if (sy!=INTEGER) gr_expect(INTEGER,gr_intstr);
	    *step = (int)atoi(token); grsy();
	    if (*step<=0) gr_error("Stepsize must be positive");
	}
    }
    else *to = *from;
}

static void gr_draw(int style)
{
    int n_rep = 0, norep = 0;
    n_curve = 0; 
    n_interval = 0;
    spline_style = style;

    while(sy==INTEGER || sy=='@' || sy=='(' || sy=='[')
    {
	int l1, l2, step=1;

	if (sy=='(' || sy=='[')
	{
	    gr_point(curve[n_curve>0 ? n_curve-1:0].x,curve[n_curve>0 ? n_curve-1:0].y,
		&curve[n_curve].x,&curve[n_curve].y);
	    n_curve++;
	    norep = 1;
	    if (sy==',') (void)grsy();
	    continue;
	}
	else if (sy=='@')
	{
	    int pop = 0;

	    grsy();
	    if (sy=='-')
	    {
		pop = 1;
		grsy();
	    }

	    if (sy!=INTEGER) gr_expect(INTEGER,gr_intstr);
	    l1 = gr_lastpoint - (int)atoi(token) + 1;
	    l2 = gr_lastpoint;
	    grsy();

	    if (pop) gr_lastpoint = l1-1;
	}
	else if (sy==INTEGER)
	{
	    gr_range(&l1, &l2, &step);
	}
	if (sy==',') grsy();
	gr_check(l1);
	gr_check(l2);
	curve[n_curve++] = point[l1];
	if (n_rep<MAX_REPPOINTS) reppoint[n_rep++] = l1;
	else norep = 1;
	while(l1!=l2)
	{
	    if (l1<l2) l1 += step>l2-l1 ? l2-l1:step;
	    else l1 -= step>l1-l2 ? l1-l2:step;
	    curve[n_curve++] = point[l1];
	    if (n_rep<MAX_REPPOINTS) reppoint[n_rep++] = l1;
	    else norep = 1;
	}
    }
    
    while(sy==':')
    {
	grsy();
	if (n_interval>=MAX_INTERVAL)
	    gr_error("Error: Too much intervals");
	interval[n_interval].t0 = expr();
	gr_expect(DOTS,"..");
	interval[n_interval++].t1 = expr();
    }
    
    if (sy==IDENTIFIER || sy==FOR_SYM)
    {
	if (sy==FOR_SYM) (void)grsy();
	gr_defloops();
	gr_loop(loop,nloops,gr_lspline);
	gr_endloops();
    }
    else if (sy==REP_SYM)
    {
	if (norep) gr_error("Curve cannot be repeated");
	if (n_rep!=n_curve) gr_error("Internal error in repeated curves");
	(void)grsy();
	while(sy==INTEGER)
	{
	    int o0, o1, step, i;
	    gr_range(&o0, &o1, &step);
	    for (; o0<o1; o0+=step)
	    {
		for (i=0; i<n_curve; i++)
		{
		    gr_check(reppoint[i]+o0);
		    curve[i] = point[reppoint[i]+o0];
		}
		spline(curve,n_curve,interval,n_interval,style);
	    }
	    for (i=0; i<n_curve; i++)
	    {
		gr_check(reppoint[i]+o1);
		curve[i] = point[reppoint[i]+o1];
	    }
	    spline(curve,n_curve,interval,n_interval,style);
	    
	    if (sy==',') grsy();
	}
    }
    else spline(curve,n_curve,interval,n_interval,style);
}

void gr_defpattern(void)
{
    int i;
    for (i=0; i<MAX_PATTERN; i++)
    {
	if (sy!=INTEGER) gr_expect(INTEGER,gr_intstr);
	gr_patmask[i] = atoi(token);
	(void)grsy();
	if (sy==',') (void)grsy();
    }
}

void gr_input(void)
{
    if (sy!=STRINGSY) gr_expect(STRINGSY,"<string>");
    spec_input(token);
    (void)grsy();
}

void gr_command(int x, int y)
{
    real a, b, c;
    int n, i, j, s;

    grsy();

    if (setjmp(gr_abort))
	while(sy!=';' && sy!=EOF) grsy();

    while(sy!=EOF)
    {
	fmt_stop();
	while(sy==';') (void)grsy();
	if (sy==EOF) break;
	form_ptr = 0;
	switch(s=grsy())
	{
	    case SETPOINT_SYM: gr_setpoint((real)x,(real)y); break;
	    case SETUNITLENGTH_SYM:
		(void)gr_parameters(1,P_DIMEN,&graph_unit);
		break;
	    case SETLINEWIDTH_SYM:
		a = DEF_LINEWIDTH;
		(void)gr_parameters(1,P_DIMEN,&a);
		setlinewidth(a*hresconv);
		break;
	    case SETDASH_SYM:
		n = gr_parameters(-MAX_DASH,P_DIMEN,dash);
		for (i=0; i<n; i++)
		{
		    if (dash[i]<=0) dash[i]=1.0;
		    dash[i] *= hresconv;
		}
		setdash(dash,n);
		break;
	    case SETARROWSIZE_SYM:
		a = DEF_ARROWWIDTH; b = DEF_ARROWLENGTH;
		gr_parameters(2,P_DIMEN,&a,P_DIMEN,&b);
		setarrowsize(a*hresconv,b*hresconv);
		break;
	    case SETARROWSHAPE_SYM:
		a = DEF_ARROW_T; b = DEF_ARROW_L;
		gr_parameters(2,P_REAL,&a,P_REAL,&b);
		setarrowshape(a,b);
		break;
	    case SETARROWLINE_SYM:
		a = DEF_ARROWLINE;
		gr_parameters(1,P_DIMEN,&a);
		setarrowline(a*hresconv);
		break;
	    case SETDOTSIZE_SYM:
		a = DEF_DOTSIZE;
		gr_parameters(1,P_DIMEN,&a);
		setdotsize(a*hresconv);
		break;
	    case SETLINECAP_SYM:
		i = j = DEF_LINECAP;
		if (sy!=EOF && sy!=';') i = gr_linecap();
		if (sy!=EOF && sy!=';') j = gr_linecap();
		setlinecap(i,j);
		break;
	    case XROT_SYM:
	    case YROT_SYM:
	    case ZROT_SYM: 
		(void)gr_parameters(1, P_REAL, &a);
		gr_testleft();
		gr_rotate(matrix[matstack], s-XROT_SYM, a);
		break;
	    case TRANS_SYM:
		(void)gr_parameters(3, P_DIMEN, &a, P_DIMEN, &b, P_DIMEN, &c);
		gr_testleft();
		gr_translate(matrix[matstack], a, b, c);
		break;
	    case SCALE_SYM:
		(void)gr_parameters(3, P_REAL, &a, P_REAL, &b, P_REAL, &c);
		gr_testleft();
		gr_scalemat(matrix[matstack], a, b, c);
		break;
	    case SETDEFAULTS_SYM: gr_defaults(); break;
	    case DEFCLIP_SYM: clip_init(); break;
	    case CLIP_SYM: gr_clip(0); break;
	    case PATCLIP_SYM: gr_clip(gr_patmask); break;
	    case WHITECLIP_SYM: gr_clip(gr_whitemask); break;
	    case ENDCLIP_SYM: clip_end(); break;
	    case DEF_SYM: gr_def(); break;
	    case UNDEF_SYM: gr_undef(); break;
	    case INITMAT_SYM: gr_initmat(matrix[matstack]); break;
	    case INPUT_SYM: gr_input(); continue;
	    case DEFPAT_SYM: gr_defpattern(); break;
	    case PUSH_SYM:
		if (matstack<MAX_MATRIX-1) 
		{
		    matstack++;
		    memcpy(matrix[matstack], matrix[matstack-1], sizeof(mat_t));
		}
		else gr_error("Matrix Stack Overflow");
		break;
	    case POP_SYM:
		if (matstack>0) matstack--;
		else gr_error("Matrix Stack Underflow");
		break;
	    case DRAW_SYM + 0: gr_draw(0); break;
	    case DRAW_SYM + 1: gr_draw(1); break;
	    case DRAW_SYM + 2: gr_draw(2); break;
	    case DRAW_SYM + 3: gr_draw(3); break;
	    case DRAW_SYM + 4: gr_draw(4); break;
	    case DRAW_SYM + 5: gr_draw(5); break;
	    case DRAW_SYM + 6: gr_draw(6); break;
	    case DRAW_SYM + 7: gr_draw(7); break;
	    default: gr_expect(';',"<command>");
	}
	if (sy!=';' && sy!=EOF)
		gr_expect(';',(char*)NULL);
    }
}

