package Net::Async::Redis;
# ABSTRACT: Redis support for IO::Async

use Full::Class qw(:v1), extends => qw(Net::Async::Redis::Commands);

our $VERSION = '6.006';
our $AUTHORITY = 'cpan:TEAM'; # AUTHORITY

=head1 NAME

Net::Async::Redis - talk to Redis servers via L<IO::Async>

=head1 SYNOPSIS

    use Net::Async::Redis;
    use Future::AsyncAwait;
    use IO::Async::Loop;
    my $loop = IO::Async::Loop->new;
    $loop->add(my $redis = Net::Async::Redis->new);
    my $value = await $redis->get('some_key');
    $value ||= await $redis->set(some_key => 'some_value');
    print "Value: $value\n";

    # You can also use ->then chaining, see L<Future> for more details
    $redis->connect->then(sub {
        $redis->get('some_key')
    })->then(sub {
        my $value = shift;
        return Future->done($value) if $value;
        $redis->set(some_key => 'some_value')
    })->on_done(sub {
        print "Value: " . shift;
    })->get;

=head1 DESCRIPTION

Provides client access for dealing with Redis servers.

6ee L<Net::Async::Redis::Commands> for the full list of commands, this list
is autogenerated from the official documentation here:

L<https://redis.io/commands>

This is intended to be a near-complete low-level client module for asynchronous Redis
support. See L<Net::Async::Redis::Server> for a (limited) Perl server implementation.

This is an unofficial Perl port, and not endorsed by the Redis server maintainers in any
way.

=head2 Supported features

Current features include:

=over 4

=item * L<all commands|https://redis.io/commands> as of 8.0-M02 (November 2024), see L<https://redis.io/commands> for the methods and parameters

=item * L<pub/sub support|https://redis.io/topics/pubsub>, see L</METHODS - Subscriptions> including sharded pubsub

=item * L<pipelining|https://redis.io/topics/pipelining>, see L</pipeline_depth>

=item * L<transactions|https://redis.io/topics/transactions>, see L</METHODS - Transactions>

=item * L<streams|https://redis.io/topics/streams-intro> and consumer groups, via L<Net::Async::Redis::Commands/XADD> and related methods

=item * L<client-side caching|https://redis.io/topics/client-side-caching>, see L</METHODS - Clientside caching>

=item * L<RESP3/https://github.com/antirez/RESP3/blob/master/spec.md> protocol for Redis 6 and above, allowing pubsub on the same connection as regular commands

=item * cluster support via L<Net::Async::Redis::Cluster>, including key specifications from L<https://redis.io/docs/reference/key-specs/> to route commands to
the correct node(s)

=item * see L<Net::Async::Redis::XS> for a faster XS version (can be 40x faster than the pure Perl version, particularly when parsing large L<Net::Async::Redis::Commands/XREADGROUP> responses)

=back

=head2 Connecting

As with any other L<IO::Async::Notifier>-based module, you'll need to
add this to an L<IO::Async::Loop>:

    my $loop = IO::Async::Loop->new;
    $loop->add(
        my $redis = Net::Async::Redis->new
    );

then connect to the server:

    use Future::AsyncAwait;
    await $redis->connect;
    # You could achieve a similar result by passing client_name in
    # constructor or ->connect parameters
    await $redis->client_setname("example client");

=head2 Key-value handling

One of the most common Redis scenarios is as a key/value store. The L</get> and L</set>
methods are typically used here:

    use Future::AsyncAwait;
    await $redis->connect;
    $redis->set(some_key => 'some value');
    my ($value) = await $redis->get('some_key');
    print "Read back value [$value]\n";

See the next section for more information on what these methods are actually returning.

=head2 Requests and responses

Requests are implemented as methods on the L<Net::Async::Redis> object.
These typically return a L<Future> which will resolve once ready:

    my $future = $redis->incr("xyz")
        ->on_done(sub {
            print "result of increment was " . shift . "\n"
        });

For synchronous code, call C<< ->get >> on that L<Future>:

    print "Database has " . $redis->dbsize->get . " total keys\n";

This means you can end up with C<< ->get >> being called on the result of C<< ->get >>,
note that these are two very different methods:

 $redis
  ->get('some key') # this is being called on $redis, and is issuing a GET request
  ->get # this is called on the returned Future, and blocks until the value is ready

Typical async code would not be expected to use the L<Future/get> method extensively;
often only calling it in one place at the top level in the code.

=head3 RESP3 and RESP2 compatibility

In RESP3 some of the responses are structured differently from RESP2.
L<Net::Async::Redis> guarantees the same structure unless you have explicitly requested the new types
using the L</configure> C<hashrefs> option, which is disabled by default.

Generally RESP3 is recommended if you have Redis version 6 or later installed: it allows
subscription operations to share the same connection as regular Redis traffic.

=cut

=head2 Error handling

Since L<Future> is used for deferred results, failure is indicated
by a failing Future with L<failure category|Future/FAILURE-CATEGORIES>
of C<redis>.

The L<Future/catch> feature may be useful for handling these:

 $redis->lpush(key => $value)
     ->catch(
         redis => sub { warn "probably an incorrect type, cannot push value"; Future->done }
     )->get;

Note that this module uses L<Future::AsyncAwait> internally.

=cut

use Future::Queue;
use IO::Async::Stream;
use Ryu::Async;
use URI;
use URI::redis;
use Cache::LRU;

use Metrics::Any qw($metrics), strict => 0;

use List::Util qw(pairmap);
use Scalar::Util qw(reftype blessed refaddr);

use Net::Async::Redis::Multi;
use Net::Async::Redis::Subscription;
use Net::Async::Redis::Subscription::Message;

=head1 CONSTANTS

=head2 OPENTRACING_ENABLED

Defaults to false, this can be controlled by the C<USE_OPENTRACING>
environment variable. This provides a way to set the default opentracing
mode for all L<Net::Async::Redis> instances - you can enable/disable
for a specific instance via L</configure>:

 $redis->configure(opentracing => 1);

When enabled, this will create a span for every Redis request. See
L<OpenTracing::Any> for details.

=cut

use constant OPENTRACING_ENABLED   => $ENV{USE_OPENTRACING} // 0;

=head2 OPENTELEMETRY_ENABLED

Defaults to false, this can be controlled by the C<USE_OPENTELEMETRY>
environment variable. This provides a way to set the default C<opentelemetry>
mode for all L<Net::Async::Redis> instances - you can enable/disable
for a specific instance via L</configure>:

 $redis->configure(opentelemetry => 1);

When enabled, this will create a span for every Redis request. See
L<OpenTelemetry> or L<https://opentelemetry.io> for details.

=cut

use constant OPENTELEMETRY_ENABLED => $ENV{USE_OPENTELEMETRY} // 0;

our $provider;
our $tracer;
BEGIN {
    if(OPENTRACING_ENABLED) {
        require OpenTracing::Any;
        OpenTracing::Any->import(qw($tracer));
    }

    if(OPENTELEMETRY_ENABLED) {
        require OpenTelemetry;
        require OpenTelemetry::Context;
        require OpenTelemetry::Trace;
        require OpenTelemetry::Constants;
        OpenTelemetry::Constants->import(qw( SPAN_STATUS_ERROR SPAN_STATUS_OK ));
        $provider = OpenTelemetry->tracer_provider;
        $tracer = $provider->tracer(
            name => __PACKAGE__,
            version => __PACKAGE__->VERSION
        );
    }
}

# These only apply to the legacy RESP2 protocol. Since RESP3, connections
# are no longer restricted once pubsub activity has started.
our %ALLOWED_SUBSCRIPTION_COMMANDS = (
    SUBSCRIBE    => 1,
    PSUBSCRIBE   => 1,
    UNSUBSCRIBE  => 1,
    PUNSUBSCRIBE => 1,
    PING         => 1,
    QUIT         => 1,
);

# Any of these commands would necessitate switching a RESP2 connection into
# a limited pubsub-only mode.
our %SUBSCRIPTION_COMMANDS = (
    SUBSCRIBE    => 1,
    PSUBSCRIBE   => 1,
    UNSUBSCRIBE  => 1,
    PUNSUBSCRIBE => 1,
    MESSAGE      => 1,
    PMESSAGE     => 1,
);
our %COMMAND_DEFINITION = %Net::Async::Redis::Commands::COMMAND_DEFINITION;

# Add support for secure Redis `rediss://...` URIs
unless(URI::rediss->can('new')) {
    push @URI::rediss::ISA, qw(URI::redis);
    *URI::rediss::secure = sub { 1 };
}

=head1 METHODS

B<NOTE>: For a full list of the Redis methods supported by this module,
please see L<Net::Async::Redis::Commands>.

=cut

=head2 configure

Applies configuration parameters - currently supports:

=over 4

=item * C<host>

=item * C<port>

=item * C<auth>

=item * C<database>

=item * C<pipeline_depth>

=item * C<stream_read_len>

=item * C<stream_write_len>

=item * C<on_disconnect>

=item * C<client_name>

=item * C<opentracing>

=item * C<opentelemetry>

=item * C<protocol> - either 'resp2' or 'resp3', default is autodetect

=item * C<hashrefs> - RESP3 (Redis 6.0+) supports more data types, currently the only difference this
makes to us is that it now supports hashrefs for key/value pairs. This is disabled by default to ensure
compatibility across newer+older versions.

=back

Note that enabling C<hashrefs> will cause connections to fail if the server does not support RESP3.

=cut

method configure (%args) {
    for (qw(
        host
        port
        username
        auth
        database
        pipeline_depth
        stream_read_len
        stream_write_len
        on_disconnect
        client_name
        opentracing
        opentelemetry
        protocol
        hashrefs
        tls_cert_file
        tls_key_file
        tls_ca_file
    )) {
        $self->{$_} = delete $args{$_} if exists $args{$_};
    }
    die 'invalid protocol requested: ' . $self->{protocol} if defined $self->{protocol} and not $self->{protocol} =~ /^resp[23]$/;

    # Be more lenient with the URI parameter, since it's tedious to
    # need the redis:// prefix every time... after all, what else
    # would we expect it to be?
    if(exists $args{uri}) {
        my $uri = delete $args{uri};
        $uri = "redis://$uri" unless ref($uri) or $uri =~ /^rediss?:/;
        $self->{uri} = $uri;
    }

    if(exists $args{client_side_cache_size}) {
        $self->{client_side_cache_size} = delete $args{client_side_cache_size};
        delete $self->{client_side_cache};
        if($self->loop) {
            my $conn = delete $self->{client_side_connection};
            $conn->remove_from_parent if $conn;
        }
    }
    my $uri = $self->{uri} = URI->new($self->{uri}) unless ref $self->uri;
    if($uri) {
        $self->{host} //= $uri->host;
        $self->{port} //= $uri->port;
    }

    die 'hashref support requires RESP3 (Redis version 6+)' if defined $self->{protocol} and $self->{protocol} eq 'resp2' and $self->{hashrefs};

    die 'opentelemetry requested but not available, set USE_OPENTELEMETRY=1 in the environment to enable' if $self->opentelemetry and not OPENTELEMETRY_ENABLED;
    $self->next::method(%args)
}

=head2 host

Returns the host or IP address for the Redis server.

=cut

method host { $self->{host} }

=head2 port

Returns the port used for connecting to the Redis server.

=cut

method port { $self->{port} }

=head2 database

Returns the database index used when connecting to the Redis server.

See the L<Net::Async::Redis::Commands/select> method for details.

=cut

method database { $self->{database} }

=head2 uri

Returns the Redis endpoint L<URI> instance.

=cut

method uri { $self->{uri} //= URI->new('redis://localhost') }

=head2 stream_read_len

Returns the buffer size when reading from a Redis connection.

Defaults to 1MB, reduce this if you're dealing with a lot of connections and
want to minimise memory usage. Alternatively, if you're reading large amounts
of data and spend too much time in needless C<epoll_wait> calls, try a larger
value.

=cut

method stream_read_len { $self->{stream_read_len} //= 1048576 }

=head2 stream_write_len

Returns the buffer size when writing to Redis connections, in bytes. Defaults to 1MB.

See L</stream_read_len>.

=cut

method stream_write_len { $self->{stream_write_len} //= 1048576 }

=head2 client_name

Returns the name used for this client when connecting.

=cut

method client_name { $self->{client_name} }

=head1 METHODS - Connection

=head2 connect

Connects to the Redis server.

Will use the L</configure>d parameters if available, but as a convenience
can be passed additional parameters which will then be applied as if you
had called L</configure> with those beforehand. This also means that they
will be preserved for subsequent L</connect> calls.

=cut

method connect (%args) {
    return $self->{connection} if $self->{connection};
    $self->{connection_in_progress} = 1;
    my $f = $self->connect_to_server(%args)->on_ready(sub ($f) {
        $log->tracef('connection call complete - %s', $f->state);
        $self->{connection_in_progress} = 0;
        delete $self->{connection} unless $f->is_done
    });
    return $self->{connection} = $f;
}

async method connect_to_server (%args) {
    $self->configure(%args) if %args;
    my $uri = $self->uri->clone;
    for (qw(host port)) {
        $uri->$_($self->$_) if defined $self->$_;
    }

    # 0 is the default anyway, no need to apply in that case
    $uri->path('/' . $self->database) if $self->database;

    my $auth = $self->{auth};
    # Our configured values are used in preference to the URI,
    # to support separation of sensitive information (pull URI
    # from config with defaults for user/password, then override
    # those with secret values taken from another source).
    my $username = $self->{username} // 'default';
    if(!defined($auth) and defined($uri->userinfo)) {
        (my $userpart, $auth) = split ':', $uri->userinfo, 2;
        $username = $userpart if length $userpart;
    }
    $log->tracef('About to start connection to %s', "$uri");
    my $tls = $self->{tls} // $uri->secure // 0;
    require IO::Async::SSL if $tls;
    await $self->loop->later;

    my $sock = await $self->loop->connect(
        service => $uri->port // 6379,
        host    => $uri->host,
        socktype => 'stream',
        $tls ? (
            extensions => ['SSL'],
            $self->ssl_options->%*,
        ) : (),
    );
    $self->{endpoint} = join ':', $sock->peerhost, $sock->peerport;
    $self->{local_endpoint} = join ':', $sock->sockhost, $sock->sockport;
    my $proto = $self->wire_protocol;
    my $stream = IO::Async::Stream->new(
        handle              => $sock,
        read_len            => $self->stream_read_len,
        write_len           => $self->stream_write_len,
        # Arbitrary multipliers for our stream values,
        # in a memory-constrained environment it's expected
        # that ->stream_read_len would be configured with
        # low enough values for this not to be a concern.
        read_high_watermark => 16 * $self->stream_read_len,
        read_low_watermark  => 2 * $self->stream_read_len,
        on_closed           => $self->curry::weak::notify_close,
        on_read             => sub {
            $proto->parse($_[1]);
            0
        }
    );
    $self->add_child($stream);
    Scalar::Util::weaken(
        $self->{stream} = $stream
    );

    try {
        # Pretend we tried and failed if the old version was specifically requested
        die 'ERR unknown command' if $self->{protocol} and $self->{protocol} eq 'resp2';

        # Try issuing a HELLO to detect RESP3 or above
        await $self->hello(
            3, defined($auth) ? (
                qw(AUTH), $username, $auth
            ) : (), defined($self->client_name) ? (
                qw(SETNAME), $self->client_name
            ) : ()
        );
        $log->tracef('RESP3 detected');
        $self->{protocol_level} = 'resp3';

        $proto->{hashrefs} = $self->{hashrefs};
        $proto->{protocol} = $self->{protocol_level};
    } catch ($e) {
        # If we had an auth failure or invalid client name, all bets are off:
        # immediately raise those back to the caller
        die $e unless $e =~ /ERR unknown command/;

        $log->tracef('Older Redis version detected, dropping back to RESP2 protocol');
        $self->{protocol_level} = 'resp2';

        die 'extended data structures require RESP3 (Redis version 6+)' if $self->{hashrefs};

        $proto->{hashrefs} = $self->{hashrefs};
        $proto->{protocol} = $self->{protocol_level};

        await $self->auth($auth) if defined $auth;
        await $self->client_setname($self->client_name) if defined $self->client_name;
    }

    if($uri->database) {
        try {
            $log->tracef('Select database %s', $uri->database);
            await $self->select($uri->database);
        } catch($e) {
            die 'Failed to switch database on Redis connection - ' . $e;
        }
    }
    if($self->is_client_side_cache_enabled) {
        try {
            $log->tracef('Client-side cache requested');
            await $self->client_side_connection;
            $log->tracef('Client-side connection established');
        } catch($e) {
            die 'This version of Redis does not support clientside caching' if $e =~ /Unknown subcommand .*tracking/i;
            $log->errorf('Clientside cache setup failure in ->connect - %s', $e);
            die 'Failed to enable clientside caching on Redis connection - ' . $e;
        }
    }
    return $stream;
}

=head2 connected

Establishes a connection if needed, otherwise returns an immediately-available
L<Future> instance.

=cut

method connected {
    return $self->{connection} if $self->{connection};
    $self->connect;
}

=head2 endpoint

The string describing the remote endpoint.

=cut

method endpoint { $self->{endpoint} }

=head2 local_endpoint

A string describing the local endpoint, usually C<host:port>.

=cut

method local_endpoint { $self->{local_endpoint} }

=head1 METHODS - Subscriptions

See L<https://redis.io/topics/pubsub> for more details on this topic.
There's also more details on the internal implementation in Redis here:
L<https://making.pusher.com/redis-pubsub-under-the-hood/>.

B<NOTE>: On Redis versions prior to 6.0, you will need a I<separate> connection
for subscriptions; you cannot share a connection for regular requests once
any of the L</subscribe> or L</psubscribe> methods have been called on an
existing connection.

With Redis 6.0, a newer protocol version (RESP3) is used by default, and
this is quite happy to support pubsub activity on the same connection
as other traffic.

=cut

=head2 psubscribe

Subscribes to a pattern.

Example:

 # Subscribe to 'info::*' channels, i.e. any message
 # that starts with the C<info::> prefix, and prints them
 # with a timestamp.
 $redis_connection->psubscribe('info::*')
    ->then(sub {
        my $sub = shift;
        $sub->map('payload')
            ->each(sub {
             print localtime . ' ' . $_ . "\n";
            })->retain
    })->get;
 # this will block until the subscribe is confirmed. Note that you can't publish on
 # a connection that's handling subscriptions due to Redis protocol restrictions.
 $other_redis_connection->publish('info::example', 'a message here')->get;

Returns a L<Future> which resolves to a L<Net::Async::Redis::Subscription> instance.

=cut

async method psubscribe ($pattern) {
    $self->{pending_subscription_pattern_channel}{$pattern} //= $self->future('pattern_subscription[' . $pattern . ']');
    await $self->execute_command(qw(PSUBSCRIBE), $pattern);
    $self->{pubsub} //= 0;
    return $self->{subscription_pattern_channel}{$pattern} //= Net::Async::Redis::Subscription->new(
        redis   => $self,
        channel => $pattern
    )
}

=head2 subscribe

Subscribes to one or more channels.

Returns a L<Future> which resolves to a L<Net::Async::Redis::Subscription> instance.

Example:

 # Subscribe to 'notifications' channel,
 # print the first 5 messages, then unsubscribe
 $redis->subscribe('notifications')
    ->then(sub {
        my $sub = shift;
        $sub->events
            ->map('payload')
            ->take(5)
            ->say
            ->completed
    })->then(sub {
        $redis->unsubscribe('notifications')
    })->get

=cut

async method subscribe (@channels) {
    my @pending = map {
        $self->{pending_subscription_channel}{$_} //= $self->future('subscription[' . $_ . ']')
    } @channels;
    await $self->execute_command(qw(SUBSCRIBE), @channels);
    $self->{pubsub} //= 0;
    await Future->wait_all(@pending);
    $log->tracef('Subscriptions established, we are go');
    return @{$self->{subscription_channel}}{@channels};
}

=head2 ssubscribe

Subscribes to one or more sharded channels.

This behaves similarly to L</subscribe>, but applies to messages received on a specific
shard. This is mostly relevant in a cluster context, where subscriptions can be localised
to one shard (group of nodes) in the cluster to improve performance.

More details are in the L<sharded pubsub documentation|https://redis.io/topics/pubsub#sharded-pubsub>.

Returns a L<Future> which resolves to a L<Net::Async::Redis::Subscription> instance.

Example:

 # Subscribe to 'notifications' channel,
 # print the first 5 messages, then unsubscribe
 $redis->subscribe('notifications')
    ->then(sub {
        my $sub = shift;
        $sub->events
            ->map('payload')
            ->take(5)
            ->say
            ->completed
    })->then(sub {
        $redis->unsubscribe('notifications')
    })->get

=cut

async method ssubscribe (@channels) {
    my @pending = map {
        $self->{pending_subscription_channel}{$_} //= $self->future('subscription[' . $_ . ']')
    } @channels;
    await $self->execute_command(qw(SSUBSCRIBE), @channels);
    $self->{pubsub} //= 0;
    await Future->wait_all(@pending);
    $log->tracef('Subscriptions established, we are go');
    return @{$self->{subscription_channel}}{@channels};
}

=head1 METHODS - Transactions

=head2 multi

Executes the given code in a Redis C<MULTI> transaction.

This will cause each of the requests to be queued on the server, then applied in a single
atomic transaction.

Note that the commands will resolve only after the transaction is committed: for example,
when the L</set> command is issued, Redis will return C<QUEUED>. This information
is B<not> used as the result - we only pass through the immediate
response if there was an error. The L<Future> representing
the response will be marked as done once the C<EXEC> command is applied and we have the
results back.

Example:

 $redis->multi(sub {
  my $tx = shift;
  $tx->incr('some::key')->on_done(sub { print "Final value for incremented key was " . shift . "\n"; });
  $tx->set('other::key => 'test data')
 })->then(sub {
  my ($success, $failure) = @_;
  return Future->fail("Had $failure failures, expecting everything to succeed") if $failure;
  print "$success succeeded\m";
  return Future->done;
 })->retain;

=cut

async method multi ($code) {
    die 'Need a coderef' unless $code and reftype($code) eq 'CODE';

    my $multi = Net::Async::Redis::Multi->new(
        redis => $self,
    );
    await $self->execute_command(qw(MULTI));
    return await $multi->exec($code);
}

=head1 METHODS - Clientside caching

Enable clientside caching by passing a true value for C<client_side_caching_enabled> in
L</configure> or L</new>. This is currently B<experimental>, and only operates on
L<Net::Async::Redis::Commands/get> requests.

See L<https://redis.io/topics/client-side-caching> for more details on this feature.

=cut

async method client_side_connection {
    return if $self->{client_side_connection};

    if($self->{protocol_level} eq 'resp3') {
        $log->tracef('Client side cache uses same connection due to RESP3');
        Scalar::Util::weaken($self->{client_side_connection} = $self);
        await $self->enable_clientside_cache($self);
        return;
    }

    $log->tracef('Client side cache needs a new connection due to RESP2');
    $self->{client_side_connection} = my $redis = ref($self)->new(
        host => $self->host,
        port => $self->port,
        auth => $self->{auth},
    );
    $self->add_child($redis);
    await $redis->connect;
    await $self->enable_clientside_cache($redis);
    return;
}

=head2 clientside_cache_events

A L<Ryu::Source> which emits key names as they are invalidated.

With client-side caching enabled, can be used to monitor which keys
are changing.

=cut

method clientside_cache_events {
    $self->{clientside_cache_events} // die 'no client-side cache available yet'
}

=head2 client_side_cache_ready

Returns a L<Future> representing the client-side cache connection status,
if there is one.

=cut

method client_side_cache_ready {
    my $f = $self->{client_side_cache_ready} or return Future->fail('client-side cache is not enabled');
    return $f->without_cancel;
}

=head2 client_side_cache

Returns the L<Cache::LRU> instance used for the client-side cache.

=cut

method client_side_cache {
    $self->{client_side_cache} //= Cache::LRU->new(
        size => $self->client_side_cache_size,
    );
}

=head2 is_client_side_cache_enabled

Returns true if the client-side cache is enabled.

=cut

method is_client_side_cache_enabled { ($self->{client_side_cache_size} // 0) > 0 }

=head2 client_side_cache_size

Returns the current client-side cache size, as a number of entries.

=cut

method client_side_cache_size { $self->{client_side_cache_size} }

# For now, we're only caching the GET/SET requests. Client-side caching
# support in the Redis server covers other commands, though: eventually
# we'll be extending this for all read commands.
async method get ($k) {
    return await $self->execute_command(qw(GET), $k) unless $self->is_client_side_cache_enabled;

    my $cache = $self->client_side_cache;
    $log->tracef('Check cache for [%s]', $k);

    my $v = $cache->get($k);
    if(ref $v) {
        $log->tracef('Key [%s] already being requested, joining', $k);
        return await $v->without_cancel;
    } else {
        return $v if exists $cache->{_entries}{$k};
    }

    $log->tracef('Key [%s] was not cached', $k);
    my $f = $self->execute_command(qw(GET), $k);
    # Set our cache entry regardless of whether it completes
    # immediately or not...
    $cache->set(
        $k => $f
    );
    # ... and then rewrite or remove the cache entry once we get a response:
    $f->on_ready(sub {
        if($f->is_done) {
            # If we had an invalidation message, we may have removed
            # our cache entry already: we shouldn't cache this value if so.
            $cache->set($k => $f->get)
                if exists $cache->{_entries}{$k};
        } else {
            # Clear up after any failure or cancellation,
            # since they may be temporary
            $cache->remove($k);
        }
    });
    return await $f;
};


=head1 METHODS - Generic

=head2 keys

=cut

method keys ($match) {
    $match //= '*';
    return $self->execute_command(qw(KEYS), $match);
}

=head2 watch_keyspace

A convenience wrapper around the keyspace notifications API.

Provides the necessary setup to establish a C<PSUBSCRIBE> subscription
on the C<__keyspace@*__> namespace, setting the configuration required
for this to start emitting events, and then calls C<$code> with each
event.

Note that this will switch the connection into pubsub mode on versions
of Redis older than 6.0, so it will no longer be available for any
other activity. This limitation does not apply on Redis 6 or above.

Use C<*> to listen for all keyspace changes.

Resolves to a L<Ryu::Source> instance.

=cut

async method watch_keyspace ($pattern, $code) {
    $pattern //= '*';
    my $sub_name = '__keyspace@*__:' . $pattern;
    $self->{have_notify} ||= await $self->config_set(
        'notify-keyspace-events', 'Kg$xe'
    );
    my $sub = await $self->psubscribe($sub_name);
    my $ev = $sub->events;
    $ev->each(sub {
        my $message = $_;
        $log->tracef('Keyspace notification for channel %s, type %s, payload %s', map $message->$_, qw(channel type payload));
        my $k = $message->channel;
        $k =~ s/^[^:]+://;
        my $f = $code->($message->payload, $k);
        $self->adopt_future($f) if blessed($f) and $f->isa('Future');
    }) if $code;
    return $ev;
}

=head2 pipeline_depth

Number of requests awaiting responses before we start queuing.
This defaults to an arbitrary value of 100 requests.

Note that this does not apply when in L<transaction|METHODS - Transactions> (C<MULTI>) mode.

See L<https://redis.io/topics/pipelining> for more details on this concept.

=cut

method pipeline_depth { $self->{pipeline_depth} //= 100 }

=head2 opentracing

Indicates whether L<OpenTracing::Any> support is enabled.

=cut

method opentracing { $self->{opentracing} }

=head2 opentelemetry

Indicates whether L<OpenTelemetry> support is enabled.

=cut

method opentelemetry { $self->{opentelemetry} }

=head1 METHODS - Deprecated

This are still supported, but no longer recommended.

=cut

method bus {
    $self->{bus} //= do {
        require Mixin::Event::Dispatch::Bus;
        Mixin::Event::Dispatch::Bus->VERSION(2.000);
        Mixin::Event::Dispatch::Bus->new
    }
}

=head1 METHODS - Internal

=cut

=head2 on_message

Called for each incoming message.

Passes off the work to L</handle_pubsub_message> or the next queue
item, depending on whether we're dealing with subscriptions at the moment.

=cut

method on_message ($data) {
    $log->tracef('Incoming message: %s, pending = %s', $data, join ',', map { $_->[0] } $self->{pending}->@*) if $log->is_trace;

    if($self->{protocol_level} eq 'resp2' and exists $self->{pubsub} and exists $SUBSCRIPTION_COMMANDS{uc $data->[0]}) {
        return $self->handle_pubsub_message(@$data);
    }

    return $self->complete_message($data);
}

method complete_message ($data) {
    my $next = shift @{$self->{pending}} or die "No pending handler";
    $self->next_in_pipeline if @{$self->{awaiting_pipeline}};

    # This shouldn't happen, preferably
    $log->errorf("our [%s] entry is ready, original was [%s]??", $data, $next->[0]) if $next->[1]->is_ready;
    $next->[1]->done($data);
    return;
}

=head2 next_in_pipeline

Attempt to process next pending request when in pipeline mode.

=cut

method next_in_pipeline {
    my $depth = $self->pipeline_depth;
    until($depth and $self->{pending}->@* >= $depth) {
        return unless my $next = shift @{$self->{awaiting_pipeline}};
        my $cmd = join ' ', @{$next->[0]};
        $log->tracef("Have free space in pipeline, sending %s", $cmd);
        push @{$self->{pending}}, [ $cmd, $next->[1] ];
        my $data = $self->wire_protocol->encode_from_client(@{$next->[0]});
        $self->stream->write($data);
    }
    # Ensure last ->write is in void context
    return;
}

=head2 on_error_message

Called when there's an error response.

=cut

method on_error_message ($data) {
    $log->tracef('Incoming error message: %s', $data);

    my $next = shift @{$self->{pending}} or die "No pending handler";
    $next->[1]->fail($data);
    $self->next_in_pipeline if @{$self->{awaiting_pipeline}};
    return;
}

=head2 handle_pubsub_message

Deal with an incoming pubsub-related message.

=cut

method handle_pubsub_message ($type, @details) {
    $type = lc $type;
    if($type eq 'message' or $type eq 'smessage') {
        my ($channel, $payload) = @details;
        if(my $sub = $self->{subscription_channel}{$channel}) {
            my $msg = Net::Async::Redis::Subscription::Message->new(
                type         => $type,
                channel      => $channel,
                payload      => $payload,
                redis        => $self,
                subscription => $sub
            );
            $sub->events->emit($msg);
        } else {
            $log->warnf('Have message for unknown channel [%s]', $channel);
        }
        $self->bus->invoke_event(message => [ $type, $channel, $payload ]) if exists $self->{bus};
        return;
    }
    if($type eq 'pmessage') {
        my ($pattern, $channel, $payload) = @details;
        if(my $sub = $self->{subscription_pattern_channel}{$pattern}) {
            my $msg = Net::Async::Redis::Subscription::Message->new(
                type         => $type,
                pattern      => $pattern,
                channel      => $channel,
                payload      => $payload,
                redis        => $self,
                subscription => $sub
            );
            $sub->events->emit($msg);
        } else {
            $log->warnf('Have message for unknown channel [%s]', $channel);
        }
        $self->bus->invoke_event(message => [ $type, $channel, $payload ]) if exists $self->{bus};
        return;
    }
    if($type =~ /invalidate$/) {
        my ($channel) = @details;
        for my $k ($channel->@*) {
            $log->tracef('have invalidation type %s with channel %s', $type, $k);
            $self->clientside_cache_events->emit($k);
        }
        return;
    }

    # Looks like this isn't a message, it's a response to (un)subscribe
    return $self->handle_pubsub_response($type, @details);
}

method handle_pubsub_response ($type, @details) {
    my ($channel, $payload) = @details;
    $type = lc $type;
    my $k = (substr $type, 0, 1) eq 'p' ? 'subscription_pattern_channel' : 'subscription_channel';
    if($type =~ /unsubscribe$/) {
        --$self->{pubsub};
        if(my $sub = delete $self->{$k}{$channel}) {
            $log->tracef('Removed subscription for [%s]', $channel);
        } else {
            $log->warnf('Have unsubscription for unknown channel [%s]', $channel);
        }
    } elsif($type =~ /subscribe$/) {
        $log->tracef('Have %s subscription for [%s]', (exists $self->{$k}{$channel} ? 'existing' : 'new'), $channel);
        ++$self->{pubsub};
        $self->{$k}{$channel} //= Net::Async::Redis::Subscription->new(
            redis => $self,
            channel => $channel
        );
        $self->{'pending_' . $k}{$channel}->done($payload) unless $self->{'pending_' . $k}{$channel}->is_done;
    } else {
        $log->warnf('have unknown pubsub message type %s with channel %s payload %s', $type, $channel, $payload);
    }

    return $self->complete_message([ $type, @details ]) unless $self->{protocol_level} eq 'resp2';
}

=head2 stream

Represents the L<IO::Async::Stream> instance for the active Redis connection.

=cut

method stream { $self->{stream} }

=head2 notify_close

Called when the socket is closed.

=cut

method notify_close {
    # Also clear our connection future so that the next request is triggered appropriately
    my $conn = delete $self->{connection};
    $self->{connection}->fail if $self->{connection} and not $self->{connection}->is_ready;

    # If we think we have an existing connection, it needs removing:
    # there's no guarantee that it's in a usable state.
    if(my $stream = delete $self->{stream}) {
        $stream->close_now;
    }

    # Clear out anything in the pending queue - we normally wouldn't expect anything to
    # have ready status here, but no sense failing on a failure. Note that we aren't
    # filtering out the list via grep because some of these Futures may be interdependent.
    $_->[1]->fail(
        'Server connection is no longer active',
        redis => 'disconnected'
    ) for grep { !$_->[1]->is_ready } splice @{$self->{pending}};

    # Subscriptions also need clearing up
    $_->cancel for values %{$self->{subscription_channel}};
    $self->{subscription_channel} = {};
    $_->cancel for values %{$self->{subscription_pattern_channel}};
    $self->{subscription_pattern_channel} = {};

    $self->maybe_invoke_event(disconnect => );
}

=head2 command_label

Generate a label for the given command list.

=cut

method command_label (@cmd) {
    return join ' ', @cmd if $cmd[0] eq 'KEYS';
    return $cmd[0];
}

=head2 span_for_future

See L<https://opentelemetry.io/docs/specs/semconv/database/redis/> for current semantic conventions around Redis.

=cut

method span_for_future ($f, %args) {
    return $f unless OPENTELEMETRY_ENABLED;

    $args{name} //= $f->label // 'Future';
    my $span = $tracer->create_span(
        %args,
        parent => OpenTelemetry::Context->current
    );

    my $context = OpenTelemetry::Trace->context_with_span($span);

    return $f->on_ready(sub {
        dynamically OpenTelemetry::Context->current = $context;
        if($f->is_done) {
            $span->set_status(
                OpenTelemetry::Constants->SPAN_STATUS_OK
            );
        } elsif($f->is_cancelled) {
            $span->set_status(
                OpenTelemetry::Constants->SPAN_STATUS_OK
            );
        } else {
            my $e = $f->failure;
            $span->record_exception($e);
            $span->set_status(
                OpenTelemetry::Constants->SPAN_STATUS_ERROR,
                $e
            );
        }
        $span->end;
    });
}

=head2 execute_command

Queues the given command for execution.

=cut

method execute_command (@cmd) {
    # This represents the completion of the command
    my $f = $self->loop->new_future->set_label(
        $self->command_label(@cmd)
    );
    $tracer->span_for_future($f) if $self->opentracing;
    $self->span_for_future(
        $f,
        attributes => {
            'db.system'               => 'redis',
            'db.redis.database_index' => $self->database,
            'db.statement'            => join(' ', map { /\s/ ? qq{"$_"} : $_ } @cmd),
        }
    ) if OPENTELEMETRY_ENABLED && $self->opentelemetry;

    my $item = [ \@cmd, $f];
    if($self->{connection_in_progress}) {
        $self->handle_command($item);
        return $f->retain;
    }
    my $queue = $self->{_is_multi} // $self->command_queue;
    # We register this as a command we want to run - it'll be
    # sent to the server once nothing else is in the way.
    # We currently don't support cancellation - any attempt to
    # do so will be ignored, the command will still be executed.
    return $queue->push($item)->then(sub { $f })->retain->without_cancel;
}

method command_queue {
    return $self->{command_queue} if $self->{command_queue};
    $self->{command_queue} = my $queue = Future::Queue->new(
        (
            $self->pipeline_depth
            ? (max_items => $self->pipeline_depth)
            : ()
        ),
        prototype => $self->future
    );
    $self->{command_processing} ||= $self->command_processing->on_ready(sub { delete $self->{command_processing} });
    return $queue;
}

async method command_processing {
    while(1) {
        await $self->connected;

        # An active MULTI always takes priority over regular commands
        if(my $queue = $self->{multi_queue}) {
            try {
                while(my $next = await $queue->shift) {
                    $self->handle_command($next);
                }
            } catch ($e) {
                $log->errorf('Failed processing MULTI commands - %s', $e);
                delete $self->{multi_queue};
                die $e;
            }
            delete $self->{multi_queue};
        } else {
            my $queue = $self->{command_queue};
            my $next = await $queue->shift
                or last;
            $self->handle_command($next);
        }
    }
}

method handle_command ($details) {
    my @cmd = $details->[0]->@*;
    my $f = $details->[1];
    if($f->is_ready) {
        $log->tracef('Ignoring command %s since it is marked as ', \@cmd, $f->state);
        return $f;
    }

    # First, the rules: pubsub or plain
    my $is_sub_command = (
        $self->{protocol_level} eq 'resp2' and exists $SUBSCRIPTION_COMMANDS{$cmd[0]}
    );

    return $f->fail(
        'Currently in pubsub mode, cannot send regular commands until unsubscribed',
        redis =>
            0 + (keys %{$self->{subscription_channel}}),
            0 + (keys %{$self->{subscription_pattern_channel}})
    ) if $self->{protocol_level} ne 'resp3' and exists $self->{pubsub} and not exists $ALLOWED_SUBSCRIPTION_COMMANDS{$cmd[0]};

    my $cmd = join ' ', @cmd;

    $log->tracef('Outgoing [%s]', $cmd);
    my $data = $self->wire_protocol->encode_from_client(@cmd);
    push @{$self->{pending}}, [ $cmd, $f ];

    # Void-context write allows IaStream to combine multiple writes on the same connection.
    $self->stream->write($data);
    if(lc($cmd[0]) eq 'multi') {
        die 'Already processing MULTI, cannot start a new one' if $self->{multi_queue};
        $self->{multi_queue} = Future::Queue->new(
            (
                $self->pipeline_depth
                ? (max_items => $self->pipeline_depth)
                : ()
            ),
            prototype => $self->future
        );
    }
    return;
}

async method xread (@args) {
    my $response = await $self->execute_command(qw(XREAD), @args);
    return [] unless ref $response;

    # protocol_level is detected while connecting checking before this point is wrong.
    return $response if $self->{protocol_level} eq 'resp2' || $self->{hashrefs};

    my $compatible_response = [ pairmap { [ $a, $b ] } $response->@* ];
    $log->tracef('Transformed response of xread/xreadgroup into RESP2 format: from %s to %s', $response, $compatible_response);

    return $compatible_response;
}

async method xreadgroup (@args) {
    my $response = await $self->execute_command(qw(XREADGROUP), @args);
    return [] unless ref $response;

    # protocol_level is detected while connecting checking before this point is wrong.
    return $response if $self->{protocol_level} eq 'resp2' || $self->{hashrefs};

    my $compatible_response = [ pairmap { [ $a, $b ] } $response->@* ];
    $log->tracef('Transformed response of xread/xreadgroup into RESP2 format: from %s to %s', $response, $compatible_response);

    return $compatible_response;
}

# These have different behaviours depending on whether we use the RESP3
# data structures (hashes etc.) or original RESP2 everything-is-an-array.
#for my $method (qw(zrange zrangebyscore zrevrange zrevrangebyscore)) {
#    around $method => async sub {
#        my ($code, $self, @args) = @_;
#        return await $self->$code(@args) if $self->{hashrefs};
#
#        my $response = await $self->$code(@args);
#        return $response unless ref $response->[0] eq 'ARRAY';
#
#        my $compatible_response = [ map { $_->@* } $response->@* ];
#        $log->tracef(
#            'Transformed response of %s into RESP2 format: from %s, to %s',
#            $method,
#            $response,
#            $compatible_response
#        );
#
#        return $compatible_response;
#    };
#}

=head2 ryu

A L<Ryu::Async> instance for source/sink creation.

=cut

method ryu {
    $self->{ryu} ||= do {
        $self->add_child(
            my $ryu = Ryu::Async->new
        );
        $ryu
    }
}

=head2 future

Factory method for creating new L<Future> instances.

=cut

method future (@args) {
    return $self->loop->new_future(@args);
}

=head2 wire_protocol

Returns the L<Net::Async::Redis::Protocol> instance used for
encoding and decoding messages.

=cut

method wire_protocol {
    $self->{wire_protocol} ||= do {
        require Net::Async::Redis::Protocol;
        Net::Async::Redis::Protocol->new(
            handler  => $self->curry::weak::on_message,
            pubsub   => $self->curry::weak::handle_pubsub_message,
            error    => $self->curry::weak::on_error_message,
            protocol => $self->{protocol_level} || 'resp3',
        )
    };
}

=head2 enable_clientside_cache

Used internally to prepare for client-side caching: subscribes to the
invalidation events.

=cut

async method enable_clientside_cache ($redis) {
    my $f = $self->{client_side_cache_ready} = $self->loop->new_future;
    try {
        $log->tracef('At this point we want an ID');
        my $id = await $redis->client_id;
        $log->tracef('Set up :invalidate sub');
        my $sub = await $redis->subscribe('__redis__:invalidate');
        $log->tracef('We now have :invalidate');
        $self->{clientside_cache_events} = $sub->events;
        $sub->events->each(sub {
            $log->tracef('Invalidating key %s', $_);
            $self->client_side_cache->remove($_);
        });

        # In a multi-connection setup, e.g. RESP2, the notifications need to be
        # delivered to the subscription connection via `REDIRECT`
        my @args = refaddr($self) != refaddr($redis)
        ? (redirect => $id)
        : ();
        $log->tracef('Enable client tracking via %s', $self->can('client_tracking'));
        await $self->client_tracking('on', @args);
        $f->done;
    } catch($e) {
        $f->fail($e) unless $f->is_ready;
        die $e;
    }
    return;
}

=head2 _init



=cut

method _init (@args) {
    $self->{protocol_level} //= 'resp2';
    $self->{pending_multi} //= [];
    $self->{pending} //= [];
    $self->{awaiting_pipeline} //= [];
    $self->{opentracing} = OPENTRACING_ENABLED;
    $self->next::method(@args);
}

=head2 _add_to_loop



=cut

method _add_to_loop ($loop) {
    delete $self->{client_side_connection};
}

=head2 retrieve_full_command_list

Iterates through all commands defined in Redis, extracting the information about
that command using C<COMMAND INFO>.

The data is formatted for internal use, converting information such as flags
into hashrefs for easier lookup.

This information is also used by L</extract_keys_from_command>.

Returns a hashref, where each key represents a method name (space-separated
commands such as C<CLUSTER NODES> are returned as C<cluster_nodes>). The values
are a restructured form of L<https://redis.io/commands/command>.

=cut

async method retrieve_full_command_list {
    my %data;
    my $commands = await $self->command_list;
    for my $command_name ($commands->@*) {
        my $method_name = $command_name =~ s/\|/_/gr;
        my ($info) = (await $self->command_info($command_name))->@*;
        my ($name, $arity, $flags, $first_key, $last_key, $step, $acl_cat, $tips, $key_spec, $subcommands) = $info->@*;
        $flags = +{ map { $_ => 1 } $flags->@* };
        $acl_cat = +{ map { $_ => 1 } $acl_cat->@* };
        $tips = +{ map { $_ => 1 } $tips->@* };
        my @key_specs;
        for my $ks ($key_spec->@*) {
            $key_spec = +{ $ks->@* };
            $key_spec->{flags} &&= +{ map { $_ => 1 } $key_spec->{flags}->@* };
            $key_spec->{begin_search} &&= +{ $key_spec->{begin_search}->@* };
            $key_spec->{begin_search}{spec} &&= +{ $key_spec->{begin_search}{spec}->@* };
            $key_spec->{find_keys} &&= +{ $key_spec->{find_keys}->@* };
            $key_spec->{find_keys}{spec} &&= +{ $key_spec->{find_keys}{spec}->@* };
            push @key_specs, $key_spec;
        }

        $data{$method_name} = {
            name        => $name,
            arity       => $arity,
            flags       => $flags,
            first_key   => $first_key,
            last_key    => $last_key,
            step        => $step,
            acl_cat     => $acl_cat,
            tips        => $tips,
            key_spec    => \@key_specs,
            subcommands => $subcommands
        };
    }
    return \%data;
}

=head2 extract_keys_for_command

Given a command arrayref and a definition for the server, this will
return a list of any keys found in that command.

Since the logic for this is slightly slow, we are caching the result
unless a specific definition is provided: this is an internal implementation
detail and not something to rely on.

(the key specification is a relatively new Redis feature - an optimised version
of this logic will be added to L<Net::Async::Redis::XS> in due course, which
should reduce the need for caching)

Returns a list of keys.

=cut

my $keyspec_cache = Cache::LRU->new(
    size => 10_000
);

sub extract_keys_for_command ($class, $command, $def = undef) {
    my $cached = $keyspec_cache->get(join "\x{01FF}", $command->@*);
    return $cached->@* if $cached;

    my $info = $class->extract_spec_for_command($command, $def);
    # Commands can have zero or more keyspecs which tell us where to find the key information. Each
    # command may have multiple keyspec definitions.
    my @keys;

    my @components = $command->@*;
    my $cmd = join ' ', $command->@*;

    KEYSPEC:
    for my $key_spec ($info->{key_spec}->@*) {
        my $type = $key_spec->{begin_search}{type}
            or next KEYSPEC;

        # Instead of trying to track index and offset, we work with a copy of the data - less efficient,
        # but easier to get accurate results.
        my @target = @components;

        # Find the starting index
        match($type : eq) {
            case('index') {
                splice @target, 0, $key_spec->{begin_search}{spec}{index} if $key_spec->{begin_search}{spec}{index};
            }
            case('keyword') {
                my $target = $key_spec->{begin_search}{spec}{keyword};
                my $start = $key_spec->{begin_search}{spec}{startfrom};
                if($start < 0) {
                    splice @target, $start, -$#target if $start < -1;
                    pop @target while @target and uc($target[-1]) ne $target;
                    next KEYSPEC unless @target;
                    @target = reverse @target;
                } else {
                    splice @target, 0, $start - 1 if $start > 1;
                    shift @target while @target and uc($target[0]) ne $target;
                    next KEYSPEC unless @target;
                    shift @target;
                }
            }
            case('unknown') {
                die 'Unknown key specification for command ' . $cmd;
            }
            default {
                die 'No key specification for command ' . $cmd;
            }
        }

        # Find the keys, starting from the index identified above
        match($key_spec->{find_keys}{type} : eq) {
            case('range') {
                my $spec = $key_spec->{find_keys}{spec};
                my $last_key = $spec->{lastkey};
                unless($last_key) {
                    push @keys, shift @target;
                    next KEYSPEC;
                }

                my $key_step = $spec->{keystep};
                my $limit = $spec->{limit};

                splice @target, (1 + $last_key) * ($key_step + 1) if $last_key < -1;
                my $target_index = 0 + @target;
                $target_index = $last_key * $key_step if $last_key > 0;
                $target_index = int($target_index / $limit) if $limit > 1;
                $target_index //= 1;

                while(@target and $target_index--) {
                    my ($next) = splice @target, 0, $key_step;
                    push @keys, $next;
                }
                next KEYSPEC;
            }
            case('keynum') {
                my $spec = $key_spec->{find_keys}{spec};
                my $key_step = $spec->{keystep};
                my $count = $target[$spec->{keynumidx}];
                splice @target, 0, $spec->{firstkey};
                while(@target and $count--) {
                    my ($next) = splice @target, 0, $key_step;
                    push @keys, $next;
                }
                next KEYSPEC;
            }
            case('unknown') {
                die 'Unknown key specification for command ' . $cmd;
            }
            default {
                die 'No key specification for command ' . $cmd;
            }
        }
    }
    $keyspec_cache->set(join("\x{01FF}", $command->@*), \@keys);
    return @keys;
}

sub extract_spec_for_command ($class, $command, $def = undef) {
    my $cache = 0;
    unless($def) {
        $def //= \%COMMAND_DEFINITION;
        ++$cache;
    }

    # The command itself is represented as a method name
    my (@components) = $command->@*;
    my ($cmd) = @components;
    my $info = $def->{lc $cmd} or die 'command not found: ' . $cmd;

    # Identify the full matching command against the known definitions
    my $idx = 0;
    while(@components && $info->{subcommands} && $info->{subcommands}->@* && $#components >= $idx) {
        my $next = "${cmd}_" . $components[++$idx];
        last unless exists $def->{lc $next};
        $cmd = $next;
        $info = $def->{lc $cmd};
    }
    return $info;
}

=head2 ssl_options

Extracts the SSL-related options as a hashref for passing
to C<< $loop->connect >>.

=cut

method ssl_options {
    return {
        map {
            defined $self->{"tls_$_"} ? (
                "SSL_$_" => $self->{"tls_$_"}
            ) : ()
        } qw(
            cert_file
            key_file
            ca_file
        )
    };
}

1;

__END__

=head1 SEE ALSO

Some other Redis implementations on CPAN:

=over 4

=item * L<Mojo::Redis2> - nonblocking, using the L<Mojolicious> framework, actively maintained

=item * L<MojoX::Redis> - changelog mentions that this was obsoleted by L<Mojo::Redis>, although there
have been new versions released since then

=item * L<RedisDB> - another synchronous (blocking) implementation, handles pub/sub and autoreconnect

=item * L<Cache::Redis> - wrapper around L<RedisDB>

=item * L<Redis::Fast> - wraps C<hiredis>, faster than L<Redis>

=item * L<Redis::Jet> - also XS-based, docs mention C<very early development stage> but appears to support
pipelining and can handle newer commands via C<< ->command >>.

=item * L<Redis> - synchronous (blocking) implementation, handles pub/sub and autoreconnect

=item * L<HiRedis::Raw> - another C<hiredis> wrapper

=back

=head1 AUTHOR

Tom Molesworth <TEAM@cpan.org>

=head1 CONTRIBUTORS

With thanks to the following for contributing patches, bug reports,
tests and feedback:

=over 4

=item * C<< BINARY@cpan.org >>

=item * C<< PEVANS@cpan.org >>

=item * C<< @eyadof >>

=item * Nael Alolwani

=item * Marc Frank

=item * C<< @pnevins >>

=back

=head1 LICENSE

Copyright Tom Molesworth and others 2015-2024.
Licensed under the same terms as Perl itself.
