package OpenInteract2::ErrorObject;

# $Id: ErrorObject.pm,v 1.7 2004/02/18 05:25:23 lachoy Exp $

use strict;
use Data::Dumper             qw( Dumper );
use File::Spec;
use Log::Log4perl            qw( get_logger );
use OpenInteract2::Constants qw( :log );
use OpenInteract2::Context   qw( CTX );
use SPOPS::Utility;
use Text::Wrap     ();

@OpenInteract2::ErrorObject::ISA     = qw( OpenInteract2::ErrorObjectPersist );
$OpenInteract2::ErrorObject::VERSION = sprintf("%d.%02d", q$Revision: 1.7 $ =~ /(\d+)\.(\d+)/);

my ( $log );

# TODO: discard?
# keeps errors during request

my @LIST = ();

my $DEFAULT_TYPE   = 'n/a';

# TODO: discard?
# Routines to manipulate the list of errors during a request
#

sub clear_listing { @LIST = () }
sub listing       { return \@LIST }
sub report        { return \@LIST }

# Class method
#

sub throw {
    my ( $class, $p ) = @_;
    $log ||= get_logger( LOG_APP );

    unless ( $p->{package} and $p->{filename} and $p->{line} ) {
        my ( $cpkg, $cfile, $cline ) = caller;
        $p->{package}  = $cpkg;
        $p->{filename} = $cfile;
        $p->{line}     = $cline;
    }
    my $err = $class->new( $p );
    $log->is_debug &&
        $log->debug( "Object after first create: ", CTX->dump( $err ) );
    push @LIST, $err; # save it for later...

    # XXX: No port for this yet...
    CTX->error_handler->catch( $err );
}


# Called when we do: OpenInteract::ErrorObject->new()
# (see SPOPS->new() for more info and how ->initialize()
# is called)

sub initialize {
    my ( $self, $p ) = @_;

    # Set pkg (etc) context if not already set

    unless ( $p->{package} and $p->{filename} and $p->{line} ) {
        my ( $cpkg, $cfile, $cline ) = caller;
        $p->{package}  = $cpkg;
        $p->{filename} = $cfile;
        $p->{line}     = $cline;
    }

    # Give a default type if not set

    $p->{type} = $DEFAULT_TYPE;

    # First call SPOPS::DBI->initialize() to set all the
    # information passed in matching up with the object
    # parameters

    $self->SUPER::initialize( $p );

    # Set meta information

    my $req = CTX->request;
    my $user_id = ( $req->is_logged_in ) ? $req->auth_user->id : undef;
    $self->{user_id}     = $user_id;
    $self->{session_id}  = $req->session->{_session_id};
    $self->{browser}     = $req->user_agent;
    $self->{error_time}  = SPOPS::Utility->now;
    $self->{url}         = $req->url_absolute;
    $self->{referer}     = $req->referer;
    return $self;
}


sub fail_save {
    my ( $self, $p ) = @_;
    $log ||= get_logger( LOG_APP );

    my $error_dir = CTX->lookup_directory( 'error' );

    # Note that even though it's not saved, the ID should still
    # have been defined in the save() process since we
    # are using random codes. Also, do not do a ->throw with any 
    # errors generated by trying to save the file, otherwise you'll
    # get in an infinite loop, which would be bad.

    my $full_file = File::Spec->catfile( $error_dir, $self->id );
    eval { open( ERROR, "> $full_file" ) || die $! };
    if ( $@ ) {
        $log->error( "LAST DITCH ERROR: Cannot save error even to ",
                     "filesystem at [$full_file]! Message: $@\n",
                     "Error:", CTX->dump( $self ), 'nodb' );
        return undef;
    }

    $self->{error_time} = scalar localtime;
    print ERROR Dumper( $self );
    close( ERROR );
    $log->warn( "Error written out to [$error_dir/$self->{error_id}]" );
    return 1;
}


sub as_string {
    my ( $self, $opt ) = @_;
    local $Text::Wrap::columns = $opt->{columns} || 65;
    my $user_msg = Text::Wrap::wrap( '', '', $self->{user_msg} );
    my $sys_msg  = Text::Wrap::wrap( '', '', $self->{system_msg } );
    my $string = <<STRING;
   Error: $self->{error_id}
    Date: $self->{error_time}
    Code: $self->{code}
    Type: $self->{type}
 Browser: $self->{browser}
 User ID: $self->{user_id}
 Session: $self->{session_id}

Messages

========
User
$user_msg

========
System
$sys_msg

STRING
    $string =~ s/\n/<br>\n/g if ( $opt->{html} );
    return $string;
}

1;

__END__

=head1 NAME

OpenInteract2::ErrorObject - Use errors as objects and enable persistence

=head1 SYNOPSIS

 my $err = OpenInteract2::ErrorObject->new( { code => 405, type => 'db' } );
 $err->save;

=head1 DESCRIPTION

The fact that each error is now an object that gets saved means that
we can create a simple (or complicated) web-based error browsing
application that shows you all errors in a certain time period plus
details, etc.

=head1 METHODS

B<throw( \%params )>

Throws an error from anywhere in the system. We create an error object
with the parameters (or with the error information already stored, see
below) and send the object onto the error handling framework within
OpenInteract.

Parameters:

=over 4

B<code> ($)

Mandatory. See
L<OpenInteract2::Error::Main|OpenInteract2::Error::Main> for what the
different codes signify. (Basically, the higher the code the less
severe the error.)

If you call throw() with only the I<code> parameter, the method will
snatch the error information from the package variables already
set. Otherwise, see L<OpenInteract::Error> for the different
parameters that can be set.

=back

=head1 TO DO

=head1 BUGS

=head1 COPYRIGHT

Copyright (c) 2002-2004, Chris Winters. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS

Chris Winters E<lt>chris@cwinters.comE<gt>