/* -*- mia-c++  -*-
 *
 * This file is part of MIA - a toolbox for medical image analysis
 * Copyright (c) Leipzig, Madrid 1999-2017 Gert Wollny
 *
 * MIA is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with MIA; if not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <mia/internal/autotest.hh>

#include <stdexcept>
#include <cmath>

#include <mia/core/waveletslopeclassifier.hh>


using namespace std;
using namespace mia;

struct SlopeClassifierFixture {
       typedef struct {
              int periodic_idx;
              int RV_idx;
              int LV_idx;
              int baseline_idx;
              int perfusion_idx;
              int RV_peak;
              int LV_peak;
       } Result;
       void run(size_t length, size_t components, const float *data, const Result& r, bool mean_stripped);
       void check_equal(int a, int b, const char *descr);
};

void SlopeClassifierFixture::check_equal(int value, int expect, const char *descr)
{
       if (expect != -2)
              BOOST_CHECK_MESSAGE( value == expect, descr << ": got '" << value << "' expect '" << expect );
}

void SlopeClassifierFixture::run(size_t length, size_t components, const float *data, const Result& result, bool mean_stripped)
{
       CWaveletSlopeClassifier::Columns columns( components );

       for (size_t c = 0; c < components; ++c)
              columns[c].resize(length);

       const float *i = data;

       for (size_t r = 0; r < length; ++r)
              for (size_t c = 0; c < components; ++c, ++i)
                     columns[c][r]  = *i;

       CWaveletSlopeClassifier c(columns, mean_stripped, -1);
       check_equal(c.get_movement_idx(), result.periodic_idx, "periodic index");
       check_equal(c.get_RV_idx(), result.RV_idx, "RV index");
       check_equal(c.get_LV_idx(), result.LV_idx, "LV index");
       check_equal(c.get_perfusion_idx(), result.perfusion_idx, "perfusion index");
       check_equal(c.get_baseline_idx(), result.baseline_idx, "baseline index");
// peak location values are now based on smoothed curves
//	check_equal(c.get_RV_peak(), result.RV_peak, "RV peak");
//	check_equal(c.get_LV_peak(), result.LV_peak, "LV peak");
}


BOOST_FIXTURE_TEST_CASE( test_classifier_mean_removed_4_2, SlopeClassifierFixture
                       )
{
       const size_t rdlength2 = 58;
       const size_t rdcomponents = 4;
       const float init_real_data5[rdlength2 * rdcomponents] = {
              -1.21697,    -12.889,    -8.8137,   -11.2628,
              7.26992,   -10.1643,   -7.71096,   -10.2105,
              3.36918,   -9.61763,    -7.5347,   -10.2059,
              -5.31023,   -10.8776,   -8.19432,   -10.7311,
              -3.61536,    -11.757,   -8.53927,    -10.786,
              7.43997,   -9.72578,   -7.70598,   -9.37773,
              3.96546,   -8.38908,   -7.17976,    -3.5115,
              -6.22638,   -9.78003,   -8.38516,    4.31378,
              -2.22855,   -11.1856,   -8.76427,    9.20779,
              8.28368,   -9.18611,   -8.94622,      13.12,
              1.34157,   -9.57345,   -7.77406,    15.3541,
              -5.89887,   -10.5877,   -7.39647,    13.6015,
              3.33047,   -10.4561,   -5.99676,    13.6439,
              11.51,   -8.87868,    -2.7022,    13.4846,
              -0.201523,    -9.3964,    4.60773,    12.5756,
              -6.46308,   -9.74944,    8.06126,    8.36199,
              2.09767,   -9.91261,    11.2294,    1.78679,
              9.98559,   -7.19383,    11.3783,    2.69818,
              -2.20947,   -6.43649,    11.8703,    2.25018,
              -6.46433,   -6.83957,    11.0092,  -0.542518,
              5.37192,   -5.87149,    12.3547,   -5.41251,
              3.82535,   -3.10461,     12.523,   -4.56304,
              -4.9578,   -1.60572,    10.4858,   -3.90724,
              -6.78449,   -1.70081,    8.48264,   -6.42681,
              6.01595,   0.314861,    8.62154,   -8.89845,
              6.47053,    3.85449,    7.31228,   -7.60484,
              -4.00454,    5.13422,    5.03714,   -5.66759,
              -7.35794,    4.92735,    2.79475,   -6.39917,
              0.160344,    4.97459,      2.087,   -7.57198,
              8.01644,    7.57877,     1.4296,   -5.21079,
              1.70258,    8.99215,   0.095665,   -2.86466,
              -5.86428,     8.2516,   -1.42558,   -2.07345,
              -7.57355,    7.06447,   -2.49541,  -0.815717,
              1.07946,     6.2852,   -3.01762,  -0.461196,
              10.2077,    7.14585,   -3.01935,    1.50698,
              2.50873,    9.08613,   -2.76672,    2.46428,
              -4.11772,    7.88459,   -2.81467,    2.72878,
              -7.30043,    6.71874,    -2.8543,    2.09026,
              -7.81432,    5.34063,   -2.81373,     2.2376,
              1.21732,     4.6602,   -2.02043,    1.28263,
              10.6291,    5.97178,    -0.9351,    1.94118,
              4.20717,    7.56745,   0.448437,    1.96171,
              -5.38515,    7.36575,   0.340722,    2.19751,
              -8.16073,    6.01278,  -0.015374,    1.08096,
              -0.268398,    4.54413,   0.434688,  -0.314944,
              10.1471,    5.79406,   0.677009,  -0.407828,
              2.88709,    7.32053,    1.45917,   0.511453,
              -6.25489,    6.58108,   0.717902,   0.687246,
              -7.81785,    5.45234,   0.138807,   0.325375,
              -5.39009,    4.31263,  -0.144036,  -0.700289,
              7.36143,    4.95953,   0.127016,   -1.10937,
              -0.0970818,    6.92727,   0.371997,   0.473863,
              -6.81938,    5.74919,  -0.475212,    1.05183,
              -8.47326,    4.63227,   -1.00542,  0.0984377,
              2.4529,    4.64438,  -0.785798,  0.0570382,
              9.60226,    6.35436,   -0.32421,   0.355114,
              -0.548125,    7.12886,  -0.522247,     1.8589,
              -7.63212,    5.34681,     -1.017,     1.7284
       };
       Result r;
       r.periodic_idx = 0;
       r.RV_idx = 3;
       r.LV_idx = 2;
       r.baseline_idx = -1;
       r.perfusion_idx = 1;
       r.RV_peak = 10;
       r.LV_peak = 21;
       run( rdlength2, rdcomponents, init_real_data5,  r, true);
}


BOOST_FIXTURE_TEST_CASE( test_classifier_4_anotherone, SlopeClassifierFixture )
{
       const size_t rdlength2 = 58;
       const size_t rdcomponents = 4;
       const float init_real_data5[rdlength2 * rdcomponents] = {
              -40.5885,   -42.0489,   -45.5256,   -1.06894,
              -45.7077,    -53.409,   -40.5529,    1.06517,
              -43.9552,    25.6663,   -41.0811,    2.05432,
              -36.844,    5.05299,   -50.4248,    1.94573,
              -39.9942,   -26.0527,   -46.9356,    2.28095,
              -45.6154,   -49.3138,   -41.4347,    2.48059,
              -47.2126,    -52.324,   -35.3429,    1.75445,
              -40.0522,    40.6544,   -27.2421,    3.12005,
              -29.7715,    35.8575,    -25.193,   0.977491,
              -31.841,    -7.7891,   -3.07592,    1.19795,
              -32.9921,    -31.021,    35.4505,    1.52449,
              -37.4308,   -59.7679,    41.8424,    1.47142,
              -36.9135,   -40.6995,    59.4962,    1.06211,
              -20.994,    106.464,    31.7577,    5.98843,
              -14.6221,    62.0819,    20.9851,    0.23603,
              -10.7321, -0.0692243,    27.4757,   0.413571,
              -17.7299,    -62.248,    34.1193,  -0.311548,
              -6.68392,    75.3428,    31.0272,    1.18152,
              13.2246,    92.2324,     21.553,    2.25978,
              20.0892,    22.3414,    15.2671,   -1.34974,
              7.86028,   -45.8267,    16.3647,   0.516328,
              17.6371,   -61.1057,    28.2191,   0.207029,
              28.6596,   -65.0605,     19.038,   -1.37892,
              5.11108,   -68.7915,     17.638,    1.79901,
              13.4148,   -75.7552,    12.8194,  -0.115909,
              14.3393,    78.6177,    2.17008,    1.95503,
              25.2623,    97.5555,   -5.16663,    3.44103,
              31.7565,    31.7876,   -11.3776,  -0.988721,
              14.8972,    -43.411,   -2.89347,  -0.513903,
              7.4624,   -64.9449,    4.15604,  -0.985499,
              7.82721,    13.4014,    2.53395,   -2.50635,
              21.7543,    81.3186,   -3.75994,    1.38046,
              25.6319,    10.6536,   -12.1314,   -2.83351,
              18.1948,   -28.0865,   -5.26292,   -1.96467,
              4.11488,   -58.1767,   0.623238,  -0.502474,
              5.59611,    32.4971,    1.39253,   -0.96725,
              18.493,    99.1702,  0.0843583,    3.00145,
              22.6629,    41.9831,   -8.01348,  -0.984234,
              22.546,   -16.2061,   -2.86221,   -2.31817,
              5.52405,   -45.3296,    1.26168,   -1.13303,
              10.0739,   -60.5249,    7.83677,   -2.10769,
              6.34081,   -3.28841,    7.01992,   -1.95952,
              9.07522,    68.8657,  -0.637501,  -0.527897,
              22.5867,    87.4689,   -5.53472,    1.61981,
              12.4519,   -21.1978,   -4.01191,   -1.42338,
              1.86497,    -61.235,    4.57168,   -1.44206,
              -0.0375902,   -31.0413,    7.91329,    -2.8754,
              14.55,    84.5825,    3.55639,    1.11224,
              25.2812,    80.1884,   -6.38595,  -0.818489,
              22.9835,    15.3067,     -6.824,   -2.93083,
              5.21044,   -48.3389,   0.447882,   -1.86792,
              10.0186,   -64.4603,    5.60138,   -1.97099,
              7.02159,   -3.22657,    2.33767,   -1.85069,
              17.6422,    100.899,   -3.03946,    2.76228,
              27.9474,    40.2603,   -14.0553,   -2.73036,
              23.8631,   -11.4303,   -9.09685,   -1.59569,
              8.58541,   -61.0184,   -5.24212,   -2.93687,
              2.16168,   -67.0509,   -1.45668,   -1.84713
       };
       Result r;
       r.periodic_idx = 1;
       r.RV_idx = 2;
       r.LV_idx = 0;
       r.baseline_idx = 3;
       r.perfusion_idx = -1;
       r.RV_peak = 12;
       r.LV_peak = 27;
       run( rdlength2, rdcomponents, init_real_data5,  r, false);
}

BOOST_FIXTURE_TEST_CASE( test_classifier_5_B0, SlopeClassifierFixture )
{
       const size_t rdlength2 = 58;
       const size_t rdcomponents = 5;
       const float init_real_data5[rdlength2 * rdcomponents] = {
              -21.5101,  -70.9349,  -54.8832,  -8.02363,  -107.674,
              62.1742,  -61.8815,  -47.4486,   4.05271,  -82.3734,
              21.1974,  -59.8629,  -45.0058,   4.71944,  -73.2937,
              -63.8195,  -64.4069,  -48.9134, -0.517728,  -82.9371,
              -44.6033,  -66.3626,  -52.0342,  -10.5712,  -93.3152,
              63.7975,  -55.3954,   -46.507, -0.653942,  -75.4164,
              28.0447,  -16.1006,  -40.4158,   10.7913,  -59.1176,
              -69.8763,   31.9211,  -45.9987,    -2.826,  -69.2282,
              -26.7364,   60.8714,  -48.6856,  -12.1556,  -85.1516,
              77.0263,   90.6266,   -49.494,   2.30612,  -72.2751,
              6.60981,   105.819,   -39.695,   2.75522,  -67.0276,
              -64.8293,   96.3348,  -37.5808,  -7.24514,   -74.002,
              30.5064,   89.9653,  -31.3594,  -18.1576,  -78.7652,
              108.75,   87.6518,  -11.9986,   12.7854,  -67.3555,
              -7.17098,   82.2409,   34.1087,    7.7757,  -60.5163,
              -66.6993,    52.693,   53.2669,  -1.92582,  -63.9904,
              20.235,   7.44192,   68.8185,  -7.64787,  -71.8108,
              95.2104,    15.592,   70.2948,   14.8668,  -50.2588,
              -25.6608,   14.8632,   75.2925,   8.50921,  -36.3162,
              -64.3794,  -5.86425,   67.8701,   -8.7853,  -44.2359,
              52.9093,   -38.592,   72.9375,  -6.64724,  -43.5992,
              36.184,  -30.9802,   75.2212,   3.90409,  -18.6681,
              -49.2223,  -26.9414,   63.1073,   -1.3883,  -6.23317,
              -64.0443,  -46.3739,   48.2332,  -17.0704,  -14.3219,
              60.7041,  -62.3946,   46.9352,  -1.94719,   -5.1516,
              63.782,  -50.9867,    40.588,   13.0022,    25.205,
              -38.3441,   -37.823,   28.2475,   7.91706,   39.2549,
              -70.9549,  -43.1368,   14.1473,   3.22068,   36.2974,
              7.2602,  -54.1956,   6.75973,  -8.44868,   27.9151,
              81.9071,  -35.6415,   4.40277,   7.70883,   50.4127,
              19.2254,  -18.7763,  -1.67137,   10.5708,   66.0645,
              -55.1381,  -13.5242,  -9.95895,   6.88407,   62.9892,
              -71.2352,  -6.11588,  -16.3596,  0.805754,   53.3075,
              17.062,  -7.76605,   -22.362,  -8.62473,   37.7436,
              105.107,   6.48572,  -21.9431,   6.54207,   44.2286,
              26.8782,   16.6228,  -17.1102,   12.4349,   67.9287,
              -38.5988,   18.7706,  -16.3778,   11.5548,   62.0762,
              -69.2257,   13.9064,  -16.8515,   3.47106,   53.1891,
              -73.425,   13.3029,  -17.3533,  -3.04657,   40.4217,
              19.0144,   3.20832,  -15.6279,  -14.6578,   26.3543,
              109.544,   9.69094,  -8.43266,   6.15825,   37.8081,
              43.812,   13.2359,    2.6147,   15.8953,   57.2712,
              -50.1025,   15.9711,   3.65279,    8.8972,   62.3557,
              -77.2073,   7.77249,  0.914186, -0.131839,   50.9678,
              4.26769,  -5.55192,   0.16415,  -9.82492,   29.7144,
              105.686,   -5.5131,  0.836226,  0.499044,   36.7304,
              31.171,   4.44616,   8.93191,   12.0494,   57.4705,
              -58.2973,   5.62527,   5.36304,   2.83262,   55.1484,
              -74.1288,   2.69363,   1.47169,  -1.29271,   45.9818,
              -45.5706,  -8.30323,  -3.35438,  -19.7296,   28.5993,
              78.246,  -10.3823,  -2.57134,  -8.38768,   30.3896,
              2.64925,   3.50841,   2.11872,  0.513459,   53.6509,
              -64.2622,   7.88266,  -1.88002, -0.417225,    48.533,
              -78.8516, -0.302667,  -6.44013,  -13.7938,   36.8173,
              30.5377,  -2.53047,  -7.10827,  -12.5386,   30.1156,
              99.094,   1.31508,  -3.91258,   4.09474,   43.7485,
              -2.82663,   14.2099,  -1.89937,   6.12335,   58.8309,
              -71.8716,    11.971,  -5.06392,  -7.18392,   45.5126
       };
       Result r;
       r.periodic_idx = 0;
       r.RV_idx = 1;
       r.LV_idx = 2;
       // these two are not yet sure
       r.baseline_idx = 3;
       r.perfusion_idx = 4;
       r.RV_peak = 10;
       r.LV_peak = 18;
       run( rdlength2, rdcomponents, init_real_data5,  r, false);
}



/* this case is wired, because it has three curves that go like the RV resp. LV slope
   and the curve with the second peak is not the curve corresponding to LV
*/
BOOST_FIXTURE_TEST_CASE( test_classifier_5_F0, SlopeClassifierFixture )
{
       const size_t rdlength2 = 58;
       const size_t rdcomponents = 5;
       const float init_real_data5[rdlength2 * rdcomponents] = {
              -3.91985,  -4.26952, -0.491522,  -1.45301,  -1.73747,
              -4.27291,  -3.92887, -0.0868448, 0.0958735,  -1.89318,
              -4.79939,  -2.82392, -0.0366393,   7.55703,  -2.65164,
              -4.96619,  -2.88971,  0.916839,   8.96919,  -2.90496,
              -4.73506,  -4.36581, -0.137766,   3.63727,  -1.64602,
              -4.2854,   -4.6991, -0.241153,  -1.12971,  -1.58393,
              -4.0678,  -4.54905, -0.317134,  -2.68932,  -1.70436,
              -3.6844,  -4.29939, -0.126088,   -2.9084,  -1.76766,
              -3.39458,  -4.11048, -0.268337,  -3.01744,  -1.73499,
              -2.23508,  -3.92186, -0.205801,  -3.10636,  -1.85866,
              -0.416026,   -3.5435, -0.307421,  -3.22974,  -1.94661,
              1.80492,  -3.33391, 0.0146373,  -1.92977,  -2.05189,
              4.43199,  -3.19202,  -0.29299,   4.66958,  -2.00937,
              4.53098,  -2.86997,  0.535387,   9.34503,  -2.71113,
              3.82912,  -3.29652,  0.217521,   8.61739,  -2.10336,
              3.67879,  -4.02331, 0.0699871,    2.2577,  -1.58575,
              4.25959,  -3.13375, -0.187782,  -2.36153,  -2.19371,
              4.39797,  -2.19324,  0.108355,  -3.37041,  -2.52931,
              4.45314,  -1.50283, -0.117822,  -3.94485,  -2.76075,
              4.02399, -0.731538,  0.138157,  -3.87451,  -2.86609,
              3.71572,  0.128108, -0.0243865,  -4.09891,  -2.82515,
              3.12605,  0.910229,  0.106374,  -4.14588,  -2.80378,
              2.6232,     1.899, -0.116851,  -4.11302,  -2.76122,
              2.59829,   2.56767, 0.0832503,  -2.98951,  -2.29169,
              2.50398,   2.93073, -0.318271,   3.91131,  -1.31003,
              1.06013,   3.88074,  0.459533,   9.74053,  -1.28968,
              0.271014,   2.77045, -0.108586,   5.79478, -0.216012,
              0.136218,   2.70895, -0.0397375, -0.381828, -0.594354,
              0.160926,   3.51572, -0.154987,  -2.38669, -0.584147,
              0.12259,   3.65125,  0.190792,  -3.05096, -0.542157,
              0.292004,   3.93639, -0.0669645,  -3.47794, -0.244408,
              0.069694,   3.81526,  0.172952,  -3.37814, -0.0115264,
              0.0191166,   3.82002, -0.0596861,  -3.55867,  0.434731,
              -0.265876,   3.46699,  0.194471,  -3.61935,  0.579506,
              -0.230492,   3.38335, 0.0760353,   -3.5608,  0.756527,
              -0.228871,   3.04027,   0.21433,  -3.56579,   1.02439,
              -0.279898,   2.68014, -0.010516,  -3.66339,   1.32698,
              -0.340424,   2.50179,  0.181271,  -2.06322,   1.72098,
              -0.619365,    2.6579, -0.0312459,    5.5936,   1.90469,
              -1.07194,   1.92646,  0.284197,   6.73248,   2.09886,
              -1.00767,  0.720694, -0.0491678,  0.288412,    2.8444,
              -0.707446,   1.08415,   0.11042,  -2.46586,   2.47987,
              -0.196771,   1.19227, -0.153958,  -3.24545,   2.42352,
              -0.0928285,   1.28496,   0.14998,  -3.26577,   2.25338,
              0.100723,   1.35567, -0.0944623,  -3.53608,   2.32708,
              0.0349242,   1.17239, 0.0965734,  -2.69898,   2.50193,
              -0.263166,    1.4302, -0.125942,   5.85864,   2.45345,
              -0.976712,   1.34324,  0.674693,   9.38254,      1.29,
              -1.12257,   0.28923,   0.18239,    7.0359,    2.7887,
              -0.997077, -0.598653, 0.0270598,  0.601905,   3.20218,
              -0.528015, -0.042121, -0.275099,  -1.83801,   3.03519,
              -0.411671,  0.125965, 0.0162738,  -1.73676,   2.87853,
              0.125577,  0.385131, -0.168422,  -2.49436,   2.83446,
              -0.013434,   0.50721, -0.00905256,  -2.60108,   2.71758,
              0.00769322,  0.469898, -0.284253,   -2.1964,   2.90225,
              -0.567674,  0.969795, -0.0690733,   6.07435,    2.5115,
              -0.93943, -0.114662, -0.0284586,   3.43186,   3.34267,
              -0.740314,  -0.08847, -0.215102,  -2.44752,   3.08165
       };
       Result r;
       r.periodic_idx = 3;
       r.RV_idx = 0;
       r.LV_idx = 1;
       // these two are not yet sure
       r.baseline_idx = -2;
       r.perfusion_idx = -2;
       r.RV_peak = 13;
       r.LV_peak = 25;
       run( rdlength2, rdcomponents, init_real_data5,  r, true);
}


BOOST_FIXTURE_TEST_CASE( test_classifier_5_O1a, SlopeClassifierFixture )
{
       const size_t rdlength2 = 58;
       const size_t rdcomponents = 5;
       const float init_real_data5[rdlength2 * rdcomponents] = {
              -0.194644,  -2.27097,  -3.36575,   -2.5364,  -5.34558,
              0.765331,   -3.8211,  -3.73159,   -1.9391,   2.65789,
              -0.892364,  -3.58818,  -4.24303,   1.22069,   5.74515,
              -1.13437,    -4.627,  -4.34544,    1.4242,   1.64877,
              -0.365101,  -12.5294,  -3.35878,  -3.83679,  -5.50722,
              1.76301,  -12.3425,  -4.54628,  -5.93627,   4.39628,
              0.261274,  -7.04237,  -5.33096,  -1.93964,   4.98745,
              -1.88556,   -1.9671,  -6.11784,  -2.90995, -0.520417,
              -0.901675,  0.215163,  -7.62535,  -5.55443,   2.47797,
              1.33391,  0.927133,  -8.08026,  -4.14495,   4.43153,
              0.172277,   1.73026,  -8.28958,  -1.04375,   1.72171,
              2.72399,    1.9474,  -6.70993,  -3.95895,  -5.91067,
              0.549728,    1.5259,  -7.29375,  -6.14811,   3.52236,
              0.452623,   1.84944,   -6.6882,  -2.86127,   6.54145,
              -1.70723,   2.82367,  -5.80605,  -3.08049,     7.237,
              0.880562,   3.20682,  -4.63637,   -2.6166,    7.6504,
              2.08701,   4.45134,  -1.44584,  -4.01091,  -4.37968,
              4.04924,   3.69677, -0.0889484,  -9.95966, -0.413756,
              0.585227,   2.99372, -0.863572,  -4.15864,   8.00943,
              -1.20141,   3.27061, -0.0777571,  -2.53295,   8.29044,
              -0.711145,   4.17774,   1.73297,  -1.58116,  -4.50367,
              2.30682,   3.01388,   2.83435,  -5.12192,  -5.33368,
              0.669934,   2.71462,   3.84647,  -8.90214,   1.42903,
              0.929072,   1.62788,   3.00085,  -3.00467,   7.24816,
              -1.2227,   1.57794,   3.68044,  -2.16732,   6.30188,
              -2.20204,  0.693496,   3.45937,   1.11622,   8.23672,
              -1.18,   1.64482,   3.73222,   3.28705,  -2.63859,
              0.368327,  0.764952,   3.90995,  -2.69701,  -5.26728,
              -1.03298,  0.210432,   4.71717,  -8.19011,  0.735514,
              0.000465238, -0.626899,   3.46267, -0.773838,   6.96087,
              1.54549, 0.0666781,   3.24886,   3.01055,  -2.64503,
              -1.73513,  0.599961,   3.47547,  -3.48053,  0.690474,
              -1.81159, -0.819506,   3.01698,    2.0462,   6.36906,
              -0.0444954,  -2.13922,   2.60177,   5.90034,   6.89727,
              -3.05218,  -1.75002,   2.45012,   7.54744,   7.11101,
              -1.08303, -0.651417,   1.47778,   11.1662,  0.288925,
              0.543421,   1.52315,  0.899992,   5.66699,  -10.7938,
              -0.624826,  0.551983,   2.02829,  -1.50053,  -7.06782,
              -1.07646, -0.238251,   2.15248,  -1.93367,  0.998062,
              -2.37593, -0.715036,    1.2021,   4.64283,   5.50439,
              -1.97033,   1.05419,   0.88446,   6.85845,   -8.2401,
              0.511671,   1.59969,  0.668511,   2.01078,  -11.1567,
              0.739805, -0.205666,   2.19111,  -1.68312,  -5.19492,
              -2.63537, -0.775507,   1.84313,   1.26641,   4.54483,
              -1.25479, -0.456243,   1.39067,   6.78639,   3.37954,
              2.46874,   3.15283,  0.408595,   7.34536,   -11.579,
              2.95337,   2.04668,   1.53156,   0.30056,  -11.7758,
              1.50927,  0.995085,   2.70701,  -2.91524,  -6.88716,
              1.11713,  -1.01366,   3.37843,  -2.06459,   1.41302,
              0.38442,  -1.39239,   2.54671,   6.78968,   5.86022,
              -0.237717,   1.78274,   1.79331,   9.10484,  -7.48854,
              1.96697,  0.595994,   2.54992,   4.37264,  -8.25502,
              0.279202,  -1.28502,   3.41409,  0.174649,   2.20281,
              -1.13495, -0.966977,   2.23015,   8.88276,   5.04827,
              -0.120631,   1.94788,    1.2486,   7.73192,  -9.82988,
              1.3372,   1.35175,   1.66208,   2.66613,  -10.8625,
              -1.55511,  0.256317,   2.85736, -0.621103,  -4.08806,
              0.0887002,  -1.36443,   2.40927,   4.48657,   5.14687

       };
       Result r;
       r.periodic_idx = 4;
       r.RV_idx = 1;
       r.LV_idx = 2;
       // these two are not yet sure
       r.baseline_idx = -2;
       r.perfusion_idx = -2;
       r.RV_peak = 16;
       r.LV_peak = 28;
       run( rdlength2, rdcomponents, init_real_data5,  r, false);
}


BOOST_FIXTURE_TEST_CASE( test_classifier_5_O1b, SlopeClassifierFixture )
{
       const size_t rdlength2 = 58;
       const size_t rdcomponents = 5;
       const float init_real_data5[rdlength2 * rdcomponents] = {
              -2.6871,  -2.28011,  -3.38407,  -5.24146, -0.433329,
              -4.05581,  -1.95256,  -3.37925,   2.80988,  0.599094,
              -3.75048,   1.29595,  -3.75867,   5.98233, -0.714227,
              -4.93509,   1.55628,  -3.81322,   1.94058,  -1.11442,
              -12.7718,  -3.99865,  -2.50314,  -5.03172,  -1.66106,
              -12.423,  -6.34533,   -3.5695,    4.7923,  0.615769,
              -7.27844,  -1.99303,  -4.66206,   5.29481, -0.0396878,
              -2.33217,  -2.42312,   -6.2368, -0.364265,  -1.80293,
              -0.19083,  -4.98816,  -8.02633,   2.50387, -0.622646,
              0.293297,  -3.72944,  -8.24063,   4.41801,     1.824,
              0.930989, -0.460923,  -8.41273,    1.7632,  0.912264,
              0.922238,  -3.49721,  -6.98733,  -5.99577,   2.99864,
              1.08277,  -5.67124,  -7.74424,   3.45292,  0.896835,
              1.45573,  -2.48071,   -6.8974,   6.50839,   1.06229,
              2.73141,  -2.56815,  -6.29424,   7.18928,  -1.04998,
              3.01166,  -2.36753,  -4.93286,   7.52773,   1.50169,
              4.07038,  -3.77088,  -2.05523,  -4.63485,   2.24413,
              3.70047,  -10.0267,  -0.89129, -0.813087,   3.70847,
              3.30206,  -4.14359,  -1.31542,    7.8118,  0.817536,
              3.75384,   -2.4407, -0.576383,   8.13343, -0.871883,
              4.26715,  -1.36786,   1.08194,   -4.7007, -0.631475,
              3.09109,  -5.21787,   2.23881,  -5.62555,   1.91847,
              3.45826,  -9.06248,   2.93734,   1.08651, 0.0848312,
              2.25296,  -3.32165,   2.74437,   7.05657,  0.803839,
              2.38633,  -2.35891,   3.30662,   6.15915,  -1.34672,
              1.50998,  0.917034,   3.37006,   8.20787,  -2.13057,
              1.88326,   3.22905,    3.6599,  -2.67486,  -1.12737,
              1.04652,  -2.83304,    3.5541,  -5.40886,  -0.13392,
              1.16108,  -8.38057,   3.95342,  0.521333,  -1.87044,
              0.0466207,  -1.15753,   3.51229,   6.90706, -0.223467,
              0.0515692,   2.71641,   3.52409,  -2.67069,   1.44024,
              1.27814,  -3.54781,   2.99009,  0.580945,  -2.13111,
              -0.177071,   1.81217,   3.15626,   6.41683,  -1.83154,
              -1.81497,   5.47261,   3.30884,   7.03806, 0.0866037,
              -1.25122,   7.35534,   3.01564,   7.32697,   -2.7325,
              -0.847239,   11.0478,   2.29227,  0.515116,  -0.50884,
              0.904234,    5.8041,    1.0846,  -10.7599,  0.766765,
              0.602109,  -1.42075,   1.70485,  -7.12815, -0.957428,
              0.20149,  -2.01366,   1.93427,  0.956927,  -1.35333,
              -0.358474,   4.57577,   1.49353,    5.6365,  -2.09492,
              0.70094,   7.10685,   1.04098,  -8.12112,  -1.63661,
              1.09538,   2.19787,  0.563649,  -11.1879,  0.513716,
              -0.165751,  -1.78239,   2.05377,   -5.2607,  0.336546,
              -0.22604,   1.21235,    1.8453,   4.61336,  -2.64461,
              -0.348253,   6.66242,   1.87904,   3.51905, -0.877165,
              2.20159,   7.44715,   0.70876,  -11.6048,   2.95761,
              1.46878,  0.291052,   1.42647,  -11.9153,   2.78655,
              0.996918,  -3.01427,   2.39401,  -7.04231,   1.05948,
              -0.600393,  -2.42861,   3.38715,   1.33696,  0.655421,
              -1.19521,    6.3803,   3.27061,   5.98266,  0.623105,
              1.30704,   9.16821,   2.18418,  -7.40491,  0.243124,
              0.192516,   4.21483,   2.84935,  -8.26614,   1.93438,
              -0.856208, -0.169894,   3.56324,   2.19267, -0.0406716,
              -0.797151,   8.62799,   2.94761,   5.22797, -0.698049,
              1.36926,   7.88262,   1.50564,  -9.76843,  0.277812,
              0.869443,   2.70616,   1.68791,  -10.9043,   1.28337,
              0.554758, -0.601956,   2.58451,  -4.11995,  -1.85009,
              -1.08934,   4.13692,   2.92538,   5.23481,   0.17826
       };
       Result r;
       r.periodic_idx = 3;
       r.RV_idx = 0;
       r.LV_idx = 2;
       // these two are not yet sure
       r.baseline_idx = 4;
       r.perfusion_idx = -2;
       r.RV_peak = 20;
       r.LV_peak = 28;
       run( rdlength2, rdcomponents, init_real_data5,  r, false);
}

BOOST_FIXTURE_TEST_CASE( test_classifier_O1_6, SlopeClassifierFixture )
{
       const size_t rdlength2 = 58;
       const size_t rdcomponents = 6;
       const float init_real_data5[rdlength2 * rdcomponents] = {
              -2.77055, -0.265872,  -5.20511,  -3.03599,  -2.36573,  -1.37281,
              -3.70677,  0.802571,   2.86372,  -2.80463,  -1.95339,  -2.51224,
              -3.77979, -0.925463,   5.92496,  -2.99664,   1.37772,  -2.25416,
              -3.04121,  -1.48487,    1.6407,  -4.22996,   2.37358,  -4.11419,
              -4.37504,  -1.05836,   -5.7074,  -4.97192,  -2.06339,  -12.6238,
              -5.32462,   1.36955,   4.39147,  -5.25371,  -4.73224,  -11.6956,
              -4.29338, -0.0281367,   5.05554,  -5.19541,  -1.03104,  -6.18687,
              -3.58669,  -1.99318, -0.303481,   -5.4649,  -2.51188, -0.749688,
              -4.52651, -0.663055,   2.95633,  -5.88969,  -5.94163,   2.41541,
              -4.61548,   1.61291,   4.93368,  -6.03006,  -4.60757,   3.27965,
              -4.16737,  0.281302,   2.13406,  -6.38532,  -1.15072,   3.80306,
              -3.46899,   2.86889,  -5.52987,  -5.45404,  -4.22922,   3.72366,
              -3.74604,  0.844147,   3.99733,  -5.69093,  -6.62785,   3.53537,
              -2.32414,  0.532771,   6.83017,  -5.68331,  -2.77011,    3.0281,
              -0.114875,  -1.84865,   7.32753,  -6.01068,  -2.36349,   2.82192,
              1.54025,  0.603296,    7.5741,  -5.75695,  -1.50922,   2.30902,
              5.18501,   1.44409,  -4.81972,  -5.00802,  -2.07493,   1.55884,
              6.17316,   3.54334, -0.858154,  -4.43865,  -8.20001,  0.342166,
              5.1087, 0.0764115,   7.59964,  -4.04693,  -2.41277,  0.170924,
              5.587,  -1.74285,   7.83363,  -3.26171, -0.752185,  0.266821,
              7.92806,  -1.63875,  -5.25932,  -3.21019,   1.01929, -0.246781,
              6.39269,   1.77972,  -5.86554,  -1.20087,  -3.57999, -0.379793,
              6.57641,  0.393975,  0.997287, 0.00417351,  -7.87877, -0.525804,
              4.82726,  0.713778,   6.89581,  0.570761,  -2.19699, -0.748064,
              4.76788,  -1.41258,   5.95744,   1.46316,  -1.49788, -0.698603,
              3.53216,  -2.33747,   7.95574,   2.16139,   1.56583, -0.965186,
              4.39952,  -1.58839,  -3.07949,   1.64209,   4.26953,   -0.6179,
              2.70297,    0.3492,  -5.44509,   2.53954,  -2.70895, -0.421934,
              2.36568, -0.712006,  0.729666,   3.71293,  -8.94077, -0.529215,
              1.08449,  0.243643,   6.95007,   3.55099,  -1.36585, -0.854529,
              1.35004,   1.52945,   -2.7861,   2.95928,   2.90249, -0.478728,
              0.837587,  -1.36629,  0.788361,   3.73275,  -4.46735,  0.731903,
              -0.214726,  -1.49478,   6.45403,   4.09836,   1.15305,  -0.31338,
              -1.3669,  0.229541,    6.9832,   4.27715,   5.03921,  -1.22923,
              -1.41967,  -2.81159,   7.19677,   4.37016,   6.72529, -0.769299,
              -0.929233,  -1.13408,  0.233934,   2.90592,   10.9949,  -0.11774,
              -1.20463,  0.672729,  -10.6882,   2.26684,   4.98174,   2.50442,
              -1.012, -0.245582,  -6.85946,   3.09461,  -2.68055,     1.562,
              -1.08954, -0.604067,   1.23592,   3.45378,  -3.23482,  0.837527,
              -1.3107,  -2.13096,   5.63291,   2.83408,   3.89145,  0.242212,
              -0.241386,    -2.074,  -8.28958,   1.53782,   6.81473,   1.29618,
              -1.00407,  0.684163,  -11.0206,   1.66135,   1.29339,   2.48209,
              -0.57366,   1.02914,  -5.06741,   2.77975,  -2.57482,  0.405277,
              -0.725153,  -2.35746,   4.66469,   2.94707,  0.477737, -0.0994434,
              -0.34211,   -1.2327,   3.35005,   2.34698,   6.57927, -0.166319,
              -0.309615,    2.4839,  -11.5508,   1.45364,   6.97442,   3.73423,
              -0.86171,   3.28761,  -11.5832,   2.62277, -0.822246,   3.06822,
              0.426132,   1.79282,  -6.79833,   2.84104,  -3.77112,   1.22864,
              -0.323964,   1.53105,    1.5485,   4.11238,  -3.20102, -0.438832,
              -0.910222,  0.598327,   5.90394,   4.05559,   6.07286, -0.687289,
              0.66744, -0.385575,  -7.62308,   2.27964,   9.15765,   1.65352,
              -0.426234,   2.12853,  -8.21457,   3.45807,   3.65076,   1.15887,
              -0.695612,  0.684047,   2.34079,   4.52003, -0.990196, -0.528606,
              -0.622126,   -1.0511,   5.02841,   3.63247,   8.46174,  -0.47157,
              0.113446, -0.206379,  -9.89127,   1.91991,   7.61286,   2.13717,
              -0.961247,   1.57599,  -10.7246,   2.81963,    1.7414,   2.29305,
              -0.0478495,  -1.27688,  -3.99191,   3.45723,  -1.50898,  0.699623,
              -1.13209,   0.38367,   5.25187,   3.93721,    3.5866, -0.492314

       };
       Result r;
       r.periodic_idx = 2;
       r.RV_idx = 5;
       r.LV_idx = 0;
       // these two are not yet sure
       r.baseline_idx = 1;
       r.perfusion_idx = -2; // 3
       r.RV_peak = 10;
       r.LV_peak = 20;
       run( rdlength2, rdcomponents, init_real_data5,  r, false);
}


BOOST_FIXTURE_TEST_CASE( test_classifier_O2_6, SlopeClassifierFixture )
{
       const size_t rdlength2 = 58;
       const size_t rdcomponents = 6;
       const float init_real_data5[rdlength2 * rdcomponents] = {
              -2.79725,  -0.361045,   -3.05947,   -3.67206,  -0.428245,   -2.91322,
              -0.776051,     -1.493,    -3.2787,    4.51879,  0.0100984,   -2.68526,
              2.02039,   -2.01242,   -3.25308,    6.60776,    1.45483,   -2.48587,
              3.39941,   -9.33322,   -1.98703,  -0.312353,   0.279883,   -3.74194,
              -2.9552,   -16.4639,    -1.9902,   -4.17009,  0.0914772,     -4.016,
              -2.01327,    -9.9362,   -2.97728,    5.26862,   0.111504,   -4.53924,
              1.90313,   -2.34704,   -3.12244,    3.82256,    1.09593,   -4.77696,
              -1.46684,  -0.520371,   -3.12888,   0.758751,  -0.493877,   -5.25192,
              -2.44908,  -0.248977,   -3.47773,    4.54051,   0.800596,   -5.30778,
              -1.0806,    1.10758,   -3.65517,    5.69498,  0.0384066,   -5.43741,
              2.18478,    1.89618,   -3.11868,   -1.26577,   0.396189,   -6.32725,
              -4.20119,    1.83785,   -2.85807,   -4.65704,   -1.75115,   -5.36033,
              -3.63532,    1.82044,   -3.43573,    5.20319,   0.386297,   -4.96366,
              -1.0216,    1.55263,   -3.02999,    6.92388,   0.113101,   -4.62775,
              -0.802921,   0.824656,   -1.88743,    7.46758,     1.1097,   -4.50375,
              0.595572,    1.80899,  -0.373035,      6.442,   0.312189,   -4.86807,
              -1.62256,    2.22764,    3.07704,   -5.18808,  -0.978168,   -5.33298,
              -6.83042,    1.02868,    4.28718,   0.483436,   -1.29708,   -4.68229,
              0.489628,    0.98984,    2.78051,    6.92249,    1.07579,   -3.38839,
              1.08606,   0.589096,    3.53981,    6.77945,   0.553666,   -2.59805,
              0.408871,    1.64269,    5.73429,   -7.15274,  -0.766158,   -3.01374,
              -5.56078,    1.89001,    5.90115,   -4.99904,   -1.86725,   -1.95768,
              -6.11383,   0.475619,    5.14128,    2.14216,  0.0682824,  -0.949274,
              -0.155035,  -0.306825,    3.88627,    6.62578,   0.509379,  0.0154463,
              -0.288651,   0.271475,     4.0659,    6.60542,    1.08157,   0.816006,
              2.06036,  -0.260089,    3.35886,    7.80388,   0.513991,    1.55238,
              4.14124,     0.3586,    4.64099,   -5.54186,  0.0748864,   0.623662,
              -2.24053,   0.817508,    3.49236,   -5.36042,     -1.051,    1.37931,
              -5.83726,   -0.68354,    3.11157,    2.92196,   0.335012,    1.84941,
              0.444395,   -1.37584,    2.14658,    7.34227,   0.777672,    2.37017,
              3.26412,    1.12235,    2.76372,   -4.86052,  -0.138705,    1.07232,
              -3.1244,   0.212633,     1.6833,    3.84253,  -0.410562,    2.34489,
              2.2996,   -1.11543,   0.855666,    6.88398,    1.62495,    3.38992,
              5.96859,  -0.523987,   0.165241,    7.16304,   0.923595,    3.64797,
              7.5891,   -1.36423,  -0.060059,    7.34145,     1.8088,    3.96869,
              8.86904,   0.553907,   0.668659,   -2.72515, -0.00300492,    1.84015,
              2.1867,    2.78165,   -1.30836,   -10.8371,   -1.50711,    2.52016,
              -2.72155,     1.0221,  -0.386451,   -6.24265,   -1.14438,    2.56111,
              -1.84138,  -0.872591,  -0.288188,    3.08501,    1.01316,    2.87331,
              5.46929,  -0.135979,  -0.375725,    4.00459,    1.01321,    2.39341,
              4.51148,    1.75543,  -0.784793,   -9.85794,  -0.572658,    1.70633,
              -1.43717,    1.76125,   -1.48393,   -10.6194,   -1.76816,    2.43781,
              -4.9326,   0.671211,  -0.191062,   -2.80398,  -0.146332,    2.70245,
              1.55751,  -0.664655,  -0.572384,    5.41251,   0.810215,    2.97117,
              7.33442,  -0.161308,   0.315256,  -0.661629,    1.55173,    1.94067,
              2.30489,    4.54933,   -2.01216,   -12.5764,   -2.70071,    2.33578,
              -6.49811,    3.44816,   -1.77391,   -11.3903,   -2.03526,    2.92323,
              -5.97034,    1.16151,   -0.40407,   -5.94628,   -1.16457,    3.26916,
              -3.47748,   0.385255,  -0.300799,    3.12343,   0.693829,    3.80919,
              5.37619,  0.0918239,  -0.577967,    4.63837,   0.889986,    4.04408,
              4.26689,    1.89123,  -0.463814,   -9.50634,  -0.206439,    3.09053,
              -3.35443,    2.02196,  -0.700084,   -6.34704,   -1.32942,    4.06389,
              -0.828717,   -0.33955,  -0.545394,     3.8502,    1.35505,    4.52391,
              7.322,  -0.238183,  -0.291989,    2.16792,   0.876632,    3.91714,
              2.8118,    3.25848,    -1.3027,    -11.201,   -1.18944,    2.87548,
              -2.26825,    2.59931,   -1.67594,   -10.0602,   -1.77894,    3.70191,
              -4.94661,   0.193411,  -0.526358,   -2.16885,   0.249542,    4.06018,
              3.38396,   0.137822,  -0.956599,    3.73577,   0.726636,    4.13762,
       };
       Result r;
       r.periodic_idx = 3;
       r.RV_idx = 1;
       r.LV_idx = 2;
       r.baseline_idx = 4;
       r.perfusion_idx = -2;
       r.RV_peak = -2;
       r.LV_peak = -2;
       run( rdlength2, rdcomponents, init_real_data5,  r, false);
}
