#                              -*- Mode: Perl -*- 
# List.pm|pkg-order-0.01/Debian/Package --- 
# Author           : Manoj Srivastava ( srivasta@tiamat.datasync.com ) 
# Created On       : Wed Jan 29 17:27:18 1997
# Created On Node  : tiamat.datasync.com
# Last Modified By : Manoj Srivastava
# Last Modified On : Sun May 16 09:06:36 1999
# Last Machine Used: glaurung.green-gryphon.com
# Update Count     : 263
# Status           : Unknown, Use with caution!
# HISTORY          : 
# Description      : 
# 
# 

use strict;
use Carp;
require 5.001;

use Debian::Package::Dependency_List;
use Debian::Package::Package;

=head1 NAME

  Debian::Package::List - List of packages.

=cut

package Debian::Package::List;
use Carp;
use FileHandle;
use IPC::Open3;


=head1 SYNOPSIS

  use Debian::Package::List;
    

=cut

=head1 DESCRIPTION

This module implements lists of packages. There is the base class
Debian::Package::List, and two derived classes, namely
Debian::Package::Installed and Debian::Package::New, corresponding to
the installed and new packages.

=cut

=head1 Debian::Package::List

This is the base class that implements lists of Packages. This
provides two virtual methods B<check> and B<extra> to be nstantiated
in the derived classes.

=cut

=head2 new

This is the constructor for the package. This takes a named parameter,
filename, which should be the path of a Packages file to use as the
data for the list. This is optional. Also, an optional parameter
'Ignored Headers' shall prevent recording those Headers (useful to
reduce memory usage by not recording descriptions). The argument of
Ignored headers is parsed to not ignore headers we consider vital,
like I<Package Status Version Provides Pre-Depends Conflicts Depends
Recommends Suggests> 


=cut

sub new {
  my $this = shift;
  my %params = @_;
  my $class = ref($this) || $this;
  my $self = {};
  
  #croak("Need filename") unless defined $params{'filename'};
  if (defined $params{'filename'} && $params{'filename'}) {
    # file name exists, had better be a real file
    croak("No such file $params{'filename'}") unless -f $params{'filename'};
  }

  bless $self => $class;
  $self->initialize(%params);
  $self->{' _Debug'} = 0;
  return $self;
}

=head2 check and extra

These are virtual functions to be instantiated in the derived
classes. They are no-ops in the base class.

=cut

sub check {
  return 1;
}

sub extra {
  return 1;
}

=head2 initialize

Internally, new uses the method B<initialize>. This routine opens the
file passed in as a named parameter filename, reads it in
paragraph mode (split on empty lines), and creates a package object
with each chunk. 

=cut

sub initialize {
  my $self = shift;
  my %params = @_;
  
  $self->{' _Vital Headers'}  = 
    q (Package Status Version Provides Pre-Depends Conflicts Depends
	Recommends Suggests); 

  my $vital_regexp = join ('|', (split(' ', $self->{' _Vital Headers'})));

  #croak("Need filename") unless defined $params{'filename'};
  if (defined $params{'Ignored Headers'} && $params{'Ignored Headers'}) {
    if ($self->{' _Ignored Headers'}) {
      $self->{' _Ignored Headers'} .= $params{'Ignored Headers'};
    }
    else {
      $self->{' _Ignored Headers'} = $params{'Ignored Headers'};
    }
    $self->{' _Ignored Headers'} =~ s/\b($vital_regexp)\b//igo;
    $self->{' _Ignored Headers'} =~ s/\s+/ /og;
  }

  $self->{' _CompareMethod'} = \&dpkg_compare_method;
  if (defined $params{'Compare Method'} && $params{'Compare Method'}) {
	$self->set_compare_method( $params{'Compare Method'} );
  }

  if (defined $params{'filename'} && $params{'filename'}) {
    # file name exists, had better be a real file
    croak("No such file $params{'filename'}") unless -f $params{'filename'};

    open(PACKAGE, "$params{'filename'}") or
      croak("Error reading $params{'filename'}");
    $/ = "";			# read in paragraph mode
    while (<PACKAGE>) {
      next if /^\s*$/;
      $self->add('Package_desc' => $_);
    }
    $/ = "\n";
    close(PACKAGE);
  }
  
  $self->{' _Targets'} = Debian::Package::List::Targets->new
    ('Package List' => $self);
  $self->{' _Separate Runs'} = "Pre-Depends";
}

=head2 add_ignored_headers

This routine adds to the current set of headers which are ignored if
they appear in the Package file. Certain headers can not be ignored,
currently I<Package Status Version Provides Pre-Depends Conflicts Depends
Recommends Suggests>. It takes in a required named parameter, 
I<Ignored Headers>, which is a space separated list of headers to be
ignored. 


=cut

sub add_ignored_headers {
  my $self = shift;
  my %params = @_;
  my $vital_regexp = join ('|', split(' ', $self->{' _Vital Headers'}));
  

  croak("Missing required paramater 'Ignored Headers'")
    unless $params{'Ignored Headers'};
  if ($self->{' _Ignored Headers'}) {
    $self->{' _Ignored Headers'} .= $params{'Ignored Headers'};
  }
  else {
    $self->{' _Ignored Headers'} = $params{'Ignored Headers'};
  }
  $self->{' _Ignored Headers'} =~ s/\b($vital_regexp)\b//igo;
  $self->{' _Ignored Headers'} =~ s/\s+/ /og;
}


=head2 delete_ignored_headers

This routine deletes from the current set of headers which are ignored if
they appear in the Package file. It takes in a required named parameter, 
I<Remove Headers>, which is a space separated list of headers to be
removed from the list. 

=cut

sub delete_ignored_headers {
  my $self = shift;
  my %params = @_;

  croak("Missing required paramater 'Remove Headers'")
    unless $params{'Remove Headers'};
  return unless $self->{' _Ignored Headers'};
  for (split (' ', $params{'Remove Headers'})) {
    next if /^\s*$/;
    $self->{' _Ignored Headers'} =~ s/$_//g;
  }
  $self->{' _Ignored Headers'} =~ s/\s+/ /og;
}


=head2 show_ignored_headers

This routine shows the current set of headers which are ignored if
they appear in the Package file.

=cut

sub show_ignored_headers {
  my $self = shift;
  my %params = @_;

  return $self->{' _Ignored Headers'};
}


=head2 set_compare_method

This method sets the routine which is used for comparing versions.
Using 'undef' as parameter selects the default method, which uses
dpkg --compare-versions.

=cut

sub set_compare_method {
  my $self = shift;
  my $new_method = shift;

  croak("New compare method is no code reference")
	  if $new_method && ref($new_method) ne "CODE";
  $self->{' _CompareMethod'} =
	  $new_method ? $new_method : \&dpkg_compare_method;
}


=head2 compare_versions

This is the generic function for comparing version numbers. It takes
three arguments, one version number, a relation, and another version
number. Relation can be what dpkg would accept as relations (i.e., <,
<<, <=, =, >=, >>, and >). The result is a boolean value.
	
=cut

sub compare_versions {
	my $self = shift;
	return &{$self->{' _CompareMethod'}}( @_ );
}


=head2 dpkg_compare_method

This is the default method for comparing versions. It simply calls
dpkg --compare-versions, which is the right thing to do in most cases.
It just does not work on non-Debian systems, so it has to be replaced
(with set_compare_method) for this case.
 	
=cut 

sub dpkg_compare_method {
	my($vers1, $rel, $vers2) = @_;
	my $rv;

	$rv = system 'dpkg', '--compare-versions', $vers1, $rel, $vers2;
	# need to invert exit status returned by system
	return !$rv if $rv == 0 || $rv == 1<<8;
	croak( "Bad exit status $rv from dpkg --compare-versions");
}


=head2 set_break_on_types

This routine adds to the current set of B<Type>s which, if they appear
as targets, force installation to break at that point into a separate
run. It takes one required named argument, I<Type List>.

=cut


sub set_break_on_types{
  my $self = shift;
  my %params = @_;
  

  croak("Missing required paramater 'Type List'")
    unless $params{'Type List'};
  $self->{' _Separate Runs'} = $params{'Type List'};
  $self->{' _Separate Runs'} =~ s/^\s*//og;
  $self->{' _Separate Runs'} =~ s/\s*$//og;
  return $self;
}

=head2 insert_breaks

This routine takes a ordered list of package names, and inserts breaks
into the list for the types specified by the internal variable array
' _Separate Runs'. It takes one required named argument, 'Ordered List'.

=cut

sub insert_breaks{
  my $self = shift;
  my %params = @_;
  my $type;
  my $index = 0;
  
  
  croak("Required parameter 'Ordered List' absent") unless
    $params{'Ordered List'}; 
  my @List = split ('\n', $params{'Ordered List'});
  
  for $type (split(' ', $self->{' _Separate Runs'})) {
    my @newlist = ();
    my $pkg;
    my $targets = '';
    
    print "\tDEBUG: \tTYPE $type\n" if $self->{' _Debug'};
    while (@List) {
      $pkg = pop @List;
      print "\tDEBUG: package $pkg \n" if $self->{' _Debug'};
      if (defined $self->{$pkg}->{'Pre-Depends'}) {
	my $target = $self->{$pkg}->{'Pre-Depends'};
	$target =~ s/\(.*\)/ /;
	$target =~ s/\s+//;
	next unless $target;
	print "\tDEBUG: $pkg found target $target\n"  if $self->{' _Debug'};
	
	# Now to handle pesky provides
	my @Targets = ();
	my @targets = split (/,/, $target);
	for (@targets) {
	  if ($self->{' _Provided'}->{$_}) {
	    push (@Targets, split (/,/, $self->{' _Provided'}->{$_}));
	  }
	  else {
	    push (@Targets, $_);
	  }
	}
	$targets .= " " . join (" ", @Targets);
      }
      my $pattern = $pkg;
      $pattern =~ s/(\W)/\\$1/g;
      if ($targets && $targets =~ /\b$pattern\b/) {
	unshift (@newlist, "START \t\t-$index");
	$index++;
	unshift (@newlist, "END \t\t-$index");
	$targets = '';
      }
      unshift (@newlist, "$pkg");
    }
    @List = @newlist;
  }
  my $ret = join ("\n", @List);
  return $ret;
}

=head2 add

Internally, B<initialize>uses the method B<add>. This routine creates
a package object with the required named argument
'Package_desc'. Next, the package object is passed to virtual method
B<check>, and, if it passes, it is added to this instance of the list,
and also adds in any symbols I<provided> as virtual packages
present. It is then passed to the virtual method B<extra> for any
extra processing of the package object.

=cut

sub add {
  my $self = shift;
  my %params = @_;
  my $pkg;
  my $ret;

  croak("Need Description") unless defined $params{'Package_desc'};
  if (defined $self->{' _Ignored Headers'} &&
	  $self->{' _Ignored Headers'}) {
    $pkg = Debian::Package::Package->new
      ('Package_desc' => $params{'Package_desc'},
       'Ignored Headers' => $self->{' _Ignored Headers'});
  }
  else {
    $pkg = Debian::Package::Package->new('Package_desc' => 
					 $params{'Package_desc'});
  }
  $ret = $self->check($pkg);
  if ($ret) {
    $self->{$pkg->{'Package'}} = $pkg;
    if (defined $pkg->{'Provides'}) {
      my @provided;
	
      @provided = split /,/, $pkg->{'Provides'};
      for (@provided) {
	s/\s//og;
	
	if (defined $self->{' _Provided'}->{$_}) {
	  $self->{' _Provided'}->{$_} .= ", $pkg->{'Package'}";
	}
	else {
	  $self->{' _Provided'}->{$_} = $pkg->{'Package'};
	}
      }
    }
    $self->extra($pkg);
  }
}


=head2 delete

This routine take a required named argument Name and deletes a package
with that name from the list if it exists. It also take care of
erasing the package from the provides list, if indeed this package
provided a virtual package.

=cut

sub delete {
  my $self = shift;
  my %params = @_;
  my $pkg;
  
  croak("Need Name") unless defined $params{'Name'};
  
  return unless defined $self->{$params{'Name'}};
  $pkg = $self->{$params{'Name'}};
  
  if (defined $pkg->{'Provides'}) {
    #Need to clean up provides
    my @provided;

    @provided = split /,/, $pkg->{'Provides'};
    for (@provided) {
      s/\s//og;
      my $pattern = $params{'Name'};
      $pattern =~ s/(\W)/\\$1/g;
      
      $self->{' _Provided'}->{$_} =~ s/, $pattern//g;
      $self->{' _Provided'}->{$_} =~ s/$pattern,//g;
      $self->{' _Provided'}->{$_} =~ s/$pattern//g;
      if ($self->{' _Provided'}->{$_} =~ /^\s*$/o) {
	delete $self->{' _Provided'}->{$_};
      }
    }
  }
  delete $self->{$params{'Name'}};
}

=head2 mark

This routine takes two required parameters, B<Package>, and B<Mark>
and marks B<Package> with B<Mark>. This is usually used to mark
packages in some fashion so they will not satisfy dependencies (for
packages in the installed list marked for removal, for example)

=cut

sub mark {
  my $self = shift;
  my %params = @_;
  my $pkg;

  croak("Need Package name") unless defined $params{'Package'};
  croak("Need Mark") unless defined $params{'Mark'};
  
  return unless defined $self->{$params{'Package'}};
  $pkg = $self->{$params{'Package'}};
  $pkg->mark("Mark" => $params{'Mark'});
}
    

=head2 unmark

This routine takes two required parameters, B<Package>, and B<Mark>
and removes B<Mark> from  B<Package>. 

=cut

sub unmark {
  my $self = shift;
  my %params = @_;
  my $pkg;

  croak("Need Package name") unless defined $params{'Package'};
  croak("Need Mark") unless defined $params{'Mark'};
  
  return unless defined $self->{$params{'Package'}};
  $pkg = $self->{$params{'Package'}};
  $pkg->unmark("Mark" => $params{'Mark'});
}
    

=head2 test_mark

This routine takes two required parameters, B<Package>, and B<Mark>
and tests to see if the B<Package> is marked with B<Mark>.

=cut

sub test_mark {
  my $self = shift;
  my %params = @_;
  my $pkg;

  croak("Need Package name") unless defined $params{'Package'};
  croak("Need Mark") unless defined $params{'Mark'};
  
  return unless defined $self->{$params{'Package'}};
  $pkg = $self->{$params{'Package'}};
  $pkg->test_mark("Mark" => $params{'Mark'});
}
    

=head2 show_mark

This routine takes one required parameter, B<Package>, and shows all
marks on it.

=cut

sub show_mark {
  my $self = shift;
  my %params = @_;
  my $pkg;

  croak("Need Package name") unless defined $params{'Package'};
  
  return unless defined $self->{$params{'Package'}};
  $pkg = $self->{$params{'Package'}};
  $pkg->show_mark();
}

=head2 list_marks

This routine collects information in a package list about all packages
with a mark on it. It returns a hash keyed on package
names.

=cut    

sub list_marks {
  my $self = shift;
  my %params = @_;
  my $name;
  my %result;

  croak("Need Mark") unless defined $params{'Mark'};
  
  foreach $name (keys %$self) {
    next if $name =~ /^\s+_/o;
    next unless $self->{$name}->{'Package'};
    my $pkg;
    $pkg = $self->{$name};
    
    $result{$name} = $name if defined
      $pkg->test_mark("Mark" => $params{'Mark'});
  }
  return %result;
}

=head2 print

This routine loops through the packages in the list, calling the print
function of each package object. It then runs through the list of
provided packages, printing the name of each package provided.

=cut

sub print {
  my $self = shift;
  my $name;
  
  #print "\t$self\n";
  #print "=" x 70, "\n";
  
  for $name (sort keys %$self) {
    my $pkg;
    next if $name =~ /^\s+_/o;
    next unless $self->{$name}->{'Package'};
    
    $pkg = $self->{$name};
    $pkg->print();
    print "-" x 70, "\n";
  }
  if (defined $self->{' _Provided'}) {
    my $field;
    printf "\nProvided Packages:\n";
    print "_" x 70, "\n";
    for $field (keys %{$self->{' _Provided'}}) {
      print "$field provided by $self->{' _Provided'}-{$field}\n";
    }
    print "_" x 70, "\n";
  }
  print "=" x 70, "\n";
}

=head2 print_name 

This routine loops through the packages in the list, printing the 
name. It then runs through the list of providedpackages, printing the
name of each package provided.

=cut

sub print_name {
  my $self = shift;
  my $name;

  #print "\t$self\n";
  #print "=" x 70, "\n";

  for $name (sort keys %$self) {
    
    next if $name =~ /^\s+_/o;
    
    #print "$name\t$self->{$name}\n\n";
    if (!$self->{$name}->{'Package'}) {
      carp("Current Package has no name [$name] ($self->{$name})\n");
    }
    #print "\tDEBUG:(list.pm):359:Name:$name\n";
    print "Name:$self->{$name}->{'Package'}\n";
  }
  if (defined $self->{' _Provided'}) {
    my $field;
    printf "\nProvided Packages:\n";
    for $field (keys %{$self->{' _Provided'}}) {
      print "$field provided by $self->{' _Provided'}->{$field}\n";
    }
    print "\n";
  }
}

=head2 names_as_string

This routine loops through the packages in the list and returns the names
of all packages as a string.

=cut

sub names_as_string {
  my $self = shift;
  my $name;
  my $string = '';

  for $name (sort keys %$self) {
      next if $name =~ /^\s+_/o;

      if (!$self->{$name}->{'Package'}) {
	carp("Current Package has no name [$name] ($self->{$name})\n");
      }

      $string .= " $self->{$name}->{'Package'}";
   }
   return $string;
}


=head2 init_depends 

As an extra processing of every package, the method B<extra> creates a
dependeny list object for the fields Depends, Pre-Depends,
Recommends, Suggests, Conflicts, and Replaces. (Stored in internal
fields whose names start with a blank followed by an underscore " _")

=cut

sub init_depends {
  my $self = shift;
  my $pkg  = shift;
  my @Types = ('Pre-Depends', 'Depends', 'Conflicts', 'Recommends', 
	       'Suggests', 'Replaces');
  
  croak("Current Package has no name $pkg->{'Package'}\n") unless
    $pkg->{'Package'};
  
  for (@Types) {
    $pkg->init_depends('Type'    => "$_");
  }
}

=head2 get_dependency_information 

This method B<extra> creates a dependeny list object for the fields
Depends, Pre-Depends, Recommends, Suggests, Conflicts, and
Replaces. (Stored in internal fields whose names start with a blank
followed by an underscore " _")

=cut

sub get_dependency_information {
  my $self = shift;
  my @Types = ('Pre-Depends', 'Depends', 'Conflicts', 'Recommends',
	       'Suggests', 'Replaces');
  my $name;

  foreach $name (sort keys %$self) {
    next if $name =~ /^\s+_/o;

    if (!$self->{$name}->{'Package'}) {
      carp("Current Package has no name($self->{$name})\n");
      next;
    }
    
    for (@Types) {
      $self->{$name}->init_depends('Type'    => "$_");
    }
  }
}


=head2 check_relations 

This routine performs the dependency checks for the new packages. For
each package in the list it calls the corresponding function in the
package, and stores the results in an associative array (keyed on
package name).

It takes a required named argument Field, whose value should be one of
Pre-Depends, Depends, Recommends, Suggests, Conflicts, or Replaces. 

It takes optional named arguments Consistent, (true if inconsistency
in the new list should cause a warning), Installed, (a package list of
installed packages), and I<Warn>, which just sends a warning, and
does not exclude a Package.

=cut

sub check_relations {
  my $self = shift;
  my %params = @_;
  my %Results;
  my $name;

  croak("Missing Parameter Field") unless $params{'Field'};
  
  foreach $name (sort keys %$self) {
    next if $name =~ /^\s+_/o;

    if (!$self->{$name}->{'Package'}) {
      carp("Current Package has no name($self->{$name})\n");
      next;
    }
    $params{'New'} = $self;
    $self->{$name}->check_relations(%params);
  }
}


=head2 order

This routine takes one named argument B<Field>, which contains the
name of the package field to generate ordering information about
(usually one of I<Pre-Depends Depends Recommends Suggests>) and
appends the ordering information so obtained to the package. This
method is typically called multiple times with different
arguments. The method B<order_string> below can be used to retrieve
the accumulated information.

=cut


sub order {
  my $self = shift;
  my %params = @_;
  my $name;
  
  croak("Missing Parameter Field") unless $params{'Field'};
  croak("Missing Parameter Installed")
    if !$params{'Installed'} && $params{'Field'} eq 'Conflicts';
  
  foreach $name (sort keys %$self) {
    my %Arguments;
    my $arg = \%Arguments;
    next if $name =~ /^\s+_/o;

    next unless $self->{$name}->{$params{'Field'}};


    if (!$self->{$name}->{'Package'}) {
      carp("Current Package has no name ($self->{$name})\n");
    }
    if (!defined $self->{$name}->{'Version'}) {
      warn "Current Package has no version" .
	" :($self->{$name}->{'Package'}): Skipping\n";
      next;
    }
    $Arguments{'Package'}   = $self->{$name};
    $Arguments{'New'}       = $self;
    $Arguments{'Installed'} = $params{'Installed'} if 
      $params{'Field'} eq 'Conflicts';
    print STDERR "\tDEBUG: List.pm 807: $self->{$name}->{'Package'}  $params{'Field'}\n"
      if $self->{' _Debug'};
    
    $self->{$name}->{" _$params{'Field'}"}->order($arg);
  }
}

=head2 order_string

This routine takes the ordering information accumulated in a package
list and returns it as a string.

=cut

sub order_string {
  my $self = shift;
  my $string = '';
  my $name;

  foreach $name (sort keys %$self) {
    next if $name =~ /^\s+_/o;
    next unless $self->{$name}->{'Package'};
    my $new_string = '';
    
    $new_string = $self->{$name}->order_string();
    print STDERR "\tDEBUG: List.pm 831 == $new_string\n"
      if $self->{' _Debug'};
    
    $string .= $new_string if $new_string;
  }
  return $string;
}


=head2 get_ordering

This routine gathers ordering information from the packages in
the list, and runs tsort on the gathered data, and returns the output
of tsort.

=cut


sub get_ordering {
  my $self = shift;
  my $string = '';
  my $pid = 0;
  my $output = '';
  my $errout = '';
  
  $string = $self->order_string();
  $pid = open3( \*Writer, \*Reader, \*ErrorPipe, "tsort" );
  Writer->autoflush(); # default here, actually
  print Writer "$string";
  Writer->close() or die "Could not close writer pipe:$!";
  
  while (<ErrorPipe>) { $errout .= $_; }
  ErrorPipe->close() or die "Could not close stderr pipe:$!";
  if ($errout) {
    print STDERR "External tsort invocation returned an error:\n";
    print STDERR $errout;
    print STDERR "";
  } ;

  while (<Reader>) { $output .= $_; }
  Reader->close() or die "Could not close reader pipe:$!";
  return $output;
}


=head2 set_fatal_failure_on_types

This routine adds to the current set of I<Show stopper> B<Type>s,
taking one required named argument, I<Type List>.

=cut

sub set_fatal_failure_on_types {
  my $self = shift;
  my %params = @_;
  my $name;
  
  croak("Missing required paramater 'Type List'")
    unless $params{'Type List'};
  
  foreach $name (sort keys %{$self}) {
    next if $name =~ /^\s+_/o;
    next unless $self->{$name}->{'Package'};
    
    $self->{$name}->{' _Results'}->set_fatal_failure_on_types
      ('Type List' => $params{'Type List'});
  }
}


=head2 print_result  

This routine prints out the current set of dependency failures for
this package list. It takes 2 named arguments, both required:
B<Type>, one of I<Conflicts Pre-Depends Depends Recommends Suggests>
B<Category>, one of I<Failed Found Warn Conflict> (B<note>, no
trailing s). 

Also, there are two special B<Type>a, I<All Critical>, which print
B<All> and B<Show Stoppers> types respectively. 

=cut

sub print_result {
  my $self = shift;
  my %params = @_;	
  my $name;

  croak("Required parameter 'Type' absent") unless
    $params{'Type'}; 
  croak("Required parameter 'Category' absent") 
    unless $params{'Category'}; 
  
  foreach $name (sort keys %$self) {
    next if $name =~ /^\s+_/o;
    next unless $self->{$name}->{'Package'};
    
    $self->{$name}->{' _Results'}->print_result
      ( 'Type' => "$params{'Type'}",
       'Category' => "$params{'Category'}");
  }
}


=head2 result_as_string

This routine returns the current set of dependency failures for
this package list as a string. It takes 2 named arguments, both required:
B<Type>, one of I<Conflicts Pre-Depends Depends Recommends Suggests>
B<Category>, one of I<Failed Found Warn Conflict> (B<note>, no
trailing s). 

Also, there are two special B<Type>a, I<All Critical>, which print
B<All> and B<Show Stoppers> types respectively. 

=cut

sub result_as_string {
  my $self = shift;
  my $string = '';
  my %params = @_;	
  my $name;

  croak("Required parameter 'Type' absent") unless
    $params{'Type'}; 
  croak("Required parameter 'Category' absent") 
    unless $params{'Category'}; 

  foreach $name (sort keys %$self) {
    next if $name =~ /^\s+_/o;
    next unless $self->{$name}->{'Package'};
    
    $string .= $self->{$name}->{' _Results'}->result_as_string
      ('Type' => "$params{'Type'}",
       'Category' => "$params{'Category'}");
  }
  return $string;
}

=head2 check_result 

This routine returns the number of the  current set of dependency
failures for this package list. It takes 2 named arguments, both required:
B<Type>, one of I<Conflicts Pre-Depends Depends Recommends Suggests>
B<Category>, one of I<Failed Found Warn Conflict> (B<note>, no
trailing s). 

Also, there are two special B<Type>a, I<All Critical>, which print
B<All> and B<Show Stoppers> types respectively. 

=cut

sub check_result {
  my $self = shift;
  my $count = 0;
  my %params = @_;	
  my $name;
  
  croak("Required parameter 'Type' absent") unless
    $params{'Type'}; 
  croak("Required parameter 'Category' absent") 
    unless $params{'Category'}; 

  foreach $name (sort keys %$self) {
    next if $name =~ /^\s+_/o;
    next unless $self->{$name}->{'Package'};
    
    $count += $self->{$name}->{' _Results'}->check
      ('Type' => $params{'Type'},
       'Category' => $params{'Category'});
  }
  return $count;
}


=head2 reset_result 

This routine recalculates the results field for all the packages in
the list.

=cut

sub reset_result {
  my $self = shift;
  my $name;
  
  # clear out old results
  foreach $name (sort keys %$self) {
    next if $name =~ /^\s+_/o;
    next unless $self->{$name}->{'Package'};
    
    $self->{$name}->reset_result();
  }
}


=head2 non_target_as_string

This routine returns the current set packages in this package list
which are not targets of the dependency checks run on this list. One
may select which of the results are reported, by selecting the
B<Type>, but, the result will be nothing unless the corresponding
I<check_relations> has been run.

=cut

sub non_target_as_string {
  my $self = shift;
  my $string = '';
  my %params = @_;	
  my $name;

  croak("Required parameter 'Type' absent") unless
    $params{'Type'}; 

  foreach $name (sort keys %$self) {
    my $ret = '';
    
    next if $name =~ /^\s+_/o;
    next unless $self->{$name}->{'Package'};
    
    $ret = $self->{' _Targets'}->is_target('Type' => "$params{'Type'}",
					    'Target' => "$name");
    $string .= " $name\n" unless $ret;
  }
  return $string;
}


=head2 target_as_string

This routine returns the current set packages in this package list
which are targets of the dependency checks run on this list. One
may select which of the results are reported, by selecting the
B<Type>, but, the result will be nothing unless the corresponding
I<check_relations> has been run.

=cut

sub target_as_string {
  my $self = shift;
  my $string = '';
  my %params = @_;	
  my $name;

  croak("Required parameter 'Type' absent") unless
    $params{'Type'}; 

  foreach $name (sort keys %$self) {
    my $ret = '';
    
    next if $name =~ /^\s+_/o;
    next unless $self->{$name}->{'Package'};

    $ret .= $self->{' _Targets'}->is_target('Type' => "$params{'Type'}",
					    'Target' => "$name");
    $string .= " $name\n\t$ret" if  $ret;
  }
  return $string;
}


=head2 test

This routine exercises the list class, initializing it with
/var/lib/dpkg/status, .and then printing the names of all packages
known. This is also a virtual function, in the sense that each derived
class should provide an implementation.

=cut

sub test {
  my ($class, $self) = @_;
  my $installed;

  $installed = Debian::Package::List->new('filename' => "/var/lib/dpkg/status");

  #$installed->print();
  $installed->print_name();
}

######################################################################
######################################################################

=head1 Debian::Package::Installed

This module is a derived class of the Debian::Package::List
class. This overrides the constructor new, the test function, and
provides an instantiation of the virtual function B<check>.

=cut

package Debian::Package::Installed;
@Debian::Package::Installed::ISA = qw(Debian::Package::List);

use Carp;

=head2 new

The constructor is over ridden to not require the named filename
parameter, instead we use the constant /var/lib/dpkg/status as a
default (the provided filename is still  used if it exists).

=cut

sub new {
  my $this = shift;
  my %params = @_;
  my $class = ref($this) || $this;
  my $self = {};
  $params{'filename'} = "/var/lib/dpkg/status" unless 
    defined $params{'filename'};

  bless $self => $class;
  $self->initialize(%params);
  return $self;
}


=head2 check

The virtual function check is instantiated, and ensures that the
package is in an ok installed status before it is included in the
package list.

=cut

sub check {
  my $self = shift;
  my $pkg = shift;
  croak("Error getting status for $pkg->{'Package'}") unless 
    defined $pkg->{'Status'};
  return $pkg->{'Status'} =~ /\s+ok\s+installed\b/o;
}

=head2 test

This routine exercises the installed list class, initializing it, and then
printing the names of all packages installed.

=cut


sub test {
  my ($class, $self) = @_;
  my $installed;
  $installed = Debian::Package::Installed->new();
  $installed->print_name();
}

######################################################################
######################################################################

=head1 Debian::Package:::New;

This module is a derived class of the Debian::Package::List
class. This provides an instantiation of the virtual function
B<extra>, and also the test method.  

=cut

package Debian::Package::New;
@Debian::Package::New::ISA = qw(Debian::Package::List);

use Carp;

=head2 extra

As an extra processing of every package, the method B<extra> creates a
dependeny list object for the fields Depends, Pre-Depends,
Recommends, Suggests, Conflicts, and Replaces. (Stored in internal
fields whose names start with a blank followed by an underscore " _")

=cut

sub extra {
  my $self = shift;
  my $pkg  = shift;
  
  croak("Current Package has no name $pkg->{'Package'}\n") unless
    $pkg->{'Package'};
  Debian::Package::List->init_depends($pkg);
}

=head2 test

This routine exercises the new list class, initializing it with a file
named Packages, and then printing the names of all packages therein.

=cut

sub test {
  my ($class, $self) = @_;
  my $candidates;

  $candidates = Debian::Package::New->new('filename' => "Packages");
  $candidates->print_name();
}

package Debian::Package::List::Targets;
use Carp;


=head1 SYNOPSIS

  use Debian::Package::List::Targets;
    

=cut


=head1 Debian::Package::List::Targets

This module is used internally to hold reverse information about
dependency (or conflict) hits..

=cut


=head2 new

This is the constructor for the package. It takes one named required
argument, 'Package List', which is a pointer to the containing package
list.

=cut



sub new {
  my $this = shift;
  my %params = @_;
  my $class = ref($this) || $this;
  my $self = {};
  
  croak("Missing required paramater 'Package List'")
    unless $params{'Package List'};
  bless $self => $class;
  $self->{' _Separate Runs'} = "Pre-Depends";
  $self->{'Package List'} = $params{'Package List'};
  return $self;
}

=head2 set_break_on_types

This routine adds to the current set of B<Type>s which, if they appear
as targets, force installation to break at that point into a separate
run. It takes one required named argument, I<Type List>.

=cut


sub set_break_on_types{
  my $self = shift;
  my %params = @_;
  

  croak("Missing required paramater 'Type List'")
    unless $params{'Type List'};
  $self->{' _Separate Runs'} = $params{'Type List'};
  $self->{' _Separate Runs'} =~ s/^\s*//og;
  $self->{' _Separate Runs'} =~ s/\s*$//og;
  return $self;
}

=head2 insert_breaks

This routine takes a ordered list of package names, and inserts breaks
into the list for the types specified by the internal variable array
' _Separate Runs'. It takes one required named argument, 'Ordered List'.

=cut

sub insert_breaks{
  my $self = shift;
  my %params = @_;
  my $type;
  my $index = 0;
  
  
  croak("Required parameter 'Ordered List' absent") unless
    $params{'Ordered List'}; 
  my @List = split ('\n', $params{'Ordered List'});
  
  for $type (split(' ', $self->{' _Separate Runs'})) {
    my @newlist = ();
    my $pkg;
    my $dependants = '';
    
    next unless defined $self->{'Targets'}->{$type};
    for $pkg (@List) {
      if (defined $self->{'Targets'}->{$type}->{$pkg}) {
	$dependants .= " $self->{'Targets'}->{$type}->{$pkg}";
      }
      my $pattern = $pkg;
      $pattern =~ s/(\W)/\\$1/g;
      if ($dependants && $dependants =~ /\b$pattern\b/) {
	push (@newlist, "END \t\t$index");
	$index++;
	push (@newlist, "START \t\t$index");
	$dependants = '';
      }
      push (@newlist, "$pkg");
    }
    @List = @newlist;
  }
  my $ret = join ("\n", @List);
  return $ret;
}

=head2 add

This routine adds to the current set of dependency targets for
this package. It takes 3 named arguments, all of them required:
B<Type>, one of I<Conflicts Pre-Depends Depends Recommends Suggests>
B<Target>, and B<Dependant>.

=cut

sub add {
  my $self = shift;
  my %params = @_;              

  croak("Required parameter 'Type' absent") unless
    $params{'Type'}; 
  croak("Required parameter 'Target' absent") 
    unless $params{'Target'}; 
  croak("Required parameter 'Dependant' absent") 
    unless $params{'Dependant'};

  if (defined $self->{'Targets'}->{$params{'Type'}}->{$params{'Target'}}) {
    $self->{'Targets'}->{$params{'Type'}}->{$params{'Target'}} .= 
      " $params{'Dependant'}";
  }
  else {
    $self->{'Targets'}->{$params{'Type'}}->{$params{'Target'}} = 
      "$params{'Dependant'}";
  }
}


=head2 remove

This routine deletes a target from the current set of dependency
targets for this package list. It takes 2 named arguments, both
required: B<Type>, one of I<Conflicts Pre-Depends Depends Recommends
Suggests> and B<Target>.

=cut

sub remove {
  my $self = shift;
  my %params = @_;              

  croak("Required parameter 'Type' absent") unless
    $params{'Type'}; 
  croak("Required parameter 'Target' absent") 
    unless $params{'Target'}; 

  if (defined $self->{'Targets'}->{$params{'Type'}}->{$params{'Target'}}) {
    delete $self->{'Targets'}->{$params{'Type'}}->{$params{'Target'}};
  }
}


=head2 is_target

This routine . It takes 2 named arguments, both required:
B<Type>, one of I<Conflicts Pre-Depends Depends Recommends Suggests>
and B<Target>. There is an undocumented B<Type> I<All>, which cycles
over it all.

=cut

sub is_target {
  my $self = shift;
  my %params = @_;              

  croak("Required parameter 'Type' absent") unless
    $params{'Type'}; 
  croak("Required parameter 'Target' absent") 
    unless $params{'Target'}; 

  if ($params{'Type'} =~ m/All/i) {
    my $dependants = undef;
    
    for (keys %{ $self->{'Targets'} }) {
      my $ret = '';
      
      $ret = 
	$self->do_is_target ('Type' => "$_",
			     'Target' => "$params{'Target'}");
      $dependants .= "$ret " if defined $ret;
    }
    return $dependants;    
  }  
  else {
    return $self->do_is_target('Type' => "$params{'Type'}",
			       'Target' => "$params{'Target'}");
  }
}

sub do_is_target {
  my $self = shift;
  my %params = @_;              

  croak("Required parameter 'Type' absent") unless
    $params{'Type'}; 
  croak("Required parameter 'Target' absent") 
    unless $params{'Target'}; 

  if (defined $self->{'Targets'}->{$params{'Type'}}->{$params{'Target'}}) {
    return $self->{'Targets'}->{$params{'Type'}}->{$params{'Target'}};
  }
  else {
    return undef;
  }
}


=head2 print_target 

This routine prints out the current set of dependency targets for this
package list. It takes one required named argument: B<Type>, one of
I<Conflicts Pre-Depends Depends Recommends Suggests>

Also, there is a special B<Type>, I<All>, which prints B<All>
types. Actually, this procedure merely handles the Special types, and
passes the work to the next routine.

=cut

sub print_target {
  my $self = shift;
  my %params = @_;      

  
  croak("Required parameter 'Type' absent") unless
    $params{'Type'}; 

  if ($params{'Type'} =~ m/All/i) {
    for (keys %{$self->{'Targets'}}) {
      $self->do_print('Type' => "$_");
    }
  }
  else {
    $self->do_print('Type' => "$params{'Type'}");
  }
}

=head2 do_print 

This routine prints out the current set of dependency targets for
this package. It takes one required named argument: B<Type>, one of
I<Conflicts Pre-Depends Depends Recommends Suggests>

=cut

sub do_print {
  my $self = shift;
  my %params = @_;      
  my $array;

  croak("Required parameter 'Type' absent") unless
    $params{'Type'}; 

  return unless defined $self->{'Targets'}->{$params{'Type'}};
  print STDERR "$params{'Type'} Targets:\n";
  for (keys %{$self->{'Targets'}->{$params{'Type'}}}) {
    print STDERR "\t$_ is the target of " . 
      "$self->{'Targets'}->{$params{'Type'}}->{$params{$_}}\n";
  }
}



=head1 CAVEATS

This is very inchoate, at the moment, and needs testing.

=cut

=head1 BUGS

None Known so far.

=cut

=head1 AUTHOR

Manoj Srivastava <srivasta@debian.org>

=cut


1;

