/*
 * Drawer.java
 *
 * created: mpichler, 19960923
 *
 * changed: kwagen, 19970917
 *
 * $Id: Drawer.java,v 1.30 1997/09/25 15:02:06 apesen Exp $
 */


package iicm.vrml.vrwave;

import iicm.vrml.pw.*;
import iicm.ge3d.GE3D;
import iicm.vrml.vrwave.pwdat.*;
import iicm.utils3d.Mat4f;
import iicm.utils3d.Vec3f;

import java.net.*;
import java.util.Vector;

/**
 * Drawer - drawing traversal
 * Copyright (c) 1996,97 IICM
 *
 * @author Michael Pichler, Karl Heinz Wagenbrunn
 * @version 0.3, latest change:  7 May 97
 */


class Drawer extends Traverser
{
  Scene scene_;
  float[] white = { 1.0f, 1.0f, 1.0f };
  int numlights_;
  int drawmode_ = 0;
  boolean ccw_ = true;
  boolean autobfc_;
  boolean bfc_ = true;
  boolean repeats_ = true;
  boolean repeatt_ = true;
  int texture_ = 0;
  URLServer urlserver_;
  Thread thread_;
  URL currbaseurl_;
  int quadslices_ = 0;
  Material currmat_ = null;
  boolean autolighting_ = true;
  boolean lighting_;

  /**
   * draw the scene graph
   */

  void drawScene (Scene s, GroupNode root)
  {
    scene_ = s;
    currbaseurl_ = scene_.baseurl_;
    drawmode_ = s.curDrawingMode ();
    // System.out.println ("=== Drawer::drawScene ===");
    // compare with VRMLScene::draw

    GE3D.setDrawMode (drawmode_);  // activate drawing mode
    GE3D.setBackgroundColor (s.getColor (Scene.COLBACKGROUND));
    GE3D.clearScreen ();

    if (root == null)
      return;

    // GE3D.fillColor3f (white);
    GE3D.defaultMaterial ();  // activate default material
    currmat_ = null;

    GE3D.loadIdentity ();  // replace current matrix with identity (for camera)

    s.setCamera ();  // set viewing camera

    if (scene_.getHeadlight () || !scene_.hasLightSource ())  // activate viewing light (headlight)
      GE3D.setHeadLight (white);

    numlights_ = 0;  // scene lights start with 1

    // (de)activate backface culling (default on)
    autobfc_ = (scene_.backfaceCulling () == Scene.TRISTATE_AUTO);
    bfc_ = (scene_.backfaceCulling () != Scene.TRISTATE_OFF);
    GE3D.hint (GE3D.HINT_BACKFACECULLING, bfc_ ? 1 : 0);
    // assert: ccw_ is true here

    // (de)activate lighting
    autolighting_ = (scene_.getLighting () == Scene.TRISTATE_AUTO);
    lighting_ = (scene_.getLighting () != Scene.TRISTATE_OFF);
    GE3D.hint (GE3D.HINT_LIGHTING, lighting_ ? 1 : 0);

    if (drawmode_ == GE3D.ge3d_texturing)
    {
      GE3D.setTextureMipmapping (scene_.getTextureMipmapping ());
      GE3D.hint (GE3D.HINT_TEXLIGHTING, scene_.getTexLighting () ? 1 : 0);
    }

    if (quadslices_ != scene_.getQuadslices ())
    {
      quadslices_ = scene_.getQuadslices ();
      GE3D.hint (GE3D.HINT_QUADSLICES, quadslices_);
    }      

    // do some initializations ...

    GE3D.pushMatrix ();

    tGroupNode (root);  // traverse scene graph for drawing

    GE3D.popMatrix ();

    // current matrix is again viewing matrix

    // restore a "reasonable" state in ge3d
    if (!ccw_)
    {
      GE3D.hint (GE3D.HINT_CCW, 1);
      ccw_ = true;
    }
    if (!bfc_)
    {
      GE3D.hint (GE3D.HINT_BACKFACECULLING, 1);
      bfc_ = true;
    }

    if (drawmode_ == GE3D.ge3d_texturing)  // turn off texturing
    {
      GE3D.alphaTest (0.0f);
      GE3D.doTexturing (0);
    }

    // turn off scene light sources (incl. headlight)
    GE3D.deactivateLights (0, numlights_);

    GE3D.fillColor3f (white);  // back to default material

  } // drawScene

// void tGroup (Group g)
// {
//   System.out.println ("draw Group");
//   tGroupNode (g);
// }

  // Grouping nodes: default behaviour - traverse children
  // void tGroup (Group g)  { tGroupNode (g); }
  // void tAnchor (Anchor g)  { tGroupNode (g); }

  // tBillboard
  protected void tBillboard (Billboard billb)
  {
    BillboardData dat = (BillboardData) billb.userdata;

    float[] mat = new float[16];
    GE3D.getMatrix (mat);

    float[] pos = { -mat[12], -mat[13], -mat[14] };

    float[] inv = new float[9];
    if (Mat4f.invertMatrix33of44 (mat, inv))  // non-singular viewing matrix
    {
      // viewer position in local coordinate system
      Vec3f v = new Vec3f (pos[0] * inv[0] + pos[1] * inv[3] + pos[2] * inv[6],
        pos[0] * inv[1] + pos[1] * inv[4] + pos[2] * inv[7],
        pos[0] * inv[2] + pos[1] * inv[5] + pos[2] * inv[8]);
      // System.out.println ("billboard-to-viewer vector: " + v);

      Vec3f y = new Vec3f (billb.axisOfRotation.getValue ());

      if (y.value_[0] == 0 && y.value_[1] == 0 && y.value_[2] == 0)  // no axis of rotation: "viewer-alignment"
      {
        // viewer's up vector
        Vec3f up = new Vec3f (inv[3], inv[4], inv[5]);
        // System.out.println ("viewer's up vector: " + up);

        v.normalize ();  // will be z-axis vector of new coordinate system
        up.normalize ();  // will be y-axis vector

        // get x-axis vector from cross product of z and y
        Vec3f x = new Vec3f ();
        x.cross (up, v);
        x.normalize ();

        // create transformation matrix
        float[] trf = dat.trfmatrix_;
        trf[0] = x.value_[0];
        trf[1] = x.value_[1];
        trf[2] = x.value_[2];
        trf[4] = up.value_[0];
        trf[5] = up.value_[1];
        trf[6] = up.value_[2];
        trf[8] = v.value_[0];
        trf[9] = v.value_[1];
        trf[10] = v.value_[2];

        GE3D.pushThisMatrix (trf);
        tGroupNode (billb);  // draw children
        GE3D.popMatrix ();
      }
      else  // use axis of rotation
      {
        // axis of rotation will be y-axis vector of new coordinate sytem
        y.normalize ();

        // normal vector of plane defined by axis of rotation and billboard-to-viewer line
        // (= x-axis vector of new coordinate sytem)
        Vec3f x = new Vec3f ();
        x.cross (y, v);
        x.normalize ();

        // get z-axis vector from cross product of x and y
        Vec3f z = new Vec3f ();
        z.cross (x, y);

        // transform z-axis vector of current coordinate system into the new
        Vec3f nz = new Vec3f (x.value_[2], y.value_[2], z.value_[2]);
        // System.out.println ("z-vector in new coordinate system: " + zz);

        // calculate angle which the z-axis vector of the current coordinate system has to be rotated around
        // the y-axis of the new coordinate system to lie in its x=0 plane
        double angle = Math.acos (nz.value_[2]);
        if (nz.value_[0] > 0)
          angle = -angle;
        // System.out.println ("angle: " + angle / Math.PI * 180);

        GE3D.pushMatrix ();
        GE3D.loadIdentity ();
        GE3D.rotatef3f (y.value_, (float) angle);
        GE3D.getMatrix (dat.trfmatrix_);
        GE3D.popMatrix ();
        GE3D.pushThisMatrix (dat.trfmatrix_);
        tGroupNode (billb);  // draw children
        GE3D.popMatrix ();
      }
    }
  } // tBillboard


  // void tCollision (Collision g)  { tGroupNode (g); }


  protected void tTransform (Transform trf)
  {
    TransformData dat = (TransformData) trf.userdata;
    GE3D.pushThisMatrix (dat.getMatrix ());
    tGroupNode (trf);  // draw children
    GE3D.popMatrix ();
  }

  // Inline
  protected void tInline (Inline inline)
  { 
    InlineData dat = (InlineData) inline.userdata;

    if (dat.inline_ != null)  // inline fetched
    {
      URL tmp = currbaseurl_;
      currbaseurl_ = dat.url_;
      tGroupNode (dat.inline_);
      currbaseurl_ = tmp;
    }
    else  // show bounding box
    {
      if (dat.bboxmin_ != null)
        GE3D.drawWireCube (dat.bboxmin_.value_, dat.bboxmax_.value_);
    }
  }

  // LOD
  protected void tLOD (LOD lod)
  {
    Vector level = lod.level.getNodes ();
    LODdata dat = (LODdata) lod.userdata;

    if (level.size () == 0)
    {
      dat.drawnchild_ = -1;
      return; 
    }

    int active = 0;  // default, if range not defined (could be adjusted to frame rate)

    if (lod.range.getValueCount () > 0)  // calculate distance and use range to choose level
    {
      float[] mat = new float[16];
      GE3D.getMatrix (mat);

// System.out.println (mat[0] + " " + mat[1] + " " + mat[2] + " " + mat[3]);
// System.out.println (mat[4] + " " + mat[5] + " " + mat[6] + " " + mat[7]);
// System.out.println (mat[8] + " " + mat[9] + " " + mat[10] + " " + mat[11]);
// System.out.println (mat[12] + " " + mat[13] + " " + mat[14] + " " + mat[15]);

      float[] pos = { -mat[12], -mat[13], -mat[14] };
      float[] inv = new float[9];
      if (Mat4f.invertMatrix33of44 (mat, inv))
      {
        Vec3f mult = new Vec3f (pos[0] * inv[0] + pos[1] * inv[3] + pos[2] * inv[6],
          pos[0] * inv[1] + pos[1] * inv[4] + pos[2] * inv[7],
          pos[0] * inv[2] + pos[1] * inv[5] + pos[2] * inv[8]);

        // System.out.println ("position in local coordinate system = " + mult);

        Vec3f center = new Vec3f (lod.center.getValue ());

        mult.decrease (center); 

        double dist = Math.sqrt (Vec3f.dot (mult, mult));
        // System.out.println ("distance = " + dist);

        float[] range = lod.range.getValueData ();
        for (active = 0; active < lod.range.getValueCount () && active < level.size () - 1 &&
             dist >= range[active]; active++);
        // System.out.println ("level = " + active);
      }
      // else
      //   System.out.prinlnt ("singular viewing matrix");
    }

    // check if choosen level node is an Inline node
    // if yes, check also, if its children have been loaded already
    while (active < level.size () - 1)
    {
      try
      {
        InlineData idat = (InlineData) ((Inline) level.elementAt (active)).userdata;
        if (idat.inline_ != null)  // show it
          break;
      }
      catch (ClassCastException e)
      {
        break;  // non-Inline node: show it
      }
      active++;
    }

    ((Node) level.elementAt (active)).traverse (this);
    dat.drawnchild_ = active;

  } // LOD


  // Switch
  protected void tSwitch (Switch sw)
  {
    Vector choice = sw.choice.getNodes ();
    int whichchoice = sw.whichChoice.getValue ();
    SwitchData dat = (SwitchData) sw.userdata;

    if (whichchoice >= 0 && whichchoice < choice.size ())
    {
      ((Node) choice.elementAt (whichchoice)).traverse (this);
      dat.drawnchild_ = whichchoice;
    }
    else
      dat.drawnchild_ = -1;
  } // Switch


  // Sound
  protected void tAudioClip (AudioClip n)  { }
  protected void tSound (Sound n)  { }


  // *** Lighting ***

  // DirectionalLight
  protected void tDirectionalLight (DirectionalLight lgt)
  {
    if (drawmode_ < GE3D.ge3d_flat_shading || !lgt.on.getValue ())
      return;

    // currently only color and intensity are used; ambient, attenuation, and radius ignored
    GE3D.activateLightSource (++numlights_,  // direction reversed in activateLightSource
      lgt.color.getValue (), lgt.intensity.getValue (), lgt.direction.getValue (),
      0.0f  // directional
    );
  }

  // PointLight
  protected void tPointLight (PointLight lgt)
  {
    if (drawmode_ < GE3D.ge3d_flat_shading || !lgt.on.getValue ())
      return;

    // currently only color and intensity are used; ambient, attenuation, and radius ignored
    GE3D.activateLightSource (++numlights_,
      lgt.color.getValue (), lgt.intensity.getValue (), lgt.location.getValue (),
      1.0f  // positional
    );
  }

  protected void tSpotLight (SpotLight n)  { }

  // *** Scripting ***
  protected void tScript (Script n)  { }

  // WorldInfo not drawn
  protected void tWorldInfo (WorldInfo n)  { }

  // *** Sensor Nodes ***
  protected void tCylinderSensor (CylinderSensor n)  { }
  protected void tPlaneSensor (PlaneSensor n)  { }
  protected void tProximitySensor (ProximitySensor n)  { }
  protected void tSphereSensor (SphereSensor n)  { }
  protected void tTimeSensor (TimeSensor n)  { }
  protected void tTouchSensor (TouchSensor n)  { }
  protected void tVisibilitySensor (VisibilitySensor n)  { }


  // *** Geometry Nodes ***

  // Shape
  protected void tShape (Shape shape)
  {
    Node geom = shape.geometry.getNode ();
    if (geom == null)  // shape without geometry not drawn
      return;

    if (drawmode_ == GE3D.ge3d_texturing)
    {
      GE3D.loadTextureIdentity ();
      GE3D.doTexturing (0);
    }

    if (autolighting_)
      GE3D.hint (GE3D.HINT_LIGHTING, (shape.appearance.getNode () != null) ? 1 : 0);

    // TODO: keep track of traversal state on changing properties
    // only have to activate when different
    if (shape.appearance.getNode () != null)
      Node.traverseNode (this, shape.appearance.getNode ());
    else
    {
      GE3D.defaultMaterial ();
      GE3D.fillColor3f (white);
      currmat_ = null;
    }

    // may have to push/pop some states here
    geom.traverse (this);  // draw geometry
  } // Shape

  // update backface culling state (only called for autobfc_)
  private void setBFC (boolean bfc)
  {
    if (bfc != bfc_)
    { 
      GE3D.hint (GE3D.HINT_BACKFACECULLING, bfc ? 1 : 0);
      bfc_ = bfc;
    }
  }

  // update CCW state
  private void setCCW (boolean ccw)
  {
    if (ccw != ccw_)
    {
      GE3D.hint (GE3D.HINT_CCW, ccw ? 1 : 0);
      ccw_ = ccw;
    }
  }

  // update texture display list
  private void setTexture (int displaylist)
  {
    if (displaylist != texture_)
    {
      GE3D.applyTexture (displaylist);
      texture_ = displaylist;
    }
  }

  // update texture wrap mode
  private void setRepeat (boolean s, boolean t)
  {
    if (s != repeats_ || t != repeatt_)
    {
      GE3D.textureRepeat (s ? 1 : 0, t ? 1 : 0);
      repeats_ = s;
      repeatt_ = t;
    }
  }

  // Box
  public void tBox (Box box)
  {
    if (autobfc_)
      setBFC (true);
    setCCW (true);

    BoxData dat = (BoxData) box.userdata;
    GE3D.drawCube (dat.min_, dat.max_);
  }

  // Cone
  protected void tCone (Cone cone)
  {
    if (autobfc_)
      setBFC (true);
    setCCW (true);

    float h = cone.height.getValue ();
    ConeData dat = (ConeData) cone.userdata;
    GE3D.drawCylinder (cone.bottomRadius.getValue (), 0, - h / 2.0f, h, dat.parts_);
  }

  // Cylinder
  protected void tCylinder (Cylinder cylinder)
  {
    if (autobfc_)
      setBFC (true);
    setCCW (true);

    float r = cylinder.radius.getValue ();
    float h = cylinder.height.getValue ();
    CylinderData dat = (CylinderData) cylinder.userdata;
    GE3D.drawCylinder (r, r, - h / 2.0f, h, dat.parts_);
  }

  protected void tElevationGrid (ElevationGrid n)  { }

  // Extrusion
  protected void tExtrusion (Extrusion extr)
  {
    // nothing to be done if spine curve or cross section not defined
    if (extr.crossSection.getValueCount () < 1 || extr.spine.getValueCount () < 1)
      return;

    ExtrusionData dat = (ExtrusionData) extr.userdata;

    if (autobfc_)
      setBFC (extr.solid.getValue ());
    setCCW (extr.ccw.getValue ());

    if (dat.convexify != null && drawmode_ != GE3D.ge3d_wireframe)
      GE3D.drawFaceSet (
        dat.coords, dat.convexify.getCount (), dat.convexify.getData (), dat.newfnormals.getData (),
        dat.texcoords, dat.newtexcinds.getCount (), dat.newtexcinds.getData (),
        0, null,
        0, null,
        GE3D.MATB_OVERALL,
        dat.normallist, (dat.normalindex != null) ? dat.normalindex.length : 0, dat.normalindex
      );
    else
      GE3D.drawFaceSet (
        dat.coords, dat.numcoordinds, dat.coordinds, dat.facenormals,
        dat.texcoords, dat.numcoordinds, dat.texcoordinds,
        0, null,
        0, null,
        GE3D.MATB_OVERALL,
        dat.normallist, (dat.normalindex != null) ? dat.normalindex.length : 0, dat.normalindex
      );
  }

  // IndexedFaceSet
  protected void tIndexedFaceSet (IndexedFaceSet faceset)
  {
    IndexedFaceSetData dat = (IndexedFaceSetData) faceset.userdata;
    MFVec3f coords = dat.coords_;
    if (coords == null)
      return;

    MFInt32 mfcindex = faceset.coordIndex;
    int[] coordinds = mfcindex.getValueData ();
    int numcoordinds = mfcindex.getValueCount ();
    if (dat.convexify_ != null && drawmode_ != GE3D.ge3d_wireframe)
    { // coord indices after triangulation
      coordinds = dat.convexify_.getData ();
      numcoordinds = dat.convexify_.getCount ();
    }

    int[] texcoordinds = null;
    int numtexinds = 0;
    if (faceset.texCoordIndex.getValueCount () > 0)
    {
      if (dat.convexify_ == null || drawmode_ == GE3D.ge3d_wireframe)
      { // texture coordinate indices after triangulation
        texcoordinds = faceset.texCoordIndex.getValueData ();
        numtexinds = faceset.texCoordIndex.getValueCount ();
      }
      else
      {
        texcoordinds = dat.newtexcoordinds_.getData ();
        numtexinds = dat.newtexcoordinds_.getCount ();
      }
    }
    else
    {
      texcoordinds = coordinds;
      numtexinds = numcoordinds;
    }

    MFColor color = null;  // defaults if field color is NULL
    int[] colorinds = null;
    int numcolorinds = 0; 
    int matb = GE3D.MATB_OVERALL;
    if (scene_.materials ())
    {
      color = dat.color_;
      if (color != null)  // field color defined (Color node)
      {
        if (faceset.colorPerVertex.getValue ())  // pervertexindexed
        {
          matb = GE3D.MATB_PERVERTEXINDEXED;
          if (faceset.colorIndex.getValueCount () > 0)  // field colorIndex contains indeces
          {
            colorinds = faceset.colorIndex.getValueData ();
            numcolorinds = faceset.colorIndex.getValueCount ();
          }
          else  // field colorIndex is empty, use coordIndex instead
          {
            colorinds = coordinds;
            numcolorinds = numcoordinds;
          }
        }
        else  // perface(indexed)
        {
          if (dat.newcolorinds_ == null || drawmode_ == GE3D.ge3d_wireframe)                        
          {
            if (faceset.colorIndex.getValueCount () > 0)  // field colorIndex contains indeces (perfaceindexed)
            {
              matb = GE3D.MATB_PERFACEINDEXED;
              colorinds = faceset.colorIndex.getValueData ();
              numcolorinds = faceset.colorIndex.getValueCount ();
            }
            else  // field colorIndex is empty (perface)
              matb = GE3D.MATB_PERFACE;
          }
          else
          {
            colorinds = dat.newcolorinds_.getData ();
            numcolorinds = dat.newcolorinds_.getCount ();
            matb = GE3D.MATB_PERFACEINDEXED;
          }
        }        
      }
    }

    float[] normallist = dat.normallist_;
    int[] normalinds = dat.normalindex_;

    if (autobfc_)
      setBFC (faceset.solid.getValue ());
    setCCW (faceset.ccw.getValue ());

    GE3D.drawFaceSet (
      coords.getValueData (), numcoordinds, coordinds, (dat.fnormals_ != null) ? dat.fnormals_.getData () : null,
      dat.texcoords_, numtexinds, texcoordinds, (color != null) ? color.getValueCount () : 0,
      (color != null) ? color.getValueData () : null, numcolorinds, colorinds, matb,
      normallist, (normalinds != null) ? normalinds.length : 0, normalinds
     );
  } // IndexedFaceSet

  // IndexedLineSet
  protected void tIndexedLineSet (IndexedLineSet lineset)
  {
    IndexedLineSetData dat = (IndexedLineSetData) lineset.userdata;
    MFVec3f coords = dat.coords_;
    if (coords == null)
      return;

    if (dat.color_ == null && currmat_ != null)
    {
      GE3D.lineColor3f (currmat_.emissiveColor.getValue ());
      currmat_ = null;
    }

    MFInt32 colorind = lineset.colorIndex;
    if (lineset.colorPerVertex.getValue () && colorind.getValueCount () == 0)
      colorind = lineset.coordIndex;

    GE3D.drawLineSet (coords.getValueData (), lineset.coordIndex.getValueCount (), lineset.coordIndex.getValueData (),
      (dat.color_ != null) ? dat.color_.getValueCount () : 0, (dat.color_ != null) ? dat.color_.getValueData () : null,
      colorind.getValueCount (), (colorind.getValueCount () > 0) ? colorind.getValueData () : null,
      lineset.colorPerVertex.getValue () ? 1 : 0
    );
  }

  // PointSet
  protected void tPointSet (PointSet pset)
  {
    PointSetData dat = (PointSetData) pset.userdata;
    MFVec3f coords = dat.coords_;
    if (coords == null)
      return;

    if (dat.color_ == null && currmat_ != null)
    {
      GE3D.lineColor3f (currmat_.emissiveColor.getValue ());
      currmat_ = null;
    }

    GE3D.drawPointSet (coords.getValueData (), coords.getValueCount (),
      (dat.color_ != null) ? dat.color_.getValueData () : null, (dat.color_ != null) ? dat.color_.getValueCount () : 0
    );
  }

  // Sphere
  protected void tSphere (Sphere sphere)
  {
    if (autobfc_)
      setBFC (true);
    setCCW (true);

    GE3D.drawSphere (sphere.radius.getValue ());
  }

  protected void tText (Text n)  { }


  // *** Geometric Properties: already associated to geometry
  protected void tColor (Color n)  { }
  protected void tCoordinate (Coordinate n)  { }
  protected void tNormal (Normal n)  { }
  protected void tTextureCoordinate (TextureCoordinate n)  { }

  // *** Appearance Nodes
  protected void tAppearance (Appearance app)
  {
    if (autolighting_)
      GE3D.hint (GE3D.HINT_LIGHTING, (app.material.getNode () != null) ? 1 : 0);

    if (!scene_.materials ())
      return;

    if (app.material.getNode () != null)
      Node.traverseNode (this, app.material.getNode ());
    else
    {
      GE3D.defaultMaterial ();
      GE3D.fillColor3f (white);
      currmat_ = null;
    }

    if (drawmode_ == GE3D.ge3d_texturing)  // texture, textureTransform
    {
      Node.traverseNode (this, app.texture.getNode ());
      Node.traverseNode (this, app.textureTransform.getNode ());
    }
  }

  protected void tFontStyle (FontStyle n)  { }

  // Material
  protected void tMaterial (Material mat)
  {
    if (mat == currmat_)
      return;
    GE3D.material (
      mat.ambientIntensity.getValue (), mat.diffuseColor.getValue (),
      mat.emissiveColor.getValue (), mat.shininess.getValue (),
      mat.specularColor.getValue (), mat.transparency.getValue ()
    );
    currmat_ = mat;
  }

  // ImageTexture (only traversed on texturing)
  protected void tImageTexture (ImageTexture texture)
  {
    ImageTextureData dat = (ImageTextureData) texture.userdata;

    if (dat.handle_ < -1)
    {
      if (urlserver_ == null)
        urlserver_ = new URLServer (scene_);

      dat.handle_ = -1;  // requested

      urlserver_.addRequest (texture, currbaseurl_);

      if (thread_ == null || !urlserver_.isActive ())
      {
        thread_ = new Thread (urlserver_);
        thread_.start ();
      }
    }

    if (dat.handle_ > 0)
    {
      setTexture (dat.handle_);
      setRepeat (texture.repeatS.getValue (), texture.repeatT.getValue ());
      GE3D.doTexturing (1);

      if (dat.hasalpha_ && scene_.getTextureTransparency ())
        GE3D.alphaTest (0.5f);

      if (dat.colored_)
      {
        GE3D.fillColor3f (white);
        currmat_ = null;
      }
    }
    else
      GE3D.doTexturing (0);
  } // ImageTexture

  protected void tMovieTexture (MovieTexture n)  { }

  // PixelTexture (only traversed on texturing)
  protected void tPixelTexture (PixelTexture texture)
  {
    PixelTextureData dat = (PixelTextureData) texture.userdata;

    if (dat.handle_ < 0)
    {
      int[] vals = texture.image.getValueData ();
      if (vals[0] == 0 || vals[1] == 0 || vals[2] == 0)
        return;

      int displaylist = GE3D.createPixelTexture (vals);
      dat.hasalpha_ = (GE3D.getTextureAlpha () > 0);

      if (displaylist < 0)
      {
        displaylist = -displaylist;
        dat.colored_ = true;
      }

      dat.handle_ = displaylist;
      if (displaylist > 0)
      {
        if (urlserver_ == null)
          urlserver_ = new URLServer (scene_);

        urlserver_.addDisplayList (displaylist);
      }
    }

    if (dat.handle_ > 0)
    {
      setTexture (dat.handle_);
      setRepeat (texture.repeatS.getValue (), texture.repeatT.getValue ());
      GE3D.doTexturing (1);

      if (dat.hasalpha_ && scene_.getTextureTransparency ())
        GE3D.alphaTest (0.5f);

      if (dat.colored_)
      {
        GE3D.fillColor3f (white);
        currmat_ = null;
      }
    }
    else
      GE3D.doTexturing (0);
  } // PixelTexture

  protected void tTextureTransform (TextureTransform trf)
  { // only traversed on texturing
    // System.out.println ("textureTransform");
    TextureTransformData dat = (TextureTransformData) trf.userdata;
    GE3D.loadTextureMatrix (dat.trfmat_);    
  }

  // *** Interpolator Nodes ***
  protected void tColorInterpolator (ColorInterpolator n)  { }
  protected void tCoordinateInterpolator (CoordinateInterpolator n)  { }
  protected void tNormalInterpolator (NormalInterpolator n)  { }
  protected void tOrientationInterpolator (OrientationInterpolator n)  { }
  protected void tPositionInterpolator (PositionInterpolator n)  { }
  protected void tScalarInterpolator (ScalarInterpolator n)  { }

  // *** Bindable Nodes ***
  protected void tBackground (Background n)  { }
  protected void tFog (Fog n)  { }
  protected void tNavigationInfo (NavigationInfo n)  { }
  protected void tViewpoint (Viewpoint n)  { }

  // *** instance of a PROTO node ***
  protected void tProtoInstance (ProtoInstance n)  { }


  // native code is contained in the library loaded in OGLCanvas.java

  public void clearThreads ()
  {
    texture_ = 0;

    if (thread_ != null)
    {
      thread_.stop ();
      thread_ = null;
    }
    if (urlserver_ != null)
    {
      urlserver_.clearAll ();  // free display lists
      urlserver_ = null;
    }
  }

} // Drawer
