/*
 *   Copyright (C) 2002,2003 by Jonathan Naylor G4KLX
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "JT6MDisplay.h"
#include "JT6MApp.h"
#include "JT6MDefs.h"

const int BORDER_SIZE = 5;

enum {
	List_Messages        = 150,
	Menu_Messages_Clear  = 250
};

BEGIN_EVENT_TABLE(CJT6MDisplay, wxPanel)
	EVT_MENU(Menu_Messages_Clear, CJT6MDisplay::onClearMessages)
	EVT_LIST_ITEM_SELECTED(List_Messages, CJT6MDisplay::onMessageSelected)
	EVT_LIST_ITEM_RIGHT_CLICK(List_Messages, CJT6MDisplay::onMessageClick)
END_EVENT_TABLE()

CJT6MDisplay::CJT6MDisplay(wxWindow* parent, int id) :
wxPanel(parent, id),
m_level(NULL),
m_audio(NULL),
m_messageListCtrl(NULL),
m_messageMenu(NULL),
m_listCount(0),
m_selected(-1),
m_messageList(NULL)
{
	wxASSERT(parent != NULL);

	m_messageMenu = createMessageMenu();

	m_messageList = new CJT6MMessageList(wxKEY_NONE);
	m_messageList->DeleteContents(true);

	wxBoxSizer* sizer = new wxBoxSizer(wxVERTICAL);

	wxBoxSizer* graphSizer = new wxBoxSizer(wxHORIZONTAL);

	m_level = new CJT6MLevelGraph(this, -1, wxDefaultPosition, wxSize(JT6M_GRAPH_WIDTH, JT6M_GRAPH_HEIGHT));
	graphSizer->Add(m_level, 0, wxRIGHT, BORDER_SIZE);

	m_audio = new CLevelGraph(this, -1, wxDefaultPosition, wxSize(JT6M_AUDIO_WIDTH, JT6M_GRAPH_HEIGHT));
	graphSizer->Add(m_audio);

	sizer->Add(graphSizer, 0, wxTOP | wxBOTTOM, BORDER_SIZE);

	m_messageListCtrl = new wxListCtrl(this, List_Messages, wxDefaultPosition, wxSize(JT6M_MESSAGES_WIDTH, JT6M_MESSAGES_HEIGHT));
	m_messageListCtrl->SetSingleStyle(wxLC_REPORT);
	m_messageListCtrl->InsertColumn(0, wxT("Id"));
	m_messageListCtrl->SetColumnWidth(0, 75);
	m_messageListCtrl->InsertColumn(1, wxT("Time"));
	m_messageListCtrl->SetColumnWidth(1, 40);
	m_messageListCtrl->InsertColumn(2, wxT("Length"));
	m_messageListCtrl->SetColumnWidth(2, 50);
	m_messageListCtrl->InsertColumn(3, wxT("Strength"));
	m_messageListCtrl->SetColumnWidth(3, 60);
	m_messageListCtrl->InsertColumn(4, wxT("DF"));
	m_messageListCtrl->SetColumnWidth(4, 30);
	m_messageListCtrl->InsertColumn(5, wxT("Text"));
	m_messageListCtrl->SetColumnWidth(5, 315);
	m_messageListCtrl->InsertColumn(6, wxT("Ave"));
	m_messageListCtrl->SetColumnWidth(6, 30);

	sizer->Add(m_messageListCtrl, 0, wxALL, 0);

	SetAutoLayout(true);
	sizer->Fit(this);
	sizer->SetSizeHints(this);
	SetSizer(sizer);
}

CJT6MDisplay::~CJT6MDisplay()
{
	delete m_messageList;
}

wxMenu* CJT6MDisplay::createMessageMenu()
{
	wxMenu* menu = new wxMenu();
	menu->Append(Menu_Messages_Clear, wxT("Clear"));

	return menu;
}

void CJT6MDisplay::onMessageSelected(const wxListEvent& event)
{
	m_selected = event.GetIndex();
}

void CJT6MDisplay::onMessageClick(const wxListEvent& event)
{
	wxASSERT(m_messageListCtrl != NULL);

	if (m_selected != -1) {
		wxPoint point = event.GetPoint();
		m_messageListCtrl->PopupMenu(m_messageMenu, point);
	}
}

void CJT6MDisplay::onClearMessages(const wxCommandEvent& event)
{
	wxASSERT(m_messageListCtrl != NULL);
	wxASSERT(m_messageList != NULL);

	m_messageListCtrl->DeleteAllItems();
	m_messageList->Clear();

	m_listCount = 0;
	m_selected  = -1;
}

void CJT6MDisplay::showMessage(CJT6MMessage* message)
{
	wxASSERT(m_messageListCtrl != NULL);
	wxASSERT(message != NULL);
	wxASSERT(m_listCount >= 0);

	wxString id = message->getId();

	int pos = id.Find(wxT('_'), true);
	if (pos != -1) {
		wxString tmp = id.Mid(pos + 1);
		id = tmp;
	}

	m_messageListCtrl->InsertItem(0, id, 0);

	wxString text = message->getText();
	int     count = message->getCount();

	wxString buffer;

	if (count == 0) {
		buffer.Printf(wxT("%.1f"), message->getTime());
		m_messageListCtrl->SetItem(0, 1, buffer);

		buffer.Printf(wxT("%d"), message->getLength());
		m_messageListCtrl->SetItem(0, 2, buffer);
	}

	buffer.Printf(wxT("%d"), message->getStrength());
	m_messageListCtrl->SetItem(0, 3, buffer);

	buffer.Printf(wxT("%d"), message->getDF());
	m_messageListCtrl->SetItem(0, 4, buffer);

	m_messageListCtrl->SetItem(0, 5, text);

	if (count > 0) {
		buffer.Printf(wxT("* %d"), count);
		m_messageListCtrl->SetItem(0, 6, buffer);
	}

	::wxGetApp().logMessage(*message);

	m_listCount++;

	pos = 0;
	m_messageList->Insert(pos, message);

	// The list is moving down and the selected message has moved down
	// by one
	if (m_selected != -1)
		m_selected++;
}

void CJT6MDisplay::showLevels(CJT6MLevels* levels) const
{
	wxASSERT(levels != NULL);
	wxASSERT(m_level != NULL);

	m_level->addData(levels);
}

void CJT6MDisplay::showAudio(double audio) const
{
	wxASSERT(m_audio != NULL);

	m_audio->addData(audio);
}
