/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2008 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "viewmanager.h"

#include <QApplication>

#include "views/view.h"
#include "views/issueview.h"
#include "views/folderview.h"
#include "views/viewfactory.h"
#include "viewerwindow.h"

using namespace WebIssues;

ViewManager* WebIssues::viewManager = NULL;

ViewManager::ViewManager()
{
}

ViewManager::~ViewManager()
{
    QList<ViewerWindow*> windows = m_views.values();

    for ( uint i = 0; i < windows.count(); i++ ) {
        ViewerWindow* window = windows.at( i );
        delete window;
    }
}

void ViewManager::openTypesView()
{
    openView( "WebIssues::TypesView", 0 );
}

void ViewManager::openUsersView()
{
    openView( "WebIssues::UsersView", 0 );
}

void ViewManager::openFolderView( int folderId )
{
    openView( "WebIssues::FolderView", folderId );
}

void ViewManager::openFolderView( int folderId, const QString& filter )
{
    FolderView* view = (FolderView*)openView( "WebIssues::FolderView", folderId );
    view->selectFilter( filter );
}

void ViewManager::openIssueView( int issueId, int itemId )
{
    IssueView* view = (IssueView*)openView( "WebIssues::IssueView", issueId );
    if ( itemId != 0 )
        view->gotoItem( itemId );
}

void ViewManager::openMembersView( int projectId )
{
    openView( "WebIssues::MembersView", projectId );
}

void ViewManager::openCommentView( int issueId )
{
    openView( "WebIssues::CommentView", issueId );
}

bool ViewManager::queryCloseViews()
{
    QList<View*> views = m_views.keys();

    for ( uint i = 0; i < views.count(); i++ ) {
        View* view = views.at( i );
        if ( !view->queryClose() )
            return false;
    }

    return true;
}

void ViewManager::postViewEvent( const char* className, ViewEvent::Action action, int id )
{
    QList<View*> views = m_views.keys();

    for ( uint i = 0; i < views.count(); i++ ) {
        View* view = views.at( i );
        if ( !className || view->inherits( className ) ) {
            ViewEvent* viewEvent = new ViewEvent( action, id );
            QApplication::postEvent( view, viewEvent );
        }
    }
}

void ViewManager::activateView( View* view )
{
    ViewerWindow* window = m_views.value( view, NULL );
    if ( !window )
        return;

    window->raise();
    window->activateWindow();
}

void ViewManager::closeView( View* view )
{
    ViewerWindow* window = m_views.value( view, NULL );
    if ( !window )
        return;

    window->deleteLater();
}

View* ViewManager::openView( const char* className, int id )
{
    View* view = findView( className, id );
    if ( view ) {
        activateView( view );
        return view;
    }
    return createView( className, id );
}

View* ViewManager::findView( const char* className, int id )
{
    QList<View*> views = m_views.keys();

    for ( uint i = 0; i < views.count(); i++ ) {
        View* view = views.at( i );
        if ( view->inherits( className ) ) {
            if ( id == 0 || view->id() == id )
                return view;
        }
    }
    return NULL;
}

View* ViewManager::createView( const char* className, int id )
{
    ViewerWindow* window = new ViewerWindow();
    View* view = ViewFactory::createView( className, window, window );

    m_views.insert( view, window );
    connect( view, SIGNAL( destroyed( QObject* ) ), this, SLOT( viewDestroyed( QObject* ) ) );

    view->setId( id );
    window->setView( view );

    view->initialUpdate();
    window->show();

    return view;
}

void ViewManager::addView( View* view )
{
    m_views.insert( view, NULL );
    connect( view, SIGNAL( destroyed( QObject* ) ), this, SLOT( viewDestroyed( QObject* ) ) );
}

void ViewManager::viewDestroyed( QObject* view )
{
    m_views.remove( (View*)view );
}
