// This may look like C code, but it is really -*- C++ -*-

//<copyright>
//
// Copyright (c) 1996
// Institute for Information Processing and Computer Supported New Media (IICM),
// Graz University of Technology, Austria.
//
// This file is part of VRweb.
//
// VRweb is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// VRweb is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with VRweb; see the file LICENCE. If not, write to the
// Free Software Foundation, Inc., 59 Temple Place - Suite 330,
// Boston, MA 02111-1307, USA.
//
// Note that the GNU General Public License does not permit incorporating
// the Software into proprietary or commercial programs. Such usage
// requires a separate license from IICM.
//
//</copyright>

//<file>
//
// Name:        polygon.h
//
// Purpose:     Interface to class Face
//
// Created:     8 Feb 1996  Georg Meszaros
//
// Modified:    11 Jun 1996 Georg Meszaros 
//
//
// Description:
//
// Declaration of class Face, which forms an indexed faceset together
// with the class FaceAttr.
// 
//</file>


//<class>
//
// Name:    Face
//
// Purpose: Builds an indexed faceset in combination with a class FaceAttr.
//
//
// Public Interface:
/*  
    - Face();
    - Face(int vertex_num,
           int* vert_index_list,
           int normal_num, 
           const vector3D* normal, 
           FaceAttr* faceattr
          );
      > links the vert_index_list into the class, no copy!
      > normal is beeing copied
      > if (normal == nil) then the normal vector is calculated
      > the plane equation is calculated
      > if (normal_num != vertex_num) then normal_num is set to 0
        else the vertex_index_list is used as normal-index-list as well
      > default status = VIS(ible)      

    - ~Face();
      > frees the vertex-index-list(by calling free) from the free store  

    - Face(const Face& face);
      > copy constructor --> creates a new vertex-index-list
    - Face& operator=(const Face& face);
      > assignment: free old instance, copy new into it 

    - void draw() const;
      > if (status != NOT_VISIBLE) forceDraw(...)

    - void draw(const point3D& position, BSPMode mode) const;
      > calculates backface-culling if (mode != BSP_PLAIN) and it is
        allowed by the attributes

    - void forceDraw() const;
      > renders the polygon with attributes set in the corresponding FaceAttr,
        material, lights

    - void print() const;
      > for test purpose, sends info to cerr

    - void printIndices() const;
      > sends the corresponding indices to cerr

    - void printVertices() const;
      > sends the corresponding vertices (looked up in FaceAttr) to cerr  

    - void printNormals() const;
      > sends the corresponding normals to cerr

    - const vector3D& normal() const;
      > get the surface normal vector of this polygon
        vertices counterclockwise is the visible side, pointing to 

    - void calcNormal();
      > calculates the normal vector out of the first 3 vertices

    - void calcPlane();
      > calculates the d value for the plane equation: ax + by + cz = d 
        (a,b,c) is the normal vector, (x,y,z) the first vertex

    - float d() const;
      > get the value of d in the plane equation
   
    - point3D* vertices() const;
      > get an array of the corresponding vertices from FaceAttr     

    - vector3D* normals() const;
      > get an array of the corresponding normals from FaceAttr

    - point3D& anyVertex() const;
      > get the first vertex  
 
    - unsigned long number() const;
      > get the current number of existing Face - instances       

    - unsigned long id() const;
      > get the unique identifier for this face (test purpose) 

    - FaceAttr* faceAttr() const;
      > get the corresponding instance of FaceAttr

    - int backfaceCulling() const;
      > get the backface-culling attribute of the FaceAttr
  
    - int* vertexIndexList() const;  
      > get the vertex index list, no copy    

    - int vertexNumber() const;
      > get the number of indices in this face, without -1 in the end

    - int normalNumber() const;
      > get the number of normals in this face, without -1 in the end

    - void calcArea();
      > calculates the area of this polygon    

    - float area() const;
      > get the area of this polygon   

    - float maxArea() const;
      > the area of the largest polygon 

    - FaceStatus status() const;
      > either VIS(ible), PARTLY_VISIBLE, or NOT_VISIBLE

    - void setStatus(FaceStatus status);
      > set the visibility status

    - void resetAttributes();
      > reset the global status information (always remembers the
        attributes of the last rendered polygon)
  
    - void reset();
      > resets the global counter of Face and FaceAttr  
*/ 


// Description:
//
// This class holds an index list for vertices and normal vectors
// together with links(pointers) to the corresponding FaceAttr.
// The FaceAttr determines the material, light, and backface-culling properties.
// The indexed face sets are built in a way that the vertex-index-list
// can alway be used as the normal-index-list if normal vectors are supported.
// Normal vectors are treated as supported if the normal_num is equal
// to the vertex_num.
// For all visibility calculations a flag in this class can be set to
// either VIS, PARTLY_VISIBLE or NOT_VISIBLE - therefore the decision
// if a polygon needs to be rendered is made within the Face-class.
// 
//
//</class>


#ifndef harmony_scene_polygon_h
#define harmony_scene_polygon_h
       
#include "faceattr.h"
#include "bsptree.h"
#include "svbsptree.h"

#include <ge3d/vectors.h>

class AnyLight;


enum FaceStatus { VIS, PARTLY_VISIBLE, NOT_VISIBLE };

class Face
{
  public:

    Face();
    Face(int vertex_num,
         int* vert_index_list,
         int normal_num, 
         const vector3D* normal, 
         FaceAttr* faceattr
         );
    ~Face();

    Face(const Face& face);
    Face& operator=(const Face& face);

    void draw() const;
    void draw(const point3D& position, BSPMode mode) const;
    void forceDraw() const;
    void print() const;
    void printIndices() const;
    void printVertices() const;
    void printNormals() const;

    const vector3D& normal() const;
    void calcNormal();
    void calcPlane();
    float d() const;

    //int inFront(const Plane& plane);
   
    point3D* vertices() const;
    vector3D* normals() const;
    point3D& anyVertex() const;

    unsigned long number() const;
     
    unsigned long id() const;
    FaceAttr* faceAttr() const;
    int backfaceCulling() const;
  
    int* vertexIndexList() const;  
    int vertexNumber() const;
    int normalNumber() const;

    void calcArea();
    float area() const;
    float maxArea() const;

    FaceStatus status() const;
    void setStatus(FaceStatus status);

    void resetAttributes();
    void reset();
    
  private:
    void free();
    void copy(const Face& face);

    static unsigned long face_number_;   // for instance counting
    static unsigned long counter_;       // for unique id
    unsigned long id_;                   // the unique id for this instance3                   

    // all index lists end with a value of -1 !!
    int* vert_index_list_;               // list of all vertex indices
    int vertex_num_;                     // length of vertex index list without -1 
    int normal_num_;                     // 0 or vertex_num_
    float area_;                         // the area of the polygon
    static float max_area_;              // area of the largest polygon

    vector3D normal_;                    // n=(a,b,c); ax+by+cz=d plane equ.
    float d_;                            // value of d
    FaceStatus status_;                  // either VIS(ible), PARTLY_VISIBLE, or NOT_VISIBLE

    FaceAttr* faceattr_;                 // face belongs to this faceset
 
}; // Face


inline float Face::d() const { return d_; }

inline unsigned long Face::number() const { return face_number_; }

inline float Face::area() const { return area_; }
inline float Face::maxArea() const { return max_area_; }
inline unsigned long Face::id() const { return id_; }
inline void Face::setStatus(FaceStatus status) { status_ = status; }
inline FaceStatus Face::status() const { return status_; }
inline FaceAttr* Face::faceAttr() const { return faceattr_; }

inline int* Face::vertexIndexList() const { return vert_index_list_; }
inline int Face::vertexNumber() const { return vertex_num_; }
inline int Face::normalNumber() const { return normal_num_; }

inline int Face::backfaceCulling() const { return faceattr_->backfaceCulling(); }
inline void Face::reset() { counter_ = 0; faceattr_->reset(); resetAttributes(); }

#endif
