/* -*- C++ -*-

  This file is part of ViPEC
  Copyright (C) 1991-2001 Johan Rossouw (jrossouw@alcatel.altech.co.za)

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU Library General Public License as
  published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU Library General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#ifndef MAINWINDOW_H
#define MAINWINDOW_H

#include <Types.h>
#include <FileBlock.h>
#include <Schematic.h>
#include <NewGraphWindow.h>
#include <GraphDefinition.h>
#include <SubstrateDefinition.h>
#include <SchematicSizeWindow.h>

#include <qmainwindow.h>
#include <qlist.h>
#include <qmap.h>
#include <qguardedptr.h> 

class QDomElement;
class GridDefinition;
class SmithDefinition;
class TableDefinition;
class HelpWindow;
class QTranslator;
class TunerWindow;
class NewGraphWindow;
class ResultDefinition;
class MicroStripCalcWindow;
class SubstrateWindow;

class MainWindow: public QMainWindow
{
  Q_OBJECT
public:
  static MainWindow* instance();
  virtual ~MainWindow();

  void setTranslator( QTranslator* );
  QTranslator* getTranslator();

  const QString& getFilename() const;
  void setActiveSchematic( const QString& name );
  void getSchematicNames( QStringList& );
  void changeSchematicSize( const QString& name );
  void setSchematicSize( const QString& name, Schematic::SchematicSize newSize );
  void setFileChanged();

  void changeVariable( const QString& name );
  void renameVariable( const QString& name );
  void deleteVariable( const QString& name );
  void updateVariable( const QString& name,
		       const QString& value );
  TReal getVariableValue( const QString& name );

  void renameSchematic( const QString& name );
  void deleteSchematic( const QString& name );
  Schematic* getSchematic( const QString& name );

  void changeSubstrate( const QString& name );  
  void updateSubstrate( const QString& name, 
			const SubstrateDefinition& definition );
  void renameSubstrate( const QString& name );
  void deleteSubstrate( const QString& name );
  SubstrateDefinition* getSubstrate( const QString& name );

  void deleteDataFile( const QString& name );
  QList<DataPoint>* getFileData( const QString& name );
  
  bool confirm( const QString& message );

  static long int getUniqueID();

  void resetCircuits();
  void calculateResponse();

  //Interface to graphical output
  GraphDefinition* findGraph( const QString& name );
 
  void addGraph( const QString& name, NewGraphWindow::GraphType );
  void renameGraph( const QString& name );
  void deleteGraph( const QString& name );

  GridDefinition* addGrid( const QString& name, const QString& title );
  GridDefinition* findGrid( const QString& name );

  SmithDefinition* addSmith( const QString& name, const QString& title );
  SmithDefinition* findSmith( const QString& name );

  TableDefinition* addTable( const QString& name, const QString& title );
  TableDefinition* findTable( const QString& name );
  
public slots:
  void quit();
  void loadSlot();
  void saveSlot();
  void saveAsSlot();
  void closeSlot();
  void printSlot();
  void helpSlot();
  void aboutSlot();
  void aboutQtSlot();

  void newSchematicSlot();
  void newVariableSlot();
  void newGraphSlot();
  void newSubstrateSlot();
  void newDataFileSlot();

  void graphFontChanged();  

  void microStripCalculator();
  void tuner();

protected:
  void closeEvent( QCloseEvent* e );

private:
  MainWindow();

  void createApplicationMenus();
  void createMainToolbar();
  void createSymbolToolbar();

  void writeFile();
  bool readFile();
  bool closeFile();

  bool readVariablesFromDOM( QDomElement& element );
  bool readDimensionsFromDOM( QDomElement& element );
  bool readSweepSettingsFromDOM( QDomElement& element );
  bool readGraphDefinitionsFromDOM( QDomElement& element );
  bool readSubstrateDefinitionsFromDOM( QDomElement& element );
  bool readFileBlocksFromDOM( QDomElement& element );
  
  bool checkSchematics();

  void removeAllGraphs();
  void showAllGraphs();
  void computeAllGraphOutputs();
  void renameGraph( const QString& oldName, const QString& newName );

  void readFileBlock( const QString& name );
  
private:
  static MainWindow* instance_;
  static long int uniqueID_;

  //Circuit file name
  QString filename_;
  bool fileChanged_;

  //Sub circuit map
  typedef QMap<QString, Schematic> SchematicMap;
  SchematicMap schematicMap_;
  
  //Variables map
  typedef QMap<QString, QString> VariableMap;
  VariableMap variableMap_;

  //Substrate map
  typedef QMap<QString, SubstrateDefinition> SubstrateMap;
  SubstrateMap substrateMap_;

  //File block map
  typedef QMap<QString, FileBlock> FileBlockMap;
  FileBlockMap fileBlockMap_;
  
  //Help browser
  QGuardedPtr<HelpWindow> helpWindow_;

  //windows
  QList<GraphDefinition> graphs_;
  QGuardedPtr<TunerWindow> tunerWindow_;
  QGuardedPtr<NewGraphWindow> newGraphWindow_;
  QGuardedPtr<MicroStripCalcWindow> microStripCalculator_;
  QGuardedPtr<SubstrateWindow> substrateWindow_;
  QGuardedPtr<SchematicSizeWindow> schematicSizeWindow_;

  //Translator object
  QTranslator* translator_;

};

#endif
