/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include "atomic_xyz.h"
#include "renderingAtomic_ascii.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <visu_rendering.h>
#include <visu_data.h>

/**
 * SECTION:atomic_xyz
 * @short_description: Method to load ascii position file.
 *
 * <para>XYZ formats are plain text format to store atomic
 * positions. This format is quite simple, first line must contain the
 * number of element, then
 * the second usually store a commentary (but this is not required),
 * then all lines that are not beginning with a '#' are tried to match
 * "label x y z":. If succeed a node is added, if not, the next line
 * is read as much time as specified on first line. This scheme can be
 * repeated as much time as required to store animation for instance.</para>
 */

static gboolean loadXyz(VisuData *data, const gchar* filename,
			FileFormat *format, int nSet, GError **error);
static int read_xyz_file(VisuData *data, GIOChannel *flux, int nSet, GError **error);

/******************************************************************************/

RenderingFormatLoad* atomicXyzInit()
{
  char *typeXYZ[] = {"*.xyz", (char*)0};
  char *descrXYZ = _("'Element x y z' format");
  RenderingFormatLoad *meth;
  
  meth = g_malloc(sizeof(RenderingFormatLoad));
  meth->name = "XYZ format (very loose).";
  meth->fmt = fileFormatNew(descrXYZ, typeXYZ);
  if (!meth->fmt)
    {
      g_error("Can't initialize the rendering atomic method, aborting...\n");
    }
  meth->priority = 100;
  meth->load = loadXyz;

  return meth;
}

static gboolean loadXyz(VisuData *data, const gchar* filename,
			FileFormat *format _U_, int nSet, GError **error)
{
  int res;
  GIOChannel *readFrom;
  GIOStatus status;

  g_return_val_if_fail(error && *error == (GError*)0, FALSE);
  g_return_val_if_fail(data && filename, FALSE);

  readFrom = g_io_channel_new_file(filename, "r", error);
  if (!readFrom)
    return FALSE;

  res = read_xyz_file(data, readFrom, nSet, error);
  status = g_io_channel_shutdown(readFrom, FALSE, (GError**)0);
  if (status != G_IO_STATUS_NORMAL)
    {
      DBG_fprintf(stderr, "Atomic XYZ: can't close file.\n");
      return FALSE;
    }

  if (res < 0)
    /* The file is not a D3 file. */
    return FALSE;
  else if (res > 0)
    /* The file is a D3 file but some errors occured. */
    return TRUE;
  /* Everything is OK. */
  *error = (GError*)0;
  return TRUE;
}

static int read_xyz_file(VisuData *data, GIOChannel *flux, int nSet, GError **error)
{
  GIOStatus status;
  GString *line;
  gsize term;
  double xu, yu, zu;
  int i;
  int res, nSets, nNodes, iNodes, nNodesSet;
  char nomloc[MAX_LINE_LENGTH];
  VisuElement *type;
  VisuElement **types, **nodeTypes;
  unsigned int *nattyp;
  int ntype;
  GHashTable *elements;
  struct dataAscii *infos;
  double xMin, yMin, zMin, xMax, yMax, zMax;
  float boxGeometry[6];
  float translation[3];
  gchar *infoUTF8;
  float *coords;
  GList *lst, *tmpLst;

  /* We read every line that corresponds to this schema : "%s %f %f %f" */
  DBG_fprintf(stderr, "Atomic xyz : reading file as an xyz file.\n");

  line = g_string_new("");

  /* Storage of number of elements per types. */
  ntype = 0;
  elements = g_hash_table_new_full(g_direct_hash, g_direct_equal, NULL, g_free);
  
  /* We read the file completely to find the number of sets of points
     and we store only the one corresponding to @nSet. */
  nSets     = 0;
  nNodesSet = 0;
  nodeTypes = (VisuElement**)0;
  coords    = (float*)0;
  lst       = (GList*)0;
  if (g_io_channel_read_line_string(flux, line, &term, error) != G_IO_STATUS_NORMAL)
    {
      g_string_free(line, TRUE);
      g_hash_table_destroy(elements);
      return -1;
    }
  do
    {
      DBG_fprintf(stderr, "Atomic xyz: read node set number %d.\n", nSets);
      /* The number of nodes. */
      if (sscanf(line->str, "%d", &nNodes) != 1)
	{
	  *error = g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_FORMAT,
			       _("Wrong XYZ format, no number on first line.\n"));
	  g_string_free(line, TRUE);
	  g_hash_table_destroy(elements);
	  tmpLst = lst;
	  while (tmpLst)
	    {
	      g_free(tmpLst->data);
	      tmpLst = g_list_next(tmpLst);
	    }
	  g_list_free(lst);
	  return -1;
	}
      DBG_fprintf(stderr, " | number of declared nodes is %d.\n", nNodes);
      /* The commentary line. */
      if (g_io_channel_read_line_string(flux, line, &term, error) != G_IO_STATUS_NORMAL)
	{
	  g_string_free(line, TRUE);
	  g_hash_table_destroy(elements);
	  tmpLst = lst;
	  while (tmpLst)
	    {
	      g_free(tmpLst->data);
	      tmpLst = g_list_next(tmpLst);
	    }
	  g_list_free(lst);
	  return -1;
	}
      g_strstrip(line->str);
      DBG_fprintf(stderr, " | set the commentary to '%s'.\n", line->str);
      if (line->str[0] == '#')
	infoUTF8 = g_locale_to_utf8(line->str + 1, -1, NULL, NULL, NULL);
      else
	infoUTF8 = g_locale_to_utf8(line->str, -1, NULL, NULL, NULL);
      if (infoUTF8)
	lst = g_list_append(lst, infoUTF8);
      else
	g_warning("Can't convert '%s' to UTF8.\n", line->str);
      /* The data lines. */
      if (nSets == nSet)
	{
	  nNodesSet = nNodes;
	  nodeTypes = g_malloc(sizeof(VisuElement*) * nNodes);
	  coords    = g_malloc(sizeof(float) * 3 * nNodes);
	}
      status = G_IO_STATUS_NORMAL;
      DBG_fprintf(stderr, " | read node coordinates.\n");
      for (iNodes = 0; iNodes < nNodes && status == G_IO_STATUS_NORMAL; iNodes++)
	{
	  /* Eat commentary lines */
	  do
	    {
	      status = g_io_channel_read_line_string(flux, line, &term, error);
	      if (status != G_IO_STATUS_NORMAL)
		{
		  if (status == G_IO_STATUS_EOF)
		    *error = g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_FORMAT,
					 _("Missing coordinates (%d read but"
					   " %d declared).\n"), iNodes, nNodes);
		    
		  DBG_fprintf(stderr, "Atomic XYZ: can't read a line (%s).\n",
			      (*error)->message);
		  g_string_free(line, TRUE);
		  g_hash_table_destroy(elements);
		  if (nSets == nSet)
		    {
		      g_free(nodeTypes);
		      g_free(coords);
		      tmpLst = lst;
		      while (tmpLst)
			{
			  g_free(tmpLst->data);
			  tmpLst = g_list_next(tmpLst);
			}
		      g_list_free(lst);
		    }
		  return 1;
		}
	    }
	  while (line->str[0] == '#');
	  /* In case of reading set, parse the values. */
	  if (nSets == nSet)
	    {
	      if (sscanf(line->str, "%s %lf %lf %lf", nomloc, &xu, &yu, &zu) != 4)
		{
		  DBG_fprintf(stderr, "Atomic XYZ: can't read line values.\n");
		  *error = g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_FORMAT,
				       _("Wrong XYZ format, 'Atom X Y Z' awaited.\n"));
		  g_string_free(line, TRUE);
		  g_hash_table_destroy(elements);
		  if (nSets == nSet)
		    {
		      g_free(nodeTypes);
		      g_free(coords);
		      tmpLst = lst;
		      while (tmpLst)
			{
			  g_free(tmpLst->data);
			  tmpLst = g_list_next(tmpLst);
			}
		      g_list_free(lst);
		    }
		  return 1;
		}
	      /* Store the coordinates. */
	      coords[iNodes * 3 + 0] = xu;
	      coords[iNodes * 3 + 1] = yu;
	      coords[iNodes * 3 + 2] = zu;
	      /* Store the elements. */
	      nomloc[8] = '\0';
	      /* adding nomloc to the hashtable */
	      type = visuElementGet_fromName(nomloc);
	      if (!type)
		{
		  type = visuElementNew_withName(nomloc);
		  if (!type)
		    {
		      g_warning("Cannot create a new type for '%s'.", nomloc);
		      g_string_free(line, TRUE);
		      g_hash_table_destroy(elements);
		      if (nSets == nSet)
			{
			  g_free(nodeTypes);
			  g_free(coords);
			  tmpLst = lst;
			  while (tmpLst)
			    {
			      g_free(tmpLst->data);
			      tmpLst = g_list_next(tmpLst);
			    }
			  g_list_free(lst);
			}
		      return 1;
		    }
		  res = visuElementAdd(type);
		  if (res)
		    {
		      g_warning("Cannot add new element, maximum has been reached.");
		      g_string_free(line, TRUE);
		      g_hash_table_destroy(elements);
		      if (nSets == nSet)
			{
			  g_free(nodeTypes);
			  g_free(coords);
			  tmpLst = lst;
			  while (tmpLst)
			    {
			      g_free(tmpLst->data);
			      tmpLst = g_list_next(tmpLst);
			    }
			  g_list_free(lst);
			}
		      return 1;
		    }
		}
	      nodeTypes[iNodes] = type;
	      infos = (struct dataAscii*)g_hash_table_lookup(elements,
							     (gconstpointer)type);
	      if (!infos)
		{
		  infos = g_malloc(sizeof(struct dataAscii));
		  infos->ele = type;
		  infos->pos = ntype;
		  infos->nbNodes = 1;
		  g_hash_table_insert(elements, (gpointer)type, (gpointer)infos);
		  ntype++;
		}
	      else
		infos->nbNodes += 1;
	    }
	}
      DBG_fprintf(stderr, " | read OK.\n");
      /* OK, one set of nodes have been read. */
      nSets += 1;
      /* Eat blank or commentary lines. */
      do
	{
	  status = g_io_channel_read_line_string(flux, line, &term, error);
	  if (status != G_IO_STATUS_NORMAL && status != G_IO_STATUS_EOF)
	    {
	      g_string_free(line, TRUE);
	      g_hash_table_destroy(elements);
	      if (nSets == nSet)
		{
		  g_free(nodeTypes);
		  g_free(coords);
		  tmpLst = lst;
		  while (tmpLst)
		    {
		      g_free(tmpLst->data);
		      tmpLst = g_list_next(tmpLst);
		    }
		  g_list_free(lst);
		}
	      return 1;
	    }
	  g_strstrip(line->str);
	}
      while (status != G_IO_STATUS_EOF &&
	     (line->str[0] == '#' || line->str[0] == '!' || line->str[0] == '\0'));
    }
  while(status != G_IO_STATUS_EOF);
  g_string_free(line, TRUE);

  /* Allocate the space for the nodes. */
  if (ntype <= 0)
    {
      g_hash_table_destroy(elements);
      if (nodeTypes)
	g_free(nodeTypes);
      if (coords)
	g_free(coords);
      tmpLst = lst;
      while (tmpLst)
	{
	  g_free(tmpLst->data);
	  tmpLst = g_list_next(tmpLst);
	}
      g_list_free(lst);
      *error = g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_FORMAT,
			   _("The file contains no atom coordinates.\n"));
      return -1;
    }
  types  = g_malloc(sizeof(VisuElement*) * ntype);
  nattyp = g_malloc(sizeof(int) * ntype);
  g_hash_table_foreach(elements, (GHFunc)putValTypeInAsciiType, (gpointer)&types);
  g_hash_table_foreach(elements, (GHFunc)putValNbInAsciiType, (gpointer)&nattyp);
  g_hash_table_destroy(elements);

  DBG_fprintf(stderr, " | begin to transfer data to VisuData.\n");
  /* Begin the storage into VisuData. */
  visuDataSet_nSet(data, nSets);

  res = visuDataSet_population(data, ntype, nattyp, types);
  if (!res)
    {
      g_error("Can't store the nodes in the VisuData object.");
    }
  DBG_fprintf(stderr, "Atomic xyz : there are %d types in this file.\n", ntype);
  if (DEBUG)
    {
      for (i = 0; i < ntype; i++)
	DBG_fprintf(stderr, " | %d atom(s) for type %d.\n", nattyp[i], i);
    }
  g_free(nattyp);
  g_free(types);
         
  /* Store the coordinates */
  xMin = 1e5;
  yMin = 1e5;
  zMin = 1e5;
  xMax = -1e5;
  yMax = -1e5;
  zMax = -1e5;
  
  for(iNodes = 0; iNodes < nNodesSet; iNodes++)
    {
      visuDataAdd_nodeFromElement(data, nodeTypes[iNodes], coords + 3 * iNodes);
      xMin = MIN(xMin, coords[iNodes * 3 + 0]);
      yMin = MIN(yMin, coords[iNodes * 3 + 1]);
      zMin = MIN(zMin, coords[iNodes * 3 + 2]);
      xMax = MAX(xMax, coords[iNodes * 3 + 0]);
      yMax = MAX(yMax, coords[iNodes * 3 + 1]);
      zMax = MAX(zMax, coords[iNodes * 3 + 2]);
    }
  g_free(nodeTypes);
  g_free(coords);

  /* Set the commentary. */
  tmpLst = lst;
  for (i = 0; i < nSets; i++)
    {
      visuDataSet_fileCommentary(data, (gchar*)tmpLst->data, i);
      g_free(tmpLst->data);
      tmpLst = g_list_next(tmpLst);
    }
  g_list_free(lst);

  DBG_fprintf(stderr, "Atomic xyz : the elements are in [%f, %f]x[%f, %f]x[%f, %f].\n",
	      xMin, xMax, yMin, yMax, zMin, zMax);
  boxGeometry[0] = xMax - xMin;
  boxGeometry[1] = 0.;
  boxGeometry[2] = yMax - yMin;
  boxGeometry[3] = 0.;
  boxGeometry[4] = 0.;
  boxGeometry[5] = zMax - zMin;
  visuDataSet_boxGeometry(data, boxGeometry, FALSE);

  translation[0] = -xMin;
  translation[1] = -yMin;
  translation[2] = -zMin;
  visuDataSet_XYZtranslation(data, translation);

  return 0;
}

