/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include "dumpToSVG.h"

#ifdef HAVE_CAIRO

#include <math.h>
#include <string.h>
#include <glib.h>
#include <cairo.h>
#include <cairo-svg.h>
#include <GL/gl.h>

#if CAIRO_VERSION_MINOR > 1

#include <visu_tools.h>
#include <visu_rendering.h>
#include <visu_pairs.h>
#include <opengl.h>
#include <openGLFunctions/light.h>
#include <openGLFunctions/view.h>
#include <extensions/box.h>
#include <extensions/axes.h>

/**
 * SECTION:dumpToSVG
 * @short_description: add an export capability into SVG files.
 * @include: extensions/box.h, extensions/axes.h, visu_pairs.h and visu_data.h
 *
 * This provides a write routine to export V_Sim views into SVG
 * files. Currently, this is an experimental feature. Not all V_Sim
 * elements are rendered, only the nodes, the box, the pairs and the
 * axes. All the characteristics are not used (no line stipple for
 * instance). In spin mode, nodes are only atomic.
 *
 * Since: 3.4
 */

struct _pairs
{
  VisuNode *node1, *node2;
  VisuElement *ele1, *ele2;
  float alpha;
};

static gboolean writeViewInSvgFormat(FileFormat *format, const char* filename,
				     int width, int height, VisuData *dataObj,
				     guchar* imageData, GError **error,
				     voidDataFunc functionWait, gpointer data);

DumpType* dumpToSVG_init()
{
  DumpType *svg;
  char *typeSVG[] = {"*.svg", (char*)0};
#define descrSVG _("Scalar Vector Graphic (SVG) file")
  FileFormat* fmt;

  svg = g_malloc(sizeof(DumpType));
  fmt = fileFormatNew(descrSVG, typeSVG);
  if (!fmt)
    {
      g_error("Can't initialize the SVG dump module, aborting.\n");
    }

  svg->bitmap    = FALSE;
  svg->hasAlpha  = FALSE;
  svg->fileType  = fmt;
  svg->writeFunc = writeViewInSvgFormat;
  
  return svg;
}

static void sort_by_z(float *coordinates, float *buffer,
		      int n, int z, int begin, int end)
{
  int i;
  int middle;

  if( begin >= end ) return;
  
  memcpy(buffer, coordinates + begin * n, sizeof(float) * n);
  memcpy(coordinates + begin * n, coordinates + (end + begin) / 2 * n, sizeof(float) * n);
  memcpy(coordinates + (end + begin) / 2 * n, buffer, sizeof(float) * n);

  middle = begin;
  for(i = begin + 1; i <= end; i++)
    {
      if ( coordinates[i * n + z] > coordinates[begin * n + z] )
	{
	  middle += 1;
	  memcpy(buffer, coordinates + i * n, sizeof(float) * n);
	  memcpy(coordinates + i * n, coordinates + middle * n, sizeof(float) * n);
	  memcpy(coordinates + middle * n, buffer, sizeof(float) * n);
	}
    }
  memcpy(buffer, coordinates + begin * n, sizeof(float) * n);
  memcpy(coordinates + begin * n, coordinates + middle * n, sizeof(float) * n);
  memcpy(coordinates + middle * n, buffer, sizeof(float) * n);
  sort_by_z(coordinates, buffer, n, z, begin, middle - 1);
  sort_by_z(coordinates, buffer, n, z, middle + 1, end);
}

gboolean writeDataToCairoSurface(cairo_surface_t *cairo_surf, VisuData *dataObj,
				 GError **error, voidDataFunc functionWait _U_,
				 gpointer user_data _U_)
{
  cairo_t *cr;
  cairo_status_t status;
  cairo_pattern_t *pat;
#define FONT_SIZE 16.
  cairo_matrix_t scale = {1., 0., 0., -1., 0., 0.};
  cairo_matrix_t scaleFont = {FONT_SIZE, 0., 0., -FONT_SIZE, 0., 0.};
  VisuDataIter iter;
  int i, nNodes;
  float xyz[3], radius, norm, *rgb;
  GLfloat *coordinates, *box, *pairs, *axes;
  GLint nValuesBox, nValues, nValuesAxes;
  OpenGLView *view;
  float modelView[16];
  int viewport[4];
  RenderingMethod *method;
  GList *lights;
  Light *light;
#define NVERT 4
#define NVALS 8
#define NPASS 1
#define NBUFF (NVERT * NVALS + NPASS * 2)
  float tmpFloat[NBUFF];
  VisuDataIter iter1, iter2;
  float d2, d2min, d2max, d2min_buffered, d2max_buffered, l;
  VisuPairData *data;
  float xyz1[3], xyz2[3], u[3], alpha;
  GList *pairsLst, *tmpLst;
  struct _pairs *pairData;
  int nPairs, iPairs;
  float tmpPairs[6];
  
  cr = cairo_create(cairo_surf);
  status = cairo_status(cr);
  if (status != CAIRO_STATUS_SUCCESS)
    {
      *error = g_error_new(VISU_ERROR_DUMP, DUMP_ERROR_FILE,
			   cairo_status_to_string(status));
      cairo_destroy(cr);
      return FALSE;
    }
  cairo_set_line_cap(cr, CAIRO_LINE_CAP_ROUND);

  visuDataIter_new(dataObj, &iter);
  view = visuDataGet_openGLView(dataObj);
  method = getRenderingMethodInUse();
  lights = lightEnvironnementGet_lightList(currentLights);
  g_return_val_if_fail(lights, FALSE);
  light = (Light*)lights->data;

  /* We create a feedback mode to get node coordinates. */
  coordinates = g_malloc(sizeof(GLfloat) * (iter.nAllStoredNodes * NBUFF));
  glFeedbackBuffer((iter.nAllStoredNodes * NBUFF), GL_3D_COLOR, coordinates);
  glRenderMode(GL_FEEDBACK);
  glGetFloatv(GL_MODELVIEW_MATRIX, modelView);
  glGetIntegerv(GL_VIEWPORT, viewport);
  scale.y0 = (double)viewport[3];
  cairo_set_matrix(cr, &scale);

  glPushMatrix();
  glTranslated(-view->box->dxxs2, -view->box->dyys2, -view->box->dzzs2);

  /* First thing is to order nodes along z axes. */
  for (visuDataIter_startVisible(dataObj, &iter); iter.node;
       visuDataIter_nextVisible(dataObj, &iter))
    {
      glBegin(GL_POINTS);
      visuDataGet_nodePosition(dataObj, iter.node, xyz);
      /* We compute the node position in the eyes coordinates. */
      glVertex3fv(xyz);
      /* We compute the node apparent radius using the real radius in
	 X direction in ModelView. */
      radius = renderingMethodGet_sizeOfElement(method, iter.element);
      glVertex3f(xyz[0] + modelView[0] * radius,
		 xyz[1] + modelView[4] * radius,
		 xyz[2] + modelView[8] * radius);
      /* We get the color of the specular point on sphere and of the
	 opposite point. */
      glCallList(visuElementGet_identifierMaterial(iter.element));
      norm = light->position[0] * light->position[0] +
	light->position[1] * light->position[1] +
	light->position[2] * light->position[2];
      norm = sqrt(norm);
      glNormal3f(light->position[0],
		 light->position[1],
		 light->position[2]);
      glVertex3f(xyz[0] + (light->position[0]) * radius / norm,
		 xyz[1] + (light->position[1]) * radius / norm,
		 xyz[2] + (light->position[2]) * radius / norm);
      glNormal3f(-light->position[0],
		 -light->position[1],
		 -light->position[2]);
      glVertex3f(xyz[0] - (light->position[0]) * radius / norm,
		 xyz[1] - (light->position[1]) * radius / norm,
		 xyz[2] - (light->position[2]) * radius / norm);
      glEnd();

      /* We compute z coordinates in eyes basis set. */
      glPassThrough(modelView[2] * xyz[0] + 
		    xyz[1] * modelView[6] + xyz[2] * modelView[10]);
    }
  glPopMatrix();

  /* We sort the coordinates along z. */
  nValues = glRenderMode(GL_RENDER);
  /* We compact coordinates to keep complete vertex list. */
  i = 0;
  nNodes = 0;
  while (i < nValues)
    {
      if (coordinates[i] == GL_POINT_TOKEN &&
	  coordinates[i + NVALS] == GL_POINT_TOKEN &&
	  coordinates[i + 2 * NVALS] == GL_POINT_TOKEN &&
	  coordinates[i + 3 * NVALS] == GL_POINT_TOKEN)
	{
/* 	  fprintf(stderr, "Found a complete node %d, at %gx%g.\n", */
/* 		  i, coordinates[i + 1], coordinates[i + 2]); */
/* 	  fprintf(stderr, " | move it from %d to %d.\n", i, nNodes); */
	  /* A complete set, copying it to nNodes location. */
	  if (nNodes != i)
	    memcpy(coordinates + nNodes, coordinates + i, sizeof(GLfloat) * NBUFF);
	  i += NBUFF;
	  nNodes += NBUFF;
	}
      else
	{
/* 	  fprintf(stderr, "Found a uncomplete node at %d.\n", i); */
	  /* Incomplete set, go on till the GL_PASS_THROUGH_TOKEN. */
	  while (coordinates[i] != GL_PASS_THROUGH_TOKEN)
	    i += 1;
	  /* Remove the GL_POINT_TOKEN. */
	  i += 2;
/* 	  fprintf(stderr, " | jump to %d.\n", i); */
	}
    }
  sort_by_z(coordinates, tmpFloat, NBUFF, 3, 0, nNodes / NBUFF - 1);

  /* Draw the pairs. */
  nPairs = -1;
  pairs = (GLfloat*)0;
  if (visuPairGet_status())
    {
      pairsLst = (GList*)0;

      visuDataIter_new(dataObj, &iter1);
      visuDataIter_new(dataObj, &iter2);
      for(visuDataIter_start(dataObj, &iter1); iter1.element;
	  visuDataIter_nextElement(dataObj, &iter1))
	{
	  if (!visuElementGet_rendered(iter1.element))
	    continue;

	  for(visuDataIter_start(dataObj, &iter2);
	      iter2.element && iter2.iElement <= iter1.iElement ;
	      visuDataIter_nextElement(dataObj, &iter2))
	    {
	      if (!visuElementGet_rendered(iter2.element))
		continue;
	  
	      for (tmpLst = visuPairGet_links(iter1.element, iter2.element);
		   tmpLst; tmpLst = g_list_next(tmpLst))
		{
		  data = (VisuPairData*)tmpLst->data;
		  if (!data->drawn)
		    continue;
		  d2min = data->minMax[PAIRS_MIN] * data->minMax[PAIRS_MIN];
		  d2max = data->minMax[PAIRS_MAX] * data->minMax[PAIRS_MAX];
		  if(d2min >= d2max || d2max <= 0.)
		    continue;

		  l = data->minMax[PAIRS_MAX] - data->minMax[PAIRS_MIN];
		  d2min_buffered = (data->minMax[PAIRS_MIN] - 0.15 * l);
		  d2min_buffered *= d2min_buffered;
		  d2max_buffered = (data->minMax[PAIRS_MAX] + 0.15 * l);
		  d2max_buffered *= d2max_buffered;

		  for(visuDataIter_restartNode(dataObj, &iter1); iter1.node;
		      visuDataIter_nextNode(dataObj, &iter1))
		    {
		      if (!iter1.node->rendered)
			continue;

		      for(visuDataIter_restartNode(dataObj, &iter2); iter2.node;
			  visuDataIter_nextNode(dataObj, &iter2))
			{
			  if (!iter2.node->rendered)
			    continue;
			  /* Don't draw the inter element pairs two times. */
			  if (iter1.element == iter2.element &&
			      iter2.node >= iter1.node)
			    break;

			  visuDataGet_nodePosition(dataObj, iter1.node, xyz1);
			  visuDataGet_nodePosition(dataObj, iter2.node, xyz2);
			  d2 = (xyz1[0] - xyz2[0]) * (xyz1[0] - xyz2[0]) + 
			    (xyz1[1] - xyz2[1]) * (xyz1[1] - xyz2[1]) + 
			    (xyz1[2] - xyz2[2]) * (xyz1[2] - xyz2[2]);
			  if(d2 <= 0. || d2 < d2min_buffered || d2 > d2max_buffered)
			    continue;

			  if (d2 < d2min)
			    alpha = (d2 - d2min_buffered) /
			      (d2min - d2min_buffered);
			  else if (d2 > d2max)
			    alpha = (d2max_buffered - d2) /
			      (d2max_buffered - d2max);
			  else
			    alpha = 1.f;

#if GLIB_MINOR_VERSION > 9
			  pairData = g_slice_alloc(sizeof(struct _pairs));
#else
			  pairData = g_malloc(sizeof(struct _pairs));
#endif
			  pairData->node1 = iter1.node;
			  pairData->node2 = iter2.node;
			  pairData->ele1  = iter1.element;
			  pairData->ele2  = iter2.element;
			  pairData->alpha = alpha;
			  pairsLst = g_list_prepend(pairsLst, pairData);
			}
		    }
		}
	    }
	}
      nPairs = g_list_length(pairsLst);
      DBG_fprintf(stderr, "Dump SVG: found %d pairs to draw.\n", nPairs);

      if (nPairs > 0)
	{
	  pairs = g_malloc(sizeof(GLfloat) * nPairs * 10);
	  glFeedbackBuffer(nPairs * 10, GL_3D, pairs);
	  glRenderMode(GL_FEEDBACK);
	  glPushMatrix();
	  glTranslated(-view->box->dxxs2, -view->box->dyys2, -view->box->dzzs2);

	  /* Render the list of pairs and free them each time. */
	  tmpLst = pairsLst;
	  while (tmpLst)
	    {
	      pairData = (struct _pairs*)tmpLst->data;

	      visuDataGet_nodePosition(dataObj, pairData->node1, xyz1);
	      visuDataGet_nodePosition(dataObj, pairData->node2, xyz2);
	      u[0] = xyz2[0] - xyz1[0];
	      u[1] = xyz2[1] - xyz1[1];
	      u[2] = xyz2[2] - xyz1[2];
	      norm = sqrt(u[0] * u[0] + u[1] * u[1] + u[2] * u[2]);
	      u[0] /= norm;
	      u[1] /= norm;
	      u[2] /= norm;
	      radius = renderingMethodGet_sizeOfElement(method, pairData->ele1);
	      xyz1[0] += radius * u[0];
	      xyz1[1] += radius * u[1];
	      xyz1[2] += radius * u[2];
	      radius = renderingMethodGet_sizeOfElement(method, pairData->ele2);
	      xyz2[0] -= radius * u[0];
	      xyz2[1] -= radius * u[1];
	      xyz2[2] -= radius * u[2];
	      glBegin(GL_POINTS);
	      glVertex3fv(xyz1);
	      glVertex3fv(xyz2);
	      glEnd();
	      /* We save alpha channel as passthrough. */
	      glPassThrough(pairData->alpha);
	  
	      /* Free the data. */
#if GLIB_MINOR_VERSION > 9
	      g_slice_free1(sizeof(struct _pairs), tmpLst->data);
#else
	      g_free(tmpLst->data);
#endif
	      tmpLst = g_list_next(tmpLst);
	    }
	  /* Free the list itself. */
	  if (pairsLst)
	    g_list_free(pairsLst);
	  /* Analyse the OpenGL results. */
	  nValues = glRenderMode(GL_RENDER);
	  DBG_fprintf(stderr, " | OpenGL returns %d.\n", nValues);
	  i = 0;
	  nPairs = 0;
	  while (i < nValues)
	    {
	  
	      if (pairs[i] == GL_POINT_TOKEN && pairs[i + 4] == GL_POINT_TOKEN &&
		  pairs[i + 8] == GL_PASS_THROUGH_TOKEN)
		{
/* 	      DBG_fprintf(stderr, " | pair %d at (%f;%f;%f) - (%f;%f;%f) %g\n", */
/* 			  nPairs, */
/* 			  pairs[i + 1], pairs[i + 2], pairs[i + 3], */
/* 			  pairs[i + 5], pairs[i + 6], pairs[i + 7], pairs[i + 9]); */
		  /* Copy all these values into the beginning of the pairs
		     array. */
		  norm = (pairs[i + 3] + pairs[i + 7]) / 2.f;
		  pairs[nPairs * 6 + 0] = pairs[i + 1];
		  pairs[nPairs * 6 + 1] = pairs[i + 2];
		  pairs[nPairs * 6 + 2] = pairs[i + 5];
		  pairs[nPairs * 6 + 3] = pairs[i + 6];
		  pairs[nPairs * 6 + 4] = pairs[i + 9];
		  pairs[nPairs * 6 + 5] = norm;
		  i += 10;
		  nPairs += 1;
		}
	      else if (pairs[i] == GL_POINT_TOKEN &&
		       pairs[i + 4] == GL_PASS_THROUGH_TOKEN)
		{
		  DBG_fprintf(stderr, "| uncomplete pair for i=%d\n", i);
		  i += 6;
		}
	      else if (pairs[i] == GL_PASS_THROUGH_TOKEN)
		{
		  DBG_fprintf(stderr, "| no pair for i=%d\n", i);
		  i += 2;
		}
	    }
	  glPopMatrix();
	  sort_by_z(pairs, tmpPairs, 6, 5, 0, nPairs - 1);
	}
    }

  /* Draw the back box. */
  nValuesBox = -1;
  box = (GLfloat*)0;
  if (boxGet_isOn())
    {
      box = g_malloc(sizeof(GLfloat) * 7000);
      glFeedbackBuffer(7000, GL_3D, box);
      glRenderMode(GL_FEEDBACK);
      OpenGLExtensionCall_list(EXT_BOX_ID);
      nValuesBox = glRenderMode(GL_RENDER);
      cairo_set_line_width(cr, (double)boxGet_lineWidth());
      rgb = boxGet_RGBValues();
      cairo_set_source_rgb(cr, rgb[0], rgb[1], rgb[2]);
      /* We draw the lines that have a boundary hidden by elements. */
      for (i = 0; i < nValuesBox; i += 7)
	if (box[i + 3] >= coordinates[nNodes - NBUFF + 3] &&
	    box[i + 6] >= coordinates[nNodes - NBUFF + 3])
	  {
	    cairo_move_to(cr, box[i + 1], box[i + 2]);
	    cairo_line_to(cr, box[i + 4], box[i + 5]);
	    cairo_stroke(cr);
	  }
    }

  cairo_set_line_width(cr, 1.);
  iPairs = 0;
  for (i = 0; i < nNodes; i+= NBUFF)
    {
      if (nPairs > 0 && iPairs < 6 * nPairs)
	{
	  /* We draw the pairs in between. */
	  cairo_set_line_width(cr, (double)boxGet_lineWidth());
	  while (pairs[iPairs + 5] > coordinates[i + 3] && iPairs < 6 * nPairs)
	    {
	      DBG_fprintf(stderr, " | pair %d at (%f;%f) - (%f;%f) %g,%g\n",
			  iPairs / 6,
			  pairs[iPairs + 0], pairs[iPairs + 1],
			  pairs[iPairs + 2], pairs[iPairs + 3],
			  pairs[iPairs + 4], pairs[iPairs + 5]);
	      cairo_set_source_rgba(cr, 0, 0, 0, pairs[iPairs + 4]);
	      cairo_move_to(cr, pairs[iPairs + 0], pairs[iPairs + 1]);
	      cairo_line_to(cr, pairs[iPairs + 2], pairs[iPairs + 3]);
	      cairo_stroke(cr);
	      iPairs += 6;
	    }
	}

      radius = (coordinates[i + NVALS + 1] - coordinates[i + 1]) *
	(coordinates[i + NVALS + 1] - coordinates[i + 1]) +
	(coordinates[i + NVALS + 2] - coordinates[i + 2]) *
	(coordinates[i + NVALS + 2] - coordinates[i + 2]);
      /*       fprintf(stderr, "%gx%g %g %g %g\n", coordinates[i + 1], coordinates[i + 2], */
      /* 	      sqrt(radius), coordinates[i + 3], coordinates[i + NBUFF - 1]); */
      radius = sqrt(radius);
      cairo_arc(cr, coordinates[i + 1], coordinates[i + 2],
		radius, 0., 2 * G_PI);
      /*       fprintf(stderr, "%gx%g %g, %gx%g %g\n", coordinates[i + 2 * NVALS + 1], */
      /* 					coordinates[i + 2 * NVALS + 2], */
      /* 					radius * 0.2f, */
      /* 					coordinates[i + 1], */
      /* 					coordinates[i + 2], */
      /* 					radius); */
      pat = cairo_pattern_create_radial(coordinates[i + 1] + 0.5f * radius,
					coordinates[i + 2] + 0.5f * radius,
					radius * 0.2f,
					coordinates[i + 1],
					coordinates[i + 2],
					radius);
      /*       fprintf(stderr,"%g %g %g\n", coordinates[i + 2 * NVALS + 4], */
      /* 	      coordinates[i + 2 * NVALS + 5], */
      /* 	      coordinates[i + 2 * NVALS + 6]); */
      cairo_pattern_add_color_stop_rgba(pat, 0,
					coordinates[i + 2 * NVALS + 4],
					coordinates[i + 2 * NVALS + 5],
					coordinates[i + 2 * NVALS + 6],
					coordinates[i + 2 * NVALS + 7]);
      /*       fprintf(stderr,"%g %g %g\n", coordinates[i + 3 * NVALS + 4], */
      /* 	      coordinates[i + 3 * NVALS + 5], */
      /* 	      coordinates[i + 3 * NVALS + 6]); */
      cairo_pattern_add_color_stop_rgba(pat, 1,
					coordinates[i + 3 * NVALS + 4],
					coordinates[i + 3 * NVALS + 5],
					coordinates[i + 3 * NVALS + 6],
					coordinates[i + 3 * NVALS + 7]);
      cairo_set_source(cr, pat);
      cairo_fill_preserve(cr);
      cairo_pattern_destroy(pat);
      cairo_set_source_rgb(cr, 0, 0, 0);
      cairo_stroke(cr);
    }
  if (nPairs > 0)
    g_free(pairs);

  /* Draw the front box. */
  if (boxGet_isOn())
    {
      cairo_set_line_width(cr, (double)boxGet_lineWidth());
      rgb = boxGet_RGBValues();
      cairo_set_source_rgb(cr, rgb[0], rgb[1], rgb[2]);
      /* We draw the lines that have a boundary hidden by elements. */
      for (i = 0; i < nValuesBox; i += 7)
	if (box[i + 3] < coordinates[nNodes - NBUFF + 3] ||
	    box[i + 6] < coordinates[nNodes - NBUFF + 3])
	  {
	    cairo_move_to(cr, box[i + 1], box[i + 2]);
	    cairo_line_to(cr, box[i + 4], box[i + 5]);
	    cairo_stroke(cr);
	  }
    }
  g_free(box);
  g_free(coordinates);

  /* We draw the axes. */
  /* We create the feedback for the axes. */
  nValuesAxes = -1;
  axes = (GLfloat*)0;
  if (axesGet_areOn())
    {
      axes = g_malloc(sizeof(GLfloat) * 7000);
      glFeedbackBuffer(7000, GL_3D, axes);
      glRenderMode(GL_FEEDBACK);
      OpenGLExtensionCall_list(EXT_AXES_ID);
      nValuesAxes = glRenderMode(GL_RENDER);

      cairo_set_line_width(cr, (double)axesGet_lineWidth());
      rgb = axesGet_RGBvalues();
      cairo_set_source_rgb(cr, rgb[0], rgb[1], rgb[2]);
      cairo_select_font_face(cr, "Serif", CAIRO_FONT_SLANT_NORMAL,
			     CAIRO_FONT_WEIGHT_NORMAL);
      cairo_set_font_size(cr, 12.0);
      /* We draw the 3 lines of the axes. */
      if (nValuesAxes >= 21 && axes[0] == GL_LINE_RESET_TOKEN &&
	  axes[7] == GL_LINE_RESET_TOKEN && axes[14] == GL_LINE_RESET_TOKEN)
	{
	  cairo_move_to(cr, axes[0 + 1], axes[0 + 2]);
	  cairo_line_to(cr, axes[0 + 4], axes[0 + 5]);
	  cairo_stroke(cr);
	  cairo_move_to(cr, axes[7 + 1], axes[7 + 2]);
	  cairo_line_to(cr, axes[7 + 4], axes[7 + 5]);
	  cairo_stroke(cr);
	  cairo_move_to(cr, axes[14 + 1], axes[14 + 2]);
	  cairo_line_to(cr, axes[14 + 4], axes[14 + 5]);
	  cairo_stroke(cr);
	}
      cairo_set_source_rgb(cr, 1.f - rgb[0], 1.f - rgb[1], 1.f - rgb[2]);
      cairo_set_font_matrix(cr, &scaleFont);
      if (nValuesAxes >= 33 && axes[21] == GL_BITMAP_TOKEN &&
	  axes[25] == GL_BITMAP_TOKEN && axes[29] == GL_BITMAP_TOKEN)
	{
	  cairo_move_to(cr, axes[21 + 1], axes[21 + 2]);
	  cairo_show_text(cr, "x");
	  cairo_move_to(cr, axes[25 + 1], axes[25 + 2]);
	  cairo_show_text(cr, "y");
	  cairo_move_to(cr, axes[29 + 1], axes[29 + 2]);
	  cairo_show_text(cr, "z");
	}
      g_free(axes);
    }
  
  cairo_show_page(cr);
  cairo_destroy(cr);

  return TRUE;
}

static gboolean writeViewInSvgFormat(FileFormat *format _U_, const char* filename,
				     int width, int height, VisuData *dataObj,
				     guchar* imageData _U_, GError **error,
				     voidDataFunc functionWait, gpointer data)
{
  cairo_surface_t *svg_surface;
  cairo_status_t status;
  OpenGLView *view;
  guint old_width, old_height;

  g_return_val_if_fail(error && !*error, FALSE);

  DBG_fprintf(stderr, "Dump SVG: begin export to SVG.\n");

  svg_surface = cairo_svg_surface_create(filename, (double)width, (double)height);
  status = cairo_surface_status(svg_surface);
  if (status != CAIRO_STATUS_SUCCESS)
    {
      *error = g_error_new(VISU_ERROR_DUMP, DUMP_ERROR_FILE,
			   cairo_status_to_string(status));
      cairo_surface_destroy(svg_surface);
      return FALSE;
    }

  view = visuDataGet_openGLView(dataObj);

  /* Change the view port to match width and height. */
  old_width = view->window->width;
  old_height = view->window->height;
  OpenGLViewSet_windowSize(view, (guint)width, (guint)height);
  writeDataToCairoSurface(svg_surface, dataObj, error, functionWait, data);
  cairo_surface_destroy(svg_surface);

  /* Set back the viewport. */
  OpenGLViewSet_windowSize(view, old_width, old_height);

  return TRUE;
}

#endif
#endif
