using System;
using System.Collections;
using NUnit.Framework;

namespace NUnit.Core.Tests
{
	/// <summary>
	/// Summary description for PlatformHelperTests.
	/// </summary>
	[TestFixture]
	public class PlatformDetectionTests
	{
		private static readonly PlatformHelper win95Helper = new PlatformHelper( 
			new OperatingSystem( PlatformID.Win32Windows , new Version( 4, 0 ) ),
			new RuntimeFramework( RuntimeType.Net, new Version( 1, 1, 4322, 0 ) ) );

		private static readonly PlatformHelper winXPHelper = new PlatformHelper( 
			new OperatingSystem( PlatformID.Win32NT , new Version( 5,1 ) ),
			new RuntimeFramework( RuntimeType.Net, new Version( 1, 1, 4322, 0 ) ) );

		private void CheckOSPlatforms( OperatingSystem os, 
			string expectedPlatforms )
		{
			CheckPlatforms(
				new PlatformHelper( os, RuntimeFramework.CurrentFramework ),
				expectedPlatforms,
				PlatformHelper.OSPlatforms );
		}

		private void CheckRuntimePlatforms( RuntimeFramework runtimeFramework, 
			string expectedPlatforms )
		{
			CheckPlatforms(
				new PlatformHelper( Environment.OSVersion, runtimeFramework ),
				expectedPlatforms,
				PlatformHelper.RuntimePlatforms + ",NET-1.0,NET-1.1,NET-2.0,MONO-1.0,MONO-2.0" );
		}

		private void CheckPlatforms( PlatformHelper helper, 
			string expectedPlatforms, string checkPlatforms )
		{
			string[] expected = expectedPlatforms.Split( new char[] { ',' } );
			string[] check = checkPlatforms.Split( new char[] { ',' } );

			foreach( string testPlatform in check )
			{
				bool shouldPass = false;

				foreach( string platform in expected )
					if ( shouldPass = platform.ToLower() == testPlatform.ToLower() )
						break;

				bool didPass = helper.IsPlatformSupported( testPlatform );
				
				if ( shouldPass && !didPass )
					Assert.Fail( "Failed to detect {0}", testPlatform );
				else if ( didPass && !shouldPass )
					Assert.Fail( "False positive on {0}", testPlatform );
			}
		}

		[Test]
		public void DetectWin95()
		{
			CheckOSPlatforms( 
				new OperatingSystem( PlatformID.Win32Windows, new Version( 4, 0 ) ),
				"Win95,Win32Windows,Win32,Win" );
		}

		[Test]
		public void DetectWin98()
		{
			CheckOSPlatforms( 
				new OperatingSystem( PlatformID.Win32Windows, new Version( 4, 10 ) ),
				"Win98,Win32Windows,Win32,Win" );
		}

		[Test]
		public void DetectWinMe()
		{
			CheckOSPlatforms( 
				new OperatingSystem( PlatformID.Win32Windows, new Version( 4, 90 ) ),
				"WinMe,Win32Windows,Win32,Win" );
		}

        // WinCE isn't defined in .NET 1.0.
		[Test, Platform(Exclude="Net-1.0")]
		public void DetectWinCE()
		{
            PlatformID winCE = (PlatformID)Enum.Parse(typeof(PlatformID), "WinCE");
			CheckOSPlatforms(
                new OperatingSystem(winCE, new Version(1, 0)),
				"WinCE,Win32,Win" );
		}

		[Test]
		public void DetectNT3()
		{
			CheckOSPlatforms( 
				new OperatingSystem( PlatformID.Win32NT, new Version( 3, 51 ) ),
				"NT3,Win32NT,Win32,Win" );
		}

		[Test]
		public void DetectNT4()
		{
			CheckOSPlatforms( 
				new OperatingSystem( PlatformID.Win32NT, new Version( 4, 0 ) ),
				"NT4,Win32NT,Win32,Win" );
		}

		[Test]
		public void DetectWin2K()
		{
			CheckOSPlatforms( 
				new OperatingSystem( PlatformID.Win32NT, new Version( 5, 0 ) ),
				"Win2K,NT5,Win32NT,Win32,Win" );
		}

		[Test]
		public void DetectWinXP()
		{
			CheckOSPlatforms( 
				new OperatingSystem( PlatformID.Win32NT, new Version( 5, 1 ) ),
				"WinXP,NT5,Win32NT,Win32,Win" );
		}

		[Test]
		public void DetectWin2003Server()
		{
			CheckOSPlatforms( 
				new OperatingSystem( PlatformID.Win32NT, new Version( 5, 2 ) ),
				"Win2003Server,NT5,Win32NT,Win32,Win" );
		}

		[Test]
		public void DetectUnix()
		{
			CheckOSPlatforms( 
				new OperatingSystem( PlatformHelper.UnixPlatformID, new Version() ),
				"UNIX,Linux" );
		}

		[Test]
		public void DetectNet10()
		{
			CheckRuntimePlatforms(
				new RuntimeFramework( RuntimeType.Net, new Version( 1, 0, 3705, 0 ) ),
				"NET,NET-1.0" );
		}

		[Test]
		public void DetectNet11()
		{
			CheckRuntimePlatforms(
				new RuntimeFramework( RuntimeType.Net, new Version( 1, 1, 4322, 0 ) ),
				"NET,NET-1.1" );
		}

		[Test]
		public void DetectNet20()
		{
			CheckRuntimePlatforms(
				new RuntimeFramework( RuntimeType.Net, new Version( 2, 0, 40607, 0 ) ),
				"Net,Net-2.0" );
		}

		[Test]
		public void DetectNetCF()
		{
			CheckRuntimePlatforms(
				new RuntimeFramework( RuntimeType.NetCF, new Version( 1, 1, 4322, 0 ) ),
				"NetCF" );
		}

		[Test]
		public void DetectSSCLI()
		{
			CheckRuntimePlatforms(
				new RuntimeFramework( RuntimeType.SSCLI, new Version( 1, 0, 3, 0 ) ),
				"SSCLI,Rotor" );
		}

		[Test]
		public void DetectMono10()
		{
			CheckRuntimePlatforms(
				new RuntimeFramework( RuntimeType.Mono, new Version( 1, 1, 4322, 0 ) ),
				"Mono,Mono-1.0" );
		}

		[Test]
		public void DetectMono20()
		{
			CheckRuntimePlatforms(
				new RuntimeFramework( RuntimeType.Mono, new Version( 2, 0, 40607, 0 ) ),
				"Mono,Mono-2.0" );
		}

		[Test]
		public void DetectExactVersion()
		{
			Assert.IsTrue( winXPHelper.IsPlatformSupported( "net-1.1.4322" ) );
			Assert.IsTrue( winXPHelper.IsPlatformSupported( "net-1.1.4322.0" ) );
			Assert.IsFalse( winXPHelper.IsPlatformSupported( "net-1.1.4323.0" ) );
			Assert.IsFalse( winXPHelper.IsPlatformSupported( "net-1.1.4322.1" ) );
		}

		[Test]
		public void ArrayOfPlatforms()
		{
			string[] platforms = new string[] { "NT4", "Win2K", "WinXP" };
			Assert.IsTrue( winXPHelper.IsPlatformSupported( platforms ) );
			Assert.IsFalse( win95Helper.IsPlatformSupported( platforms ) );
		}

		[Test]
		public void PlatformAttribute_Include()
		{
			PlatformAttribute attr = new PlatformAttribute( "Win2K,WinXP,NT4" );
			Assert.IsTrue( winXPHelper.IsPlatformSupported( attr ) );
			Assert.IsFalse( win95Helper.IsPlatformSupported( attr ) );
			Assert.AreEqual("Only supported on Win2K,WinXP,NT4", win95Helper.Reason);
		}

		[Test]
		public void PlatformAttribute_Exclude()
		{
			PlatformAttribute attr = new PlatformAttribute();
			attr.Exclude = "Win2K,WinXP,NT4";
			Assert.IsFalse( winXPHelper.IsPlatformSupported( attr ) );
			Assert.AreEqual( "Not supported on Win2K,WinXP,NT4", winXPHelper.Reason );
			Assert.IsTrue( win95Helper.IsPlatformSupported( attr ) );
		}

		[Test]
		public void PlatformAttribute_IncludeAndExclude()
		{
			PlatformAttribute attr = new PlatformAttribute( "Win2K,WinXP,NT4" );
			attr.Exclude = "Mono";
			Assert.IsFalse( win95Helper.IsPlatformSupported( attr ) );
			Assert.AreEqual( "Only supported on Win2K,WinXP,NT4", win95Helper.Reason );
			Assert.IsTrue( winXPHelper.IsPlatformSupported( attr ) );
			attr.Exclude = "Net";
			Assert.IsFalse( win95Helper.IsPlatformSupported( attr ) );
			Assert.AreEqual( "Only supported on Win2K,WinXP,NT4", win95Helper.Reason );
			Assert.IsFalse( winXPHelper.IsPlatformSupported( attr ) );
			Assert.AreEqual( "Not supported on Net", winXPHelper.Reason );
		}

		[Test]
		public void PlatformAttribute_InvalidPlatform()
		{
			PlatformAttribute attr = new PlatformAttribute( "Net-1.0,Net11,Mono" );
			Assert.IsFalse( winXPHelper.IsPlatformSupported( attr ) );
			Assert.AreEqual( "Invalid platform name: Net11", winXPHelper.Reason );
		}
	}
}
