#!/usr/bin/perl
#
# See LICENSE for copyright information
#
# check_disk.pl <host> <disk> [warn] [critical] [port]
#
# NetSaint host script to get the disk usage from a client that is running
# netsaint_statd.
#

require 5.003;
BEGIN { $ENV{PATH} = '/bin' }
use Socket;
use POSIX;
use strict;

sub usage;

my $TIMEOUT = 15;

my %ERRORS = ('UNKNOWN', '-1',
		'OK', '0',
		'WARNING', '1',
		'CRITICAL', '2');

my $remote = shift || &usage(%ERRORS);
my $disk = shift || &usage(%ERRORS);
my $warn = shift || 85;
my $crit = shift || 95;
my $port = shift || 1040;

my $remoteaddr = inet_aton("$remote");
my $paddr = sockaddr_in($port, $remoteaddr) || die "Can't create info for connection: #!\n";;
my $proto = getprotobyname('tcp');
socket(Server, PF_INET, SOCK_STREAM, $proto) || die "Can't create socket: $!";
setsockopt(Server, SOL_SOCKET, SO_REUSEADDR, 1);
connect(Server, $paddr) || die "Can't connect to server: $!";

my $state = "OK";
my $answer = undef;

# Just in case of problems, let's not hang NetSaint
$SIG{'ALRM'} = sub { 
        close(Server);
        select(STDOUT);
	print "No Answer from Client\n";
	exit $ERRORS{"UNKNOWN"};
};
alarm($TIMEOUT);

select(Server);
$| = 1;

print Server "disk $disk\n";
my ($servanswer) = <Server>;
alarm(0);
close(Server);

select(STDOUT);

if ($servanswer =~ /^[\w\/\d:\.\-\=]* (\d*) (\d*) ([\w\/\-]*)/)
	{
	my ($avail) = $1;
	my ($capper) = $2;
	my ($mountpt) = $3;
	if (int($avail / 1024) > 0)
		{
		$avail = int($avail / 1024);
		if (int($avail /1024) > 0)
			{
			$avail = (int(($avail / 1024)*100))/100;
			$avail = $avail."G";
			}
		else
		{
		$avail = $avail."M";
		}
		}
	else
		{
		$avail = $avail."K";
		}
	if ((100 - $capper) < $warn)
		{ 
		$answer = "Disk ok - $avail ($capper%) free on $mountpt\n";
		}
	elsif ((100 - $capper) < $crit)
		{
		$state = "WARNING";
		$answer = "Only $avail ($capper%) free on $mountpt\n";
		}
	else
		{
		$state = "CRITICAL";
		$answer = "Only $avail ($capper%) free on $mountpt\n";
		}
	}
# If the disk is missing:
elsif ($servanswer eq "not found")
	{
	$state = "CRITICAL";
	$answer = "Disk $disk not mounted or nonexistant\n";
	}
else
	{
	$state = "UNKNOWN";
	$answer = "Unknown status on disk $disk\n";
	}

print $answer;
exit $ERRORS{$state};

sub usage {
	print "Minimum arguments not supplied!\n";
	print "\n";
	print "Perl Check Disk plugin for NetSaint\n";
	print "Copyright (c) 1999 Charlie Cook\n";
	print "\n";
	print "Usage: $0 <host> <disk> [<warn> [<crit> [<port>]]]\n";
	print "\n";
	print "<warn> = Percent of used space at which a warning message will be generated.\n	Defaults to 85.\n";
	print "<crit> = Percent of used space at which a critical message will be generated.\n	Defaults to 95.\n";
	print "<port> = Port that the status daemon is running on <host>.\n	Defaults to 1040.\n";
	exit $ERRORS{"UNKNOWN"};
}
