/* MenuBuilderVisitor.java
 * =========================================================================
 * This file is part of the SWIRL Library - http://swirl-lib.sourceforge.net
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 * 
 */

package be.ugent.caagt.swirl.menus;

import be.ugent.caagt.swirl.GenericSelectionGroup;
import be.ugent.caagt.swirl.GenericSelectionModel;
import be.ugent.caagt.swirl.actions.Description;
import javax.swing.AbstractButton;
import javax.swing.Action;
import javax.swing.ButtonModel;
import javax.swing.JMenu;

/**
 * Common superclass of visitors that are used to setup a menu hierarchy.
 */
abstract class MenuBuilderVisitor implements ConfigurationVisitor {

    //
    private final MenuBuilder builder;

    public MenuBuilderVisitor(MenuBuilder builder) {
        this.builder = builder;
    }

    public void visitSeparator(SeparatorConfiguration config) {
        if (config.satisfies(builder.getPredicates())) {
            addSeparator(config.isGlue());
        }
    }

    private void visitActionAux(ActionConfiguration config) throws MenuConfigurationException {
        String id = config.getId();
        Action action = builder.getAction(id);
        if (action == null) {
            throw new MenuConfigurationException("No action registered for key: " + config.getId());
        }
        // configure action if not yet configured
        if (action.getValue(Action.NAME) == null) {
            config.getDescription().init(action);
        }
        addAction(action).setName(id);
    }

    public void visitAction(ActionConfiguration config) {
        if (config.satisfies(builder.getPredicates())) {
            visitActionAux(config);
        }
    }

    public void visitActionReference(ActionReferenceConfiguration config) {
        if (config.satisfies(builder.getPredicates())) {
            visitActionAux(config.getActionConfiguration());
        }
    }

    private void visitToggleAux(ToggleConfiguration config) throws MenuConfigurationException {
        String id = config.getId();
        ButtonModel bm = builder.getToggleModel(id);
        if (bm == null) {
            throw new MenuConfigurationException("No model registered for toggle key: " + config.getId());
        }
        addToggle(bm, config.getDescription()).setName(id);
    }

    public void visitToggle(ToggleConfiguration config) {
        if (config.satisfies(builder.getPredicates())) {
            visitToggleAux(config);
        }
    }

    public void visitToggleReference(ToggleReferenceConfiguration config) {
        if (config.satisfies(builder.getPredicates())) {
            visitToggleAux(config.getToggleConfiguration());
        }
    }

    private void visitGroupAux(GroupConfiguration config) throws MenuConfigurationException {
        String id = config.getId();
        GroupInfo groupInfo = builder.getGroupInfo(id);
        if (groupInfo == null) {
            throw new MenuConfigurationException("No model registered for group key: " + config.getId());
        }
        @SuppressWarnings("unchecked")
        GenericSelectionModel<Object> model = groupInfo.getSelectionModel();
        GenericSelectionGroup<Object> gsg 
                = new GenericSelectionGroup<Object>(model, config.isClearable());
        for (ButtonConfiguration bc : config.getEntries()) {
            String bid = bc.getId();
            Object obj = groupInfo.get(bid);
            if (obj == null) {
                throw new MenuConfigurationException("No button key '" + bid + "' in group '" + id + "'");
            }
            if (bc.satisfies(builder.getPredicates())) {
                AbstractButton button = addButton(bc.getDescription());
                button.setName(bc.getId());
                gsg.add(button, obj);
            }
        }
    }

    public void visitGroup(GroupConfiguration config) {
        if (config.satisfies(builder.getPredicates())) {
            visitGroupAux(config);
        }
    }

    public void visitGroupReference(GroupReferenceConfiguration config) {
        if (config.satisfies(builder.getPredicates())) {
            visitGroupAux(config.getGroupConfiguration());
        }
    }

    public void visitMenu(MenuConfiguration config) {
        if (config.satisfies(builder.getPredicates())) {
            JMenu menu = new JMenu();
            config.getDescription().init(menu);
            menu.setName(config.getName());
            addMenu(menu, config.getEntries());
        }
    }

    protected abstract void addSeparator(boolean glue);

    protected abstract AbstractButton addAction(Action action);

    protected abstract AbstractButton addButton(Description description);

    protected abstract AbstractButton addToggle(ButtonModel model, Description description);

    protected abstract void addMenu(JMenu menu, Iterable<EntryConfiguration> entries);
}
