/*
*   pam_abl - a PAM module and program for automatic blacklisting of hosts and users
*
*   Copyright (C) 2005 Andy Armstrong andy@hexten.net
*   Copyright (C) 2009 Chris Tasma pam-abl@deksai.com
*
*   This program is free software: you can redistribute it and/or modify
*   it under the terms of the GNU General Public License as published by
*   the Free Software Foundation, either version 2 of the License, or
*   (at your option) any later version.
*
*   This program is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU General Public License for more details.
*
*   You should have received a copy of the GNU General Public License
*   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "../../config.h"
#include "pam_abl.h"
#include <inttypes.h>
#include <getopt.h>

#define MAXNAMES 200

static int relative  = 0;
static int verbose   = 0;
const char *users[MAXNAMES];
const char *hosts[MAXNAMES];
int command=SHOW,num_users=0, num_hosts=0;

#define PAD "    "

static void usage(const char *prg) {
    printf("Usage: %s [OPTION] [CONFIG]\n", prg);
    printf("Perform maintenance on the databases used by the pam_abl (auto blacklist)\n"
           "module. CONFIG is the name of the pam_abl config file (defaults to\n"
           CONFIG "). The config file is read to discover the names\n"
           "of the pam_abl databases and the rules that control purging of old data\n"
           "from them. The following options are available:\n\n"
           "MAINTENANCE\n"
           "  -h, --help              See this message.\n"
           "  -p, --purge             Purge databases based on rules in config.\n"
           "  -r, --relative          Display times relative to now.\n"
           "  -v, --verbose           Verbose output.\n"
           "\n"
           "NON-PAM INTERACTION\n"
           "  -f  --fail\n"
           "      Fail user or host.\n"
           "  -w  --whitelist\n"
           "      Perform whitelisting (remove from blacklist, does not provide immunity).\n"
           "  -c  --check\n"
           "      Check status.  Returns non-zero if currently blocked\n"
           "      Prints 'name: status' if verboseness is specified.\n"
           "  -s  --service\n"
           "      Operate in context of specified service.  Defaults to 'none'.\n"
           "  -U  --user\n"
           "      Operate on user (wildcards are ok for whitelisting).\n"
           "  -H  --host\n"
           "      Operate on host (wildcards are ok for whitelisting).\n"
           "\n"
           "Please send comments, questions and bug reports to " PACKAGE_BUGREPORT".\n");
    exit(0);
}

static void mention(const char *msg, ...) {
    if (verbose > 0) {
        va_list ap;
        va_start(ap, msg);
        vprintf(msg, ap);
        printf("\n");
        va_end(ap);
    }
}

static int wildmatch(const char *key, const char *value, const char *end) {
    for (;;) {
        switch (*key) {
        case '\0':
            return value == end;
        case '*':
            key++;
            for (;;) {
                if (wildmatch(key, value, end)) {
                    return 1;
                }
                if (value == end) {
                    return 0;
                }
                value++;
            }
        case '?':
            if (value == end) {
                return 0;
            }
            break;
        default:
            if (value == end || *value++ != *key) {
                return 0;
            }
            break;
        }
        key++;
    }
}

static int iswild(const char *key) {
    return NULL != strchr(key, '*') || NULL != strchr(key, '?');
}

static void reltime(long t) {
    long days    = t / (24 * 60 * 60);
    long hours   = t / (60 * 60) % 24;
    long minutes = t / 60 % 60;
    long seconds = t % 60;
    printf(PAD PAD "%ld/%02ld:%02ld:%02ld\n", days, hours, minutes, seconds);
}

static void showblocking(const abl_args *args, const char *rule, const time_t *history, size_t histsz, time_t now) {
    int op = 0;
    while (*rule) {
        const char *up;
        const char *colon = strchr(rule, ':');
        if (NULL == colon) {
            break;
        }
        up = rule;
        rule = colon + 1;
        if (rule_matchperiods(args, history, histsz, now, &rule)) {
            if (!op) {
                printf(PAD PAD "Blocked based on rule [");
                op = 1;
            } else {
                printf("], [");
            }
            while (up != colon) {
               putchar(*up++);
            } 
        }
        while (*rule != '\0' && !isspace(*rule)) {
            rule++;
        }
        while (*rule != '\0' && isspace(*rule)) {
            rule++;
        }
    }

    if (op) {
        printf("]\n");
    } else {
        printf(PAD PAD "Not blocking\n");
    }
}

static int doshow(const abl_args *args, abl_info *info) {
    DB *db=NULL;
    DBT key, data;
    DBC *cursor;
    int err = 0;
    int bsz = 0;
    int cnt = 0;
    char *buf = NULL, *thing = NULL;
    const char *rule;
    time_t now = time(NULL);
	u_int32_t fail_count;  // Matches db.h


    if (args == NULL || info== NULL) {
        return 0;
    }

    if( info->subject == HOST ) {
        db = info->htdb;
        thing = "hosts";
        rule = args->host_rule;
    }
    else {
        db = info->utdb;
        thing = "users";
        rule = args->user_rule;
    }


    //Section header for output
    printf("Failed %s:\n", thing);

    memset(&key,  0, sizeof(key));
    memset(&data, 0, sizeof(data));

    DB_TXN *tid = NULL;
    if ((err = info->dbenv->txn_begin(info->dbenv, NULL, &tid, 0)) != 0) {
        log_db_error(err, "starting transaction");
        goto doshow_fail;
    }
    
    if (err = db->cursor(db, tid, &cursor, 0), 0 != err) {
        log_db_error(err, "creating cursor");
        goto doshow_fail;
    }

    for (;;) {
        time_t *tm;
        size_t ntm;
        
        err = cursor->c_get(cursor, &key, &data, DB_NEXT);
        if (DB_NOTFOUND == err) {
            break;
        } else if (0 != err) {
            log_db_error(err, "iterating cursor");
            goto doshow_fail;
        }

        /* Print it out */
        if (bsz < key.size + 1) {
            char *nb;
            int ns = key.size + 80;
            if (nb = realloc(buf, ns), NULL == nb) {
                log_sys_error(ENOMEM, "displaying item");
                goto doshow_fail;
            }
            buf = nb;
            bsz = ns;
        }

        memcpy(buf, key.data, key.size);
        buf[key.size] = '\0';
		fail_count = data.size / sizeof(time_t);
        printf(PAD "%s (%"PRIu32")\n", buf, fail_count);
        cnt++;

        tm = (time_t *) data.data;
        ntm = data.size / sizeof(time_t);
        if (verbose) {
            while (ntm != 0) {
                ntm--;
                if (relative) {
                    reltime((long) difftime(now, tm[ntm]));
                } else {
                    printf(PAD PAD "%s", ctime(&tm[ntm]));
                }
            }
        } else if (NULL != rule) {
            showblocking(args, rule, tm, ntm, now);  
        }
    }

    if (0 == cnt) {
        printf("   <none>\n");
    }

    /* Cleanup */
doshow_fail:
#if DB_VERSION_MAJOR < 5
    if (cursor != NULL) 
        cursor->c_close(cursor);
#else
    if (cursor != NULL) 
        cursor->close(cursor);
#endif
    if (tid) {
        tid->commit(tid, 0);
    }
    free(buf);
    return err;
}

static int dopurge(const abl_args *args, abl_info *info) {
    DB *tdb = NULL;
    DB *sdb = NULL;
    DBT key, data;
    DBC *cursor;
    int err;
    long maxage = 0;
    const void **thing;
    time_t now = time(NULL);

    if ( args == NULL || info == NULL ) {
        return 0;
    }

    if(info->subject == HOST) {
        tdb = info->htdb;
        sdb = info->hsdb;
        maxage = args->host_purge;
        thing = &info->host;
    }
    else {
        tdb = info->utdb;
        sdb = info->usdb;
        maxage = args->user_purge;
        thing = &info->user;
    }
    /*Cheat, because get_dbname is broken starting on some version lower than 4.8 */
    mention("Purging %s", tdb->fname);

    memset(&key,  0, sizeof(key));
    memset(&data, 0, sizeof(data));

    DB_TXN *tid = 0;
    if ((err = info->dbenv->txn_begin(info->dbenv, NULL, &tid, 0)) != 0) {
        log_db_error(err, "starting transaction");
        return err;
    }
    if (err = tdb->cursor(tdb, tid, &cursor, 0), 0 != err) {
        log_db_error(err, "creating cursor");
        goto dopurge_fail;
    }

    for (;;) {
        err = cursor->c_get(cursor, &key, &data, DB_NEXT);
        if (DB_NOTFOUND == err) {
            break;
        } else if (0 != err) {
            log_db_error(err, "iterating cursor");
            goto dopurge_fail;
        }

        if (rule_purge(&data, maxage, now)) {
            if (data.size == 0) {
                err = ( cursor->c_del(cursor, 0) || sdb->del(sdb, tid, &key, 0) ); 
                if (err) {
                    goto dopurge_fail;
                }
                info->state = CLEAR;
                *thing = (char*) key.data;
                update_status(args, info, tid);
                if (info->state_change) run_command(args,info);

            } else {
                if (err = cursor->c_put(cursor, &key, &data, DB_CURRENT), 0 != err) {
                    goto dopurge_fail;
                }
            }
        }
    }

    /* Cleanup */
dopurge_fail:
#if DB_VERSION_MAJOR < 5
    if (cursor != NULL) 
        cursor->c_close(cursor);
#else
    if (cursor != NULL) 
        cursor->close(cursor);
#endif

    if (tid) {
        if (err) {
            tid->abort(tid);
        } else {
            tid->commit(tid, 0);
        }
    }
    return err;
}

static int whitelist(const abl_args *args, abl_info *info, const char **permit, int count) {
    DB *db;
    int err = 0;
    DBT key, data;
    DBC *cursor = NULL;
    int wild = 0;
    int n, hit, del = 0;
    int *done;
    const void **thing;

    if (args == NULL || info == NULL) {
        return 0;
    }

    if(info->subject == HOST) {
        db = info->htdb;
        thing = &info->host;
    }
    else {
        db = info->utdb;
        thing = &info->user;
    }

    if (done = malloc(sizeof(int) * count), NULL == done) {
        log_sys_error(ENOMEM, "allocating buffer");
    }
    
    for (n = 0; n < count; n++) {
        /* Any wildcards? */
        if (iswild(permit[n])) {
            wild = 1;
        }
        //Initialize done
        done[n] = 0;
    }

    memset(&key,  0, sizeof(key));
    //key.flags = DB_DBT_REALLOC;
    memset(&data, 0, sizeof(data));
    //data.flags = DB_DBT_REALLOC;

    DB_TXN *tid = 0;
    if ((err = info->dbenv->txn_begin(info->dbenv, NULL, &tid, 0)) != 0) {
        log_db_error(err, "starting transaction");
        goto whitelist_fail;
    }
    if (wild) {
        /* wildcard delete */

        if (err = db->cursor(db, tid, &cursor, 0), 0 != err) {
            log_db_error(err, "creating cursor");
            goto whitelist_fail;
        }

        for (;;) {
            err = cursor->c_get(cursor, &key, &data, DB_NEXT);
            if (DB_NOTFOUND == err) {
                break;
            } else if (0 != err) {
                log_db_error(err, "iterating cursor");
                goto whitelist_fail;
            }

            for (n = 0, hit = 0; n < count; n++) {
                if (wildmatch(permit[n], (const char *) key.data, (const char *) key.data + key.size)) {
                    done[n] = 1;
                    hit = 1;
                }
            }

            if (hit) {
                if (err = cursor->c_del(cursor, 0), 0 != err) {
                    goto whitelist_fail;
                }
                info->state = CLEAR;
                *thing = permit[n];
                update_status(args, info, tid);
                if (info->state_change) run_command(args,info);
                del++;
            }
        }
    } else {
        /* delete individual items */
        for (n = 0; n < count; n++) {
            if (!done[n]) {
                make_key(&key, permit[n]);
                err = db->del(db, tid, &key, 0);
                if (0 == err) {
                    int m;
                    done[n] = 1;
                    del++;
                    info->state = CLEAR;
                    *thing = permit[n];
                    update_status(args, info, tid);
                    if (info->state_change) run_command(args,info);
                    /* Slightly pedantic - knock out any duplicates so we don't flag an error when
                     * we can't delete them later. Arguably if you supply duplicate arguments you
                     * should expect an error but this keeps the semantics the same as the wildcard
                     * case.
                     */
                    for (m = n + 1; m < count; m++) {
                        if (strcmp(permit[n], permit[m]) == 0) {
                            done[m] = 1;
                        }
                    }
                } else if (DB_NOTFOUND == err) {
                    /* shrug */
                } else {
                    goto whitelist_fail;
                }
            }
        }
    }

    if (verbose) {
        hit = 0;
        if (del) {
            printf("Deleted %d item%s\n", del, del == 1 ? "" : "s");
        }
        for (n = 0; n < count; n++) {
            if (!done[n]) {
                if (!hit) {
                    printf("These keys could not be matched:\n");
                    hit++;
                }
                printf(PAD "%s\n", permit[n]);
            }
        }
    }

    /* Cleanup */
whitelist_fail:
    free(done);
#if DB_VERSION_MAJOR < 5
    if (cursor != NULL) {
        cursor->c_close(cursor);
    }
#else
    if (cursor != NULL) {
        cursor->close(cursor);
    }
#endif
    if (tid) {
        if (err) {
            tid->abort(tid);
        } else {
        	tid->commit(tid, 0);
        }
    }
    return err;
}

static int fail(const abl_args *args, abl_info *info, const char **prohibitions, int count, int max_age) {
    int n, err = 0, add = 0;
    int *done;
    time_t tm = time(NULL);

    if (args == NULL || info == NULL) {
        return 0;
    }

    if (done = malloc(sizeof(int) * count), NULL == done) {
        log_sys_error(ENOMEM, "allocating buffer");
    }
    
    /* Any wildcards? */
    for (n = 0; n < count; n++) {
        if (iswild(prohibitions[n]) && command == FAIL) {
            die("Wildcards don't make sense when using -f or --fail.  Aborting.");
        }
        //Initialize done
        done[n] = 0;
    }

    /* add individual items */
    for (n = 0; n < count; n++) {
        if (!done[n]) {
            if(info->subject == HOST) {
                info->host = prohibitions[n];
                err = record(args,info,tm,max_age);
                if(!err) check_host(args,info,tm);
            } else {
                info->user = prohibitions[n];
                err = record(args,info,tm,max_age);
                if(!err) check_user(args,info,tm);
            }

            if (0 == err) {
                int m;
                done[n] = 1;
                add++;
                /* Slightly pedantic - knock out any duplicates so we don't flag an error when
                 * we can't delete them later. Arguably if you supply duplicate arguments you
                 * should expect an error but this keeps the semantics the same as the wildcard
                 * case.
                 */
                for (m = n + 1; m < count; m++) {
                    if (strcmp(prohibitions[n], prohibitions[m]) == 0) {
                        done[m] = 1;
                    }
                }
            }
        }
    }

    if (verbose) {
        if (add) {
            printf("Failed %d item%s\n", add, add == 1 ? "" : "s");
        }
    }

    /* Cleanup */
    free(done);

    return err;
}

int main(int argc, char **argv) {
    int err, n, c;
    char *conf = NULL, *service = "none";
    abl_args args;
    abl_info info;
    DB *utdb;
    DB *usdb;
    DB *htdb;
    DB *hsdb;
    DB_ENV *dbenv;

    while (1) {

        int option_index = 0;
        static struct option long_options[] = {
            {"help",        0, 0,    'h' },
            {"purge",       0, 0,    'p' },
            {"relative",    0, 0,    'r' },
            {"verbose",     0, 0,    'v' },
            {"fail",        1, 0,    'f' },
            {"whitelist",   1, 0,    'w' },
            {"check",       1, 0,    'c' },
            {"user",        1, 0,    'U' },
            {"host",        1, 0,    'H' },
            {"service",     1, 0,    's' },
            {0,             0, 0,     0  }
        };

        c = getopt_long(argc, argv, "hrvpfwcU:H:s:",
                long_options, &option_index);
        if (c == -1)
            break;
        switch (c) {
            case 'h':
                usage(argv[0]);
                break;
            case 'p':
                command = PURGE;
                break;
            case 'r':
                relative=1;
                break;
            case 'v':
                verbose=1;
                break;
            case 'f':
                command = FAIL;
                break;
            case 'w':
                command = WHITELIST;
                break;
            case 'c':
                command = CHECK;
                break;
            case 'U':
                users[num_users++]=optarg;
                break;
            case 'H':
                hosts[num_hosts++]=optarg;
                break;
            case 's':
                service=optarg;
                break;
            case '?':
                break;

            default:
                printf("Unknown error parsing command line arguments: code 0%o\n", c);
        }
    }

    if (optind < argc) {
        conf = argv[optind++];
        while (optind < argc)
            mention("Ignored command line argument: %s",argv[optind++]);
        printf("\n");
    }
    if (NULL == conf) {
        conf = CONFIG;
    }

    config_clear(NULL, &args);
    mention("Reading config from %s", conf);
    if (err = config_parse_file(conf, &args), 0 != err) {
        return err;
    }

    if (NULL == args.user_db) {
        mention("No user_db in %s", conf);
    }

    if (NULL == args.host_db) {
        mention("No host_db in %s", conf);
    }
    dump_args(&args);

    for(n=0;n<num_users;n++){
        log_debug(&args,"user: %s",users[n]);
    }
    for(n=0;n<num_hosts;n++){
        log_debug(&args,"host: %s",hosts[n]);
    }
    memset(&info, 0, sizeof(abl_info));

    /* Most everything should be set, and it should be safe to open the
     * databases. */
    err = dbenvironment(args.db_home, &dbenv);
    if (err) 
        return err;
    info.dbenv = dbenv;
    err = dbopen(&args, args.user_db, DB_TIME, dbenv, &utdb);
    if (err) return err;
    info.utdb = utdb;
    err = dbopen(&args, args.user_db, DB_STATE, dbenv, &usdb);
    if (err) return err;
    info.usdb = usdb;
    err = dbopen(&args, args.host_db, DB_TIME, dbenv, &htdb);
    if (err) return err;
    info.htdb = htdb;
    err = dbopen(&args, args.host_db, DB_STATE, dbenv, &hsdb);
    if (err) return err;
    info.hsdb = hsdb;



    if (command == WHITELIST) {
        if (num_users > 0) {
            info.subject = USER;
            whitelist(&args, &info, users, num_users);
        }
        if (num_hosts > 0) {
            info.subject = HOST;
            whitelist(&args, &info, hosts, num_hosts);
        }
        if (num_users == 0 && num_hosts == 0) {
            die("Asked to whitelist but no hosts or users given!");
        }
    }
    else if (command == FAIL) {
        if (num_users > 0) {
            info.subject = USER;
            fail(&args, &info, users, num_users, args.user_purge);
        }
        if (num_hosts > 0) {
            info.subject = HOST;
            fail(&args, &info, hosts, num_hosts, args.host_purge);
        }
        if (num_users == 0 && num_hosts == 0) {
            die("Asked to fail someone, but no hosts or users given!");
        }
    }
    
    if (command == CHECK) {
        int  err = 0;
        time_t tm = time(NULL);

        if (num_users == 0 && num_hosts == 0) {
            die("Asked to check but no hosts or users given!");
        }
        if (num_hosts > 1) {
            log_warning(&args,"More than one host specified.  Only the first one will be used!");
        }
        if (num_users > 1) {
            log_warning(&args,"More than one user specified.  Only the first one will be used!");
        }

        info.service = service ? service : NULL;

        if (num_hosts > 0) {
            info.host = hosts[0];
            info.subject = HOST;
            err = check_host(&args, &info, tm);
            if(err) {
                log_warning(&args,"Could not check host.");
            }
            else {
                if (info.state == BLOCKED) exit(info.state);
            }
        }

        if (num_users > 0) {
            info.user = users[0];
            info.subject = USER;
            err = check_user(&args, &info, tm);
            if(err) {
                log_warning(&args,"Could not check user.");
            }
            else {
                if (info.state == BLOCKED) exit(info.state);
            }
        }
        return 0;
    }


    if (command == PURGE) {
        info.subject = USER;
        dopurge(&args, &info);
        info.subject = HOST;
        dopurge(&args, &info);
    } else if (num_users == 0 && num_hosts == 0) {
        info.subject = USER;
        doshow(&args, &info);
        info.subject = HOST;
        doshow(&args, &info);
    }

    config_free(&args);
    err = utdb->close(utdb,0);
    err |= usdb->close(usdb,0);
    err |= htdb->close(htdb,0);
    err |= hsdb->close(hsdb,0);
    if (dbenv)
        dbenv->close(dbenv, 0);
    return 0;
}

/*
    //XXX temp for debuging database sections
    DBC *dbc;
    memset(&key,0,sizeof(key));
    memset(&data,0,sizeof(data));
    db->cursor(db,NULL,&dbc,0);
    while(1) {
        err = dbc->c_get(dbc, &key, &data, DB_NEXT);
        if (err == DB_NOTFOUND) break;
        printf("key '%s:%d', lookup '%s:%d' data %d\n",key.data,key.size,permit[0],strlen(permit[0]) + 1,data.data != NULL ? *(int *) data.data: 0);
    }
    err=0;
    //XXX /temp
*/
