/* 

                          Firewall Builder

                 Copyright (C) 2000 Vadim Kurland

  Author:  Vadim Kurland     vadim@vk.crocodile.org

  $Id: Interface.cc,v 1.3 2001/12/19 12:46:48 lord Exp $


  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include <fwbuilder/Interface.hh>
#include <fwbuilder/InterfacePolicy.hh>
#include <fwbuilder/XMLTools.hh>

using namespace std;
using namespace libfwbuilder;

const char *Interface::TYPENAME={"Interface"};

Interface::Interface(const Interface &i):FWObject((FWObject&)i)
{
    bcast_bits       = i.bcast_bits       ; 
    ostatus          = i.ostatus          ;
    snmp_type        = i.snmp_type        ;
}

Interface::Interface(const Interface *i):FWObject((FWObject&)(*i))
{
    bcast_bits       = i->bcast_bits       ; 
    ostatus          = i->ostatus          ;
    snmp_type        = i->snmp_type        ;
}

Interface::Interface():FWObject()
{
    setName("unknown");
    setStr("address", "");
    setStr("netmask", "");
    setBool("dyn",false);
    setStr("physAddress", "");
    setInt("security_level",0);

    bcast_bits       = 1    ;
    ostatus          = true ;
    snmp_type        = -1   ;     
}

Interface::Interface(const string& n,const string& a,const string& nm, 
                     int _bcast_bits, int _snmp_type, bool _ostatus,
		     bool ext,bool dyn, string _physical_address):FWObject()
{
    setName(n);

    setStr  ("address" , a   );
    setStr  ("netmask" , nm  );
    setBool ("dyn"     , dyn );
    setStr  ("physAddress", _physical_address);
    setExt  ( ext );

    bcast_bits       = _bcast_bits       ;
    snmp_type        = _snmp_type        ;
    ostatus          = _ostatus          ;
} 

Interface::Interface(const string& n,const string& a,const string& nm,
		     bool ext,const string& dyn, 
		     string _physical_address):FWObject()
{
    setName (n);
    setStr  ("address",a);
    setStr  ("netmask",nm);
    setExt  ( ext );
    setBool ("dyn", (dyn=="dyn") );
    setStr  ("physAddress", _physical_address);

    bcast_bits       = 1    ;
    ostatus          = true ;
    snmp_type        = -1   ; 
}

Interface::Interface(const string& n,const string& a,const string& nm,
		     int slevel,const string& dyn, 
		     string _physical_address):FWObject()
{
    setName (n);
    setStr  ("address",a);
    setStr  ("netmask",nm);
    setInt  ("security_level", slevel  );
    setBool ("dyn", (dyn=="dyn") );
    setStr  ("physAddress", _physical_address);

    bcast_bits       = 1    ;
    ostatus          = true ;
    snmp_type        = -1   ; 
}

Interface::~Interface() {}

Interface& Interface::operator=(const Interface &o)
{
//    duplicate(&o);
    FWObject::operator=(o);

    bcast_bits       = o.bcast_bits       ; 
    ostatus          = o.ostatus          ;
    snmp_type        = o.snmp_type        ;

    return *this;
}

const IPAddress Interface::getIPAddress() const throw(FWException)
{
    return IPAddress(getStr("address"));
}

void Interface::fromXML(xmlNodePtr root)
{
    FWObject::fromXML(root);

    const char *n=FROMXMLCAST(xmlGetProp(root,TOXMLCAST("address")));
    assert(n!=NULL);
    setStr("address", n);

    n=FROMXMLCAST(xmlGetProp(root,TOXMLCAST("netmask")));
    assert(n!=NULL);
    setStr("netmask", n);

    n=FROMXMLCAST(xmlGetProp(root,TOXMLCAST("physAddress")));
    if (n!=NULL) setStr("physAddress", n);

    n=FROMXMLCAST(xmlGetProp(root,TOXMLCAST("security_level")));
    if (n!=NULL) setStr("security_level",n);

    n=FROMXMLCAST(xmlGetProp(root,TOXMLCAST("dyn")));
    if (n!=NULL) setStr("dyn",n);

    n=FROMXMLCAST(xmlGetProp(root,TOXMLCAST("label")));
    if (n!=NULL) setStr("label",n);
}

int  Interface::getSecurityLevel()
{
    return getInt("security_level") ;
}

void Interface::setSecurityLevel(int level)
{
    setInt("security_level",level);
}

bool Interface::isExt() const 
{ 
    return( getInt("security_level")==0 );
}

/**
 *   if parameter value is true, then security level is set to 0 (least secure,
 *   or "outside")
 *   if parameter value is false, then current security level is set to 100
 */
void Interface::setExt(bool external)
{
    setInt("security_level",(external)?0:100);
}

bool Interface::isDyn() const { return(getBool("dyn")); }

bool  Interface::validateChild(FWObject *o)
{
    string otype=o->getTypeName();
    return (FWObject::validateChild(o) && 
            (otype==InterfacePolicy::TYPENAME ));
}

IPNetwork Interface::getIPNetwork() const throw(FWException)
{
    return IPNetwork(IPAddress(getStr("address")), 
                     Netmask(getStr("netmask")),
                     bcast_bits
    );
}

const string&    Interface::getPhysicalAddress () const
{
    return getStr("physAddress");
}

const string &Interface::getLabel() const
{
    return getStr("label");
}

void          Interface::setLabel(const string& n)
{
    setStr("label",n);
}
    
