/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.commons.codec.language;

import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;

/**
 * Tests {@link DoubleMetaphone}.
 * <p>
 * The test data was extracted from Stephen Woodbridge's
 * <a href="http://swoodbridge.com/DoubleMetaPhone/surnames.txt">php test program</a>.
 * 
 * @see http://swoodbridge.com/DoubleMetaPhone/surnames.txt
 * @version $Id: DoubleMetaphone2Test.java 800153 2009-08-02 22:45:30Z ggregory $
 */
public class DoubleMetaphone2Test extends TestCase {


    public static Test suite() {
        return new TestSuite(DoubleMetaphone2Test.class);
    }

    private DoubleMetaphone doubleMetaphone = null;

    /**
     * Construct a new test case.
     *
     * @param name The name of the test
     */
    public DoubleMetaphone2Test(String name) {
        super(name);
    }

    /**
     * Set up.
     */
    public void setUp() throws Exception {
        super.setUp();
        this.doubleMetaphone = new DoubleMetaphone();
    }

    /**
     * Tear Down.
     */
    public void tearDown() throws Exception {
        super.tearDown();
        this.doubleMetaphone = null;
    }

    /**
     * Test primary encoding.
     */
    public void testDoubleMetaphonePrimary() {
        String value = null;
        for (int i = 0; i < TEST_DATA.length; i++) {
            value = TEST_DATA[i][0];
            assertEquals("Test [" + i + "]=" + value, TEST_DATA[i][1], doubleMetaphone.doubleMetaphone(value, false));
        }
    }

    /**
     * Test alternative encoding.
     */
    public void testDoubleMetaphoneAlternate() {
        String value = null;
        for (int i = 0; i < TEST_DATA.length; i++) {
            value = TEST_DATA[i][0];
            assertEquals("Test [" + i + "]=" + value, TEST_DATA[i][2], doubleMetaphone.doubleMetaphone(value, true));
        }
    }

    /** Test values and their expected primary & alternate Double Metaphone encodings */
    private static final String[][] TEST_DATA = new String[][] {
        new String[] {"ALLERTON",    "ALRT", "ALRT"},
        new String[] {"Acton",       "AKTN", "AKTN"},
        new String[] {"Adams",       "ATMS", "ATMS"},
        new String[] {"Aggar",       "AKR",  "AKR"},
        new String[] {"Ahl",         "AL",   "AL"},
        new String[] {"Aiken",       "AKN",  "AKN"},
        new String[] {"Alan",        "ALN",  "ALN"},
        new String[] {"Alcock",      "ALKK", "ALKK"},
        new String[] {"Alden",       "ALTN", "ALTN"},
        new String[] {"Aldham",      "ALTM", "ALTM"},
        new String[] {"Allen",       "ALN",  "ALN"},
        new String[] {"Allerton",    "ALRT", "ALRT"},
        new String[] {"Alsop",       "ALSP", "ALSP"},
        new String[] {"Alwein",      "ALN",  "ALN"},
        new String[] {"Ambler",      "AMPL", "AMPL"},
        new String[] {"Andevill",    "ANTF", "ANTF"},
        new String[] {"Andrews",     "ANTR", "ANTR"},
        new String[] {"Andreyco",    "ANTR", "ANTR"},
        new String[] {"Andriesse",   "ANTR", "ANTR"},
        new String[] {"Angier",      "ANJ",  "ANJR"},
        new String[] {"Annabel",     "ANPL", "ANPL"},
        new String[] {"Anne",        "AN",   "AN"},
        new String[] {"Anstye",      "ANST", "ANST"},
        new String[] {"Appling",     "APLN", "APLN"},
        new String[] {"Apuke",       "APK",  "APK"},
        new String[] {"Arnold",      "ARNL", "ARNL"},
        new String[] {"Ashby",       "AXP",  "AXP"},
        new String[] {"Astwood",     "ASTT", "ASTT"},
        new String[] {"Atkinson",    "ATKN", "ATKN"},
        new String[] {"Audley",      "ATL",  "ATL"},
        new String[] {"Austin",      "ASTN", "ASTN"},
        new String[] {"Avenal",      "AFNL", "AFNL"},
        new String[] {"Ayer",        "AR",   "AR"},
        new String[] {"Ayot",        "AT",   "AT"},
        new String[] {"Babbitt",     "PPT",  "PPT"},
        new String[] {"Bachelor",    "PXLR", "PKLR"},
        new String[] {"Bachelour",   "PXLR", "PKLR"},
        new String[] {"Bailey",      "PL",   "PL"},
        new String[] {"Baivel",      "PFL",  "PFL"},
        new String[] {"Baker",       "PKR",  "PKR"},
        new String[] {"Baldwin",     "PLTN", "PLTN"},
        new String[] {"Balsley",     "PLSL", "PLSL"},
        new String[] {"Barber",      "PRPR", "PRPR"},
        new String[] {"Barker",      "PRKR", "PRKR"},
        new String[] {"Barlow",      "PRL",  "PRLF"},
        new String[] {"Barnard",     "PRNR", "PRNR"},
        new String[] {"Barnes",      "PRNS", "PRNS"},
        new String[] {"Barnsley",    "PRNS", "PRNS"},
        new String[] {"Barouxis",    "PRKS", "PRKS"},
        new String[] {"Bartlet",     "PRTL", "PRTL"},
        new String[] {"Basley",      "PSL",  "PSL"},
        new String[] {"Basset",      "PST",  "PST"},
        new String[] {"Bassett",     "PST",  "PST"},
        new String[] {"Batchlor",    "PXLR", "PXLR"},
        new String[] {"Bates",       "PTS",  "PTS"},
        new String[] {"Batson",      "PTSN", "PTSN"},
        new String[] {"Bayes",       "PS",   "PS"},
        new String[] {"Bayley",      "PL",   "PL"},
        new String[] {"Beale",       "PL",   "PL"},
        new String[] {"Beauchamp",   "PXMP", "PKMP"},
        new String[] {"Beauclerc",   "PKLR", "PKLR"},
        new String[] {"Beech",       "PK",   "PK"},
        new String[] {"Beers",       "PRS",  "PRS"},
        new String[] {"Beke",        "PK",   "PK"},
        new String[] {"Belcher",     "PLXR", "PLKR"},
        new String[] {"Benjamin",    "PNJM", "PNJM"},
        new String[] {"Benningham",  "PNNK", "PNNK"},
        new String[] {"Bereford",    "PRFR", "PRFR"},
        new String[] {"Bergen",      "PRJN", "PRKN"},
        new String[] {"Berkeley",    "PRKL", "PRKL"},
        new String[] {"Berry",       "PR",   "PR"},
        new String[] {"Besse",       "PS",   "PS"},
        new String[] {"Bessey",      "PS",   "PS"},
        new String[] {"Bessiles",    "PSLS", "PSLS"},
        new String[] {"Bigelow",     "PJL",  "PKLF"},
        new String[] {"Bigg",        "PK",   "PK"},
        new String[] {"Bigod",       "PKT",  "PKT"},
        new String[] {"Billings",    "PLNK", "PLNK"},
        new String[] {"Bimper",      "PMPR", "PMPR"},
        new String[] {"Binker",      "PNKR", "PNKR"},
        new String[] {"Birdsill",    "PRTS", "PRTS"},
        new String[] {"Bishop",      "PXP",  "PXP"},
        new String[] {"Black",       "PLK",  "PLK"},
        new String[] {"Blagge",      "PLK",  "PLK"},
        new String[] {"Blake",       "PLK",  "PLK"},
        new String[] {"Blanck",      "PLNK", "PLNK"},
        new String[] {"Bledsoe",     "PLTS", "PLTS"},
        new String[] {"Blennerhasset","PLNR", "PLNR"},
        new String[] {"Blessing",    "PLSN", "PLSN"},
        new String[] {"Blewett",     "PLT",  "PLT"},
        new String[] {"Bloctgoed",   "PLKT", "PLKT"},
        new String[] {"Bloetgoet",   "PLTK", "PLTK"},
        new String[] {"Bloodgood",   "PLTK", "PLTK"},
        new String[] {"Blossom",     "PLSM", "PLSM"},
        new String[] {"Blount",      "PLNT", "PLNT"},
        new String[] {"Bodine",      "PTN",  "PTN"},
        new String[] {"Bodman",      "PTMN", "PTMN"},
        new String[] {"BonCoeur",    "PNKR", "PNKR"},
        new String[] {"Bond",        "PNT",  "PNT"},
        new String[] {"Boscawen",    "PSKN", "PSKN"},
        new String[] {"Bosworth",    "PSR0", "PSRT"},
        new String[] {"Bouchier",    "PX",   "PKR"},
        new String[] {"Bowne",       "PN",   "PN"},
        new String[] {"Bradbury",    "PRTP", "PRTP"},
        new String[] {"Bradder",     "PRTR", "PRTR"},
        new String[] {"Bradford",    "PRTF", "PRTF"},
        new String[] {"Bradstreet",  "PRTS", "PRTS"},
        new String[] {"Braham",      "PRHM", "PRHM"},
        new String[] {"Brailsford",  "PRLS", "PRLS"},
        new String[] {"Brainard",    "PRNR", "PRNR"},
        new String[] {"Brandish",    "PRNT", "PRNT"},
        new String[] {"Braun",       "PRN",  "PRN"},
        new String[] {"Brecc",       "PRK",  "PRK"},
        new String[] {"Brent",       "PRNT", "PRNT"},
        new String[] {"Brenton",     "PRNT", "PRNT"},
        new String[] {"Briggs",      "PRKS", "PRKS"},
        new String[] {"Brigham",     "PRM",  "PRM"},
        new String[] {"Brobst",      "PRPS", "PRPS"},
        new String[] {"Brome",       "PRM",  "PRM"},
        new String[] {"Bronson",     "PRNS", "PRNS"},
        new String[] {"Brooks",      "PRKS", "PRKS"},
        new String[] {"Brouillard",  "PRLR", "PRLR"},
        new String[] {"Brown",       "PRN",  "PRN"},
        new String[] {"Browne",      "PRN",  "PRN"},
        new String[] {"Brownell",    "PRNL", "PRNL"},
        new String[] {"Bruley",      "PRL",  "PRL"},
        new String[] {"Bryant",      "PRNT", "PRNT"},
        new String[] {"Brzozowski",  "PRSS", "PRTS"},
        new String[] {"Buide",       "PT",   "PT"},
        new String[] {"Bulmer",      "PLMR", "PLMR"},
        new String[] {"Bunker",      "PNKR", "PNKR"},
        new String[] {"Burden",      "PRTN", "PRTN"},
        new String[] {"Burge",       "PRJ",  "PRK"},
        new String[] {"Burgoyne",    "PRKN", "PRKN"},
        new String[] {"Burke",       "PRK",  "PRK"},
        new String[] {"Burnett",     "PRNT", "PRNT"},
        new String[] {"Burpee",      "PRP",  "PRP"},
        new String[] {"Bursley",     "PRSL", "PRSL"},
        new String[] {"Burton",      "PRTN", "PRTN"},
        new String[] {"Bushnell",    "PXNL", "PXNL"},
        new String[] {"Buss",        "PS",   "PS"},
        new String[] {"Buswell",     "PSL",  "PSL"},
        new String[] {"Butler",      "PTLR", "PTLR"},
        new String[] {"Calkin",      "KLKN", "KLKN"},
        new String[] {"Canada",      "KNT",  "KNT"},
        new String[] {"Canmore",     "KNMR", "KNMR"},
        new String[] {"Canney",      "KN",   "KN"},
        new String[] {"Capet",       "KPT",  "KPT"},
        new String[] {"Card",        "KRT",  "KRT"},
        new String[] {"Carman",      "KRMN", "KRMN"},
        new String[] {"Carpenter",   "KRPN", "KRPN"},
        new String[] {"Cartwright",  "KRTR", "KRTR"},
        new String[] {"Casey",       "KS",   "KS"},
        new String[] {"Catterfield", "KTRF", "KTRF"},
        new String[] {"Ceeley",      "SL",   "SL"},
        new String[] {"Chambers",    "XMPR", "XMPR"},
        new String[] {"Champion",    "XMPN", "XMPN"},
        new String[] {"Chapman",     "XPMN", "XPMN"},
        new String[] {"Chase",       "XS",   "XS"},
        new String[] {"Cheney",      "XN",   "XN"},
        new String[] {"Chetwynd",    "XTNT", "XTNT"},
        new String[] {"Chevalier",   "XFL",  "XFLR"},
        new String[] {"Chillingsworth","XLNK", "XLNK"},
        new String[] {"Christie",    "KRST", "KRST"},
        new String[] {"Chubbuck",    "XPK",  "XPK"},
        new String[] {"Church",      "XRX",  "XRK"},
        new String[] {"Clark",       "KLRK", "KLRK"},
        new String[] {"Clarke",      "KLRK", "KLRK"},
        new String[] {"Cleare",      "KLR",  "KLR"},
        new String[] {"Clement",     "KLMN", "KLMN"},
        new String[] {"Clerke",      "KLRK", "KLRK"},
        new String[] {"Clibben",     "KLPN", "KLPN"},
        new String[] {"Clifford",    "KLFR", "KLFR"},
        new String[] {"Clivedon",    "KLFT", "KLFT"},
        new String[] {"Close",       "KLS",  "KLS"},
        new String[] {"Clothilde",   "KL0L", "KLTL"},
        new String[] {"Cobb",        "KP",   "KP"},
        new String[] {"Coburn",      "KPRN", "KPRN"},
        new String[] {"Coburne",     "KPRN", "KPRN"},
        new String[] {"Cocke",       "KK",   "KK"},
        new String[] {"Coffin",      "KFN",  "KFN"},
        new String[] {"Coffyn",      "KFN",  "KFN"},
        new String[] {"Colborne",    "KLPR", "KLPR"},
        new String[] {"Colby",       "KLP",  "KLP"},
        new String[] {"Cole",        "KL",   "KL"},
        new String[] {"Coleman",     "KLMN", "KLMN"},
        new String[] {"Collier",     "KL",   "KLR"},
        new String[] {"Compton",     "KMPT", "KMPT"},
        new String[] {"Cone",        "KN",   "KN"},
        new String[] {"Cook",        "KK",   "KK"},
        new String[] {"Cooke",       "KK",   "KK"},
        new String[] {"Cooper",      "KPR",  "KPR"},
        new String[] {"Copperthwaite","KPR0", "KPRT"},
        new String[] {"Corbet",      "KRPT", "KRPT"},
        new String[] {"Corell",      "KRL",  "KRL"},
        new String[] {"Corey",       "KR",   "KR"},
        new String[] {"Corlies",     "KRLS", "KRLS"},
        new String[] {"Corneliszen", "KRNL", "KRNL"},
        new String[] {"Cornelius",   "KRNL", "KRNL"},
        new String[] {"Cornwallis",  "KRNL", "KRNL"},
        new String[] {"Cosgrove",    "KSKR", "KSKR"},
        new String[] {"Count of Brionne","KNTF", "KNTF"},
        new String[] {"Covill",      "KFL",  "KFL"},
        new String[] {"Cowperthwaite","KPR0", "KPRT"},
        new String[] {"Cowperwaite", "KPRT", "KPRT"},
        new String[] {"Crane",       "KRN",  "KRN"},
        new String[] {"Creagmile",   "KRKM", "KRKM"},
        new String[] {"Crew",        "KR",   "KRF"},
        new String[] {"Crispin",     "KRSP", "KRSP"},
        new String[] {"Crocker",     "KRKR", "KRKR"},
        new String[] {"Crockett",    "KRKT", "KRKT"},
        new String[] {"Crosby",      "KRSP", "KRSP"},
        new String[] {"Crump",       "KRMP", "KRMP"},
        new String[] {"Cunningham",  "KNNK", "KNNK"},
        new String[] {"Curtis",      "KRTS", "KRTS"},
        new String[] {"Cutha",       "K0",   "KT"},
        new String[] {"Cutter",      "KTR",  "KTR"},
        new String[] {"D'Aubigny",   "TPN",  "TPKN"},
        new String[] {"DAVIS",       "TFS",  "TFS"},
        new String[] {"Dabinott",    "TPNT", "TPNT"},
        new String[] {"Dacre",       "TKR",  "TKR"},
        new String[] {"Daggett",     "TKT",  "TKT"},
        new String[] {"Danvers",     "TNFR", "TNFR"},
        new String[] {"Darcy",       "TRS",  "TRS"},
        new String[] {"Davis",       "TFS",  "TFS"},
        new String[] {"Dawn",        "TN",   "TN"},
        new String[] {"Dawson",      "TSN",  "TSN"},
        new String[] {"Day",         "T",    "T"},
        new String[] {"Daye",        "T",    "T"},
        new String[] {"DeGrenier",   "TKRN", "TKRN"},
        new String[] {"Dean",        "TN",   "TN"},
        new String[] {"Deekindaugh", "TKNT", "TKNT"},
        new String[] {"Dennis",      "TNS",  "TNS"},
        new String[] {"Denny",       "TN",   "TN"},
        new String[] {"Denton",      "TNTN", "TNTN"},
        new String[] {"Desborough",  "TSPR", "TSPR"},
        new String[] {"Despenser",   "TSPN", "TSPN"},
        new String[] {"Deverill",    "TFRL", "TFRL"},
        new String[] {"Devine",      "TFN",  "TFN"},
        new String[] {"Dexter",      "TKST", "TKST"},
        new String[] {"Dillaway",    "TL",   "TL"},
        new String[] {"Dimmick",     "TMK",  "TMK"},
        new String[] {"Dinan",       "TNN",  "TNN"},
        new String[] {"Dix",         "TKS",  "TKS"},
        new String[] {"Doggett",     "TKT",  "TKT"},
        new String[] {"Donahue",     "TNH",  "TNH"},
        new String[] {"Dorfman",     "TRFM", "TRFM"},
        new String[] {"Dorris",      "TRS",  "TRS"},
        new String[] {"Dow",         "T",    "TF"},
        new String[] {"Downey",      "TN",   "TN"},
        new String[] {"Downing",     "TNNK", "TNNK"},
        new String[] {"Dowsett",     "TST",  "TST"},
        new String[] {"Duck?",       "TK",   "TK"},
        new String[] {"Dudley",      "TTL",  "TTL"},
        new String[] {"Duffy",       "TF",   "TF"},
        new String[] {"Dunn",        "TN",   "TN"},
        new String[] {"Dunsterville","TNST", "TNST"},
        new String[] {"Durrant",     "TRNT", "TRNT"},
        new String[] {"Durrin",      "TRN",  "TRN"},
        new String[] {"Dustin",      "TSTN", "TSTN"},
        new String[] {"Duston",      "TSTN", "TSTN"},
        new String[] {"Eames",       "AMS",  "AMS"},
        new String[] {"Early",       "ARL",  "ARL"},
        new String[] {"Easty",       "AST",  "AST"},
        new String[] {"Ebbett",      "APT",  "APT"},
        new String[] {"Eberbach",    "APRP", "APRP"},
        new String[] {"Eberhard",    "APRR", "APRR"},
        new String[] {"Eddy",        "AT",   "AT"},
        new String[] {"Edenden",     "ATNT", "ATNT"},
        new String[] {"Edwards",     "ATRT", "ATRT"},
        new String[] {"Eglinton",    "AKLN", "ALNT"},
        new String[] {"Eliot",       "ALT",  "ALT"},
        new String[] {"Elizabeth",   "ALSP", "ALSP"},
        new String[] {"Ellis",       "ALS",  "ALS"},
        new String[] {"Ellison",     "ALSN", "ALSN"},
        new String[] {"Ellot",       "ALT",  "ALT"},
        new String[] {"Elny",        "ALN",  "ALN"},
        new String[] {"Elsner",      "ALSN", "ALSN"},
        new String[] {"Emerson",     "AMRS", "AMRS"},
        new String[] {"Empson",      "AMPS", "AMPS"},
        new String[] {"Est",         "AST",  "AST"},
        new String[] {"Estabrook",   "ASTP", "ASTP"},
        new String[] {"Estes",       "ASTS", "ASTS"},
        new String[] {"Estey",       "AST",  "AST"},
        new String[] {"Evans",       "AFNS", "AFNS"},
        new String[] {"Fallowell",   "FLL",  "FLL"},
        new String[] {"Farnsworth",  "FRNS", "FRNS"},
        new String[] {"Feake",       "FK",   "FK"},
        new String[] {"Feke",        "FK",   "FK"},
        new String[] {"Fellows",     "FLS",  "FLS"},
        new String[] {"Fettiplace",  "FTPL", "FTPL"},
        new String[] {"Finney",      "FN",   "FN"},
        new String[] {"Fischer",     "FXR",  "FSKR"},
        new String[] {"Fisher",      "FXR",  "FXR"},
        new String[] {"Fisk",        "FSK",  "FSK"},
        new String[] {"Fiske",       "FSK",  "FSK"},
        new String[] {"Fletcher",    "FLXR", "FLXR"},
        new String[] {"Folger",      "FLKR", "FLJR"},
        new String[] {"Foliot",      "FLT",  "FLT"},
        new String[] {"Folyot",      "FLT",  "FLT"},
        new String[] {"Fones",       "FNS",  "FNS"},
        new String[] {"Fordham",     "FRTM", "FRTM"},
        new String[] {"Forstner",    "FRST", "FRST"},
        new String[] {"Fosten",      "FSTN", "FSTN"},
        new String[] {"Foster",      "FSTR", "FSTR"},
        new String[] {"Foulke",      "FLK",  "FLK"},
        new String[] {"Fowler",      "FLR",  "FLR"},
        new String[] {"Foxwell",     "FKSL", "FKSL"},
        new String[] {"Fraley",      "FRL",  "FRL"},
        new String[] {"Franceys",    "FRNS", "FRNS"},
        new String[] {"Franke",      "FRNK", "FRNK"},
        new String[] {"Frascella",   "FRSL", "FRSL"},
        new String[] {"Frazer",      "FRSR", "FRSR"},
        new String[] {"Fredd",       "FRT",  "FRT"},
        new String[] {"Freeman",     "FRMN", "FRMN"},
        new String[] {"French",      "FRNX", "FRNK"},
        new String[] {"Freville",    "FRFL", "FRFL"},
        new String[] {"Frey",        "FR",   "FR"},
        new String[] {"Frick",       "FRK",  "FRK"},
        new String[] {"Frier",       "FR",   "FRR"},
        new String[] {"Froe",        "FR",   "FR"},
        new String[] {"Frorer",      "FRRR", "FRRR"},
        new String[] {"Frost",       "FRST", "FRST"},
        new String[] {"Frothingham", "FR0N", "FRTN"},
        new String[] {"Fry",         "FR",   "FR"},
        new String[] {"Gaffney",     "KFN",  "KFN"},
        new String[] {"Gage",        "KJ",   "KK"},
        new String[] {"Gallion",     "KLN",  "KLN"},
        new String[] {"Gallishan",   "KLXN", "KLXN"},
        new String[] {"Gamble",      "KMPL", "KMPL"},
        new String[] {"Garbrand",    "KRPR", "KRPR"},
        new String[] {"Gardner",     "KRTN", "KRTN"},
        new String[] {"Garrett",     "KRT",  "KRT"},
        new String[] {"Gassner",     "KSNR", "KSNR"},
        new String[] {"Gater",       "KTR",  "KTR"},
        new String[] {"Gaunt",       "KNT",  "KNT"},
        new String[] {"Gayer",       "KR",   "KR"},
        new String[] {"Gerken",      "KRKN", "JRKN"},
        new String[] {"Gerritsen",   "KRTS", "JRTS"},
        new String[] {"Gibbs",       "KPS",  "JPS"},
        new String[] {"Giffard",     "JFRT", "KFRT"},
        new String[] {"Gilbert",     "KLPR", "JLPR"},
        new String[] {"Gill",        "KL",   "JL"},
        new String[] {"Gilman",      "KLMN", "JLMN"},
        new String[] {"Glass",       "KLS",  "KLS"},
        new String[] {"Goddard\\Gifford","KTRT", "KTRT"},
        new String[] {"Godfrey",     "KTFR", "KTFR"},
        new String[] {"Godwin",      "KTN",  "KTN"},
        new String[] {"Goodale",     "KTL",  "KTL"},
        new String[] {"Goodnow",     "KTN",  "KTNF"},
        new String[] {"Gorham",      "KRM",  "KRM"},
        new String[] {"Goseline",    "KSLN", "KSLN"},
        new String[] {"Gott",        "KT",   "KT"},
        new String[] {"Gould",       "KLT",  "KLT"},
        new String[] {"Grafton",     "KRFT", "KRFT"},
        new String[] {"Grant",       "KRNT", "KRNT"},
        new String[] {"Gray",        "KR",   "KR"},
        new String[] {"Green",       "KRN",  "KRN"},
        new String[] {"Griffin",     "KRFN", "KRFN"},
        new String[] {"Grill",       "KRL",  "KRL"},
        new String[] {"Grim",        "KRM",  "KRM"},
        new String[] {"Grisgonelle", "KRSK", "KRSK"},
        new String[] {"Gross",       "KRS",  "KRS"},
        new String[] {"Guba",        "KP",   "KP"},
        new String[] {"Gybbes",      "KPS",  "JPS"},
        new String[] {"Haburne",     "HPRN", "HPRN"},
        new String[] {"Hackburne",   "HKPR", "HKPR"},
        new String[] {"Haddon?",     "HTN",  "HTN"},
        new String[] {"Haines",      "HNS",  "HNS"},
        new String[] {"Hale",        "HL",   "HL"},
        new String[] {"Hall",        "HL",   "HL"},
        new String[] {"Hallet",      "HLT",  "HLT"},
        new String[] {"Hallock",     "HLK",  "HLK"},
        new String[] {"Halstead",    "HLST", "HLST"},
        new String[] {"Hammond",     "HMNT", "HMNT"},
        new String[] {"Hance",       "HNS",  "HNS"},
        new String[] {"Handy",       "HNT",  "HNT"},
        new String[] {"Hanson",      "HNSN", "HNSN"},
        new String[] {"Harasek",     "HRSK", "HRSK"},
        new String[] {"Harcourt",    "HRKR", "HRKR"},
        new String[] {"Hardy",       "HRT",  "HRT"},
        new String[] {"Harlock",     "HRLK", "HRLK"},
        new String[] {"Harris",      "HRS",  "HRS"},
        new String[] {"Hartley",     "HRTL", "HRTL"},
        new String[] {"Harvey",      "HRF",  "HRF"},
        new String[] {"Harvie",      "HRF",  "HRF"},
        new String[] {"Harwood",     "HRT",  "HRT"},
        new String[] {"Hathaway",    "H0",   "HT"},
        new String[] {"Haukeness",   "HKNS", "HKNS"},
        new String[] {"Hawkes",      "HKS",  "HKS"},
        new String[] {"Hawkhurst",   "HKRS", "HKRS"},
        new String[] {"Hawkins",     "HKNS", "HKNS"},
        new String[] {"Hawley",      "HL",   "HL"},
        new String[] {"Heald",       "HLT",  "HLT"},
        new String[] {"Helsdon",     "HLST", "HLST"},
        new String[] {"Hemenway",    "HMN",  "HMN"},
        new String[] {"Hemmenway",   "HMN",  "HMN"},
        new String[] {"Henck",       "HNK",  "HNK"},
        new String[] {"Henderson",   "HNTR", "HNTR"},
        new String[] {"Hendricks",   "HNTR", "HNTR"},
        new String[] {"Hersey",      "HRS",  "HRS"},
        new String[] {"Hewes",       "HS",   "HS"},
        new String[] {"Heyman",      "HMN",  "HMN"},
        new String[] {"Hicks",       "HKS",  "HKS"},
        new String[] {"Hidden",      "HTN",  "HTN"},
        new String[] {"Higgs",       "HKS",  "HKS"},
        new String[] {"Hill",        "HL",   "HL"},
        new String[] {"Hills",       "HLS",  "HLS"},
        new String[] {"Hinckley",    "HNKL", "HNKL"},
        new String[] {"Hipwell",     "HPL",  "HPL"},
        new String[] {"Hobart",      "HPRT", "HPRT"},
        new String[] {"Hoben",       "HPN",  "HPN"},
        new String[] {"Hoffmann",    "HFMN", "HFMN"},
        new String[] {"Hogan",       "HKN",  "HKN"},
        new String[] {"Holmes",      "HLMS", "HLMS"},
        new String[] {"Hoo",         "H",    "H"},
        new String[] {"Hooker",      "HKR",  "HKR"},
        new String[] {"Hopcott",     "HPKT", "HPKT"},
        new String[] {"Hopkins",     "HPKN", "HPKN"},
        new String[] {"Hopkinson",   "HPKN", "HPKN"},
        new String[] {"Hornsey",     "HRNS", "HRNS"},
        new String[] {"Houckgeest",  "HKJS", "HKKS"},
        new String[] {"Hough",       "H",    "H"},
        new String[] {"Houstin",     "HSTN", "HSTN"},
        new String[] {"How",         "H",    "HF"},
        new String[] {"Howe",        "H",    "H"},
        new String[] {"Howland",     "HLNT", "HLNT"},
        new String[] {"Hubner",      "HPNR", "HPNR"},
        new String[] {"Hudnut",      "HTNT", "HTNT"},
        new String[] {"Hughes",      "HS",   "HS"},
        new String[] {"Hull",        "HL",   "HL"},
        new String[] {"Hulme",       "HLM",  "HLM"},
        new String[] {"Hume",        "HM",   "HM"},
        new String[] {"Hundertumark","HNTR", "HNTR"},
        new String[] {"Hundley",     "HNTL", "HNTL"},
        new String[] {"Hungerford",  "HNKR", "HNJR"},
        new String[] {"Hunt",        "HNT",  "HNT"},
        new String[] {"Hurst",       "HRST", "HRST"},
        new String[] {"Husbands",    "HSPN", "HSPN"},
        new String[] {"Hussey",      "HS",   "HS"},
        new String[] {"Husted",      "HSTT", "HSTT"},
        new String[] {"Hutchins",    "HXNS", "HXNS"},
        new String[] {"Hutchinson",  "HXNS", "HXNS"},
        new String[] {"Huttinger",   "HTNK", "HTNJ"},
        new String[] {"Huybertsen",  "HPRT", "HPRT"},
        new String[] {"Iddenden",    "ATNT", "ATNT"},
        new String[] {"Ingraham",    "ANKR", "ANKR"},
        new String[] {"Ives",        "AFS",  "AFS"},
        new String[] {"Jackson",     "JKSN", "AKSN"},
        new String[] {"Jacob",       "JKP",  "AKP"},
        new String[] {"Jans",        "JNS",  "ANS"},
        new String[] {"Jenkins",     "JNKN", "ANKN"},
        new String[] {"Jewett",      "JT",   "AT"},
        new String[] {"Jewitt",      "JT",   "AT"},
        new String[] {"Johnson",     "JNSN", "ANSN"},
        new String[] {"Jones",       "JNS",  "ANS"},
        new String[] {"Josephine",   "JSFN", "HSFN"},
        new String[] {"Judd",        "JT",   "AT"},
        new String[] {"June",        "JN",   "AN"},
        new String[] {"Kamarowska",  "KMRS", "KMRS"},
        new String[] {"Kay",         "K",    "K"},
        new String[] {"Kelley",      "KL",   "KL"},
        new String[] {"Kelly",       "KL",   "KL"},
        new String[] {"Keymber",     "KMPR", "KMPR"},
        new String[] {"Keynes",      "KNS",  "KNS"},
        new String[] {"Kilham",      "KLM",  "KLM"},
        new String[] {"Kim",         "KM",   "KM"},
        new String[] {"Kimball",     "KMPL", "KMPL"},
        new String[] {"King",        "KNK",  "KNK"},
        new String[] {"Kinsey",      "KNS",  "KNS"},
        new String[] {"Kirk",        "KRK",  "KRK"},
        new String[] {"Kirton",      "KRTN", "KRTN"},
        new String[] {"Kistler",     "KSTL", "KSTL"},
        new String[] {"Kitchen",     "KXN",  "KXN"},
        new String[] {"Kitson",      "KTSN", "KTSN"},
        new String[] {"Klett",       "KLT",  "KLT"},
        new String[] {"Kline",       "KLN",  "KLN"},
        new String[] {"Knapp",       "NP",   "NP"},
        new String[] {"Knight",      "NT",   "NT"},
        new String[] {"Knote",       "NT",   "NT"},
        new String[] {"Knott",       "NT",   "NT"},
        new String[] {"Knox",        "NKS",  "NKS"},
        new String[] {"Koeller",     "KLR",  "KLR"},
        new String[] {"La Pointe",   "LPNT", "LPNT"},
        new String[] {"LaPlante",    "LPLN", "LPLN"},
        new String[] {"Laimbeer",    "LMPR", "LMPR"},
        new String[] {"Lamb",        "LMP",  "LMP"},
        new String[] {"Lambertson",  "LMPR", "LMPR"},
        new String[] {"Lancto",      "LNKT", "LNKT"},
        new String[] {"Landry",      "LNTR", "LNTR"},
        new String[] {"Lane",        "LN",   "LN"},
        new String[] {"Langendyck",  "LNJN", "LNKN"},
        new String[] {"Langer",      "LNKR", "LNJR"},
        new String[] {"Langford",    "LNKF", "LNKF"},
        new String[] {"Lantersee",   "LNTR", "LNTR"},
        new String[] {"Laquer",      "LKR",  "LKR"},
        new String[] {"Larkin",      "LRKN", "LRKN"},
        new String[] {"Latham",      "LTM",  "LTM"},
        new String[] {"Lathrop",     "L0RP", "LTRP"},
        new String[] {"Lauter",      "LTR",  "LTR"},
        new String[] {"Lawrence",    "LRNS", "LRNS"},
        new String[] {"Leach",       "LK",   "LK"},
        new String[] {"Leager",      "LKR",  "LJR"},
        new String[] {"Learned",     "LRNT", "LRNT"},
        new String[] {"Leavitt",     "LFT",  "LFT"},
        new String[] {"Lee",         "L",    "L"},
        new String[] {"Leete",       "LT",   "LT"},
        new String[] {"Leggett",     "LKT",  "LKT"},
        new String[] {"Leland",      "LLNT", "LLNT"},
        new String[] {"Leonard",     "LNRT", "LNRT"},
        new String[] {"Lester",      "LSTR", "LSTR"},
        new String[] {"Lestrange",   "LSTR", "LSTR"},
        new String[] {"Lethem",      "L0M",  "LTM"},
        new String[] {"Levine",      "LFN",  "LFN"},
        new String[] {"Lewes",       "LS",   "LS"},
        new String[] {"Lewis",       "LS",   "LS"},
        new String[] {"Lincoln",     "LNKL", "LNKL"},
        new String[] {"Lindsey",     "LNTS", "LNTS"},
        new String[] {"Linher",      "LNR",  "LNR"},
        new String[] {"Lippet",      "LPT",  "LPT"},
        new String[] {"Lippincott",  "LPNK", "LPNK"},
        new String[] {"Lockwood",    "LKT",  "LKT"},
        new String[] {"Loines",      "LNS",  "LNS"},
        new String[] {"Lombard",     "LMPR", "LMPR"},
        new String[] {"Long",        "LNK",  "LNK"},
        new String[] {"Longespee",   "LNJS", "LNKS"},
        new String[] {"Look",        "LK",   "LK"},
        new String[] {"Lounsberry",  "LNSP", "LNSP"},
        new String[] {"Lounsbury",   "LNSP", "LNSP"},
        new String[] {"Louthe",      "L0",   "LT"},
        new String[] {"Loveyne",     "LFN",  "LFN"},
        new String[] {"Lowe",        "L",    "L"},
        new String[] {"Ludlam",      "LTLM", "LTLM"},
        new String[] {"Lumbard",     "LMPR", "LMPR"},
        new String[] {"Lund",        "LNT",  "LNT"},
        new String[] {"Luno",        "LN",   "LN"},
        new String[] {"Lutz",        "LTS",  "LTS"},
        new String[] {"Lydia",       "LT",   "LT"},
        new String[] {"Lynne",       "LN",   "LN"},
        new String[] {"Lyon",        "LN",   "LN"},
        new String[] {"MacAlpin",    "MKLP", "MKLP"},
        new String[] {"MacBricc",    "MKPR", "MKPR"},
        new String[] {"MacCrinan",   "MKRN", "MKRN"},
        new String[] {"MacKenneth",  "MKN0", "MKNT"},
        new String[] {"MacMael nam Bo","MKML", "MKML"},
        new String[] {"MacMurchada", "MKMR", "MKMR"},
        new String[] {"Macomber",    "MKMP", "MKMP"},
        new String[] {"Macy",        "MS",   "MS"},
        new String[] {"Magnus",      "MNS",  "MKNS"},
        new String[] {"Mahien",      "MHN",  "MHN"},
        new String[] {"Malmains",    "MLMN", "MLMN"},
        new String[] {"Malory",      "MLR",  "MLR"},
        new String[] {"Mancinelli",  "MNSN", "MNSN"},
        new String[] {"Mancini",     "MNSN", "MNSN"},
        new String[] {"Mann",        "MN",   "MN"},
        new String[] {"Manning",     "MNNK", "MNNK"},
        new String[] {"Manter",      "MNTR", "MNTR"},
        new String[] {"Marion",      "MRN",  "MRN"},
        new String[] {"Marley",      "MRL",  "MRL"},
        new String[] {"Marmion",     "MRMN", "MRMN"},
        new String[] {"Marquart",    "MRKR", "MRKR"},
        new String[] {"Marsh",       "MRX",  "MRX"},
        new String[] {"Marshal",     "MRXL", "MRXL"},
        new String[] {"Marshall",    "MRXL", "MRXL"},
        new String[] {"Martel",      "MRTL", "MRTL"},
        new String[] {"Martha",      "MR0",  "MRT"},
        new String[] {"Martin",      "MRTN", "MRTN"},
        new String[] {"Marturano",   "MRTR", "MRTR"},
        new String[] {"Marvin",      "MRFN", "MRFN"},
        new String[] {"Mary",        "MR",   "MR"},
        new String[] {"Mason",       "MSN",  "MSN"},
        new String[] {"Maxwell",     "MKSL", "MKSL"},
        new String[] {"Mayhew",      "MH",   "MHF"},
        new String[] {"McAllaster",  "MKLS", "MKLS"},
        new String[] {"McAllister",  "MKLS", "MKLS"},
        new String[] {"McConnell",   "MKNL", "MKNL"},
        new String[] {"McFarland",   "MKFR", "MKFR"},
        new String[] {"McIlroy",     "MSLR", "MSLR"},
        new String[] {"McNair",      "MKNR", "MKNR"},
        new String[] {"McNair-Landry","MKNR", "MKNR"},
        new String[] {"McRaven",     "MKRF", "MKRF"},
        new String[] {"Mead",        "MT",   "MT"},
        new String[] {"Meade",       "MT",   "MT"},
        new String[] {"Meck",        "MK",   "MK"},
        new String[] {"Melton",      "MLTN", "MLTN"},
        new String[] {"Mendenhall",  "MNTN", "MNTN"},
        new String[] {"Mering",      "MRNK", "MRNK"},
        new String[] {"Merrick",     "MRK",  "MRK"},
        new String[] {"Merry",       "MR",   "MR"},
        new String[] {"Mighill",     "ML",   "ML"},
        new String[] {"Miller",      "MLR",  "MLR"},
        new String[] {"Milton",      "MLTN", "MLTN"},
        new String[] {"Mohun",       "MHN",  "MHN"},
        new String[] {"Montague",    "MNTK", "MNTK"},
        new String[] {"Montboucher", "MNTP", "MNTP"},
        new String[] {"Moore",       "MR",   "MR"},
        new String[] {"Morrel",      "MRL",  "MRL"},
        new String[] {"Morrill",     "MRL",  "MRL"},
        new String[] {"Morris",      "MRS",  "MRS"},
        new String[] {"Morton",      "MRTN", "MRTN"},
        new String[] {"Moton",       "MTN",  "MTN"},
        new String[] {"Muir",        "MR",   "MR"},
        new String[] {"Mulferd",     "MLFR", "MLFR"},
        new String[] {"Mullins",     "MLNS", "MLNS"},
        new String[] {"Mulso",       "MLS",  "MLS"},
        new String[] {"Munger",      "MNKR", "MNJR"},
        new String[] {"Munt",        "MNT",  "MNT"},
        new String[] {"Murchad",     "MRXT", "MRKT"},
        new String[] {"Murdock",     "MRTK", "MRTK"},
        new String[] {"Murray",      "MR",   "MR"},
        new String[] {"Muskett",     "MSKT", "MSKT"},
        new String[] {"Myers",       "MRS",  "MRS"},
        new String[] {"Myrick",      "MRK",  "MRK"},
        new String[] {"NORRIS",      "NRS",  "NRS"},
        new String[] {"Nayle",       "NL",   "NL"},
        new String[] {"Newcomb",     "NKMP", "NKMP"},
        new String[] {"Newcomb(e)",  "NKMP", "NKMP"},
        new String[] {"Newkirk",     "NKRK", "NKRK"},
        new String[] {"Newton",      "NTN",  "NTN"},
        new String[] {"Niles",       "NLS",  "NLS"},
        new String[] {"Noble",       "NPL",  "NPL"},
        new String[] {"Noel",        "NL",   "NL"},
        new String[] {"Northend",    "NR0N", "NRTN"},
        new String[] {"Norton",      "NRTN", "NRTN"},
        new String[] {"Nutter",      "NTR",  "NTR"},
        new String[] {"Odding",      "ATNK", "ATNK"},
        new String[] {"Odenbaugh",   "ATNP", "ATNP"},
        new String[] {"Ogborn",      "AKPR", "AKPR"},
        new String[] {"Oppenheimer", "APNM", "APNM"},
        new String[] {"Otis",        "ATS",  "ATS"},
        new String[] {"Oviatt",      "AFT",  "AFT"},
        new String[] {"PRUST?",      "PRST", "PRST"},
        new String[] {"Paddock",     "PTK",  "PTK"},
        new String[] {"Page",        "PJ",   "PK"},
        new String[] {"Paine",       "PN",   "PN"},
        new String[] {"Paist",       "PST",  "PST"},
        new String[] {"Palmer",      "PLMR", "PLMR"},
        new String[] {"Park",        "PRK",  "PRK"},
        new String[] {"Parker",      "PRKR", "PRKR"},
        new String[] {"Parkhurst",   "PRKR", "PRKR"},
        new String[] {"Parrat",      "PRT",  "PRT"},
        new String[] {"Parsons",     "PRSN", "PRSN"},
        new String[] {"Partridge",   "PRTR", "PRTR"},
        new String[] {"Pashley",     "PXL",  "PXL"},
        new String[] {"Pasley",      "PSL",  "PSL"},
        new String[] {"Patrick",     "PTRK", "PTRK"},
        new String[] {"Pattee",      "PT",   "PT"},
        new String[] {"Patten",      "PTN",  "PTN"},
        new String[] {"Pawley",      "PL",   "PL"},
        new String[] {"Payne",       "PN",   "PN"},
        new String[] {"Peabody",     "PPT",  "PPT"},
        new String[] {"Peake",       "PK",   "PK"},
        new String[] {"Pearson",     "PRSN", "PRSN"},
        new String[] {"Peat",        "PT",   "PT"},
        new String[] {"Pedersen",    "PTRS", "PTRS"},
        new String[] {"Percy",       "PRS",  "PRS"},
        new String[] {"Perkins",     "PRKN", "PRKN"},
        new String[] {"Perrine",     "PRN",  "PRN"},
        new String[] {"Perry",       "PR",   "PR"},
        new String[] {"Peson",       "PSN",  "PSN"},
        new String[] {"Peterson",    "PTRS", "PTRS"},
        new String[] {"Peyton",      "PTN",  "PTN"},
        new String[] {"Phinney",     "FN",   "FN"},
        new String[] {"Pickard",     "PKRT", "PKRT"},
        new String[] {"Pierce",      "PRS",  "PRS"},
        new String[] {"Pierrepont",  "PRPN", "PRPN"},
        new String[] {"Pike",        "PK",   "PK"},
        new String[] {"Pinkham",     "PNKM", "PNKM"},
        new String[] {"Pitman",      "PTMN", "PTMN"},
        new String[] {"Pitt",        "PT",   "PT"},
        new String[] {"Pitts",       "PTS",  "PTS"},
        new String[] {"Plantagenet", "PLNT", "PLNT"},
        new String[] {"Platt",       "PLT",  "PLT"},
        new String[] {"Platts",      "PLTS", "PLTS"},
        new String[] {"Pleis",       "PLS",  "PLS"},
        new String[] {"Pleiss",      "PLS",  "PLS"},
        new String[] {"Plisko",      "PLSK", "PLSK"},
        new String[] {"Pliskovitch", "PLSK", "PLSK"},
        new String[] {"Plum",        "PLM",  "PLM"},
        new String[] {"Plume",       "PLM",  "PLM"},
        new String[] {"Poitou",      "PT",   "PT"},
        new String[] {"Pomeroy",     "PMR",  "PMR"},
        new String[] {"Poretiers",   "PRTR", "PRTR"},
        new String[] {"Pote",        "PT",   "PT"},
        new String[] {"Potter",      "PTR",  "PTR"},
        new String[] {"Potts",       "PTS",  "PTS"},
        new String[] {"Powell",      "PL",   "PL"},
        new String[] {"Pratt",       "PRT",  "PRT"},
        new String[] {"Presbury",    "PRSP", "PRSP"},
        new String[] {"Priest",      "PRST", "PRST"},
        new String[] {"Prindle",     "PRNT", "PRNT"},
        new String[] {"Prior",       "PRR",  "PRR"},
        new String[] {"Profumo",     "PRFM", "PRFM"},
        new String[] {"Purdy",       "PRT",  "PRT"},
        new String[] {"Purefoy",     "PRF",  "PRF"},
        new String[] {"Pury",        "PR",   "PR"},
        new String[] {"Quinter",     "KNTR", "KNTR"},
        new String[] {"Rachel",      "RXL",  "RKL"},
        new String[] {"Rand",        "RNT",  "RNT"},
        new String[] {"Rankin",      "RNKN", "RNKN"},
        new String[] {"Ravenscroft", "RFNS", "RFNS"},
        new String[] {"Raynsford",   "RNSF", "RNSF"},
        new String[] {"Reakirt",     "RKRT", "RKRT"},
        new String[] {"Reaves",      "RFS",  "RFS"},
        new String[] {"Reeves",      "RFS",  "RFS"},
        new String[] {"Reichert",    "RXRT", "RKRT"},
        new String[] {"Remmele",     "RML",  "RML"},
        new String[] {"Reynolds",    "RNLT", "RNLT"},
        new String[] {"Rhodes",      "RTS",  "RTS"},
        new String[] {"Richards",    "RXRT", "RKRT"},
        new String[] {"Richardson",  "RXRT", "RKRT"},
        new String[] {"Ring",        "RNK",  "RNK"},
        new String[] {"Roberts",     "RPRT", "RPRT"},
        new String[] {"Robertson",   "RPRT", "RPRT"},
        new String[] {"Robson",      "RPSN", "RPSN"},
        new String[] {"Rodie",       "RT",   "RT"},
        new String[] {"Rody",        "RT",   "RT"},
        new String[] {"Rogers",      "RKRS", "RJRS"},
        new String[] {"Ross",        "RS",   "RS"},
        new String[] {"Rosslevin",   "RSLF", "RSLF"},
        new String[] {"Rowland",     "RLNT", "RLNT"},
        new String[] {"Ruehl",       "RL",   "RL"},
        new String[] {"Russell",     "RSL",  "RSL"},
        new String[] {"Ruth",        "R0",   "RT"},
        new String[] {"Ryan",        "RN",   "RN"},
        new String[] {"Rysse",       "RS",   "RS"},
        new String[] {"Sadler",      "STLR", "STLR"},
        new String[] {"Salmon",      "SLMN", "SLMN"},
        new String[] {"Salter",      "SLTR", "SLTR"},
        new String[] {"Salvatore",   "SLFT", "SLFT"},
        new String[] {"Sanders",     "SNTR", "SNTR"},
        new String[] {"Sands",       "SNTS", "SNTS"},
        new String[] {"Sanford",     "SNFR", "SNFR"},
        new String[] {"Sanger",      "SNKR", "SNJR"},
        new String[] {"Sargent",     "SRJN", "SRKN"},
        new String[] {"Saunders",    "SNTR", "SNTR"},
        new String[] {"Schilling",   "XLNK", "XLNK"},
        new String[] {"Schlegel",    "XLKL", "SLKL"},
        new String[] {"Scott",       "SKT",  "SKT"},
        new String[] {"Sears",       "SRS",  "SRS"},
        new String[] {"Segersall",   "SJRS", "SKRS"},
        new String[] {"Senecal",     "SNKL", "SNKL"},
        new String[] {"Sergeaux",    "SRJ",  "SRK"},
        new String[] {"Severance",   "SFRN", "SFRN"},
        new String[] {"Sharp",       "XRP",  "XRP"},
        new String[] {"Sharpe",      "XRP",  "XRP"},
        new String[] {"Sharply",     "XRPL", "XRPL"},
        new String[] {"Shatswell",   "XTSL", "XTSL"},
        new String[] {"Shattack",    "XTK",  "XTK"},
        new String[] {"Shattock",    "XTK",  "XTK"},
        new String[] {"Shattuck",    "XTK",  "XTK"},
        new String[] {"Shaw",        "X",    "XF"},
        new String[] {"Sheldon",     "XLTN", "XLTN"},
        new String[] {"Sherman",     "XRMN", "XRMN"},
        new String[] {"Shinn",       "XN",   "XN"},
        new String[] {"Shirford",    "XRFR", "XRFR"},
        new String[] {"Shirley",     "XRL",  "XRL"},
        new String[] {"Shively",     "XFL",  "XFL"},
        new String[] {"Shoemaker",   "XMKR", "XMKR"},
        new String[] {"Short",       "XRT",  "XRT"},
        new String[] {"Shotwell",    "XTL",  "XTL"},
        new String[] {"Shute",       "XT",   "XT"},
        new String[] {"Sibley",      "SPL",  "SPL"},
        new String[] {"Silver",      "SLFR", "SLFR"},
        new String[] {"Simes",       "SMS",  "SMS"},
        new String[] {"Sinken",      "SNKN", "SNKN"},
        new String[] {"Sinn",        "SN",   "SN"},
        new String[] {"Skelton",     "SKLT", "SKLT"},
        new String[] {"Skiffe",      "SKF",  "SKF"},
        new String[] {"Skotkonung",  "SKTK", "SKTK"},
        new String[] {"Slade",       "SLT",  "XLT"},
        new String[] {"Slye",        "SL",   "XL"},
        new String[] {"Smedley",     "SMTL", "XMTL"},
        new String[] {"Smith",       "SM0",  "XMT"},
        new String[] {"Snow",        "SN",   "XNF"},
        new String[] {"Soole",       "SL",   "SL"},
        new String[] {"Soule",       "SL",   "SL"},
        new String[] {"Southworth",  "S0R0", "STRT"},
        new String[] {"Sowles",      "SLS",  "SLS"},
        new String[] {"Spalding",    "SPLT", "SPLT"},
        new String[] {"Spark",       "SPRK", "SPRK"},
        new String[] {"Spencer",     "SPNS", "SPNS"},
        new String[] {"Sperry",      "SPR",  "SPR"},
        new String[] {"Spofford",    "SPFR", "SPFR"},
        new String[] {"Spooner",     "SPNR", "SPNR"},
        new String[] {"Sprague",     "SPRK", "SPRK"},
        new String[] {"Springer",    "SPRN", "SPRN"},
        new String[] {"St. Clair",   "STKL", "STKL"},
        new String[] {"St. Claire",  "STKL", "STKL"},
        new String[] {"St. Leger",   "STLJ", "STLK"},
        new String[] {"St. Omer",    "STMR", "STMR"},
        new String[] {"Stafferton",  "STFR", "STFR"},
        new String[] {"Stafford",    "STFR", "STFR"},
        new String[] {"Stalham",     "STLM", "STLM"},
        new String[] {"Stanford",    "STNF", "STNF"},
        new String[] {"Stanton",     "STNT", "STNT"},
        new String[] {"Star",        "STR",  "STR"},
        new String[] {"Starbuck",    "STRP", "STRP"},
        new String[] {"Starkey",     "STRK", "STRK"},
        new String[] {"Starkweather","STRK", "STRK"},
        new String[] {"Stearns",     "STRN", "STRN"},
        new String[] {"Stebbins",    "STPN", "STPN"},
        new String[] {"Steele",      "STL",  "STL"},
        new String[] {"Stephenson",  "STFN", "STFN"},
        new String[] {"Stevens",     "STFN", "STFN"},
        new String[] {"Stoddard",    "STTR", "STTR"},
        new String[] {"Stodder",     "STTR", "STTR"},
        new String[] {"Stone",       "STN",  "STN"},
        new String[] {"Storey",      "STR",  "STR"},
        new String[] {"Storrada",    "STRT", "STRT"},
        new String[] {"Story",       "STR",  "STR"},
        new String[] {"Stoughton",   "STFT", "STFT"},
        new String[] {"Stout",       "STT",  "STT"},
        new String[] {"Stow",        "ST",   "STF"},
        new String[] {"Strong",      "STRN", "STRN"},
        new String[] {"Strutt",      "STRT", "STRT"},
        new String[] {"Stryker",     "STRK", "STRK"},
        new String[] {"Stuckeley",   "STKL", "STKL"},
        new String[] {"Sturges",     "STRJ", "STRK"},
        new String[] {"Sturgess",    "STRJ", "STRK"},
        new String[] {"Sturgis",     "STRJ", "STRK"},
        new String[] {"Suevain",     "SFN",  "SFN"},
        new String[] {"Sulyard",     "SLRT", "SLRT"},
        new String[] {"Sutton",      "STN",  "STN"},
        new String[] {"Swain",       "SN",   "XN"},
        new String[] {"Swayne",      "SN",   "XN"},
        new String[] {"Swayze",      "SS",   "XTS"},
        new String[] {"Swift",       "SFT",  "XFT"},
        new String[] {"Taber",       "TPR",  "TPR"},
        new String[] {"Talcott",     "TLKT", "TLKT"},
        new String[] {"Tarne",       "TRN",  "TRN"},
        new String[] {"Tatum",       "TTM",  "TTM"},
        new String[] {"Taverner",    "TFRN", "TFRN"},
        new String[] {"Taylor",      "TLR",  "TLR"},
        new String[] {"Tenney",      "TN",   "TN"},
        new String[] {"Thayer",      "0R",   "TR"},
        new String[] {"Thember",     "0MPR", "TMPR"},
        new String[] {"Thomas",      "TMS",  "TMS"},
        new String[] {"Thompson",    "TMPS", "TMPS"},
        new String[] {"Thorne",      "0RN",  "TRN"},
        new String[] {"Thornycraft", "0RNK", "TRNK"},
        new String[] {"Threlkeld",   "0RLK", "TRLK"},
        new String[] {"Throckmorton","0RKM", "TRKM"},
        new String[] {"Thwaits",     "0TS",  "TTS"},
        new String[] {"Tibbetts",    "TPTS", "TPTS"},
        new String[] {"Tidd",        "TT",   "TT"},
        new String[] {"Tierney",     "TRN",  "TRN"},
        new String[] {"Tilley",      "TL",   "TL"},
        new String[] {"Tillieres",   "TLRS", "TLRS"},
        new String[] {"Tilly",       "TL",   "TL"},
        new String[] {"Tisdale",     "TSTL", "TSTL"},
        new String[] {"Titus",       "TTS",  "TTS"},
        new String[] {"Tobey",       "TP",   "TP"},
        new String[] {"Tooker",      "TKR",  "TKR"},
        new String[] {"Towle",       "TL",   "TL"},
        new String[] {"Towne",       "TN",   "TN"},
        new String[] {"Townsend",    "TNSN", "TNSN"},
        new String[] {"Treadway",    "TRT",  "TRT"},
        new String[] {"Trelawney",   "TRLN", "TRLN"},
        new String[] {"Trinder",     "TRNT", "TRNT"},
        new String[] {"Tripp",       "TRP",  "TRP"},
        new String[] {"Trippe",      "TRP",  "TRP"},
        new String[] {"Trott",       "TRT",  "TRT"},
        new String[] {"True",        "TR",   "TR"},
        new String[] {"Trussebut",   "TRSP", "TRSP"},
        new String[] {"Tucker",      "TKR",  "TKR"},
        new String[] {"Turgeon",     "TRJN", "TRKN"},
        new String[] {"Turner",      "TRNR", "TRNR"},
        new String[] {"Tuttle",      "TTL",  "TTL"},
        new String[] {"Tyler",       "TLR",  "TLR"},
        new String[] {"Tylle",       "TL",   "TL"},
        new String[] {"Tyrrel",      "TRL",  "TRL"},
        new String[] {"Ua Tuathail", "AT0L", "ATTL"},
        new String[] {"Ulrich",      "ALRX", "ALRK"},
        new String[] {"Underhill",   "ANTR", "ANTR"},
        new String[] {"Underwood",   "ANTR", "ANTR"},
        new String[] {"Unknown",     "ANKN", "ANKN"},
        new String[] {"Valentine",   "FLNT", "FLNT"},
        new String[] {"Van Egmond",  "FNKM", "FNKM"},
        new String[] {"Van der Beek","FNTR", "FNTR"},
        new String[] {"Vaughan",     "FKN",  "FKN"},
        new String[] {"Vermenlen",   "FRMN", "FRMN"},
        new String[] {"Vincent",     "FNSN", "FNSN"},
        new String[] {"Volentine",   "FLNT", "FLNT"},
        new String[] {"Wagner",      "AKNR", "FKNR"},
        new String[] {"Waite",       "AT",   "FT"},
        new String[] {"Walker",      "ALKR", "FLKR"},
        new String[] {"Walter",      "ALTR", "FLTR"},
        new String[] {"Wandell",     "ANTL", "FNTL"},
        new String[] {"Wandesford",  "ANTS", "FNTS"},
        new String[] {"Warbleton",   "ARPL", "FRPL"},
        new String[] {"Ward",        "ART",  "FRT"},
        new String[] {"Warde",       "ART",  "FRT"},
        new String[] {"Ware",        "AR",   "FR"},
        new String[] {"Wareham",     "ARHM", "FRHM"},
        new String[] {"Warner",      "ARNR", "FRNR"},
        new String[] {"Warren",      "ARN",  "FRN"},
        new String[] {"Washburne",   "AXPR", "FXPR"},
        new String[] {"Waterbury",   "ATRP", "FTRP"},
        new String[] {"Watson",      "ATSN", "FTSN"},
        new String[] {"WatsonEllithorpe","ATSN", "FTSN"},
        new String[] {"Watts",       "ATS",  "FTS"},
        new String[] {"Wayne",       "AN",   "FN"},
        new String[] {"Webb",        "AP",   "FP"},
        new String[] {"Weber",       "APR",  "FPR"},
        new String[] {"Webster",     "APST", "FPST"},
        new String[] {"Weed",        "AT",   "FT"},
        new String[] {"Weeks",       "AKS",  "FKS"},
        new String[] {"Wells",       "ALS",  "FLS"},
        new String[] {"Wenzell",     "ANSL", "FNTS"},
        new String[] {"West",        "AST",  "FST"},
        new String[] {"Westbury",    "ASTP", "FSTP"},
        new String[] {"Whatlocke",   "ATLK", "ATLK"},
        new String[] {"Wheeler",     "ALR",  "ALR"},
        new String[] {"Whiston",     "ASTN", "ASTN"},
        new String[] {"White",       "AT",   "AT"},
        new String[] {"Whitman",     "ATMN", "ATMN"},
        new String[] {"Whiton",      "ATN",  "ATN"},
        new String[] {"Whitson",     "ATSN", "ATSN"},
        new String[] {"Wickes",      "AKS",  "FKS"},
        new String[] {"Wilbur",      "ALPR", "FLPR"},
        new String[] {"Wilcotes",    "ALKT", "FLKT"},
        new String[] {"Wilkinson",   "ALKN", "FLKN"},
        new String[] {"Willets",     "ALTS", "FLTS"},
        new String[] {"Willett",     "ALT",  "FLT"},
        new String[] {"Willey",      "AL",   "FL"},
        new String[] {"Williams",    "ALMS", "FLMS"},
        new String[] {"Williston",   "ALST", "FLST"},
        new String[] {"Wilson",      "ALSN", "FLSN"},
        new String[] {"Wimes",       "AMS",  "FMS"},
        new String[] {"Winch",       "ANX",  "FNK"},
        new String[] {"Winegar",     "ANKR", "FNKR"},
        new String[] {"Wing",        "ANK",  "FNK"},
        new String[] {"Winsley",     "ANSL", "FNSL"},
        new String[] {"Winslow",     "ANSL", "FNSL"},
        new String[] {"Winthrop",    "AN0R", "FNTR"},
        new String[] {"Wise",        "AS",   "FS"},
        new String[] {"Wood",        "AT",   "FT"},
        new String[] {"Woodbridge",  "ATPR", "FTPR"},
        new String[] {"Woodward",    "ATRT", "FTRT"},
        new String[] {"Wooley",      "AL",   "FL"},
        new String[] {"Woolley",     "AL",   "FL"},
        new String[] {"Worth",       "AR0",  "FRT"},
        new String[] {"Worthen",     "AR0N", "FRTN"},
        new String[] {"Worthley",    "AR0L", "FRTL"},
        new String[] {"Wright",      "RT",   "RT"},
        new String[] {"Wyer",        "AR",   "FR"},
        new String[] {"Wyere",       "AR",   "FR"},
        new String[] {"Wynkoop",     "ANKP", "FNKP"},
        new String[] {"Yarnall",     "ARNL", "ARNL"},
        new String[] {"Yeoman",      "AMN",  "AMN"},
        new String[] {"Yorke",       "ARK",  "ARK"},
        new String[] {"Young",       "ANK",  "ANK"},
        new String[] {"ab Wennonwen","APNN", "APNN"},
        new String[] {"ap Llewellyn","APLL", "APLL"},
        new String[] {"ap Lorwerth", "APLR", "APLR"},
        new String[] {"d'Angouleme", "TNKL", "TNKL"},
        new String[] {"de Audeham",  "TTHM", "TTHM"},
        new String[] {"de Bavant",   "TPFN", "TPFN"},
        new String[] {"de Beauchamp","TPXM", "TPKM"},
        new String[] {"de Beaumont", "TPMN", "TPMN"},
        new String[] {"de Bolbec",   "TPLP", "TPLP"},
        new String[] {"de Braiose",  "TPRS", "TPRS"},
        new String[] {"de Braose",   "TPRS", "TPRS"},
        new String[] {"de Briwere",  "TPRR", "TPRR"},
        new String[] {"de Cantelou", "TKNT", "TKNT"},
        new String[] {"de Cherelton","TXRL", "TKRL"},
        new String[] {"de Cherleton","TXRL", "TKRL"},
        new String[] {"de Clare",    "TKLR", "TKLR"},
        new String[] {"de Claremont","TKLR", "TKLR"},
        new String[] {"de Clifford", "TKLF", "TKLF"},
        new String[] {"de Colville", "TKLF", "TKLF"},
        new String[] {"de Courtenay","TKRT", "TKRT"},
        new String[] {"de Fauconberg","TFKN", "TFKN"},
        new String[] {"de Forest",   "TFRS", "TFRS"},
        new String[] {"de Gai",      "TK",   "TK"},
        new String[] {"de Grey",     "TKR",  "TKR"},
        new String[] {"de Guernons", "TKRN", "TKRN"},
        new String[] {"de Haia",     "T",    "T"},
        new String[] {"de Harcourt", "TRKR", "TRKR"},
        new String[] {"de Hastings", "TSTN", "TSTN"},
        new String[] {"de Hoke",     "TK",   "TK"},
        new String[] {"de Hooch",    "TK",   "TK"},
        new String[] {"de Hugelville","TJLF", "TKLF"},
        new String[] {"de Huntingdon","TNTN", "TNTN"},
        new String[] {"de Insula",   "TNSL", "TNSL"},
        new String[] {"de Keynes",   "TKNS", "TKNS"},
        new String[] {"de Lacy",     "TLS",  "TLS"},
        new String[] {"de Lexington","TLKS", "TLKS"},
        new String[] {"de Lusignan", "TLSN", "TLSK"},
        new String[] {"de Manvers",  "TMNF", "TMNF"},
        new String[] {"de Montagu",  "TMNT", "TMNT"},
        new String[] {"de Montault", "TMNT", "TMNT"},
        new String[] {"de Montfort", "TMNT", "TMNT"},
        new String[] {"de Mortimer", "TMRT", "TMRT"},
        new String[] {"de Morville", "TMRF", "TMRF"},
        new String[] {"de Morvois",  "TMRF", "TMRF"},
        new String[] {"de Neufmarche","TNFM", "TNFM"},
        new String[] {"de Odingsells","TTNK", "TTNK"},
        new String[] {"de Odyngsells","TTNK", "TTNK"},
        new String[] {"de Percy",    "TPRS", "TPRS"},
        new String[] {"de Pierrepont","TPRP", "TPRP"},
        new String[] {"de Plessetis","TPLS", "TPLS"},
        new String[] {"de Porhoet",  "TPRT", "TPRT"},
        new String[] {"de Prouz",    "TPRS", "TPRS"},
        new String[] {"de Quincy",   "TKNS", "TKNS"},
        new String[] {"de Ripellis", "TRPL", "TRPL"},
        new String[] {"de Ros",      "TRS",  "TRS"},
        new String[] {"de Salisbury","TSLS", "TSLS"},
        new String[] {"de Sanford",  "TSNF", "TSNF"},
        new String[] {"de Somery",   "TSMR", "TSMR"},
        new String[] {"de St. Hilary","TSTL", "TSTL"},
        new String[] {"de St. Liz",  "TSTL", "TSTL"},
        new String[] {"de Sutton",   "TSTN", "TSTN"},
        new String[] {"de Toeni",    "TTN",  "TTN"},
        new String[] {"de Tony",     "TTN",  "TTN"},
        new String[] {"de Umfreville","TMFR", "TMFR"},
        new String[] {"de Valognes", "TFLN", "TFLK"},
        new String[] {"de Vaux",     "TF",   "TF"},
        new String[] {"de Vere",     "TFR",  "TFR"},
        new String[] {"de Vermandois","TFRM", "TFRM"},
        new String[] {"de Vernon",   "TFRN", "TFRN"},
        new String[] {"de Vexin",    "TFKS", "TFKS"},
        new String[] {"de Vitre",    "TFTR", "TFTR"},
        new String[] {"de Wandesford","TNTS", "TNTS"},
        new String[] {"de Warenne",  "TRN",  "TRN"},
        new String[] {"de Westbury", "TSTP", "TSTP"},
        new String[] {"di Saluzzo",  "TSLS", "TSLT"},
        new String[] {"fitz Alan",   "FTSL", "FTSL"},
        new String[] {"fitz Geoffrey","FTSJ", "FTSK"},
        new String[] {"fitz Herbert","FTSR", "FTSR"},
        new String[] {"fitz John",   "FTSJ", "FTSJ"},
        new String[] {"fitz Patrick","FTSP", "FTSP"},
        new String[] {"fitz Payn",   "FTSP", "FTSP"},
        new String[] {"fitz Piers",  "FTSP", "FTSP"},
        new String[] {"fitz Randolph","FTSR", "FTSR"},
        new String[] {"fitz Richard","FTSR", "FTSR"},
        new String[] {"fitz Robert", "FTSR", "FTSR"},
        new String[] {"fitz Roy",    "FTSR", "FTSR"},
        new String[] {"fitz Scrob",  "FTSS", "FTSS"},
        new String[] {"fitz Walter", "FTSL", "FTSL"},
        new String[] {"fitz Warin",  "FTSR", "FTSR"},
        new String[] {"fitz Williams","FTSL", "FTSL"},
        new String[] {"la Zouche",   "LSX",  "LSK"},
        new String[] {"le Botiller", "LPTL", "LPTL"},
        new String[] {"le Despenser","LTSP", "LTSP"},
        new String[] {"le deSpencer","LTSP", "LTSP"},
        new String[] {"of Allendale","AFLN", "AFLN"},
        new String[] {"of Angouleme","AFNK", "AFNK"},
        new String[] {"of Anjou",    "AFNJ", "AFNJ"},
        new String[] {"of Aquitaine","AFKT", "AFKT"},
        new String[] {"of Aumale",   "AFML", "AFML"},
        new String[] {"of Bavaria",  "AFPF", "AFPF"},
        new String[] {"of Boulogne", "AFPL", "AFPL"},
        new String[] {"of Brittany", "AFPR", "AFPR"},
        new String[] {"of Brittary", "AFPR", "AFPR"},
        new String[] {"of Castile",  "AFKS", "AFKS"},
        new String[] {"of Chester",  "AFXS", "AFKS"},
        new String[] {"of Clermont", "AFKL", "AFKL"},
        new String[] {"of Cologne",  "AFKL", "AFKL"},
        new String[] {"of Dinan",    "AFTN", "AFTN"},
        new String[] {"of Dunbar",   "AFTN", "AFTN"},
        new String[] {"of England",  "AFNK", "AFNK"},
        new String[] {"of Essex",    "AFSK", "AFSK"},
        new String[] {"of Falaise",  "AFFL", "AFFL"},
        new String[] {"of Flanders", "AFFL", "AFFL"},
        new String[] {"of Galloway", "AFKL", "AFKL"},
        new String[] {"of Germany",  "AFKR", "AFJR"},
        new String[] {"of Gloucester","AFKL", "AFKL"},
        new String[] {"of Heristal", "AFRS", "AFRS"},
        new String[] {"of Hungary",  "AFNK", "AFNK"},
        new String[] {"of Huntington","AFNT", "AFNT"},
        new String[] {"of Kiev",     "AFKF", "AFKF"},
        new String[] {"of Kuno",     "AFKN", "AFKN"},
        new String[] {"of Landen",   "AFLN", "AFLN"},
        new String[] {"of Laon",     "AFLN", "AFLN"},
        new String[] {"of Leinster", "AFLN", "AFLN"},
        new String[] {"of Lens",     "AFLN", "AFLN"},
        new String[] {"of Lorraine", "AFLR", "AFLR"},
        new String[] {"of Louvain",  "AFLF", "AFLF"},
        new String[] {"of Mercia",   "AFMR", "AFMR"},
        new String[] {"of Metz",     "AFMT", "AFMT"},
        new String[] {"of Meulan",   "AFML", "AFML"},
        new String[] {"of Nass",     "AFNS", "AFNS"},
        new String[] {"of Normandy", "AFNR", "AFNR"},
        new String[] {"of Ohningen", "AFNN", "AFNN"},
        new String[] {"of Orleans",  "AFRL", "AFRL"},
        new String[] {"of Poitou",   "AFPT", "AFPT"},
        new String[] {"of Polotzk",  "AFPL", "AFPL"},
        new String[] {"of Provence", "AFPR", "AFPR"},
        new String[] {"of Ringelheim","AFRN", "AFRN"},
        new String[] {"of Salisbury","AFSL", "AFSL"},
        new String[] {"of Saxony",   "AFSK", "AFSK"},
        new String[] {"of Scotland", "AFSK", "AFSK"},
        new String[] {"of Senlis",   "AFSN", "AFSN"},
        new String[] {"of Stafford", "AFST", "AFST"},
        new String[] {"of Swabia",   "AFSP", "AFSP"},
        new String[] {"of Tongres",  "AFTN", "AFTN"},
        new String[] {"of the Tributes","AF0T", "AFTT"},
        new String[] {"unknown",     "ANKN", "ANKN"},
        new String[] {"van der Gouda","FNTR", "FNTR"},
        new String[] {"von Adenbaugh","FNTN", "FNTN"},
        new String[] {"ARCHITure",   "ARKT", "ARKT"},
        new String[] {"Arnoff",      "ARNF", "ARNF"},
        new String[] {"Arnow",       "ARN",  "ARNF"},
        new String[] {"DANGER",      "TNJR", "TNKR"},
        new String[] {"Jankelowicz", "JNKL", "ANKL"},
        new String[] {"MANGER",      "MNJR", "MNKR"},
        new String[] {"McClellan",   "MKLL", "MKLL"},
        new String[] {"McHugh",      "MK",   "MK"},
        new String[] {"McLaughlin",  "MKLF", "MKLF"},
        new String[] {"ORCHEStra",   "ARKS", "ARKS"},
        new String[] {"ORCHID",      "ARKT", "ARKT"},
        new String[] {"Pierce",      "PRS",  "PRS"},
        new String[] {"RANGER",      "RNJR", "RNKR"},
        new String[] {"Schlesinger", "XLSN", "SLSN"},
        new String[] {"Uomo",        "AM",   "AM"},
        new String[] {"Vasserman",   "FSRM", "FSRM"},
        new String[] {"Wasserman",   "ASRM", "FSRM"},
        new String[] {"Womo",        "AM",   "FM"},
        new String[] {"Yankelovich", "ANKL", "ANKL"},
        new String[] {"accede",      "AKST", "AKST"},
        new String[] {"accident",    "AKST", "AKST"},
        new String[] {"adelsheim",   "ATLS", "ATLS"},
        new String[] {"aged",        "AJT",  "AKT"},
        new String[] {"ageless",     "AJLS", "AKLS"},
        new String[] {"agency",      "AJNS", "AKNS"},
        new String[] {"aghast",      "AKST", "AKST"},
        new String[] {"agio",        "AJ",   "AK"},
        new String[] {"agrimony",    "AKRM", "AKRM"},
        new String[] {"album",       "ALPM", "ALPM"},
        new String[] {"alcmene",     "ALKM", "ALKM"},
        new String[] {"alehouse",    "ALHS", "ALHS"},
        new String[] {"antique",     "ANTK", "ANTK"},
        new String[] {"artois",      "ART",  "ARTS"},
        new String[] {"automation",  "ATMX", "ATMX"},
        new String[] {"bacchus",     "PKS",  "PKS"},
        new String[] {"bacci",       "PX",   "PX"},
        new String[] {"bajador",     "PJTR", "PHTR"},
        new String[] {"bellocchio",  "PLX",  "PLX"},
        new String[] {"bertucci",    "PRTX", "PRTX"},
        new String[] {"biaggi",      "PJ",   "PK"},
        new String[] {"bough",       "P",    "P"},
        new String[] {"breaux",      "PR",   "PR"},
        new String[] {"broughton",   "PRTN", "PRTN"},
        new String[] {"cabrillo",    "KPRL", "KPR"},
        new String[] {"caesar",      "SSR",  "SSR"},
        new String[] {"cagney",      "KKN",  "KKN"},
        new String[] {"campbell",    "KMPL", "KMPL"},
        new String[] {"carlisle",    "KRLL", "KRLL"},
        new String[] {"carlysle",    "KRLL", "KRLL"},
        new String[] {"chemistry",   "KMST", "KMST"},
        new String[] {"chianti",     "KNT",  "KNT"},
        new String[] {"chorus",      "KRS",  "KRS"},
        new String[] {"cough",       "KF",   "KF"},
        new String[] {"czerny",      "SRN",  "XRN"},
        new String[] {"deffenbacher","TFNP", "TFNP"},
        new String[] {"dumb",        "TM",   "TM"},
        new String[] {"edgar",       "ATKR", "ATKR"},
        new String[] {"edge",        "AJ",   "AJ"},
        new String[] {"filipowicz",  "FLPT", "FLPF"},
        new String[] {"focaccia",    "FKX",  "FKX"},
        new String[] {"gallegos",    "KLKS", "KKS"},
        new String[] {"gambrelli",   "KMPR", "KMPR"},
        new String[] {"geithain",    "K0N",  "JTN"},
        new String[] {"ghiradelli",  "JRTL", "JRTL"},
        new String[] {"ghislane",    "JLN",  "JLN"},
        new String[] {"gough",       "KF",   "KF"},
        new String[] {"hartheim",    "HR0M", "HRTM"},
        new String[] {"heimsheim",   "HMSM", "HMSM"},
        new String[] {"hochmeier",   "HKMR", "HKMR"},
        new String[] {"hugh",        "H",    "H"},
        new String[] {"hunger",      "HNKR", "HNJR"},
        new String[] {"hungry",      "HNKR", "HNKR"},
        new String[] {"island",      "ALNT", "ALNT"},
        new String[] {"isle",        "AL",   "AL"},
        new String[] {"jose",        "HS",   "HS"},
        new String[] {"laugh",       "LF",   "LF"},
        new String[] {"mac caffrey", "MKFR", "MKFR"},
        new String[] {"mac gregor",  "MKRK", "MKRK"},
        new String[] {"pegnitz",     "PNTS", "PKNT"},
        new String[] {"piskowitz",   "PSKT", "PSKF"},
        new String[] {"queen",       "KN",   "KN"},
        new String[] {"raspberry",   "RSPR", "RSPR"},
        new String[] {"resnais",     "RSN",  "RSNS"},
        new String[] {"rogier",      "RJ",   "RJR"},
        new String[] {"rough",       "RF",   "RF"},
        new String[] {"san jacinto", "SNHS", "SNHS"},
        new String[] {"schenker",    "XNKR", "SKNK"},
        new String[] {"schermerhorn","XRMR", "SKRM"},
        new String[] {"schmidt",     "XMT",  "SMT"},
        new String[] {"schneider",   "XNTR", "SNTR"},
        new String[] {"school",      "SKL",  "SKL"},
        new String[] {"schooner",    "SKNR", "SKNR"},
        new String[] {"schrozberg",  "XRSP", "SRSP"},
        new String[] {"schulman",    "XLMN", "XLMN"},
        new String[] {"schwabach",   "XPK",  "XFPK"},
        new String[] {"schwarzach",  "XRSK", "XFRT"},
        new String[] {"smith",       "SM0",  "XMT"},
        new String[] {"snider",      "SNTR", "XNTR"},
        new String[] {"succeed",     "SKST", "SKST"},
        new String[] {"sugarcane",   "XKRK", "SKRK"},
        new String[] {"svobodka",    "SFPT", "SFPT"},
        new String[] {"tagliaro",    "TKLR", "TLR"},
        new String[] {"thames",      "TMS",  "TMS"},
        new String[] {"theilheim",   "0LM",  "TLM"},
        new String[] {"thomas",      "TMS",  "TMS"},
        new String[] {"thumb",       "0M",   "TM"},
        new String[] {"tichner",     "TXNR", "TKNR"},
        new String[] {"tough",       "TF",   "TF"},
        new String[] {"umbrella",    "AMPR", "AMPR"},
        new String[] {"vilshofen",   "FLXF", "FLXF"},
        new String[] {"von schuller","FNXL", "FNXL"},
        new String[] {"wachtler",    "AKTL", "FKTL"},
        new String[] {"wechsler",    "AKSL", "FKSL"},
        new String[] {"weikersheim", "AKRS", "FKRS"},
        new String[] {"zhao",        "J",    "J"}
    };
}
