/***************************************************************************
 $RCSfile: cardcommander.cpp,v $
 -------------------
 cvs         : $Id: cardcommander.cpp,v 1.6 2003/05/08 11:01:22 aquamaniac Exp $
 begin       : Fri Dec 13 2002
 copyright   : (C) 2002 by Martin Preuss
 email       : martin@libchipcard.de

 ****************************************************************************
 * This program is free software; you can redistribute it and/or modify     *
 * it under the terms of the GNU General Public License as published by     *
 * the Free Software Foundation; either version 2 of the License, or        *
 * (at your option) any later version.                                      *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA *
 ****************************************************************************/


#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

/* Internationalization */
#ifdef HAVE_GETTEXT_ENVIRONMENT
# include <libintl.h>
# include <locale.h>
# define I18N(m) gettext(m)
#else
# define I18N(m) m
#endif
#define I18NT(m) m


#include <stdio.h>
#include <errno.h>
#include <chipcard.h>
#include <chameleon/chameleon.h>
#include <cctype>


#define k_PRG "cardcommander"
#define k_PRG_VERSION_INFO \
    "cardcommander v0.2  (part of libchipcard v"k_CHIPCARD_VERSION_STRING")\n"\
    "(c) 2003 Martin Preuss<martin@libchipcard.de>\n" \
    "This program is free software licensed under GPL.\n"\
    "See COPYING for details.\n"


void usage(string name) {
  fprintf(stderr,
	  "CardCommander - A command line too to manipulate a chip card.\n"
	  "(c) 2003 Martin Preuss<martin@libchipcard.de>\n"
	  "This program is free software; you can redistribute it and/or\n"
	  "modify it under the terms of the GNU General Public\n"
	  "License as published by the Free Software Foundation; either\n"
	  "version 2.1 of the License, or (at your option) any later version.\n"
	  "\n"
	  "Usage:\n"
	  "%s [-C CONFIGFILE]\n"
	  " OPTIONS\n"
	  " -C CONFIGFILE - which configuration file to use (defaults to\n"
	  "                 "CHIPCARDC_CFGFILE")\n"
	  ,
	  name.c_str());
}


struct s_args {
  string configFile;    // -C
  list<string> params;
};


int checkArgs(s_args &args, int argc, char **argv) {
  int i;
  string tmp;

  i=1;
  args.configFile=CHIPCARDC_CFGFILE;

  while (i<argc){
    tmp=argv[i];
    if (tmp=="-C") {
      i++;
      if (i>=argc)
	return 1;
      args.configFile=argv[i];
    }
    else if (tmp=="-h" || tmp=="--help") {
      usage(argv[0]);
      return -1;
    }
    else if (tmp=="-V" || tmp=="--version") {
      fprintf(stdout,k_PRG_VERSION_INFO);
      return -1;
    }
    else
      // otherwise add param
      args.params.push_back(tmp);
    i++;
  } // while
  // that's it
  return 0;
}


string dumpString(const string &s) {
  unsigned int i;
  unsigned int j;
  unsigned int pos;
  string r;

  pos=0;
  fprintf(stderr,"String size is %d:\n",s.length());
  while(pos<s.length()) {
    r+=CTMisc::num2string((unsigned int)pos, "%04x");
    r+=" ";
    j=pos+16;
    if (j>=s.length())
      j=s.length();

    // show hex dump
    for (i=pos; i<j; i++) {
      r+=CTMisc::num2string((unsigned char)(s[i]),"%02x ");
    }
    if (j-pos<16)
      for (i=0; i<16-(j-pos); i++)
        r+="   ";
    // show text
    for (i=pos; i<j; i++) {
      if (s[i]<32)
	r+=".";
      else
	r+=s[i];
    }
    r+="\n";
    pos+=16;
  }
  return r;
}


int execCommand(s_args args,
		CTPointer<CTCardTrader> trader,
		CTPointer<CTCard> &card,
		const string &cmd) {
  unsigned int i;
  string cm;
  CTError err;
  static string lastAPDU;
  CTCard *cp;

  // skip leading blanks
  i=0;
  while(i<cmd.length()) {
    if (!isspace(cmd[i]))
      break;
    i++;
  } //while

  // now read command
  while(i<cmd.length()) {
    if (isspace(cmd[i]))
      break;
    cm+=cmd[i];
    i++;
  } //while

  // ok, we have the command now, check it
  if (CTMisc::compare(cm,"open")) {
    // open card

    if (card.isValid()) {
      fprintf(stderr, I18N("Card is already open, try \"close\" first.\n"));
      return 3;
    }

    fprintf(stdout,I18N("Waiting for a card to be inserted...\n"));
    err=trader.ref().getNext(cp, 20);
    if (!err.isOk()) {
      fprintf(stderr,
	      I18N("No card inserted, timeout (%s)\n"),
	      err.errorString().c_str());
      return 3;
    }
    card=cp;

    err=card.ref().openCard();
    if (!err.isOk()) {
      fprintf(stderr,I18N("Error: %s\n"),err.errorString().c_str());
      return 3;
    }
    fprintf(stdout,
	    I18N("Card is open in \"%s\" , info follows:\n"),
	    card.ref().readerDescription().name);
    fprintf(stdout,
	    I18N(" Card Type: "));
    if (card.ref().isProcessorCard())
      fprintf(stdout,I18N("Processor card\n"));
    else
      fprintf(stdout,I18N("Memory card (%d bytes)\n"),
	      card.ref().memorySize());
    fprintf(stdout,
	    I18N(" ATR:       %s\n"),
	    CTMisc::bin2hex(card.ref().atr()).c_str());
  }
  else if (CTMisc::compare(cm,"close")) {
    if (!card.isValid()) {
      fprintf(stderr, I18N("Card is not open, try \"open\" first.\n"));
      return 3;
    }

    fprintf(stdout,I18N("Closing card.\n"));
    err=card.ref().closeCard();
    if (!err.isOk()) {
      fprintf(stderr,I18N("Error: %s\n"),err.errorString().c_str());
      return 3;
    }
    card=0;
  }
  else if (CTMisc::compare(cm,"apdu")) {
    string apdu;
    string response;

    if (!card.isValid()) {
      fprintf(stderr, I18N("Card is not open, try \"open\" first.\n"));
      return 3;
    }
    apdu=CTMisc::hex2bin(cmd.substr(i));
    if (apdu.length()<4) {
      fprintf(stderr,I18N("An APDU needs at least 4 bytes.\n"));
      return 2;
    }
    lastAPDU=apdu;
    err=card.ref().sendAPDU(apdu, response);
    if (!err.isOk()) {
      fprintf(stderr,"Error: %s\n",err.errorString().c_str());
      return 3;
    }
    fprintf(stdout, I18N("Result: %02x/%02x\nResponse: "),
	    err.subcode1(), err.subcode2());
    fprintf(stdout,"%s",dumpString(response).c_str());
  }
  else if (CTMisc::compare(cm,"last")) {
    string apdu;
    string response;

    apdu=lastAPDU;
    if (!card.isValid()) {
      fprintf(stderr, I18N("Card is not open, try \"open\" first.\n"));
      return 3;
    }
    if (apdu.empty()) {
      fprintf(stderr, I18N("No last APDU.\n"));
      return 2;
    }
    fprintf(stdout, I18N("Resending APDU: %s\n"),
	    apdu.c_str());
    err=card.ref().sendAPDU(apdu, response);
    if (!err.isOk()) {
      fprintf(stderr,I18N("Error: %s\n"),err.errorString().c_str());
      return 3;
    }
    fprintf(stdout, I18N("Result: %02x/%02x\nResponse: "),
	    err.subcode1(), err.subcode2());
    fprintf(stdout,"%s",dumpString(response).c_str());
  }
  else if (CTMisc::compare(cm,"info")){
    unsigned int u;

    if (!card.isValid()) {
      fprintf(stderr, I18N("Card is not open, try \"open\" first.\n"));
      return 3;
    }
    fprintf(stdout, I18N("Reader Information:\n"));
    fprintf(stdout, I18N(" Name  : %s\n"),
            card.ref().readerDescription().name);
    fprintf(stdout, I18N(" Type  : %s\n"),
	    card.ref().readerDescription().type);
    u=card.ref().readerDescription().flags;
    fprintf(stdout, I18N(" Flags :"));
    if (u==0)
      fprintf(stdout,I18N(" none"));
    else {
      if (u & CHIPCARD_READERFLAGS_KEYPAD)
	fprintf(stdout, I18N(" KEYPAD"));
      if (u & CHIPCARD_READERFLAGS_DISPLAY)
	fprintf(stdout, I18N(" DISPLAY"));

    }
    fprintf(stdout, "\n");
    u=card.ref().readerStatus();

    fprintf(stdout, I18N(" Status:"));
    if (u==0)
      fprintf(stdout,I18N(" none"));
    else {
      if (u & CHIPCARD_STATUS_INSERTED)
	fprintf(stdout, I18N(" inserted"));
      if (u & CHIPCARD_STATUS_CONNECTED)
	fprintf(stdout, I18N(" connected"));
      if (u & CHIPCARD_STATUS_LOCKED_BY_OTHER)
	fprintf(stdout, I18N(" locked"));
    }
    fprintf(stdout, "\n");
  }
  else if (CTMisc::compare(cm,"quit") ||
	   CTMisc::compare(cm,"exit")) {

    if (card.isValid()) {
      fprintf(stdout,I18N("Closing card before exiting...\n"));
      err=card.ref().closeCard();
      if (!err.isOk()) {
	fprintf(stderr,I18N("Error: %s\n"),err.errorString().c_str());
      }
      card=0;
    }
    fprintf(stdout,I18N("Exiting.\n"));
    return -1;
  }

  else if (CTMisc::compare(cm,"help") ||
	   CTMisc::compare(cm,"?") ||
	   CTMisc::compare(cm,"h")) {
    fprintf(stdout,
	    I18N("List of commands:\n"
		 "open  - connects the card\n"
		 "close - disconnects the card \n"
		 "apdu xx xx xx xx [xx...] - sends a command to the card\n"
		 "info  - shows some information about the reader the \n"
		 "        currently open card is inserted in\n"
		 "help  - shows this little help screen\n"
		 "quit  - exits\n"
		)
	   );
  }
  else {
    fprintf(stderr,"Unknown command \"%s\"\n",cm.c_str());
    return 1;
  }
  return 0;
}





int main(int argc, char **argv) {
  s_args args;
  int rv;
  CTPointer<CTCard> card;
  string cmdstring;
  char *cmd;
  char buffer[256];
  CTPointer<CTCardTrader> trader;
  CTError err;

#ifdef HAVE_GETTEXT_ENVIRONMENT
  setlocale(LC_ALL,"");
  if (bindtextdomain("cardcommander", I18N_PATH)==0) {
    fprintf(stderr," Error bindtextdomain()\n");
  }
  if (textdomain("cardcommander")==0) {
    fprintf(stderr," Error textdomain()\n");
  }
#endif

  rv=checkArgs(args,argc,argv);
  if (rv==-1)
    return 0;
  else if (rv)
    return rv;

  rv=ChipCard_Init(args.configFile.c_str(),0);
  if (rv!=CHIPCARD_SUCCESS) {
    fprintf(stderr,
	    "Error initializing libchipcard (%d), aborting.\n",rv);
    return 2;
  }

  rv=0;
  trader=new CTCardTrader(false,
			  0,
			  0,
			  CHIPCARD_STATUS_INSERTED,
			  CHIPCARD_STATUS_INSERTED |
			  CHIPCARD_STATUS_LOCKED_BY_OTHER,
			  CHIPCARD_STATUS_LOCKED_BY_OTHER|
			  CHIPCARD_STATUS_INSERTED);
  err=trader.ref().start();
  if (!err.isOk()) {
    fprintf(stderr, I18N("Could not initialize trader"));
    return 2;
  }

  while(rv!=-1) {
    fprintf(stdout,"Card: ");
    cmd=fgets(buffer,sizeof(buffer),stdin);
    cmdstring=cmd;
    rv=execCommand(args, trader, card, cmdstring);
  }

  ChipCard_Fini();
  return rv;
}







