/* 
    TDA8083 - QPSK demodulator

    Copyright (C) 2001 Convergence Integrated Media GmbH <ralph@convergence.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/    

#include <linux/module.h>
#include <linux/init.h>
#include <linux/delay.h>
#include <linux/slab.h>
#include <linux/poll.h>
#include <asm/io.h>
#include <linux/i2c.h>

#include "dvb_frontend.h"

#ifdef MODULE
MODULE_PARM(debug,"i");
MODULE_DESCRIPTION("TDA8083 DVB demodulator driver");
MODULE_AUTHOR("Ralph Metzler");
#ifdef MODULE_LICENSE
MODULE_LICENSE("GPL");
#endif
#endif

static int debug = 0;
#define dprintk	if (debug) printk

static struct i2c_driver dvbt_driver;
static struct i2c_client client_template;

struct tda8083 {
        u32 srate;
        u8 fec;
        u8 inv;
};

static u8
init_tab[] = {
	0x04, 0x00, 0x4a, 0x79, 0x04, 0x00, 0xff, 0xea,
	0x48, 0x42, 0x79, 0x60, 0x70, 0x52, 0x9a, 0x10,
	0x0e, 0x10, 0xf2, 0xa7, 0x93, 0x0b, 0x05, 0xc8,
	0x9d, 0x00, 0x42, 0x80, 0x00, 0x60, 0x40, 0x00,
	0x00, 0x75, 0x00, 0xe0, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00
};

static int 
writereg(struct i2c_client *client, int reg, int data)
{
        int ret;
        unsigned char msg[] = {0x00, 0x00};
        
        msg[0]=reg; msg[1]=data;
        ret=i2c_master_send(client, msg, 2);
        if (ret!=2) 
	        printk("writereg error\n");
        return ret;
}

static u8 
readreg(struct i2c_client *client, u8 reg)
{
        struct i2c_adapter *adap=client->adapter;
        unsigned char mm1[] = {reg};
        unsigned char mm2[] = {0x00};
        struct i2c_msg msgs[2];
        
        msgs[0].flags=0;
        msgs[1].flags=I2C_M_RD;
        msgs[0].addr=msgs[1].addr=client->addr;
        msgs[0].len=1; msgs[1].len=1;
        msgs[0].buf=mm1; msgs[1].buf=mm2;
        i2c_transfer(adap, msgs, 2);
        
        return mm2[0];
}

static int 
init(struct i2c_client *client)
{
        struct tda8083 *tda=(struct tda8083 *) client->data;
	int i;
	
        dprintk("tda8083: init chip\n");
        tda->srate=0;
        tda->fec=0;
        tda->inv=0;
        for (i=0; i<44; i++)
		writereg(client, i, init_tab[i]);
        return 0;
}

static inline 
void ddelay(int i) 
{
        current->state=TASK_INTERRUPTIBLE;
        schedule_timeout((HZ*i)/100);
}

static int 
SetFEC(struct i2c_client *client, u8 fec)
{
        return 0;
}


static int 
SetSymbolrate(struct i2c_client *client, u32 srate)
{
        struct tda8083 *tda=(struct tda8083 *) client->data;
        u32 ratio;
	u32 tmp;
	u8 filter;

	if (srate>32000000)
                srate=32000000;
        if (srate<500000)
                srate=500000;
        tda->srate=srate;

	filter=0;
	if (srate<24000000)
		filter=2;
	if (srate<16000000)
		filter=3;

	tmp=31250<<16;
	ratio=tmp/srate;
        
	tmp=(tmp%srate)<<8;
	ratio=(ratio<<8)+tmp/srate;
        
	tmp=(tmp%srate)<<8;
	ratio=(ratio<<8)+tmp/srate;
	
	dprintk("tda8083: ratio=%08x\n", ratio);

	writereg(client, 0x05, filter);
	writereg(client, 0x02, (ratio>>16)&0xff);
	writereg(client, 0x03, (ratio>>8)&0xff);
	writereg(client, 0x04, ratio&0xff);
	
	writereg(client, 0x00, 0x3c);
	writereg(client, 0x00, 0x04);
	return 1;
}

static int attach_adapter(struct i2c_adapter *adap)
{
        struct tda8083 *tda;
        struct i2c_client *client;
        
        client_template.adapter=adap;
        
        if (i2c_master_send(&client_template,NULL,0))
                return -1;
        
        client_template.adapter=adap;
        
        if ((readreg(&client_template, 0x00))!=0x05) /* FIXME: guess */
                return -1;
        	  
        if (NULL == (client = kmalloc(sizeof(struct i2c_client), GFP_KERNEL)))
                return -ENOMEM;
        memcpy(client, &client_template, sizeof(struct i2c_client));
        
        client->data=tda=kmalloc(sizeof(struct tda8083),GFP_KERNEL);
        if (tda==NULL) {
                kfree(client);
                return -ENOMEM;
        }
       
        i2c_attach_client(client);
        init(client);

        dprintk("tda8083: attaching tda8083 at 0x%02x ", (client->addr)<<1);
        dprintk("to adapter %s\n", adap->name);
	MOD_INC_USE_COUNT;
        return 0;
}

static int detach_client(struct i2c_client *client)
{
        dprintk("tda8083: detach_client\n");
        i2c_detach_client(client);
        kfree(client->data);
        kfree(client);
	MOD_DEC_USE_COUNT;
        return 0;
}

static int dvb_command(struct i2c_client *client, unsigned int cmd, void *arg)
{
        //struct tda8083 *tda=(struct tda8083 *) client->data;
                
        switch (cmd) 
        {
		
        case FE_READ_STATUS:
	{
		FrontendStatus *status=(FrontendStatus *) arg;
		int sync;

		*status=0;
                sync=readreg(client,0x02);

		if (sync&0x01) // FIXME: find criteria for having signal 
			*status|=FE_HAS_SIGNAL;
		if (sync&0x01)
			*status|=FE_HAS_CARRIER;
		if (sync&0x02)
			*status|=FE_HAS_VITERBI;
		if (sync&0x10)
			*status|=FE_HAS_SYNC;
		if ((sync&0x1f)==0x1f)
			*status|=FE_HAS_LOCK;
		break;
	}
        case FE_WRITEREG:
        {
                u8 *msg = (u8 *) arg;
                writereg(client, msg[0], msg[1]);
                break;
        }
        case FE_READREG:
        {
                u8 *msg = (u8 *) arg;

                msg[1]=readreg(client, msg[0]);
                break;
        }
        case FE_INIT:
        {
                init(client);
                break;
        }
        case FE_SET_FRONTEND:
        {
		FrontendParameters *param = (FrontendParameters *) arg;

                SetFEC(client, param->u.qpsk.FEC_inner);
                SetSymbolrate(client, param->u.qpsk.SymbolRate);
                break;
        }
        case FE_RESET:
        {
		writereg(client, 0x00, 0x3c);
		writereg(client, 0x00, 0x04);
                break;
        }

        default:
                return -1;
        }
        
        return 0;
} 

static void inc_use (struct i2c_client *client)
{
#ifdef MODULE
        MOD_INC_USE_COUNT;
#endif
}

static void dec_use (struct i2c_client *client)
{
#ifdef MODULE
        MOD_DEC_USE_COUNT;
#endif
}

static struct i2c_driver dvbt_driver = {
        "tda8083 DVB demodulator",
        I2C_DRIVERID_TDA8083,
        I2C_DF_NOTIFY,
        attach_adapter,
        detach_client,
        dvb_command,
        inc_use,
        dec_use,
};

static struct i2c_client client_template = {
        name:    "tda8083",
        id:      I2C_DRIVERID_TDA8083,
        flags:   0,
        addr:    (0xd0 >> 1),
        adapter: NULL,
        driver:  &dvbt_driver,
};


static __init int
init_tda8083(void) {
        int res;
        
        if ((res = i2c_add_driver(&dvbt_driver))) 
        {
                printk("tda8083: I2C driver registration failed\n");
                return res;
        }
        
        dprintk("tda8083: init done\n");
        return 0;
}

static __exit void
exit_tda8083(void)
{
        int res;
        
        if ((res = i2c_del_driver(&dvbt_driver))) 
        {
                printk("dvb-tuner: Driver deregistration failed, "
                       "module not removed.\n");
        }
        dprintk("tda8083: cleanup\n");
}

module_init(init_tda8083);
module_exit(exit_tda8083);
