#include <fcntl.h>
#include <gtk/gtk.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>
#include <stdio.h>

#include "plugin.h"
#include "configfile.h"

#include "dmx/dmx.h"

static int dmxfd = -1;
static gint slots = 32;
static gint oldslots=32;
static gint rows = 1;
static gint columns = 0; /* defaults to slots. */
static gint effect_type = 0;
static gint column_offset = 0;

#define NUM_EFFEKTS (7)

unsigned char dmxbuffer[4096];


static gchar *PLUGINNAME="DMX4Linux";



static void read_config()
{
  ConfigFile *cfile = xmms_cfg_open_default_file();
  if (cfile)
    {
      xmms_cfg_read_int(cfile, PLUGINNAME, "rows",    &rows);
      xmms_cfg_read_int(cfile, PLUGINNAME, "columns", &columns);
      xmms_cfg_read_int(cfile, PLUGINNAME, "slots",   &slots);
      xmms_cfg_read_int(cfile, PLUGINNAME, "effect_type",   &effect_type);
      xmms_cfg_read_int(cfile, PLUGINNAME, "column_offset",   &column_offset);

      xmms_cfg_free(cfile);
    }
}


static void write_config()
{
  ConfigFile *cfile = xmms_cfg_open_default_file();
  if (cfile)
    {
      xmms_cfg_write_int(cfile, PLUGINNAME, "rows",    rows);
      xmms_cfg_write_int(cfile, PLUGINNAME, "columns", columns);
      xmms_cfg_write_int(cfile, PLUGINNAME, "slots",   slots);
      xmms_cfg_write_int(cfile, PLUGINNAME, "effect_type", effect_type);
      xmms_cfg_write_int(cfile, PLUGINNAME, "column_offset", column_offset);

      xmms_cfg_write_default_file(cfile);
      xmms_cfg_free(cfile);
    }
}

double max_val = 0.0;
double intensities[256];
double avg_intensities[256];

static void blackout()
{
  printf ("dmx4linux plugin stopped\n");

  if(dmxfd)
    {
      memset(dmxbuffer, 0, sizeof(dmxbuffer));
      lseek(dmxfd, 0, SEEK_SET);
      write(dmxfd, dmxbuffer, sizeof(dmxbuffer));
    }
}

/*
 * Initialize the plugin.
 */
static void plug_init()
{
  int  i;

  printf ("dmx4linux plugin started\n");

  dmxfd = open("/proc/dmxdummy/in0", O_WRONLY);  /* try to open the input of the dummy plugin first */
  if (dmxfd < 0)                                 /* this feeds the data back to the input universe of the dmxdummy driver */
    {
      const char *s=DMXdev(0, 0);
      if(s==NULL)
        return;
      dmxfd = open(s, O_WRONLY);
    }
  if(dmxfd<0)
    return;

  for (i=0; i<256; i++)
    avg_intensities[i] = 0.0;

  read_config();
}


static void plug_cleanup(void)
{
#if 0 /* what does the following call do? Do we need it? */
  dmx4linux_vp.disable_plugin(&dmx4linux_vp);
#endif
  blackout();
  if(dmxfd>=0)
    close(dmxfd);
}



static void plug_render_freq(gint16 data[2][256])
{
  double max_limit = 0.0;
  double avg_value = 0.0;

  int z=1;

  intensities[0] = avg_intensities[0] = 0.0; /* absolute part. */
  for (z=1; z<256; z++)
    {
      double v = sqrt(data[0][z] * data[0][z] + data[1][z] * data[1][z]);

      double n = ((double)z / 8.0);
      if (n < 1.0) n = 1.0;

      intensities[z] = v * (1.5 - (1.0 / n)); /* z := 1..255 */

      if (intensities[z] > avg_intensities[z])
	avg_intensities[z] = intensities[z];
      else
	avg_intensities[z] = (avg_intensities[z] * 0.9)  +  (intensities[z] * 0.1);

      avg_value += avg_intensities[z];
    }
  avg_value /= 255.0;

  if (avg_value > max_val)
    max_val = avg_value;

  max_limit = max_val; /* the maximum limit. */


  if (dmxfd)
    {
      int col = 0, row = 0;

      memset(dmxbuffer, 0, sizeof(dmxbuffer));

      for (col=0; col < columns && col*2 < 256; col++)
	{
	  for (row=0; row < rows; row++)
	    {
	      int channel = row*columns + col + column_offset;

	      double value        = (intensities[2*col] + intensities[2*col+1]) / 2.0 / max_limit; /* 0..1 */
	      double normalvalue  = value * rows; /* 0..rows */

	      double avalue       = (avg_intensities[2*col] + avg_intensities[2*col+1]) / 2.0 / max_limit; /* 0..1 */
	      double anormalvalue = avalue * rows; /* 0..rows */

	      int    vcell = rows - row;

	      if (normalvalue < 0.0)
		normalvalue = 0.0;


	      switch (effect_type)
		{
		case 0:
		  if ((int)anormalvalue > vcell)
		    dmxbuffer[channel] = 60;

		  else if ((int)anormalvalue < vcell)
		    dmxbuffer[channel] = 0;

		  else if ((int)anormalvalue == vcell)
		    {
		      double  v = anormalvalue - (int)anormalvalue;
		      dmxbuffer[channel] = (int)(v * 255);
		    }
		  break;

		case 1:
		  if (anormalvalue == vcell)
		    dmxbuffer[channel] = 255;
		  else
		    dmxbuffer[channel] = 0;
		  break;

		case 2:
		  if (((int)anormalvalue-1 == vcell) || ((int)anormalvalue+1 == vcell))
		    dmxbuffer[channel] = 255;

		  else if ((int)anormalvalue == vcell)
		    {
		      double  v = anormalvalue - (int)anormalvalue;
		      dmxbuffer[channel] = (int)(v * 255);
		    }
		  else
		    dmxbuffer[channel] = 0;
		  break;

		case 3:
		  if ((int)anormalvalue == vcell)
		    {
		      double  v = anormalvalue - (int)anormalvalue;
		      dmxbuffer[channel] = (int)(v * 255);
		    }
		  else
		    dmxbuffer[channel] = 0;
		  break;

		case 4:
		  if ((int)anormalvalue-1 == vcell)
		    dmxbuffer[channel] = 255;

		  else if ((int)anormalvalue == vcell)
		    {
		      double  v = anormalvalue - (int)anormalvalue;
		      dmxbuffer[channel] = (int)(v * 255);
		    }
		  else
		    dmxbuffer[channel] = 0;
		  break;

		case 5:
		  if ((int)anormalvalue-1 == vcell)
		    dmxbuffer[channel] = 255;

		  else if ((int)anormalvalue == vcell)
		    {
		      double  v = anormalvalue - (int)anormalvalue;
		      dmxbuffer[channel] = (int)(v * 255);
		    }
		  else if ((int)normalvalue > vcell)
		    dmxbuffer[channel] = 200;
		  else
		    dmxbuffer[channel] = 0;
		  break;
		}
	    }
	}

      lseek(dmxfd, 0, SEEK_SET);
      write(dmxfd, dmxbuffer, rows*columns);
    }
}

static void dmx4linux_xmms_about()
{
  static GtkWidget *dmx4linux_about_win=NULL;
  if (!dmx4linux_about_win)
    {
      GtkWidget *vbox,
	*frame,
	*label,
	*box,
	*button;
      dmx4linux_about_win = gtk_window_new(GTK_WINDOW_DIALOG);

      gtk_signal_connect        (GTK_OBJECT(dmx4linux_about_win), "destroy", GTK_SIGNAL_FUNC(gtk_widget_destroyed), &dmx4linux_about_win);
      gtk_window_set_title      (GTK_WINDOW(dmx4linux_about_win), "About");
      gtk_window_set_policy     (GTK_WINDOW(dmx4linux_about_win), FALSE, FALSE, FALSE);
      gtk_window_set_position   (GTK_WINDOW(dmx4linux_about_win), GTK_WIN_POS_MOUSE);
      gtk_container_border_width(GTK_CONTAINER(dmx4linux_about_win), 10);

      vbox = gtk_vbox_new(FALSE, 10);
      gtk_container_add(GTK_CONTAINER(dmx4linux_about_win), vbox);

      frame = gtk_frame_new("XMMS dmx4linux Plugin:");
      gtk_container_border_width(GTK_CONTAINER(frame), 5);
      gtk_box_pack_start(GTK_BOX(vbox), frame, FALSE, FALSE, 0);

      label = gtk_label_new("\n"
			    "DMX4Linux XMMS Plugin\n"
			    "(C)2000 Michael Stickel\n"
			    "This software is distributed under the\n"
                            "GPL(General Public Licence)\n"
			    "http://llg.cubic.org/\n"
			    "michael@cubic.org\n"
			    );

      gtk_container_add(GTK_CONTAINER(frame), label);

      box = gtk_hbutton_box_new();
      gtk_button_box_set_spacing(GTK_BUTTON_BOX(box), 5);
      gtk_box_pack_start(GTK_BOX(vbox), box, FALSE, FALSE, 0);

      button = gtk_button_new_with_label("Ok");
      gtk_signal_connect_object(GTK_OBJECT(button), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(dmx4linux_about_win));
      GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
      gtk_box_pack_start(GTK_BOX(box), button, TRUE, TRUE, 0);
      gtk_widget_grab_default(button);

      gtk_widget_show(button);
      gtk_widget_show(box);
      gtk_widget_show(frame);
      gtk_widget_show(label);
      gtk_widget_show(vbox);
      gtk_widget_show(dmx4linux_about_win);
    }
}


static GtkWidget *d4l_config_win=NULL;
static GtkWidget *slots_spinner=NULL;
static GtkWidget *rows_spinner=NULL;
static GtkWidget *columns_spinner=NULL;
static GtkWidget *effekt_spinner=NULL;
static GtkWidget *coloffs_spinner=NULL;

static void update_values ()
{
  slots = gtk_spin_button_get_value_as_int( GTK_SPIN_BUTTON(slots_spinner));
  if(slots>4096)
    slots=4096;
  if(slots<1)
    slots=1;

  rows    = gtk_spin_button_get_value_as_int( GTK_SPIN_BUTTON(rows_spinner));
  columns = gtk_spin_button_get_value_as_int( GTK_SPIN_BUTTON(columns_spinner));

  effect_type = gtk_spin_button_get_value_as_int( GTK_SPIN_BUTTON(effekt_spinner));

  column_offset = gtk_spin_button_get_value_as_int( GTK_SPIN_BUTTON(coloffs_spinner));
  if (column_offset + rows*columns > 4096)
    column_offset = 4096 - rows*columns;
}

static void config_ok_cb(GtkButton * button, gpointer data)
{
  update_values();
  gtk_widget_destroy(GTK_WIDGET(d4l_config_win));
  d4l_config_win = NULL;
  write_config();
}


static void config_cancel_cb(GtkButton * button, gpointer data)
{
  gtk_widget_destroy(GTK_WIDGET(d4l_config_win));
  d4l_config_win = NULL;
  slots=oldslots;
}


static void config_apply_cb(GtkButton * button, gpointer data)
{
  update_values ();
}






static void dmx4linux_xmms_configure()
{
  oldslots=slots;
  if (!d4l_config_win)
    {
      GtkWidget *label, *label2, *label3, *label4, *label5, *hbox;

      d4l_config_win = gtk_dialog_new ();
      if (d4l_config_win)
        {
          char *cb_names[] = {"OK","Cancel","Apply"};
          void (*cb_funcs[])(GtkButton * button, gpointer data) = {config_ok_cb, config_cancel_cb, config_apply_cb};
          int i;

	  gtk_signal_connect(GTK_OBJECT(d4l_config_win), "destroy", GTK_SIGNAL_FUNC(gtk_widget_destroyed), &d4l_config_win);
          gtk_window_set_title (GTK_WINDOW(d4l_config_win), "Configure - DMX4Linux Plugin");

          hbox = gtk_hbox_new(FALSE, 5);
	  if (hbox)
	    {
	      gtk_box_pack_start(GTK_BOX(GTK_DIALOG(d4l_config_win)->vbox),hbox, TRUE,TRUE,5);
	      gtk_widget_show(hbox);


	      label4 = gtk_label_new( "Effekt:" );
	      if (label4)
		{
		  gtk_box_pack_start(GTK_BOX(hbox), label4, TRUE, TRUE, 5);
		  gtk_widget_show(label4);
		}

	      effekt_spinner = gtk_spin_button_new (GTK_ADJUSTMENT(gtk_adjustment_new(effect_type, 0, NUM_EFFEKTS-1, 1, 1, 0 )), 0, 0);
	      if (effekt_spinner)
		{
		  gtk_box_pack_start(GTK_BOX(hbox), effekt_spinner, TRUE, TRUE, 5);
		  gtk_widget_show(effekt_spinner);
		}


	      label = gtk_label_new( "Slots:" );
	      if (label)
		{
		  gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 5);
		  gtk_widget_show(label);
		}

	      slots_spinner = gtk_spin_button_new (GTK_ADJUSTMENT(gtk_adjustment_new(slots, 0, 256, 1, 10, 0 )), 0, 0);
	      if (slots_spinner)
		{
		  gtk_box_pack_start(GTK_BOX(hbox), slots_spinner, TRUE, TRUE, 5);
		  gtk_widget_show(slots_spinner);
		}

	      label2 = gtk_label_new( "Rows:" );
	      if (label2)
		{
		  gtk_box_pack_start(GTK_BOX(hbox), label2, TRUE, TRUE, 5);
		  gtk_widget_show(label2);
		}

	      rows_spinner = gtk_spin_button_new (GTK_ADJUSTMENT(gtk_adjustment_new(rows, 0, 256, 1, 10, 0 )), 0, 0);
	      if (rows_spinner)
		{
		  gtk_box_pack_start(GTK_BOX(hbox), rows_spinner, TRUE, TRUE, 5);
		  gtk_widget_show(rows_spinner);
		}

	      label3 = gtk_label_new( "Columns:" );
	      if (label2)
		{
		  gtk_box_pack_start(GTK_BOX(hbox), label3, TRUE, TRUE, 5);
		  gtk_widget_show(label3);
		}

	      columns_spinner = gtk_spin_button_new (GTK_ADJUSTMENT(gtk_adjustment_new(columns, 0, 256, 1, 10, 0 )), 0, 0);
	      if (columns_spinner)
		{
		  gtk_box_pack_start(GTK_BOX(hbox), columns_spinner, TRUE, TRUE, 5);
		  gtk_widget_show(columns_spinner);
		}



	      label5 = gtk_label_new( "Column Offset:" );
	      if (label5)
		{
		  gtk_box_pack_start(GTK_BOX(hbox), label5, TRUE, TRUE, 5);
		  gtk_widget_show(label5);
		}

	      coloffs_spinner = gtk_spin_button_new (GTK_ADJUSTMENT(gtk_adjustment_new(column_offset, 0, 256, 1, 10, 0 )), 0, 0);
	      if (coloffs_spinner)
		{
		  gtk_box_pack_start(GTK_BOX(hbox), coloffs_spinner, TRUE, TRUE, 5);
		  gtk_widget_show(coloffs_spinner);
		}

	    }


	  /*-------------------------------------------------------------------------*/

          for (i=0; i<3; i++)
            {
              GtkWidget *button = gtk_button_new_with_label (cb_names[i]);
              if (button)
                {
                  gtk_box_pack_start (GTK_BOX(GTK_DIALOG(d4l_config_win)->action_area), button, TRUE, TRUE, 0);
                  gtk_signal_connect (GTK_OBJECT(button), "clicked", GTK_SIGNAL_FUNC(cb_funcs[i]), NULL);
                  GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
                  if (i==0)
                    gtk_widget_grab_default(button);
                  gtk_widget_show(button);
                }
            }
          gtk_widget_show (d4l_config_win);
        }
    }
}


VisPlugin dmx4linux_vp =
  {
    description:         "XMMS DMX4Linux Spectrum",
    num_freq_chs_wanted: 2,
    init:                plug_init,
    cleanup:             plug_cleanup,
    about:               dmx4linux_xmms_about,
    configure:           dmx4linux_xmms_configure,
    render_freq:         plug_render_freq,
    playback_stop:       blackout,
  };

VisPlugin *get_vplugin_info(void)
{
  return &dmx4linux_vp;
}
