/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2009 Christian Schallhart <christian@schallhart.net>,
 *                    Michael Tautschnig <tautschnig@forsyte.de>
 *               2008 model.in.tum.de group, FORSYTE group
 *               2006-2007 model.in.tum.de group
 *               2002-2005 Christian Schallhart
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/frame/level.hpp
 *
 * @brief [LEVEL: beta] definition of the diagnostic levels: @ref diagnostics::Level_t
 *
 * $Id: level.hpp,v 1.15 2005/06/23 09:54:19 esdentem Exp $
 *
 * @author Christian Schallhart
 *
 * @test library/interface/frame/level.t.cpp
 */

#ifndef DIAGNOSTICS__FRAME__LEVEL_HPP__INCLUDE_GUARD
#define DIAGNOSTICS__FRAME__LEVEL_HPP__INCLUDE_GUARD

#include <diagnostics/frame/namespace.hpp>

DIAGNOSTICS_NAMESPACE_BEGIN;

/**
 * @brief The diagnostic levels (production, debug, audit, test, and system)
 *
 * The levels are used in three contexts: 
 * 
 * @arg First, each @ref ::diagnostics::Record (which describes some
 * event) has a diagnostic level. If the record has been generated by
 * an annotation (a predefined macro), then the level is identical to
 * the level of this annotation.
 * 
 * @arg Second, a binary is built with a specific debugging
 * level. audit annotations are ignored in production and debugging
 * builds, and audit and debug annotations are ignored in production
 * builds.
 *
 * @arg this, a @ref diagnostics::unittest::Test_Case can be
 * specified for different build debugging levels, for example, a
 * Test_Case might be supposed to work correctly under @ref
 * diagnostics::LEVEL_DEBUG and @ref diagnostics::LEVEL_AUDIT but not
 * under @ref diagnostics::LEVEL_PROD (see the detailed description of
 * the levels). 
 *
 * The third item might be confusing: But consider an annotation
 * 
 * DEBUG_CHECK(Null_Pointer,ptr!=NULL)
 *
 * This annotation will be ignored in a production build, i.e., the
 * programmer means, that this check is not necessary in production
 * code. Consequently, if the check finds a null pointer in code which
 * is supposed to be production code, it is an error. Therefore, the
 * programmer must specify the build levels which are suitable for a
 * test case.
 *
 */
typedef enum {
    /**
     * The production level is meant for the release builds. Thus,
     * only those checks are included into the build, which are
     * expected to be violated from time to time by production
     * code. For example a check whether a file exists might still be
     * necessary in a production build.
     *
     * In other words, each module assumes that it is works properly
     * and that it is used properly. Only those checks are included
     * which guard the code against external errors such as input
     * kludge.
     *
     * @attention This means, that a module in the production build is 
     * behaving undefined, if its debug-preconditions are not met.
     */
    LEVEL_PROD=0,
    /**
     * In the debug level, each module has to check all inputs for
     * their correctness, i.e., if a module is misused by another
     * module, the module should throw an exception. The debug level
     * should be fairly efficient tough.
     *
     * Summarized, each moduels checks that it is used properly --
     * under the assumption that it works internally correctly.
     */
    LEVEL_DEBUG=1,
    /**
     * In the audit level, each module has to check its internal
     * consistency in addition to preconditions, i.e., it has to
     * assert its class invariances, loop invariances, and post
     * conditions.  
     *
     * @attention If the module is working correctly, then the outside
     * observable behavior under LEVEL_DEBUG and LEVEL_AUDIT should be
     * the same (functionally).
     *
     * In addition, the audit level provides massive tracing of the
     * controll flow. For this reason, and for the invariance
     * checking, the performance of the audit build is low. Thus this
     * build type is mainly aimed at module development and testing.
     *
     * Summarized, each moduel checks that it is used properly and
     * works properly.
     */
    LEVEL_AUDIT=2,
    /**
     * All annotation which are written for testing only, i.e.,
     * test_cases, should use this diagnostic level.
     *
     * While LEVEL_PROD, LEVEL_DEBUG, and LEVEL_AUDIT are ment for
     * code which is to be tested and ultimately deployed, LEVEL_TEST
     * is used by code which tests the code to be tested.
     *
     * All messages which have LEVEL_TEST are test-relevant, i.e., if
     * such a record occurs during a @ref
     * ::diagnostics::unittest::Test_Case, it will be part of the test
     * report (represented by @ref
     * ::diagnostics::unittest::Test_Run_Result).
     */
    LEVEL_TEST=3,
    /**
     * All messages which are generated by the diagnostic environment
     * (and by the test system) and which are related to the
     * configuration of this environment are classified with this
     * diagnostic level.
     */
    LEVEL_SYSTEM=4
} Level_t;

/**
 * @brief The maximum integer value of @ref diagnostics::Level_t
 */
#define LEVEL_MAX LEVEL_SYSTEM

/**
 * @brief Returns meaningful string for a diagnostic level. If the
 * level is unkown, i.e., a value not defined in the enum is passed,
 * "UNK" is returned.
 *
 * @throw never 
 */
inline char const * level_to_string(Level_t const level) 
{
    char const * const level_map[]= {"PRO","DEB","AUD","TES","SYS","UNK"};
    return (level>LEVEL_MAX ? level_map[LEVEL_MAX+1] : level_map[level]);
}


DIAGNOSTICS_NAMESPACE_END;

#endif

// vim:ts=4:sw=4
