/*
 * Detect a Library for hardware detection
 *
 * Copyright (C) 1998-2000 MandrakeSoft
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#include <ctype.h>
#include <stdarg.h>
#include <sys/utsname.h>

#include "detect.h"
#include "utils.h"

/* comments have to stay there to have a translation! */
char s_unknown[] = "Unknown";
// _("Unknown")
char s_not_available[] = "Not Available";
// _("Not Available")
char s_none[] = "(none)";
// _("(none)")
char s_ignore[] = "ignore";
// _("ignore")

/**********************************************************************/
/* kernel_verif:  Check if current kernel version correspond to the   */
/*                needed kernel version.                              */
/*                                                                    */
/*     major: major number of the kernel.                             */
/*     minor: minor number of the kernel.                             */
/*     patch: patch number of the kernel.                             */
/**********************************************************************/
extern int kernel_verif(int major, int minor, int patch){
  int cur_major = 0, cur_minor = 0, cur_patch = 0;
  struct utsname kernel;

  if(uname(&kernel)){
    fprintf(stderr, KERNEL_ERROR);
    exit(-1);
  }/*endif*/
  sscanf(kernel.release, "%d.%d.%d", &cur_major, &cur_minor, &cur_patch);

  if(cur_major < major || (cur_major == major && cur_minor < minor)
                       || (cur_major == major && cur_minor == minor 
                                              && cur_patch < patch)){
    fprintf(stderr, KERNEL_VERSION_ERROR);
    return -1;
  }/*endif*/
  return 0;
}/*endfunc kernel_verif*/


/**********************************************************************/
/* my_calloc:  Allocates memory for an array of nmemb elements of     */
/*             size bytes each and returns a pointer to the allocated */
/*             memory. The memory is set to zero.                     */
/*                                                                    */
/*     nmemb: Number of elements in the array.                        */
/*     size: Byte length of each array element.                       */
/**********************************************************************/
extern void *my_calloc(size_t nmemb, size_t size){
  void *ptr = NULL;

  if(!size){
    return NULL;
  }/*endif*/
  if((ptr = calloc(nmemb, size)) == NULL){
    fprintf(stderr, MALLOC_ERROR);
    exit(-2);
  }/*endif*/
  return ptr;
}/*endfunc my_calloc*/


/**********************************************************************/
/* my_malloc:  Allocates size bytes and returns a pointer to the      */
/*             allocated memory.  The memory is not cleared.          */
/*                                                                    */
/*     size: Number of allocated bytes.                               */
/**********************************************************************/
extern void *my_malloc(size_t size){
  void *ptr = NULL;

  if(!size){
    return NULL;
  }/*endif*/
  if((ptr = malloc(size)) == NULL){
    fprintf(stderr, MALLOC_ERROR);
    exit(-2);
  }/*endif*/
  return ptr;
}/*endfunc my_malloc*/


/**********************************************************************/
/* cut_word: Cut a len byte word, beginning at start, out of string.  */
/*                                                                    */
/*     string: The string containing the word we want to extract.     */
/*     start: First character to be extracted.                        */
/*     len: Number of characters to be extracted.                     */
/**********************************************************************/
extern char *cut_word(char *string, int start, int len){
  char *buffer = NULL;
  int tmp1 = 0;
  int tmp2 = 0;

  buffer = (char *)my_malloc(len + 1);
  for(tmp1 = start; tmp1 < start + len; tmp1++){
    buffer[tmp2] = string[tmp1];
    tmp2++;
  }/*next tmp1*/
  buffer[tmp2] = '\0';
  return buffer;
}/*endfunc cut_word*/


/**********************************************************************/
/* extract_word:  Extract a word from a delim separated string.       */
/*                                                                    */
/*     string: String to be processed.                                */
/*     pos: Position of the word (beginning with 1).                  */
/*     delim: Delimiters between the fields.                          */
/*                                                                    */
/* Example:      delim=":"    string="wwwww:zzz:hhh:llllll"           */
/*                            pos   = [1]   [2] [3] [4]               */
/**********************************************************************/
extern char *extract_word(char *string, int pos, char *delim){
  char *temp = (char *)NULL;
  char *left = (char *)NULL;
  char *found = (char *)NULL;
  static char *res = (char *)NULL;
  int current=1;
  
  /* first copy original string to keep it ... */
  temp = (char *)my_malloc(strlen(string)+1);
  strcpy(temp, string);
  /* now work on the copy */
  left = temp;
  
  for(found = strstr(left, delim); found; found = strstr(left, delim)){
    if(current == pos){
      found[0]='\0';
      trim(left);
      res = (char *)my_malloc(strlen(left)+1);
      strcpy(res, left);
      free(temp);
      return res;
    }else{
      current+=1;
      left=found + strlen(delim);
    }/*endif*/  
  }/*next found*/
  if(current== pos){
    trim(left);
    res = (char *)my_malloc(strlen(left)+1);
    strcpy(res, left);
  }/*else*/
  return res;
}/*endfunc extract_word*/


/**********************************************************************/
/* find_word:  Extract a word from a file.                            */
/*                                                                    */
/*     word: Word to be found.                                        */
/*     file: Filename to be processed.                                */
/**********************************************************************/
extern char *find_word(char *word, char *file){
  FILE *f;
  char *line;
  size_t len = 0;

  if(!(f = fopen(file, "r"))) {
    fprintf(stderr, FOPEN_READ_ERROR, file);
    return NULL;
  }/*endif*/
  while(getline(&line, &len, f) >= 0){
    if(strstr(line, word)) {
      if(strstr(word, "\n")        ){
	fclose(f);
        return cut_word(line, 0, strlen(line) - 1);
      }else{
	fclose(f);
        return line;
      }/*endif*/
    }/*endif*/
  }/*endwhile*/
  len = 0;
  free(line);
  fclose(f);
  return NULL;
}/*endfunc find_word*/


/**********************************************************************/
/* trim:  Trim a string (remove whitespaces behind the text).         */
/*                                                                    */
/*     str: String to be trimmed.                                     */
/**********************************************************************/
extern char *trim(char *string){
  int i;

  for(i=strlen(string)-1; i>0; i-=1){
    if((string[i] == ' ') 
        || (string[i] == '\t') 
        || (string[i] == '\n')){
      string[i]='\0';
    }else{
      break;
    }/*endif*/
  }/*endif*/
  return string;
}/*endfunc trim*/


extern char *ToUpper(char *str){
  char *res;
  int i;

  res = (char *)my_malloc(strlen(str) + 1);
  strcpy(res, str);
  for(i= 0; i < strlen(res); i++){
    res[i] = toupper(*(res+i));
  }/*next i*/
  return res;
}/*endfunc ToUpper*/


extern char *ToLower(char *str){
  char *res;
  int i;

  res = (char *)my_malloc(strlen(str) + 1);
  strcpy(res, str);
  for(i= 0; i < strlen(res); i++){
    res[i] = tolower(*(res+i));
  }/*next i*/
  return res;
}/*endfunc ToLower*/
