/*
 * Detect a Library for hardware detection
 *
 * Copyright (C) 1998-2000 MandrakeSoft
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */



#include "detect.h"
#include "utils.h"


typedef struct{
  unsigned long id;
  char vendor[256];		/* Vendor name                        */
  char model[256];		/* Model name                         */
  char module[32];		/* Name of the module to load         */
  DeviceType type;
} TableEntry;



static int read_table(FILE *tablefile, TableEntry *entry){
  char *line;
  char class[12];
  int dumb;
  size_t len = 0;
  
  if(getline(&line, &len, tablefile) < 0) {
    return EOF;
  }/*endif*/
  
  if(line[0] == '\t'){
    sscanf(line, "%08lx\t%12s\t%32s\t%256[^\n]\n",
	   (unsigned long *)&entry->id, 
	   (char *)&class,
	   (char *)&entry->module, 
	   (char *)&entry->model); 
    entry->type = str2device(class);
  }else{
    sscanf(line, "%04x %256[^\n]\n", 
           (int *)&dumb,
           (char *)&entry->vendor);
    len = 0;
    free(line);
    return read_table(tablefile, entry);
  }/*endif*/

  len = 0;
  free(line);

  return 0;
}/*endfunc read_table*/


extern struct cards_lst *init_lst(char *isa_lst, char *pci_lst, 
				  char *pcmcia_lst, char *usb_lst){
  FILE *f;
  char *line;
  struct cards_lst *lst, *first;
  char options[256];
  char class[12];
  TableEntry entry;
  size_t len = 0;

  first = lst = (struct cards_lst *)my_malloc(sizeof(struct cards_lst));

  /********************************************************************/
  /************************ ISA HARDWARE LIST *************************/
  /********************************************************************/
  if(isa_lst){
    if(debug)
    fprintf(stdout, "Reading ISA hardware database...\n");

    /*First we add all ISA cards recognized*/
    if(!(f = fopen(isa_lst, "r"))){
      fprintf(stderr, FOPEN_READ_ERROR, isa_lst);
    }else{
      while(getline(&line, &len, f) >= 0) {
        lst->dev_id = (char *) malloc(7);
        lst->vendor = (char *) malloc(32);
        lst->model = (char *) malloc(256);    
        lst->modulename = (char *)malloc(32);
        sscanf(line, "%7s\t%32[^\t]\t%12[^\t]\t%32[^\t]\t%256[^\t]"
	       "\t%256[^\n]\n",
               lst->dev_id,
               class,
               lst->modulename,
               lst->vendor,
               lst->model,
               options);
        lst->type = str2device(class);
        lst->options = str2options(options);
        if((lst->options & IS_SERIAL) > 0){
          lst->bus = SERIAL;
        }else{
          lst->bus = ISA;
        }/*endif*/
        lst->next = (struct cards_lst *) 
	  my_malloc(sizeof(struct cards_lst));
        lst = lst->next;
      }/*endwhile*/ 
      fclose(f);
    }/*endif*/
  }/*endif*/

  /********************************************************************/
  /************************ PCI HARDWARE LIST *************************/
  /********************************************************************/
  if(pci_lst){  
    if(debug)
      fprintf(stdout, "Reading PCI hardware database...\n");

    /*PCI list processing*/
    if(!(f = fopen(pci_lst, "r"))){
      fprintf(stderr, FOPEN_READ_ERROR, pci_lst);
    }else {
      while(read_table(f, &entry) != EOF) {
        lst->vendor = (char *)my_malloc(256);
        lst->model = (char *)my_malloc(256);
        lst->modulename = (char *)my_malloc(30);
        lst->long_id = entry.id;
	lst->type = entry.type;
        strcpy(lst->vendor, entry.vendor);
        strcpy(lst->model, entry.model);
        strcpy(lst->modulename, entry.module);
        lst->bus = PCI;
        lst->next = (struct cards_lst *) 
	  my_malloc(sizeof(struct cards_lst));
        lst = lst->next;
      }/*endwhile*/
      fclose(f);
    }/*endif*/
  }/*endif*/


  /********************************************************************/
  /************************ USB HARDWARE LIST *************************/
  /********************************************************************/
  
  if(usb_lst) {
    if(debug)
      fprintf(stdout, "Reading USB hardware database...\n");

    /*USB list processing*/
    if(!(f = fopen(usb_lst, "r"))){
      fprintf(stderr, FOPEN_READ_ERROR, usb_lst);
    }else {
      while(read_table(f, &entry) != EOF) {
	lst->vendor = (char *)my_malloc(256);
	lst->model = (char *)my_malloc(256);
	lst->modulename = (char *)my_malloc(30);
	lst->long_id = entry.id;
	lst->type = entry.type;
	strcpy(lst->vendor, entry.vendor);
	strcpy(lst->model, entry.model);
	strcpy(lst->modulename, entry.module);
	lst->bus = USB;
	lst->next = (struct cards_lst *)
	  my_malloc(sizeof(struct cards_lst));
	lst = lst->next;
      }/*endwhile*/
      fclose(f);
    }/*endif*/
  }/*endif*/
  
  /********************************************************************/
  /********************** PCMCIA HARDWARE LIST ************************/
  /********************************************************************/
  if(pcmcia_lst){
    if(debug)
      fprintf(stdout, "Reading PCMCIA hardware database...\n");

   /*PCMCIA list processing*/
    if(!(f = fopen(pcmcia_lst, "r"))){
      fprintf(stderr, FOPEN_READ_ERROR, pcmcia_lst);
    }else {
      while(read_table(f, &entry) != EOF) {
        lst->vendor = (char *)my_malloc(256);
        lst->model = (char *)my_malloc(256);
        lst->modulename = (char *)my_malloc(30);
        lst->long_id = entry.id;
	lst->type = entry.type;
        strcpy(lst->vendor, entry.vendor);
        strcpy(lst->model, entry.model);
        strcpy(lst->modulename, entry.module);
        lst->bus = PCMCIA;
        lst->next = (struct cards_lst *)
	  my_malloc(sizeof(struct cards_lst));
        lst = lst->next;
      }/*endwhile*/
      fclose(f);
    }/*endif*/
  }/*endif*/

  len = 0;
  free(line);

  lst->next = NULL;
  lst = first;
  return lst;
}/*endfunc init_lst*/
