/* commandfuncs.c
 * functions invoked by user keypresses
 *
 * for Denemo, a gtk+ frontend to GNU Lilypond
 * (c) 1999, 2000, 2001 Matthew Hiller
 */

#include <string.h>

#include "calculatepositions.h"
#include "chordops.h"
#include "contexts.h"
#include "datastructures.h"
#include "draw.h"
#include "measureops.h"
#include "midi.h"
#include "objops.h"
#include "moveviewport.h"
#include "staffops.h"
#include "selectops.h"
#include "utils.h"
#include "tupletops.h"
#include "graceops.h"

#define declarecurmudelaobj \
mudelaobject *curmudelaobj = si->currentobject ? \
  si->currentobject->data : NULL

static void
beamandstemdirhelper (struct scoreinfo *si)
{
  calculatebeamsandstemdirs
    (si->currentmeasure->data, &(si->curmeasureclef), &(si->cursortime1),
     &(si->cursortime2), &(si->curmeasure_stem_directive));
}

void
setcurrents (struct scoreinfo *si)
{
  si->currentmeasure =
    g_list_nth (firstmeasurenode (si->currentstaff),
		si->currentmeasurenum - 1);
  si->cursor_x = 0;
  si->currentobject = si->currentmeasure->data;
  if (si->currentobject)
    si->cursor_appending = FALSE;
  else
    si->cursor_appending = TRUE;
  calcmarkboundaries (si);
}

void
nudgerightward (struct scoreinfo *si)
{
  set_rightmeasurenum (si);
  while (si->currentmeasurenum > si->rightmeasurenum)
    {
      si->leftmeasurenum++;
      set_rightmeasurenum (si);
    }
  find_leftmost_allcontexts (si);
  update_hscrollbar (si);
}

void
nudge_downward (struct scoreinfo *si)
{
  set_bottom_staff (si);

  while (si->currentstaffnum > si->bottom_staff)
    {
      si->top_staff++;
      set_bottom_staff (si);
    }
  update_vscrollbar (si);
}

void
set_width_to_work_with (struct scoreinfo *si)
{
  si->widthtoworkwith
    = (si->scorearea->allocation.width
       - (RIGHT_MARGIN + KEY_MARGIN + si->maxkeywidth + SPACE_FOR_TIME));
}

void
adjustmeasurewidth (struct scoreinfo *si, gint amount)
{
  si->measurewidth += amount;
  if (si->measurewidth < 10)
    si->measurewidth = 10;
  if (si->widthtoworkwith < si->measurewidth + SPACE_FOR_BARLINE)
    si->measurewidth = si->widthtoworkwith - SPACE_FOR_BARLINE;
  find_xes_in_all_measures (si);
  nudgerightward (si);
}

/* Determines whether the closer jump will be up or down */

gint jumpcursor (gint cursor_y, gint fromnote, gint tonote)
{
  int distance;

  distance = (tonote - fromnote + 7) % 7;
  if (distance <= 3)		/* an upward jump is good */
    return cursor_y + distance;
  else				/* jump down */
    return cursor_y - 7 + distance;
}

void
object_insert (struct scoreinfo *si, mudelaobject * mudela_obj_new)
{
  declarecurmudelaobj;

  /* First, check to see if the operation would add something before an
   * indicator of a time signature change. This would be bad, so don't
   * allow it to happen */
  if (curmudelaobj && curmudelaobj->type == TIMESIG && !si->cursor_appending)
    {
      si->cursor_x++;
      if (si->currentobject->next)
	si->currentobject = si->currentobject->next;
      else
	si->cursor_appending = TRUE;
    }

  si->currentmeasure->data = g_list_insert (si->currentmeasure->data,
					    mudela_obj_new, si->cursor_x);
  si->cursor_x++;
  if (si->cursor_appending)
    si->currentobject = g_list_last (si->currentmeasure->data);
  else
    si->currentobject = g_list_nth (si->currentmeasure->data, si->cursor_x);
  beamandstemdirhelper (si);
  showwhichaccidentals (si->currentmeasure->data, si->curmeasurekey,
			si->curmeasureaccs);
  find_xes_in_measure (si, si->currentmeasurenum,
		       si->cursortime1, si->cursortime2);
  nudgerightward (si);
  si->haschanged = TRUE;
  si->markstaffnum = 0;
}



void
adjuststaffheight (struct scoreinfo *si, gint amount)
{
  si->staffspace += amount;
  if (si->staffspace < 2 * STAFF_HEIGHT)
    si->staffspace = 2 * STAFF_HEIGHT;
  nudge_downward (si);
}

void
measureleft (struct scoreinfo *si)
{
  if (!si->cursor_x && si->currentmeasure->prev)
    {
      si->currentmeasurenum--;
      isoffleftside (si);
    }
  setcurrents (si);
}

void
measureright (struct scoreinfo *si)
{
  if (si->currentmeasure->next)
    {
      si->currentmeasurenum++;
      isoffrightside (si);
      setcurrents (si);
    }
}

void
staffup (struct scoreinfo *si)
{
  if (si->currentstaff->prev)
    {
      si->currentstaffnum--;
      si->currentstaff = si->currentstaff->prev;
      setcurrentprimarystaff (si);
      setcurrents (si);
      move_viewport_up (si);
    }
}

void
staffdown (struct scoreinfo *si)
{
  if (si->currentstaff->next)
    {
      si->currentstaffnum++;
      si->currentstaff = si->currentstaff->next;
      setcurrentprimarystaff (si);
      setcurrents (si);
      move_viewport_down (si);
    }
}

void
cursorleft (struct scoreinfo *si)
{
  if (!si->cursor_x)
    {
      /* also the only situation where si->currentobject == NULL */
      if (si->currentmeasure->prev)
	{
	  /* Go to end of preceding measure */
	  si->cursor_appending = TRUE;
	  si->currentmeasure = si->currentmeasure->prev;
	  si->currentmeasurenum--;
	  isoffleftside (si);
	  si->currentobject =
	    g_list_last ((objnode *) si->currentmeasure->data);
	  /* The preceding statement will set currentobject to
	   * NULL if appropriate */
	  si->cursor_x = g_list_length ((objnode *) si->currentmeasure->data);
	  /* Despite appearances, there is not an off-by-one error in the
	   * preceding command */
	}
    }
  else if (si->cursor_appending)
    {
      /* Can back off from appending */
      si->cursor_appending = FALSE;
      si->cursor_x--;
    }
  else
    {
      /* Can go back in the measure */
      si->currentobject = si->currentobject->prev;
      si->cursor_x--;
    }
  calcmarkboundaries (si);
}

void
cursorright (struct scoreinfo *si)
{
  if (si->cursor_appending && si->currentmeasure->next)
    {
      /* Go to the next measure */
      si->currentmeasure = si->currentmeasure->next;
      si->currentmeasurenum++;
      isoffrightside (si);
      si->currentobject = (objnode *) si->currentmeasure->data;
      si->cursor_x = 0;
      if (si->currentobject)
	si->cursor_appending = FALSE;
    }
  else if (si->currentobject)
    {
      /* See if we should go to appending position. If not, go to the
       * next note (if possible) */
      if (!si->cursor_appending && !si->currentobject->next)
	{
	  /* Go to appending position */
	  si->cursor_appending = TRUE;
	  si->cursor_x++;
	}
      else if (si->currentobject->next)
	{
	  si->currentobject = si->currentobject->next;
	  si->cursor_x++;
	}
    }
  calcmarkboundaries (si);
}

void
cursorup (struct scoreinfo *si)
{
  si->cursor_y++;
  si->staffletter_y = (si->staffletter_y + 1) % 7;
}

void
cursordown (struct scoreinfo *si)
{
  si->cursor_y--;
  si->staffletter_y = (si->staffletter_y + 6) % 7;
}

void
shiftcursor (struct scoreinfo *si, gint note_value)
{
  gint oldstaffletter_y = si->staffletter_y;

  si->staffletter_y = note_value;
  si->cursor_y = jumpcursor (si->cursor_y, oldstaffletter_y,
			     si->staffletter_y);
}

void
insertchord (struct scoreinfo *si, gint duration, gboolean rest)
{
  gboolean next_measure;
  mudelaobject *mudela_obj_new;

  /* First, check to see if the insertion'll cause the cursor to
   * jump to the next measure. (Denemo will implicitly create it
   * if it doesn't exist already.) */

  next_measure = si->cursoroffend && si->cursor_appending
    && (!si->currentmeasure->next || !si->currentmeasure->next->data);
  if (next_measure)
    {
      if (!si->currentmeasure->next)
	/* Add a measure and make it currentmeasure */
	si->currentmeasure = addmeasures (si, si->currentmeasurenum, 1);
      else
	si->currentmeasure = si->currentmeasure->next;
      /* Now the stuff that needs to be done for each case */
      si->currentmeasurenum++;
      si->currentobject = NULL;
      si->cursor_x = 0;
      memcpy (si->cursoraccs, si->nextmeasureaccs, SEVENGINTS);
      memcpy (si->curmeasureaccs, si->nextmeasureaccs, SEVENGINTS);
      si->curmeasureclef = si->cursorclef;
    }

  /* Now actually create the chord */
  mudela_obj_new = newchord (duration, 0);

  if (!si->rest_mode && (rest != TRUE))
    addtone (mudela_obj_new, si->cursor_y, si->cursoraccs[si->staffletter_y],
	     si->cursorclef);
  else
    si->rest_mode = 0;

  if (si->is_grace_mode)
    mudela_obj_new->u.chordval.is_grace = TRUE;


  /* Insert the new note into the score.  Note that while we may have
     added a measure above, object_insert will invoke nudgerightward,
     which will in turn invoke update_hscrollbar, so we
     don't need to invoke that here.  */

  object_insert (si, mudela_obj_new);

  playnotes (si->prefs->immediateplayback, mudela_obj_new->u.chordval);
}

void
inserttuplet (struct scoreinfo *si, gint duration)
{
  mudelaobject *mudela_obj_new;

  switch (duration)
    {
    case 0:
    case 1:
    case 2:
      mudela_obj_new = newtupopen (2, 3);
      tupletchangedialog (mudela_obj_new, si->scorearea);
      break;
    case 3:
      mudela_obj_new = newtupopen (2, 3);
      break;
    case 5:
      mudela_obj_new = newtupopen (4, 5);
      break;
    case 6:
      mudela_obj_new = newtupopen (4, 6);
      break;
    case 7:
      if (si->cursortime2 == 8)
	{
	  mudela_obj_new = newtupopen (8, 7);
	}
      else
	mudela_obj_new = newtupopen (4, 7);
      break;
    case 9:
      mudela_obj_new = newtupopen (8, 9);
      break;
    default:
      mudela_obj_new = newtupopen (2, 3);
      break;
    }
  object_insert (si, mudela_obj_new);

  /* Add the closing bracket */
  object_insert (si, newtupclose ());
  si->cursor_x--;

  /* And clean up */
  si->currentobject = g_list_nth (si->currentmeasure->data, si->cursor_x);
  si->cursor_appending = FALSE;
}

void
insertgrace (struct scoreinfo *si)
{
  mudelaobject *mudela_obj_new;

  mudela_obj_new = newgracestart ();

  object_insert (si, mudela_obj_new);

  object_insert (si, newgraceend ());
  si->cursor_x--;
  si->currentobject = g_list_nth (si->currentmeasure->data, si->cursor_x);
  si->cursor_appending = FALSE;
  si->is_grace_mode = TRUE;
}

void
changeduration (struct scoreinfo *si, gint duration)
{
  declarecurmudelaobj;

  if (curmudelaobj && curmudelaobj->type == CHORD)
    {
      changedur (curmudelaobj, duration, 0);
      beamandstemdirhelper (si);
      find_xes_in_measure (si, si->currentmeasurenum,
			   si->cursortime1, si->cursortime2);
      nudgerightward (si);
      si->haschanged = TRUE;
    }
}

void
tonechange (struct scoreinfo *si, gboolean remove)
{
  declarecurmudelaobj;

  if (curmudelaobj && curmudelaobj->type == CHORD)
    {
      if (remove == TRUE)
	removetone (curmudelaobj, si->cursor_y, si->cursorclef);
      else
	addtone (curmudelaobj, si->cursor_y,
		 si->cursoraccs[si->staffletter_y], si->cursorclef);
      beamandstemdirhelper (si);
      showwhichaccidentals (si->currentmeasure->data, si->curmeasurekey,
			    si->curmeasureaccs);
      find_xes_in_measure (si, si->currentmeasurenum,
			   si->cursortime1, si->cursortime2);
      nudgerightward (si);
      si->haschanged = TRUE;
      playnotes (si->prefs->immediateplayback, curmudelaobj->u.chordval);
    }
}

/* Change the enharmonic shift of the tone closest to the cursor. The
 * function double-checks to be sure that it's passed a chord, though
 * at the moment (28 July 2000), it probably doesn't strictly need to */

void
changeenshift (struct scoreinfo *si, gint amount)
{
  declarecurmudelaobj;

  if (curmudelaobj && curmudelaobj->type == CHORD)
    {
      shiftpitch (curmudelaobj, si->cursor_y, amount == 1);
      showwhichaccidentals (si->currentmeasure->data, si->curmeasurekey,
			    si->curmeasureaccs);
      find_xes_in_measure (si, si->currentmeasurenum,
			   si->cursortime1, si->cursortime2);
      nudgerightward (si);
      si->haschanged = TRUE;
      playnotes (si->prefs->immediateplayback, curmudelaobj->u.chordval);
    }
}

void
change_stem_directive (struct scoreinfo *si, gint amount)
{
  declarecurmudelaobj;

  if (curmudelaobj && curmudelaobj->type == STEMDIRECTIVE)
    {
      curmudelaobj->u.stemval.type += amount;
      curmudelaobj->u.stemval.type =
	MAX (curmudelaobj->u.stemval.type, STEMDOWN);
      curmudelaobj->u.stemval.type =
	MIN (curmudelaobj->u.stemval.type, STEMUP);
      beamsandstemdirswholestaff (si->currentstaff->data);
      find_xes_in_all_measures (si);
      nudgerightward (si);
      si->haschanged = TRUE;
      gtk_widget_draw (si->scorearea, NULL);
    }
}

void
changedots (struct scoreinfo *si, gint amount)
{
  declarecurmudelaobj;

  if (curmudelaobj && curmudelaobj->type == CHORD)
    {
      changenumdots (curmudelaobj, amount);
      beamandstemdirhelper (si);
      find_xes_in_measure (si, si->currentmeasurenum,
			   si->cursortime1, si->cursortime2);
      nudgerightward (si);
      si->haschanged = TRUE;
    }
}

void
insertmeasures (struct scoreinfo *si, gint number)
{
  si->currentmeasure = addmeasures (si, si->currentmeasurenum - 1, number);
  si->cursor_x = 0;
  si->cursor_appending = TRUE;
  si->currentobject = NULL;
  set_rightmeasurenum (si);
  si->haschanged = TRUE;
  si->markstaffnum = 0;
  calcmarkboundaries (si);
  update_hscrollbar (si);
}

void
appendmeasures (struct scoreinfo *si, gint number)
{
  addmeasures (si, g_list_length (firstmeasurenode (si->currentstaff)),
	       number);
  /* Reset these two variables because si->currentmeasure and
   * si->currentobject may now be pointing to dead data */
  si->currentmeasure =
    g_list_nth (firstmeasurenode (si->currentstaff),
		si->currentmeasurenum - 1);
  si->currentobject =
    g_list_nth (si->currentmeasure->data,
		si->cursor_x - (si->cursor_appending == TRUE));
  set_rightmeasurenum (si);
  si->haschanged = TRUE;
  update_hscrollbar (si);
}

void
deletestaff (struct scoreinfo *si)
{
  if (g_list_length (si->thescore) > 1)
    {
      removestaff (si, si->currentstaffnum - 1, 1);
      setcurrents (si);
      find_xes_in_all_measures (si);
      nudgerightward (si);
      si->haschanged = TRUE;
      si->markstaffnum = 0;
    }
}

void
deletemeasure (struct scoreinfo *si)
{
  si->currentmeasure = removemeasures (si, si->currentmeasurenum - 1, 1);
  isoffleftside (si);
  /* In case that was the last measure we just deleted, which'd cause
   * the current measure to be the left of what's displayed */
  nudgerightward (si);
  setcurrents (si);
  si->haschanged = TRUE;
  si->markstaffnum = 0;
  update_hscrollbar (si);
}

static void
remove_object (measurenode * cur_measure, objnode * cur_objnode)
{
  cur_measure->data = g_list_remove_link (cur_measure->data, cur_objnode);
  freeobject (cur_objnode->data);
  g_list_free_1 (cur_objnode);
}

static void
reset_cursor_stats (struct scoreinfo *si)
{
  si->currentobject = g_list_nth (si->currentmeasure->data, si->cursor_x);
  if (!si->currentobject)
    {
      si->currentobject = g_list_last (si->currentmeasure->data);
      si->cursor_appending = TRUE;
    }
}

static void
delete_object_helper (struct scoreinfo *si)
{
  remove_object (si->currentmeasure, si->currentobject);
  reset_cursor_stats (si);
}

void
deleteobject (struct scoreinfo *si)
{
  declarecurmudelaobj;
  staffnode *curstaff;
  measurenode *curmeasure;

  if (!si->cursor_appending)
    {
      switch (curmudelaobj->type)
	{
	case CHORD:
	  delete_object_helper (si);
	  beamandstemdirhelper (si);
	  showwhichaccidentals (si->currentmeasure->data, si->curmeasurekey,
				si->curmeasureaccs);
	  find_xes_in_measure (si, si->currentmeasurenum, si->cursortime1,
			       si->cursortime2);
	  break;
	case TUPOPEN:
	case TUPCLOSE:
	  /* TODO - add code that will automatically delete a tupbracket's
	   * corresponding bracket */
	  delete_object_helper (si);
	  beamandstemdirhelper (si);
	  find_xes_in_measure (si, si->currentmeasurenum, si->cursortime1,
			       si->cursortime2);
	  break;
	case CLEF:
	  delete_object_helper (si);
	  fixnoteheights (si->currentstaff->data);
	  beamsandstemdirswholestaff (si->currentstaff->data);
	  find_xes_in_all_measures (si);
	  break;
	case KEYSIG:
	  /* Doesn't automatically delete sibling key signatures, though
	   * I probably will have it do so soon */
	  delete_object_helper (si);
	  beamsandstemdirswholestaff (si->currentstaff->data);
	  showwhichaccidentalswholestaff (si->currentstaff->data);
	  find_xes_in_all_measures (si);
	  break;
	case TIMESIG:
	  /* For time signatures, deletion is linked to all
	   * the staffs on the score */
	  for (curstaff = si->thescore; curstaff; curstaff = curstaff->next)
	    {
	      curmeasure = g_list_nth (firstmeasurenode (curstaff),
				       si->currentmeasurenum - 1);
	      remove_object (curmeasure, curmeasure->data);
	      beamsandstemdirswholestaff (curstaff->data);
	    }
	  reset_cursor_stats (si);
	  find_xes_in_all_measures (si);
	  break;
	case STEMDIRECTIVE:
	  delete_object_helper (si);
	  beamsandstemdirswholestaff (si->currentstaff->data);
	  find_xes_in_all_measures (si);
	  break;
	case DYNAMIC:
	  delete_object_helper (si);
	  beamandstemdirhelper (si);
	  find_xes_in_measure (si, si->currentmeasurenum, si->cursortime1,
			       si->cursortime2);
	  break;
	case LILYDIRECTIVE:
	  delete_object_helper (si);
	  beamandstemdirhelper (si);
	  find_xes_in_measure (si, si->currentmeasurenum, si->cursortime1,
			       si->cursortime2);
	  break;
	case GRACE_START:
	case GRACE_END:
	  delete_object_helper (si);
	  beamandstemdirhelper (si);
	  find_xes_in_measure (si, si->currentmeasurenum, si->cursortime1,
			       si->cursortime2);
	  break;
	case BARLINE:
	case COMMENT:
	case MEASUREBREAK:
	  break;
	}
      nudgerightward (si);
      si->haschanged = TRUE;
      si->markstaffnum = 0;
    }
}

void
insertclone (struct scoreinfo *si)
{
  declarecurmudelaobj;

  if (curmudelaobj && curmudelaobj->type == CHORD)
    object_insert (si, clone_object (curmudelaobj));
}

/* Go to end of score */
void
toend (struct scoreinfo *si, gint callback_action, GtkWidget * widget)
{
  si->currentmeasurenum = si->leftmeasurenum = si->rightmeasurenum =
    g_list_length (((staff *) si->currentstaff->data)->measures);
  setcurrents (si);
  find_leftmost_allcontexts (si);
  update_hscrollbar (si);
  gtk_widget_draw (si->scorearea, NULL);
}

/* Go to beginning of score */
void
tohome (struct scoreinfo *si, gint callback_action, GtkWidget * widget)
{
  si->currentmeasurenum = si->leftmeasurenum = 1;
  set_rightmeasurenum (si);
  setcurrents (si);
  find_leftmost_allcontexts (si);
  update_hscrollbar (si);
  gtk_widget_draw (si->scorearea, NULL);
}

void
stem_directive_insert (struct scoreinfo *si, gint callback_action,
		       GtkWidget * widget)
{
  object_insert (si, stem_directive_new (STEMBOTH));
  /* This sets beams and stem directions in the measure, but that's
   * not sufficient */
  beamsandstemdirswholestaff (si->currentstaff->data);
  find_xes_in_all_measures (si);
  nudgerightward (si);
  si->haschanged = TRUE;
  gtk_widget_draw (si->scorearea, NULL);
}

void
toggle_begin_slur (struct scoreinfo *si)
{
  declarecurmudelaobj;

  if (curmudelaobj && curmudelaobj->type == CHORD)
    {
      curmudelaobj->u.chordval.slur_begin_p
	= !curmudelaobj->u.chordval.slur_begin_p;
      si->haschanged = TRUE;
      gtk_widget_draw (si->scorearea, NULL);
    }
}

void
toggle_end_slur (struct scoreinfo *si)
{
  declarecurmudelaobj;

  if (curmudelaobj && curmudelaobj->type == CHORD)
    {
      curmudelaobj->u.chordval.slur_end_p
	= !curmudelaobj->u.chordval.slur_end_p;
      si->haschanged = TRUE;
      gtk_widget_draw (si->scorearea, NULL);
    }
}


void
toggle_start_crescendo (struct scoreinfo *si)
{
  declarecurmudelaobj;

  if (curmudelaobj && curmudelaobj->type == CHORD)
    {
      curmudelaobj->u.chordval.crescendo_begin_p
	= !curmudelaobj->u.chordval.crescendo_begin_p;
      si->haschanged = TRUE;
      gtk_widget_draw (si->scorearea, NULL);
    }
}

void
toggle_end_crescendo (struct scoreinfo *si)
{
  declarecurmudelaobj;

  if (curmudelaobj && curmudelaobj->type == CHORD)
    {
      curmudelaobj->u.chordval.crescendo_end_p
	= !curmudelaobj->u.chordval.crescendo_end_p;
      si->haschanged = TRUE;
      gtk_widget_draw (si->scorearea, NULL);
    }
}

void
toggle_start_diminuendo (struct scoreinfo *si)
{
  declarecurmudelaobj;

  if (curmudelaobj && curmudelaobj->type == CHORD)
    {
      curmudelaobj->u.chordval.diminuendo_begin_p
	= !curmudelaobj->u.chordval.diminuendo_begin_p;
      si->haschanged = TRUE;
      gtk_widget_draw (si->scorearea, NULL);
    }
}

void
toggle_end_diminuendo (struct scoreinfo *si)
{
  declarecurmudelaobj;

  if (curmudelaobj && curmudelaobj->type == CHORD)
    {
      curmudelaobj->u.chordval.diminuendo_end_p
	= !curmudelaobj->u.chordval.diminuendo_end_p;
      si->haschanged = TRUE;
      gtk_widget_draw (si->scorearea, NULL);
    }
}
