/** Translate a Decaf parse tree to MIPS assembler.
 * @author Shaun Jackman <sdj@sfu.ca>
 * @copyright Copyright 2004 Shaun Jackman
 */


#include "decaf.h"
#include "dictionary.h"
#include "parsetree.h"
#include "stack.h"
#include "syscall.h"
#include "util.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>


/** Register names. */
static const char* reg[] = { "$zero",
	"$at", "$v0", "$v1", "$a0", "$a1", "$a2", "$a3",
	"$t0", "$t1", "$t2", "$t3", "$t4", "$t5", "$t6", "$t7", 
	"$s0", "$s1", "$s2", "$s3", "$s4", "$s5", "$s6", "$s7", 
	"$t8", "$t9", "$k0", "$k1", "$gp", "$sp", "$fp", "$ra"
};
#define A0 4
#define S0 16
#define S7 23
#define NO_REG -1


/** Size of a word. */
#define SIZEOF_WORD 4


/** Returns a unique label. */
static int
new_label( void)
{
	static int label;
	return ++label;
}


/** Returns the value of this constant. */
static int
constant( const Tree* tree, const void* in)
{
	switch( $token) {
		case CONSTANT:
			return $(1,constant);
		case CHAR_CONSTANT: {
			char c = $lexeme[1];
			if( c == '\\') {
				c = $lexeme[2];
				if( strchr( "\\\"'", c) == NULL)
					c = unescape( c);
			}
			return c; }
		case INT_CONSTANT:
			return strtol( $lexeme, NULL, 0);
		case BOOL_CONSTANT: // FALSE | TRUE
			return $(1,token) == TRUE;
		default: die( "unexpected production");
	}
}


/** Returns the instruction for this binary operator. */
static const char*
op( const Tree* tree, const void* in)
{
	switch( $token) {
		case BIN_OP: // arith_op | rel_op | eq_op | cond_op
		case ARITH_OP: case REL_OP: case EQ_OP: case COND_OP:
			return $(1,op);
		case PLUS: return "add";
		case MINUS: return "sub";
		case TIMES: return "mul";
		case DIVIDE: return "div";
		case LSHIFT: return "sll";
		case RSHIFT: return "sra";
		case ROT: return "rol";
		case MOD: return "rem";
		case LT: return "slt";
		case GT: return "sgt";
		case LE: return "sle";
		case GE: return "sge";
		case EQ: return "seq";
		case NE: return "sne";
		case AND: return "and";
		case OR: return "or";
		case NOT: return "seq";
		default: die( "unexpected production");
	}
}


/** Symbol table. */
static Dictionary symbols;

/** Maximum number of symbols. */
#define SYMBOLS 1000

/** Offset of this symbol. */
static Stack offset[SYMBOLS];


/** Returns the offset of the specified symbol, or zero if it is a
 * global symbol. */
static int
get_offset( const char* lexeme)
{
	int i = find( &symbols, lexeme);
	return i == 0 || offset[i].count == 0 ? 0 :
		top( &offset[i]) * SIZEOF_WORD;
}


/** Checks that the specified register is in bounds. */
static inline void
check_reg( int reg)
{
	assumex( reg <= S7, "exhausted temporary registers");
}


/** Inherited attributes of an expression. */
typedef struct {
	/** Available temporary register. */
	int reg;

	/** Next argument register. */
	int arg;
} ExpressionAttr;


static int expression( const Tree* tree, ExpressionAttr* in);


/** Deal with read_int which uses pass by reference. */
static void
read_int( const Tree* tree, ExpressionAttr* in)
{
#define $1 $(1,read_int)
#define $2 $(2,read_int)
#define $4 $(4,read_int)
	switch( $token) {
	case METHOD_CALL: switch( $length) {
		case 5: // CALLOUT LPAREN STRING_CONSTANT callout_arg_list RPAREN
			printf( "\tli $v0, %d\n"
					"\tsyscall\n",
					READ_INT);
			$4;
			return;
		default: die( "unexpected production"); }
	case CALLOUT_ARG_LIST: switch( $length) {
		case 0: // epsilon
			die( "read_int takes one argument");
		case 3: // COMMA callout_arg callout_arg_list
			$2;
			return;
		default: die( "unexpected production"); }
	case EXPR: switch( $length) {
		case 1: switch( $(1,token)) {
			case LVALUE:
				printf( "\tsw $v0, 0(%s)\n", reg[$(1,expression)]);
				return; }
		default:
			die( "the first argument of read_int must be an lvalue"); }
	default: $all(read_int); }
#undef $1
#undef $2
#undef $4
}


/** Returns the register this expression resides in. */
static int
expression( const Tree* tree, ExpressionAttr* in)
{
#define $1 $(1,expression)
#define $2 $(2,expression)
#define $3 $(3,expression)
#define $4 $(4,expression)
	switch( $token) {
	case OPTIONAL_EXPR: switch( $length) {
		case 0: /* epsilon */ return NO_REG;
		case 1: /* expr */ return $1;
		default: die( "unexpected production"); }
	case METHOD_CALL: switch( $length) {
		case 4: // ID LPAREN optional_expr_list RPAREN
			{ ExpressionAttr* prev = in; ExpressionAttr In, *in = &In;
				in->reg = prev->reg;
				in->arg = A0;
				$3;
			}
			printf( "\tjal _%s\n"
					"\tmove %s, $v0\n",
					$(1,lexeme), reg[in->reg]);
			return in->reg;
		case 5: // CALLOUT LPAREN STRING_CONSTANT callout_arg_list RPAREN
			{ int syscall = find( &syscall_strings, $(3,lexeme));
				if( syscall != READ_INT) {
					{ ExpressionAttr* prev = in; ExpressionAttr In, *in = &In;
						in->reg = prev->reg;
						in->arg = A0;
						$4;
					}
					printf( "\tli $v0, %d\n"
							"\tsyscall\n"
							"\tmove %s, $v0\n",
							syscall, reg[in->reg]);
				} else
					$$(read_int);
			}
			return in->reg;
		default: die( "unexpected production"); }
	case OPTIONAL_EXPR_LIST:
		$all(expression);
		return NO_REG;
	case EXPR_LIST: { // expr
		int src = $1;
		int dst = in->arg++;
		if( $length > 1) {
			// expr COMMA expr_list
			check_reg( ++in->reg);
			$3;
			in->reg--;
		}
		printf( "\tmove %s, %s\n", reg[dst], reg[src]);
		return NO_REG; }
	case CALLOUT_ARG_LIST: switch( $length) {
		case 0: // epsilon
			return NO_REG;
		case 3: // COMMA callout_arg callout_arg_list
			printf( "\tmove %s, %s\n", reg[A0], reg[$2]);
			$3;
			return NO_REG;
		default: die( "unexpected production"); }
	case CALLOUT_ARG: // expr | STRING_CONSTANT
		return $1;
	case LVALUE: switch( $length) {
		case 1: // ID
			return $1;
		case 4: { // ID LBRACKET expr RBRACKET
			int dst = $3;
			printf( "\tsll %s, %s, 2\n", reg[dst], reg[dst]);
			check_reg( ++in->reg);
			printf( "\taddu %s, %s, %s\n", reg[dst], reg[dst], reg[$1]);
			in->reg--;
			return dst; }
		default: die( "unexpected production"); }
	case EXPR: switch( $length) {
		int dst;
		case 1: switch( $(1,token)) {
			case LVALUE:
				dst = $1;
				printf( "\tlw %s, 0(%s)\n", reg[dst], reg[dst]);
				return dst;
			case METHOD_CALL:
			case CONSTANT:
				return $1;
			default: die( "unexpected production"); }
		case 2: // MINUS expr | NOT expr
			dst = $2;
			printf( "\t%s %s, $zero, %s\n", $(1,op),
					reg[dst], reg[dst]);
			return dst;
		case 3:
			if( $(2,token) == BIN_OP) {
				// expr bin_op expr
				dst = $1;
				check_reg( ++in->reg);
				printf( "\t%s %s, %s, %s\n", $(2,op),
						reg[dst], reg[dst], reg[$3]);
				in->reg--;
				return dst;
			} else
			if( $(1,token) == LPAREN)
				// LPAREN expr RPAREN
				return $2;
			else
		default: die( "unexpected production"); }
	case STRING_CONSTANT: {
		int label = new_label();
		printf( "\t.data\n"
				"$L%d:\n"
				"\t.asciiz %s\n"
				"\t.text\n"
				"\tla %s, $L%d\n",
				label, $lexeme, reg[in->reg], label);
		return in->reg; }
	case CONSTANT: // INT_CONSTANT | CHAR_CONSTANT | bool_constant
		printf( "\tli %s, %d\n", reg[in->reg], $(1,constant));
		return in->reg;
	case ID: {
		int offset = get_offset( $lexeme);
		if( offset > 0)
			printf( "\tsubu %s, $fp, %d\n", reg[in->reg], offset);
		else
			printf( "\tla %s, _%s\n", reg[in->reg], $lexeme);
		return in->reg; }
	default: die( "unexpected production"); }
#undef $1
#undef $2
#undef $3
#undef $4
}


/** Inherited attributes of blocks. */
typedef struct {
	/** Depth of this frame. */
	int depth;

	/** Continue label. */
	int continue_label;

	/** Break label. */
	int break_label;
} BlockAttr;


static void block( const Tree* tree, BlockAttr* in);


/** Translates this statement. */
static void
statement( const Tree* tree, BlockAttr* in)
{
	switch( $token) {
	case ASSIGNMENT_LIST:
		$all( statement);
		return;
	case ASSIGNMENT: // lvalue ASSIGN expr
		{ ExpressionAttr In, *in = &In;
			int src, dst;
			in->reg = S0;
			src = $(3,expression);
			check_reg( ++in->reg);
			dst = $(1,expression);
			printf( "\tsw %s, 0(%s)\n", reg[src], reg[dst]);
		}
		return;
	case METHOD_CALL:
		{ ExpressionAttr In, *in = &In;
			in->reg = S0;
			$$(expression);
		}
		return;
	case BREAK:
		assumex( in->break_label > 0,
				"break outside of control block");
		printf( "\tj $L%d\n", in->break_label);
		return;
	case CONTINUE:
		assumex( in->continue_label > 0,
				"continue outside of control block");
		printf( "\tj $L%d\n", in->continue_label);
		return;
	case COMMA:
		return;
	case STATEMENT:
		break;
	default: die( "unexpected production"); }

	// This token is a statement.
	switch( $length) {
	case 1: // block
		$(1,block);
		return;
	case 2: // assignment SEMICOLON | method_call SEMICOLON |
		// BREAK SEMICOLON | CONTINUE SEMICOLON
		putchar( '#'); print_lexemes( tree); putchar( '\n');
		$(1,statement);
		return;
	case 3: // RETURN optional_expr SEMICOLON
		putchar( '#'); print_lexemes( tree); putchar( '\n');
		{ ExpressionAttr In, *in = &In;
			int src;
			in->reg = S0;
			src = $(2,expression);
			if( src != NO_REG)
				printf( "\tmove $v0, %s\n", reg[src]);
		}
		puts( "\tj return");
		return;
	case 5: // WHILE LPAREN expr RPAREN block
		{ BlockAttr* prev = in; BlockAttr In, *in = &In;
		in->depth = prev->depth;
		in->continue_label = new_label();
		in->break_label = new_label();
		printf( "$L%d:\n", in->continue_label);
		{ ExpressionAttr In, *in = &In;
			in->reg = S0;
			printf( "\tbeqz %s, ", reg[$(3,expression)]);
		}
		printf( "$L%d\n", in->break_label);
		$(5,block);
		printf( "\tj $L%d\n"
				"$L%d:\n", in->continue_label, in->break_label);
		return; }
	case 6: { // IF LPAREN expr RPAREN block optional_else_clause
		int else_label = new_label();
		int end_label = new_label();
		{ ExpressionAttr In, *in = &In;
			in->reg = S0;
			printf( "\tbeqz %s, $L%d\n",
					reg[$(3,expression)], else_label);
		}
		$(5,block);
		printf( "\tj $L%d\n", end_label);
		printf( "$L%d:\n", else_label);
		$(6,block);
		printf( "$L%d:\n", end_label);
		return; }
	case 9: // FOR LPAREN assignment_list SEMICOLON expr SEMICOLON
			//   assignment_list RPAREN block
		{ BlockAttr* prev = in; BlockAttr In, *in = &In;
		int loop_label = new_label();
		in->depth = prev->depth;
		in->continue_label = new_label();
		in->break_label = new_label();
		$(3,statement);
		printf( "$L%d:\n", loop_label);
		{ ExpressionAttr In, *in = &In;
			in->reg = S0;
			printf( "\tbeqz %s, ", reg[$(5,expression)]);
		}
		printf( "$L%d\n", in->break_label);
		$(9,block);
		printf( "$L%d:\n", in->continue_label);
		$(7,statement);
		printf( "\tj $L%d\n"
				"$L%d:\n", loop_label, in->break_label);
		return; }
	default: die( "unexpected production"); }
}


/** Adds this symbol to the symbol table. */
static void
add_symbol( const char* lexeme, int depth)
{
	int i = insert( &symbols, lexeme);
	assumex( i < SYMBOLS, "maximum of %d symbols reached", SYMBOLS);
	if( offset[i].max == 0)
		clear_stack( &offset[i]);
	push( &offset[i], depth);
}


/** Removes these symbols from the symbol table. */
static void
unscope( const Tree* tree, const void* in)
{
	switch( $token) {
	case ID: {
			int i = find( &symbols, $lexeme);
			assumex( i > 0, "unknown symbol");
			pop( &offset[i], 1);
		return; }
	default: $all( unscope); }
}


/** Translates this block. */
static void
block( const Tree* tree, BlockAttr* in)
{
#define $1 $(1,block)
#define $2 $(2,block)
#define $3 $(3,block)
	switch( $token) {
	case BLOCK: // LBRACE var_decls statements RBRACE
		$2;
		$3;
		$(2,unscope);
		return;
	case STATEMENT:
		$$(statement);
		return;
	case ID:
		add_symbol( $lexeme, ++in->depth);
		return;
	default: $all( block); }
#undef $1
#undef $2
#undef $3
}


/** Translates these method parameters. */
static void
parameters( const Tree* tree, BlockAttr* in)
{
	switch( $token) {
	case ID:
		printf( "\tsw %s, -%d($fp)\n",
				reg[A0 + in->depth], (in->depth+1) * SIZEOF_WORD);
		add_symbol( $lexeme, ++in->depth);
		return;
	default: $all( parameters); }
}


/** Returns the number of words needed by this frame. */
static int
frame( const Tree* tree, const void* in)
{
#define $4 $(4,frame)
#define $6 $(6,frame)
	switch( $token) {
		case METHOD_DECL:
			// type ID LPAREN optional_param_list RPAREN block
			return $4 + $6;
		case OPTIONAL_PARAM_LIST:
		case PARAM_LIST:
		case BLOCK:
		case VAR_DECLS:
		case VAR_DECL:
		case ID_LIST:
			return $sum( frame);
		case STATEMENTS:
		case STATEMENT:
		case OPTIONAL_ELSE_CLAUSE:
			return $max( frame);
		case PARAM:
		case ID:
			return 1;
		default:
			return 0;
	}
#undef $4
#undef $6
}


/** Translates this program. */
static void
translate( const Tree* tree, const void* in)
{
#define $1 $(1,translate)
#define $2 $(2,translate)
#define $3 $(3,translate)
#define $4 $(4,translate)
#define $5 $(5,translate)
	switch( $token) {
	case PROGRAM:
		// CLASS class_name LBRACE field_decls method_decls RBRACE
		puts( "\t.data");
		$4;
		puts( "\t.text");
		$5;
		return;
	case FIELD_DECL: switch( $length) {
		case 3: // type field_list SEMICOLON
			$2;
			return;
		case 5: // type ID ASSIGN constant SEMICOLON
			$2;
			printf( "\t.word %d\n", $(4,constant));
			return;
		default: die( "unexpected production");
	}
	case FIELD: switch( $length) {
		case 1: // ID
			$1;
			puts( "\t.word 0");
			return;
		case 4: // ID LBRACKET INT_CONSTANT RBRACKET
			$1;
			printf( "\t.space %d\n", SIZEOF_WORD * $(3,constant));
			return;
		default: die( "unexpected production");
	}
	case METHOD_DECL:
		// type ID LPAREN optional_param_list RPAREN block |
		// VOID ID LPAREN optional_param_list RPAREN block
		$2;
		puts( "\tsw $ra -4($sp)\n"
				"\tjal save");
		{
			int n = $$(frame);
			if( n > 0)
				printf( "\tsubu $sp, %d\n", n * SIZEOF_WORD);
		}
		{ BlockAttr In, *in = &In;
			in->depth = 0;
			$(4,parameters);
			in->continue_label = 0;
			in->break_label = 0;
			$(6,block);
		}
		$(4,unscope);
		puts( "\tj return");
		return;
	case ID:
		printf( "_%s:\n", $lexeme);
		return;
	default: $all( translate); }
#undef $1
#undef $2
#undef $3
#undef $4
#undef $5
}


/** Translates the specified Decaf parse tree to MIPS assembler. */
void
mips_translate_decaf( const Tree* tree)
{
	clear_dictionary( &symbols);
	translate( tree, NULL);

	// Support code.
	puts( "\t.text\n"
			"save:\n"
			"\tsubu $sp, 40\n"
			"\tsw $s0, 0($sp)\n"
			"\tsw $s1, 4($sp)\n"
			"\tsw $s2, 8($sp)\n"
			"\tsw $s3, 12($sp)\n"
			"\tsw $s4, 16($sp)\n"
			"\tsw $s5, 20($sp)\n"
			"\tsw $s6, 24($sp)\n"
			"\tsw $s7, 28($sp)\n"
			"\tsw $fp, 32($sp)\n"
			"\tmove $fp, $sp\n"
			"\tjr $ra\n"
			"return:\n"
			"\tmove $sp, $fp\n"
			"\tlw $s0, 0($sp)\n"
			"\tlw $s1, 4($sp)\n"
			"\tlw $s2, 8($sp)\n"
			"\tlw $s3, 12($sp)\n"
			"\tlw $s4, 16($sp)\n"
			"\tlw $s5, 20($sp)\n"
			"\tlw $s6, 24($sp)\n"
			"\tlw $s7, 28($sp)\n"
			"\tlw $fp, 32($sp)\n"
			"\tlw $ra, 36($sp)\n"
			"\taddu $sp, 40\n"
			"\tjr $ra\n"
			"\t.globl main\n"
			"main:\n"
			"\tj _main");
}
