#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/file.h>
#include <signal.h>
#include <math.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <linux/sem.h>		/* for the value of SEMVMX */
#include <dirent.h>
#include <string.h>
#include <gnome.h>

#include "misc_gtk.h"
#include "init_fnc.h"
#include "main.h"
#include "do_connect.h"
#include "dctc_process.h"
#include "status.h"
#include "gui_layout.h"
#include "bookmark.h"
#include "bdb.h"

GtkStyle *fast_style=NULL;				/* style for fast user */

char *lbl_chat[]={"pchat_label1", "pchat_label2", "pchat_label3",
						"pchat_label4", "pchat_label5", "pchat_label6",
						"pchat_label7", "pchat_label8", "pchat_label9"};

char *chat_text[]={	"chat1_text", "chat2_text", "chat3_text",
							"chat4_text", "chat5_text", "chat6_text",
							"chat7_text", "chat8_text", "chat9_text"};

/**********************************************************/
/* copie the given file (filename) into another file (fd) */
/* output: 0=ok, 1=error                                  */
/**********************************************************/
static int copy_file(char *filename, int dest_fd)
{
	char buf[512];
	FILE *f;
	int ln;
	int err=0;

	f=fopen(filename,"rb");
	if(f==NULL)
		return 1;

	while( (ln=fread(buf,1,sizeof(buf),f))!=0)
	{
		if(write(dest_fd,buf,ln)!=ln)
		{
			err=1;
			break;
		}
	}
	fclose(f);
	return err;
}

/*********************************************************************************************/
/* merge .done files into done+exited if there is no socket linked with them and remove them */
/*********************************************************************************************/
static void	clean_directory(char *base_dir)
{
	int fd;
	GString *p;
	GString *q;

	p=g_string_new(base_dir);
	q=g_string_new(NULL);
	g_string_sprintfa(p,"/done+exited");

	fd=open(p->str,O_CREAT|O_RDWR,0666);
	if(fd==-1)
	{
		perror("clean_directory - open fails");
	}
	else
	{
		if(flock(fd,LOCK_EX)!=0)
		{
			perror("clean_directory - lock fails");
		}
		else
		{
			DIR *dir;
			struct stat st;

			lseek(fd,0,SEEK_END);

			dir=opendir(base_dir);
			if(dir!=NULL)
			{
				struct dirent *obj;

				while((obj=readdir(dir))!=NULL)
				{
					int a;

					if(strncmp(obj->d_name,"dctc-",5))
						continue;

					a=strlen(obj->d_name);
					if(a<5)
						continue;

					if(!strcmp(obj->d_name+a-4,".udp"))		/* ignore .udp file */
						continue;

					if(!strcmp(obj->d_name+a-5,".done"))
					{
						g_string_sprintf(q,"%s/%s",base_dir,obj->d_name);
						q=g_string_truncate(q,q->len-5);		/* remove the .done */
	
						if(stat(q->str,&st))
						{
							process_no_more_exist:
							g_string_sprintf(q,"%s/%s",base_dir,obj->d_name);
							/* copy q->str into fd */
							if(!copy_file(q->str,fd))
								unlink(q->str);
						}
						else
						{
							/* unix socket exist but does the process exists ? */
							pid_t num;
	
							/* extract the number */
							if(sscanf(obj->d_name+5,"%08X",&num)==1)
							{
								if(kill(num,SIGQUIT)!=0)
								{
									/* process is dead, remove the socket */
									unlink(q->str);

									/* and the udp socket */
									q=g_string_append(q,".udp");
									unlink(q->str);

									del_client_status(num);

									goto process_no_more_exist;
								}
							}
						}
					}
					else
					{
						/* well, the file start with dctc but has no .done at the end */
						/* it is the socket */
						pid_t num;
	
						/* extract the number */
						if(sscanf(obj->d_name+5,"%08X",&num)==1)
						{
							if(kill(num,SIGQUIT)!=0)
							{
								g_string_sprintf(q,"%s/%s",base_dir,obj->d_name);
								/* process is dead, remove the socket */
								unlink(q->str);

								/* and the udp socket */
								{
									GString *p;

									p=g_string_new(q->str);
									p=g_string_append(p,".udp");
									unlink(p->str);
									g_string_free(p,TRUE);
								}

								q=g_string_append(q,".done");
								/* copy q->str into fd */
								if(!copy_file(q->str,fd))
									unlink(q->str);

								del_client_status(num);
							}
						}
					}
				}
				closedir(dir);
			}
			flock(fd,LOCK_UN);
		}
		close(fd);
	}

	g_string_free(p,TRUE);
	g_string_free(q,TRUE);
}

/***********************************************************************/
/* search inside the "hub_recent_clist" a hub having the given address */
/***********************************************************************/
static char *get_hubname_from_recent(char *hub_addr)
{
	GtkWidget *rhcw;

	rhcw=get_widget_by_widget_name("hub_recent_clist");
	if(rhcw)
	{
		GtkCList *clst;
		int i;

		clst=GTK_CLIST(rhcw);
		for(i=0;i<clst->rows;i++)
		{
			char *t;

			gtk_clist_get_text(clst,i,3,&t);
			if((t!=NULL)&&(!strcmp(t,hub_addr)))
			{
				gtk_clist_get_text(clst,i,0,&t);
				return t;
			}
		}
	}
	return NULL;
}

/*********************************************************************************************/
/* fill the clist named "csearch_clist" with the list of all files of the download directory */
/*********************************************************************************************/
void fill_content_search_clist(void)
{
	GtkWidget *rhcw;

	rhcw=get_widget_by_widget_name("csearch_clist");
	if(rhcw)
	{
		const char *cur_dir;

		GtkCList *clst;

		clst=GTK_CLIST(rhcw);
		gtk_clist_freeze(clst);
		gtk_clist_clear(clst);

		cur_dir=get_var("dl_path");
		if(cur_dir!=NULL)
		{
			DIR *dir;

			dir=opendir(cur_dir);
			if(dir!=NULL)
			{
				struct dirent *obj;

				while((obj=readdir(dir))!=NULL)
				{
					char *ent[2];
					unsigned long fsize;
					char fsize_txt[64];

					if(obj->d_name[0]=='.')
						continue;

					fsize=get_file_size(cur_dir,obj->d_name);

					sprintf(fsize_txt,"%lu",fsize);
					ent[0]=obj->d_name;
					ent[1]=fsize_txt;

					gtk_clist_append(clst,ent);
				}
				closedir(dir);
			}
		}

		gtk_clist_sort(clst);
		gtk_clist_thaw(clst);
	}
}

/********************************************************************************/
/* fill the clist named "running_hub_clist" with the list of all running client */
/********************************************************************************/
void fill_running_hub_clist(void)
{
	GtkWidget *rhcw;

	rhcw=get_widget_by_widget_name("running_hub_clist");
	if(rhcw)
	{
		gtk_clist_freeze(GTK_CLIST(rhcw));
		gtk_clist_clear(GTK_CLIST(rhcw));

		clean_directory(dctc_dir->str);

		/* we read the directory ~/.dctc/running */
		/* each entry has the format "dctc-xxxxxxxx-aaaaaaaaaaaa" where xxxxxxxx is the dctc pid (8 hexdigits) and aaaaaaaaa is the hubname */
		{
			DIR *dir;

			dir=opendir(dctc_dir->str);
			if(dir!=NULL)
			{
				struct dirent *obj;

				while((obj=readdir(dir))!=NULL)
				{
					char *ent[6];
					int a;
					char *hn;
					int row_num;
					int stt;
					unsigned int the_pid;
					char str_gdl[20];
					char str_users[20];
					unsigned long cli_param[NB_LONG_PER_ENTRY];
					GdkColor *row_col;

					if(strncmp(obj->d_name,"dctc-",5))
						continue;

					a=strlen(obj->d_name);
					if((a>5)&&(!strcmp(obj->d_name+a-5,".done")))
						continue;
					if((a>4)&&(!strcmp(obj->d_name+a-4,".udp")))
						continue;

					hn=get_hubname_from_recent(obj->d_name+5+8+1);

					ent[0]=obj->d_name+5+8+1;
					ent[1]=obj->d_name;

					if(hn!=NULL)
						ent[2]=hn;
					else
						ent[2]="";

					sscanf(obj->d_name+5,"%08X",&the_pid);
					stt=get_client_status(the_pid,cli_param);
					switch(stt)
					{
						default:
						case NOT_EXIST:	ent[3]="";
												ent[4]="";
												ent[5]="";
												row_col=&white;
												break;

						case IS_OFFLINE:
												ent[3]="0";
												ent[4]=str_gdl;
												ent[5]=(((FLAG1_STRUCT)(cli_param[GSTATUS_FLAG1])).bf.is_clock_master)?_("Master"):"";
												sprintf(str_gdl,"%lu",cli_param[GSTATUS_NB_GDL]);
												row_col=&light_orange;
												break;

						case IS_ONLINE:
												ent[3]=str_users;
												sprintf(str_users,"%lu",cli_param[GSTATUS_NB_USERS]);
												ent[4]=str_gdl;
												ent[5]=(((FLAG1_STRUCT)(cli_param[GSTATUS_FLAG1])).bf.is_clock_master)?_("Master"):"";
												sprintf(str_gdl,"%lu",cli_param[GSTATUS_NB_GDL]);
												row_col=&green;
												break;

					}
					row_num=gtk_clist_append(GTK_CLIST(rhcw),ent);
					gtk_clist_set_background(GTK_CLIST(rhcw),row_num,row_col);
				}
				closedir(dir);
			}
		}

		gtk_clist_thaw(GTK_CLIST(rhcw));
	}

	colorize_favorite(GTK_CLIST(get_widget_by_widget_name("hub_favorite_clist")),GTK_CLIST(rhcw));
}

/*******************************************************************************/
/* fill the clist named "hub_recent_clist" with the list of all running client */
/*******************************************************************************/
void fill_recent_hub_clist(void)
{
	GtkWidget *rhcw;

	rhcw=get_widget_by_widget_name("hub_recent_clist");
	if(rhcw)
	{
		FILE *f;
		GString *s;
		char *path;

		gtk_clist_freeze(GTK_CLIST(rhcw));
		gtk_clist_clear(GTK_CLIST(rhcw));
		gtk_clist_set_auto_sort(GTK_CLIST(rhcw),TRUE);

		s=g_string_new(NULL);
		path=getenv("HOME");
		g_string_sprintf(s,"%s/.dctc/recent",(path!=NULL)?path:".");

		/* we read the file ~/.dctc/recent */
		f=fopen(s->str,"rb");
		if(f!=NULL)
		{
			char *ent[4];
			char buf[51200];
			char *t;
			GStringChunk *gsc;
			GPtrArray *gpa;

			gpa=g_ptr_array_new();
			gsc=g_string_chunk_new(48);

			ent[1]="";
			ent[2]="";
			ent[3]=buf;
		
			while(fgets(buf,sizeof(buf),f)!=NULL)
			{
				ent[0]=strchr(buf,'|');
				if(ent[0]!=NULL)
				{
					int i;
					int fnd=0;

					*(ent[0])++='\0';
					t=strchr(ent[0],'\n');
					if(t!=NULL)
						*t='\0';

					/* site already added ? */
					for(i=0;i<gpa->len;i++)
					{
						char *p;

						p=g_ptr_array_index(gpa,i);
						if(!strcmp(p,ent[0]))
						{
							fnd=1;
							break;
						}
					}

					if(!fnd)
					{
						gtk_clist_append(GTK_CLIST(rhcw),ent);

						/* add the site name to the list of already added site */
						g_ptr_array_add(gpa,g_string_chunk_insert(gsc,ent[0]));
					}
				}
			}

			if(gpa!=NULL)
				g_ptr_array_free(gpa,TRUE);
			if(gsc!=NULL)
				g_string_chunk_free(gsc);

			fclose(f);
		}
		g_string_free(s,TRUE);
		gtk_clist_sort(GTK_CLIST(rhcw));
		gtk_clist_thaw(GTK_CLIST(rhcw));
	}
}

/*************************************************************************************/
/* this function updates the gnome vector containing seen hub to include the new one */
/*************************************************************************************/
void update_seen_hub_list(GPtrArray *new_seen)
{
	int i;
	GString *tmp;

	tmp=g_string_new("");

	for(i=0;i<new_seen->len;i++)
	{
		char *v;
		char *toadd;

		toadd=g_ptr_array_index(new_seen,i);

		/* build the key */
		v=strchr(toadd,'|')+1;
		tmp=g_string_assign(tmp,v);

		v=strchr(tmp->str,'|');
		tmp=g_string_truncate(tmp,v-tmp->str);

		/* to ease future usage of values, the key and its value is stored with the trailing '\0' (==C string format)*/
		set_key_data(seen_hub_db,tmp->str,tmp->len+1,toadd,strlen(toadd)+1);
	}
}

/**********************************************************/
/* build hublist command line according to GUI parameters */
/**********************************************************/
static GString *build_list_cmd_line(void)
{
	GString *cmdline;
	int valid_len;

	int i;
	GtkWidget *w;
	char *t;
	char *cnxtype;
	char *hubcnxtype[3]={_("No proxy needed"), _("Use SOCKS parameters"), _("Use Web proxy")};

	cmdline=g_string_new("hublist");
	cnxtype=gtk_entry_get_text(GTK_ENTRY(get_widget_by_widget_name("hublist_cnxtype_entry")));
	for(i=0;i<3;i++)
	{
		if(!strcmp(cnxtype,hubcnxtype[i]))
			break;
	}

	valid_len=cmdline->len;
	switch(i)
	{
		case 0:		/* no proxy needed */
						break;

		case 1:		/* use socks param */
						cmdline=g_string_append(cmdline," --socks");
						{
							/* add socks proxy address */
							w=get_widget_by_widget_name("socks_address_entry");
							if(w==NULL)
								break;
							
							t=gtk_entry_get_text(GTK_ENTRY(w));
							if((t==NULL)||(strlen(t)==0))
								break;

							cmdline=g_string_append_c(cmdline,' ');
							cmdline=g_string_append(cmdline,t);
						}
						{
							/* add socks proxy port */
							w=get_widget_by_widget_name("socks_port_entry");
							if(w==NULL)
								break;
							
							t=gtk_entry_get_text(GTK_ENTRY(w));
							if((t==NULL)||(strlen(t)==0))
								break;

							cmdline=g_string_append_c(cmdline,' ');
							cmdline=g_string_append(cmdline,t);

							valid_len=cmdline->len;
						}
						{
							/* add socks user ID (if exist) */
							w=get_widget_by_widget_name("socks_userid_entry");
							if(w==NULL)
								break;
							
							t=gtk_entry_get_text(GTK_ENTRY(w));
							if((t==NULL)||(strlen(t)==0))
								break;

							cmdline=g_string_append_c(cmdline,' ');
							cmdline=g_string_append(cmdline,t);

							valid_len=cmdline->len;
						}
						break;

		case 2:		/* use web proxy */
						cmdline=g_string_append(cmdline," --proxy");
						{
							/* add web proxy address */
							w=get_widget_by_widget_name("web_proxy_host_entry");
							if(w==NULL)
								break;
							
							t=gtk_entry_get_text(GTK_ENTRY(w));
							if((t==NULL)||(strlen(t)==0))
								break;

							cmdline=g_string_append_c(cmdline,' ');
							cmdline=g_string_append(cmdline,t);
						}
						{
							/* add socks proxy port */
							w=get_widget_by_widget_name("web_proxy_port_entry");
							if(w==NULL)
								break;
							
							t=gtk_entry_get_text(GTK_ENTRY(w));
							if((t==NULL)||(strlen(t)==0))
								break;

							cmdline=g_string_append_c(cmdline,' ');
							cmdline=g_string_append(cmdline,t);

							valid_len=cmdline->len;
						}
						break;

		default:		/* eh ??? */
						break;
	}

	/* discard any incomplete parameter */
	g_string_truncate(cmdline,valid_len);
	return cmdline;
}

/**************************************************************************/
/* fill the clist named "hub_public_clist" with the list of existing hubs */
/**************************************************************************/
void fill_pub_hub_clist(int flag)
{
	GtkWidget *rhcw;
	GStringChunk *gsc;
	GPtrArray *gpa;

	gpa=g_ptr_array_new();
	gsc=g_string_chunk_new(128);

	rhcw=get_widget_by_widget_name("hub_public_clist");
	if(rhcw)
	{
		FILE *f;
		GString *hublist_cmd_line;

		if((flag==FALSE)&&(GTK_CLIST(rhcw)->rows!=0))
			return;

		gtk_clist_freeze(GTK_CLIST(rhcw));
		gtk_clist_clear(GTK_CLIST(rhcw));

		hublist_cmd_line=build_list_cmd_line();
		f=popen(hublist_cmd_line->str,"r");
		g_string_free(hublist_cmd_line,TRUE);
		if(f!=NULL)
		{
			char *ent[4];
			char buf[51200];
			char *t;

			while(fgets(buf,sizeof(buf),f)!=NULL)
			{
				char buf_cpy[51200];

				strcpy(buf_cpy,buf);

				ent[0]=buf;

				t=strchr(buf,'|');
				if(t==NULL)
					continue;
				*t++='\0';

				ent[3]=t;
				t=strchr(t,'|');
				if(t==NULL)
					continue;
				*t++='\0';

				ent[2]=t;
				t=strchr(t,'|');
				if(t==NULL)
					continue;
				*t++='\0';

				ent[1]=t;
				t=strchr(t,'|');
				if(t==NULL)
					continue;
				*t++='\0';

				gtk_clist_append(GTK_CLIST(rhcw),ent);

				/* perform all pending redraw, not very useful but it is a lot nicer :) */
				while (g_main_iteration(FALSE));

				g_ptr_array_add(gpa,g_string_chunk_insert(gsc,buf_cpy));
			}
			pclose(f);
		}
		gtk_clist_sort(GTK_CLIST(rhcw));
		gtk_clist_thaw(GTK_CLIST(rhcw));
	}

	if(gpa)
		update_seen_hub_list(gpa);

	if(gpa)
		g_ptr_array_free(gpa,TRUE);
	if(gsc)
		g_string_chunk_free(gsc);
}

/********************************************************************/
/* fill the clist named "seen_hub_clist" with the list of seen hubs */
/********************************************************************/
void fill_seen_hub_clist(int flag)
{
	GtkWidget *rhcw;

	rhcw=get_widget_by_widget_name("seen_hub_clist");
	if(rhcw)
	{
		DBC *cursor;
		int ret;

		if((flag==FALSE)&&(GTK_CLIST(rhcw)->rows!=0))
			return;

		gtk_clist_freeze(GTK_CLIST(rhcw));
		gtk_clist_clear(GTK_CLIST(rhcw));

		/* we must get all keys of seen_hub berkeley Database */
		/* the value of the keys are to put in buf and to process */
		ret=seen_hub_db->cursor(seen_hub_db,NULL,&cursor,0);
		if(ret==0)
		{
			DBT key;
			DBT data;
			char buf1[8192];
			char buf2[8192];

			memset(&key,0,sizeof(key));
			memset(&data,0,sizeof(data));

			key.data=buf1;
			key.ulen=sizeof(buf1)-1;
			key.flags=DB_DBT_USERMEM;

			data.data=buf2;
			data.ulen=sizeof(buf2)-1;
			data.flags=DB_DBT_USERMEM;

			ret=cursor->c_get(cursor,&key,&data,DB_FIRST);
			while(ret==0)
			{
				char *ent[4];
				char *t;

				/* should not be useful but who knows what appends if someone enters invalid keys */
				buf2[data.size]='\0';

				ent[0]=buf2;
	
				t=strchr(buf2,'|');
				if(t==NULL)
					continue;
				*t++='\0';

				ent[3]=t;
				t=strchr(t,'|');
				if(t==NULL)
					continue;
				*t++='\0';

				ent[2]=t;
				t=strchr(t,'|');
				if(t==NULL)
					continue;
				*t++='\0';

				ent[1]=t;
				t=strchr(t,'|');
				if(t==NULL)
					continue;
				*t++='\0';

				gtk_clist_append(GTK_CLIST(rhcw),ent);

				ret=cursor->c_get(cursor,&key,&data,DB_NEXT);
			}

			/* end of scan, destroy the cursor */
			cursor->c_close(cursor);
		}

		gtk_clist_sort(GTK_CLIST(rhcw));
		gtk_clist_thaw(GTK_CLIST(rhcw));
	}
}


static gint hub_pub_comp(GtkCList *clist, gconstpointer p1, gconstpointer p2)
{
	GtkCListRow *row1 = (GtkCListRow *) p1;
	GtkCListRow *row2 = (GtkCListRow *) p2;
	char *text1=NULL;
	char *text2=NULL;
	int col;

	switch((col=clist->sort_column))
	{
		case 1:	/* #users */
					text1=GTK_CELL_TEXT(row1->cell[col])->text;
					text2=GTK_CELL_TEXT(row2->cell[col])->text;
					if(text1&&text2)
					{
						unsigned long v1,v2;
						sscanf(text1,"%lu",&v1);
						sscanf(text2,"%lu",&v2);
						/* don't replace the following code by return v1-v2; because v1 and v2 are unsigned */
						if(v1<v2)
							return -1;
						if(v1==v2)
							return 0;
						return 1;
					}
					return -1;


		default:
					col=0;		/* use hubname as default */
		case 0:	/* hubname */
		case 2:	/* description */
		case 3:	/* hub address */
					text1=GTK_CELL_TEXT(row1->cell[col])->text;
					text2=GTK_CELL_TEXT(row2->cell[col])->text;
					if(text1&&text2)
						return strcmp(text1,text2);
					return -1;
	}
}

static gint dl_clist_comp(GtkCList *clist, gconstpointer p1, gconstpointer p2)
{
	GtkCListRow *row1 = (GtkCListRow *) p1;
	GtkCListRow *row2 = (GtkCListRow *) p2;
	char *text1=NULL;
	char *text2=NULL;
	int col;

	switch((col=clist->sort_column))
	{
		case 2:	/* size */
					text1=GTK_CELL_TEXT(row1->cell[col])->text;
					text2=GTK_CELL_TEXT(row2->cell[col])->text;
					if(text1&&text2)
					{
						unsigned long v1,v2;
						sscanf(text1,"%lu",&v1);
						sscanf(text2,"%lu",&v2);
						/* don't replace the following code by return v1-v2; because v1 and v2 are unsigned */
						if(v1<v2)
							return -1;
						if(v1==v2)
							return 0;
						return 1;
					}
					return -1;

		default:
					col=1;		/* unknown column, use the nick */

		case 0:	/* thread ID (even if the thread is a number, due to the fact */
					/* all thread ID have the same length, we can do a string compare */
		case 1:	/* nick */
		case 3:	/* speed */
		case 4:	/* filename */
		case 5:	/* local file */
					text1=GTK_CELL_TEXT(row1->cell[col])->text;
					text2=GTK_CELL_TEXT(row2->cell[col])->text;
					if(text1&&text2)
						return strcmp(text1,text2);
					return -1;
	}
}

static gint user_clist_comp(GtkCList *clist, gconstpointer p1, gconstpointer p2)
{
  GtkCListRow *row1 = (GtkCListRow *) p1;
  GtkCListRow *row2 = (GtkCListRow *) p2;
  char *text1=NULL;
  char *text2=NULL;
  int col;

  switch((col=clist->sort_column))
  {
    case 2:	/* size */
      text1=GTK_CELL_TEXT(row1->cell[col])->text;
      text2=GTK_CELL_TEXT(row2->cell[col])->text;
      if(text1&&text2)
      {
        float v1=0;
        float v2=0;
        unsigned int i;
        char unit1[3];
        char unit2[3];
        double scale[4]={1.0,1024.0, 1024.0*1024.0, 1024.0*1024.0*1024.0};
        char *unit_name[4]={"B","KB","MB","GB"};
        if(sscanf(text1,"%f%2s",&v1,unit1))
        {
          for(i=0;i<4;i++)
          {
            if(strcmp(unit1,unit_name[i]) == 0)
            {
              v1=v1*scale[i];
            }
          }
        }
        if(sscanf(text2,"%f%2s",&v2,unit2))
        {
          for(i=0;i<4;i++)
          {
            if(strcmp(unit2,unit_name[i]) == 0)
            {
              v2=v2*scale[i];
            }
          }
        }
        /* don't replace the following code by return v1-v2; because v1 and v2 are unsigned */
        if(v1<v2)
          return -1;
        if(v1==v2)
          return 0;
        return 1;
      }
      return -1;
    default:
      text1=GTK_CELL_TEXT(row1->cell[col])->text;
      text2=GTK_CELL_TEXT(row2->cell[col])->text;
      if(text1&&text2)
        return strcmp(text1,text2);
      return -1;
 }
}

static gint clist_comp_num_is_2(GtkCList *clist, gconstpointer p1, gconstpointer p2)
{
	GtkCListRow *row1 = (GtkCListRow *) p1;
	GtkCListRow *row2 = (GtkCListRow *) p2;
	char *text1=NULL;
	char *text2=NULL;
	int col;

	switch((col=clist->sort_column))
	{
		case 2:	/* size */
					text1=GTK_CELL_TEXT(row1->cell[col])->text;
					text2=GTK_CELL_TEXT(row2->cell[col])->text;
					if(text1&&text2)
					{
						unsigned long v1,v2;
						sscanf(text1,"%lu",&v1);
						sscanf(text2,"%lu",&v2);
						/* don't replace the following code by return v1-v2; because v1 and v2 are unsigned */
						if(v1<v2)
							return -1;
						if(v1==v2)
							return 0;
						return 1;
					}
					return -1;

		default:
					text1=GTK_CELL_TEXT(row1->cell[col])->text;
					text2=GTK_CELL_TEXT(row2->cell[col])->text;
					if(text1&&text2)
						return strcmp(text1,text2);
					return -1;
	}
}

static gint clist_comp_num_is_1(GtkCList *clist, gconstpointer p1, gconstpointer p2)
{
	GtkCListRow *row1 = (GtkCListRow *) p1;
	GtkCListRow *row2 = (GtkCListRow *) p2;
	char *text1=NULL;
	char *text2=NULL;
	int col;

	switch((col=clist->sort_column))
	{
		case 1:	/* size */
					text1=GTK_CELL_TEXT(row1->cell[col])->text;
					text2=GTK_CELL_TEXT(row2->cell[col])->text;
					if(text1&&text2)
					{
						unsigned long v1,v2;
						sscanf(text1,"%lu",&v1);
						sscanf(text2,"%lu",&v2);
						/* don't replace the following code by return v1-v2; because v1 and v2 are unsigned */
						if(v1<v2)
							return -1;
						if(v1==v2)
							return 0;
						return 1;
					}
					return -1;

		default:
					text1=GTK_CELL_TEXT(row1->cell[col])->text;
					text2=GTK_CELL_TEXT(row2->cell[col])->text;
					if(text1&&text2)
						return strcmp(text1,text2);
					return -1;
	}
}

static gint ctree_comp_num_is_1(GtkCList *clist, gconstpointer p1, gconstpointer p2)
{
	GtkCListRow *row1 = (GtkCListRow *) p1;
	GtkCListRow *row2 = (GtkCListRow *) p2;
	char *text1=NULL;
	char *text2=NULL;
	int col;

	switch((col=clist->sort_column))
	{
		case 1:	/* size */
					if( (GTK_CELL_TEXT(row1->cell[col])->type==GTK_CELL_TEXT) &&
						 (GTK_CELL_TEXT(row2->cell[col])->type==GTK_CELL_TEXT) )
					{
						/* the size can be CELL_EMPTY (for non leaf node), that's why we verify the cell type here */
						text1=GTK_CELL_TEXT(row1->cell[col])->text;
						text2=GTK_CELL_TEXT(row2->cell[col])->text;
						if(text1&&text2)
						{
							unsigned long v1,v2;

							sscanf(text1,"%lu",&v1);
							sscanf(text2,"%lu",&v2);
							/* don't replace the following code by return v1-v2; because v1 and v2 are unsigned */
							if(v1<v2)
								return -1;
							if(v1==v2)
								return 0;
							return 1;
						}
					}
					col=0;		/* else, sort by filename */
					

		default:
					text1=GTK_CELL_PIXTEXT(row1->cell[col])->text;
					text2=GTK_CELL_PIXTEXT(row2->cell[col])->text;
					if(text1&&text2)
						return strcmp(text1,text2);
					return -1;
	}
}


static gint	gdl_ctree_sort_fnc(GtkCList *clist, gconstpointer p1, gconstpointer p2)
{
	GtkCListRow *row1 = (GtkCListRow *) p1;
	GtkCListRow *row2 = (GtkCListRow *) p2;
	char *text1=NULL;
	char *text2=NULL;
	int col;

	switch((col=clist->sort_column))
	{
		case 0:	/* 1st column: be careful, on a CTREE, it is always a PIXTEXT, not a TEXT */
					and_by_name:
					text1=GTK_CELL_PIXTEXT(row1->cell[col])->text;
					text2=GTK_CELL_PIXTEXT(row2->cell[col])->text;
					if(text1&&text2)
					{
						int l1,l2;

						l1=strlen(text1);
						l2=strlen(text2);

						if((l1==0)&&(l2==0))
							return 0;

						if((l1==0)&&(l2!=0))
							return 1;
						if((l1!=0)&&(l2==0))
							return -1;

						return strcmp(text1,text2);
					}
					return -1;
				
		case 2:	/* size and range column */
					text1=GTK_CELL_TEXT(row1->cell[col])->text;
					text2=GTK_CELL_TEXT(row2->cell[col])->text;
					if(text1&&text2)
					{
						if((text1[0]!='[')&&(text2[0]!='['))		/* size only */
						{
							unsigned long v1,v2;
							sscanf(text1,"%lu",&v1);
							sscanf(text2,"%lu",&v2);
							/* don't replace the following code by return v1-v2; because v1 and v2 are unsigned */
							if(v1<v2)
								return -1;
							if(v1==v2)
								goto and_by_name;			/* sort by size and then by name if size are equal */
							return 1;
						}
						else if((text1[0]!='[')&&(text2[0]=='['))      /* 1st is size 2nd is range */
						{
							return -1;	/* size first */
						}
						else if((text1[0]=='[')&&(text2[0]!='['))      /* 2nd is size 1st is range */
						{
							return 1;	/* size first */
						}
						else if((text1[0]=='[')&&(text2[0]=='['))		/* range only (don't simplify this test, there are other posible strings */
						{
							unsigned long v1,v2;
							sscanf(text1+1,"%lu",&v1);
							sscanf(text2+1,"%lu",&v2);
							/* don't replace the following code by return v1-v2; because v1 and v2 are unsigned */
							if(v1<v2)
								return -1;
							if(v1==v2)
								goto and_by_name;			/* sort by size and then by name if size are equal */
							return 1;
						}
					}
					return -1;

		default:
					text1=GTK_CELL_TEXT(row1->cell[col])->text;
					text2=GTK_CELL_TEXT(row2->cell[col])->text;
					if(text1&&text2)
					{
						int ret=strcmp(text1,text2);
						if(ret==0)
							goto and_by_name;
						return ret;
					}
					return -1;
	}
}

/********************************************************************************************************************/
/* the following function makes some adjustements that cannot be done inside glade (or I don't know how to do them) */
/********************************************************************************************************************/
void init_clist(void)
{
	GtkWidget *w;
	int i;

	w=get_widget_by_widget_name("running_hub_clist");
	if(w!=NULL)
	{
		gtk_clist_set_auto_sort(GTK_CLIST(w),TRUE);
		gtk_clist_set_column_visibility(GTK_CLIST(w),1,FALSE);		/* hide process_id column */
		gtk_clist_set_column_justification(GTK_CLIST(w),3,GTK_JUSTIFY_RIGHT);
		gtk_clist_set_column_justification(GTK_CLIST(w),4,GTK_JUSTIFY_RIGHT);
	}

	w=get_widget_by_widget_name("download_clist");
	if(w!=NULL)
	{
		gtk_clist_freeze(GTK_CLIST(w));
		gtk_clist_clear(GTK_CLIST(w));
		gtk_clist_set_column_visibility(GTK_CLIST(w),0,FALSE);
		gtk_clist_set_compare_func(GTK_CLIST(w),dl_clist_comp);
		gtk_clist_set_auto_sort(GTK_CLIST(w),TRUE);
		gtk_clist_set_column_justification(GTK_CLIST(w),2,GTK_JUSTIFY_RIGHT);
		gtk_clist_set_column_justification(GTK_CLIST(w),3,GTK_JUSTIFY_RIGHT);
		gtk_clist_thaw(GTK_CLIST(w));
	}
	w=get_widget_by_widget_name("upload_clist");
	if(w!=NULL)
	{
		gtk_clist_freeze(GTK_CLIST(w));
		gtk_clist_clear(GTK_CLIST(w));
		gtk_clist_set_column_visibility(GTK_CLIST(w),0,FALSE);
		gtk_clist_set_compare_func(GTK_CLIST(w),dl_clist_comp);			/* works because order of columns are identical, only less columns */
		gtk_clist_set_auto_sort(GTK_CLIST(w),TRUE);
		gtk_clist_set_column_justification(GTK_CLIST(w),2,GTK_JUSTIFY_RIGHT);
		gtk_clist_set_column_justification(GTK_CLIST(w),3,GTK_JUSTIFY_RIGHT);
		gtk_clist_thaw(GTK_CLIST(w));
	}
	w=get_widget_by_widget_name("queue_clist");
	if(w!=NULL)
	{
		gtk_clist_freeze(GTK_CLIST(w));
		gtk_clist_clear(GTK_CLIST(w));
		gtk_clist_set_column_visibility(GTK_CLIST(w),0,FALSE);
		gtk_clist_set_auto_sort(GTK_CLIST(w),TRUE);
		gtk_clist_set_column_visibility(GTK_CLIST(w),3,FALSE);						/* hide speed column (it is empty) */
#if 0
		gtk_clist_set_column_justification(GTK_CLIST(w),3,GTK_JUSTIFY_RIGHT);	/* right align size column */
#endif
		gtk_clist_thaw(GTK_CLIST(w));
	}

	w=get_widget_by_widget_name("done_clist");
	if(w!=NULL)
	{
		gtk_clist_set_column_visibility(GTK_CLIST(w),2,FALSE);						/* hide speed column (it is empty) */
		gtk_clist_set_column_justification(GTK_CLIST(w),1,GTK_JUSTIFY_RIGHT);	/* right align size column */
	}

	{
		GString *str;

		str=g_string_new(dctc_dir->str);
		g_string_sprintfa(str,"/done+exited");
		load_done_xfer(str->str,0);
		g_string_free(str,TRUE);
	}

	w=get_widget_by_widget_name("find_result");
	if(w)
	{
#if 0
		gtk_clist_set_auto_sort(GTK_CLIST(w),TRUE);
#endif
		gtk_clist_set_column_justification(GTK_CLIST(w),2,GTK_JUSTIFY_RIGHT);
		gtk_clist_set_compare_func(GTK_CLIST(w),clist_comp_num_is_2);
	}

	w=get_widget_by_widget_name("csearch_clist");
	if(w)
	{
		gtk_clist_set_compare_func(GTK_CLIST(w),clist_comp_num_is_1);
		gtk_clist_set_column_justification(GTK_CLIST(w),1,GTK_JUSTIFY_RIGHT);
	}

	w=get_widget_by_widget_name("user_file_list_clist");
	if(w)
	{
		gtk_clist_set_compare_func(GTK_CLIST(w),ctree_comp_num_is_1);
		gtk_clist_set_column_justification(GTK_CLIST(w),1,GTK_JUSTIFY_RIGHT);
		gtk_clist_set_column_visibility(GTK_CLIST(w),2,FALSE);						/* hide the full filename column */
		gtk_clist_set_column_visibility(GTK_CLIST(w),3,FALSE);						/* hide the original nick column */
	}

	w=get_widget_by_widget_name("gdl_ctree");
	if(w)
	{
		gtk_clist_set_compare_func(GTK_CLIST(w),gdl_ctree_sort_fnc);
		gtk_clist_set_column_justification(GTK_CLIST(w),2,GTK_JUSTIFY_RIGHT);
	}

	/* create a special style for fast user */

	w=get_widget_by_widget_name("user_clist");
	if(w)
	{
		fast_style=gtk_style_copy(GTK_WIDGET (w)->style);
		fast_style->bg[GTK_STATE_NORMAL]=green;
  		fast_style->bg[GTK_STATE_ACTIVE]=green;
  		fast_style->bg[GTK_STATE_PRELIGHT]=green;
  		fast_style->bg[GTK_STATE_SELECTED]=green;
  		fast_style->bg[GTK_STATE_INSENSITIVE]=green;
		fast_style->base[GTK_STATE_NORMAL]=green;
  		fast_style->base[GTK_STATE_ACTIVE]=green;
  		fast_style->base[GTK_STATE_PRELIGHT]=green;
  		fast_style->base[GTK_STATE_SELECTED]=green;
  		fast_style->base[GTK_STATE_INSENSITIVE]=green;
    gtk_clist_set_compare_func(GTK_CLIST(w),user_clist_comp);
	}

	for(i=0;i<9;i++)
	{
		w=get_widget_by_widget_name(lbl_chat[i]);
		if(w)
		{
			gtk_label_set(GTK_LABEL(w),_("empty"));
		}
	}

	w=get_widget_by_widget_name("hub_public_clist");
	if(w)
	{
		gtk_clist_set_compare_func(GTK_CLIST(w),hub_pub_comp);
		gtk_clist_set_column_justification(GTK_CLIST(w),1,GTK_JUSTIFY_RIGHT);
	}

	w=get_widget_by_widget_name("seen_hub_clist");
	if(w)
	{
		gtk_clist_set_compare_func(GTK_CLIST(w),hub_pub_comp);
		gtk_clist_set_column_justification(GTK_CLIST(w),1,GTK_JUSTIFY_RIGHT);
	}

	w=get_widget_by_widget_name("gdl_ctree");
	if(w)
	{
		gtk_clist_set_column_justification(GTK_CLIST(w),2,GTK_JUSTIFY_RIGHT);
	}
}

static void clist_add_uniq_row(GtkCList *w, char *nw[5])
{
	int i;
	char *t;

	for(i=0;i<w->rows;i++)
	{
		gtk_clist_get_text(w,i,0,&t);
		if(t==NULL)
			continue;
		if(strcmp(t,nw[0]))
			continue;

		gtk_clist_get_text(w,i,1,&t);
		if(t==NULL)
			continue;
		if(strcmp(t,nw[1]))
			continue;

		gtk_clist_get_text(w,i,2,&t);
		if(t==NULL)
			continue;
		if(strcmp(t,nw[2]))
			continue;

		gtk_clist_get_text(w,i,3,&t);
		if(t==NULL)
			continue;
		if(strcmp(t,nw[3]))
			continue;

		gtk_clist_get_text(w,i,4,&t);
		if(t==NULL)
			continue;
		if(strcmp(t,nw[4]))
			continue;


		/* we have found an identical row */
		return;
	}
	gtk_clist_append(GTK_CLIST(w),nw);
}

/********************************************/
/* load the .done file list into done_clist */
/********************************************/
void load_done_xfer(char *fname,int with_clear)
{
	GtkWidget *w;
	FILE *f;
	char buf[51200];

	w=get_widget_by_widget_name("done_clist");

	if(w==NULL)
		return;

	if(with_clear==1)
		gtk_clist_clear(GTK_CLIST(w));

	f=fopen(fname,"rb");
	if(f==NULL)
		return;

	gtk_clist_freeze(GTK_CLIST(w));

	while(fgets(buf,sizeof(buf),f)!=NULL)
	{
		char *t;
		char *u;
		char *nw[5];

		t=strrchr(buf,'\n');
		if(t!=NULL)
			*t='\0';

		/* string format is: nick|DL/remote_fic$v|local_fic|start_pos|filesize|start_time */
		/*               or: nick|XDL|local_fic|filesize */

		t=strstr(buf,"|DL/");
		if(t==NULL)
		{
			/* not a |DL/, maybe a |XDL| */
			t=strstr(buf,"|XDL|");
			if(t!=NULL)
			{
				gchar **fields;

				fields=g_strsplit(buf,"|",0);
				if(gchar_array_len(fields)==4)
				{
					nw[0]=fields[0];		/* nickname */
					nw[1]=fields[3];		/* local filesize */
					nw[2]="";
					nw[3]="XDL (multiple sources)";			/* remote filename */
					nw[4]=fields[2];		/* local filename */
					clist_add_uniq_row(GTK_CLIST(w),nw);
				}
				g_strfreev(fields);
			}
		}
		else
		{
			/* it is a |DL/ */
			*t='\0';

			nw[0]=buf;

			nw[3]=strtok_r(t+4,"|",&u);
			if(nw[3]==NULL)
				continue;

			nw[4]=strtok_r(NULL,"|",&u);
			if(nw[4]==NULL)
				continue;
		
			t=strtok_r(NULL,"|",&u);
			if(t==NULL)
				continue;

			nw[1]=strtok_r(NULL,"|",&u);
			if(nw[1]==NULL)
				continue;

			nw[2]="";
	
			t=strrchr(nw[3],'$');
			if(t!=NULL)
				*t='\0';

			clist_add_uniq_row(GTK_CLIST(w),nw);
		}
	}
	gtk_clist_thaw(GTK_CLIST(w));
	fclose(f);
}

typedef struct
{
	char *widget_name;
	char *var_name;
} LNK;

/************************************************************/
/* set all fields of preference_box according to vars value */
/************************************************************/
void fix_pref_window(void)
{
	static const LNK v[]={	{"nickname_entry","nickname"},
								{"user_description_entry","user_desc"},
								{"cnx_type_entry","cnx_type"},
								{"e_mail_entry","email"},
								{"incoming_port_number_entry","com_port"},
								{"xfer_host_ip_entry","hostip"},
								{"dl_dir_entry","dl_path"},
								{"socks_address_entry","socks_ip"},
								{"socks_port_entry","socks_port"},
								{"socks_userid_entry","socks_name"},
								{"unodeport_entry","unode_port"},
								{"min_gdl_wake_up_delay_entry","min_gdl_wake_up_delay"},
								{"vshare_dir_entry","vshare_dir"},
								{NULL,NULL}};

	static const LNK tgl_bt[]={	{"enable_upload_checkbutton","dl_on"},
											{"follow_forcemove_checkbutton","follow_force_move"},
											{"md5sum_computation_checkbutton","with_md5sum"},
											{"use_done_dir_checkbutton","when_done"},
											{"ddl_checkbutton","with_ddl"},
											{"dctclink_checkbutton","with_dctclink"},
											{"force_dl_checkbutton","with_dl_force"},
											{"hide_abscheckbutton","hide_absolute"},
											{"hide_kick_checkbutton","hide_kick"},
											{"abort_upload_checkbutton","abort_upload_when_user_leaves"},
											{"lazykc_checkbutton","lazy_key_check"},
											{"incoming_wake_up_checkbutton","with_incoming_wake_up"},
											{"sr_wake_up_checkbutton","with_sr_wake_up"},
										{NULL,NULL}};

	int i;
	const char *t;

	/* initialize gtk entry from base */
	i=0;
	while(v[i].widget_name!=NULL)
	{
		t=get_var(v[i].var_name);
		if(t!=NULL)
		{
			gtk_entry_set_text(GTK_ENTRY(get_widget_by_widget_name(v[i].widget_name)), t);
		}
		else
		{
			if(current_dctc!=NULL)
				gtk_entry_set_text(GTK_ENTRY(get_widget_by_widget_name(v[i].widget_name)), "");
		}
		i++;
	}

	/* adjust toggle buttons */
	i=0;
	while(tgl_bt[i].widget_name!=NULL)
	{
		t=get_var(tgl_bt[i].var_name);
		if(t!=NULL)
		{
			gtk_toggle_button_set_active(
							GTK_TOGGLE_BUTTON(
									get_widget_by_widget_name(tgl_bt[i].widget_name)),
							atoi(t));
		}
		i++;
	}

	t=get_var("offset");
	if(t!=NULL)
	{
		double val;
		double scale[4]={1.0,1024.0, 1024.0*1024.0, 1024.0*1024.0*1024.0};
		char *unit_name[4]={_("Bytes"), _("KBytes"), _("MBytes"), _("GBytes")};
		int i=0;
		char buf[512];

		val=strtod(t,NULL);

		while(i<3)
		{
			double res,ent;

			res=modf(val/scale[i+1],&ent);
			if(res!=0.0)
				break;
			i++;
		}

		gtk_entry_set_text(GTK_ENTRY(get_widget_by_widget_name("size_offset_unit_entry")), unit_name[i]);
		sprintf(buf,"%.0f",val/scale[i]);
		gtk_entry_set_text(GTK_ENTRY(get_widget_by_widget_name("size_offset_entry")), buf);
		
	}

	t=get_var("dl_slot");
	if(t!=NULL)
	{
		gtk_adjustment_set_value(
					gtk_range_get_adjustment(GTK_RANGE(
							get_widget_by_widget_name("sim_dl_hscale"))),
					atoi(t));
	}

	t=get_var("recon_delay");
	if(t!=NULL)
	{
		gtk_adjustment_set_value(
					gtk_range_get_adjustment(GTK_RANGE(
							get_widget_by_widget_name("reconnect_delay_scale"))),
					atoi(t));
	}

	t=get_var("auto_rebuild_delay");
	if(t!=NULL)
	{
		gtk_adjustment_set_value(
					gtk_range_get_adjustment(GTK_RANGE(
							get_widget_by_widget_name("rebuild_delay_scale"))),
					atoi(t)/60);
	}

	t=get_var("max_running_source_per_gdl");
	if(t!=NULL)
	{
		gtk_spin_button_set_value(GTK_SPIN_BUTTON(
							get_widget_by_widget_name("maxrunspinbutton")),
					atoi(t));
	}

	t=get_var("disable_gdl_as_when_enough_running");
	if(t!=NULL)
	{
		gtk_spin_button_set_value(GTK_SPIN_BUTTON(
							get_widget_by_widget_name("maxasoffspinbutton")),
					atoi(t));
	}

	t=get_var("behind_fw");
	if(t!=NULL)
	{
		if(atoi(t)==1)
		{
			gtk_toggle_button_set_active(
						GTK_TOGGLE_BUTTON(
								get_widget_by_widget_name("passive_mode_radio_button")),
						TRUE);
		}
		else
		{
			gtk_toggle_button_set_active(
						GTK_TOGGLE_BUTTON(
								get_widget_by_widget_name("active_mode_radio_button")),
						TRUE);
		}
	}

	{
		GtkWidget *w;

		w=get_widget_by_widget_name("shared_dir_clist");
		if(w!=NULL)
		{
			GtkWidget *w2;
			char *pt, *pos,*nxt;

			w2=get_widget_by_widget_name("hidden_shared_dir_entry");

			gtk_clist_freeze(GTK_CLIST(w));
			gtk_clist_clear(GTK_CLIST(w));

			t=get_var("ul_path");
			if(t==NULL)
			{	/* if ul_path does not exist, we use "hidden_shared_dir_entry" content */
				t=gtk_entry_get_text(GTK_ENTRY(w2));
			}

			pt=strdup(t);
				
			/* we put its value into "hidden_shared_dir_entry" so it will be saved */
			gtk_entry_set_text(GTK_ENTRY(w2),pt);

			pos=strtok_r(pt,"|",&nxt);
			while(pos!=NULL)
			{
				gtk_clist_append(GTK_CLIST(w),&pos);
				pos=strtok_r(NULL,"|",&nxt);
			}
			free(pt);

			gtk_clist_thaw(GTK_CLIST(w));
		}
	}

   /* set xbl values */
   {
      struct
      {
         char *cmd_name;
         char *check_widget;
         char *val_widget;
      } sema_bl[]={  {"ubl","ubl_checkbutton","ubl_entry"},
                     {"dbl","dbl_checkbutton","dbl_entry"},
                     {"gbl","gbl_checkbutton","gbl_entry"},
                     {NULL,NULL,NULL}};

      i=0;
      while(sema_bl[i].cmd_name!=NULL)
      {
			t=get_var(sema_bl[i].cmd_name);
			if(t!=NULL)
			{
				if(atoi(t)==SEMVMX)
				{
					gtk_toggle_button_set_active(
									GTK_TOGGLE_BUTTON(
											get_widget_by_widget_name(sema_bl[i].check_widget)),FALSE);
				}
				else
				{
					gtk_toggle_button_set_active(
									GTK_TOGGLE_BUTTON(
											get_widget_by_widget_name(sema_bl[i].check_widget)),TRUE);
					gtk_entry_set_text(GTK_ENTRY(get_widget_by_widget_name(sema_bl[i].val_widget)), t);
				}
			}
			i++;
		}
	}

	t=get_var("cnx_opt");
	if(t!=NULL)
	{
		unsigned int v=atoi(t);

		gtk_toggle_button_set_active(
						GTK_TOGGLE_BUTTON(
								get_widget_by_widget_name("away_togglebutton")),
						((v&2)?TRUE:FALSE));
	}

	t=get_var("cnx_status");
	if(t!=NULL)
	{
		/* we have a connection status */
		GtkWidget *w;

		w=get_widget_by_widget_name("appbar1");
		if(w!=NULL)
		{
			unsigned int a;

			a=atoi(t);
		
			gnome_appbar_clear_stack(GNOME_APPBAR(w));

			switch(a&3)
			{
				case 3:	/* green, connection established */
							gnome_appbar_set_status(GNOME_APPBAR(w),_("Online"));
							break;
						
				case 1:	/* orange, connection in progress */
							gnome_appbar_set_status(GNOME_APPBAR(w),_("Trying to establish connection with hub"));
							break;

				default:	/* red, connection lost */
				case 2:
				case 0:
							t=get_var("main_sck");
							if((t==NULL)||(t[0]!='=')||(t[1]!='>'))
							{
								gnome_appbar_set_status(GNOME_APPBAR(w),"Offline");
							}
							else
							{
								char buf_time[512];
								struct tm tm;
								GString *rcn_msg;
								time_t tt;

								tt=atoi(t+2);
								localtime_r(&tt,&tm);
								strftime(buf_time,sizeof(buf_time)-1,"%c",&tm);

								rcn_msg=g_string_new(_("Offline, reconnect time: "));
								rcn_msg=g_string_append(rcn_msg,buf_time);

								gnome_appbar_set_status(GNOME_APPBAR(w),rcn_msg->str);

								g_string_free(rcn_msg,TRUE);
							}
							w=get_widget_by_widget_name("user_clist");
							if(w!=NULL)
								gtk_clist_clear(GTK_CLIST(w));
							break;
			}
		}
	}

	{
		GtkWidget *w;

		w=get_widget_by_widget_name("current_hub_address_label");

		if(w!=NULL)
		{
			t=get_var("hub_addr");
			if(t==NULL)
			{
				gtk_label_set(GTK_LABEL(w),"");
			}
			else
			{
				gtk_label_set(GTK_LABEL(w),t);
			}
		}
	}
	{
		GtkWidget *w;

		w=get_widget_by_widget_name("dctc_version_label");

		if(w!=NULL)
		{
			t=get_var("version");
			if(t==NULL)
			{
				gtk_label_set(GTK_LABEL(w),"");
			}
			else
			{
				gtk_label_set(GTK_LABEL(w),t);
			}
		}
	}}

