<?php
/**
 * Class to handle all the news system.
 *
 * daCode http://www.dacode.org/
 * src/phplib/news.php3
 * $Id: news.php3,v 1.216.2.33 2003/02/08 21:19:53 ruffy Exp $
 *
 * Depends: Config Db Utils Html Redirect Cache
 */
Class News {
	/**
	 * call to the class Db
	 *@var object Db
	 */
	var $db;
	
	/**
	 * call to the class Session
	 *@var object Session
	 */
	var $session;
	
	/**
	 * call to the class Utils
	 *@var object Utils
	 */
	var $utils;
	
	/**
	 * call to the class Html
	 *@var object Html -- in the classtree of the HTML class
	 */
	var $html;
	
	/**
	 * call to the class Redirect
	 *@var object Redirect
	 */
	var $redirect;

	/**
	 * call to the class Cache
	 *@var object Cache
	 */
	var $cache;
	
	/**
	 * call to the class News
	 *@var object News
	 */
	var $news;
	
	/**
	 * call to the class Comments
	 *@var object Comments
	 */
	var $comments;
	
	/**
	 * average of the score during last week
	 *@var integer
	 */
	var $last_week_score;
	
	/**
	 * Cache for the common SQL queries
	 *@var array $sqlCache [id => row]
	 */
	var $sqlCache;

	/**
	 * Boolean to indicate if a query has been cached or not
	 *@var boolean $isCached
	 */
	var $isCached = true;

	/**
	 * Class constructor
	 */
	Function News() {
		global $config;
		$this->db = LoadClass('Db');
		$this->session = LoadClass('Session');
		$this->utils = LoadClass('Utils');
		$this->html  = LoadClass('Html');
		$this->redirect = LoadClass('Redirect');
		$this->cache = LoadClass('Cache');
		//$this->comments = LoadClass('Comments');

		if ($config->moderation_type == 2) {
			// First we get the last week average score ...
			$sqlc_q = "SELECT value FROM ".$config->tables['general_infos'].
				" WHERE name='lastweek_score'";
			$ret = $this->db->query($sqlc_q);
			if (!$ret) {
				echo lecho("It looks like admins are working on the server");
				exit;
			}
			if ($this->db->num_rows()>=1) {
				$c = $this->db->fetch_array();
				$config->last_week_score = $c['value'];
			}
			$this->db->free();

			// We change the score according to that value.
			$config->oldscore = $config->score;
			switch($config->score) {
				case "-1":
					// Means all comments. -10000 should be ok :)
					$config->newscore = "-10000";
					break;
				case "0":
					$config->newscore = 0;
					break;
				case "1":
					$config->newscore = $config->last_week_score;
					break;
				case "2":
					$config->newscore = ( 2 * $config->last_week_score );
					break;
				case "3":
					$config->newscore = ( 3 * $config->last_week_score );
					break;
				case "4":
					$config->newscore = ( 4 * $config->last_week_score );
					break;
				case "5":
					$config->newscore = ( 5 * $config->last_week_score );
					break;
			}
		}


		$this->news = array ();

	}

	/**
	 * Add an entry to the news table.
	 * Need an array as input which needs to be like the following.
	 * FIXME: But where  is it used
	 * $row['topic_id']    is the id of the topic.
	 * $row['section_id']  is the id the section.
	 * $row['author_id']   is the id of the author.
	 * $row['title']       is the title of the news.
	 * $row['departement'] is the departement.
	 * $row['body']        is the body of the news.
	 * $row['links']       is an array (see the following)
	 * $row['files']       is an array which is an array itself (see the following)
	 *
	 *@access private
	 *@param array news info
	 *@return boolean -1 if db error, else nothing
	 */
	Function add($row) {
		global $config;

		$texte = "";
		$sqlc_q = "INSERT INTO ".$config->tables['news']." ".
			"(topic,section,author_id,title,departement,body) VALUES ('".
			addslashes($row['topic'])."','".
			addslashes($row['section'])."','".
			addslashes($row['author_id'])."','".
			addslashes($row['title'])."','".
			addslashes($row['departement'])."','".
			addslashes($row['body'])."')";
		$ret = $this->db->query($sqlc_q);
		if (!$ret) {
			$this->utils->debug("News:".__LINE__." ".
				lecho("SQL Failed: ").$this->db->error().
				" SQL: ".$sqlc_q."\n");
			return -1;
		}

		$last_insert_id = $this->db->last_insert_id();

		/*
			We INSERT all the submitted urls
			$row['links'] must be an array which looks like the following.

			$row['links'] = array(
				"name of url"   => "http://www.url.com",
				"name of url2"  => "http://www.url2.com"
			);
		*/
		$links = $row['links'];
		reset($links);

		while (current($links)) {
			$c_links = current($links);
			$k_links = key($links);
			$sqlc_q = "INSERT INTO ".$config->tables['url']." ".
				"(news_id, name, url) VALUES ('".
				addslashes($last_insert_id)."','".
				addslashes($c_links)."','".
				addslashes($k_links)."')";

			$ret = $this->db->query($sqlc_q);
			if (!$ret) {
				$this->utils->debug("News:".__LINE__." ".
					lecho("SQL Failed: ").$this->db->error().
					" SQL: ".$sqlc_q."\n");
				return -1;
			}

			next($links);
		}

		/*
			We INSERT all the submitted attachements
			$row['files'] must be an array of array which looks like the following.

			$tmp[1] = array(
				"size"  =>  "1000",
				"name"  =>  "file1",
				"content_type"  =>  "text/plain",
				"content" =>  "This is the text from the file"
			);

			$tmp[2] = array(
				"size"  =>  "1200",
				"name"  =>  "file2",
				"content_type"  =>  "text/plain",
				"content" =>  "This is the text from the file"
			);

			$row['files'] = $tmp;

		*/
		$files = $row['files'];
		reset ($files);

		while(current($files)) {
			$files_tmp = current($files);
			$files_tmp['content'] =
				(empty($files_tmp['content']) || $files_tmp['content'] == 'unknown' ?
				'text/plain' : $files_tmp['content']);
			/*
			  We shall use LOB interface

			$sqlc_q = "INSERT INTO ".$config->tables['attach']." ".
				"(news_id, content_type, size, name, attachement) VALUES ('".
				addslashes($last_insert_id)."','".
				addslashes($files_tmp['content_type'])."','".
				addslashes($files_tmp['size'])."','".
				addslashes($files_tmp['name'])."','".
				(get_magic_quotes_gpc()?$files_tmp['content']:
				addslashes($files_tmp['content']))."')";
			$ret = $this->db->query($sqlc_q);
			*/
			$this->db->insertLOB($config->tables['attach'], 
				array( 'news_id' => "'" . addslashes($last_insert_id) . "'",
					   'content_type' => "'" .addslashes($files_tmp['content_type'])."'" ,
					   'size' => "'" . addslashes($files_tmp['size']) . "'",
					   'name' => "'" . addslashes($files_tmp['name']) . "'", 
					   'attachement' => (get_magic_quotes_gpc()?stripslashes($files_tmp['content']):
										 $files_tmp['content']) 
					   ),
				array('attachement')
			);

			
			$ret = $this->db->query($sqlc_q);
			if (!$ret) {
				$this->utils->debug("News:".__LINE__." ".
					lecho("SQL Failed: ").$this->db->error().
					" SQL: ".$sqlc_q."\n");
				return -1;
			}

			next($files);
		}
	}




	/**
	 * Get a number of news. Return an array of array.
	 *
	 *@access public
	 *@param integer $nb number of news 
	 *@param string $section news from a specific section
	 *@param string $topic news from a specific topic
	 *@param integer $state state of news
	 *@param integer $view 1 for main page, 2 for others
	 *@param integer $news_id to find a specific news
	 *@param boolean $body to include news body or not
	 *@param string $start_date timestamp14
	 *@param integer $date the date of the news can't be superior to $date 
	 *@param string $start_nb timestamp14
	 *@return array all the news in an array of array.
	 */
	Function get_news($nb=10,$section="",$topic="",$state=1,$view=1,$news_id=0,$body=true,$start_date="",$date=0,$start_nb="0") {
		global $config;

		/* We build the SELECT command - Damned long ! :-) */
		$sqlc_q = "SELECT ".$config->tables['news'].".id,".$config->tables['news'].
			".timestamp,".$config->tables['news'].".title,".$config->tables['news'].
			".departement,".$config->tables['news'].".author_id, ";

		$sqlc_q .=$config->tables['topics'].".topic,".$config->tables['sections'].
			".section,".$config->tables['author'].".name,".$config->tables['author'].
			".contact FROM ".
			$config->tables['news'].",";

		$sqlc_q .= $config->tables['topics'].",".$config->tables['sections'].",".
			$config->tables['author']." ".
			"WHERE ";
		$sqlc_q .= $config->tables['news'].".author_id=".
			$config->tables['author'].".id AND ".$config->tables['news'].
			".topic_id=".$config->tables['topics'].".id AND ".
			$config->tables['news'].".section_id=".$config->tables['sections'].
			".id AND ".$config->tables['news'].".state='".addslashes($state)."' ";

		/*
		 * Removed to fix bug when url is called by normal url
		 * but doesn't have main_page at 1
		 */
		if ($view == 2) {
			$sqlc_q .= "AND ".$config->tables['news'].".main_page='0' ";
		} elseif ($view && $news_id==0) {
			$sqlc_q .= "AND ".$config->tables['news'].".main_page='1' ";
		}


		if ($section) {
			$sqlc_q .= "AND ".$config->tables['sections'].".section='".
				addslashes($section)."' ";
		}
		if ($topic) {
			$sqlc_q .= "AND ".$config->tables['topics'].".topic='".
				addslashes($topic)."' ";
		}
		if ($news_id) {
			$sqlc_q .= "AND ".$config->tables['news'].".id='".
				addslashes($news_id)."' ";
		}
		if ($start_date && $start_nb==0) {
			$sqlc_q .= "AND ".$config->tables['news'].".timestamp<'".
				$this->db->timestamp14_to_dbms_stamp(addslashes($start_date))."' ";
		}
		if ($date) {
			$sqlc_q .= "AND ".$config->tables['news'].".timestamp LIKE '".
				$this->db->date8_to_dbms_date(addslashes($date))."%' ";
		}

		$sqlc_q .= "ORDER BY ".$config->tables['news'].
			".timestamp DESC ";

		if ((!$news_id && !$nb) || $start_nb != 0) {
			$sqlc_q .= $this->db->compat_limit($nb,$start_nb);
		} elseif (!$date) {
			$sqlc_q .= $this->db->compat_limit($nb);
		}
		/* built ! */

		$ret = $this->db->query($sqlc_q);
		if (!$ret) {
			$this->utils->debug("News:".__LINE__." ".
				lecho("SQL Failed: ").$this->db->error().
				" SQL: ".$sqlc_q."\n");
			return -1;
		}

		//$i = 1;
		$out = array();
		/* fetch_array is probably faster than query */
		/*
		while ($row = $this->db->fetch_array()) {
			$out[$i] = $row;
			$i++;
		}
		*/
		//OK then, let's optimise a bit... :)
		$i = 0;
		$tmp = array();
		while ( $out[++$i] = $this->db->fetch_array()) {
	        $tmp[] = $out[$i]['id'];
		}
		// "optimization" adds an empty item at the end...
		unset($out[$i]);

		$tmp = 'id IN (' . implode(" , ", $tmp) . ')';
		if ($this->db->num_rows() == 0) {
			//   There are no news matching the request.
			//   In order to prevent flooding disk with
			//   non-existing news, no files will be written
			//   to cache
			$config->nosave = 1;
			$out = ''; //we need the empty string, since elsewhere in the
			// code, an array is assumed non-empty.
		}
		$this->db->free();

		//We want the body, but we have to make sure there are news 
		//to display
		if ($body=='1' && !isset($config->nosave)) {
			$sqlc_q = "SELECT id,body,content_type FROM ".
				 $config->tables['news_body'].
				 " WHERE $tmp";
			$ret = $this->db->query($sqlc_q);
			if (!$ret) {
				$this->utils->debug("News:".__LINE__." ".
									"Erreur SQL&nbsp;: ".$this->db->error().
									" SQL: ".$sqlc_q."\n");
				return -1;
			}
			$i='1';reset($out);
			while($row_tmp = $this->db->fetch_array()) {
				reset($out);
				while(list($a,$b) = each ($out)) {
					if ($b['id'] == $row_tmp[0]) {
						$out[$a]['body'] = $row_tmp[1];
						$out[$a]['content_type'] = $row_tmp[2];
						break;
					}
				}
				$i++;
			}
			$this->db->free();
		}

		return $out;
	}

	/**
	 * Get all the urls of the given news_id.
	 *
	 *@access public
	 *@param integer id of the news 
	 *@return mixed array with all the urls, or -1 if db error
	 */
	Function get_urls($id) {
		global $config;

		$out = array();
		/* Let's make the SQL command */
		$sqlc_q = "SELECT ".$config->tables['lang'].".lang,".
			$config->tables['news_urls'].
			".name,".$config->tables['urls'].".url,".
			$config->tables['news_urls'].".id,".
			$config->tables['urls_hits'].".hits,".$config->tables['news_urls'].
			".news_id,".
			$config->tables['urls'].".id AS urls_id FROM ".
			$config->tables['lang'].",".$config->tables['news_urls'].",".
			$config->tables['urls'].",".$config->tables['urls_hits'].
			" WHERE ".
			$config->tables['news_urls'].".lang_id=".$config->tables['lang'].
			".id AND ".
			$config->tables['news_urls'].".urls_id=".
			$config->tables['urls'].".id AND ".
			$config->tables['urls_hits'].".news_urls_id=".
			$config->tables['news_urls'].".id ";

		if (gettype($id) == "array") {
			if (sizeof($id) > 0) {
				reset($id);
				$sqlc_q .= "AND (";
				while($tmpo=current($id)) {
					$sqlc_q .= $config->tables['news_urls'].".news_id='".
						addslashes($tmpo)."'";
					if (next($id)) {
						$sqlc_q .= " OR ";
					}
				}
				$sqlc_q .= ")";
			}
		} else {
			$sqlc_q .= "AND ".$config->tables['news_urls'].".news_id='".addslashes($id)."'";
		}

		$sqlc_q .= " ORDER BY ".$config->tables['news_urls'].".id";

		$ret = $this->db->query($sqlc_q);

		if (!$ret) {
			$this->utils->debug("News:".__LINE__." ".
				lecho("SQL Failed: ").$this->db->error().
				" SQL: ".$sqlc_q."\n");
			return -1;
		}
		$i = 1;
		/*
		while ($row = $this->db->fetch_array()) {
			$out[$i] = $row;
			$i++;
		}
		*/
		$i = 0;
		$tmp = array();
		while ( $out[++$i] = $this->db->fetch_array()) {
	        $tmp[] = $out[$i]['id'];
		}
		$tmp = 'id IN (' . implode(" , ", $tmp) . ')';

		//Abusive cut'n'paste??? FIXME:
        //if ($body=='1') {
		if ( $i > 1) { //Got a news at least
            $sqlc_q = "SELECT id,body,content_type FROM ".
                $config->tables['news_body'].
                " WHERE $tmp";
            $ret = $this->db->query($sqlc_q);
            if (!$ret) {
                $this->utils->debug("News:".__LINE__." ".
                    "Erreur SQL&nbsp;: ".$this->db->error().
                    " SQL: ".$sqlc_q."\n");
                return -1;
            }
            $i='1';reset($out);
            while($row_tmp = $this->db->fetch_array()) {
                reset($out);
                while(list($a,$b) = each ($out)) {
                    if ($b['id'] == $row_tmp[0]) {
                        $out[$a]['body'] = $row_tmp[1];
                        $out[$a]['content_type'] = $row_tmp[2];
                        break;
                    }
                }
                $i++;
            }
        }

		$this->db->free();
		return $out;
	}

	/**
	 * Get all the attachements of the given news_id.
	 *
	 *@access public
	 *@param integer id of the news 
	 *@param boolean with or without body
	 *@return mixed array with all the urls, or -1 if db error
	 */
	Function get_attach($id,$w=false) {
		global $config;
		//No need to use LOB interface here, unless w is set 
		$out = array();
		$sqlc_q = "SELECT id,news_id,content_type,size,name";

		if ($w) {
			$sqlc_q .= ",attachement";
		}

		$sqlc_q .= " FROM ". $config->tables['attach'] ." WHERE ";

		if (gettype($id) == "array") {
			if (sizeof($id) > 0) {
				reset($id);
				$sqlc_q .= "(";
				while($tmpo=current($id)) {
					$sqlc_q .= "news_id='".addslashes($tmpo)."'";
					if (next($id)) {
						$sqlc_q .= " OR ";
					}
				}
				$sqlc_q .= ")";
			} else {
				$sqlc_q .= "news_id='-1'";
			}
		} else {
			$sqlc_q .= "news_id='".addslashes($id)."'";
		}

		$ret = $this->db->query($sqlc_q);
		if (!$ret) {
			$this->utils->debug("News:".__LINE__." ".
				lecho("SQL Failed: ").$this->db->error().
				" SQL: ".$sqlc_q."\n");
			return -1;
		}
		$i = 1;
		//If $w is set, we need to use LOB interface here.
		while ($row = ($w? $this->db->fetchLOB(array('attachement')) : $this->db->fetch_array()) ) {
			$this->utils->debug("News" . __LINE__);
			$out[$i] = $row;
			$i++;
		}
		$this->db->free();
		return $out;
	}

	/**
	 * Get attachement for save as command under netscape
	 *
	 * Call to header() to send attachment.
	 * Warning : Exit with -1 if db error
	 *
	 *@access public
	 *@param integer id of the news 
	 */
	Function view_attach($id) {
		global $config;

		$sqlc_q = "SELECT content_type,size,name,attachement FROM ".
			$config->tables['attach'] ." WHERE id='".addslashes($id)."'";
		$ret = $this->db->query($sqlc_q);
		if (!$ret) {
			$this->utils->debug("News:".__LINE__." ".
				lecho("SQL Failed: ").$this->db->error().
				" SQL: ".$sqlc_q."\n");
			return -1;
		}

		if ($row = $this->db->fetchLOB(array('attachement'))) {
			header('Content-type: '.$row['content_type']);
			header('Content-Length: '.$row['size']);
			header('Content-disposition: inline; filename='.$row['name']);
			echo $row['attachement'];
			exit;
		} else {
			echo lecho("No attachment");
		}

		$this->db->free();
	}

	/**
	 * Return the number of news in the last week for the given section
	 *
	 * This function could work with string or array
	 * Warning : return -1 if DB error
	 * FIXME: used only in Sidebox
	 *
	 *@access public
	 *@param array  list of the name section
	 *@return array contains number of news,in the section
	 */
	Function count_section($section) {
		global $config;

		$sqlc_q = "SELECT COUNT(*) AS nb,".$config->tables['sections'].
			".section FROM ".$config->tables['news'].",".
			$config->tables['sections'] . " WHERE ".
			$config->tables['news'] . ".section_id=".
			$config->tables['sections']. ".id AND ".
			$config->tables['news'].".state='1' AND ";

		if (gettype($section) == "array" && sizeof($section) > 0) {
			reset($section);
			$sqlc_q .= "(";
			while($tmpo=current($section)) {
				$sqlc_q .= $config->tables['sections'].".section='".
					addslashes($tmpo['section'])."'";
				if (next($section)) {
					$sqlc_q .= " OR ";
				}
			}
			$sqlc_q .= ") ";
		} else {
			$sqlc_q .= $config->tables['sections'].".section='".
				addslashes($tmpo)."' ";
		}

		$sqlc_q .= "GROUP BY ".$config->tables['sections'].".section";

		$ret = $this->db->query($sqlc_q);
		if (!$ret) {
			$this->utils->debug("News:".__LINE__." ".
				lecho("SQL Failed: ").$this->db->error().
				" SQL: ".$sqlc_q."\n");
			return -1;
		}

		$out = array();
		$i=1;
		while ($row = $this->db->fetch_array()) {
			$out[$i] = $row;
			$i++;
		}
		$this->db->free();

		if (gettype($section) == "array") {
			return $out;
		} elseif (isset($out[1])) {
			return $out[1]['nb'];
		} else {
			return 0;
		}
	}

	/**
	 * Get title
	 * Gets the title (HTML-escaped) of the news whose id is $news_id. 
	 * Warning : Exit with -1 if db error
	 *
	 *@access public
	 *@param mixed integer the ID of the news or empty string -- dirty
	 *@return string the title HTML escaped.
	 */
	Function show_title($news_id="") {
		global $config;

		if (empty($news_id)) {
			return $config->titledefault;
		}
		//Check if it's already in memory
		if (gettype($this->news) == "array") {
			reset($this->news);
			while ($cur=current($this->news)) {
				if ($cur['id'] == $news_id) {
					return htmlspecialchars($cur['title']);
				}
				next($this->news);
			}
		}
		//Not in memory, DB query
		$sqlc_q = "SELECT title FROM ".$config->tables['news'].
			" WHERE id='".addslashes($news_id)."'";
		$ret = $this->db->query($sqlc_q);
		if (!$ret) {
			$this->utils->debug("News:".__LINE__." ".
				lecho("SQL Failed: ").$this->db->error().
				" SQL: ".$sqlc_q."\n");
			return -1;
		}
		$row = $this->db->fetch_array();
		if ($row['title'] == "") {
			$row['title'] = $config->newstitledefault;
		} 
		$this->db->free();
		$this->news[0]['title'] = $row['title'];
		$this->news[0]['id'] = $news_id;

		return htmlspecialchars($row['title']);
	}

	/**
	 * View only one news... for reminder.
	 * Part of the implementation of the resource interface.
	 *
	 *@access public
	 *@param string $news_id the id of the news to view
	 *@return string the HTML for this news.
	 */
	Function viewResource($news_id) {
		return $this->show_news(10,"","",1,0,$news_id);
	}


	/**
	/* Html code to display the news in their beautiful boxes
	 *
	 *@access public
	 *@param integer $nb number of news 
	 *@param string $section news from a specific section
	 *@param string $topic news from a specific topic
	 *@param integer $state state of news
	 *@param integer $view 1 for main page, 2 for others
	 *@param integer $news_id to find a specific news
	 *@param integer $date the date of the news can't be superior to $date 
	 *@return string the HTML to display
	 */
	Function show_news($nb=10,$section="",$topic="",
			$state=1,$view=1,$news_id=0,$date=0) {
		global $config;

		if ($news_id) {
			$boxname = 'news_show';
			$cachetmp = array ($news_id,$section,$topic,$config->ext,
							$config->score,$config->hide_sig,$this->session->level,
							$state,$view,$date);
		} else {
			$boxname = 'news_index';
			$cachetmp = array ($section,$topic,$config->ext,$config->score,
							$config->hide_sig,$this->session->level,$state,$view,$date);
		}

		$fcontents = $this->cache->check_box("300",$boxname,$cachetmp);
		if (!empty($fcontents)) {
			return $fcontents;
		}

		$out = '';
		$this->news = $this->get_news($nb,$section,
			$topic,$state,$view,$news_id,1,0,$date);
		if (gettype($this->news) == "array") {

			reset($this->news);

			$i=0;
			$tempo = array();
			while ($cur=current($this->news)) {
				$tempo[$i] = $cur['id'];
				$i++;
				next($this->news);
			}
			reset($this->news);

			/* We get the urls for all the news */
			$tempo_urls = (array) $this->get_urls($tempo);

			if ($news_id) {
				$tempo_attach = (array) $this->get_attach($tempo,1);
			} else {
				$tempo_attach = (array) $this->get_attach($tempo);
			}

			$i=1;
			while ($cur=current($this->news)) {
				$tempo2 = array();
				reset($tempo_urls);
				$j=1;
				while($cur2=current($tempo_urls)) {
					if ($cur2['news_id'] == $cur['id']) {
						$tempo2[$j] = $cur2;
					}
					$j++;
					next($tempo_urls);
				}
				$this->news[$i]['links'] = $tempo2;
				$i++;
				next($this->news);
			}
			reset($this->news);

			$i=1;
			while ($cur=current($this->news)) {
				$tempo = array();
				reset($tempo_attach);
				$j=1;
				while($cur2=current($tempo_attach)) {
					if ($cur2['news_id'] == $cur['id']) {
						$tempo[$j] = $cur2;
					}
					$j++;
					next($tempo_attach);
				}
				$this->news[$i]['attach'] = $tempo;
				$i++;
				next($this->news);
			}
			reset($this->news);

			//Protect against some pbs if comments not loaded...
			$this->comments = LoadClass('Comments');
			//var_dump($config->obj);
			$score_comment = (array) $this->comments->count_comments_array($this->news, 1);
			reset($this->news);
			while ($cur=current($this->news)) {

				/*
				 * We get the number of comments for this news
				 */
				$n = $cur[0];
				$cur['comment'] = $score_comment['n'.$n]['s0'];
				if ($config->moderation_type==2) {
					$cur['score_comment'] = $score_comment['n'.
						$n]['s'.$config->oldscore];
				} else {
					$cur['score_comment'] = $score_comment['n'.$n]['s'.$config->score];
				}

				$out .= $this->html->newsbox($cur);

				/*
				 * We archive the last news id for archive
				 */
				$config->last_news_timestamp = $cur['timestamp'];

				if ($news_id) {
					if (gettype($cur['attach']) == "array") {
						reset($cur['attach']);
						$accepted_view = array ("text/plain", "text/html");
						while ($attach = current($cur['attach'])) {
							$ok=0;
							reset($accepted_view);
							while($accepted_view_a = current($accepted_view)) {
								if ($attach['content_type'] == $accepted_view_a) {
									$ok =1;
								}
								next($accepted_view);
							}

							if ($ok==1) {
								// Important!
								// $attach['attachement']=htmlentities($attach['attachement']);
								$attach['attachement'] = str_replace("<?","&lt;?",
									$attach['attachement']);
								$attach['attachement'] = str_replace("?>","?&gt;",
									$attach['attachement']);
								$out_tmp = "<b>".$attach['name']." - ";
								$out_tmp .= $attach['size']<=1?
									$attach['size']." ".lecho("byte"):
									$this->html->size_unit($attach['size']).lecho("bytes");
								$out_tmp .= "</b><br /><br />\n";

								if ($attach['content_type'] == "text/plain") {
									$out_tmp = $out_tmp .
										nl2br(htmlspecialchars($attach['attachement']));
								} else {
									$out_tmp = $out_tmp . $attach['attachement'];
								}
								$out .= $this->html->simplebox($out_tmp);
								$out .= "<br />";
							}
							next($cur['attach']);
						}
					}
				}

				next($this->news);
			}
			$this->cache->write_box($boxname,$cachetmp,$out);
			return $out;
		}
	}

	/**
	 * This is to show archives of the news in an archive box
	 *
	 *@access public
	 *@param integer $nb number of news 
	 *@param string $section news from a specific section
	 *@param string $topic news from a specific topic
	 *@param integer $state state of news
	 *@param integer $view 1 for main page, 2 for others
	 *@param integer $others Is it the "other" box?
	 *@param string $title title of a news
	 *@return string the HTML to display
	 *
	 */
	Function show_archive($nb=10,$section="",$topic="",$state=1,$view=1,$others=0,$title="") {
		global $config,$date;

		/*
		 * If the archive is called from the main page, and not for the others
		 * box, we start at 10.
		 */
		if (empty($date) && $others == '0') {
			$start_nb=10;
			$last_news_timestamp = 0;
			$date_news = 0;
		} else {
			/*
			 * If the archive is called from a date page, the last news to
			 * refer is the first one of the day which has been called.
			 */
			if (!empty($date)
#				 (!isset($config->last_news_timestamp) ||
#				 $config->last_news_timestamp == 0)
					) {
				$last_news_timestamp = $date."235959";
				$date_news = $date;
			} else {
				$last_news_timestamp = 0;
				$date_news = 0;
			}
			/*
			 * Else we start at 0. Probably other boxes.
			 */
			$start_nb=0;
		}

		$cachetmp = array ($nb,$section,$topic,$config->ext,$config->score,
				$config->hide_sig,$state,$view,
				$last_news_timestamp,$title,$others,$start_nb,$date,$config->depth);

		$fcontents = $this->cache->check_box("600","news_show_archive",$cachetmp);
		if (!empty($fcontents)) {
			return $fcontents;
		}

		$out = '';

		$tmp = $this->get_news($nb,$section,$topic,$state,$view,0,0,
				$last_news_timestamp,$date_news,$start_nb);
		if (!empty($tmp)) {
			$yr_old = $m_old = $d_old = '';

			//Protect against some pbs if comments not loaded...
			$this->comments = LoadClass('Comments');

			$score_comment = (array) $this->comments->count_comments_array($tmp);
			while(current($tmp)) {
				$cur = current($tmp);

				$cur['title'] = $cur['title']
						?htmlentities($cur['title'])
						:htmlentities($config->newstitledefault);

				$regs = $this->utils->stamp2array( $cur['timestamp'] );
				if ($yr_old != $regs[1] || $m_old != $regs[2] || $d_old != $regs[3]) {
					if (!empty($yr_old)) $out .= "<br />";
					$mregs = (int)$regs[2] - 1 ;
					$out .= "<b><a href=\"".
						$this->session->newsurls("","","",$regs[1],$regs[2],$regs[3],-2,0).
						"\"><span class=\"boxtext\">".
						$this->utils->stamp2date($cur['timestamp'],"short").
						"</span></a></b><br />\n";
				}
				$yr_old = $regs[1];
				$m_old = $regs[2];
				$d_old = $regs[3];


				$title_tmp = $cur['title'];
				$regs = $this->utils->stamp2array( $cur['timestamp'] );
				$url_tmp = $this->session->newsurls($cur['id'], "", "",
					$regs[1], $regs[2], $regs[3],-2,0);
				$out .= "&nbsp;<b>-</b>&nbsp;[".
					$score_comment['n'.$cur['id']]['s'.$config->score]."/".
					$score_comment['n'.$cur['id']]['s0'].
					"]&nbsp;".$this->redirect->make($title_tmp,$url_tmp,"zop=").
					"<br />";

				next($tmp);
				}
			} else {
				$out = lecho("No archives yet");
			}

			if (empty($title)) {
				if (!$section && !$topic) {
					$title = lecho("Archives");
				} else {
					if ($section) {
						$title = $section;
						if ($topic)
							$title .= "::".$topic;
					} else {
						$title = $topic;
					}
				}
			}
			$out = $this->html->sidebox($title,$out);

			$this->cache->write_box("news_show_archive",$cachetmp,$out);
			return $out;
	}

	/**
	 * Give <a href="url">name</a> of the previous or next news.
	 *
	 * If $prevnext=1, previous news is returned, otherwise next.
	 * Warnig : return -1 id db error
	 *
	 *@access private
	 *@param integer news id
	 *@param integer previous or next
	 *@param string $section news from a specific section
	 *@param string $topic news from a specific topic
	 *@param integer $view 1 for main page, 2 for others
	 *@return string html to display
	 */
	Function get_prev_next_news($news_id,$prevnext,$section="",
			$topic="",$view=1) {
		global $config;

		$out = '';

		$sqlc_q = "SELECT timestamp FROM ".$config->tables['news']." WHERE id='".
			addslashes($news_id)."'";
		$ret = $this->db->query($sqlc_q);
		if (!$ret) {
			$this->utils->debug("News:".__LINE__." ".
				lecho("SQL Failed: ").$this->db->error().
				" SQL: ".$sqlc_q."\n");
			return -1;
		}
		$row = $this->db->fetch_array();
		$timestamp = $row['timestamp'];
		$this->db->free();

		$sqlc_q = "SELECT ".$config->tables['news'].".id,".
			$config->tables['news'].".title,".
			$config->tables['news'].".timestamp FROM ".$config->tables['news'];

		if ($section) {
			$sqlc_q .= ",".$config->tables['sections'];
		}

		if ($topic) {
			$sqlc_q .= ",".$config->tables['topics'];
		}

		$sqlc_q .= " WHERE ";

		if ($view) {
			$sqlc_q .= $config->tables['news'].".main_page='".addslashes($view).
				"' AND ";
		}

		$sqlc_q .= $config->tables['news'].".state='1' ";

		if ($section) {
			$sqlc_q .= "AND ".$config->tables['sections'].".section='".
				addslashes($section)."' AND ".
				$config->tables['sections'].".id=".
				$config->tables['news'].".section_id ";
		}

		if ($topic) {
			$sqlc_q .= "AND ".$config->tables['topics'].".topic='".
				addslashes($topic)."' AND ".
				$config->tables['topics'].".id=".$config->tables['news'].
				".topic_id ";
		}

		$sqlc_q .= "AND ".$config->tables['news'].".timestamp ";
		$date = $this->utils->stamp2datetime( $timestamp);
		if ($prevnext) {
			$sqlc_q .= "< '".addslashes($date)."' ORDER BY timestamp DESC ". $this->db->compat_limit(1);

		} else {
			$sqlc_q .= "> '".addslashes($date)."' ORDER BY timestamp ". $this->db->compat_limit(1);
		}

		$ret = $this->db->query($sqlc_q);

		if (!$ret) {
			$this->utils->debug("News:".__LINE__." ".
				lecho("SQL Failed: ").$this->db->error().
				" SQL: ".$sqlc_q."\n");
			return -1;
		}

		if ($row = $this->db->fetch_array()) {
			$row['title'] = $row['title']?$row['title']:$config->newstitledefault;
			$regs = $this->utils->stamp2array( $row['timestamp'] );
			$out .= '<a href="'.
				$this->session->newsurls($row['id'], $section, $topic,
																$regs[1],
																$regs[2],
																$regs[3],-2,0).'">';
			if ($prevnext) {
				$out .= htmlentities("<< ".$row['title']);
			} else {
				$out .= htmlentities($row['title']." >>");
			}
			$out .= '</a>';
		}
		$this->db->free();

		return $out;
	}

	/**
	 * Give navigation box with <-- previous_news  next_news--> 
	 *
	 * if top=0 "title" is written
	 * if top=1 "top of the page" is written
	 * Warning : return -1 id db error
	 *
	 *@access public
	 *@param integer news id
	 *@param string $section news from a specific section
	 *@param string $topic news from a specific topic
	 *@param integer $view 1 for main page, 2 for others
	 *@param integer $top "top of the page" if 1
	 *@return string html to display
	 */
	Function show_prev_next_news($news_id,$section="",$topic="",$view=1,$top=0) {
		global $config;
		$cachetmp = array ($news_id,$section,$topic,
			$config->ext,$config->score,$view,$top);

		$fcontents = $this->cache->check_box("180","news_prev_next",$cachetmp);
		if (!empty($fcontents)) {
			return $fcontents;
		}

		$out = '<table width="100%" border="0">'.
			'<tr><td class="newstext" width="33%">'.
			$this->get_prev_next_news($news_id,1,$section,$topic,$view).'&nbsp;';

		$out .= '</td><td class="newstext" width="33%">';
		if ($top==1) {
			$out .= '<div align="center"><a href="#top">'
			.lecho("top of the page").'</a></div>';
		} else {
			$out .= '&nbsp;';
		}

		$out .= '</td><td class="rightnewstext" width="33%">&nbsp;'.
			$this->get_prev_next_news($news_id,0,$section,$topic,$view).
			'</td></tr></table>'."\n";

		$out = $this->html->simplebox($out);
		$this->cache->write_box("news_prev_next",$cachetmp,$out);
		return $out;
	}

	/**
	 * Verify that the author has been created. If not then it creates it,
	 *
	 * Warning: return -1 if db error
	 *
	 *@access private
	 *@param string name of an author 
	 *@param string a contact of an author
	 *@return integer id of the author (allready exist or created)
	 */
	Function verify_author($name,$contact) {
		global $config;

		$sqlc_q = "SELECT id FROM ".$config->tables['author']." WHERE name like '".
			addslashes($name)."' ";
		$sqlc_q .= "AND contact='".addslashes($contact)."'";

		$ret = $this->db->query($sqlc_q);
		if (!$ret) {
			$this->utils->debug("News:".__LINE__." ".
				lecho("SQL Failed: ").$this->db->error().
				" SQL: ".$sqlc_q."\n");
			return -1;
		}

		if ($this->db->num_rows()) {
			$id_tempo = $this->db->fetch_array();
			return $id_tempo['id'];
		} else {
			$sqlc_q = "INSERT INTO ". $config->tables['author'].
				" (name,contact) VALUES ('".
				addslashes($name)."','".
				addslashes($contact)."')";
			$ret = $this->db->query($sqlc_q);
			if (!$ret) {
				$this->utils->debug("News:".__LINE__." ".
					lecho("SQL Failed: ").$this->db->error().
					" SQL: ".$sqlc_q."\n");
				return -1;
			}
			return $this->db->last_insert_id();
		}
	}

	/**
	 * Give an html list of a specific news moderators
	 *
	 * Warning: return -1 if db error
	 *
	 *@access public
	 *@param integer id of the news 
	 *@return string html to display with modero1<br />modero2....
	 */
	Function view_modero($news_id) {
		global $config;

		$out = "";

		$sqlc_q = "SELECT ".$config->tables['users'].".fname,".
			$config->tables['users'].".lname,".
			$config->tables['users'].".login".
			" FROM ".$config->tables['users'].",".
			$config->tables['news_moderated']." WHERE ".
			$config->tables['users'].".id=".
			$config->tables['news_moderated'].".user_id AND ".
			$config->tables['news_moderated'].".news_id='".addslashes($news_id)."'".
			" ORDER BY ".$config->tables['news_moderated'].".timestamp";

		$ret = $this->db->query($sqlc_q);
		if (!$ret) {
			$this->utils->debug("News:".__LINE__." ".
				lecho("SQL Failed: ").$this->db->error().
				" SQL: ".$sqlc_q."\n");
			return -1;
		}
		if ($this->db->num_rows()) {
			while($row=$this->db->fetch_array()) {
				if (!empty($row['fname']) || !empty($row['lname'])) {
					$out .= '&nbsp;-&nbsp;' . $row['fname']." ".$row['lname']."<br />\n";
				} else {
					$out .= '&nbsp;-&nbsp;' . $row['login']."<br />\n";
				}

			}
		}
		$this->db->free();

		return $out;
	}


	/**
	 * Controls wether the news may be commented.
	 * Part of the implementation of the resource interface.
	 *
	 *@access public
	 *@param integer $news_id the ID of the news to control
	 *@return mixed NULL if it may be commented; an error string otherwise.
	 */
	Function checkResource($news_id) {
		// We check if the news exists.
		$sqlc_q = "SELECT state FROM news WHERE id='".
			addslashes($news_id)."'";
		$ret = $this->db->query($sqlc_q);
		if (!$ret) {
			$this->utils->debug("News:".__LINE__." ".
				lecho("SQL Failed: ").$this->db->error().
				" SQL: ".$sqlc_q."\n");
			return -1;
		}
		
		list($state_of_news) = $this->db->fetch_array();
		$this->db->free();
		if ($state_of_news != 1) {
			return lecho("Sorry, no results.").
				" : News ".
				lecho("Not Found");
		}

		return ;
	}

	/**
	 * Give preview of a news in a newsbox.
	 * Warning: return -1 if db error
	 *
	 *@access private
	 *@param array info about the future news (topic, section)
	 *@return string html to be displayed.
	 */
	Function preview($table) {
		global $config;

		$sqlc_q = "SELECT topic FROM ".$config->tables['topics'].
			" WHERE id='".addslashes($table['topic_id'])."'";
		$ret = $this->db->query($sqlc_q);
		if (!$ret) {
			$this->utils->debug("News:".__LINE__." ".
				lecho("SQL Failed: ").$this->db->error().
				" SQL: ".$sqlc_q."\n");
			return -1;
		}

		$topics_tmp=$this->db->fetch_array();
		$table['topic'] = $topics_tmp['topic'];

		$sqlc_q = "SELECT section FROM ".$config->tables['sections'].
			" WHERE id='".addslashes($table['section_id'])."'";
		$ret = $this->db->query($sqlc_q);
		if (!$ret) {
			$this->utils->debug("News:".__LINE__." ".
				lecho("SQL Failed: ").$this->db->error().
				" SQL: ".$sqlc_q."\n");
			return -1;
		}

		$sections_tmp=$this->db->fetch_array();
		$table['section'] = $sections_tmp['section'];

		return $this->html->newsbox($table,1) . "<br /><br />\n";
	}

	/**
	 * A form to fill to submit a news
	 * Warning: return -1 if db error
	 *
	 *@access public
	 *@param integer size of the box to write url
	 *@return string html to be displayed, or error message
	 */
	Function submit($size_url=50) {
		global $config;
		global $HTTP_POST_VARS,$HTTP_SERVER_VARS,$name_cookie,$mail_cookie;

		$pass = 1;

		$table = '';
		if (isset ($HTTP_POST_VARS['table'])) {
			$table = $HTTP_POST_VARS['table'];
		}

		$table2 = '';
		if (isset ($HTTP_POST_VARS['table2'])) {
			$table2 = $HTTP_POST_VARS['table2'];
		}

		/*
		 * We verify if there is a section and a topic to post in.
		 */

		$sqlc_q = "SELECT state FROM ".$config->tables['topics'].
			" WHERE state='1'";
		$ret = $this->db->query($sqlc_q);
		if (!$ret) {
			$this->utils->debug("News:".__LINE__." ".
				lecho("SQL Failed: ").$this->db->error().
				" SQL: ".$sqlc_q."\n");
			return -1;
		}

		if (!$this->db->num_rows()) {
			return lecho("We are sorry but you can't submit news now, ").
				lecho("because there is no topic where to post.")."<br /><br />";
		}
		$sqlc_q = "SELECT state FROM ".$config->tables['sections'].
			" WHERE state='1'";
		$ret = $this->db->query($sqlc_q);
		if (!$ret) {
			$this->utils->debug("News:".__LINE__." ".
				lecho("SQL Failed: ").$this->db->error().
				" SQL: ".$sqlc_q."\n");
			return -1;
		}
		if (!$this->db->num_rows()) {
			return lecho("We are sorry but you can't submit news now, ").
				lecho("because there is no section where to post.")."<br /><br />";
		}

		if (is_array($table)) {
			//  First check for referer
			if ($HTTP_SERVER_VARS['HTTP_REFERER'] != $config->baseurl.'submit.'.
				$config->php && !($this->session->param & pow(2,3))) {
				return $this->utils->warn_referer();
			}

			//  Check whether $table and $table2 are identical
			//  If box is unchecked, this variable is unset
			if (!is_array($table2)) {
				$table2 = array();
			}
			if (isset($table2['attachments']) &&
					$table2['attachments'] != $table['attachments']) {
				$pass = 2;
			} elseif ($this->utils->cmp_array($table,$table2)) {
				if (isset ($table2['attachments'])) {
					return $this->submit_upload($table);
				} else {
					return $this->submit_confirm($table);
				}
			} else {
				$pass = 2;
			}
		}

		$out = '';

		if ($pass == 2) {
			$out = $this->preview($table);
		}

		$out .= '<form method="post" action="'.
			$config->basehref.'submit.'.$config->php.'">'."\n";

		//  Previous answers are stored in hidden fields
		if (is_array($table)) {
			reset($table);
			while(list($key,$value)=each($table)) {
				if ($key != 'section' && $key != 'topic') {
					$out .= '<input type="hidden" name="table2['.$key.']"'.
						' value="'.htmlentities($table[$key]).'">'."\n";
				}
			}
		} else {
			$table = array();
		}

		//  Name
		$out .= '<div class="newstext">'.lecho("Your name (Leave Blank to be Anonymous)").'</div>';
		$out .= '<br /><input type="text" name="table[name]" value="';
		if (!empty($table['name']))
			$out .= htmlentities($table['name']);
		elseif (!empty($name_cookie))
			$out .= htmlentities("$name_cookie");
		else
			$out .= htmlentities(lecho("Anonymous"));
		$out .= '" size="50" maxlength="50" /><br /><br />'."\n";

		//  E-mail or homepage
		$out .= '<div class="newstext">'.lecho("Your email or homepage (Must looks like http://)").'</div>';
		$out .= '<br /><input type="text" name="table[contact]" value="';
		if (!empty($table['contact']))
			$out .= htmlentities($table['contact']);
		elseif (!empty($mail_cookie))
			$out .= htmlentities("$mail_cookie");
		$out .= '" size="50" maxlength="100" /><br /><br />'."\n";

		//  Subject
		$out .= '<div class="newstext">'.lecho("Subject (Be descriptive, clear, and simple!)").'</div>';
		$out .= '<br /><input type="text" name="table[title]" value="';
		if (!empty($table['title']))
			$out .= htmlentities($table['title']);
		$out .= '" size="50" maxlength="150" /><br /><br />'."\n";

		//  Topic
		$out .= '<div class="newstext">'.lecho("Topic, section, and departement (free speech space)").'</div>';
		$out .= '<br /><select name="table[topic_id]">';
		$sqlc_q = "SELECT id,topic,longname FROM ".$config->tables['topics'].
			" WHERE state='1' ORDER BY topic";
		$ret = $this->db->query($sqlc_q);
		if (!$ret) return $this->db->error();

		while($topics_tmp=$this->db->fetch_array()) {

			if (isset($topics_tmp['longname']) && $topics_tmp['longname'] != "") {
				$topics_tmp['topic'] = $topics_tmp['longname'];
			}

			$topics_tmp['topic'] = htmlentities($topics_tmp['topic']);

			if (isset($table['topic_id'])
					&& $table['topic_id'] == $topics_tmp['id']) {
				$out .= '<option value="'.$topics_tmp['id'].'" ';
				$out .= 'selected="selected">'.$topics_tmp['topic'].'</option>'."\n";
			} else {
				$out .= '<option value="'.$topics_tmp['id'].'">'.$topics_tmp['topic'].
					'</option>'."\n";
			}
		}
		$this->db->free();
		$out .= '</select>'."\n";

		//  Section
		$out .= '<select name="table[section_id]">';
		$sqlc_q = "SELECT id,section FROM ".$config->tables['sections'].
			" WHERE state='1' ORDER BY section";
		$ret = $this->db->query($sqlc_q);
		if (!$ret) {
			return $this->db->error();
		}

		while($sections_tmp=$this->db->fetch_array()) {

			$sections_tmp['section'] = htmlentities($sections_tmp['section']);

			if (isset($table['section_id'])
					&& $table['section_id'] == $sections_tmp['id']) {
				$out .= '<option value="'.$sections_tmp['id'].'" ';
				$out .= 'selected="selected">'.$sections_tmp['section'].'</option>'.
					"\n";
			} else {
				$out .= '<option value="'.$sections_tmp['id'].'">';
				$out .= $sections_tmp['section'].'</option>'."\n";
			}
		}
		$this->db->free();
		$out .= '</select>'."\n";

		//  Department
		$out .= '<input type="text" name="table[departement]" value="';
		if (!empty($table['departement']))
			$out .= htmlentities($table['departement']);
		$out .= '" size="30" maxlength="100" /><br /><br />'."\n";

		//  Body
		$out .= '<div class="newstext">'.lecho("The scoop (HTML is fine, but double check ").
			lecho("thoses URLs and HTML tags!)").'</div>';
		$out .= '<select name="table[content_type]">';
		if (isset($table['content_type']) && $table['content_type'] == 'text/html') {
			$out .= '<option value="text/plain">'.lecho("text").'</option>'.
				'<option value="text/html" selected="selected">'.
				lecho("html").'</option>';
		} else {
			$out .= '<option value="text/plain" selected="selected">'.
				lecho("text").
				'</option><option value="text/html">'.lecho("html").'</option>';
		}
		$out .= '</select><br />'."\n";

		$out .= '<textarea name="table[body]" rows="10" cols="70">';
		if (isset($table['body']))
			$out .= htmlentities($table['body']);
		$out .= '</textarea><br /><br />'."\n";

		$out .= '<div class="newstext">'.lecho("Attached File:").'</div>';
		$out .= ' <input type="checkbox" name="table[attachments]"';
		if (isset($table['attachments']) &&
				$table['attachments'] == "on") {
			$out .= ' checked="checked"';
		}
		$out .= ' /><br /><br />'."\n";

		//  Links
		$out .= '<div class="newstext">'.lecho("HTML Links. <b>Don't forget to choose the lang!</b>").'</div>';
		$out .= '<br /><table width="100%">'.
			'<tr><td>'.'<div class="newstext">'.lecho("Name").'</div>'.'</td><td>'.
			'<div class="newstext">'.lecho("Urls (Must looks like http://)")."</div></td><td>&nbsp;</td></tr>\n";
		$sqlc_q = "SELECT id,lang FROM ".$config->tables['lang'] ." ORDER BY lang";
		$ret = $this->db->query($sqlc_q);
		if (!$ret) return $this->db->error();
		while($lang_tmp = $this->db->fetch_array()) {
			$lang_id       = $lang_tmp['id'];
			$lang_lan      = $lang_tmp['lang'];
			$lan[$lang_id] = $lang_lan;
		}
		$this->db->free();

		for ($k=0;$k<$config->numberoflinks;$k++) {
			$nomk = isset($table["nom$k"])?$table["nom$k"]:'';
			$urlk = isset($table["url$k"])?$table["url$k"]:'';
			$lank = ereg_replace(':.*$', '', isset($table["lan$k"])?$table["lan$k"]:'');

			$out .= '<tr><td><input type="text" name="table[nom'.$k.
				']" size="20" ';
			if (!empty($nomk))
				$out .= 'value="'.htmlentities($nomk).'"';
			$out .= ' /></td><td><input type="text" name="table[url'.$k.
				']" size="'.$size_url.'" maxlength="200" ';
			if (!empty($urlk))
				$out .= 'value="'.htmlentities($urlk).'"';
			$out .= ' /></td><td><select name="table[lan'.$k.']">';
			$out .= '<option value="0">'.lecho("Language").'</option>'."\n";
			reset($lan);
			while (current($lan)) {
				$out .= '<option value="'.key($lan).':'.current($lan) .'"'.
					($lank==key($lan) ? ' selected="selected"' : '').
					">".current($lan).'</option>'."\n";
				next($lan);
			}
			$out .= '</select></td></tr>'."\n";
		}
		$out .= '</table>'."\n";

		$out .= '<input type="submit" name="submit" value="'.
			lecho("Next").'" /></form>'."\n";
		$out .= '<div align="center">'.lecho("Step").' '.$pass.'</div>';
		return $out;
	}

	/**
	 * A form to fill to submit an attached file of a news
	 * Warning: return -1 if db error
	 *
	 *@access private
	 *@param array all the news info
	 *@return string html to be displayed
	 */
	Function submit_upload($table) {
		global $config;

		$out = '<div class="newstext">'.lecho("Attached Files:").'</div>';
		$out .= '<br /><form method="post" action="'.
			$config->basehref."submit.".$config->php.
			'" enctype="multipart/form-data">';

		$out .= '<input type="hidden" name="MAX_FILE_SIZE" value="2097152">';
		//  Pass variables through hidden files so that they can be
		//  treated in pass 4.
		reset($table);
		while(list($key,$value)=each($table)) {
			if ($key != 'section' && $key != 'topic' && $key != 'attachments') {
				$out .= '<input type="hidden" name="table['.$key.']"'.
					' value="'.htmlentities($table[$key]).'">'."\n".
					'<input type="hidden" name="table2['.$key.']"'.
					' value="'.htmlentities($table[$key]).'">'."\n";
			}
		}
		for ($k=0;$k<$config->numberoffiles;$k++) {
			$out .= '<input type="file" size="50" name="files'.$k.'"><br />'."\n";
		}
		$out .= '<input type="submit" name="submit" value="'.
			lecho("Next").'"></form>'."\n";
		$out .= '<p><div align="center" class="newstext">'.lecho("Step").' 3</div>';
		return $out;
	}

	/**
	 * Message when news has been successfully fulfilled and added to db
	 * Warning: return -1 if db error
	 *
	 *@access private
	 *@param array all the news info
	 *@return string html to be displayed
	 */
	Function submit_confirm($table) {
		global $config;

		$out = '';

		for ($k=0;$k<$config->numberoffiles;$k++) {
			if (isset ($_FILES['files'.$k]['name'])) {
				$out .= '<input type="hidden" name="table2[files'.$k.']"'.
					' value="'.htmlentities($_FILES['files'.$k]['name']).'">'."\n";
			}
		}

		/* We verify the user exist, if it doesn't exist we
		 * create the entry in the database. In any case the
		 * function return the id of the user
		 */

		$id_author = $this->verify_author($table['name'],$table['contact']);
		/* We INSERT the title of the news and the rest in news table */
		$sqlc_q = "SELECT id FROM ".$config->tables['lang'].
			" WHERE lang='".$config->defaultlang."'";
		$ret = $this->db->query($sqlc_q);
		if (!$ret) {
			$this->utils->debug("News:".__LINE__." ".
				lecho("SQL Failed: ").$this->db->error().
				" SQL: ".$sqlc_q."\n");
			return -1;
		}
		$lang_tmp = $this->db->fetch_array();

		$sqlc_q = "INSERT INTO ".$config->tables['news'].
			" (topic_id,section_id,author_id,lang_id,title,departement,ip,user_id) ".
			"VALUES ('".
			addslashes($table['topic_id'])."','".
			addslashes($table['section_id'])."','".
			addslashes($id_author)."','".
			addslashes($lang_tmp['id'])."','".
			addslashes($table['title'])."','".
			addslashes($table['departement'])."','".
			addslashes($this->utils->ip())."','".
			addslashes($this->session->user_id)."')";

		$ret = $this->db->query($sqlc_q);
		if (!$ret) {
			$this->utils->debug("News:".__LINE__." ".
				lecho("SQL Failed: ").$this->db->error().
				" SQL: ".$sqlc_q."\n");
			return -1;
		}
		$last_insert_id = $this->db->last_insert_id();

		/* We INSERT the BODY of the NEWS */

		/* This replace is needed so that users don'h have to mess up
		 * with XHTML,HTML etc etc
		 */
		if($table['content_type'] == 'text/html') {
			$table['body'] = preg_replace('/<br>/i', '<br />', $table['body']);
		}
		
		$sqlc_q = "INSERT INTO ".$config->tables['news_body'].
		" (id,body,content_type) VALUES ('".
			addslashes($last_insert_id)."','".
			addslashes($table['body'])."','".
			addslashes($table['content_type'])."')";

		$ret = $this->db->query($sqlc_q);
		if (!$ret) {
			$this->utils->debug("News:".__LINE__." ".
				lecho("SQL Failed: ").$this->db->error().
				" SQL: ".$sqlc_q."\n");
			return -1;
		}

		/* We INSERT ALL the LINKS */
		for ($k=0;$k<$config->numberoflinks;$k++) {
			$nomk = $table["nom$k"];
			$urlk = $table["url$k"];
			$lank = ereg_replace(':.*$', '', $table["lan$k"]);
			if (!empty($nomk)) {

				/* We verify urls is in $config->tables['urls'] */
				$sqlc_q = "SELECT id FROM ".$config->tables['urls']." WHERE url like ".
					"'".addslashes($urlk)."'";
				$ret = $this->db->query($sqlc_q);
				if (!$ret) {
					$this->utils->debug("News:".__LINE__." ".
						lecho("SQL Failed: ").$this->db->error().
						" SQL: ".$sqlc_q."\n");
					return -1;
				}
				if (!$this->db->num_rows()) {
					$sqlc_q = "INSERT INTO ".$config->tables['urls']." (url) VALUES ".
						"('".addslashes($urlk)."')";
					$ret = $this->db->query($sqlc_q);
					if (!$ret) {
						$this->utils->debug("News:".__LINE__." ".
							lecho("SQL Failed: ").$this->db->error().
							" SQL: ".$sqlc_q."\n");
						return -1;
					}
					$url_id = $this->db->last_insert_id();
				} else {
					$url_id_tmp = $this->db->fetch_array();
					$url_id = $url_id_tmp['id'];
				}


				/* We insert the news_urls */
				$sqlc_q = "INSERT INTO ".$config->tables['news_urls'].
					" (news_id,lang_id,urls_id,name) ".
					"VALUES ('".
					addslashes($last_insert_id)."','".
					addslashes($lank)."','".
					addslashes($url_id)."','".
					addslashes($nomk)."')";

				$ret = $this->db->query($sqlc_q);
				if (!$ret) {
					$this->utils->debug("News:".__LINE__." ".
						lecho("SQL Failed: ").$this->db->error().
						" SQL: ".$sqlc_q."\n");
					return -1;
				}

				$tempo_id = $url_id = $this->db->last_insert_id();

				$sqlc_q = "INSERT INTO ".$config->tables['urls_hits'].
					" (news_urls_id,hits) VALUES ('".
					addslashes($tempo_id)."', '0')";
				$ret = $this->db->query($sqlc_q);
				if (!$ret) {
					$this->utils->debug("News:".__LINE__." ".
						lecho("SQL Failed: ").$this->db->error().
						" SQL: ".$sqlc_q."\n");
					return -1;
				}

			}
		}

		/* We INSERT all the attached files */
		for ($k=0;$k<$config->numberoffiles;$k++) {
			if (isset ($_FILES['files'.$k])) {
				$filesk = $_FILES['files'.$k];
				if (!empty($filesk) && $filesk['tmp_name'] != "none") {
					$files_tmp_name = $filesk['name'];
					$files_tmp_size = $filesk['size'];
					$files_tmp_type = $filesk['type'];

					$fd = fopen( $filesk['tmp_name'], "rb" );
					$contents = fread( $fd, $files_tmp_size );
					fclose( $fd );

					/*
					  Here we need to use LOB interface
					$sqlc_q = "INSERT INTO ".$config->tables['attach'].
						" (news_id,content_type,".
						"size,name,attachement) VALUES ('".
						addslashes($last_insert_id)."','".
						addslashes($files_tmp_type)."','".
						addslashes($files_tmp_size)."','".
						addslashes($files_tmp_name)."','".
						addslashes($contents)."')";
					$ret = $this->db->query($sqlc_q);
					*/

					$this->db->insertLOB( $config->tables['attach'], 
						array('news_id' => "'" . addslashes($last_insert_id) ."'",
							  'content_type' => "'". addslashes($files_tmp_type) . "'",
							  'size' => addslashes($files_tmp_size) ,
							  'name' => "'". addslashes($files_tmp_name) . "'",
							  'attachement' => $contents),
						array('attachement')
					);
					if (!$ret) {
						$this->utils->debug("News:".__LINE__." ".
							lecho("SQL Failed: ").$this->db->error().
							" SQL: ".$sqlc_q."\n");
						return -1;
					}
				}
			}
		}
		$out = '<div class="bgtext">'.
			lecho("Your message has been recorded and will be processed").
			lecho(" by one of the moderators.")." ".
			lecho("If approved, it will be published on our website:").
			' <a href="'.$config->basehref.'">'.$config->baseurl.'</a></div>'.
			'<div align="center">'.lecho("Step").' ';
		if (isset($files_tmp_name)) {
			$out .= "4";
		} else {
			$out .= "3";
		}
		$out .= '</div>';

		if ($config->moderation_warn) {
			$text = "$config->name - $config->baseurl\n--------------------\n";
			$text .= lecho("News have been posted. It's waiting for moderation.");
			$title = lecho("Work for moderators");

			$this->utils->send_email($config->moderation_from, $title, $text);
		}
		
		return $out;
	}

	/**
	 * We print the form to send an email to a friend about a news
	 *
	 *@access public
	 *@param integer id of the news
	 *@return mixed NULL or html form
	 */
	Function send_email_view($news_id=0) {
		global $config;

		if ($news_id == '0' ||
			$this->is_approved($news_id) == "0") {
			return;
		}

		$out = '';
		$out .= '<form action="'.$config->basehref.'news/send_email.'.
				$config->php.'" type="post">'."\n";
		$out .= '<center><b>'.lecho("Send to a friend").'</b></center><br />'.
			'<br />'.lecho("Enter an email address").':<br />';
		$out .= '<input name="email_address" value="" type="text" '.
			'size="20" /><br /><br />'."\n";
		$out .= lecho("Enter a comment").':<br />';
		$out .= '<textarea name="body" cols="20" rows="3"></textarea><br />'.
			'<br />'."\n";
		$out .= '<center><input type="submit" value="'.lecho("Submit").
			'" /></center>'."\n";
		$out .= '<input name="news_id" value="'.$news_id.'" type="hidden" />'."\n";
		$out .= '</form>';

		return $out;
	}

	/**
	 * We send email if to a friend if everything is ok
	 *
	 *@access public
	 *@param integer id of the news
	 *@param string email adress
	 *@param string email body
	 *@return mixed NULL or html to diplay
	 */
	Function send_email($news_id=0,$email_address="",$body="") {
		global $config;

		if ($news_id=='0' || empty($email_address) ||
			$this->is_approved($news_id) == "0") {
			return;
		}

		$out = "";

		// TODO Make a real RFC822 compliant check
		if (! preg_match('/^[\w_\-\.\+\']+@[\w_\-\.]+$/',$email_address)) {
			$out .= lecho("Email looks wrong");
			$out .= ': '.$email_address.'-'.$news_id;
		} else {

			$ts = $this->utils->gettimestampnews($news_id);
			$news_url = $this->session->newsurls($news_id,"","",$ts[0],$ts[1],
				$ts[2],-2,1);
			$title = $this->show_title($news_id);

			$text = lecho("Hello,\n\n");

			if ($this->session->checked) {
				if (($this->session->param & pow(2,0)) &&
					(!empty($this->session->fname) ||
						!empty($this->session->lname))) {
					$text .= ucfirst($this->session->fname) ." ".
						ucfirst($this->session->lname)." ";
				} else {
					$text .= ucfirst($this->session->login)." ";
				}
			} else {
				$text .= lecho("Someone, ");
			}

			if ($this->session->checked) {
				$text .= lecho("whose email address is ").$this->session->email;
			} else {
				$text .= lecho("whose ip address is ");
				$text .= $config->REMOTE_ADDR;
			}

			$text .= lecho("\nthought you should be interested by ").
				lecho("the news that you can find at:\n\n");
			$text .= $news_url."\n\n\n";
			$text .= lecho("The title of this news is:\n\n");
			$text .= $title."\n\n";

			if (!empty($body)) {
				$text .= lecho("He also wanted to add the following comment:\n\n").
				ucfirst($body).
					"\n\n";
			}

			$text .= lecho("If someone is abusing with this service, please\n");
			$text .= lecho("contact us at: ").$config->mailadmin."\n";


			if ($this->utils->send_email($email_address,$title,$text)) {
				$out .= lecho("Sent!");
				$out .= '<script language="javascript">self.close()</script>'."\n";
			} else {
				$out .= lecho("Error, the email couldn't be send.<br />Perhaps the mail server is not up?");
			}
		}

		return $out;
	}

	/**
	 * Returns a simplebox with icon (printer and email) inside if $news_id != 0
	 * Returns nothing if $news_id=0
	 *
	 *@access public
	 *@param integer id of the news
	 *@return mixed NULL or html to diplay
	 */
	Function show_print($news_id=0) {
		global $config;

		if ($news_id == 0) {
			return;
		}

		$out = '<center><a href="'.$config->basehref.'news/send_email_view.'.
			$config->php.'?news_id='.$news_id.'" onclick="window.open(\''.
			$config->basehref."news/send_email_view.".$config->php."?news_id=".
			$news_id."','M',"."'toolbar=no,location=no,directories=no,status=no,".
			"alwaysRaised=yes,dependant=no,menubar=no,scrollbars=no,".
			"resizable=no,width=200,height=300'); return false\" ".
			"onmouseout=\"javascript: window.status=''; return true;\" ".
			"onmouseover=\"javascript: window.status='".lecho("Send to a friend").
			"'; return true;\"><img src=\"".$this->session->imgurl("icon_send").
			"\" border=\"0\" alt=\"".lecho("Send to a friend")."\" /></a>";


		reset($config->listofthemes);
		while(list($key,$val) = each ($config->listofthemes)) {
			if ($val == "printable") {
				$theme_print=$key;
				break;
			}
		}

		if (isset($theme_print)) {
			$backup_ext = $config->ext;
			$config->ext = ','.$theme_print;
			if ($this->session->checked) {
				$config->ext .= '.'.$config->php;
			} else {
				$config->ext .= '.html';
			}

			$ts = $this->utils->gettimestampnews($news_id);
			$news_url = $this->session->newsurls($news_id, "", "",
														$ts[0], $ts[1],
														$ts[2],-2,0);

			$out .= "<br /><a href=\"".$news_url."\" onmouseout=\"javascript: ".
				"window.status=''; return true;\" ".
				"onmouseover=\"javascript: window.status='".lecho("Print").
				"'; return true;\">".
				"<img src=\"".$this->session->imgurl("icon_print")."\" alt=\"".
				lecho("Print").
				"\" border=\"0\" /></a>";

			$config->ext = $backup_ext;
		}
		$out .= "</center>\n";

		return $this->html->simplebox($out);
	}

	/**
	 * test if news is approved
	 * Returns 1 if the news is approved (state=1)
	 * Returns 0 if the news isn't approved neither doesn't exists.
	 *
	 *@access private
	 *@param integer id of the news
	 *@return boolean 1, 0, -1 if db error
	 */
	Function is_approved($news_id) {
		global $config;

		$sqlc_q = "SELECT state FROM ".$config->tables['news'].
			" WHERE id='".addslashes($news_id)."'";

		$ret = $this->db->query($sqlc_q);
		if (!$ret) {
			$this->utils->debug("News:".__LINE__." ".
				lecho("SQL Failed: ").$this->db->error().
				" SQL: ".$sqlc_q."\n");
			return -1;
		}

		if (!$this->db->num_rows()) {
			return 0;
		} else {
			$state_tmp = $this->db->fetch_array();
			$this->db->free();
			if ($state_tmp['state'] == "1") {
				return 1;
			} else {
				return 0;
			}
		}
	}


	/**
	 * Returns the URL where the news && comments can be found
	 *
	 *@access public
	 *@param integer $news_id the ID of news viewed
	 *@param integer $absolute URL is absolute or not.
	 *@param integer $score the threshold for displaying comments
	 *@param mixed $order TODO
	 *@param integer $theme TODO
	 *@return string the URL where to view it.
	 */
	Function getTopicUrl($news_id, $absolute=1, $score=-2, $order="", $theme=0) {
		$this->comments = LoadClass('Comments');

		$this->utils->debug("News:".__LINE__." getTopicUrl: score=".$score.";\n");
		$ts = $this->comments->utils->gettimestampnews($news_id);

		if ($absolute==1) {
			$news_url = $this->comments->session->newsurls($news_id, "",
						"", $ts[0], $ts[1], $ts[2], $score);
		} else {
			$news_url = $this->comments->session->newsurls($news_id, "",
						"", $ts[0], $ts[1], $ts[2],-2,0);
		}

		//Still I don't understand the need of this one...
		//$news_url = ereg_replace(',[^/]+$', '', $news_url);
		return $news_url;
	}

	/**
	 * Preloads a few things...
	 * FIXME : very basic version
	 *
	 *@access public
	 *@param integer id of the news
	 */
	Function loadResources($news_ids) {
		$sqlc_q = "SELECT id, topic_id, section_id ".
			"timestamp, title, departement, autor_id ".
			"FROM " . $config->tables['news'] .
			" WHERE id IN (" . join(',', $news_id) . ")";

		$ret = $this->db->query($sqlc_q);
		if (!$ret) {
			$this->utils->debug("News:".__LINE__." ".
				lecho("SQL Failed: ").$this->db->error().
				" SQL: ".$sqlc_q."\n");
			return -1;
		}

		while ($row = $this->db->fetch_array()) {
			$this->sqlCache[$row['id']] = $row;
		}
		$this->isCached = 1;
		$this->db->free();
	}

	/**
	 * Displays the title, using the SQL cache
	 *
	 *@access public
	 *@param integer id of the news
	 *@return string news title
	 */
	Function viewShort($news_id) {
		// apparently, isCached can be true and sqlCache undefined...
		// appended with users/post.php3
		if ($this->isCached && isset( $this->sqlCache[$news_id])) {
			$out = $this->sqlCache[$news_id]['title'];
		} else {
			$out = $this->show_title($news_id);
		}
		return $out;
	}

}

?>
