/*
 * The Apache Software License, Version 1.1
 *
 *
 * Copyright (c) 1999 The Apache Software Foundation.  All rights 
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:  
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Xalan" and "Apache Software Foundation" must
 *    not be used to endorse or promote products derived from this
 *    software without prior written permission. For written 
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache",
 *    nor may "Apache" appear in their name, without prior written
 *    permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation and was
 * originally based on software copyright (c) 1999, Lotus
 * Development Corporation., http://www.lotus.com.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */
package org.apache.xalan.xpath;

import java.util.*;
import java.io.*;
import java.net.*;

import org.w3c.dom.*;
import org.xml.sax.*;

/**
 * <meta name="usage" content="internal"/>
 * Class handling an extension namespace for XPath. Provides functions
 * to test a function's existence and call a function
 *
 * @author Sanjiva Weerawarana (sanjiva@watson.ibm.com)
 */
public class ExtensionFunctionHandler 
{
	/**
	 * Patch from Costin Manolache
	 * Remove BSF Specific code, create new Interface ExtensionLiaison
	 * and move BSF Specific code to XSLTJavaClassEngine
   */	
  protected ExtensionLiaison extensionLiaison;
	public String namespaceUri;  // uri of the extension namespace
	// scripting language of implementation
  public String scriptLang = "javaclass";    
  public String scriptSrc;     // script source to run (if any)
  public String scriptSrcURL;  // URL of source of script (if any)
  protected Hashtable functions = new Hashtable (); // functions of namespace
  protected Hashtable elements = new Hashtable (); // elements of namespace
  protected boolean componentStarted; // true when the scripts in a
  // component description (if any) have
  // been run
	
	public static final String DEFAULT_EXTENSION_LIAISON =
    "org.apache.xalan.xpath.XSLTJavaClassEngine";

  /////////////////////////////////////////////////////////////////////////
  // Constructors
  /////////////////////////////////////////////////////////////////////////

  /**
   * Construct a new extension namespace handler for a given extension NS.
   * This doesn't do anything - just hang on to the namespace URI.
   * 
   * @param namespaceUri the extension namespace URI that I'm implementing
   */
	public ExtensionFunctionHandler (String namespaceUri) 
	{
		this.namespaceUri = namespaceUri;
		try 
		{
			Class c=Class.forName(DEFAULT_EXTENSION_LIAISON);
			extensionLiaison=(ExtensionLiaison)c.newInstance();
		} 
		catch( Exception ex ) 
		{
			extensionLiaison=null;
		}
	}

  /////////////////////////////////////////////////////////////////////////

  /**
   * Construct a new extension namespace handler given all the information
   * needed. 
   * 
   * @param namespaceUri the extension namespace URI that I'm implementing
   * @param funcNames    string containing list of functions of extension NS
   * @param lang         language of code implementing the extension
   * @param srcURL       value of src attribute (if any) - treated as a URL
   *                     or a classname depending on the value of lang. If
   *                     srcURL is not null, then scriptSrc is ignored.
   * @param scriptSrc    the actual script code (if any)
   */
  public ExtensionFunctionHandler (String namespaceUri, String funcNames, 
                                   String lang, String srcURL, String src) 
  {
    this (namespaceUri);
    setFunctions (funcNames);
    if( extensionLiaison!=null)
      extensionLiaison.setScript (lang, srcURL, src);
  }

  /////////////////////////////////////////////////////////////////////////
  // Main API
  /////////////////////////////////////////////////////////////////////////

  /**
   * Set function local parts of extension NS.
   *
   * @param functions whitespace separated list of function names defined
   *        by this extension namespace.
   */
  public void setFunctions (String funcNames) 
  {
    if (funcNames == null) 
    {
      return;
    }
    StringTokenizer st = new StringTokenizer (funcNames, " \t\n\r", false);
    Object junk = new Object ();
    while (st.hasMoreTokens ()) 
    {
      String tok = st.nextToken ();
      functions.put (tok, junk); // just stick it in there basically
    }
  }

  /**
   * Set element local parts of extension NS.
   *
   * @param elements whitespace separated list of element names defined
   *        by this extension namespace.
   */
  public void setElements (String elemNames) 
  {
    if (elemNames == null) 
    {
      return;
    }
    StringTokenizer st = new StringTokenizer (elemNames, " \t\n\r", false);
    Object junk = new Object ();
    while (st.hasMoreTokens ()) 
    {
      String tok = st.nextToken ();
      elements.put (tok, junk); // just stick it in there basically
    }
  }
  
  /////////////////////////////////////////////////////////////////////////

  /**
   * Set the script data for this extension NS. If srcURL is !null then
   * the script body is read from that URL. If not the scriptSrc is used
   * as the src. This method does not actually execute anything - that's
   * done when the component is first hit by the user by an element or 
   * a function call.
   *
   * @param lang      language of the script.
   * @param srcURL    value of src attribute (if any) - treated as a URL
   *                  or a classname depending on the value of lang. If
   *                  srcURL is not null, then scriptSrc is ignored.
   * @param scriptSrc the actual script code (if any)
   */
  public void setScript (String lang, String srcURL, String scriptSrc) 
  {
    if (extensionLiaison != null)
			extensionLiaison.setScript(lang, srcURL, scriptSrc);  
  }

  /////////////////////////////////////////////////////////////////////////

  /**
   * Tests whether a certain function name is known within this namespace.
   *
   * @param function name of the function being tested
   *
   * @return true if its known, false if not.
   */
  public boolean isFunctionAvailable (String function) 
  {
    return (functions.get (function) != null);
  }
  
  /**
   * Tests whether a certain element name is known within this namespace.
   *
   * @param function name of the function being tested
   *
   * @return true if its known, false if not.
   */
  public boolean isElementAvailable (String element) 
  {
    return (elements.get (element) != null);
  }


  //Hashtable m_cachedMethods = null;
  
  /**
   * call the named method on the object that was loaded by eval. 
   * The method selection stuff is very XSLT-specific, hence the
   * custom engine.
   *
   * @param object ignored - should always be null
   */
  public Object callJava (Object object, String method, Object[] args, 
                          Object methodKey) 
    throws XPathException //BSFException
  {    
		if( extensionLiaison==null)
      return null;
    return extensionLiaison.callJava( object, method, args, methodKey );
  }

  /////////////////////////////////////////////////////////////////////////

  /**
   * Process a call to a function.
   *
   * @param funcName Function name.
   * @param args     The arguments of the function call.
   *
   * @return the return value of the function evaluation.
   *
   * @exception XSLProcessorException thrown if something goes wrong 
   *            while running the extension handler.
   * @exception MalformedURLException if loading trouble
   * @exception FileNotFoundException if loading trouble
   * @exception IOException           if loading trouble
   * @exception SAXException          if parsing trouble
   */
  public Object callFunction (String funcName, Vector args, Object methodKey, Class javaClass)
    throws XPathException 
  {
    if (!componentStarted) 
    {
      startupComponent (javaClass);
    }
  
		if( extensionLiaison==null)
      return null;
		
    return  extensionLiaison.callFunction( funcName, args,
		 		 		 		 		       methodKey, javaClass );
  }

  /**
   * Start the component up by executing any script that needs to run
   * at startup time. This needs to happen before any functions can be
   * called on the component. 
   * 
   * @exception XPathProcessorException if something bad happens.
   */
  protected void startupComponent (Class classObj) throws  XPathProcessorException 
  {   
		if( extensionLiaison == null)
      return;  
		
		extensionLiaison.checkInit();
    extensionLiaison.startupComponent( classObj );
    componentStarted=true;
  } 
	
	/**
   * Interface for connecting extension functions to xalan.
   * Based on ExtensionFunctionHandler
   *
   * @author Sanjiva Weerawarana (sanjiva@watson.ibm.com)
   * @author Costin@eng.sun.com
   */
  public static interface  ExtensionLiaison
  {
    /** Make sure the component is initialized
     */
    public void checkInit()
      throws  XPathProcessorException;
		
    /**
     * Start the component up by executing any script that needs to run
     * at startup time. This needs to happen before any functions can be
     * called on the component. 
     * 
     * @exception XPathProcessorException if something bad happens.
     */
    public void startupComponent (Class classObj)
      throws  XPathProcessorException;

    /**
     * Process a call to a function.
     *
     * @param funcName Function name.
     * @param args     The arguments of the function call.
     *
     * @return the return value of the function evaluation.
     *
     * @exception XSLProcessorException thrown if something goes wrong 
     *            while running the extension handler.
     * @exception MalformedURLException if loading trouble
     * @exception FileNotFoundException if loading trouble
     * @exception IOException           if loading trouble
     * @exception SAXException          if parsing trouble
     */
    public Object callFunction (String funcName, Vector args, Object methodKey,
		 		 		 		 Class javaClass)
      throws  XPathException;
    
    /**
     * call the named method on the object that was loaded by eval. 
     * The method selection stuff is very XSLT-specific, hence the
     * custom engine.
     *
     * @param object ignored - should always be null
     */
    public Object callJava (Object object, String method, Object[] args, 
		 		 		     Object methodKey) 
      throws XPathException;
		
		/**
     * Set the script data for this extension NS. If srcURL is !null then
     * the script body is read from that URL. If not the scriptSrc is used
     * as the src. This method does not actually execute anything - that's
     * done when the component is first hit by the user by an element or 
     * a function call.
     *
     * @param lang      language of the script.
     * @param srcURL    value of src attribute (if any) - treated as a URL
     *                  or a classname depending on the value of lang. If
     *                  srcURL is not null, then scriptSrc is ignored.
     * @param scriptSrc the actual script code (if any)
     */
    public void setScript (String lang, String srcURL, String scriptSrc);

   }


}
