/* SingletonBroker.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.util;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.grinvin.xml.XMLUtils;

import org.jdom.Element;

/**
 * An object of this type keeps track of a set of singleton objects of
 * type T which can be requested by id. The list of available singleton objects
 * is instantiated at initialization time. An XML configuration file
 * determines which objects should be instantiated.
 */
public class SingletonBroker<T> {
    
    /**
     * Maps an id to the corresponding singleton.
     */
    private final Map<String,T> singletons;
    
    /**
     * Ordered list of ids.
     */
    private final List<String> ids;
    
    
    /**
     * Instantiates a broker object and loads the corresponding objects.
     * @param configPath file name in the class path of the XML file with
     * the requested objects
     */
    public SingletonBroker(String configPath) {
        try {
            this.singletons = new HashMap<String,T> ();
            this.ids = new ArrayList<String> ();
            Element element = XMLUtils.loadFromClassPath(configPath);
            if (element != null) {
                for (Object o : element.getChildren("package")) {
                    Element pkg = (Element)o;
                    String packageName = pkg.getAttributeValue("name");
                    for (Object obj : pkg.getChildren("id")) {
                        Element child = (Element)obj;
                        String id = child.getTextTrim();
                        String className = child.getAttributeValue("class");
                        if (className == null) {
                            int pos = id.lastIndexOf('.');
                            className = id.substring(pos+1);
                        }
                        try {
                            String fullClassName = packageName + "." + className;
                            singletons.put(id, (T)Class.forName(fullClassName).newInstance());
                            ids.add(id);
                        } catch (Exception ex) {
                            Logger.getLogger("org.grinvin.util")
                            .log(Level.WARNING,
                                    "Failed to create singleton object '" + id + "' of type '" + className +"'",
                                    ex);
                        }
                        
                    }
                }
            }
        } catch (IOException ex) {
            Logger.getLogger("org.grinvin.util")
            .log(Level.SEVERE, "Failed to create singleton broker", ex);
            throw new RuntimeException("Unable to initialize singleton broker", ex);
        }
    }
    
    /**
     * Add all singleton objects to the given list.
     */
    public void addToList(List<T> list) {
        for (String id : ids)
            list.add(singletons.get(id));
        // list.addAll (singletons.values()) cannot be used because it loses the order
    }
    
    /**
     * Return the list of all ids, in the form of an <code>Iterable</code>.
     */
    public Iterable<String> getIds() {
        return ids;
    }
    
    /**
     * Return the object with the given id.
     */
    public T get(String id) {
        return singletons.get(id);
    }
}
