package com.clarkware.junitperf;

import junit.framework.AssertionFailedError;
import junit.framework.Test;
import junit.framework.TestResult;
import junit.extensions.TestDecorator;

/**
 * The <code>TimedTest</code> is a test decorator that 
 * runs a test and measures the elapsed time of the test.
 * <p>
 * A <code>TimedTest</code> is constructed with a specified
 * maximum elapsed time.  By default, a <code>TimedTest</code> 
 * will wait for the completion of its decorated test and
 * then fail if the maximum elapsed time was exceeded.
 * Alternatively, a <code>TimedTest</code> can be constructed 
 * to immediately signal a failure when the maximum elapsed time 
 * of its decorated test is exceeded.  In other words, the 
 * <code>TimedTest</code> will <b>not</b> wait for its decorated 
 * test to run to completion if the maximum elapsed time is 
 * exceeded.
 * <p>
 * For example, to decorate the <code>ExampleTest</code> 
 * as a <code>TimedTest</code> that waits for the 
 * <code>ExampleTest</code> to run to completion and 
 * then fails if the maximum elapsed time of 2 seconds 
 * is exceeded, use: 
 * <blockquote>
 * <pre>
 * Test timedTest = new TimedTest(new ExampleTest(), 2000);
 * </pre>
 * </blockquote> 
 * or, alternatively, use: 
 * <blockquote>
 * <pre>
 * Test timedTest = new TimedTest(new ExampleTest(), 2000, true);
 * </pre>
 * </blockquote> 
 * <p>
 * Alternatively, to decorate the <code>ExampleTest</code>
 * as a <code>TimedTest</code> that fails immediately when
 * the maximum elapsed time of 2 seconds is exceeded, use:
 * <blockquote>
 * <pre>
 * Test timedTest = new TimedTest(new ExampleTest(), 2000, false);
 * </pre>
 * </blockquote>
 *
 * @author <a href="mailto:mike@clarkware.com">Mike Clark</a>
 * @author <a href="http://www.clarkware.com">Clarkware Consulting, Inc.</a>
 * @author Ervin Varga
 *
 * @see junit.extensions.TestDecorator
 */

public class TimedTest extends TestDecorator {

	private final long _maxElapsedTime;
	private final boolean _waitForCompletion;
	private boolean _maxElapsedTimeExceeded;

	/**
	 * Constructs a <code>TimedTest</code> to decorate the 
	 * specified test with the specified maximum elapsed time.
	 * <p>
	 * The <code>TimedTest</code> will wait for the completion 
	 * of its decorated test and then fail if the maximum elapsed 
	 * time was exceeded.
	 * 
	 * @param test Test to decorate.
	 * @param maxElapsedTime Maximum elapsed time (ms).
	 */
	public TimedTest(Test test, long maxElapsedTime) {
		this(test, maxElapsedTime, true);
	}
	
	/**
	 * Constructs a <code>TimedTest</code> to decorate the 
	 * specified test with the specified maximum elapsed time.
	 * 
	 * @param test Test to decorate.
	 * @param maxElapsedTime Maximum elapsed time (ms).
	 * @param waitForCompletion <code>true</code> (default) to 
	 *        indicate that the <code>TimedTest</code> should wait 
	 *        for its decorated test to run to completion and then 
	 *        fail if the maximum elapsed time was exceeded; 
	 *        <code>false</code> to indicate that the 
	 *        <code>TimedTest</code> should immediately signal 
	 *        a failure when the maximum elapsed time is exceeded.
	 */
	public TimedTest(Test test, long maxElapsedTime, boolean waitForCompletion) {
		super(test);
		_maxElapsedTime = maxElapsedTime;
		_waitForCompletion = waitForCompletion;
		_maxElapsedTimeExceeded = false;
	}

	/**
	 * Returns the number of tests in this test.
	 *
	 * @return Number of tests.
	 */
	public int countTestCases() {
		return super.countTestCases();
	}
	
	/**
	 * Determines whether the maximum elapsed time of
	 * the test was exceeded.
	 *
	 * @return <code>true</code> if the max elapsed time
	 *         was exceeded; <code>false</code> otherwise.
	 */
	public boolean outOfTime() {
		return _maxElapsedTimeExceeded;		
	}

	/**
	 * Runs the test.
	 *
	 * @param result Test result.
	 */
	public void run(TestResult result) {
		//
		// TODO: May require a strategy pattern
		//       if other algorithms emerge.
		// 
		if (_waitForCompletion) {
			runUntilTestCompletion(result);
		} else {
			runUntilTimeExpires(result);
		}
	}
	
	/**
	 * Runs the test until test completion and then signals
	 * a failure if the maximum elapsed time was exceeded.
	 *
	 * @param result Test result.
	 */
	protected void runUntilTestCompletion(TestResult result) {

		long beginTime = System.currentTimeMillis();

		super.run(result);

		long endTime = System.currentTimeMillis();
		long elapsedTime = endTime - beginTime;

		System.out.println(toString() + ": " + elapsedTime + " ms");
		System.out.flush();

		//
		// TODO:
		// Use getTest() rather than fTest for JUnit 3.5 and above.
		//
		if (elapsedTime > _maxElapsedTime) {	
			_maxElapsedTimeExceeded = true;			
			result.addFailure(fTest,
				new AssertionFailedError("Maximum elapsed time exceeded!" +
					" Expected " + _maxElapsedTime + "ms, but was " +
					elapsedTime + "ms."));
		}
	}
	
	/**
	 * Runs the test and immediately signals a failure 
	 * when the maximum elapsed time is exceeded.
	 *
	 * @param result Test result.
	 */
	protected void runUntilTimeExpires(final TestResult result) {
	
		Thread t = new Thread(new Runnable() {
			public void run() {
				TimedTest.super.run(result);
			}
		});

		long beginTime = System.currentTimeMillis();
		
		t.start();

		try {
		
			t.join(_maxElapsedTime);
		
		} catch(InterruptedException ignored) {}

		long endTime = System.currentTimeMillis();
		long elapsedTime = endTime - beginTime;

		System.out.println(toString() + ": " + elapsedTime + " ms");
		System.out.flush();
	
		if (t.isAlive()) {				
			_maxElapsedTimeExceeded = true;
			result.addFailure(fTest,
				new AssertionFailedError("Maximum elapsed time (" + _maxElapsedTime + " ms) exceeded!"));
		}	
	}

	/**
	 * Returns the test description.
	 *
	 * @return Description.
	 */
	public String toString() {
		if (_waitForCompletion) {
			return "TimedTest (WAITING): " + super.toString();
		} else {
			return "TimedTest (NON-WAITING): " + super.toString();
		}
	}
}
