/* filename: rlpr-net.c
 * project: rlpr
 * author: meem  --  meem@sherilyn.wustl.edu
 * version: $Id: rlpr-net.c,v 1.14 1996/10/12 09:20:28 meem Exp $
 * contents: network-related parts of rlpr (all the socket-based functions)
 *
 * Time-stamp: <1996/10/11 16:23 -- meem@sherilyn.wustl.edu>
 */

/* copyright (c) 1996 meem, meem@gnu.ai.mit.edu
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 1, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 */

#include "config.h"

#include <sys/types.h>  
#include <netinet/in.h>                 /* struct socketaddr_in definition */
#include <sys/socket.h>                 /* all the different socket types */
#include <sys/stat.h>
#include <fcntl.h>
#include <netdb.h>                      /* network-specific functions here */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/time.h>                   /* for timeval in select() calls */
#include <unistd.h>                     /* for unlink(), etc */
#include <errno.h>
#include "rlpr-common.h"
#include "rlpr-net.h"

/* LOCAL FUNCTIONS */
static void get_and_verify_ack(int sock, char *caller);

static int        sockfd;       	/* socket we're using.. this should
					   probably be encapsulated sometime */
void open_connection(void) {
  struct sockaddr_in sin;               /* socket struct with system info */
  struct sockaddr_in sin_local;   	/* to bind a socket to a number */
  struct hostent *hp;                   /* ptr to struct given by gethostbyname() */
  int i;		                /* scrap variable */

  /* zero out structs to get the unused part zeroed */
  memset(&sin,       0, sizeof(sin));

  
  if ((hp = gethostbyname(net_.localhost)) == NULL)
    rlpr_fatal("hostname \"%s\" does not seem to exist!", net_.localhost);

  sin_local.sin_family = AF_INET;
  memcpy(&sin_local.sin_addr, hp->h_addr, hp->h_length);
    
  /* obtain a socket descriptor */
  if ((sockfd = socket(AF_INET, SOCK_STREAM, 0)) < 0) /* 0 == auto proto config */
    rlpr_fatal("socket: %s", ERRNO);

  /* here if we're setuid root we can just bind to the right port
     immediately and be done with it. otherwise we need to use a
     proxy. if the proxy variable isn't set we assume they have it
     installed setuid root. */

  if (!*net_.proxyhost) {
    if (geteuid() != 0)
      rlpr_fatal("not setuid root. see %s(1) about using a proxyhost", name);
    
    for (i = LO_LPD_FROM_PORT; i <= HI_LPD_FROM_PORT; i++) {
      sin_local.sin_port = htons(i);
      if (bind(sockfd, (struct sockaddr *) &sin_local, sizeof(sin_local)) == 0)
	break;
      else if (i == HI_LPD_FROM_PORT)
	rlpr_fatal("bind to ports %hi-%hi: %s", LO_LPD_FROM_PORT, HI_LPD_FROM_PORT, ERRNO);
    }
  }

  /* either we're connecting to the proxy or the real host */

  if (!*net_.proxyhost) /* USE PRINTHOST */ {

    if (!(hp = gethostbyname(net_.printhost)))
      rlpr_fatal("hostname \"%s\" does not seem to exist!", net_.printhost);
    sin.sin_port = htons(LPD_TO_PORT);  /* put it in network byte order */
        
  } else /* USE PROXY */ {

    if (!(hp = gethostbyname(net_.proxyhost)))
      rlpr_fatal("hostname \"%s\" does not seem to exist!", net_.proxyhost);
    sin.sin_port = htons(net_.port);    /* put it in network byte order */
  }

  /* fill in sin struct, and connect  */
  memcpy(&sin.sin_addr, hp->h_addr, hp->h_length); /* already in nbo */
  sin.sin_family = AF_INET;

  if (connect(sockfd, (struct sockaddr *) &sin, sizeof(sin)) < 0)
    if (errno == ECONNREFUSED)
      rlpr_fatal("connection to %s refused on port %hi! is the daemon up?",
		 *net_.proxyhost ? net_.proxyhost : net_.printhost, ntohs(sin.sin_port));
    else rlpr_fatal("connect: %s", ERRNO);

  if (*net_.proxyhost) {	/* if proxy, send out final destination */
    safe_writen(sockfd, net_.printhost, strlen(net_.printhost));
    safe_writen(sockfd, "\n", 1);
  }
}


void close_connection(void) {
  if (close(sockfd) < 0)
    rlpr_fatal("close: %s", ERRNO);
}

/* i know sprintf is neater. this is more accurate though because
   i don't need to malloc() or guess a bufsiz for sprintf */

void send_recvj_req(void) {
  char c = RECVJ;
  safe_writen(sockfd, &c, 1);
  safe_writen(sockfd, opts_.printer, strlen(opts_.printer));
  safe_writen(sockfd, "\n", 1);
  get_and_verify_ack(sockfd, __FUNCTION__);
}

static void get_and_verify_ack(int sock, char *caller) {
  static char ack[1];
  
  if (read(sock, ack, 1) < 0) /* check acknolwedgement */
    rlpr_fatal("%s: while reading ack from server: %s", caller, ERRNO);

  if (*ack != 0)
    rlpr_fatal("%s: lpd gave negative ack, bailing out!", caller);
}


void send_cf(int cfd, char *cfname) {
  char buf[BUFSIZ];
  int count;

  if (lseek(cfd, 0, SEEK_SET) < 0)
    rlpr_fatal("lseek on control file %s: %s", cfname, ERRNO);

  /* send header for control file */
  sprintf(buf, "%c%lu %s\n", RECVCF, filesz(cfd), cfname);
  safe_writen(sockfd, buf, strlen(buf));
  get_and_verify_ack(sockfd, __FUNCTION__);
  
  /* send control file */
  while ((count = read(cfd, buf, BUFSIZ)) > 0)
    safe_writen(sockfd, buf, count);

  if (count < 0)
    rlpr_fatal("read on control file %s: %s", cfname, ERRNO);

  safe_writen(sockfd, "\0", 1);	/* just being explicit */
  get_and_verify_ack(sockfd, __FUNCTION__);
}

void send_df(char *filename, char *dfname) {
  char buf[BUFSIZ];		/* temporary buffer */
  int count;
  int dfd;			/* descriptor for datafile */
  
  if (!filename) /* STDIN */ {

    /* we need to make a dummy file to send, so that we know the size
     * of stdin for machines that are not local. this is annoying but
     * apparently required for most lpd's (even though RFC 1179 seems
     * to say otherwise)
     */

    filename = rlpr_malloc(strlen(dfname) + strlen(opts_.tmpdir) + 2);
    sprintf(filename, "%s/%s", opts_.tmpdir, dfname);

    if ((dfd = open(filename, O_RDWR|O_TRUNC|O_CREAT, 0600)) < 0)
      rlpr_fatal("%s - cannot create: %s",filename, ERRNO);
    unlink(filename);

    while ((count = read(STDIN_FILENO, buf, BUFSIZ)) > 0)
      safe_writen(dfd, buf, count);
    if (count < 0) 
      rlpr_fatal("read on stdin: %s", ERRNO);

    lseek(dfd, 0, SEEK_SET);

    /* free filename and reset to NULL both so we prevent a memory leak
       and so we know that the user is reading from stdin down below */
    
    free(filename);
    filename = NULL;
    
    /* else have a filename from the commandline */

  } else if ((dfd = open(filename, O_RDONLY)) < 0)
    rlpr_fatal("%s - cannot open: %s", filename, ERRNO);

  /* COMMON */

  sprintf(buf, "%c%lu %s\n", RECVDF, filesz(dfd), dfname);
  safe_writen(sockfd, buf, strlen(buf));
  get_and_verify_ack(sockfd, __FUNCTION__);
  
  while ((count = read(dfd, buf, BUFSIZ)) > 0)
    safe_writen(sockfd, buf, count);
  if (count < 0)
    rlpr_fatal("read on data file %s: %s", filename ? filename : "stdin", ERRNO);

  safe_writen(sockfd, "\0", 1);
  get_and_verify_ack(sockfd, __FUNCTION__);
}
