/*
 * Programm XBLAST V2.0 or higher
 * (C) by Oliver Vogel (e-mail: vogel@ikp.uni-koeln.de)
 * May 16th 1996
 * started August 1993
 *
 * File: maze.c
 * level management
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public Licences as by published
 * by the Free Software Foundation; either version 2; or (at your option)
 * any later version
 *
 * This program is distributed in the hope that it will entertaining,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILTY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
 * Publis License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 */



#include <stdio.h>
#include <stdlib.h>

#include "include.h"
#include "mytypes.h"
#include "const.h"
#include "func.h"
#include "data.h"
#include "graphics.h"
#include "maze.h"
#include "main.h"
#include "expl.h"
#include "intro.h"
#include "shrink.h"

#define BOMB_TIME 64
#define SHORT_FUSE 32
#define LONG_FUSE 128

#define BOMB_ERROR_PROB 16
#define BOMB_DELAY 5

/*
 * local prototypes
 */
#ifdef __STDC__
static void my_swap (int *a, int *b);
static void one_expl_at (int x, int y, int ra, int ri, int outer, int inner);
static void blast_extra_block (int x, int y);
static void del_explosion (Explosion *ptr);
static void bombdirshift (Explosion *ptr, int dir, int WhenUp, int WhenLeft, 
			  int WhenDown, int WhenRight);
static void bombrandomdir (Explosion *ptr);
static void do_player_click (Explosion *bomb, int player, BMPlayer *ps);
static void do_wall_click (Explosion *bomb);
static void do_bomb_click (Explosion *bomb);
static void sprex (int lx, int ly, int range, int type, int type_extr);
static void do_special_bomb_types (Explosion *ptr);
#else
static void my_swap ();
static void one_expl_at ();
static void blast_extra_block ();
static void del_explosion ();
static void bombdirshift ();
static void bombrandomdir ();
static void do_player_click ();
static void do_wall_click ();
static void do_bomb_click ();
static void sprex ();
static void do_special_bomb_types ();
#endif

/*
 * global variables
 */ 
int redraw_maze[MAZE_W][MAZE_H + 2];

/*
 * local variables
 */
static BMLevelData *cur_level;

static int expl_maze[MAZE_W][MAZE_H];
static int old_expl_maze[MAZE_W][MAZE_H];
static Explosion *bomb_maze[MAZE_W][MAZE_H];
static int maze[MAZE_W][MAZE_H];
static int extra[MAZE_W][MAZE_H];
static int dist_extra[MAZE_W][MAZE_H];
static int dist_x[MAZE_W*MAZE_H], dist_y[MAZE_W*MAZE_H];
static int block[MAX_BLOCK];
static Explosion *expl_list;
static Explosion *expl_end;
static int num_expl = 0 ;
static int shadow_mode;
static int distrib_extras;
static int BC_mode;
static int WC_mode;
static int PC_mode;
static fuse_times[NUM_FUSES] = {SHORT_FUSE, BOMB_TIME, LONG_FUSE};
static int evilBMT;
static int initial_bomb_dir;


/* 
 * Shuffling start positions 
 */
static int sposswap[MAX_PLAYER];

#ifdef __STDC__
void 
init_shuffle_startpos (void)
#else
void 
init_shuffle_startpos ()
#endif
{
  int i,x;
  int j;
  int a;

  for(i=0; i<MAX_PLAYER; i++) {
    sposswap[i] = i;
  }
  for(x=0; x<(MAX_PLAYER-1); x++) {
    for(i=0; i<MAX_PLAYER; i++) {
      j = (rand()>>4) % MAX_PLAYER;
      a           = sposswap[i];
      sposswap[i] = sposswap[j];
      sposswap[j] = a;
    }
  }
}

  /* public function load_maze */

#ifdef __STDC__
void 
load_maze (BMPlayer *stat,
	  int level,   
	  XBConfig *config,
	  XBSettings *setup,
	  int *min_range, 
	  int *min_bombs)
#else
void 
load_maze (stat, level, config, setup, min_range, min_bombs )
     BMPlayer *stat;
     int level; 
     XBConfig *config;
     XBSettings *setup;
     int *min_range, *min_bombs;
#endif
{
  int x,y,i,player;
  int rnd;
  static int tmpx[MAX_PLAYER];
  static int tmpy[MAX_PLAYER];


  expl_list = NULL;
  expl_end = NULL;
  num_expl = 0;

  cur_level = maze_data[level];

  init_scramble(&(cur_level->scrdraw), &(cur_level->scrdel));
  (*cur_level->shrink_func)();

  special_init_function = cur_level->init_func;
  special_game_function = cur_level->game_func;
  special_extra_function = cur_level->extra_func;
  special_key_function = cur_level->key_func;

  shadow_mode = cur_level->shadow;
  distrib_extras = cur_level->distrib_extras;
  BC_mode = cur_level->bomb_click;
  WC_mode = cur_level->wall_click;
  PC_mode = cur_level->player_click;

  initial_bomb_dir = cur_level->bomb_dir;

  *min_range = cur_level->range;
  *min_bombs = cur_level->bombs;

  cur_bomb_time = fuse_times[cur_level->fuse_time];
  defaultBMT = cur_level->defaultBMT;
  buttonBMT = cur_level->buttonBMT;
  evilBMT = cur_level->evilBMT;

  /* Must load for all players as we scramble the x/y positions */
  for (player = 0; player < MAX_PLAYER; player ++) {
    if ( GM_NoGrid & cur_level->game_mode) {
      stat[player].y = cur_level->position[player].y;
      stat[player].x = cur_level->position[player].x;
    } else {
      stat[player].y = (cur_level->position[player].y-1)	* BLOCK_HEIGHT;
      stat[player].x =  cur_level->position[player].x * BLOCK_WIDTH;
    }
    stat[player].anime = 0;
    stat[player].invincible = NEW_INVINCIBLE;
    stat[player].next_blink = 0;
    stat[player].illness = cur_level->init_health;
    stat[player].illtime = 0;
    stat[player].junkie = FALSE;
    stat[player].junkie_time = MAX_JUNKIE_TIME;
    stat[player].health = cur_level->init_health;
    stat[player].dying = 0;
    stat[player].stunned = 0;
    stat[player].lives = setup->max_lives;
    stat[player].range = cur_level->range;
    stat[player].bombs = cur_level->bombs;
    stat[player].extra_flags = cur_level->init_flags;
    stat[player].special_bombs = 0;
    if (IF_RC & stat[player].extra_flags) {
      stat[player].remote_control = 1;
    } else {
      stat[player].remote_control = 0;
    }
    if (IF_Teleport & stat[player].extra_flags) {
      stat[player].teleport = 1;
    } else {
      stat[player].teleport = 0;
    }      
    if (IF_Kick & stat[player].extra_flags) {
      stat[player].kick = 1;
    } else {
      stat[player].kick = 0;
    }
    stat[player].air_button = FALSE;
    stat[player].num_extras = 0;
    stat[player].abort = FALSE;
    stat[player].d_ist = GoStop;
    stat[player].d_soll= GoStop;
    stat[player].d_look= GoDown;
  }

    /* Shuffle start positions (if desired) */

    if ( (setup->random_spos) 
	&& ( (cur_level->game_mode) & GM_Random) ) {
      int i;

      for(i=0;i<MAX_PLAYER;i++) {
        tmpx[i] = stat[i].x;
        tmpy[i] = stat[i].y;
      }
      for(i=0;i<MAX_PLAYER;i++) {
        stat[i].x = tmpx[sposswap[i]];
        stat[i].y = tmpy[sposswap[i]];
      }
    }

  for (x = 0; x < MAZE_W; x++) {
    for (y = 0; y < MAZE_H; y++) {
      bomb_maze[x][y] = NULL;
      expl_maze[x][y] = 0;
      old_expl_maze[x][y] = 0;
      maze[x][y] = cur_level->maze[x][y];

      if (maze[x][y] == BTExtra) {
	rnd = rand() % 64;
	if (rnd < cur_level->prob.bomb) {
	  extra[x][y] = BTBomb;
	} else if (rnd < cur_level->prob.range) {
	  extra[x][y] = BTRange;
	} else if (rnd < cur_level->prob.ill) {
	  extra[x][y] = BTSick;
	} else if (rnd < cur_level->prob.invinc) {
	  extra[x][y] = BTSpecial;
	} else if (rnd < cur_level->prob.evil) {
	  extra[x][y] = BTEvil;
	} else {
	  extra[x][y] = BTFree;
	}
      } else {
	extra[x][y] = BTFree;
	
	if (maze[x][y] == BTFree) {
	  /* set shadow */
	  switch( maze[x-1][y] ) {
	  case BTBlock:
	    if ( shadow_mode & ShadowBlock)
	      maze[x][y] = BTShadow;
	    break;
	    
	  case BTExtra:
	    if ( shadow_mode & ShadowExtra)
	      maze[x][y] = BTShadow;
	    break;
	  }
	}
      }
      
      redraw_maze[x][y] = FALSE;
    }
  }

  for (i = 0; i < MAX_BLOCK; i++) {
    block[i] = cur_level->block[i].id;
  }

  for (player = 0; player < config->num_disp; player++) {
    for (i=0; i< MAX_BLOCK; i++) {
      init_block(player, block[i], i, 
		 cur_level->block[i].fg, 
		 cur_level->block[i].bg,
		 cur_level->block[i].add);
    }
  }

  for (player = 0; player < config->num_disp; player++)  {
    init_explosion_blocks(player);
  }
}



/* public function show_levels */
#ifdef __STDC__
void 
show_levels (void)
#else
void 
show_levels ()
#endif
{
  int i;

  printf("%3s  %-36s %-30s\n","Lvl","Level Name","Author");  
  for (i=0; i < LEVEL_MAX; i++) {
    printf("%3d  %-36s %-30s\n",i,
           maze_data[i]->name, maze_data[i]->author);
  }
}



/* public function set_victories */
#ifdef __STDC__
void 
set_victories (int num_victories)
#else
void 
set_victories (num_victories)
     int num_victories;
#endif
{
 int x,y;

 for (x = num_victories + 6; x < MAZE_W; x ++) {
   for (y = 4; y < 12; y++) {
     maze_data[LEVEL_MAX]->maze[x][y] = 6;
   }
 }
}



/* public function get_level_name */
#ifdef __STDC__
char *
get_level_name (int level)
#else
char *
get_level_name(level)
     int level;
#endif
{
  return maze_data[level]->name;
}


#ifdef __STDC__
char *
get_level_res_name (int level)
#else
char *
get_level_res_name(level)
     int level;
#endif
{
  return maze_data[level]->resource;
}


#ifdef __STDC__
char *
get_level_author (int level)
#else
char *
get_level_author (level)
     int level;
#endif
{
  static char result[256];

  sprintf(result, "created by %s", maze_data[level]->author);
  return result;
}



#ifdef __STDC__
char *
get_level_tip (int level)
#else
char *
get_level_tip (level)
     int level;
#endif
{
  return maze_data[level]->DSCtips;
}



/* public function get_game_mode */
#ifdef __STDC__
int 
get_game_mode (int level)
#else
int 
get_game_mode (level)
     int level;
#endif
{
  return(maze_data[level]->game_mode);
}


/* public unload_blocks */
#ifdef __STDC__
void 
unload_blocks (int num_disp)
#else
void 
unload_blocks (num_disp)
     int num_disp;
#endif
{
  int player, i;

  for(player = 0; player < num_disp; player ++) {
    for (i = 0; i< MAX_BLOCK; i++) {
	free_block(player, i);
      }
    free_explosion_blocks(player);
  }
}



/* public function draw_maze */
#ifdef __STDC__
void 
draw_maze (int player)
#else
void 
draw_maze (player)
     int player;
#endif
{
  int x,y;

  for (x=0; x<MAZE_W; x++) {
    for (y=0; y<MAZE_H; y++) {
      draw_block(player, x, y, (int) maze[x][y]);
    }
  }
}



/* public function redraw_rectangles */

#ifdef __STDC__
void 
set_redraw_rectangles (void)
#else
void 
set_redraw_rectangles ()
#endif
{
  int x,y;

  for (y=0; y<MAZE_H; y++) {
    for (x=0; x<MAZE_W; x++) { 
      if (expl_maze[x][y] != old_expl_maze[x][y]) {
	redraw_maze[x][y] = TRUE;
      }
    }
  }
  
  for (y=0; y<MAZE_H+2; y++) {
    for (x=0; x<MAZE_W; x++) { 
      if (redraw_maze[x][y]) {
	add_rectangle(x, y);
      }
    }
  }
}



/* public function mark_maze */
#ifdef __STDC__
void 
mark_maze (int x1,
	   int y1, 
	   int x2, 
	   int y2)
#else
void 
mark_maze (x1,y1, x2, y2)
     int x1, y1, x2, y2;
#endif
{
  int x,y;

  for (x=MAX(x1,0); x<=MIN(x2,MAZE_W-1); x++) {
    for (y=MAX(y1,0); y<=MIN(y2,MAZE_H+1); y++) {
      redraw_maze[x][y] = TRUE;
    }
  }
}



/* public function clear_redraw_map */
#ifdef __STDC__
void 
clear_redraw_map (void)
#else
void 
clear_redraw_map ()
#endif
{
  int x,y;

  for(x=0; x<MAZE_W; x++) {
    for(y=0; y<(MAZE_H+2); y++) {
      redraw_maze[x][y] = FALSE;
    }
  }

  for(x=0; x<MAZE_W; x++) {
    for(y=0; y<MAZE_H; y++) {
      old_expl_maze[x][y] = expl_maze[x][y];
      expl_maze[x][y] = 0;
    }
  }
}



/* public function update_maze */
#ifdef __STDC__
void 
update_maze (int player)
#else
void 
update_maze (player)
     int player;
#endif
{
  int x,y;

  for (x=0; x<MAZE_W; x++) {
    for (y=0; y<MAZE_H; y++) {
      if (redraw_maze[x][y] == TRUE) {
	if (expl_maze[x][y] & 0x10) {
	  draw_explosion(player, x, y, expl_maze[x][y] & 0xf);
	} else {
	  draw_block(player, x, y, (int) maze[x][y]);
	}
      }
    }
  }
}


#ifdef __STDC__
void 
update_expl (int player)
#else
void 
update_expl (player)
     int player;
#endif
{
  int x,y;

  for (x=0; x<MAZE_W; x++) {
    for (y=0; y<MAZE_H; y++) {
      if ( (redraw_maze[x][y]) && (expl_maze[x][y] & 0x10) ) {
	draw_explosion_sprite(player, x, y, expl_maze[x][y] & 0xf);
      }
    }
  }
}



/* public function check_maze */
#ifdef __STDC__
int 
check_maze (int x, 
	    int y)
#else
int 
check_maze (x,y)
     int x,y;
#endif
{
  return( (maze[x][y]==BTBlock) 
	 || (maze[x][y]==BTBlockRise) 
	 || (maze[x][y]==BTExtra) 
	 || (maze[x][y]==BTExtraOpen ) 
	 || (maze[x][y]==BTVoid ) 
	 || (maze[x][y]<0) );
}



/* public function check_maze_free */
#ifdef __STDC__
int 
check_maze_free (int x,
		 int y)
#else
int 
check_maze_free (x,y)
     int x,y;
#endif
{
  return( (maze[x][y]==BTFree) || (maze[x][y]==BTShadow) );
}

/* public function check_maze_wall */
#ifdef __STDC__
int 
check_maze_wall (int x,
		 int y)
#else
int 
check_maze_wall (x,y)
     int x,y;
#endif
{
  return( (maze[x][y]==BTBlock) || (maze[x][y]==BTBlockRise) );
}

/* public function set_maze_block */
#ifdef __STDC__
void 
set_maze_block (int x,
		int y, 
		int block)
#else
void 
set_maze_block (x,y,block)
     int x,y, block;
#endif
{
  int rnd;
  
  maze[x][y] = block;

  /* set shadows */
  if ( (block == BTBlock) 
      && (shadow_mode & ShadowBlock) 
      && (x != (MAZE_W-1) )
      && (maze[x+1][y] == BTFree) ) {
    maze[x+1][y] = BTShadow;
    mark_maze(x,y,x+1,y);
  } else if ( (block == BTExtra) 
	     && (shadow_mode & ShadowExtra) 
	     && (x != (MAZE_W-1) )
	     && (maze[x+1][y] == BTFree) ) {
    maze[x+1][y] = BTShadow;
    mark_maze(x,y,x+1,y);
  /* Corrected shadow code by Chris Doherty */
  } else if ( (block == BTFree)
	&&  (x != (MAZE_W-1) ) )
  {
     if (maze[x+1][y] == BTShadow)
     {
        maze[x+1][y] = BTFree;
        mark_maze(x,y,x+1,y);
     }
     if ( ((maze[x-1][y] == BTBlock) && (shadow_mode & ShadowBlock))
           ||  ((maze[x-1][y] == BTExtra) && (shadow_mode & ShadowExtra)) )
     {
        maze[x][y] = BTShadow;
        mark_maze(x,y,x,y);
     }
     else
     {
        maze[x][y] = BTFree;
        mark_maze(x,y,x,y);
     }
  }
  else
  {
    mark_maze(x,y,x,y);
  }

  /* create new extra */
  if (maze[x][y] == BTExtra) {
    rnd = rand() % 64;
    if (rnd < cur_level->prob.bomb) {
      extra[x][y] = BTBomb;
    } else if (rnd < cur_level->prob.range) {
      extra[x][y] = BTRange;
    } else if (rnd < cur_level->prob.ill) {
      extra[x][y] = BTSick;
    } else if (rnd < cur_level->prob.invinc) {
      extra[x][y] = BTSpecial;
    } else if (rnd < cur_level->prob.evil) {
      extra[x][y] = BTEvil;
    } else {
      extra[x][y] = BTFree;
    }
  } else {
    extra[x][y] = BTFree;
  }

}


/* public function get_extra */
#ifdef __STDC__
int 
get_extra (int invincible,
	   int x,
	   int y)
#else
int 
get_extra (invincible, x, y)
     int invincible, x, y;
#endif
{
  int extra_block;

  extra_block = ( maze[x][y]<=BTExtraOpen ? 0 : maze[x][y]);
  if ( (invincible>0)  && (extra_block == BTSick)) {
    extra_block = 0;
  }

  if (extra_block != 0) {
    redraw_maze[x][y] = TRUE;
    if ( ( (shadow_mode & ShadowBlock) && (maze[x-1][y]==BTBlock) )
	|| ( (shadow_mode & ShadowExtra) 
	    && ( (maze[x-1][y]==BTExtra) || (maze[x-1][y]==BTExtraOpen) ) ) ) {
      maze[x][y] = BTShadow;
    } else {
      maze[x][y] = BTFree;
    }
  }

  return extra_block ;
}



#ifdef __STDC__
static void 
my_swap (int *a, 
	 int *b)
#else
static void 
my_swap (a, b)
     int *a, *b;
#endif
{
  int s;

  s = *a;
  *a = *b;
  *b = s;
}



/* Do we distribute special bombs? */
#ifdef __STDC__
int 
distrib_special (void)
#else
int 
distrib_special ()
#endif
{
  return (distrib_extras == DEspecial);
}



/* global function */
#ifdef __STDC__
void 
distribute_extras (int bombs, 
		   int range, 
		   int extras, 
		   int specials)
#else
void 
distribute_extras(bombs, range, extras, specials)
     int bombs, range, extras, specials;
#endif
{
  int x,y;
  Explosion *ptr;
  int ra;
  int i, where;
  int free_blocks = 0;
  int non_free = 0;
  int n_extras = 0;

    
  /* Create Extra Distribution Map */
  if (bombs + range + extras + specials != 0)  {
    /* First check for free Blocks */
    for (x=0; x < MAZE_W; x++) {
      for (y=0; y < MAZE_H; y++) {
	if ( (maze[x][y] == BTShadow) || (maze[x][y] == BTFree) ) {
	  dist_extra[x][y] = TRUE;
	  free_blocks ++;
	} else {
	  dist_extra[x][y] = FALSE;
	  non_free ++;
	}
      }
    }

    /* Go through explosions */
    for (ptr = expl_list; ptr != NULL; ptr = ptr->next) {
      if (ptr->count >=0) {
	ra = MIN(ptr->range, ptr->count + 1);
	
	for (x = ptr->x; (x <= (ptr->x + ra)) && (x <= MAZE_W); x++) {
	  if (dist_extra[x][ptr->y]) {
	    free_blocks --;
	  }
	  dist_extra[x][ptr->y] = FALSE;
	}
	for (x = ptr->x; (x >= (ptr->x - ra)) && (x >= 0); x--)	{
	  if (dist_extra[x][ptr->y]) {
	    free_blocks --;
	  }
	  dist_extra[x][ptr->y] = FALSE;
	}
	for (y = ptr->y; (y <= (ptr->y + ra)) && (y <= MAZE_H); y++) {
	  if (dist_extra[ptr->x][y]) {
	    free_blocks --;
	  }
	  dist_extra[ptr->x][y] = FALSE;
	}
	for (y = ptr->y; (y >= (ptr->y - ra)) && (y >= 0); y--)	{
	  if (dist_extra[ptr->x][y]) {
	    free_blocks --;
	  }
	  dist_extra[ptr->x][y] = FALSE;
	}
      }
    }
    
    /* fill dist_koord array */
    i = 0;
    for (x=0; x < MAZE_W; x++) {
      for (y=0; y < MAZE_H; y++) {
	if (dist_extra[x][y] == TRUE) {
	  dist_x[i] = x;
	  dist_y[i] = y;
	  i ++;
	}
      }
    }
    
    /* Special extras */
    
    /* How many extras ?*/
    switch (distrib_extras) {
    case DEnone:
      n_extras = 0;
      break;

    case DEsingle:
      n_extras = ((extras>0) ? (1) : (0));
      break;

    case DEall:
      n_extras = extras;
      break;

    case DEspecial:
      n_extras = (specials + 2)/3;
      break;
    }

    /* Distribute special extras */
    if (free_blocks > 0) {
      for(i = 0; (free_blocks > 0) && (i < n_extras); i++) {
	where = (rand()>>4) % free_blocks;
	set_maze_block(dist_x[where],dist_y[where], BTSpecial);
	
	free_blocks--;
	/* this position is used */
	if (where != free_blocks) {
	  my_swap( dist_x + where, dist_x + free_blocks );
	  my_swap( dist_y + where, dist_y + free_blocks );
	}
      }
    }
    
    if (free_blocks > 0) {
      for(i = 0; (free_blocks > 0) && (i < range); i++) {
	where = (rand()>>4) % free_blocks;
	set_maze_block(dist_x[where],dist_y[where], BTRange);
	
	free_blocks--;
	/* this position is used */
	if (where != free_blocks)  {
	  my_swap( dist_x + where, dist_x + free_blocks );
	    my_swap( dist_y + where, dist_y + free_blocks );
	}
      }
    }
    
    if (free_blocks > 0) {
      for(i = 0; (free_blocks > 0) && (i < bombs); i++) {
	where = (rand()>>4) % free_blocks;
	  set_maze_block(dist_x[where],dist_y[where], BTBomb);
	
	free_blocks--;
	/* this position is used */
	if (where != free_blocks) {
	  my_swap( dist_x + where, dist_x + free_blocks );
	  my_swap( dist_y + where, dist_y + free_blocks );
	}
      }
    }
  }
}



/* local function one_expl_at */
#ifdef __STDC__
static void 
one_expl_at (int x, 
	    int y, 
	    int ra, 
	    int ri, 
	    int outer, 
	    int inner)
#else
static void 
one_expl_at (x, y, ra, ri, outer, inner)
     int x, y, ra, ri, outer, inner;
#endif
{
  int i;

  /* right */
  for (i = 0; (i <= ra) && (maze[x+i][y] <= BTShadow) ; i ++) {
    if (i >= ri) {
      if (i != ri && i != ra) {
	expl_maze[x+i][y] |= 0x1a;
      } else if (i != ri) {
	expl_maze[x+i][y] |= 0x18;
      } else if (i == ra) {
	expl_maze[x+i][y] |= 0x10;
      } else {
	expl_maze[x+i][y] |= 0x12;
      }
    }
  }
  if ( (i <= ra) && (maze[x+i][y] == BTExtra) ) {
    maze[x+i][y] = BTExtraOpen;
    redraw_maze[x+i][y] = TRUE;
  }


  /* left */
  for (i = 0; (i <= ra) && (maze[x-i][y] <= BTShadow); i ++) {
    if (i >= ri) {
      if (i != ri && i != ra) {
	expl_maze[x-i][y] |= 0x1a;
      } else if (i != ri) {
	expl_maze[x-i][y] |= 0x12;
      } else if (i == ra) {
	expl_maze[x-i][y] |= 0x10;
      } else {
	expl_maze[x-i][y] |= 0x18;
      }
    }
  }
  if ( (i <= ra ) && (maze[x-i][y] == BTExtra) ) {
    maze[x-i][y] = BTExtraOpen;
    redraw_maze[x-i][y] = TRUE;
  }


  /* up */
  for (i = 0; (i <= ra) && (maze[x][y-i] <= BTShadow); i ++) {
    if (i >= ri) {
      if (i != ri && i != ra) {
	expl_maze[x][y-i] |= 0x15;
      } else if (i != ri) {
	expl_maze[x][y-i] |= 0x14;
      } else if (i == ra) {
	expl_maze[x][y-i] |= 0x10;
      } else {
	expl_maze[x][y-i] |= 0x11;
      }
    }
  }
  if ( (i <= ra) && ( maze[x][y-i] == BTExtra) ) {
    maze[x][y-i] = BTExtraOpen;
    redraw_maze[x][y-i] = TRUE;
  }

  /* down */
  for (i = 0; (i <= ra) && (maze[x][y+i] <= BTShadow); i ++) {
    if (i >= ri) {
      if (i != ri && i != ra) {
	expl_maze[x][y+i] |= 0x15;
      } else if (i != ri) {
	expl_maze[x][y+i] |= 0x11;
      } else if (i == ra) {
      expl_maze[x][y+i] |= 0x10;
      } else {
	expl_maze[x][y+i] |= 0x14;
      }
    }
  }
  if ( (i <= ra) && ( maze[x][y+i] == BTExtra) ) {
    maze[x][y+i] = BTExtraOpen;
    redraw_maze[x][y+i] = TRUE;
  }
}


/* local function blast_extra_block */
#ifdef __STDC__
static void
blast_extra_block (int x, 
		   int y)
#else
static void
blast_extra_block (x, y)
     int x, y;
#endif
{
  switch(maze[x][y]) {

    /* open extra block */
  case BTExtraOpen:
    if (extra[x][y] != BTEvil) {
      maze[x][y]= extra[x][y];
    } else {
      maze[x][y] = BTFree;
      new_explosion(MAX_PLAYER,x,y,3,FALSE,FALSE,evilBMT,0,GoDefault);
    }
    redraw_maze[x][y] = TRUE;
	
    /* Redraw Shadow on the right */
    if ( ( shadow_mode & ShadowExtra )
	&& ( maze[x+1][y] == BTShadow ) ) {
      maze[x+1][y] = BTFree;
      redraw_maze[x+1][y] = TRUE;
    }

    /* Redraw for empty Floor */
    if ( maze[x][y] == BTFree ) {
      if ( (shadow_mode & ShadowExtra) 
	  && ( (maze[x-1][y] == BTExtra) 
	      || (maze[x-1][y] == BTExtraOpen) ) ) {
	maze[x][y] = BTShadow;
      }
      if ( (shadow_mode & ShadowBlock) 
	    && (maze[x-1][y] == BTBlock)  ) {
	maze[x][y] = BTShadow;
      }
    }
    break;

    /* blast away extra and correct shadow */
  case BTSick:
  case BTBomb:
  case BTRange:
  case BTSpecial:
    if ( ( (shadow_mode & ShadowBlock) && (maze[x-1][y]==BTBlock) )
	|| ( (shadow_mode & ShadowExtra) 
	    && ( (maze[x-1][y]==BTExtra) || (maze[x-1][y]==BTExtraOpen) ) ) ) {
      maze[x][y] = BTShadow;
    } else {
      maze[x][y]= BTFree;
    }
    redraw_maze[x][y] = TRUE;
    break;
  }
}

/* local function del_explosion */
#ifdef __STDC__
static void 
del_explosion (Explosion *ptr)
#else
static void 
del_explosion (ptr)
     Explosion *ptr;
#endif
{
  Explosion *hilf;
  int i,x,y,r;

#ifdef DEBUG
  if (ptr == NULL) {
    fprintf (stderr, "Warning null pointer given to del_explosion\n");
    return;
  }
#endif

  add_player_num_bomb(ptr->player);

  x = ptr->x;
  y = ptr->y;
  r = ptr->range;
  bomb_maze[x][y]=NULL;

  num_expl --;

  /* rechts */
  for (i=0; i<=r && ((maze[x+i][y] <= BTShadow)); i++);
  if (i <= r) {
    blast_extra_block(x+i,y);
  }

  /* links */
  for (i=0; i<=r && ((maze[x-i][y] <= BTShadow)); i++);
  if (i <= r) {
    blast_extra_block(x-i,y);
  }

  /* unten */
  for (i=0; i<=r && ((maze[x][y+i] <= BTShadow)); i++);
  if (i <= r) {
    blast_extra_block(x,y+i);
  }
  /* oben */
  for (i=0; i<=r && ((maze[x][y-i] <= BTShadow)); i++);
  if (i <= r) {
    blast_extra_block(x,y-i);
  }

  /* aus liste aushaengen */
  if (ptr == expl_list) {
    expl_list = ptr->next;
    if (expl_list == NULL) {
      expl_end = NULL;
    }
    free(ptr);
  } else {
    for (hilf = expl_list; (hilf->next != ptr) && (hilf->next != NULL);
	 hilf = hilf->next);
    
    if (hilf->next != NULL) {
      if (expl_end == ptr) {
	expl_end = hilf;
      }
      hilf->next = hilf->next->next;
      free(ptr);
    }
  }
}



/* Player, wall, and bomb click functions (Garth Denley) */

#ifdef __STDC__
static void 
bombdirshift (Explosion *ptr,
	      int dir,
	      int WhenUp, 
	      int WhenLeft, 
	      int WhenDown, 
	      int WhenRight)
#else
static void 
bombdirshift (ptr, dir, WhenUp, WhenLeft, WhenDown, WhenRight)
     Explosion *ptr;
     int dir;
     int WhenUp, WhenLeft, WhenDown, WhenRight;
#endif
{
  int newdir = GoStop;

  switch (dir) {
  case GoUp: 
    newdir = WhenUp; 
    break;
  case GoLeft:
    newdir = WhenLeft;
    break;
  case GoDown:
    newdir = WhenDown;
    break;
  case GoRight:
    newdir = WhenRight;
    break;
  }

  ptr->dir = newdir;
  switch (newdir) {
  case GoUp:
  case GoDown:
    ptr->dx = 0;
    break;
  case GoLeft:
    case GoRight:
    ptr->dy = 0;
    break;
  }
} 



#ifdef __STDC__
static void 
bombrandomdir (Explosion *ptr)
#else
static void 
bombrandomdir (ptr)
     Explosion *ptr;
#endif
{
  int newdir = GoStop;

  switch (rand()%4)
  {
    case 0: newdir = GoUp;    break;
    case 1: newdir = GoLeft;  break;
    case 2: newdir = GoDown;  break;
    case 3: newdir = GoRight; break;
  }
  ptr->dir = newdir;
  switch (newdir)
  {
    case GoUp:
    case GoDown:
      ptr->dx = 0;
      break;
    case GoLeft:
    case GoRight:
      ptr->dy = 0;
      break;
  }
}


#ifdef __STDC__
static void 
do_player_click (Explosion *bomb, 
		 int player, 
		 BMPlayer *ps)
#else
static void 
do_player_click (bomb,player,ps)
     Explosion *bomb;
     int player;
     BMPlayer *ps;
#endif
{
  int dir;

  dir = bomb->dir;

  switch (PC_mode)  {
    
  case PC_StunStop:
    if (0 == ps[player].stunned) {
      ps[player].stunned = STUN_TIME;
    }
    bomb->dx = 0;
    bomb->dy = 0;
    bomb->dir = GoStop;
    break;

    /* Oliver's stun-through code */
  case PC_StunThruInit:
    if (0 == ps[player].stunned) {
      ps[player].stunned = STUN_TIME;
    }
    switch(initial_bomb_dir) {
    case GoStop:
      bomb->dx = 0;
      bomb->dy = 0;
      break;
    case GoRight:
      if (check_maze_free(bomb->x+1, bomb->y)) {
	bomb->dx = BOMB_VX;
      }
      bomb->dy = 0;
      break;
    case GoLeft:
      if (check_maze_free(bomb->x-1, bomb->y)) {
	bomb->dx = -BOMB_VX;
      }
      bomb->dy = 0;
      break;
    case GoDown:
      bomb->dx = 0;
      if (check_maze_free(bomb->x, bomb->y+1)) {
	bomb->dy = BOMB_VY;
      }
      break;
    case GoUp:
      bomb->dx = 0;
      if (check_maze_free(bomb->x, bomb->y-1)) {
	bomb->dy = -BOMB_VY;
      }
      break;
    }
    bomb->dir = initial_bomb_dir;
    break;

  case PC_StunThru:
    if (0 == ps[player].stunned) {
      ps[player].stunned = STUN_TIME;  
    }
    break;
    
  case PC_Contact:
    bomb->dx = 0;
    bomb->dy = 0;
    bomb->dir = GoStop;
    bomb->count = 0;
    break;
    
  case PC_Rebound:
    if (0 == ps[player].stunned) {
      ps[player].stunned = STUN_TIME;
    }
    bombdirshift(bomb,dir,GoDown,GoRight,GoUp,GoLeft);
    break;
  }
}



#ifdef __STDC__
static void 
do_wall_click (Explosion *bomb)
#else
static void 
do_wall_click (bomb)
     Explosion *bomb;
#endif
{
  int dir;

  dir = bomb-> dir;
  bomb->dir = GoStop;

  switch (WC_mode) {
  case WC_Rebound:
    bombdirshift(bomb,dir,GoDown,GoRight,GoUp,GoLeft);
    break;
  case WC_Contact:
    bomb->count=0;
    break;
  case WC_Clockwise:
    bombdirshift(bomb,dir,GoRight,GoUp,GoLeft,GoDown);
    break;
  case WC_Anticlockwise:
    bombdirshift(bomb,dir,GoLeft,GoDown,GoRight,GoUp);
    break;
  case WC_Random:
    bombrandomdir(bomb);
    break;
  }
}



#ifdef __STDC__
static void 
do_bomb_click (Explosion *bomb)
#else
static void 
do_bomb_click(bomb)
     Explosion *bomb;
#endif
{
  int dir;
  int dx, dy;

  dir = bomb-> dir;
  bomb->dir = GoStop;

  dx = bomb->dx;
  dy = bomb->dy;
  bomb->dx = 0;
  bomb->dy = 0;

  switch (BC_mode) {
  case BC_Snooker:
    switch (dir) {
    case GoUp:
      move_bomb(bomb->x,bomb->y-1,dir);
      break;
      case GoLeft:
      move_bomb(bomb->x-1,bomb->y,dir);
	break;
    case GoDown:
      move_bomb(bomb->x,bomb->y+1,dir);
      break;
    case GoRight:
	move_bomb(bomb->x+1,bomb->y,dir);
	break;
    }
    break;

  case BC_Rebound:
    bombdirshift(bomb,dir,GoDown,GoRight,GoUp,GoLeft);
    break;
  case BC_Contact:
    bomb->count = 0;
    break;
  case BC_Clockwise:
    bombdirshift(bomb,dir,GoRight,GoUp,GoLeft,GoDown);
    break;
  case BC_Anticlockwise:
    bombdirshift(bomb,dir,GoLeft,GoDown,GoRight,GoUp);
    break;
  case WC_Random:
    bombrandomdir(bomb);
    break;
  }
}

/* public function do_bombs() */
#ifdef __STDC__
void 
do_bombs (void)
#else
void 
do_bombs ()
#endif
{
  Explosion *ptr;
  int draw_flag;

  for (ptr = expl_list; ptr != NULL; ptr = ptr->next) {
    if (ptr->count < 0) {
      draw_flag = TRUE;

      switch(ptr->dir) {

      case GoStop:
	draw_flag = FALSE;
	break;

      case GoUp:
	if  ( (ptr->dy == 0) 
	     && (maze[ptr->x][ptr->y-1] != BTFree)
	     && (maze[ptr->x][ptr->y-1] != BTShadow) )  {
	  do_wall_click(ptr);
	} else if ( (ptr->dy <= 0) && check_bomb(ptr->x,ptr->y-1) ) {
	  do_bomb_click(ptr);
	} else {
	  ptr->dy -= BOMB_VY;
	  if (ptr->dy <= -BLOCK_HEIGHT/2) {
	    bomb_maze[ptr->x][ptr->y] = NULL;
	    ptr->dy += BLOCK_HEIGHT;
	    ptr->y -= 1;
	    bomb_maze[ptr->x][ptr->y] = ptr;
	  }
	}
	break;

      case GoDown:
	if ( (ptr->dy == 0) 
	    && (maze[ptr->x][ptr->y+1] != BTFree)
	    && (maze[ptr->x][ptr->y+1] != BTShadow) ) {
	  do_wall_click(ptr);
	} else if ( (ptr->dy >= 0) && check_bomb(ptr->x,ptr->y+1)) {
	  do_bomb_click(ptr);
	} else {
	  ptr->dy += BOMB_VY;
	  if (ptr->dy >= BLOCK_HEIGHT/2) {
	    bomb_maze[ptr->x][ptr->y] = NULL;
	    ptr->dy -= BLOCK_HEIGHT;
	    ptr->y += 1;
	    bomb_maze[ptr->x][ptr->y] = ptr;
	  }
	}
	break;

      case GoRight:
	if ( (ptr->dx == 0)  
	    && (maze[ptr->x+1][ptr->y] != BTFree)
	    && (maze[ptr->x+1][ptr->y] != BTShadow) ) {
	  do_wall_click(ptr);
	} else if ( (ptr->dx >= 0) && check_bomb(ptr->x+1,ptr->y)) {
	  do_bomb_click(ptr);
	} else {
	  ptr->dx += BOMB_VX;
	  if (ptr->dx >= BLOCK_WIDTH/2) {
	    bomb_maze[ptr->x][ptr->y] = NULL;
	    ptr->dx -= BLOCK_WIDTH;
	    ptr->x += 1;
	    bomb_maze[ptr->x][ptr->y] = ptr;
	  }
	}
	break;

      case GoLeft:
	if ( (ptr->dx == 0)  
	    && (maze[ptr->x-1][ptr->y] != BTFree)
	    && (maze[ptr->x-1][ptr->y] != BTShadow) ) {
	  do_wall_click(ptr);
	} else if ( (ptr->dx <= 0) && check_bomb(ptr->x-1,ptr->y)) {
	  do_bomb_click(ptr);
	} else {
	  ptr->dx -= BOMB_VX;
	  if (ptr->dx <= -BLOCK_WIDTH/2) {
	    bomb_maze[ptr->x][ptr->y] = NULL;
	    ptr->dx += BLOCK_WIDTH;
	    ptr->x -= 1;
	    bomb_maze[ptr->x][ptr->y] = ptr;
	  }
	}
	break;
      }

      /* draw a bomb */
      if ( (ptr->dir == GoUp) || (ptr->dir == GoDown) ) {
	if (ptr->dy <= 0) {
	  redraw_maze[ptr->x][ptr->y-1] = TRUE;
	}
	if (ptr->dy >= 0) {
	  redraw_maze[ptr->x][ptr->y+1] = TRUE;
	}
      }
	  
      if ( (ptr->dir == GoLeft) || (ptr->dir == GoRight) ) {
	if (ptr->dx < 0) {
	  redraw_maze[ptr->x-1][ptr->y] = TRUE;
	} else {
	  redraw_maze[ptr->x+1][ptr->y] = TRUE;
	}
      }

      if (expl_maze[ptr->x][ptr->y]!=0) {
	ptr->count = 0;
      }

      if (ptr->blink + ptr->count == 0) {
	redraw_maze[ptr->x][ptr->y] = TRUE;
	add_bomb_to_sprite_list(ptr->x*BLOCK_WIDTH + ptr->dx, 
				ptr->y*BLOCK_HEIGHT + ptr->dy,
				((ptr->range == 1) ? BB_MINI : BB_NORMAL),
				FALSE);
	ptr->blink = ptr->blink >> 1;
      } else {
	if ( draw_flag || ( (ptr->blink<<1) + ptr->count == 1) ) 
	  redraw_maze[ptr->x][ptr->y] = TRUE;
	add_bomb_to_sprite_list(ptr->x*BLOCK_WIDTH + ptr->dx, 
				ptr->y*BLOCK_HEIGHT + ptr->dy,
				((ptr->range == 1) ? BB_MINI : BB_NORMAL),
				TRUE);
      }

      /* Bomb malfunction, random or illness */
      if ( (ptr->count == -3) 
	  && (ptr->malfunction || ((rand()>>2)%BOMB_ERROR_PROB == 0) ) )
	{
	  ptr->malfunction = 0;
	  ptr->count = -BOMB_TIME *(2+(rand()>>2)%BOMB_DELAY);
	  ptr->blink = (BOMB_TIME >>1);
	}
    }
  }
}



/* public function ignite_players_bombs */
#ifdef __STDC__
int 
ignite_players_bombs (int player)
#else
int 
ignite_players_bombs (player)
     int player;
#endif
{
  Explosion *ptr;
  int number_of_bombs = 0;

  for (ptr = expl_list; ptr != NULL; ptr = ptr->next) {
    if (ptr->count < 0) {
      /* draw a bomb */
      if ( ptr->player == player ) {
	ptr->count = 0;
	number_of_bombs ++;
      }
    }
  }
  return(number_of_bombs);
}



/* public function ignite_bombs */
#ifdef __STDC_
void 
ignite_bombs (void)
#else
void 
ignite_bombs ()
#endif
{
  Explosion *ptr;

  for (ptr = expl_list; ptr != NULL; ptr = ptr->next) {
    if (ptr->count < 0) {
      /* draw a bomb */
      if ( (expl_maze[ptr->x][ptr->y] & 0x0f) !=0) {
	ptr->count = 0;
      }
    }
  }
}



/* public functions do_explosion */
#ifdef __STDC__
int 
do_explosions (void)
#else
int 
do_explosions ()
#endif
{
  Explosion *ptr, *save;
  register int hilf;
  int explode_flag = FALSE;

  ptr = expl_list;;
  while (ptr != NULL) {
    /* check if bomb is exploding */
    if (ptr->count >= 0) {
      /* check if bomb has burned out */
      if ( ptr->count  >= (2*ptr->range + 2) ) {
	save = ptr->next;
	del_explosion(ptr);
	ptr = save;
	continue;
      } else {
	/* get exploding time */
	if ( (hilf = ptr->count) == 0) {
	  /* this is needed for triggering the bell */
	  explode_flag = TRUE;
	}
	/* now do the explosion */
	one_expl_at(ptr->x, ptr->y,
		    MIN(ptr->range,hilf), MAX(0,( (hilf) - ptr->range)),
		    hilf, hilf - ptr->range);
      }
    }
    do_special_bomb_types(ptr);
    ptr->count ++;
    ptr = ptr->next;
  }

  return explode_flag;
}

/* Special bomb code (Garth Denley) */

/* Used to spread an explosion out */
#ifdef __STDC__
static void 
sprex (int lx,
       int ly,
       int range,
       int type,
       int type_extr)
#else
static void 
sprex (lx, ly, range, type, type_extr)
     int lx;
     int ly;
     int range;
     int type;
     int type_extr;
#endif
{
  if ( (lx < MAZE_W) && (lx > -1) && (ly < MAZE_H) && (ly > -1)
       && (maze[lx][ly] != BTBlock) && (maze[lx][ly] != BTBlockRise)
       && (maze[lx][ly] != BTExtra) && (maze[lx][ly] != BTExtraOpen)) {
    new_explosion(MAX_PLAYER, lx, ly, range, FALSE, FALSE,
                  type, type_extr, GoStop);
  }
}

#ifdef __STDC__
static void
test_and_kill (int x,
               int y)
#else
static void
test_and_kill (x,y)
     int x;
     int y;
#endif
{
  BMPlayer *ps;
  int player;

  for (player = 0; player < num_player; player ++) {
    ps = player_stat + player;
    if ( ps->lives > 0 ) {
      if (    (ps->x < (x+1)*BLOCK_WIDTH)
           && (ps->x > (x-1)*BLOCK_WIDTH)
           && (ps->y < (y)  *BLOCK_HEIGHT)
           && (ps->y > (y-2)*BLOCK_HEIGHT) ) {
        ps->lives = 1;
        ps->dying = DEAD_TIME;
      }
    }
    delete_bomb_at(x,y);
  }
}

#ifdef __STDC__
static void
move_block_from_to (int sx,
                    int sy,
                    int dx,
                    int dy)
#else
static void
move_block_from_to (sx, sy, dx, dy)
     int sx;
     int sy;
     int dx;
     int dy;
#endif
{
  if (    (sx > 0) && (sx < MAZE_W-1)
       && (sy > 0) && (sy < MAZE_H-1)
       && (dx > 0) && (dx < MAZE_W-1)
       && (dy > 0) && (dy < MAZE_H-1)
       && check_maze_wall(sx, sy)
       && check_maze_free(dx, dy) ) {

    set_maze_block(sx, sy, BTFree);
    set_maze_block(dx, dy, BTBlock);
    test_and_kill (dx, dy);
  }
}

#ifdef __STDC__
static void 
do_special_bomb_types (Explosion *ptr)
#else
static void
do_special_bomb_types (ptr)
     Explosion *ptr;
#endif
{
  int i,j, k;
  int nasty;
  int x,y;

  switch (ptr->type) {

  case BMTnapalm:
    if (ptr->count == -1) {
      ptr->type = BMTnormal;
      for (i= -2; i<=2; i++) {
	sprex (ptr->x+i, ptr->y, ptr->range/(ABS(i)+1), BMTblastnow, 0);
	sprex (ptr->x, ptr->y+i, ptr->range/(ABS(i)+1), BMTblastnow, 0);
      }
    }
    break;
    
  case BMTfirecracker:
  case BMTfirecracker2:
    if (ptr->count >= 1) {
      if ((ptr->type == BMTfirecracker) && ((rand() % 10) == 0)) {
	ptr->type_extr = -5;
      }
      nasty = ptr->type_extr;
      for (i= -1; i<=1; i++) {
	if (nasty<2 || (!(rand() % (1 + nasty)))) {
	  sprex (ptr->x+i, ptr->y, 1, BMTfirecracker2, nasty + 1);
	}
	if (nasty<2 || (!(rand() % (1 + nasty)))) {
	  sprex (ptr->x, ptr->y+i, 1, BMTfirecracker2, nasty + 1);
	}
      }
      ptr->type = BMTnormal;
    }
    break;
    
  case BMTconstruction:
    if (ptr->count == 1) {
      x = ptr->x;
      y = ptr->y;
      del_explosion(ptr);
      if (check_b_near(x,y) == 1) {
	set_maze_block(x,y,BTExtra);
	extra[x][y] = BTFree;
      }
    }
    break;
  case BMTfungus:
    x = ptr->x;
    y = ptr->y;
    if (ptr->count == (-cur_bomb_time)*3/5) {
      for (i=-1; i<=1; i++) {
        sprex(x+i,y,1,BMTfungus,0);
        sprex(x,y+i,1,BMTfungus,0);
      }
    }
    break;
  case BMTthreebombs:
    if (ptr->count == -cur_bomb_time) {
      sprex(ptr->x-2,ptr->y,ptr->range,defaultBMT,0);
      sprex(ptr->x+2,ptr->y,ptr->range,defaultBMT,0);
    }
    break;
  case BMTgrenade:
    if (ptr->range > 0) {
      if (ptr->count == -1) {
        x = ptr->x;
        y = ptr->y;
        if (ptr->range == 1) {
          sprex(ptr->x-1, ptr->y-1, 0, BMTblastnow, 0);
          sprex(ptr->x+1, ptr->y-1, 0, BMTblastnow, 0);
          sprex(ptr->x-1, ptr->y+1, 0, BMTblastnow, 0);
          sprex(ptr->x+1, ptr->y+1, 0, BMTblastnow, 0);
        } else {
          for (i=-((ptr->range)-1); i<=((ptr->range)-1); i++)
            for (j=-((ptr->range)-1); j<=((ptr->range)-1); j++)
              sprex (ptr->x+i, ptr->y+j, 1, BMTblastnow, 0);
        }
      }
    }
    break;
  case BMTtrianglebombs:
    if (ptr->count == -cur_bomb_time + 2) {
      i = (rand () % 2) * 4 - 2;
      j = (rand () % 2) * 4 - 2;
      sprex(ptr->x+i, ptr->y,   ptr->range, BMTnormal, 0);
      sprex(ptr->x,   ptr->y+j, ptr->range, BMTnormal, 0);
      ptr->type = BMTnormal;
    }
    break;
  case BMTdestruction:
    x = ptr->x;
    y = ptr->y;
    if (ptr->count == 1) {
      del_explosion(ptr);
      for (i=-1; i<=1; i++) {
        if ( (x+i < (MAZE_W-1)) && check_maze(x+i,y) && (x+i > 0) )
          set_maze_block(x+i,y,BTFree);
        if ( (y+i < (MAZE_H-1)) && check_maze(x,y+i) && (y+i > 0) )
          set_maze_block(x,y+i,BTFree);
      }
    }
    break;
  case BMTrenovation:
    x = ptr->x;
    y = ptr->y;
    if (ptr->count == 1) {
      move_block_from_to ( x-1, y,   x-2, y   );
      move_block_from_to ( x+1, y,   x+2, y   );
      move_block_from_to ( x,   y-1, x,   y+1 );
      move_block_from_to ( x-1, y-1, x,   y+2 );
    }
    break;
  case BMTpyro:
  case BMTpyro2:
    if (ptr->count == 1) {
      for (k=0;k<5;k++) {
        x = ptr->x + (rand() % 3) - 1;
        y = ptr->y + (rand() % 3) - 1;  
        if ((!bomb_maze[x][y]) && (maze[x][y] == BTFree)) {
          sprex(x, y, 1, BMTpyro2, 0);
          break;
        }
      }
    }
    break;
  }
}

/* public function new_explosion */

#ifdef __STDC__
void 
new_explosion (int player, 
	       int x, 
	       int y, 
	       int range, 
	       int remote_controlled, 
	       int malfunction, 
	       int type, 
	       int type_extr, 
	       int initialdir)
#else
void 
new_explosion (player, x, y, range, remote_controlled, malfunction, type, 
	       type_extr, initialdir)
     int player, x, y, range, remote_controlled, malfunction,
     type,type_extr,initialdir;
#endif
{
  Explosion *new;

  if (NULL == bomb_maze[x][y]) {
    redraw_maze[x][y] = TRUE;
    
    num_expl ++;
    
    new = (Explosion *)calloc(1,sizeof(Explosion));
    bomb_maze[x][y]=new;
    
    if (expl_list == NULL) {
      expl_list = new;
    } else {
      expl_end->next = new;
    
    }
    
    new->player = player;
    new->x = x;
    new->y = y;
    new->dx = 0;
    new->dy = 0;
    new->malfunction = malfunction;
    
    if (initialdir == GoDefault) {
      new->dir = initial_bomb_dir;
    } else {
      new->dir = initialdir;
    }

    if (new->type == BMTrandom) {
      switch (rand () % 5) {
        case 0: new->type = BMTnapalm;      break;
        case 1: new->type = BMTfirecracker; break;
        case 2: new->type = BMTgrenade;     break;
        case 3: new->type = BMTfungus;      break;
        case 4: new->type = BMTpyro;        break;
      }
    }

    new->type = type;
    new->type_extr = type_extr;
    
    if (type != BMTclose) {
      switch(new->dir) {
      case GoDown:
	if(check_maze_free(new->x,new->y+1)) {
	  new->dy= BOMB_VY;
	}
	break;

      case GoUp:
	if(check_maze_free(new->x,new->y-1)) {
	  new->dy= -BOMB_VY;
	}
	break;
      case GoLeft:
	if(check_maze_free(new->x-1,new->y)) {
	  new->dx= -BOMB_VX;
	}
	break;

      case GoRight:
	if(check_maze_free(new->x+1,new->y)) {
	  new->dx= BOMB_VX;
	}
	break;
      }
    }
    if ( (type == BMTfirecracker) || (type == BMTfungus)
         || (type == BMTpyro) ) {
      new->range = 1;
    } else if (type == BMTconstruction) {
      new->range = 0;
    } else if (type == BMTgrenade) {
      new->range = new->range / 2;
    } else {
     new->range = range;
    }
    if ((type == BMTblastnow) || (type == BMTfirecracker2)
         || (type == BMTpyro2) ) {
      new->count = 0;
    } else if (remote_controlled) {
      new->count = -GAME_TIME;
    } else {
      new->count = -cur_bomb_time;
    }
    new->blink = BOMB_TIME >>1;
    expl_end = new;

  } else {
    add_player_num_bomb(player);
  }
}



/* public function stun_players */
#ifdef __STDC__
void 
stun_players (BMPlayer *ps, 
	      int num_player)
#else
void 
stun_players (ps, num_player)
     BMPlayer *ps;
     int num_player;
#endif
{
  Explosion *ptr;
  int player;
  int click_flags;

  for (ptr = expl_list; ptr != NULL; ptr = ptr->next) {
    if (ptr->dir != GoStop) {
      click_flags = 0;
      for (player = 0; player < num_player; player ++) {
	if ( (ps[player].invincible == 0)
	    && (ABS(ptr->x*BLOCK_WIDTH + ptr->dx - ps[player].x ) 
		< BOMB_STUN_X )
	    && (ABS(ptr->y*BLOCK_HEIGHT + ptr->dy 
		    - ps[player].y - BLOCK_HEIGHT ) < BOMB_STUN_Y ) ) {
	  if (ptr->dx == 0) {
	    if (ptr->dy < 0) {
	      redraw_maze[ptr->x][ptr->y-1] = TRUE;
	    }
	    if (ptr->dy > 0) {
	      redraw_maze[ptr->x][ptr->y+1] = TRUE;
	    }
	  }
	  if (ptr->dy == 0) {
	    if (ptr->dx < 0) {
	      redraw_maze[ptr->x-1][ptr->y-1] = TRUE;
	    }
	    if (ptr->dx > 0) {
	      redraw_maze[ptr->x+1][ptr->y-1] = TRUE;
	    }
	  }
	  click_flags |= (1<<player);
	}
      }
      /* do player click after all players are checked */
      if (click_flags) {
	for (player=0; player<num_player; player++) {
	  if (click_flags & (1<<player)) {
	    do_player_click(ptr,player,ps);
	  }
	}
      }
    }
  }
}


/* public function check_explosion */
#ifdef __STDC__
int 
check_explosion (int x,
		 int y)
#else
int 
check_explosion (x,y)
     int x,y;
#endif
{
  return ((int)(expl_maze[x][y]!=0));
}



/* public function check_bomb */
#ifdef __STDC__
int 
check_bomb (int x, 
	    int y)
#else
int 
check_bomb (x,y)
     int x,y;
#endif
{
#if 0
  return ((int)(bomb_maze[x][y] != NULL));
#else
  return (int)( (bomb_maze[x][y] != NULL) && (bomb_maze[x][y]->count < 0));
#endif
}



/* public function number_of_explosions */
#ifdef __STDC__
int 
number_of_explosions (void)
#else
int 
number_of_explosions ()
#endif
{
  return(num_expl);
}


#ifdef __STDC__
void 
delete_ring_bombs (int range)
#else
void 
delete_ring_bombs (range)
     int range;
#endif
{
  int x,y;

  for (x=range; x<MAZE_W-range; x++) {
    if (bomb_maze[x][range]!=NULL) {
      del_explosion(bomb_maze[x][range]);
    }
    if (bomb_maze[x][MAZE_H-range-1]!=NULL) {
      del_explosion(bomb_maze[x][MAZE_H-range-1]);
    }
  }
  for (y=range; y<MAZE_H-range; y++) {
    if (bomb_maze[range][y]!=NULL) {
      del_explosion(bomb_maze[range][y]); 
    }
    if (bomb_maze[MAZE_W-range-1][y]!=NULL) {
      del_explosion(bomb_maze[MAZE_W-range-1][y]);
    }
  }
}



/*
 * public function delete_row_bombs
 */
#ifdef __STDC__
void
delete_row_bombs (int pos)
#else
void
delete_row_bombs (pos)
     int pos;
#endif
{
  int x;

  for (x=0; x<MAZE_W; x++) {
    if (bomb_maze[x][pos]!=NULL) {
      del_explosion(bomb_maze[x][pos]);
    }
  }
}



/*
 * public function delete_row_bombs
 */
#ifdef __STDC__
void
delete_column_bombs (int pos)
#else
void
delete_column_bombs (pos)
     int pos;
#endif
{
  int y;

  for (y=0; y<MAZE_H; y++) {
    if (bomb_maze[pos][y]!=NULL) {
      del_explosion(bomb_maze[pos][y]);
    }
  }
}


/* public function */
#ifdef __STDC__
void 
delete_bomb_at (int x, int y)
#else
void 
delete_bomb_at (x, y)
     int x,y;
#endif
{
  if (bomb_maze[x][y]) {
    del_explosion(bomb_maze[x][y]);
  }
}



/* public function */
#ifdef __STDC__
void 
move_bomb (int x,
	   int y,
	   int dir)
#else
void 
move_bomb (x,y,dir)
     int x,y,dir;
#endif
{
  Explosion *ptr;
  
  if(NULL != (ptr = bomb_maze[x][y]) ) {
    if (ptr->dir == GoStop) {
      ptr->dir = dir;
      switch(dir) {
      case GoUp:
      case GoDown:
	ptr->dx = 0;
	break;
	
      case GoLeft:
      case GoRight:
	ptr->dy = 0;
	break;
      }
    }
  }
}


/* public copy_expl_block */
#ifdef __STDC__
void 
copy_expl_block (int x,
		 int y,
		 int block[CHARH][CHARW])
#else
void 
copy_expl_block (x, y, block)
     int x,y;
     int block[CHARH][CHARW];
#endif
{
  int xp,yp;

  for (xp = x; xp < x+CHARW; xp ++) {
    for (yp = y; yp < y+CHARH; yp ++) {
      expl_maze[xp][yp] = block[yp-y][xp-x];
      redraw_maze[xp][yp] = TRUE;
    }
  }
}



/* contributed functions */

/* 
 * public function haunt_kick 
 * prob is an integer >= 6
 * the lower the number, the more vicious the haunted bombs are
 */

#ifdef __STDC__
void 
haunt_kick (int prob)
#else
void 
haunt_kick (prob)
     int prob;
#endif
{
  int di;
  int bnum;
  Explosion *ptr;
  int t;

  ptr = expl_list;
  bnum = (rand() % prob);

  if (bnum < 6)  {
    if (bnum > 0) {
      for (t=0; t<bnum ; t++ ) {
	if (ptr != NULL) {
	  ptr = ptr->next;
	}
      }
    }
    
    if (ptr != NULL) {
      if (ptr->dir == GoStop) {
	if (check_b_near(ptr->x,ptr->y) == 1) {
	  di = rand() % 4;

	  switch (di) {
	  case 0:
	    ptr->dir = GoUp;
	    ptr->dx = 0;
	    break;
	  case 1:
	    ptr->dir = GoDown;
	    ptr->dx = 0;
	    break;
	  case 2:
	    ptr->dir = GoLeft;
	    ptr->dy = 0;
	    break;
	  case 3:
	    ptr->dir = GoRight;
	    ptr->dy = 0;
	    break;
	  }
	}
      }
    }
  }
}

/* public function do_air (Garth Denley)*/

/* Shoots bombs away if within 2 square radius */
/* Direction based on angle from bomb */

#ifdef __STDC__
void 
do_air (BMPlayer *ps, int player)
#else
void 
do_air (ps, player)
     BMPlayer *ps;
     int player;
#endif
{
  Explosion *ptr;
  int x,y,ex,ey;

  for (ptr = expl_list; ptr != NULL; ptr = ptr->next) {
    if ((ptr->dir == GoStop))  {
      x = (ptr->x) * BLOCK_WIDTH;
      y = (ptr->y-1) * BLOCK_HEIGHT;
      ex = x - ps->x;
      ey = y - ps->y;
      if ((ABS(ex)<BLOCK_WIDTH*2)&&(ABS(ey)<BLOCK_HEIGHT*2)
	  && ((ex!=0)||(ey!=0))) {
        if (ABS(ex)*BLOCK_HEIGHT >= ABS(ey)*BLOCK_WIDTH)  {
          if (ex<0) {
	    ptr->dir=GoLeft; 
	  } else { 
	    ptr->dir=GoRight;
	  }
        } else {
          if (ey<0) {
	    ptr->dir=GoUp; 
	  } else {
	    ptr->dir=GoDown;
	  }
        }
      }
    }
  }
}

/*
 * end of file maze.c
 */
