/*
 * pam_private.h
 *
 * $Id: pam_private.h,v 1.5 1996/07/07 23:58:01 morgan Exp $
 *
 * This is the Linux-PAM Library Private Header. It contains things
 * internal to the Linux-PAM library. Things not needed by either an
 * application or module.
 *
 * Please see end of file for copyright.
 *
 * Creator: Marc Ewing.
 * 
 * $Log: pam_private.h,v $
 * Revision 1.5  1996/07/07 23:58:01  morgan
 * added debug function as a statically defined one here. Also added
 * pam_fail_delay structures/prototypes
 *
 * Revision 1.4  1996/06/02 08:01:48  morgan
 * added StrTok prototype
 *
 * Revision 1.3  1996/05/26 03:58:09  morgan
 * Michael Johnson's changes for static libpam/module support
 *
 *
 */

#ifndef _PAM_PRIVATE_H
#define _PAM_PRIVATE_H

/* this is not used at the moment --- AGM */
#define LIBPAM_VERSION 0.50

#include <security/pam_appl.h>
#include <security/pam_modules.h>

/* the Linux-PAM configuration file */

#define PAM_CONFIG    "/etc/pam.conf"

/* components of the pam_handle structure */

struct handler {
    int (*func)(pam_handle_t *pamh, int flags, int argc, char **argv);
    int control_flag;
    int argc;
    char **argv;
    struct handler *next;
};

struct loaded_module {
    char *name;
    int type; /* PAM_STATIC_MOD or PAM_DYNAMIC_MOD */
    void *dl_handle;
};

#define PAM_DYNAMIC_MOD 0
#define PAM_STATIC_MOD  1

struct handlers {
    struct handler *authenticate;
    struct handler *setcred;
    struct handler *acct_mgmt;
    struct handler *open_session;
    struct handler *close_session;
    struct handler *chauthtok;
};

struct service {
    struct loaded_module *module; /* Only used for dynamic loading */
    int modules_allocated;
    int modules_used;
    int handlers_loaded;

    struct handlers conf;        /* the configured handlers -- AGM */
    struct handlers other;       /* the default handlers    -- AGM */
};

#ifdef PAM_FAIL_DELAY_ON

/* warning this code is likely to change. It was added for .50 and is
   a rather crude attempt to provide some time delay support */

#include <sys/time.h>

typedef enum { PAM_FALSE, PAM_TRUE } _pam_boolean;

struct _pam_fail_delay {
    _pam_boolean set;
    struct timeval begin;
    struct timeval delay;
};

#endif /* PAM_FAIL_DELAY_ON */

struct _pam_handle {
    char *service_name;
    char *user;
    char *tuser;                 /* for use by pam_get_user() -- AGM */
    char *prompt;                /* for use by pam_get_user() -- AGM */
    char *tty;
    char *rhost;
    char *ruser;
    char *authtok;
    char *oldauthtok;
    struct pam_conv *pam_conversation;
    struct pam_data *data;
    struct service handlers;
#ifdef PAM_FAIL_DELAY_ON
    struct _pam_fail_delay fail_delay;   /* helper function for easy delays */
#endif /* PAM_FAIL_DELAY_ON */
};

/* Values for select arg to _pam_dispatch() */
#define PAM_AUTHENTICATE  0
#define PAM_SETCRED       1
#define PAM_ACCOUNT       2
#define PAM_OPEN_SESSION  3
#define PAM_CLOSE_SESSION 4
#define PAM_CHAUTHTOK     5

/* Values for control_flag in struct handler */
#define PAM_REQUIRED   0
#define PAM_SUFFICIENT 1
#define PAM_OPTIONAL   2

/* internally defined functions --- these should not be directly
 * called by applications or modules */
int _pam_dispatch(pam_handle_t *pamh, int flags, int choice);

/* Free various allocated structures and dlclose() the libs */
int _pam_free_handlers(pam_handle_t *pamh);

/* Parse config file, allocate handler structures, dlopen() */
int _pam_init_handlers(pam_handle_t *pamh);

/* Set all hander stuff to 0/NULL - called once from pam_start() */
void _pam_start_handlers(pam_handle_t *pamh);

#ifdef LINUX

/* these functions deal with failure delays as required by the
   authentication modules and application. Their interface is likely
   to remain the same although their function is hopefully going to
   improve */

/* reset the timer to no-delay */
void _pam_reset_timer(pam_handle_t *pamh);

/* this sets the clock ticking */
void _pam_start_timer(pam_handle_t *pamh);

/* this waits for the clock to stop ticking if status != PAM_SUCCESS */
void _pam_await_timer(pam_handle_t *pamh, int status);

#endif

#ifdef PAM_STATIC

/* The next two in ../modules/_pam_static/pam_static.c */

/* Return pointer to data structure used to define a static module */
struct pam_module * _pam_open_static_handler(char *path);

/* Return pointer to function requested from static module */

typedef void (*voidfunc(void))(void);
voidfunc *_pam_get_static_sym(struct pam_module *mod, const char *symname);

#endif

/* For now we just use a stack and linear search for module data. */
/* If it becomes apparent that there is a lot of data, it should  */
/* changed to either a sorted list or a hash table.               */

struct pam_data {
    char *name;
    void *data;
    int (*cleanup)(pam_handle_t *pamh, void *data, int error_status);
    struct pam_data *next;
};

void _pam_free_data(pam_handle_t *pamh, int status);

int strCMP(const char *s, const char *t);
char *StrTok(char *from, const char *format, char **next);

char *strdup(const char *s);

int _pam_mkargv(char *s, char ***argv, int *argc);

void _pam_log_error(const char *format, ...);

#ifdef DEBUG
#include <stdarg.h>
#include <stdio.h>
static void _pam_debug(const char *format, ...)
{
    va_list args;

    va_start(args, format);
    vfprintf(stderr, format, args);
    fprintf(stderr, "\n");
}
#endif

#define DROP(X) \
if (X) { \
    free(X); \
    X=NULL; \
}

/* XXX - Take care with this. It could confuse the logic of a trailing
   else */

#define IF_NO_PAMH(X,pamh,ERR) \
if ((pamh) == NULL) { \
    _pam_log_error(X ": NULL pam handle passed"); \
    return ERR; \
}

/* Definition for the default username prompt used by pam_get_user() */

#define PAM_DEFAULT_PROMPT "Please enter username: "

/* Good policy to strike out passwords with some characters not just
   free the memory */

#define _pam_overwrite(x) \
{ \
     register char *xx; \
     if ((xx=x)) \
	  while (*xx) \
	       *xx++ = '\0'; \
}

/*
 * Copyright (C) 1995 by Red Hat Software, Marc Ewing
 * All rights reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, and the entire permission notice in its entirety,
 *    including the disclaimer of warranties.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote
 *    products derived from this software without specific prior
 *    written permission.
 * 
 * ALTERNATIVELY, this product may be distributed under the terms of
 * the GNU Public License, in which case the provisions of the GPL are
 * required INSTEAD OF the above restrictions.  (This clause is
 * necessary due to a potential bad interaction between the GPL and
 * the restrictions contained in a BSD-style copyright.)
 * 
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#endif /* _PAM_PRIVATE_H_ */
