/*
 * Dolphin.c
 *
 * This file contains the Dolphin specific hardware stuff.
 *
 * This file is based on the dutch documentation mailed to me by
 * Pascal <pern@ramoth.xs4all.nl>. Since I don't have a Dolphin
 * modem, I cannot test nor debug this driver. But I'm quite
 * optimistic that it will work.
 *
 * Marc
 *
 */

#include "../include/voice.h"

char *libvoice_Dolphin_c = "$Id: Dolphin.c,v 1.7 1996/08/06 19:50:22 marc Exp $";

/*
 * Here we save the current mode of operation of the voice modem when
 * switching to voice mode, so that we can restore it afterwards.
 */

static char mode_save[VOICE_BUF_LEN] = "";

/*
 * Internal status variables for aborting some voice modem actions.
 */

static int stop_dialing;
static int stop_playing;
static int stop_recording;
static int stop_waiting;

/*
 * The Dolphin samples with 9600 samples per second with a maximum of 2 bit
 * per sample. We want to buffer voice data for 0.1 second, so we need a
 * buffer less or equal to 9600 * 0.25 * 0.1 = 240 bytes.
 */

#define DOLPHIN_BUFFER_SIZE 240
static int dolphin_buffer_size = DOLPHIN_BUFFER_SIZE;

/*
 * This function handles the <DLE> shielded codes.
 */

#define ST_NO_INPUT (0x00)
#define ST_GOT_DLE  (0x01)

static int modem_answer_state = ST_NO_INPUT;

static void handle_modem_answer(char new_byte)
     {

     switch (modem_answer_state)
          {
          case ST_NO_INPUT:

               switch (new_byte)
                    {
                    case DLE:
                         modem_answer_state = ST_GOT_DLE;
                         break;
                    case XON:
                         lprintf(L_WARN, "%s: Received XON",
                          program_name, new_byte);
                         break;
                    case XOFF:
                         lprintf(L_WARN, "%s: Received XOFF",
                          program_name, new_byte);
                         break;
                    case NL:
                    case CR:
                         break;
                    default:
                         lprintf(L_ERROR, "%s: Illegal modem answer 0x%2x",
                          program_name, new_byte);
                    };

               return;
          case ST_GOT_DLE:

               switch (new_byte)
                    {
                    case '0':
                    case '1':
                    case '2':
                    case '3':
                    case '4':
                    case '5':
                    case '6':
                    case '7':
                    case '8':
                    case '9':
                    case '*':
                    case '#':
                         voice_handle_event(RECEIVED_DTMF,
                          (event_data) new_byte);
                         break;
                    case 'b':
                         voice_handle_event(BUSY_TONE, (event_data) 0);
                         break;
                    case 'c':
                         voice_handle_event(FAX_CALLING_TONE, (event_data) 0);
                         break;
                    case 'q':
                         voice_handle_event(SILENCE_DETECTED, (event_data) 0);
                         break;
                    case 's':
                         voice_handle_event(NO_VOICE_ENERGY, (event_data) 0);
                         break;
                    default:
                         lprintf(L_ERROR,
                          "%s: Unknown <DLE> shielded code 0x%x",
                          program_name, new_byte);
                    };

               modem_answer_state = ST_NO_INPUT;
               return;
          };

     };

/*
 * This is the main handle event routine for the Dolphin.
 */

int Dolphin_handle_event(int event, event_data data)
     {
     char buffer[VOICE_BUF_LEN];

     switch (event)
          {
          case VOICE_ANSWER_PHONE:
               return(voice_command("ATA", "VCON"));
          case VOICE_BEEP:
               sprintf(buffer, "AT+VTS=[%d,0,%d]", data.beep.frequency,
                data.beep.length);

               if (voice_command(buffer, "OK") != VMA_USER_1)
                    return(ERROR);

               return(OK);
          case VOICE_DIAL:
               {
               int result = ERROR;

               voice_modem_state = DIALING;
               stop_dialing = FALSE;
               sprintf(buffer, "ATD%s", (char*) data.p);

               if (voice_command(buffer, "") != OK)
                    return(ERROR);

               while (!stop_dialing)
                    {
                    voice_read(buffer);
                    result = voice_analyze(buffer, NULL);

                    switch (result)
                         {
                         case VMA_BUSY:
                         case VMA_FAIL:
                         case VMA_ERROR:
                         case VMA_NO_ANSWER:
                         case VMA_NO_CARRIER:
                         case VMA_NO_DIAL_TONE:
                              stop_dialing = TRUE;
                              result = ERROR;
                              break;
                         case VMA_VCON:
                              stop_dialing = TRUE;
                              result = OK;
                              break;
                         };

                    };

               voice_modem_state = IDLE;
               return(result);
               };
          case VOICE_INIT:
               lprintf(L_MESG, "initializing Dolphin voice modem");
               voice_modem_state = INITIALIZING;

               /*
                * AT+VNH=1 - Disable automatic hangup.
                */

               if (voice_command("AT+VNH=1", "OK") != VMA_USER_1)
                    lprintf(L_WARN, "disabling automatic hangup didn't work");

               /*
                * AT+VSD=x,y - Set silence threshold and duration.
                */

               sprintf(buffer, "AT+VSD=%d,%d",
                cvd.rec_silence_threshold.d.i * 31 / 100,
                 cvd.rec_silence_len.d.i);

               if (voice_command(buffer, "OK") != VMA_USER_1)
                    lprintf(L_WARN,
                     "setting recording preferences didn't work");

               voice_modem_state = IDLE;
               return(OK);
          case VOICE_MESSAGE_LIGHT_OFF:
               return(voice_command("ATS0=0", "OK"));
          case VOICE_MESSAGE_LIGHT_ON:
               return(voice_command("ATS0=255", "OK"));
          case VOICE_MODE_OFF:
               sprintf(buffer, "AT+FCLASS=%s", mode_save);
               voice_command(buffer, "OK");
               return(OK);
          case VOICE_MODE_ON:
               voice_command("AT+FCLASS?", "");
               voice_read(mode_save);
               voice_flush(1);
               voice_command("AT+FCLASS=8", "OK");
               return(OK);
          case VOICE_PLAY_FILE:
               {
               TIO tio_save;
               TIO tio;
               char input_buffer[DOLPHIN_BUFFER_SIZE];
               char output_buffer[2 * DOLPHIN_BUFFER_SIZE];
               int i;
               int bytes_in;
               int bytes_out;
               int bytes_written;

               stop_playing = FALSE;
               modem_answer_state = ST_NO_INPUT;
               voice_modem_state = PLAYING;
               tio_get(voice_fd, &tio);
               tio_save = tio;
               tio_set_flow_control(voice_fd, &tio, FLOW_XON_OUT);
               tio_set(voice_fd, &tio);
               voice_command("AT+VTX", "CONNECT");

               while (!stop_playing)
                    {

                    if ((bytes_in = read(data.i, input_buffer,
                     dolphin_buffer_size)) <= 0)
                         break;

                    bytes_out = 0;

                    for(i = 0; i < bytes_in; i++)
                         {
                         output_buffer[bytes_out] = input_buffer[i];

                         if (output_buffer[bytes_out++] == DLE)
                              output_buffer[bytes_out++] = DLE;

                         };

                    lprintf(L_JUNK, "%s: <DATA %d bytes>", program_name,
                     bytes_out);

                    errno = 0;
                    bytes_written = 0;

                    while (((bytes_written += write(voice_fd,
                     &output_buffer[bytes_written], bytes_out -
                     bytes_written)) != bytes_out) && (errno == 0))
                         ;

                    if (bytes_written != bytes_out)
                         lprintf(L_ERROR,
                          "%s: could only write %d bytes of %d bytes (errno 0x%x)",
                          program_name, bytes_written, bytes_out, errno);

                    while (check_for_input(voice_fd))
                         {
                         char modem_byte;

                         if (read(voice_fd, &modem_byte, 1) != 1)
                              lprintf(L_ERROR,
                               "%s: could not read byte from voice modem",
                               program_name);
                         else
                              handle_modem_answer(modem_byte);

                         };

                    };

               sprintf(output_buffer, "%c%c", DLE, ETX);
               lprintf(L_JUNK, "%s: <DLE><ETX>", program_name);
               write(voice_fd, output_buffer, strlen(output_buffer));
               tio_set(voice_fd, &tio_save);
               voice_command("", "VCON");
               voice_command("AT", "OK");
               voice_modem_state = IDLE;

               if (stop_playing)
                    return(INTERRUPTED);

               return(OK);
               };
          case VOICE_RECORD_FILE:
               {
               TIO tio_save;
               TIO tio;
               char input_buffer[DOLPHIN_BUFFER_SIZE];
               char output_buffer[DOLPHIN_BUFFER_SIZE];
               int i = 0;
               int bytes_in = 0;
               int bytes_out;
               int got_DLE_ETX = FALSE;
               int was_DLE = FALSE;

               stop_recording = FALSE;
               modem_answer_state = ST_NO_INPUT;
               voice_modem_state = RECORDING;
               tio_get(voice_fd, &tio);
               tio_save = tio;
               tio_set_flow_control(voice_fd, &tio, FLOW_XON_IN);
               tio.c_cc[VMIN] = (dolphin_buffer_size > 0xff) ? 0xff :
                dolphin_buffer_size;
               tio.c_cc[VTIME] = 1;
               tio_set(voice_fd, &tio);
               voice_command("AT+VRX", "CONNECT");

               while (!got_DLE_ETX)
                    {

                    if ((bytes_in = read(voice_fd, input_buffer,
                     dolphin_buffer_size)) <= 0)
                         {
                         lprintf(L_ERROR,
                          "%s: could not read byte from voice modem",
                          program_name);
                         return(FAIL);
                         };

                    bytes_out = 0;

                    for (i = 0; (i < bytes_in) && !got_DLE_ETX; i++)
                         {

                         if (was_DLE)
                              {
                              was_DLE = FALSE;

                              switch (input_buffer[i])
                                   {
                                   case DLE:
                                        output_buffer[bytes_out++] = DLE;
                                        break;
                                   case ETX:
                                        got_DLE_ETX = TRUE;
                                        lprintf(L_JUNK, "%s: <DATA %d bytes>",
                                         voice_modem_name, bytes_out);
                                        lprintf(L_JUNK, "%s: <DLE><ETX>",
                                         voice_modem_name);
                                        break;
                                   default:
                                        handle_modem_answer(DLE);
                                        handle_modem_answer(input_buffer[i]);
                                   };

                              }
                         else
                              {

                              if (input_buffer[i] == DLE)
                                   was_DLE = TRUE;
                              else
                                   output_buffer[bytes_out++] =
                                    input_buffer[i];

                              };

                         };

                    write(data.i, output_buffer, bytes_out);

                    if (!got_DLE_ETX)
                         lprintf(L_JUNK, "%s: <DATA %d bytes>",
                          voice_modem_name, bytes_out);

                    };

               tio_set(voice_fd, &tio_save);

               if (voice_analyze(&input_buffer[i], "\r\nVCON") == VMA_USER_1)
                    lprintf(L_JUNK, "%s: VCON", voice_modem_name);
               else
                    {
                    int j;

                    lprintf(L_JUNK, "%s: data left in buffer: ",
                     voice_modem_name);

                    for (j = i; j < bytes_in ; j++)
                         lputc(L_JUNK, input_buffer[j]);

                    voice_command("AT", "OK");
                    };

               voice_command("AT", "OK");
               voice_modem_state = IDLE;
               return(OK);
               };
          case VOICE_SET_COMPRESSION:

               switch (data.i)
                    {
                    case 0:
                    case 2:
                         dolphin_buffer_size = DOLPHIN_BUFFER_SIZE;
                         voice_command("AT+VSM=2", "OK");
                         return(OK);
                    };

               lprintf(L_WARN,
                "Dolphin handle event: Illeagal voice compression method (%d)",
                data.i);
               return(FAIL);
          case VOICE_SET_DEVICE:

               switch (data.i)
                    {
                    case NO_DEVICE:
                         voice_write("AT+VLS=0");

                         if (voice_command("", "AT+VLS=0|OK") == VMA_USER_1)
                              voice_command("", "OK");

                         return(OK);
                    case DIALUP_LINE:
                         voice_command("AT+VLS=2", "VCON");
                         return(OK);
                    case INTERNAL_SPEAKER:
                         voice_command("AT+VLS=16", "VCON");
                         return(OK);
                    };

               lprintf(L_WARN,
                "Dolphin handle event: Unknown output device (%d)",
                data.i);
               return(FAIL);
          case VOICE_STOP_DIALING:
               stop_dialing = TRUE;
               return(OK);
          case VOICE_STOP_PLAYING:
               stop_playing = TRUE;
               return(OK);
          case VOICE_STOP_RECORDING:
               stop_recording = TRUE;

               if (write(voice_fd, "Q", 1) != 1)
                    {
                    lprintf(L_ERROR, "could not write to voice modem device");
                    return(FAIL);
                    };

               return(OK);
          case VOICE_STOP_WAITING:
               stop_waiting = TRUE;
               return(OK);
          case VOICE_SWITCH_TO_DATA_FAX:
               sprintf(buffer, "AT+FCLASS=%s", (char *) data.p);
               return(voice_command(buffer, "OK"));
          case VOICE_WAIT:
               {
               stop_waiting = FALSE;
               modem_answer_state = ST_NO_INPUT;
               voice_modem_state = WAITING;
               alarm(data.i);

               while (!stop_waiting)
                    {

                    while (check_for_input(voice_fd))
                         {
                         char modem_byte;

                         if (read(voice_fd, &modem_byte, 1) != 1)
                              lprintf(L_ERROR,
                               "%s: could not read byte from voice modem",
                               program_name);
                         else
                              handle_modem_answer(modem_byte);

                         };

                    delay(100);
                    };

               voice_modem_state = IDLE;
               alarm(0);
               return(OK);
               };
          };

     lprintf(L_WARN, "Dolphin handle event: Unknown event %04x", event);
     return(FAIL);
     };
