/*
 * rockwell.c
 *
 * Conversion snd <--> Rockwell "adpcm" format
 *
 * Torsten Duwe <duwe@informatik.uni-erlangen.de>
 *
 * very remotely derived from Rockwell's d.asm.
 * Converted to C and simplified by Torsten Duwe 1995
 * Floating-Point routines are C++-proof to allow replacement
 * by Fixed-Point class (not working yet).
 *
 */

#include "../include/voice.h"

#ifndef __GNU_C__
# define inline
#endif

const char *rockwell_c = "$Id: rockwell.c,v 1.4 1996/07/25 19:22:07 marc Exp $";

/* #################### start of C++-proof part #################### */
/* this can be made a seperate file for compilation as C++ */

/* this looks like -*- C -*- , but may use C++ classes (fixpt) */
#undef USE_FIXEDPOINT
#ifdef USE_FIXEDPOINT
#include "fixpt.h"
#else
#include <math.h>
typedef double fixpt;

static inline int 
clip _P1((d), double * d)
{
  if (*d > 1.0){ *d = 1.0; return(1); }
  else if (*d < -1.0){ *d = -1.0; return(1); }
  return(0);
}
#define SAMPLE_TO_FIXPT(s) ((s)/32768.0)
#define FIXPT_TO_SAMPLE(f) ((int)((f)*32768.0))
#endif

#ifdef __cplusplus
extern "C" {             /* make this linkable with C code */
char encode1(short X);
void reset_model();
short decode1(char Q);
}
#endif

typedef unsigned char uch;
#define A1 0
#define B1 2
static fixpt _pzTable[8];
static fixpt Qdata[8];

static fixpt lastX;
extern char bps_2;       /* bps - 2 */
static
fixpt Xpect = (fixpt)0;

static
fixpt Zeta[3][16] = {
  {-1.510,  -0.4528,  0.4528,  1.510, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
  {-2.152, -1.344, -0.7560, -0.2451, 0.2451, 0.7560, 1.344, 2.152,
   0, 0, 0, 0, 0, 0, 0, 0},
  {-2.733,  -2.069,  -1.618,  -1.256,  -0.9424,  -0.6568,  -0.3881,  -0.1284,
   0.1284,  0.3881,  0.6568,  0.9424,  1.256,  1.618,  2.069,  2.733}
};

static
fixpt mul[3][16] = {
  {1.6,  0.8,  0.8,  1.6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
  {1.75,  1.25,  0.9,  0.9,  0.9,  0.9,  1.25,  1.75, 0, 0, 0, 0, 0, 0, 0, 0},
  {2.4,  2.0,  1.6,  1.2,  0.9,  0.9,  0.9,  0.9,
   0.9,  0.9,  0.9,  0.9,  1.2,  1.6,  2.0,  2.4}
};

static
fixpt QDelayMax[3] = { 0x2ED5/32768.0, 0x3B7A/32768.0, 0x54C4/32768.0 };

/* #define QDLMIN ((fixpt)(0x1F/32768.0)) */
#define QDLMIN (9.46044921875e-4)
static fixpt Nu = QDLMIN;
#define DEMPCF ((fixpt)(0.4))
#define PDEMPCF     ((fixpt)(0.4))

static
fixpt cd[3][7] = {
  {0.9816, 0, 0, 0, 0, 0, 0},
  {0.5006,  1.050,  1.748, 0, 0, 0, 0},
  {0.2582,  0.5224,  0.7996,  1.099,  1.437,  1.844,  2.401}
};

/* reset the expectation model after silence expansion
 */
static
void reset_model()
{
  int i;

  for (i=0; i<8; i++)
    _pzTable[i] = Qdata[i] = 0;
  Xpect = 0;             /* = _pzTable * Qdata */
  Nu = QDLMIN;
  lastX = 0;
}

/* update the expectation according to the codeword found/read
 */
static void update_model _P2((NewQdata, Q), fixpt NewQdata, uch Q)
{
  int i;
  for (i=0; i<8; i++)

    if (Qdata[i]*NewQdata > 0)
      _pzTable[i] = fabs((i<B1)? (fixpt).012: (fixpt).006)
       + _pzTable[i]*(fixpt).98;
    else
      _pzTable[i] = -fabs((i<B1)? (fixpt).012: (fixpt).006)
       + _pzTable[i]*(fixpt).98;

  for (i=7; i; i--)
    Qdata[i] = Qdata[i-1];

  Qdata[A1] = NewQdata + Xpect; /* clip(Qdata+A1); */
  Qdata[B1] = NewQdata;

  Nu *= mul[(int) bps_2][Q];
  if (Nu < QDLMIN) Nu = QDLMIN;
  else if (Nu > QDelayMax[(int) bps_2]) Nu = QDelayMax[(int) bps_2];

  Xpect = 0;
  for (i=0; i<8; i++)
    Xpect += _pzTable[i] * Qdata[i];
  clip(&Xpect);
}

/* encode a single 16-Bit sample into a {2,3,4}-Bit word
 */
static
char encode1 _P1((X), short X)
{
  uch Q, idx;
  fixpt a, b;
  fixpt NewQdata;

  b = fabs(a = SAMPLE_TO_FIXPT(X) - Xpect - PDEMPCF*lastX);
  lastX = SAMPLE_TO_FIXPT(X);
  for(idx = 0; idx<(1<<(bps_2+1))-1; idx++)
    if (Nu*cd[(int) bps_2][idx] > b) break;
  Q = (1<<(bps_2+1)) + ((a<(fixpt)0) ? -(idx+1) : idx);
  NewQdata = Nu*Zeta[(int) bps_2][Q];
  update_model(NewQdata, Q);
  return(Q);
}

/* decode a single {2,3,4}-Bit word into a 16-Bit sample
 */
static
short decode1 _P1((Q), uch Q)
{
  fixpt X, NewQdata;

  NewQdata = Nu*Zeta[(int) bps_2][Q];
  X = NewQdata + Xpect + DEMPCF*lastX; clip(&X);
  lastX = X;
  update_model(NewQdata, Q);
  return((short)FIXPT_TO_SAMPLE(X));
}

/* #################### end of C++-proof part #################### */

static
unsigned short silence_word[3] = {0x13ec, 0x23de, 0xc11c};

static int rom_version = 150;
char bps_2 = 1;               /* Bits Per Sample -2, used for indexing */

/* get a codeword from the rmd-file,
 * byte-order depending on rom_revision (?)
 */
static inline int _getcw _P1((in), FILE * in)
{
  int c1, c2;
  if ((c1=getc(in)) == EOF) return (c1);
  if ((c2=getc(in)) == EOF) c2 = 0;
  return ((rom_version < 150) ? c1 << 8 | c2 : c2 << 8 | c1);
}

/* get a sample from the snd-file, preserving EOF
 */
static inline int _getw _P1((in), FILE * in)
{
  int c1, c2;
  if ((c1=getc(in)) == EOF) return (c1);
  if ((c2=getc(in)) == EOF) c2 = 0;
  return (c1 << 8 | c2);
}

/* put a codeword to the rmd-file,
 * byte-order depending on rom_revision (?)
 */
static inline void _putcw _P2((s, out), unsigned short s, FILE * out)
{
  if (rom_version < 150)
    {                    /* Can anyone confirm this ? */
      putc(s>>8, out);
      putc(s&255, out);
    }
  else
    {
      putc(s&255, out);
      putc(s>>8, out);
    }
}

/* put a sample to the .snd-file
 */
static inline void _putw _P2((s, out), unsigned short s, FILE * out)
{
  putc(s>>8,  out);
  putc(s&255, out);
}

static
short silence[256];      /* XXX - hope this goes to the bss an is 0 */

/* expand a silence codeword
 * note that silence compression isn't implemented here
 */
static
void put_silence _P2((num_samples, out), int num_samples, FILE *out)
{
  int nw;
  num_samples *= 2;      /* make it bytes */

  while (num_samples)
    {
      if ((nw=fwrite(silence, 1,
              (num_samples > 256) ? 256 : num_samples, out)) == -1)
     {
       perror("write silence");
       exit(1);
     }
      else
     num_samples -= nw;
    }
}

extern int write_snd_hdr _PROTO((FILE *, int, int));
extern int write_rmd_hdr _PROTO((char *, int, int, FILE *));

/* decode an entire file - rmd-header has already been read, values
 * fed as parameters. SND-Header will be written here.
 */
void
rockwtosnd _P4((nbits, rom_release, in, out), int nbits, int rom_release,
        FILE *in, FILE* out)
{
  int w;            /* codeword */

  /* "accumulator" and number of bits valid in it */
  int a=0, valbits=0;

  bps_2 = nbits - 2;
  rom_version = rom_release;

  if (write_snd_hdr(out, 7200, 0) != OK)
    {
      perror("write snd header");
      exit(1);
    }

  while ((w = _getcw(in)) != EOF)
    {
      if (w==silence_word[(int) bps_2])
     {
       put_silence(_getw(in), out);
       reset_model();
       valbits = 0;
       continue;
     }
      a |= w << valbits;
      valbits +=16;
      while (valbits > nbits)
     {
       _putw(decode1((uch)(a&((1<<nbits)-1))), out);
       a >>= nbits;
       valbits -= nbits;
     }
    }
}

/* encode an entire file - .snd-header has already been read, values
 * fed as parameters. RMD-Header will be written here.
 */
void
sndtorockw _P4((nbits, rom_release, in, out), int nbits, int rom_release,
        FILE *in, FILE* out)
{
  int w;            /* codeword */

  /* "accumulator" and number of bits valid in it */
  int a=0, valbits=0;

  bps_2 = nbits - 2;
  rom_version = rom_release;

  if (write_rmd_hdr("Rockwell", nbits, rom_release, out) != OK)
    {
      perror("write rmd header");
      exit(1);
    }

  while ((w = _getw(in)) != EOF)
    {
      a |= encode1(w)<<valbits;
      if ((valbits += nbits) >= 16)
     {
       _putcw(a&0xffff, out);
       a >>= 16;
       valbits -= 16;
     }
    }
  if (valbits)
    _putcw(a, out);
}
