/***************************************
  $Header: /home/amb/wwwoffle/RCS/parse.c 2.21 1998/03/30 17:55:33 amb Exp $

  WWWOFFLE - World Wide Web Offline Explorer - Version 2.1b.
  Functions to parse the HTTP requests.
  ******************/ /******************
  Written by Andrew M. Bishop

  This file Copyright 1996,97,98 Andrew M. Bishop
  It may be distributed under the GNU Public License, version 2, or
  any higher version.  See section COPYING of the GNU Public license
  for conditions under which this file may be redistributed.
  ***************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <time.h>
#include <unistd.h>

#include "wwwoffle.h"
#include "misc.h"
#include "proto.h"
#include "errors.h"
#include "config.h"


/*++++++++++++++++++++++++++++++++++++++
  Parse the request to the server.

  char *ParseRequest Returns the URL or NULL if it failed.

  int fd The file descriptor to read the request from.

  char **request_head Return the header of the request.

  char **request_body Return the body of the request.
  ++++++++++++++++++++++++++++++++++++++*/

char *ParseRequest(int fd,char **request_head,char **request_body)
{
 char *url=NULL,*line=NULL;
 int length=-1;

 *request_head=NULL;
 *request_body=NULL;

 while((line=read_line_or_timeout(fd,line)))
   {
    if(!strncmp("Content-length:",line,15) || !strncmp("Content-Length:",line,15))
       length=atoi(&line[15]);

    if(!*request_head)          /* first line */
      {
       *request_head=(char*)malloc(strlen(line)+1);
       strcpy(*request_head,line);

       url=(char*)malloc(strlen(line));
       if(sscanf(line,"%*s %s",url)!=1)
         {free(url);return(NULL);}
      }
    else
      {
       *request_head=(char*)realloc((void*)*request_head,strlen(line)+strlen(*request_head)+1);
       strcat(*request_head,line);
      }

    if(*line=='\r' || *line=='\n')
       break;
   }

 if(!*request_head)
    return(NULL);

 if(line)
    free(line);

 if(!strncasecmp("POST",*request_head,4))
   {
    if(length==-1)
      {free(url);return(NULL);}

    if(length)
      {
       int m,l=length;

       *request_body=(char*)malloc(length+1);

       do
         {
          m=read_data_or_timeout(fd,&(*request_body)[length-l],l);
         }
       while(m>0 && (l-=m));

       if(l)
         {free(url);return(NULL);}

       (*request_body)[length]=0;
      }

    url=(char*)realloc((void*)url,strlen(url)+32);

    if(strchr(url,'?'))
      {
       char *from=url+strlen(url),*to=from+1;
       while(*from!='?')
          *to--=*from--;
       *to='!';
      }
    else
       strcat(url,"?");

    sprintf(url+strlen(url),"!POST:%s",MakeHash(*request_body));
   }

 return(url);
}


/*++++++++++++++++++++++++++++++++++++++
  Modify the request to ask for changes since the spooled file.

  int RequestChanges Returns 1 if the file needs changes made, 0 if not, or -1 in case of an error.

  int fd The file descriptor of the spooled file.

  char **request_head The head of the HTTP request to modify.
  ++++++++++++++++++++++++++++++++++++++*/

int RequestChanges(int fd,char **request_head)
{
 struct stat buf;
 char *reply;
 int status=0;
 
 reply=read_line(fd,NULL);

 if(reply)
   {
    sscanf(reply,"%*s %d",&status);
    free(reply);
   }

 if(status==0)
   {
    PrintMessage(Debug,"Empty or no status");
    return(-1);
   }
 else if(status>=200 && status<400 && !fstat(fd,&buf))
   {
    if((time(NULL)-buf.st_mtime)<RequestChanged)
      {
       PrintMessage(Debug,"Too new");
       return(0);
      }
    else
      {
       char *if_mod=(char*)malloc(64);
       char *copy=(char*)malloc(strlen(*request_head)+64);
       char *eol=strchr(*request_head,'\n');

       sprintf(if_mod,"If-Modified-Since: %s",RFC822Date(buf.st_mtime,1));
       *eol=0; eol++;

       strcpy(copy,*request_head);
       strcat(copy,"\n");
       strcat(copy,if_mod);
       strcat(copy,"\r\n");
       strcat(copy,eol);

       free(*request_head);
       free(if_mod);

       *request_head=copy;
      }
   }

 return(1);
}


/*++++++++++++++++++++++++++++++++++++++
  Return the location that the URL has been moved to.

  char *MovedLocation Returns the new URL.

  URL *Url The original URL.

  char *reply_head The head of the original HTTP reply.
  ++++++++++++++++++++++++++++++++++++++*/

char *MovedLocation(URL *Url,char *reply_head)
{
 char *location,*eol,oldeol;
 char *new;

 location=strstr(reply_head,"\nLocation:");
 if(!location)
    return(NULL);

 location+=11;
 eol=strchr(location,'\n');
 if(eol[-1]=='\r')
    eol--;
 oldeol=*eol;
 *eol=0;

 new=(char*)malloc(strlen(location)+strlen(Url->name));

 if(!strchr(location,'/'))
   {
    char *newpath=(char*)malloc(strlen(Url->path)+1),*p;
    strcpy(newpath,Url->path);
    p=newpath+strlen(newpath);
    while(p>newpath && *--p!='/')
       *p=0;
    sprintf(new,"%s://%s/%s%s",Url->proto,Url->host,newpath,location);
    free(newpath);
   }
 else if(*location=='/')
    sprintf(new,"%s://%s%s",Url->proto,Url->host,location);
 else
    strcpy(new,location);

 *eol=oldeol;

 return(new);
}


/*++++++++++++++++++++++++++++++++++++++
  Create a new request for a page.

  char *RequestURL Ask for a page.

  char *url The URL to get.

  char *referer The Refering URL or NULL if none.
  ++++++++++++++++++++++++++++++++++++++*/

char *RequestURL(char *url,char *referer)
{
 char *new=(char*)malloc(strlen(url)+(referer?strlen(referer):0)+64);

 sprintf(new,"GET %s HTTP/1.0\r\n",url);
 if(referer)
    sprintf(&new[strlen(new)],"Referer: %s\r\n",referer);
 strcat(new,"Accept: */*\r\n"
            "\r\n");

 return(new);
}


/*++++++++++++++++++++++++++++++++++++++
  Modify the request taking into account censoring of header and modified URL.

  char *ModifyRequest Return the new request.

  URL *Url The actual URL.

  char *request_head The original head of the HTTP request possibly with a different URL.
  ++++++++++++++++++++++++++++++++++++++*/

char *ModifyRequest(URL *Url,char *request_head)
{
 char *new;
 char *hostheader=(char*)malloc(strlen(Url->host)+16),*closeheader;
 char *bol,*to,http[16];

 /* Make up the new headers. */

 sprintf(hostheader,"Host: %s\r\n",Url->host);
 closeheader="Connection: close\r\n";

 new=(char*)malloc(strlen(request_head)+strlen(closeheader)+strlen(hostheader)+strlen(Url->name));

 /* Parse the old header and create a new one. */

 sscanf(request_head,"%s %*s %s",new,http);
 strcat(new," ");
 strcat(new,Url->name);

 /* Remove the false arguments from POSTed URLs. */

 if(!strncasecmp(new,"POST",4))
   {
    char *pling=strstr(new,"?!");
    char *pling2=strchr(++pling+1,'!');

    if(pling2)
       for(;pling<pling2;pling++)
          *pling=*(pling+1);

    *(pling-1)=0;
   }

 strcat(new," ");
 strcat(new,http);
 strcat(new,"\r\n");

 strcat(new,hostheader);

 /* Check for HTTP 1.1 and add a Connection header */

 if(!strncmp(http,"HTTP/1.1",8))
    strcat(new,closeheader);

 /* Censor the header */

 to=new+strlen(new);
 bol=strchr(request_head,'\n')+1;

 while(*bol)
   {
    if(!strncmp("Host:",bol,5) || !strncmp("Connection:",bol,11) || !strncmp("Proxy-Connection:",bol,17))
       bol=strchr(bol,'\n');
    else if(IsCensoredHeader(bol))
      {
       char *eol=strchr(bol,'\n');

       if(*(eol-1)=='\r')
          *(eol-1)=0;
       else
          *eol=0;

       PrintMessage(Debug,"Censored '%s'",bol);

       bol=eol;
      }
    else if(!strncmp("Referer:",bol,8))
      {
       char *eol=strchr(bol,'\n');
       char *pling=strstr(bol,"?!");

       if(*(eol-1)=='\r')
          *(eol-1)=0;
       else
          *eol=0;

       if(pling)
         {
          char *pling2=strchr(++pling+1,'!');

          if(pling2)
             for(;pling<pling2;pling++)
                *pling=*(pling+1);

          *(pling-1)=0;
         }

       while(*bol)
          *to++=*bol++;
       *to++='\r';
       *to++='\n';

       bol=eol;
      }
    else
      {
       while(*bol && *bol!='\n')
          *to++=*bol++;
       *to++=*bol;
      }

    bol++;
   }

 *to=0;

 /* tidy up and exit. */

 free(hostheader);
 free(request_head);

 return(new);
}


/*++++++++++++++++++++++++++++++++++++++
  Change the request to one that contains an authorisation string if required.

  char *MakeRequestAuthorised Returns a new request with the authorisation if required or else the old request.

  char *proxy The name of the proxy.

  char *request_head The old HTTP request head.
  ++++++++++++++++++++++++++++++++++++++*/

char *MakeRequestAuthorised(char *proxy,char *request_head)
{
 char *new=request_head;
 char *userpass=WhatProxyAuth(proxy);

 if(userpass)
   {
    char *userpassencoded=Base64Encode(userpass,strlen(userpass));
    char *auth=(char*)malloc(strlen(userpassencoded)+32);
    char *bol,*eol;

    sprintf(auth,"Proxy-Authorization: Basic %s\r\n",userpassencoded);

    new=(char*)malloc(strlen(request_head)+strlen(auth)+1);

    if((bol=strstr(request_head,"\nProxy-Authorization:")))
       eol=strchr(++bol,'\n')+1;
    else
       bol=eol=strchr(request_head,'\n')+1;

    strncpy(new,request_head,bol-request_head);
    new[bol-request_head]=0;
    strcat(new,auth);
    strcat(new,eol);

    free(auth);
   }

 return(new);
}


/*++++++++++++++++++++++++++++++++++++++
  Change the request from one to a proxy to a normal one.

  char *MakeRequestNonProxy Return a new request that is suitable for a non-proxy server.

  char *request_head The buffer containing the head of the HTTP request.
  ++++++++++++++++++++++++++++++++++++++*/

char *MakeRequestNonProxy(char *request_head)
{
 char *new=(char*)malloc(strlen(request_head)),*r=request_head,*n=new;

 /* The URL is already in canonical form because of the ModifyRequest() function. */

 while(*r!=' ')                 /* 'GET ' */
    *n++=*r++;
 *n++=*r++;

 while(*r!=':')                 /* 'http://' */
    r++;
 r+=3;

 while(*r!='/')                 /* 'www.host.domain/' */
    r++;

 strcpy(n,r);

 return(new);
}


/*++++++++++++++++++++++++++++++++++++++
  Parse the reply from the server.

  int ParseReply Return the numeric status of the reply.

  URL *Url The URL that we are reading.

  char *reply_head Return the head of the HTTP reply.
  ++++++++++++++++++++++++++++++++++++++*/

int ParseReply(URL *Url,char **reply_head)
{
 char *line=NULL;
 int status=0;

 *reply_head=NULL;

 while((line=(Url->Protocol->readhead)(line)))
   {
    if(!*reply_head)
      {
       *reply_head=(char*)malloc(strlen(line)+1);
       strcpy(*reply_head,line);
      }
    else
      {
       *reply_head=(char*)realloc((void*)*reply_head,strlen(line)+strlen(*reply_head)+1);
       strcat(*reply_head,line);
      }

    if(*line=='\r' || *line=='\n')
       break;
   }

 if(!line)
   return(0);

 if(sscanf(*reply_head,"%*s %d",&status)!=1)
    status=0;

 return(status);
}
