#include <pwd.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <signal.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/utsname.h>

#include "arena.h"

#ifndef ARENA_NO_LOCALE
#  ifdef HAVE_LOCALE_H
#    include <locale.h>
#  endif
#  ifdef HAVE_LIBINTL_H
#    include <libintl.h>
#  endif
#endif

#ifndef NFDBITS
#  define NFDBITS 0
#endif

#include "defs.h"
#include "arenaXdefaults.h"
#ifdef BOOKMARKS
#  include "bookmark.h"
#endif
#ifdef ARENA_DEBUG
#  include "debug.h"
#endif
#include "error.h"
#include "applet.h"
#include "bridge.h"
#include "cmdline.h"
#include "colour.h"
#include "display.h"
#include "forms.h"
#include "html.h"
#include "icon.h"
#include "image.h"
#include "main_i.h"
#include "main.h"
#include "menu.h"
#include "neweditor.h"
#include "parsecommand.h"
#include "parsehtml.h"
#include "popup.h"
#include "scrollbar.h"
#include "spawn.h"
#include "status.h"
#include "style.h"
#include "toolbar.h"
#include "util.h"
#include "x11.h"

#include "HTChunk.h"


#ifdef ARENA_FLYBY_HELP
/* Hinting data from toolbar.c */
extern FlyByButtonData FlyByData[];

/* Default(arena.h) "--flyby (none|hint|terse|verbose)" to alter its value.
 * Global!  toolbar.c
 */
int FlyByHints = ARENA_FLYBY_HOW_MUCH;

/* Pick up compiled in "delay" timer incrementer from arena.h
 * Each time a button has its hint displayed, the amount of time the user has
 * to "sit still" over THAT button is increased... Of course, this is to remove
 * the silly hint as the user keeps using THAT button more and more!
 */
static int FlyByTimerIncrement = ARENA_FLYBY_TIMER_INCREMENT;

/* Internal static data                                                      */
static int FlyByStoredX = -2;
static int FlyByStoredY = -2;
static int FlyByShowing = 0;
static int FlyByAlarm = 0;
static Window FlyByWindow;
static GC FlyByGC;
static XFontStruct *FlyByFont = NULL;

static long FlyByEventMask = (ExposureMask | LeaveWindowMask |
			      KeyPressMask | KeyReleaseMask |
			      ButtonPressMask | ButtonReleaseMask |
			      ButtonMotionMask | PointerMotionMask |
			      StructureNotifyMask | PropertyChangeMask);

RETSIGTYPE ArenaSIGALRMHandler(int);
int FlyByCreate(int x, int y, char *text);
void FlyByDestroy(void);
#endif   /* ARENA_FLYBY_HELP */

#ifndef XK_DeleteChar
#  define XK_DeleteChar XK_Delete
#endif

#define BITMAPDEPTH 1

/* values for window_size in main, is window big enough to be useful */

#define SMALL 1
#define OK 0

Bool initialised = False;	/* avoid X output until this is True! */
Bool UsePaper = True;
int fontsize = 0;

char *CacheRoot = NULL;	/* QingLong.06-12-96 */ /* Dynamically allocated */
Bool Caching = True;    /* glk.07-01-97      */

HTUserProfile* UserProfile = NULL;              /* QingLong.13-02-97     */

char *ArenaUser = NULL;	/* QingLong.30-06-97:
			   the username */
char *Printer = NULL;	/* howcome 5/10/94:
                           command line option enables printing */
char *Editor = NULL;	/* howcome 5/10/94:
                           command line option turns the editor on */
char *EmailAddr = NULL; /* glk 26/06/97:
			   command line option specifies our email address */
int  Badflags = 5;	/* howcome 5/10/94:
                           how many badflags do we want in the output */
char *Icon = NULL;	/* howcome 5/10/94:
                           command line option turns the use of icon */
Bool UseMailcap = True;

Bool NoStyle = False;	/* howcome 2/3/95 */
char *UserSpecifiedStyle = NULL;

int sbar_width = 15;    /* command line option, used in scrollbar.c */
Atom ARENA_LOCATION;
Atom ARENA_COMMAND;

long main_stack[256];
long *stack;

#if defined(ARENA_DEBUG)	/* QingLong.08-12-96 */
#  ifndef LIBWWW_TRACE_FLAGS
#    define LIBWWW_TRACE_FLAGS "abcdefghijklmnopqrstuwvxyzABCDEFGHIJKLMNOPQRSTUWVXYZ";
#  endif
#  ifndef LIBWWW_TRACE_FLAGS_LENGTH
#     define LIBWWW_TRACE_FLAGS_LENGTH 66
#  endif
char *LibWWW_traceflags = NULL;	/* QingLong.09-12-96 */	/* Dynamic */
#  if defined(LIBWWW_TRACE_MASK)
int library_trace = LIBWWW_TRACE_MASK;
#   else
int library_trace = 255;
#  endif
# else
int library_trace = 0;	/* howcome 20/11/94 */
#endif

Doc* CurrentDoc;
Context* context;

/* Display and screen are used as arguments to nearly every Xlib routine,
 * so it lamerizes and simplifies routine calls to declare them global.
 */
Display* display;
Visual* visual;
int screen;
int depth;
Window win;
int ExposeCount;   /* used to monitor GraphicsExpose events during scrolling */
char *Application;        /* name of invoking program */
int default_pixmap_width, default_pixmap_height;
Pixmap default_pixmap;

Cursor hourglass;
int shape;

#ifdef SELECTION
unsigned long statusSelectColour; /* howcome */
static char *selectStr = NULL;		/* howcome 2/8/94 */
#endif

int charWidth, charHeight, charAscent;
long lineHeight;
unsigned int win_width, win_height;

int gadget;
int gatewayport = 3000;     /* GATEWAYPORT; */
char *gateway;              /* gateway if can't access server directly */
char *HelpURL = NULL;
char *startwith = NULL;     /* default initial document */

Bool RepeatButtonDown = False;

XFontStruct* fixed_font;
XFontStruct* Fonts[FONTS];  /* array of fonts */
char *FontsName[FONTS];
int LineSpacing[FONTS];
int BaseLine[FONTS];
int StrikeLine[FONTS];
int LineThickness[FONTS];

int ListIndent1, ListIndent2;

GC gc_fill;
static char *display_name = NULL;
static int window_size = 0;    /* OK or SMALL to display contents */
static int button_x, button_y;
static XSizeHints size_hints;
static XWMHints* WinMan_hints = NULL;

/* For delete window messages -- shared between main() and GuiEvents() */
#define Arena_xa_WM_DELETE_WINDOW 0
#define Arena_xa_WM_TAKE_FOCUS    1
#define Arena_xa_WM_PROTOCOlS     2
#define Arena_xa_WM_protocols_N   3
static Atom  Arena_WM_xa_list[Arena_xa_WM_protocols_N];
static char* Arena_WM_xa_names_list[Arena_xa_WM_protocols_N] =
             { "WM_DELETE_WINDOW", "WM_TAKE_FOCUS", "WM_PROTOCOLS" };

unsigned int display_width, display_height;
double display_pt2px;

struct itimerval itimer, old_itimer;


 /* pause for delay milliseconds */
 /* howcome moved this from tcp.c 20/9/94 */
void Pause(int delay)
{
 struct timeval timer;

 timer.tv_sec = 0;
 timer.tv_usec = delay * 1000;

 select(NFDBITS, 0, 0, 0, &timer);
}


RETSIGTYPE CatchCore(int signo)
{
#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
 Arena_TracePrint(" CatchCore", " pid %d\a\n", (int)getpid());
#endif
 Exit(1);
}


RETSIGTYPE CatchQuit(void)
{
#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
 Arena_TracePrint("CatchQuit", " CATCH_QUIT\n");
#endif
 Exit(1);
}


void CatchX(Display* err_dpy_p, XErrorEvent* err_event)
{
#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
 Arena_TracePrint("CatchX", " pid %d\n", (int)getpid());
#endif
 return;
}


static Context* NewContext(void)
{
 Context* context_new;

 if ((context_new = (Context*)HT_CALLOC(1, sizeof (Context))) == NULL)
   HT_OUTOFMEM("Context_new");
 return context_new;
}


static void DeleteContext(Context* thecontext)
{
 if (thecontext == NULL) return;

 HTList_delete(thecontext->conversions);
 HTList_delete(thecontext->encodings);

/*
 * What must one do to ``HTParentAnchor* home_anchor'' to free it?
 */

/*
 * Free ``StyleSheet* style''? But how?!
 */

/*
 * How must one do free with that wart named ``History* current_history''?
 */
 HT_FREE(thecontext);
}


/*
 * howcome 4/10/94: need to clean the library cache..
 */
RETSIGTYPE Exit(int signo)
{
#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
 if (VERBOSE_TRACE) Arena_TracePrint("Exit(main.c)", " Exit(%i)\n", signo);
#endif

 /* Clean up any "running" children we have created */
 SpawnCleanUpChildren();

 /* Now clean up libwww */
 libExit(0);
 Free(CacheRoot);
 ArenaBridgeQuit();
 DeleteContext(context);

 /* Free some X stuff */
 XFree(WinMan_hints);

#ifndef ARENA_DEBUG
 /* internationalized */
 printf("\n");
 printf(_("Thanks for using %s %s."
	  " The authors -- Dave, Hakon, Henrik \nand Yves --"
	  " can be reached at arena@w3.org.\n"
	  " Arena project coordinator (QingLong) at Yggdrasil\n"
	  " can be reached at %s.\n"
	  " Please check %s \nbefore reporting bugs.\n"),
        BANNER, ARENA_VERSION, DEVELOPERS_MAILTO, HELP_URL);
#endif

 exit(signo);
}



 /* find best visual for specified visual class */
Visual* BestVisual(int class, int *depth)
{
 long visual_info_mask;
 int number_visuals, i, best_depth;
 XVisualInfo *visual_array, visual_info_template;
 Visual *best_visual;

 visual_info_template.class = class;
 visual_info_template.screen = DefaultScreen(display);
 visual_info_mask = VisualClassMask | VisualScreenMask;

 visual_array = XGetVisualInfo(display, visual_info_mask,
                               &visual_info_template,
                               &number_visuals);

 *depth = 0;
 best_depth = 0; /* *depth */
 best_visual = 0;

 for (i = 0; i < number_visuals; ++i)
   {
    if (visual_array[i].depth > *depth)
      {
       best_visual = visual_array[i].visual;
       *depth = visual_array[i].depth;
      }
    /* End if */
    if (*depth == best_depth) break;
   }
 /* End for */

 XFree((void *)visual_array);
 return best_visual;
}


#ifdef SELECTION

void SetSelection(char *s) /* howcome 2/8/94 */
{
 Free(selectStr);
 selectStr = strdup(s);
 XSetSelectionOwner(display, XA_PRIMARY, win, CurrentTime);
 XStoreBytes(display, selectStr, Arena_StrLen(selectStr));
}

#endif


/*
 * send myself an expose event to force redrawing of give rectangle
 */
void Redraw(int x, int y, int w, int h)
{
 XExposeEvent event;

 event.type = Expose;
 event.serial = 0;
 event.send_event = 1;
 event.display = display;
 event.window = win;
 event.x = x;
 event.y = y;
 event.width = w;
 event.height = h;
 event.count = 0;

#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
 if (X_TRACE) Arena_TracePrint("Redraw", " redraw %dx%d%+d%+d.\n", w, h, x, y);
#endif
 XSendEvent(display, win, 0, ExposureMask, (XEvent*)&event);
}


void SetBanner(char *title)
{
 XTextProperty textprop;

 /* Support multiple <TITLE>s in a document */
 Free(CurrentDoc->title);

 CurrentDoc->title = strdup(title);
 XStoreName(display, win, title);
 if (CurrentDoc->url)
   {
    char *v;
    int v_size;

              /* 13: 'URL='(string)\0'TITLE='(string)\0 \0 */ 
    v_size = Arena_StrLen(title) + Arena_StrLen(CurrentDoc->url) + 13;
    v = (char *)Arena_CAlloc(v_size, sizeof(char), False);
    sprintf(v, "URL=%s%cTITLE=%s", CurrentDoc->url, 0, title);
    XChangeProperty(display, win, ARENA_LOCATION, XA_STRING, 8,
		    PropModeReplace, v, v_size);
    Free(v);
   }

#if 0
 if (Arena_StrLen(title) > 8 && (p = ARENA_rIndex(title, '/')))
   {
    if (p[1] != '\0') ++p;
    title = p;
   }
#endif

 textprop.value = (unsigned char *)title;
 textprop.encoding = XA_STRING;
 textprop.format = 8;
 textprop.nitems = Arena_StrLen(title);

 XSetWMIconName(display, win, &textprop);
}


void LoadFont(int idx, char *font_name, char *fall_back)
{
 int direction_hint, font_ascent, font_descent;
 XFontStruct *font;
 XCharStruct overall;
 char *test = "Testing";
#ifdef ARENA_DEBUG
 char Iam[] = "LoadFont";
#endif


 if ((Fonts[idx] = XLoadQueryFont(display, font_name)) == NULL)
   {
#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
    (void) Arena_TracePrint(Iam,
			    " Cannot open \"%s\" font.\n", font_name);
# else
    (void) Arena_PrintError(_("Cannot open \"%s\" font.\n"), font_name);
#endif

    if ((Fonts[idx] = XLoadQueryFont(display, fall_back)) == NULL)
      {
#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
       (void) Arena_TracePrint(Iam,
			       " Cannot open alternate font: \"%s\".\n",
			       fall_back);
# else
       (void) Arena_PrintError(_("Cannot open alternate font: \"%s\".\n"),
			       fall_back);
#endif
       Exit(1);
      }

#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
    Arena_TracePrint(Iam,
		     " Using alternate font: \"%s\".\n", fall_back);
# else
    Arena_PrintError(_("Using alternate font: \"%s\".\n"), fall_back);
#endif
   }

 font = Fonts[idx];

 XTextExtents(font, test, Arena_StrLen(test),
	      &direction_hint,
	      &font_ascent,
	      &font_descent,
	      &overall);

 charWidth = overall.width/Arena_StrLen(test);
 charHeight = overall.ascent + overall.descent;
 charAscent = overall.ascent;
 lineHeight = charHeight + overall.ascent/4 + 2;

 LineSpacing[idx] = SPACING(font);
 BaseLine[idx] = BASELINE(font);
 StrikeLine[idx] = STRIKELINE(font);
}


/*
 * Resize Area main window and upate all subwindows sizes appropriately.
 */
void ResizeArenaWindow(unsigned int newWidth, unsigned int newHeight)
{
 Bool window_width_changed = (win_width != newWidth);
 XRectangle aRect;

 window_size = (((newWidth  >= size_hints.min_width) &&
		 (newHeight >= size_hints.min_height)) ?
                                                    OK : SMALL);
 win_width  = newWidth;
 win_height = newHeight;

 if (window_width_changed)
   {
    UpdateToolBarGeometry();
    UpdateScrollBarGeometry();
    UpdateDocDispWinGeometry();

    aRect = GetUpdateArenaIconGeometry();
    XMoveResizeWindow(display, IconWin,
		      aRect.x, aRect.y, aRect.width, aRect.height);

    aRect = GetToolBarGeometry();
    XMoveResizeWindow(display, ToolBarWin,
		      aRect.x, aRect.y, aRect.width, aRect.height);

    aRect = GetArenaStatusLineGeometry();
    XMoveResizeWindow(display, StatusWin,
		      aRect.x, aRect.y, aRect.width, aRect.height);
   }

 aRect = GetScrollXBarGeometry();
 XMoveResizeWindow(display, ScrollXBarWin,
		   aRect.x, aRect.y, aRect.width, aRect.height);
 aRect = GetScrollYBarGeometry();
 XMoveResizeWindow(display, ScrollYBarWin,
		   aRect.x, aRect.y, aRect.width, aRect.height);

 aRect = GetDocDispWinGeometry();
 XMoveResizeWindow(display, DocDispWin,
		   aRect.x, aRect.y, aRect.width, aRect.height);

 DisplayAll(CurrentDoc, True);
}


void ButtonDown(Window theWindow,
		unsigned int button, unsigned int state,
		int x, int y, int x_root, int y_root)
{
#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
 char Iam[] = "ButtonDown";
#endif

#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
 if (X_TRACE)
   Arena_TracePrint(Iam,
		    " ButtonPress %i at %d(%d, %d).\n",
		    button, theWindow, x, y);
#endif

 if (theWindow == DocDispWin)
   {
    switch (button)
      {
       case Button1:
	 {
	 /* Do NOT remove focus here! */
	  gadget = WindowButtonDown(DocDispGC, GetDefaultDocViewWindow(),
				    button, x, y);
	 }
	 break;

       case Button2:
	 FinishCurrentFieldEditing(DocDispGC, GetDefaultDocViewWindow());
	 ShowLink(DocDispGC, GetDefaultDocViewWindow(), x, y);
	 break;

#ifdef POPUP
       case Button3:
	 FinishCurrentFieldEditing(DocDispGC, GetDefaultDocViewWindow());
	 if (HistoryMode(HISTORY_QUERY) == HISTORY_TYPE_NONE)
	   Beep();
	 else
	   Popup(DocDispGC, GetDefaultDocViewWindow(), x_root, y_root);
	 break;
#endif

#ifdef ARENA_DEBUG
       default:
	 if (X_TRACE) Arena_TracePrint(Iam, " WEIRD! Button %d.\n", button);
	 break;
#endif
      }
    /* End ``switch (button)'' */
   }
  else
   {
    FinishCurrentFieldEditing(DocDispGC, GetDefaultDocViewWindow());

    if (theWindow == ToolBarWin)
      {
       gadget = ToolBarButtonDown(button,x, y);
      }
     else
      if (theWindow == StatusWin)
	gadget = StatusButtonDown(button, x, y);
       else
	if (theWindow == ScrollXBarWin)
	  {
	   button_x = x;
	   button_y = y;
	   gadget = ScrollXButtonDown(x, y);
	  }
	 else
	  if (theWindow == ScrollYBarWin)
	    {
	     button_x = x;
	     button_y = y;
	     gadget = ScrollYButtonDown(x, y);
	    }
	   else
	    {
	     gadget = VOID;
	    }
   }
}


void ButtonUp(Window theWindow,
	      unsigned int button, unsigned int state, int x, int y)
{
 int shifted;
#ifdef ARENA_DEBUG
 char Iam[] = "ButtonUp";
#endif


#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
	 if (X_TRACE)
	   Arena_TracePrint(Iam,
			    " ButtonRelease %i at %d(%d, %d).\n",
			    button, theWindow, x, y);
#endif

 RepeatButtonDown = False;
 shifted = ShiftMask & state;

 if (theWindow == DocDispWin)
   WindowButtonUp(DocDispGC, GetDefaultDocViewWindow(),
		  button, shifted, x, y);
  else
   {
    if (gadget == SCROLLBAR)
      ScrollButtonUp(x, y);
     else
      if (gadget == TOOLBAR)
	ToolBarButtonUp(button,x, y);
       else
	if (gadget == STATUS)
	  StatusButtonUp(x, y);
   }
}

#ifdef ARENA_FLYBY_HELP
/* Fly by help alarm handlers
 * glk.10.23.97
 * This code is concentrated here since GuiEvents() is where we use it.
 * There is also some code in toolbar.h & .c to locate button data.
 */
RETSIGTYPE ArenaSIGALRMHandler(int signo)
{
 static int Lx, Ly;


 FlyByAlarm = 0;
 alarm(FlyByAlarm);
 signal(SIGALRM, SIG_DFL);
 if (FlyByHints == ARENA_FLYBY_NONE) return;

 /* If NOT already showing a "hopefully" helpful hint */
 if (FlyByShowing == 0)
   {
    /* see if within one of the buttons... >0 is button... i-1 is subscript! */
    int i = ToolBarWhichButton(FlyByStoredX, FlyByStoredY);
    if (i > 0)
      {
       /* Is it the same as "last" time we were here? */
       if (Lx == FlyByStoredX && Ly == FlyByStoredY)
	 {
	  char *text = ToolBarFlyByText(i);
	  FlyByCreate(FlyByStoredX, FlyByStoredY+20, text ? text :"(null)");
	  FlyByShowing = i;
	  FlyByData[i-1].displayed_times += FlyByTimerIncrement;
	 }
       else
	 {
	  /* NO it is different... so re-arm the alarm... if we come back
	   * and it's the same, we'll diplay it then!  Use "last" timer
	   * delay... but NOT zero!
	   */
	  FlyByAlarm = 1;
	  signal(SIGALRM, ArenaSIGALRMHandler);

	  if (FlyByData[i-1].displayed_times == 0)
	    FlyByData[i-1].displayed_times = 1;
	  alarm(FlyByData[i-1].displayed_times);
	 }
      }
   }
 else
   {
    /* We should never be here... so destroy the hint! */
    FlyByDestroy();
   }

 /* always remember the Last x,y since we are here anyway! */
 Lx = FlyByStoredX;
 Ly = FlyByStoredY;
 return;
}

/* FlyByCreate
 * At x,y... display the text string in a popup help window.
 * Will take care of multi line text... Maximum number of lines = 20!
 */
int FlyByCreate(int x, int y, char *text)
{
  char *t;
  char *temp = NULL;
  char *line[20] = { NULL };
  int lwid, mwid, cwid, nctext, nline, i;
  unsigned int w, h;
  XSetWindowAttributes setwinattr;
  Cursor menucursor;
  unsigned long valuemask;

  /* Don't do anything if there's nothing to do...
   * Boy that's some helpful comment... huh?
   */
  nctext = Arena_StrLen(text);
  if (nctext == 0 || FlyByHints == ARENA_FLYBY_NONE) return EXIT_FAILURE;

  if (FlyByFont == NULL)
    FlyByFont = Fonts[IDX_FIXEDFONT];

  /* Make working copy of input text...
   * Calculate the number of lines and keep pointers to each.
   */
  StrAllocCopy(temp, text);
  if (*(temp + nctext - 1) != '\n') StrAllocCat(temp, "\n");

  for (nline = 0, t = temp; *t ; t++)
    {
     if (*t == '\n')
       {
	*t = 0;
	if (nline >= 20)
	  break;
	nline++;
       }
     else
       {
	if (line[nline] == NULL)
	  line[nline] = t;
       }
    }

  /* Caclulate the pixel length of each line...
   * And the width of the Maximum length line.
   */
  for (i = 0, mwid = 0; i < nline; i++)
    {
     nctext = Arena_StrLen(line[i]);
     if (nctext > 0)
       {
	lwid = XTextWidth(FlyByFont, line[i], nctext);
	mwid = max(lwid, mwid);
       }
    }
  if (mwid <= 2)
    {
     Free(temp);
     return EXIT_FAILURE;
    }


  /* Ok, now we can calculate the size of the popup window! */
  cwid = XTextWidth(FlyByFont, " ", 1);
  w = mwid + cwid;
  h = SPACING(FlyByFont) * nline;

  /* decide whether x and y are ok or need to be modified */
  x = (x + w > display_width  ? display_width  - w - 1 : x);
  y = (y + h > display_height ? display_height - h - 1 : y);

  /* create window */
  FlyByWindow = XCreateSimpleWindow(display, win, x, y, w, h, 1,
				    labelColour, flybyColour);

  /* define a cursor  */
  menucursor = XCreateFontCursor(display, XC_arrow);
  
  /* set winattrs */
  XDefineCursor(display, FlyByWindow, menucursor);

  valuemask =  CWOverrideRedirect;
  setwinattr.override_redirect = True;

  if(ColourStyle != COLOUR888)
    {
      valuemask |= CWColormap;
      setwinattr.colormap = colourmap;
    }

  XChangeWindowAttributes(display, FlyByWindow, valuemask, &setwinattr);
  
  /* create GC for pop */
  FlyByGC = XCreateGC(display, FlyByWindow, 0, NULL);

  /* set up the font */
  if (FlyByFont == NULL) FlyByFont = Fonts[IDX_FIXEDFONT];
  XSetFont(display, FlyByGC, FlyByFont->fid);
  XSetForeground(display, FlyByGC, labelColour);

  /* map the popup window */
  XMapWindow(display, FlyByWindow);

  /* display each line of text */
  for (i = 0; i < nline; i++)
    {
     if ((nctext = Arena_StrLen(line[i])) > 0)
       {
	 XDrawString(display, FlyByWindow, FlyByGC,
		     cwid * 0.5 ,
		     BASELINE(FlyByFont) + (i * SPACING(FlyByFont)),
		     line[i], Arena_StrLen(line[i]));
       }
    }

  /* flush the display buffer, and relase temps */
  XFlush(display);
  Free(temp);
  return EXIT_SUCCESS;
}

/* FlyByDestroy
 * Used to get rid of any help text window that may be showing
 * AND to remove any alarms... Any alarms needed must be reinstated.
 */
void FlyByDestroy(void)
{
 if (FlyByAlarm != 0)
   {
    FlyByAlarm = 0;
    alarm(FlyByAlarm);
    FlyByStoredX = FlyByStoredY = -1;
    signal(SIGALRM, SIG_DFL);
   }

 if (FlyByShowing)
   {
    FlyByShowing = 0;
    XUndefineCursor(display, FlyByWindow);
    XUnmapWindow(display, FlyByWindow);
    XDestroyWindow(display, FlyByWindow);
    XFlush(display);
   }
}
#endif /* ARENA_FLYBY_HELP */


/*
 * Main X11 event loop. Called from various places...
 * This code can be used while a child process is running in "WAIT_GUI" mode.
 * This code can ONLY allow those events that keep the Gui looking good...
 */
int GuiEvents(int theSocket, HTRequest* theRequest, int theSocketOps)
{
 extern GC DocDispGC;

 XEvent event;
 KeySym key;
 XComposeStatus theComposeStatus;
 int count, i;
#define ARENA_GUIevents_KeyBuf_Size 64
 int  keybufsize = ARENA_GUIevents_KeyBuf_Size - 1;
 char keybuf[ARENA_GUIevents_KeyBuf_Size];
#undef  ARENA_GUIevents_KeyBuf_Size
#ifdef ARENA_DEBUG
 char Iam[] = "GuiEvents";
#endif

 for (i = XEventsQueued(display, QueuedAfterReading); i; i--)
   {
    XNextEvent(display, &event);

    switch (event.type)
      {
       case ClientMessage:
         if (event.xclient.message_type
	     == Arena_WM_xa_list[Arena_xa_WM_PROTOCOlS])
	   {
	    if (event.xclient.data.l[0]
		== Arena_WM_xa_list[Arena_xa_WM_TAKE_FOCUS])
	      {
#ifdef ARENA_DEBUG	/* QingLong.21-01-98 */
	       if (X_TRACE)
		 Arena_TracePrint(Iam,
				  " WM sets input focus to window %d%s.\n",
				  event.xclient.window,
				  ((event.xclient.window == win) ?
				   " (Arena main)" : ""));
#endif
	       if (event.xclient.window != win)
		 {
#ifdef ARENA_DEBUG	/* QingLong.21-01-98 */
		  if (X_TRACE)
		    Arena_TracePrint(Iam,
				     " re-set  input focus to window %d"
				     " (Arena main).\n",
				     win);
#endif
		  XSetInputFocus(display, win, RevertToParent, CurrentTime);
		 }
	      }
	     else
	      if (event.xclient.data.l[0]
		  == Arena_WM_xa_list[Arena_xa_WM_DELETE_WINDOW])
		{
#ifdef ARENA_DEBUG	/* QingLong.21-01-98 */
		 if (X_TRACE)
		   Arena_TracePrint(Iam,
				    " WM deletes window %d%s.\n",
				    event.xclient.window,
				    ((event.xclient.window == win) ?
				     " (Arena main)" : ""));
#endif
		 XCloseDisplay(display);
		 Exit(0);
		}
	   }

	 StatusLineDialog(&event);
	 break;


       case Expose:
	 if (event.xexpose.window == DocDispWin)
	   {
	   /* only redraw pieces that were exposed */
	    XRectangle theExposedArea = { event.xexpose.x,
					  event.xexpose.y,
					  event.xexpose.width,
					  event.xexpose.height };

	    DisplayDocArea(CurrentDoc, theExposedArea);
	   }
	  else
	   if (event.xexpose.window == ToolBarWin)
	     DisplayToolBar();
	    else
	     if (event.xexpose.window == StatusWin)
	       DisplayStatusLine();
	      else
	       if (event.xexpose.window == IconWin)
		 DisplayIcon();
	       else
		 if ((event.xexpose.window == ScrollXBarWin) ||
		     (event.xexpose.window == ScrollYBarWin))
		   DisplayScrollBar();
#ifdef ARENA_DEBUG
		 else
		   {
		    if (X_TRACE)
		      Arena_TracePrint(Iam,
				       " ERROR!"
				       " \"Expose\" unknown window %d.\n",
				       event.xexpose.window);
		   }
#endif

	 break;


       case GraphicsExpose:
	 if (event.xgraphicsexpose.drawable == DocDispWin)
	   {
	   /* only redraw pieces that were exposed */
	    XRectangle theExposedArea = { event.xgraphicsexpose.x,
					  event.xgraphicsexpose.y,
					  event.xgraphicsexpose.width,
					  event.xgraphicsexpose.height };

	    DisplayDocArea(CurrentDoc, theExposedArea);
	   }
	  else
	   if (event.xgraphicsexpose.drawable == ToolBarWin)
	     DisplayToolBar();
	    else
	     if (event.xgraphicsexpose.drawable == StatusWin)
	       DisplayStatusLine();
	      else
	       if (event.xgraphicsexpose.drawable == IconWin)
		 DisplayIcon();
	       else
		 if ((event.xgraphicsexpose.drawable == ScrollXBarWin) ||
		     (event.xgraphicsexpose.drawable == ScrollYBarWin))
		   DisplayScrollBar();
#ifdef ARENA_DEBUG
		  else
		   {
		    if (X_TRACE)
		      Arena_TracePrint(Iam,
				       " ERROR!"
				       " \"Expose\" unknown window %d.\n",
				       event.xgraphicsexpose.drawable);
		   }
#endif

	 ExposeCount = event.xgraphicsexpose.count;
	 break;


       case NoExpose:
	 ExposeCount = 0;
	 break;


       case ConfigureNotify:
	 if ((event.xconfigure.window == win) &&
	     (event.xconfigure.width  != win_width ||
	      event.xconfigure.height != win_height))
	   {
	    FinishCurrentFieldEditing(DocDispGC, GetDefaultDocViewWindow());
	    ResizeArenaWindow(event.xconfigure.width, event.xconfigure.height);
	    DisplaySizeChanged(CurrentDoc, True);
	   }
	 break;


	 /* The following is a failed attempt to exit tidily
	  * when the user forces an exit through the system menu.
          */
	 /* I need to do better! */
       case DestroyNotify:
	 if (event.xdestroywindow.window == win)
	   {
#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
	     if (X_TRACE) Arena_TracePrint(Iam, " DestroyNotify.\n");
#endif
	     Exit(1);
	   }


       case MapNotify:
#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
	 if (X_TRACE) Arena_TracePrint(Iam, " MapNotify.\n");
#endif
	 initialised = True;   /* allow status display now its safe */
	 break;


       case UnmapNotify:
#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
	 if (X_TRACE) Arena_TracePrint(Iam, " UnmapNotify.\n");
#endif
	 break;


       case ReparentNotify:
	 break;


      case PropertyNotify:
	{
	 /* thanks to Colas Nahaboo */
	 Atom            actual_type;
	 int             actual_format;
	 unsigned long   nitems;
	 unsigned long   bytes_after;
	 unsigned char  *buffer = 0;

	 if (Success == XGetWindowProperty(display, win, ARENA_COMMAND,
					   0, 8000, True,
					   AnyPropertyType, &actual_type,
					   &actual_format, &nitems,
					   &bytes_after, &buffer))
	   {
	    if (buffer)
	      {
	       buffer[nitems] = '\0';
	       if (!(actual_type == XA_STRING && ((signed long)nitems >= 0))) 
		 {
		  XFree(buffer);
		  buffer = 0;
		 } /* else ok */
	      }
	    else
	      {
	       if (buffer) XFree(buffer);
	       buffer = 0;
	      }
	   }

	 /* If we have a "command line", go parse and execute it!
	  * Currently only GOTO URL= and GOTO NEW URL= are supported
	  */
	 if (buffer)
	   {
	    ParseCommandLine(buffer, nitems);
	    XFree(buffer);
	    buffer = 0;
	   }
	}
	break;


#ifdef SELECTION
	 /* selection event handling added by howcome 2/8/94 */

       case SelectionClear:
	 ClearStatusSelection();
	 DisplayStatusLine();
	 break;

       case SelectionNotify:
#  ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
	 Arena_TracePrint(Iam, " SelectionNotify.\n");
#  endif
	 break;

       case SelectionRequest:
	 {
	  XSelectionEvent selEv; 

	  if (event.xselectionrequest.target != XA_STRING)
	    {
	     selEv.property = None;
	    }
	  else
	    {
	     selEv.property = event.xselectionrequest.property;
	     XChangeProperty(event.xselectionrequest.display,
			     event.xselectionrequest.requestor,
			     event.xselectionrequest.property,
			     event.xselectionrequest.target,
			     8, PropModeReplace, selectStr,
			     Arena_StrLen(selectStr));
	    }

	  selEv.type = SelectionNotify;
	  selEv.display = event.xselectionrequest.display;
	  selEv.requestor = event.xselectionrequest.requestor;
	  selEv.selection = event.xselectionrequest.selection;
	  selEv.target = event.xselectionrequest.target;
	  selEv.time = event.xselectionrequest.time;

	  XSendEvent(event.xselectionrequest.display,
		     event.xselectionrequest.requestor,
		     False, 0, (XEvent*) &selEv);
	 }
	 break;

#endif /* SELECTION */

       case ButtonPress:
#ifdef ARENA_FLYBY_HELP
	 FlyByDestroy();
#endif
	 ButtonDown(event.xbutton.window,
		    event.xbutton.button, event.xbutton.state,
		    event.xbutton.x, event.xbutton.y,
		    event.xbutton.x_root, event.xbutton.y_root);
	 break;


       case ButtonRelease:
	 ButtonUp(event.xbutton.window,
		  event.xbutton.button, event.xbutton.state,
		  event.xbutton.x, event.xbutton.y);
	 break;


       case LeaveNotify:
#ifdef ARENA_FLYBY_HELP
	 FlyByDestroy();
#endif
	 break;

       case MotionNotify:
	 /* ignore event if still repairing holes from earlier copy operation*/
	 if (ExposeCount) break;

#ifdef ARENA_FLYBY_HELP
	 /* Remove any fly by help and its alarm... if necessary */
	 FlyByDestroy();

	 /* See if button 1 down...
	  * If so, then check if this is a scrollbar...
	  * If NOT, then we are moving around with some
	  * other button down, or with NO button down!
	  */
	 if (event.xmotion.state & Button1MotionMask)
	   {
	   /* Here because of Button 1 Down and Motion Event.
	    * That can only be valid in our case...
	    * If doing a "selection" of text, or if scrolling.
	    * Get rid of all other B1 MotionNotify events on the queue.
	    */
	    while (XCheckMaskEvent(display, Button1MotionMask, &event)) ;

	    if (event.xmotion.window == ScrollXBarWin ||
		event.xmotion.window == ScrollYBarWin)
	      {
	       ScrollButtonDrag(event.xbutton.x, event.xbutton.y);
	      }
#ifdef SELECTION
	     else
	      if (event.xmotion.window == StatusWin)
		{
		 SelectStatus(event.xbutton.x, event.xbutton.y);
		}
#endif
	   }
	  else
	   {
	    if (event.xmotion.state &
		(ButtonMotionMask  | Button1MotionMask | Button2MotionMask |
		 Button3MotionMask | Button4MotionMask | Button5MotionMask))
	      {
	       /* Blow off any button down motion notifications */
	       while (XCheckMaskEvent(display,
				      (ButtonMotionMask |
				       Button1MotionMask | Button2MotionMask |
				       Button3MotionMask | Button4MotionMask |
				       Button5MotionMask),
				      &event));  /* watch out for no-op */
	       break;
	      }

	    /* Moving around with NO button down... Iff we are "hinting"...
	     * See if within the toolbar. If so, then see if within a button.
	     * Also, don't do anything if Arena is busy with other stuff!
	     * If ok, remember x,y and set our alarmclock.  If the alarm
	     * goes off twice with the same x,y... IT will display the help!
	     */
	    while (XCheckTypedEvent(display, event.type, &event)) ;

	    if (FlyByHints != ARENA_FLYBY_NONE)
	      {
	       FlyByStoredX = event.xmotion.x;
	       FlyByStoredY = event.xmotion.y;

	       if ((event.xmotion.window == ToolBarWin) && !Busy() )
		 {
		  int i = ToolBarWhichButton(FlyByStoredX, FlyByStoredY);
		  if (i > 0)
		    {
		     FlyByAlarm = 1;
		     signal(SIGALRM, ArenaSIGALRMHandler);
		     alarm(FlyByAlarm);
		    }
		 }
	      }
	   }
#endif  /* ARENA_FLYBY_HELP */

	 break;


       case KeyPress:
#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
	 if (X_TRACE) Arena_TracePrint(Iam, " KeyPress.\n");
#endif

#ifdef ARENA_FLYBY_HELP
	 FlyByDestroy();
#endif
	 count = XLookupString((XKeyEvent*)&event,
			       keybuf, keybufsize, &key, &theComposeStatus);
	 keybuf[count] = 0;

	 switch (key)
	   {
	   case XK_F2:
	     count = 1;
	     break;

	   case XK_F4:
#if 0
	     ShowPaint(background.top-paint, 14);
#endif
	     break;

	   case XK_F6:
	     ReportVisuals();
	     break;

	   case XK_F7:
	     ReportStandardColourMaps(XA_RGB_DEFAULT_MAP);
	     ReportStandardColourMaps(XA_RGB_BEST_MAP);
	     break;

	   case XK_F8:
	     XCloseDisplay(display);
	     Exit(1);
	     break;

	   case XK_v:
	     if ((event.xkey.state & ControlMask))
	       {
	        MoveDownPage(CurrentDoc);
		break;
	       }
	     else
	       if ((event.xkey.state & Mod1Mask))
		 {
		  MoveUpPage(CurrentDoc);
		  break;
		 }
	     /* Endif */
	     /* No break! */

	   default:
	     if (EditDocFormField(DocDispGC, GetDefaultDocViewWindow(),
				  CurrentDoc, key, keybuf))
	       break;

	     /* QingLong.25-09-97: handle status line separately */
	     if (StatusLineDialog(&event)) break;

	     if (key == XK_Up && !ExposeCount)
	       {
		if (event.xkey.state & ControlMask)
		  MoveUpPage(CurrentDoc);
		 else
		  MoveUpLine(CurrentDoc);

		XFlush(display);   /* to avoid scrollbar flashing */
	       }
	      else
	       if (key == XK_Down && !ExposeCount)
		 {
		  if (event.xkey.state & ControlMask)
		    MoveDownPage(CurrentDoc);
		   else
		    MoveDownLine(CurrentDoc);

		  XFlush(display); /* to avoid scrollbar flashing */
		 }
	        else
		 if (key == XK_Left && !ExposeCount)
		   {
		    if (event.xkey.state & (Mod4Mask | ControlMask) &&
			  SpawnIsWaitGui() == False)
		      BackDoc(DocDispGC, GetDefaultDocViewWindow());
		     else
		      MoveLeftLine(CurrentDoc);

		    XFlush(display);/* to avoid scrollbar flashing */
		   }
		  else
		   if (key == XK_Right && !ExposeCount)
		     {
		      if (event.xkey.state & (Mod4Mask | ControlMask) &&
			  SpawnIsWaitGui() == False)
			ForwardDoc(DocDispGC, GetDefaultDocViewWindow());
		       else
			MoveRightLine(CurrentDoc);

		      /* to avoid scrollbar flashing */
		      XFlush(display);
		     }
		   else
		     if (key == XK_Prior && !ExposeCount)
		       MoveUpPage(CurrentDoc);
		      else
		       if ((key == XK_space || key == XK_Next) && !ExposeCount)
			 MoveDownPage(CurrentDoc);
		        else
			 if ((key == XK_Begin || key == XK_Home) &&
			     !ExposeCount)
			   {
			    if (event.xkey.state & ShiftMask)
			      MoveToEnd(CurrentDoc);
			     else
			      MoveToStart(CurrentDoc);
			   }
			  else
			   if (key == XK_End && !ExposeCount)
			     MoveToEnd(CurrentDoc);
			    else
			      if (Busy())
				Beep();
			       else
				if (key == XK_F2 && SpawnIsWaitGui() == False)
				  BackDoc(DocDispGC,
					  GetDefaultDocViewWindow());
				 else
				  if (key == XK_F3 && SpawnIsWaitGui()== False)
				    FindString(CurrentDoc, NULL);
	     break;
	   }
	 /* End ``switch (key)'' */
	 break;

       case KeyRelease:
	 count = XLookupString((XKeyEvent*)&event,
			       keybuf, keybufsize, &key, &theComposeStatus);
	 keybuf[count] = 0;
	 break;

       default:  /* handle all other events */
#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
	 if (X_TRACE)
	   Arena_TracePrint(Iam,
			    " ERROR! Unexpected Event %d.\n", event.type);
#endif
	 break;
       }
    XFlush(display);
   }

 HideBusy();
 return HT_WOULD_BLOCK; /* howcome 24/7/95:
			   must do this to not insult library */
}


char* DetermineValue(char *param, char *def)
{
 extern Display* display;
 extern char* Application;
 char* value;

 if ((value = XGetDefault(display, Application, param)) == NULL) value = def;

 return value;
}


int DetermineColour(char *param, int r, int g, int b, unsigned long *pix)
{
 extern Display* display;
 extern char* Application;
 char *theColourName, *p1, *p2;

 theColourName = XGetDefault(display, Application, param);

 if (theColourName)  /* parse and GetColour() or GetNamedColour() */
   {
   /* "red:green:blue" e.g."205:184:157" or named colour */

    p1 = ARENA_Index(theColourName, ':');
    p2 = ARENA_rIndex(theColourName, ':');

    if (p1 && p2 && p1 != p2)
      {
       sscanf(theColourName, "%d:", &r);
       sscanf(p1 + 1, "%d:", &g);
       sscanf(p2 + 1, "%d", &b);

       return GetColour(r, g, b, pix);
      }

    return GetNamedColour(theColourName, pix);
   }

 return GetColour(r, g, b, pix);
}


/*
 * get font/colour resources
 */
void GetResources(void)
{
 if (fontsize >= 0)
   {
    int i, theFontSize = ((fontsize <= 2) ? fontsize : 2);

    for (i = ArenaFontsNumber; i--;)
      LoadFont(ArenaFonts[i].idx,
	       DetermineValue(ArenaFonts[i].resource_name,
			      ArenaFonts[i].font[theFontSize].def),
	       ArenaFonts[i].font[theFontSize].fallback);
   }

 fixed_font = Fonts[IDX_FIXEDFONT];

 /* list indents for ordered/unordered lists */

 ListIndent1 = XTextWidth(Fonts[IDX_NORMALFONT], "ABCabc", 6)/6;
 ListIndent2 = ListIndent1;
 ListIndent1 = 2 * ListIndent1;

 /* check for monchrome displays */

 if (DefaultDepth(display, screen) == 1)
   {
    strikeColour = labelColour = textColour = BlackPixel(display, screen);
    transparent = statusColour = windowColour = WhitePixel(display, screen);
    flybyColour = windowColour;
    windowShadow = windowTopShadow = windowBottomShadow = textColour;
   }
  else   /* try for colour but degrade as sensibly as possible */
   {
    if (!DetermineColour("windowColour", 220, 209, 186, &windowColour))
      {
       flybyColour = windowColour;
       if (GetNamedColour("gray", &windowColour))
	 {
	  windowBottomShadow = BlackPixel(display, screen);

	  if (!GetNamedColour("dim gray", &windowShadow))
	    windowShadow = windowColour;

	  if (!GetNamedColour("light gray", &windowTopShadow))
	    windowTopShadow = WhitePixel(display, screen);

	  strikeColour = textColour = labelColour = WhitePixel(display, screen);
	 }
       else
	 {
	  labelColour = strikeColour = textColour = BlackPixel(display, screen);
	  windowShadow = windowColour = WhitePixel(display, screen);
	  windowShadow = windowTopShadow = windowBottomShadow = textColour;
	  flybyColour = windowColour;
	 }
      }
    else
      {
       DetermineColour("textColour", 0, 0, 100, &textColour);
       DetermineColour("labelColour", 0, 100, 100, &labelColour);
       DetermineColour("strikeColour", 170, 0, 0, &strikeColour);
       DetermineColour("flybyColour", 238, 238, 0, &flybyColour);
       DetermineColour("windowShadow", 200, 188, 169, &windowShadow);
       DetermineColour("windowTopShadow", 255, 242, 216, &windowTopShadow);
       DetermineColour("windowBottomShadow", 180, 170, 152, &windowBottomShadow);

       if (windowTopShadow == windowColour)
	 windowTopShadow = WhitePixel(display, screen);

       if (windowBottomShadow == windowColour)
	 windowBottomShadow = BlackPixel(display, screen);

       if (windowColour == textColour)
	 {
	  windowShadow = windowColour = BlackPixel(display, screen);
	  strikeColour = windowTopShadow = windowBottomShadow = textColour;
	 }
      }

    transparent = windowColour;
    statusColour = windowShadow;
   }
}


/* create clone of self by forking and duplicating resources
 *
 * creates duplicate of history file and
 * closes all open files, then duplicates stdio ...
 * returns 0 for parent, and 1 for clone
 */
int CloneSelf(void)
{
#if 0
 int childpid, tty;
 int x = 0, y = 0;               /* window position */
 unsigned int border_width = 4;  /* border four pixels wide */
 /* unsigned int display_width, display_height;*/
 char *window_name = BANNER;
 char *icon_name = BANNER;
 int i, fh, depth;
 unsigned int class;
 Visual *visual;
 unsigned long valuemask, where;
 XSetWindowAttributes attributes;

 if ((tty = open("/dev/tty", 2)) == -1 && (tty = open("/dev/null", 2)) == -1)
   {
    Warn("Can't open /dev/tty");
    return 0;
   }

 /* ensure that children won't become zombies */

#ifdef SIGCHLD
 signal(SIGCHLD, SIG_IGN);
# else
 signal(SIGCLD, SIG_IGN);
#endif

 if ((childpid = fork()) < 0)
   {
    close(tty);
    Warn("Can't fork new process");
    return 0;
   }

 if (childpid != 0)
   {
    close(tty);
    return 0;
   }

 /* ok child process - so dup stdio  - this leaves other files open! */

#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
 Arena_TracePrint("CloneSelf", " child pid: %d\n", childpid);
#endif

 close(0); dup(tty);
 close(1); dup(tty);
 close(2); dup(tty);
 close(tty);

 close(ConnectionNumber(display));  /* close TCP connection to X server */

 /* now clone history file and close original */

 initialised = 0;  /* avoid X output until this is true! */

/*    XCloseDisplay(display); */ /* close connection for parent display */

 /* connect to X server */

 if ( (display = XOpenDisplay(display_name)) == NULL )
   {
#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
    (void) Arena_TracePrint("CloneSelf",
			    " cannot connect to X server %s\n",
			    XDisplayName(display_name));
# else
    (void) Arena_PrintError(_("Cannot connect to X server %s\n"),
			    XDisplayName(display_name));
#endif
    exit(-1);
   }

 /* free memory but not pixmaps which we no longer own! */
 FreeImages(1);
 FreeForms();   /* is this right? */

 /* try to allocate 128 fixed colours + 16 grey scales */
 if (InitImaging(ColourStyle) == MONO) UsePaper = False;

/*    GetResources();  *//* load font and colour resources */

 if (ColourStyle == MONO) statusColour = windowColour;

 /* get screen size from display structure macro */

 screen = DefaultScreen(display);

 depth = DisplayPlanes(display, screen);
 class = InputOutput;    /* window class*/
 visual = DefaultVisual(display, screen);
 valuemask = CWColormap | CWBorderPixel | CWBitGravity |
             CWBackingStore | CWBackingPlanes;
 attributes.colormap = colourmap;
 attributes.bit_gravity = ForgetGravity;
 attributes.backing_planes = 0;
 attributes.backing_store = NotUseful;

 win = XCreateWindow(display, RootWindow(display, screen),
		     x, y, win_width, win_height, border_width,
		     depth, class, visual, valuemask, &attributes);

 XSetWindowBackground(display, win, windowColour);


 /* Create default pixmap for use when we can't load images */
 default_pixmap = XCreatePixmapFromBitmapData(display, win,
					      www_bits, www_width, www_height,
					      textColour, transparent,
					      depth);

 default_pixmap_width  = www_width;
 default_pixmap_height = www_height;


 /* initialize size hint property for window manager */

 size_hints.flags = PPosition | PSize | PMinSize;
 size_hints.x = x;
 size_hints.y = y;
 size_hints.width  = win_width;
 size_hints.height = win_height;
 size_hints.min_width  = 440;
 size_hints.min_height = 250;

 /* set properties for window manager (always before mapping) */
 XSetStandardProperties(display, win, window_name, icon_name,
			ArenaIconImage ? ArenaIconImage->pixmap : NULL,
			(char **)0, 0, &size_hints);

 /* select events wanted */
 XSelectInput(display, win,
	      ExposureMask |
	      KeyPressMask | KeyReleaseMask |
	      Button1MotionMask | ButtonPressMask | ButtonReleaseMask |
	      StructureNotifyMask | SubstructureNotifyMask |
	      PropertyChangeMask);


 /* create hourglass cursor */
 hourglass = XCreateFontCursor(display, XC_watch);

 /* create paper (gc_fill background) */
 UsePaper = MakePaper();
 font = -1;

 if (DocHTML(CurrentDoc)) SetBanner(CurrentDoc->url);

 /* refresh paint buffer to ensure that images resources are created */

 where = viewWindow.rect.y;
 viewWindow.rect.y = 0;
 ParseHTML(&i);

 if (where > 0) DeltaHTMLPosition(&(DocViewWin.rect), where);

 /* Map Display Window */

 XMapWindow(display, win);
#endif

 return 1;
}


void myAlarm(void)
{
#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
 Arena_TracePrint("myAlarm", " \a!!!\n");
#endif
 return;
}


void Push(void *item)
{
 *stack = (long)item;
 stack += sizeof(long);
}


void *Pop(void)
{
 if (stack == main_stack) return NULL;

 stack -= sizeof(long);
 return((void *)*stack);
}


/* Read in an rc file for arena.
 * Returns the number of options specified...
 * AND, a new char *argv[].  Here we define the absolute maximum number of
 * args... 4 times the number specified in the CMDLSwitchDefs table.
 * This is enough to handle a system, user, debug, and command line entries!
 */
static int ARENA_argc = 0;
static int ARENA_max_argc = (sizeof(CMDLSwitchDefs)/sizeof(CMDLSwitchDefs[0])) * 1;
static char **ARENA_argv = NULL;

int ReadArenaRC(char *filespec)
{
 char *ep;
 char c, option[256];
 int ich, isquoted, istat;
 HTChunk *chnk;
 FILE *f;

#ifdef ARENA_DEBUG
 char Iam[] = "ReadArenaRC";
#endif

 /* Temporary usage for short variable names... will be returned */
 int tac = ARENA_argc;
 char **tav = &ARENA_argv[tac];

 /* Well, open the file */
 if (filespec == NULL) return EXIT_FAILURE;
 f = fopen(filespec, "r");
 if (f == NULL) return EXIT_SUCCESS;     /* we have to assume this is ok     */

#ifdef ARENA_DEBUG
    if (OPTION_TRACE) Arena_TracePrint(Iam, " FILE arenarc=%s\n", filespec);
#endif

 /* set up a chunky variable */
 chnk = HTChunk_new(512);

 /* NOW, blow through this input file  */
 for (;;)
   {
    istat = fscanf(f, " %255s ", option); /* please note leading & trailing  */
    if (istat == EOF) break;              /* whitespace is taken care of!    */

    /* See if comment */
    if (*option == '#') 
      {
       while (((istat = fgetc(f)) != EOF) && (istat != '\n'))
	 ; /* very null procedure */
       continue;
      }
    /* can we handle another option + value? */
    if (tac+2 > ARENA_max_argc)
      {
       Arena_PrintError(_("Way too many options specified!\n"
			"\tTry `arena --help' for more information.\n"));
       HTChunk_delete(chnk);
       fclose(f);
       return EXIT_FAILURE;
      }

    /* Put the option into new command line table*/
#ifdef ARENA_DEBUG
    if (OPTION_TRACE) Arena_TracePrint(Iam, " Add option %s\n", option);
#endif
    *tav = NULL;
    StrAllocCopy(*tav, "--");
    StrAllocCat(*tav, option);
    tac++;
    tav++;
       
    /* See if we have a "value" for the --option */
    if ((ich = fgetc(f)) == EOF) break;
    if (ich == '#')
      {
       ungetc(ich, f);
       continue;  /* with forever */
      }

    /* Ok, we have a value specified on this line! */
    HTChunk_clear(chnk);
    isquoted = 0;
    if (ich == '"')
      isquoted = ich;
    else
      ungetc(ich, f);

    /* Copy everything until EOF, or an ending double quote  (if quoted)
     * Or, until the end of line if NOT quoted
     */
    while ((ich = fgetc(f)) != EOF)
      {
       /* Quoted? */
       if (isquoted != 0)                         /* are we quoted?          */
	 {
	  if (ich == '"' && isquoted != '\\')     /* at ending quote.        */
	    {
	      while ((ich = fgetc(f)) != EOF && ich != '\n')
		istat = fgetc(f);                 /* get to end of line      */
	     break;
	    }
	  isquoted = ich;
	 }
       else
	 {
	  if (ich == '\n') break;     /* unquoted, and end of line    */
	 }
       HTChunk_putc(chnk, (char)ich); /* put char to our CHUNKY variable */
      } /* end while */
 
    /* Now put our chunky value in the table */
    HTChunk_terminate(chnk);
    istat = HTChunk_size(chnk) - 1;
    if (istat > 0 || isquoted)
      {
       *tav = NULL;
       *tav = StrAllocCopy(*tav, HTChunk_data(chnk));
       if (isquoted == 0)             /* get rid of trailing blanks ONLY for */
	 {                            /* the unquoted values!                */
	  ep = *tav + istat - 1;
	  while ((c = *ep), isspace(c)) *ep-- = 0;
	 }
#ifdef ARENA_DEBUG
       if (OPTION_TRACE) Arena_TracePrint(Iam, " value %s\n", *tav);
#endif
       tac++;
       tav++;
      }
    else
      {
       Arena_PrintError(_("BUG! ZERO length value to option `%s'\n"
			"\tFrom %s\n"
			"\tPlease contact %s\n"),
			option, filespec, DEVELOPERS_MAILTO);
       HTChunk_delete(chnk);
       fclose(f);
       return EXIT_FAILURE;
      }
    if (ich == EOF) break;
   } /* end for(;;) */

 /* Return the correct count of arguments */
 ARENA_argc = tac;

 HTChunk_delete(chnk);
 fclose(f);
 return EXIT_SUCCESS;
}


/*
 * Standard error message output routine for use during command line parsing.
 */
void CMDLineBadArg(char *s, char *a)
{
 char *txt = " Try `arena --help' for more information.\n";
 if (!s)
   Arena_PrintError(_("Unrecognized option `%s'\n%s"), a, txt);
 else
   Arena_PrintError(" %s `%s'\n%s", s, a, txt);
}


/*
 * See if next argument is one of our option switches?
 */
CLSwitch *CMDLineFindSwitch(char *arg)
{
 int alen, olen;
 int match_count = 0;
 CLSwitch *cls = CMDLSwitchDefs;
 CLSwitch *return_cls = NULL;

 /* Must be something to test! */
 if (arg == NULL) return NULL;
 if ((alen = Arena_StrLen(arg)) <= 0) return NULL;

 /* See if it is a switch...
  * If not, we will assume it is a URL!
  * Of course, it may be an invalid  parameter, but...
  */
 if (*arg != '-') return(CMDLSwitchDefs + CMDLSwitchURL);

 /* Search the table for match of the argument...
  * Test 2nd char directly to keep from doing the strncmp (if possible).
  * Then if lengths are equal, do a full comparison of values.
  * If not, allow partial matches...  BUT, force a full search of the
  * table!!  That way we can force only 1 match!  Probably could sort the
  * table and make this thing quicker, but... Then we have to keep it sorted!
  */
 while (cls != NULL && cls->name != NULL)
   {
    if (*(cls->name+1) == *(arg+1))
      {
       olen = Arena_StrLen(cls->name); /* length of "option" from table */
       if (olen >= alen)
	 {
	  if (strncmp(arg, cls->name, alen) == 0)
	    {
	     return_cls = cls;
	     if (olen == alen) return return_cls;   /* exact match!     */
	     match_count++;
	    }
	 }
      }
    cls++;
   }

 if (match_count > 1) return_cls = NULL;    /* found too many matches */
 return return_cls;
}


/* Check the NEXT command line argument.
 * If the current switch has optional or required argument, return it!
 * If the switch requires an argument, output error and return null!
 * NOTE:  You get NULL back for wrong thing next! Error generated!
 * User may enter --option "" which will return a 0 length string.
 * NO error message is generated for the 0 length string!
 */
char *CMDLineNextArg(CLSwitch *pCLS, int *argc, char ***av)
{
 char **argv = *av;

 if (!pCLS) return NULL;

 switch (pCLS->arg)
   {
   default:
     Arena_PrintError(_("Whoa! Invalid pCLS->arg for `%s' (ERROR)\n"),
		      pCLS->name);
     break;

   case CLA_NO:
     /* Could check that the next arg is a switch, but...
      * if it isn't, we can't do anything about that.... It will be
      * used as the initial document URL.
      */
     break;

   case CLA_OPTIONAL:
     /* If the NEXT arg begins with a -
      * Then we ASSUME that there is no argument to current option switch.
      * NOTE: that means you can't specify -1.5 as a value to an option
      * with optional value!
      */
     if (*argc > 1)
       {
	if (*argv[1] != '-')
	  {
	   *argc -= 1;
	   *av = ++argv;
	   return *argv;
	  }
       }
     break;

   case CLA_REQUIRED:
     /* We've been told that we MUST have an argument to this option!
      * If the NEXT arg begins with -     Then problem!
      * Again... cannot specify negative value to option!
      */
     if (*argc > 1)
       {
	if (*argv[1] != '-')
	  {
	   *argc -= 1;
	   *av = ++argv;
	   return *argv;
	  }
       }
     CMDLineBadArg("missing argument to", *argv);
     break;
   } /* endswitch */
 return NULL;
}


/* Simple look up a string in a table of strings...
 * then return value from value.  Will now allow partial (but unique) entry!
 * The major problem here is that -1 is the error flag...
 */
long CMDLineFindOptionArg(char *lookfor, CLOptTable *table)
{
 int tlen;
 int match_count = 0;
 long return_val = -1;
 int llen = Arena_StrLen(lookfor);

 if (lookfor && table && llen > 0)
   {
    while (table->name != NULL)
      {
       tlen = Arena_StrLen(table->name);
       if (tlen >= llen)
	 {
	  if (strncmp(lookfor, table->name, llen) == 0)
	    {
	     return_val = table->val;
	     if (llen == tlen) return return_val;
	     match_count++;
	    }
	 }
       table++;
      }
    if (match_count > 1) return_val = -1;
   }

 if (return_val == -1) CMDLineBadArg("invalid option argument", lookfor);
 return return_val;
}


/* MAIN Command line argument processing... AND arenarc files (2 of them)!
 * Many defaults already ASSUMED, or #defined may be overridden here!
 * To add new options... Add a new value to the CLOpt enum in cmdline.h
 * Then,  alter CMDLineSwitchDefs[] structure (also found in cmdline.h)
 * to include your new option.  Then, come here and add code for your
 * option in the large SWITCH below!
 */
Bool ArenaProcessCommandLine(int argc, char **argv, int *x, int *y)
{
 CLSwitch *pCLS = NULL;
 char *sT = NULL;
 int i;

#ifdef ARENA_DEBUG
 char Iam[] = "ArenaProcessCommandLine";
#endif


 /* Allocate our version of the char *argv[] stuff,
  * and put the original argv[0] in it!
  */
 ARENA_argv = (char **)Arena_MAlloc(ARENA_max_argc * sizeof(char *), True);
 ARENA_argv[0] = argv[0];
 ARENA_argc = 1;

 /* Now read the initilization files... which will add more stuff to
  * OUR version of argv[].
  */
 if (ReadArenaRC(ARENA_RC_DATA_LOCATION"/arenarc")) return EXIT_FAILURE;

 StrAllocCopy(sT, ALibHomePath());
 StrAllocCat(sT, "/.");
 StrAllocCat(sT, Application);
 StrAllocCat(sT, "/arenarc");
 if (ReadArenaRC(sT)) return EXIT_FAILURE;
 Free(sT);

 /* NOW, add the args from the user command line to the end of OUR argv[] */
 if ((argc + ARENA_argc) >= ARENA_max_argc)
   {
    Arena_PrintError(_("Too many options specified arenarc + commandline!\n"));
    return EXIT_FAILURE;
   }
 i = 1;
 while (i < argc && ARENA_argc < ARENA_max_argc)
    ARENA_argv[ARENA_argc++] = argv[i++];


 /* OK, now process OUR VERSION of the command line */
 argc = ARENA_argc;
 argv = ARENA_argv;

 while (argc > 1)
   {
#ifdef ARENA_DEBUG
     if (OPTION_TRACE)
       Arena_TracePrint(Iam, " argv[1] = \"%s\"\n",
			argv[1] ? argv[1] : "(null)");
#endif

    /* Try and find the next argument in the list of valid switches          */
    pCLS = CMDLineFindSwitch(argv[1]);
    if (pCLS == NULL)
      {
       CMDLineBadArg(NULL, argv[1]);
       return EXIT_FAILURE;
       break;
      }

    /* Found a match... so reduce argument counter and move pointer to next  */
    if (pCLS->option != CLO_UNKNOWN)
      {
       --argc;
       ++argv;
      }

    /* Process each switch separately!
     * We will warn if any value in the structures is NOT workable!
     */
    switch (pCLS->option)
      {
      default:
	Arena_PrintError(_("BUG! Bad switch value for `%s'\n"
			 "\tPlease contact %s\n"),
			 pCLS->name, DEVELOPERS_MAILTO);
	break;

      case CLO_USAGE:
	{
	 int col = 14;
	 CLSwitch *pcls = CMDLSwitchDefs;
	 printf("Usage: arena [");
	 while (pcls != NULL && pcls->name != NULL)
	   {
	     if (pcls->text) /* option text supplied! */
	       {
		col += (Arena_StrLen(pcls->text) + 1);
		if (col > 70)
		  {
		    printf("\n              ");
		    col = Arena_StrLen(pcls->text) + 14;
		  }
		printf("%s ", pcls->text);
	       }
	     pcls++;
	   }
	printf(" URL ]\nTry `arena --help' for more information.\n");
	}
	return EXIT_FAILURE;
	break;

      case CLO_HELP:
	{
	 int col = 5;
	 CLSwitch *pcls = CMDLSwitchDefs;
	 printf("Options: " BANNER"-"ARENA_VERSION"\n     ");
	 while (pcls != NULL && pcls->name != NULL)
	   {
	    if (pcls->text)                         /* option text supplied! */
	       {
		col += (Arena_StrLen(pcls->text) + 1);
		if (col > 75)
		  {
		    printf("\n     ");
		    col = Arena_StrLen(pcls->text) + 5;
		  }
		printf("%s ", pcls->text);
	       }
	    if (pcls->help != NULL)
	      {
	       printf("\n          %s\n     ", pcls->help);
	       col = 5;
	      }
	    pcls++;
	   }
	}
	return EXIT_FAILURE;
	break;

      case CLO_VERSION:
	printf("arena --version "ARENA_VERSION"\n");
	return EXIT_FAILURE;
	break;

      case CLO_UNKNOWN:
	Arena_PrintError(_("BUG! Bad switch value for `%s'\n"
			 "\tPlease contact %s\n"),
			 pCLS->name, DEVELOPERS_MAILTO);
	return EXIT_FAILURE;
	break;

      case CLO_FONTSIZE:
	{
	 CLOptTable tbl[] = { { "large", 1 },
			      { "giant", 2 },
			      { NULL,    1 } };
	 sT = CMDLineNextArg(pCLS, &argc, &argv);
	 if (sT) fontsize = CMDLineFindOptionArg(sT, (CLOptTable *)tbl);
	 if (!sT || fontsize == -1) return EXIT_FAILURE;
	}
	break;

      case CLO_LARGE:
	fontsize = 1;
	break;

      case CLO_GIANT:
	fontsize = 2;
	break;

      case CLO_COLOUR:
	{
	 CLOptTable tbl[] = { { "true",   COLOUR888 },
			      { "dither", COLOUR232 },
			      { "mono",   MONO },
			      { "grey",   GREY4 },
			      { "gray",   GREY4 },
			      { NULL,     COLOUR888 } };
	 sT = CMDLineNextArg(pCLS, &argc, &argv);
	 if (sT) ColourStyle = CMDLineFindOptionArg(sT, (CLOptTable *)tbl);
	 if (!sT || ColourStyle == -1) return EXIT_FAILURE;
	}
	break;

      case CLO_MONO:
	ColourStyle = MONO;
	break;

      case CLO_GREY:
	ColourStyle = GREY4;
	break;

      case CLO_COLOUR888:
	ColourStyle = COLOUR888;
	break;

      case CLO_COLOUR232:
	ColourStyle = COLOUR232;
	break;

      case CLO_PLAIN:
	UsePaper = False;
	break;

      case CLO_NOCACHE:
	Free(CacheRoot);
	Caching = False;
	break;

      case CLO_NOMAILCAP:
	UseMailcap = False;
	break;

      case CLO_NOSTYLE:
	NoStyle = True;
	break;

      case CLO_STYLE:
	UserSpecifiedStyle = CMDLineNextArg(pCLS, &argc, &argv);
	if (UserSpecifiedStyle == NULL) return EXIT_FAILURE;
	if (!*UserSpecifiedStyle)     /* user specified --style-sheet ""     */
	  {                           /* make same as --no-style             */
	   UserSpecifiedStyle = NULL;
	   NoStyle = True;
	  }
	Arena_PrintError(_("Command Line style sheets NOT YET implemented\n"));
	break;

      case CLO_CM:
	OwnColourMap = INSTALL;
	break;

      case CLO_NOCM:
	OwnColourMap = INHIBIT;
	break;

      case CLO_DISPLAY:
	display_name = CMDLineNextArg(pCLS, &argc, &argv);
	if (!display_name || *display_name == 0) return EXIT_FAILURE;
	break;

	/* howcome 5/10/94: added cache command line option                  */
	/* glk forced access and allowed NULL (e.g. "") same as -nocache     */
      case CLO_CACHE:
	Caching = True;
	sT = CMDLineNextArg(pCLS, &argc, &argv);
	if (sT == NULL) return EXIT_FAILURE;

	StrAllocCopy(CacheRoot, sT);
	if (*sT == '~')                  /* put in the HOMEY                 */
	  {
	   StrAllocCopy(CacheRoot, ALibHomePath());
	   StrAllocCat(CacheRoot, (sT+2));
	  }

	if (!CacheRoot || *CacheRoot == 0 || access(CacheRoot, R_OK | W_OK))
	  {
	   Free(CacheRoot);
	   Caching = False;
	  }
#ifdef ARENA_DEBUG
	if (OPTION_TRACE)
	  {
	   Arena_TracePrint(Iam, " cache dir = \"%s\"\n", 
			    CacheRoot ? CacheRoot : "(null)");
	   Arena_TracePrint(Iam, " cache flg = %d\n", Caching);
	  }
#endif
	break;

      case CLO_EDITOR:    /* howcome 10/10/94: added editor cmd line option  */
	Editor = CMDLineNextArg(pCLS, &argc, &argv);
	if (!Editor) return EXIT_FAILURE;
	if (*Editor == 0) Editor = NULL;
	break;

      case CLO_BADFLAGS:    /* howcome 10/10/94: added badflags line option */
	sT = CMDLineNextArg(pCLS, &argc, &argv);
	if (!sT) return EXIT_FAILURE;
	Badflags = atoi(sT);                           /* FIXME */
	Badflags = max(min(Badflags,50),0);
	break;

      case CLO_ICON:    /* howcome 13/10/94: added icon command line option */
	Icon = CMDLineNextArg(pCLS, &argc, &argv);
	if (!Icon) return EXIT_FAILURE;
	if (*Icon == 0) Icon = NULL;
	break;

      case CLO_EMAIL:
	EmailAddr = CMDLineNextArg(pCLS, &argc, &argv);
	if (!EmailAddr) return EXIT_FAILURE;
	if (!*EmailAddr) EmailAddr = NULL;
	break;

      case CLO_PRINTER:    /* howcome 10/10/94: added command line option */
	Printer = CMDLineNextArg(pCLS, &argc, &argv);
	if (!Printer) return EXIT_FAILURE;
	if (!*Printer) Printer = NULL;
	break;

      case CLO_GAMMA:    /* howcome 7/11/94: added command line option */
	sT = CMDLineNextArg(pCLS, &argc, &argv);
	if (!sT) return EXIT_FAILURE;
	Gamma = (double) atof(sT);            /* FIXME */
	break;


      case CLO_DEBUG:
#ifdef ARENA_DEBUG
#  ifdef ARENA_TRACE_MASK
	debug = ARENA_TRACE_MASK;
#   else
	debug = 0xffffffff;
#  endif
# else
	Arena_PrintError(_("Sorry, this binary has no debug stuff.\n"));
#endif
	/* FALL through to the client-trace stuff.  This allows optional
	 * argument to -debug and --debug
	 */

      case CLO_CT:
	sT = CMDLineNextArg(pCLS, &argc, &argv);
	if (sT)
	  if (*sT)
	    {
#ifdef ARENA_DEBUG
	     debug = 0;
	     if (isdigit(sT[0]))
	       {
	       /* Allow dec, octal, hex input...
		* NOT perfect, but a little easier than before
		*/
		int istat;
		int ilen = strlen(sT);
		char *fmt = " %lu ";
		if (sT[0] == '0' && ilen > 1)
		  {
		   fmt = " %lo ";
		   if (sT[1] == 'x') fmt = " %lx ";
		  }

		istat = sscanf(sT, fmt, &debug);
		if (istat != 1 || (debug == 0 && ilen > 0))
		  Arena_TracePrint(Iam,
				   " Invalid Client Trace value %s\n", sT);
	       }
	      else
	       {
	       /* Let's allow --debug "SPAWN|X|VERBOSE"
		* NOTE: this will require the quoted string...
		*/
		char *b;
		long lTemp = 0;

		CLOptTable tbl[] = {
		  { "AUTHENTICATION", AUTHENTICATION_TRACE_BITMASK },
		  { "BADFLAG",               BADFLAG_TRACE_BITMASK },
		  { "BOOKMARKS",           BOOKMARKS_TRACE_BITMASK },
		  { "BRIDGE",                 BRIDGE_TRACE_BITMASK },
		  { "COLOUR",                 COLOUR_TRACE_BITMASK },
		  { "CONNECT",               CONNECT_TRACE_BITMASK },
		  { "FONT",                     FONT_TRACE_BITMASK },
		  { "FORM",                     FORM_TRACE_BITMASK },
		  { "HISTORY",               HISTORY_TRACE_BITMASK },
		  { "IMAGE",                   IMAGE_TRACE_BITMASK },
		  { "MAILCAP",               MAILCAP_TRACE_BITMASK },
		  { "MAILTO",                 MAILTO_TRACE_BITMASK },
		  { "MENU",                     MENU_TRACE_BITMASK },
		  { "OPTION",                 OPTION_TRACE_BITMASK },
		  { "PAINTSTREAM",       PAINTSTREAM_TRACE_BITMASK },
		  { "PARSEHTML",           PARSEHTML_TRACE_BITMASK },
		  { "REGISTER",             REGISTER_TRACE_BITMASK },
		  { "SCROLL",                 SCROLL_TRACE_BITMASK },
		  { "STATUS",                 STATUS_TRACE_BITMASK },
		  { "STYLE",                   STYLE_TRACE_BITMASK },
		  { "TABLE",                   TABLE_TRACE_BITMASK },
		  { "TAG",                       TAG_TRACE_BITMASK },
		  { "X",                           X_TRACE_BITMASK },
		  { "SPAWN",                   SPAWN_TRACE_BITMASK },
		  { "VERBOSE",               VERBOSE_TRACE_BITMASK },
		  { NULL,                            0             } };

		b = strtok(sT, "|");
		while (b != NULL)
		  {
		   lTemp = CMDLineFindOptionArg(b, (CLOptTable *)tbl);
		   if (lTemp == -1) return EXIT_FAILURE;
		   debug |= lTemp;
		   b = strtok(NULL, "|");
		  }
	       }
	     /* end of if (isdigit())... else... */

	     if (ANY_TRACE)
	       Arena_TracePrint(Iam, " Client trace = 0x%lx.\n", debug);
# else
	     Arena_PrintError(_("Sorry, this binary has no trace stuff.\n"));
#endif
	     break;
	    }
	/* End ``if (sT) if (*sT)'' */

	if (pCLS->option == CLO_CT) return EXIT_FAILURE;

	break;


      case CLO_LENS:    /* howcome 9/5/95: added command line option */
	sT = CMDLineNextArg(pCLS, &argc, &argv);
	if (sT == NULL) return EXIT_FAILURE;
	lens_factor = (int) atof(sT);
	fontsize = (int) (lens_factor + 0.5);
	break;

      case CLO_SBARTHICK:
	sT = CMDLineNextArg(pCLS, &argc, &argv);
	if (sT == NULL) return EXIT_FAILURE;
	sbar_width = atoi(sT);
	sbar_width = max(min(sbar_width, 50), 0);
	break;

      case CLO_LT:    /* howcome 7/11/94: added command line option */
	sT = CMDLineNextArg(pCLS, &argc, &argv);
	if (sT == NULL)
	  {
#  ifdef LIBWWW_TRACE_MASK
	   library_trace = LIBWWW_TRACE_MASK;
#   else
	   library_trace = 255;
#  endif
	  }
	else
	  {
	   char *p = sT;
#ifndef HAVE_STRNCPY
	   char c = 0;
#endif
	   library_trace = 0;
#ifdef HAVE_STRNCPY
	   for(; *p; p++)
# else
	   for(c = 0; *p; p++, c++)
#endif
	     {
	      if (*p == 'v')
		{
		 library_trace = SHOW_ALL_TRACE;
		 break;
		}
	      /* End if */
	     }
	   /* End for */
	  }
#ifdef ARENA_DEBUG
#  ifdef HAVE_STRNDUP
	LibWWW_traceflags = strndup(((library_trace == SHOW_ALL_TRACE) ?
				     LIBWWW_TRACE_FLAGS : *argv),
				    LIBWWW_TRACE_FLAGS_LENGTH);
#   else
	LibWWW_traceflags = strdup(((library_trace == SHOW_ALL_TRACE) ?
				    LIBWWW_TRACE_FLAGS : *argv));
#  endif
#endif
	break;


      case CLO_GEOMETRY:    /* pdd 31/10/95: added crippled window position  */
	sT = CMDLineNextArg(pCLS, &argc, &argv);
	if (!sT) return EXIT_FAILURE;
	sscanf(sT,"+%d+%d", x, y);
	/* this part may not be necessary... but can't hurt */
	if (*x < 0 || *x >= 2048 || *y < 0 || *y >= 1536)
	  {
	   CMDLineBadArg("`-geometry' only supports", "+X+Y");
	   *x = 0;
	   *y = 0;
	   return EXIT_FAILURE;
	  }
	break;

#ifdef ARENA_FLYBY_HELP
      case CLO_FLYBY:
	{
	 CLOptTable tbl[] = { { "none",    ARENA_FLYBY_NONE    },
			      { "hint",    ARENA_FLYBY_HINT    },
			      { "terse",   ARENA_FLYBY_TERSE   },
			      { "verbose", ARENA_FLYBY_VERBOSE },
			      { NULL,      ARENA_FLYBY_NONE    } };
	 sT = CMDLineNextArg(pCLS, &argc, &argv);
	 if (sT) FlyByHints = CMDLineFindOptionArg(sT, (CLOptTable *)tbl);
	 if (!sT || FlyByHints == -1) return EXIT_FAILURE;
	}
	break;

      case CLO_NOFLYBY:
	sT = CMDLineNextArg(pCLS, &argc, &argv);
	if (!sT) return EXIT_FAILURE;
	FlyByHints = ARENA_FLYBY_NONE;
	break;
#endif

      case CLO_HISTORY:
	{
	 int mode = -1;
	 CLOptTable tbl[] = { { "none",    HISTORY_TYPE_NONE               },
			      { "chain",   HISTORY_TYPE_HYPERCHAIN         },
			      { "delete",  HISTORY_TYPE_DELETE             },
			      { "semi",    HISTORY_TYPE_SEMI_HYPERCHAIN    },
			      { NULL,      HISTORY_TYPE_NONE               } };
	 sT = CMDLineNextArg(pCLS, &argc, &argv);
	 if (sT) mode = CMDLineFindOptionArg(sT, (CLOptTable *)tbl);
	 if (!sT || mode == -1) return EXIT_FAILURE;
	 HistoryMode(mode);
	}
	break;

      case CLO_URL:         /* pdd 31/10/95: treat first argument not        */
	startwith = *argv;  /* starting with a "-" as a new starting URL     */
	break;
      } /* end of switch (pCLS->option) */
   } /* end of while (argc > 1)   */ 


 return EXIT_SUCCESS;
}


int main(int argc, char **argv)
{
 extern Image* ArenaIconImage;

 int x = 0, y = 0;               /* window position */
 unsigned int border_width = 2;  /* border four pixels wide */
 char *window_name = BANNER;
 char *icon_name = BANNER;
 int best_depth;
 /* unsigned long *pixels;  janet 21/07/95:  not used */
 unsigned int class;
 unsigned long valuemask;
 XSetWindowAttributes attributes;
 XClassHint class_hints;
 mode_t cmask;
#ifdef ARENA_DEBUG
 char Iam[] = "main";
#endif


#ifndef ARENA_NO_LOCALE
#  ifdef HAVE_SETLOCALE
#    ifdef ARENA_DEBUG
 if (setlocale(LC_CTYPE, "") == NULL)
   Arena_TracePrint(Iam, " Failed to set CTYPE locale.\n");
#     else
 setlocale(LC_CTYPE, "");
#    endif
#    ifdef HAVE_LIBINTL
#    ifdef ARENA_DEBUG
 if (setlocale(LC_MESSAGES, "") == NULL)
   Arena_TracePrint(Iam, " Failed to set MESSAGES locale.\n");
#     else
 setlocale(LC_MESSAGES, "");
#    endif
 bindtextdomain(ARENA_ApplicationName, ARENA_LOCALE_DIR);
 textdomain(ARENA_ApplicationName);
#    endif
#  endif
#endif


#ifdef SIGWAITING_IGN
 /* howcome: solaris binaries receive this signal, but why? */
 signal(SIGWAITING, SIG_IGN);
#endif
 signal(SIGABRT, SIG_IGN);
 signal(SIGPIPE, SIG_IGN);
#if !defined __QNX__
 /* howcome 4/10/94: trap C-C so that we can delete cache */
 signal(SIGINT, Exit);
#endif
 signal(SIGIOT, CatchCore);
#ifdef CatchCoreIfSegmentationFault
 signal(SIGSEGV, CatchCore);
#endif
#ifdef ARENA_FLYBY_HELP
 signal(SIGALRM, ArenaSIGALRMHandler);
#endif

 /* Set up some of the internal flags and storage areas.
  * I think these are all hardwired here, and are not altered willy nilly.
  */
 cmask = umask(GeneralUMASK);
 stack = main_stack;

 {
  char* theHome = NULL;

  if ((ArenaUser = GetArenaUser()) && (theHome = ALibHomePath()))
    {
     Free(theHome);
    }
   else
    {
#ifdef ARENA_DEBUG
     Arena_TracePrint(Iam, " Arena needs some basic environment!\n");
# else
     Arena_PrintError(_("Arena needs some basic environment!\n"));
#endif
     Free(ArenaUser);
     Free(theHome);
     Exit(1);
    }
 }

 /* Define our "default" internal setup here.  Some or all of these may be
  * altered later from the #defines coming from arena.h, or via the runtime
  * switches which we will process below.
  */
 UsePaper = True;
 ColourStyle = COLOUR888;
 Application = ARENA_ApplicationName;
 Editor = getenv("EDITOR");     /* howcome 16/10/94: */


 /* Go get ALL the command line arguments
  * This routine is returning 99% of its results via the global
  * pointers at the top of this module!  How this works:
  * 1) Read ARENA_RC_DATA_LOCATION/arenarc
  *    Adding switches to OUR version of argv[]
  * 2) Read ~/.Arena/arenarc... doing the same.
  * 3) Add the user's command line args to the end of OUR argv[].
  * 4) Proccess command line! 
  */
 if (ArenaProcessCommandLine(argc, argv, &x, &y)) Exit(1);
 Free(ARENA_argv); /* Get rid of OUR argv[] stuff.  Just the vector of ptrs. */
 ARENA_argc = 0;   /* This leaves data behind.. Lots of global ptrs to it.   */

 /* initialise ISO character entity definitions */
 InitEntities();

 /* connect to X server */
 if ( (display = XOpenDisplay(display_name)) == NULL )
   {
#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
    (void) Arena_TracePrint(Iam,
			    " cannot connect to X server %s\n",
			    XDisplayName(display_name));
# else
    (void) Arena_PrintError(_("Cannot connect to X server %s\n"),
			    XDisplayName(display_name));
#endif
    Exit(-1);
   }

 {
  char *dpy_name, *tmp1, *tmp2;
  struct utsname os_info;

  dpy_name = DisplayString(display);
  uname(&os_info);

  tmp1 = (char *)Arena_MAlloc(Arena_StrLen(dpy_name)+1, False);
  strcpy(tmp1, dpy_name);
  for (tmp2 = tmp1; *tmp2; tmp2++) if (*tmp2 == '.') *tmp2 = 0;

  if (!strcasecmp(tmp1, os_info.nodename) || (*tmp1 == ':')) /* it is local */
    {
     if (!strcasecmp(os_info.sysname, "IRIX"))
       {
	FILE *gammafile;

	gammafile = fopen("/etc/config/system.glGammaVal","r");

	if (gammafile)
	  {
	   float local_gamma;

	   fscanf(gammafile, "%f", &local_gamma);

	   Gamma = Gamma / local_gamma;
#ifdef ARENA_DEBUG
	   if(X_TRACE)
	     Arena_TracePrint(Iam,
			      " GAMMA = %f (local was %f)\n",
			      Gamma, local_gamma);
#endif
	  }
       }
    }
  Free(tmp1);
 }


/* for debuging  - disable buffering of output queue */
#if 0
#  ifdef ARENA_DEBUG	/* QingLong.08-12-96 */
 XSynchronize(display, 1); /* 0 enable, 1 disable */
#   else
 XSynchronize(display, 0); /* 0 enable, 1 disable */
#  endif
#endif

/*    gateway = DetermineValue("gateway", GATEWAY_HOST); */
 HelpURL = DetermineValue("HelpURL", HELP_URL);

 {
  Bool StartingURLset = False;


 /*
  * If the user has defined the starting URL, do nothing!
  */
  if (startwith) if (*startwith) StartingURLset = True;
  if (!StartingURLset)
    {
     startwith = getenv(ARENA_HOME_URL_ENV_VARIABLE);
     if (startwith) if (*startwith) StartingURLset = True;

     if (!StartingURLset)
       startwith = DetermineValue("DefaultURL", DEFAULT_URL);
    }
 }

 /* get screen size from display structure macro */
 screen = DefaultScreen(display);
 depth = DisplayPlanes(display, screen);

 if (depth == 1) ColourStyle = MONO;

 visual = BestVisual(TrueColor, &best_depth);
 if (best_depth == 24 || best_depth == 16)
   {
   /* From Scott Nelson <snelson@canopus.llnl.gov> 24bit colour: 12/5/94 */
   /* visual = BestVisual(DirectColor, &best_depth); */ 
    visual = BestVisual(TrueColor, &best_depth); 
    CalcPixelShift();
   }
  else
   {
    visual = BestVisual(DirectColor, &best_depth);
   }

 if (ColourStyle != COLOUR888 || best_depth <= depth)
   {
    colourmap = DefaultColormap(display, screen);
    visual = DefaultVisual(display, screen);

    if ((ColourStyle = InitImaging(ColourStyle)) == MONO) UsePaper = False;
   }
  else
   {
    InitImaging(ColourStyle);
    depth = best_depth;
    colourmap = XCreateColormap(display, RootWindow(display, screen),
			       visual, AllocNone);
   }

 display_width = DisplayWidth(display, screen);
 display_height = DisplayHeight(display, screen);
 display_pt2px =  (25.4 / 72.0) * 
   (((double)DisplayWidth(display, screen)) / ((double)DisplayWidthMM(display, screen)));

 GetResources();  /* load font and colour resources */

 if (ColourStyle == MONO) statusColour = windowColour;

 /* size widow with enough room for text */

 charWidth  = XTextWidth(Fonts[IDX_NORMALFONT], "ABCabc", 6)/6; 
 charHeight = SPACING(Fonts[IDX_NORMALFONT]);

 win_width  = min((88 * charWidth), (display_width * 0.9));
 win_height = min((33 * charHeight + 7), (display_height));

 class = InputOutput;    /* window class*/
 valuemask = (CWColormap | CWBackPixel | CWBitGravity | CWEventMask |
              CWBackingPlanes | CWBackingStore);
 attributes.colormap         = colourmap;
 attributes.background_pixel = windowColour;
 attributes.bit_gravity      = ForgetGravity;
 attributes.event_mask       = (ExposureMask |
				KeyPressMask | KeyReleaseMask |
				ButtonPressMask | ButtonReleaseMask |
				Button1MotionMask |
				StructureNotifyMask | PropertyChangeMask);
 attributes.backing_planes   = 0;
 attributes.backing_store    = NotUseful;

 /*
  * Create main Arena window.
  */
 win = XCreateWindow(display, RootWindow(display, screen),
		     x, y, win_width, win_height, border_width,
		     depth, class, visual, valuemask, &attributes);

 class_hints.res_name  = (char*)Arena_CAlloc(Arena_StrLen("Browser") + 1,
					     sizeof(char),
					     False);
 strcpy(class_hints.res_name, "Browser");
 class_hints.res_class = (char*)Arena_CAlloc(Arena_StrLen(BANNER) + 1,
					     sizeof(char),
					     False);
 strcpy(class_hints.res_class, BANNER);
 XSetClassHint(display, win, &class_hints);


 /* inter client comm */
 ARENA_LOCATION = XInternAtom(display, "ARENA_LOCATION", False);
 ARENA_COMMAND  = XInternAtom(display, "ARENA_COMMAND" , False);


 /* initialize size hint property for window manager */
 size_hints.flags = PPosition | PSize | PMinSize;
 size_hints.x = x;
 size_hints.y = y;
 size_hints.width  = win_width;
 size_hints.height = win_height;
 size_hints.min_width  = 100;
 size_hints.min_height = 150;

 /*
  * Initialize all the context fields, but `style' (needs `paper').
  */
 context = NewContext();
 context->history = HTList_new();
#if 1	/* 16-04-97 Juergen Ruehle <jr@falbala.informatik.uni-kiel.de> */
 context->current_history = NewHistory();
 context->history_pos = -1;
#endif
 context->conversions = HTList_new();
 HTFormat_setConversion(context->conversions);
 context->encodings = HTList_new();

 ArenaBridgeInit();

 /*
  * Before we can call XSetStandardProperties we need the icon.
  * Befor we can get the icon, we need to initialize the library.
  */
 /*
  * Up to now, we should be using Application for the app name.
  * HTLib_appName() has not been inited yet.
  * I did NOT want to alter HTLib.c within libwww to fix it
  * so we could use HTLib_appName().
  * Also, libEntry() will create a correct UserProfile for us,
  * set up our temporary files path, and CacheRoot will be initialized.
  */

 libEntry(); /* howcome: library init */

 /*
  * Now it's time to initialize BookMarx stuff constants.
  */
 if (!BMinit()) Exit(-1);

 LoadIcon();


 /*
  * Fill in window manager hints structure values.
  */
 WinMan_hints = XAllocWMHints();
 WinMan_hints->flags           = InputHint;
 WinMan_hints->input           = True;
 if (ArenaIconImage)
   {
    WinMan_hints->flags       |= (IconPixmapHint | IconMaskHint);
    WinMan_hints->icon_pixmap  = ArenaIconImage->pixmap;
    WinMan_hints->icon_mask    = ArenaIconImage->mask;
   }

 /*
  * Set properties for window manager. (always before mapping)
  */
 XSetStandardProperties(display,
			win, window_name,
			icon_name,
			ArenaIconImage ?
	        ArenaIconImage->pixmap : default_pixmap,
			argv, argc, &size_hints);

 XSetWMHints(display, win, WinMan_hints);

#ifdef HAVE_XINTERNATOMS
 XInternAtoms(display,
	      Arena_WM_xa_names_list, Arena_xa_WM_protocols_N, True,
	      Arena_WM_xa_list);
# else
 {
  int ai;

  for (ai = 0; ai < Arena_xa_WM_protocols_N; ai++)
    Arena_WM_xa_list[ai] = XInternAtom(display,
				       Arena_WM_xa_names_list[ai],
				       True);
 }
#endif
 XSetWMProtocols(display, win, Arena_WM_xa_list, Arena_xa_WM_protocols_N - 1);


 /* create hourglass cursor */
 hourglass = XCreateFontCursor(display, XC_watch);


         SetToolBarFont(Fonts[IDX_LABELFONT]);
 SetArenaStatusLineFont(Fonts[IDX_LABELFONT]);

   ToolBarInit(win);        /* also init's status bar geometry              */
    StatusInit(ToolBarWin); /* must be called after toolbar initialization  */
      IconInit(win);        /* must be called after toolbar and status init */
 ScrollBarInit(win);        /* must be called after toolbar and status init */
   DocDispInit(win);        /* must go after toolbar, status and scrollbars */

#ifdef POPUP
 initPopup(Fonts[IDX_LABELFONT]);
#endif

#ifdef ARENA_FLYBY_HELP
 /*
  * Select events wanted.
  * Use the simpler set of event if the FLYBY stuff not compiled in
  * OR if NONE is specified for hinting.
  */
 if (FlyByHints != ARENA_FLYBY_NONE)
   XSelectInput(display, ToolBarWin, FlyByEventMask);
#endif


 UsePaper = MakePaper(); /* create gc_fill for textured paper */

 /*
  * Now we have `paper' and thus can initialize context style to it's default.
  */
 context->style = StyleGetInit();

 MakeIcons(depth);       /* create images for icons */

 /* create pixmap for testing initial <img> implementation */

 /* Map Display Window */
 XMapWindow(display, win);
 XMapWindow(display, IconWin);
 XMapWindow(display, ToolBarWin);
 XMapWindow(display, StatusWin);
 XMapWindow(display, ScrollXBarWin);
 XMapWindow(display, ScrollYBarWin);
 XMapWindow(display, DocDispWin);

 /* get buffer with named file in it */
 hdrlen = 0;
 buffer = NULL;

 libEventLoop(ConnectionNumber(display), startwith);

    /* This point is unreachable! */
#ifdef ARENA_DEBUG
 Arena_TracePrint(Iam,
		  " -----!     !  !\n");
# else
 Arena_PrintError(" -----!     !  !\n");
#endif

 return -1;
}
