/* -*-Mode: C;-*-
 * XDELTA - RCS replacement and delta generator
 * Copyright (C) 1997  Josh MacDonald
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: xdelta.h 1.4.1.8.1.3 Mon, 13 Oct 1997 19:22:14 -0700 jmacd $
 */

#ifndef _XDELTA_H_
#define _XDELTA_H_

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <errno.h>
#include <time.h>

#undef DEBUG_HASH
#undef DEBUG_XD
#undef DEBUG_PARAMS
#undef DEBUG_CKSUM
#undef DEBUG_INST
#undef DEBUG_GROW
#undef DEBUG_XDP

#undef assert
#define assert(arg) (void)0

#ifndef _GNUC_
#define inline
#endif

#define XDELTA_PREFIX "%XDELTA%"
#define XDELTA_PREFIX_LEN (strlen(XDELTA_PREFIX))

#ifndef g_new
#include "fakeglib.h"
#endif
#include "gdbm.h"

#ifdef _WIN32
#define FOPEN_READ_ARG "rb"
#define FOPEN_WRITE_ARG "wb"
#else
#define FOPEN_READ_ARG "r"
#define FOPEN_WRITE_ARG "w"
#endif

typedef struct _CKList          CKList;
typedef struct _ChecksumArray   ChecksumArray;
typedef struct _ChecksumTable   ChecksumTable;
typedef struct _Checksum        Checksum;
typedef struct _MatchQuery      MatchQuery;
typedef struct _Match           Match;
typedef struct _MatchLevel      MatchLevel;
typedef enum   _XdeltaInstructionType XdeltaInstructionType;
typedef struct _XdeltaInstruction XdeltaInstruction;
typedef struct _FromSegment     FromSegment;
typedef struct _MappedFile      MappedFile;
typedef struct _XdFile          XdFile;

struct _XdFile {
  GDBM_FILE dbf;
  gint versions;
};

struct _MappedFile {
  guint8* seg;
  gint    len;
#ifndef _WIN32
  gint    fd;
#endif
};

struct _ChecksumArray {
  gint ck_count; /* number of elts in cksums. */
  gint ck_width; /* number of atoms each cksum represents. */
  Checksum* cksums; /* array of cksums. */

  FromSegment* from;
};

struct _Checksum {
  guint16 low;
  guint16 high;
};

struct _CKList {
  CKList* next;
  Checksum* cksum;
  ChecksumArray* array;
};

struct _FromSegment {
  const guint8* real_seg;
  gint          real_len;

  /* Don't touch below here. */
  ChecksumArray* ckarray;

  gint* multibyte_map;
  gint multibyte_alloc;

  gint segment_index;

  union {
    const guint8  *ob;
    const guint16 *tb;
  } seg;

  gint len;
};

struct _ChecksumTable {
  CKList         **buckets;
  guint            mask;
  gint             bits;
  GMemChunk       *cklist_chunk;
#ifdef DEBUG_HASH
  gint   hash_coll;
  gint   hash_items;
  guint8 high_checksums [1<<16];
  guint8 low_checksums  [1<<16];
#endif
};

struct _Match {
  gint from_real_offset;
  gint from_segment_index;
  gint to_low;
  gint to_high;
  gint low_neg_leftover;  /* partial match. */
  gint high_pos_leftover; /* partial match. */
  gint length;
};

struct _MatchLevel {
  /* Transient data (each call to lookup_checksum). */
  Match     *match;

  /* Permanent storage */
  GMemChunk *chunk;

#ifdef DEBUG_HASH
  gint       false_alarms;
  gint       useless_matches;
  gint       matches_width;
#endif

  /* Match data. */
  GSList    *to_intervals;
};

enum _XdeltaInstructionType {
  CopyInstruction   = 'C', /* C LENGTH, INDEX */
  InsertInstruction = 'I'  /* I LENGTH, TEXT */
};

struct _XdeltaInstruction {
  XdeltaInstructionType type;
  gint from;
  gint from_seg;
  gint to;
};

struct _MatchQuery {
  /* 6 things you set. */

  /* 1. If you want a segmented or aligned delta, set this to a function
   * which will return true if it should break at this segment index. */
  gint        (*break_segment) (MatchQuery* query,
				const guint8* seg,
				gint index,
				gint length);
  /* 2-3. Actual TO data. */
  const guint8 *real_to_seg;
  gint          real_to_len;

  /* 4. Base 2 log of the segment len to consider. */
  gint          size;

  /* 5-6. From segments and count. */
  FromSegment  **from;
  gint           from_count;

  /* Return value. */
  GSList *xdelta;

  /* Things you don't set. */
  gint* multibyte_to_map;
  gint multibyte_to_alloc;

  gint        (*grow_match) (MatchQuery *query,
			     CKList     *cksum,
			     gint        segment_len,
			     gint        lower_to_limit,
			     gint        upper_to_limit,
			     gint        to_index,
			     Match      *match);

  MatchLevel* (*find_matches) (MatchQuery *query,
			       gint  width,
			       gint  low_to_index,
			       gint  high_to_index);

  gint (* index_to_real_offset) (gint index, gint* map);

  ChecksumTable *table;
  GMemChunk     *xdelta_chunk;

  gint current_output_point;

  /* To Segment. */
  union {
    const guint8  *ob;
    const guint16 *tb;
  }    to_seg;
  gint to_len;
};

			    /* CHASH.C */

/* Simple Hash Table Stuff. */
ChecksumTable* c_hash_table_new    (gint                 bits);
void           c_hash_table_free   (ChecksumTable       *table);
gint           c_hash              (const ChecksumTable *table,
				    gint                 high,
				    gint                 low);
void           c_hash_table_insert (ChecksumTable *table,
				    Checksum      *checksum,
				    ChecksumArray *array);
CKList*        c_hash_table_lookup (const ChecksumTable *table,
				    const Checksum      *checksum);
#ifdef DEBUG_HASH
void           c_hash_stats        (ChecksumTable       *table);
#endif
gint           ilog2               (gint                 arg);
#define        ipow2(arg)          (1<<(arg))

			    /* CKSUM.C */

/* Hashing base ranges of bytes and shorts. */
extern const guint16 single_hash[256];
void init_checksum_1 (const guint8  *buf,
		      const gint     len,
		      Checksum      *cksum);
void init_checksum_2 (const guint16 *buf,
		      const gint     len,
		      Checksum      *cksum);

			    /* GENCK.C */

/* Produce one-byte checksums on the buffer. */
ChecksumArray* generate_checksums_1 (const guint8   *segment,
				     gint            segment_len,
				     gint            cksum_width,
				     FromSegment    *from);
/* Produce two-byte checksums on the buffer. */
ChecksumArray* generate_checksums_2 (const guint16  *segment,
				     gint            segment_len,
				     gint            cksum_width,
				     FromSegment    *from);

			      /* CM.C */

void        match_level_free (MatchLevel *match_level);
MatchLevel* find_matches_1   (MatchQuery *query,
			      gint  width,
			      gint  low_to_index,
			      gint  high_to_index);
MatchLevel* find_matches_2   (MatchQuery *query,
			      gint  width,
			      gint  low_to_index,
			      gint  high_to_index);

			     /* GROW.C */

gint grow_1 (MatchQuery *query,
	     CKList     *cksum,
	     gint        segment_len,
	     gint        lower_to_limit,
	     gint        upper_to_limit,
	     gint        to_index,
	     Match      *match);
gint grow_2 (MatchQuery *query,
	     CKList     *cksum,
	     gint        segment_len,
	     gint        lower_to_limit,
	     gint        upper_to_limit,
	     gint        to_index,
	     Match      *match);

			     /* EMIT.C */
void emit_match  (MatchQuery*, Match *match);
void emit_rest   (MatchQuery*);
datum xdelta_to_bytes (MatchQuery* query);
gint  xdelta_bytes_seg_count (datum dat);
datum xpatch (datum delta, datum* segs, gint nsegs);

			     /* MAP.C */

MappedFile* map_file   (const gchar* filename);
void        unmap_file (MappedFile* map);

			    /* XDELTA.C */
GSList* xdelta (MatchQuery* query);

void        match_query_free (MatchQuery *query);
MatchQuery* match_query_new (gint (*break_segment) (MatchQuery* query,
						    const guint8* seg,
						    gint index,
						    gint length),
			     const guint8 *real_to_seg,
			     gint          real_to_len,
			     gint          size,
			     FromSegment  **from,
			     gint           from_count);
FromSegment* from_segment_new (const guint8 *real_to_seg,
			       gint          real_to_len);
void         from_segment_free ();

			     /* MISC.C */
char*       time_t_to_rfc822(time_t t);
gboolean    fs_file_exists(gchar* P);
gchar*      strip_leading_path (gchar* p);
time_t      get_utc_time_t();
void        md5_buffer (gchar* buffer, gint buflen, gchar* digest);

			      /* DB.C */
XdFile*  xd_create (gchar* name);
XdFile*  xd_open_read (gchar* name);
XdFile*  xd_open_write (gchar* name);
void     xd_close (XdFile* xd);

gchar* xd_get_date_str (XdFile* xd, gint ver);
gchar* xd_get_md5_str  (XdFile* xd, gint ver);
gchar* xd_get_md5_raw  (XdFile* xd, gint ver);
gint   xd_get_len      (XdFile* xd, gint ver);
datum  xd_get_latest   (XdFile* xd);

gboolean xd_set_date (XdFile* xd, gint ver, time_t t);
gboolean xd_set_md5  (XdFile* xd, gint ver, gchar raw_md5[16]);
gboolean xd_set_len  (XdFile* xd, gint ver, gint len);
gboolean xd_set_latest (XdFile* xd, guint8* seg, gint len);
gboolean xd_set_versions (XdFile* xd, gint versions);

datum xd_get_delta   (XdFile* xd, gint delta);
gboolean xd_append_delta (XdFile* xd, guint8* seg, gint len);

gint xd_checkin (XdFile*, gchar* file);
gint xd_checkout (XdFile*, gchar* file, gint ver);

#define MAX_PAST_DELTAS 10

#endif
