/* 
 * xwave - an interactive audio player, recorder, editor 
 * for the XWindow System
 * 
 * Copyright (C) 1996 Kai Kollmorgen
 * (kkollmor@informatik.uni-rostock.de)
 *
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 */

#include <signal.h>		   
#include <stdlib.h>
#include <unistd.h>		      
#include <stdio.h>
#include <dirent.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <X11/Intrinsic.h>	/* Intrinsics Definitions */
#include <X11/StringDefs.h>	/* Standard Name-String definitions */
#include <X11/Shell.h>     	/* Shell Definitions */

#include <X11/Xaw/Command.h>	/* Athena Command Widget */
#include <X11/Xaw/MenuButton.h>
#include <X11/Xaw/SimpleMenu.h>
#include <X11/Xaw/SmeBSB.h>

#include <X11/Xaw/Form.h>
#include <X11/Xaw/AsciiText.h>
#include <X11/Xaw/List.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/Viewport.h>

#include "types.h"
#include "xwave.h"
#include "xwave_widget.h"
#include "audio_file.h"
#include "misc.h"
#include "audio.h"
#include "graphics.h"
#include "sample_settings.h"
#include "filebrowseP.h"
#include "filebrowse.h"

#include "bitmaps/pull.xbm"

static void cancel_action(Widget w,XtPointer client_data,XtPointer call_data);
static void ok_action(Widget w, XtPointer client_data, XtPointer call_data);
static int (*browse_return) (char *fname,int type,int comp);
static void hilight_file ();
static void hilight_dir ();
static void tab_expansion();
static void clean_up (Widget popup);
static void dir_action(Widget w, XtPointer client_data, XtPointer call_data);
static void set_dir(char *directory);
static void play_it(Widget w, XtPointer client_data, XtPointer call_data);
static int strqsortcmp(char **a, char **b);
static int select_it(char *fn,char *mask);
static void set_dir_label(char *directory);
static void display_permissions(char *filename); 
static void uh_action(Widget w, XtPointer *junk, XKeyPressedEvent *event,
		      Boolean *flg);
static void free_stringlist(Widget list,String *strlist);
static void mask_menu_cb(Widget w, XtPointer client_data, XtPointer call_data);
static void filetype_cb(Widget w, XtPointer client_data, XtPointer call_data);
static void ft_return(int type,int comp);

/* application resources are defined in xwave.c */
extern AppResources app_resources;

/* MD is global in xwave.c */

extern Main_Data *MD;

/* we need some globals */

static Widget popup,form,dirlabel,text,mtext,status,wavestat,playbutton,
okbutton;
static Widget filelist=NULL,dirlist=NULL;
static char file_buf[MAX_NAMELENGTH],mask_buf[MAX_NAMELENGTH],
test_buf[MAX_NAMELENGTH];
static String *dateien;
static String *verzeich;
static int pid=-1;
static bool playing=FALSE;
static struct sigaction *sig;
static int browsetype=LOAD;
static char* last_expansion=NULL;
static int default_ftype,default_comp;
static char ft_label[20];

void play_it(Widget w, XtPointer client_data, XtPointer call_data)
{
   char filename[MAX_NAMELENGTH];
   XawListReturnStruct *item=XawListShowCurrent(filelist);
   Widget popup=(Widget) client_data;
   int pid_status;
   bool sense;
   
   XtVaGetValues(playbutton,XtNsensitive,&sense,NULL);
   if (!sense) return;

   if (!playing) {
      playing=TRUE;
      getcwd(filename,MAX_NAMELENGTH);
      strcat(filename,"/");
      strcat(filename,item->string);
      XtVaSetValues(playbutton,XtNlabel, "Stop", NULL);
      pid= fork();
      if (!pid) {
	 XtFree((char *) dateien);
	 XtFree((char *) verzeich);
	 XtDestroyWidget(popup);
	 XtDestroyApplicationContext(XtWidgetToApplicationContext(popup));
	 play_file(filename,MD->mb);   
	 exit(0);
      } else if (pid<=0) {
	 playing=FALSE;
	 XtVaSetValues(wavestat,XtNlabel,"Error! Couldn't create bg process !",NULL);
	 XtVaSetValues(playbutton,XtNlabel, "Play", NULL);
	 
	 return;
      }
   } else {
      kill((pid_t)pid,SIGKILL);
      waitpid((pid_t)pid, &pid_status, 0);
      XtVaSetValues(playbutton,XtNlabel, "Play", NULL);
      pid=-1;
      playing=FALSE;
      return;
   }
   
   return;
}

static int select_it(char *fn,char *mask)
{
   int i=0,j=0,lfn,lm;
   Boolean found=FALSE;
   
   lfn=strlen(fn);lm=strlen(mask);
   if (lm>lfn) return(-1);
   if ((mask[0]=='*') /*&& (mask[1]=='.')*/) {
      while (i<lfn) {
	 if (fn[i]==mask[1]) {	
	    
	    /* don't look at asterisk '*' */
	    
	    for (j=1;j<lm;) {
	       if (fn[i]=='\0') break;
	       if (mask[j]=='\0') break;
	       if (fn[i]!=mask[j]) break;
	       i++;j++;
	    }
	    if ((j==lm) && (i==lfn)) {
	       found=TRUE;
	       break;
	    }
	 }
	 i++;
      }
   }
   else {
      while (i<lfn) {
	 for (j=0;j<lm;) {
	    if (fn[i]=='\0') break;
	    if (mask[j]=='\0') break;
	    if (mask[j]=='*') {
	       found=TRUE;
	       break;
	    }
	    if (fn[i]!=mask[j]) break;
	    i++;j++;
	 }
	 if (j==lm) {
	    found=TRUE;
	    break;
	 }
	 i++;
      }
   }
   if(found) return(0);
   return(-1);
} 


static int strqsortcmp(char **a, char **b)
{
   String	astr = *a;
   String	bstr = *b;
   String	aend = astr + strlen(astr) - 1;
   String	bend = bstr + strlen(bstr) - 1;
   
   if (strncmp(astr, "../", 3) == 0)
     return -1;
   if (strncmp(bstr, "../", 3) == 0)
     return  1;
   
   if (*aend == '/' && *bend != '/')
     return -1;
   if (*aend != '/' && *bend == '/')
     return  1;
   return strcmp(astr, bstr);
}

void clean_up (Widget popup)
{
   int pid_status;

   if (playing) {
      kill((pid_t)pid,SIGKILL);
      waitpid((pid_t)pid, &pid_status, 0);
      pid=-1;
      playing=FALSE;
      XtVaSetValues(playbutton,XtNlabel, "Play", NULL);
   }
   XtPopdown(popup);
   XFlush(XtDisplay(popup));
   return;
}

void cancel_action(Widget w, XtPointer client_data, XtPointer call_data)
{
   Widget popup=(Widget) client_data;
   
   clean_up(popup);
   return;
}

void ok_action(Widget w, XtPointer client_data, XtPointer call_data)
{
   struct stat buffer;
   static char filename[MAX_NAMELENGTH];
   static char dirname[MAX_NAMELENGTH];
   XawListReturnStruct *item=XawListShowCurrent(filelist);
   Widget popup=(Widget) client_data;

   getcwd(dirname,MAX_NAMELENGTH);
   strcat(dirname,"/");
   if (strcmp(file_buf,"")!=0) {
      strcpy(filename,dirname);
      if (strncmp(file_buf,"/",1)==0) 
      	strcpy(filename,file_buf);
      else    
      	strcat(filename,file_buf);
      
      sprintf((char*)MD->mg->fbuf,"%s\n\"%s\"\n",
	      app_resources.err_openfile,filename);
      if (lstat(filename,&buffer)==-1) {
	 if (browsetype==LOAD) {
	    warn_popup(popup,(char*)MD->mg->fbuf);
	    return;
	 } else {
	    if (!can_write(filename)) {
	       warn_popup(popup,(char*)MD->mg->fbuf);
	       return;
	    }
	    browse_return(filename,default_ftype,default_comp);
	    clean_up(popup);
	    return;
	 }
      }
      if (S_ISDIR(buffer.st_mode)) {

	 /* dir not changed */
	 if ((strcmp(dirname,filename)==0)&&(strcmp(test_buf,mask_buf)==0)) {
	    return; 		
	 }
	 set_dir(filename);
	 return;
      }		
      if (S_ISREG(buffer.st_mode)) {
	 if (strcmp(test_buf,mask_buf)!=0) {
	    set_dir(dirname);
	    return;
	 }

	 if ((browsetype!=LOAD)&&(!can_write(filename))) {
	    warn_popup(popup,(char*)MD->mg->fbuf);
	    return;
	 }
	 XtPopdown(popup);
	 XFlush(XtDisplay(popup));
	 if (browse_return(filename,default_ftype,default_comp)==-1) {
	    XtPopup(popup,XtGrabExclusive);
	    XFlush(XtDisplay(popup));
	    warn_popup(popup,(char*)MD->mg->fbuf);
	    return;
	 }
	 clean_up(popup);
	 return;
      }
   }
   if (strcmp(test_buf,mask_buf)!=0) {
      set_dir(dirname);
      return;
   }
   if (strcmp(item->string,"")==0) return;
   strcat(dirname,item->string);
   
   sprintf((char*)MD->mg->fbuf,"%s\n\"%s\"\n",
	   app_resources.err_openfile,dirname);
   if ((browsetype!=LOAD)&&(!can_write(dirname))) {
      warn_popup(popup,(char*)MD->mg->fbuf);
      return;
   }
   XtPopdown(popup);
   XFlush(XtDisplay(popup));
   if (browse_return(dirname,default_ftype,default_comp)==-1) {
      XtPopup(popup,XtGrabExclusive);
      XFlush(XtDisplay(popup));
      warn_popup(popup,(char*)MD->mg->fbuf);
      return;
   }	
   clean_up(popup);
   return;
}

void dir_action(Widget w, XtPointer client_data, XtPointer call_data)
{
   char dirname[MAX_NAMELENGTH];
   XawListReturnStruct *item=(XawListReturnStruct*) call_data;
   getcwd(dirname,MAX_NAMELENGTH);
   strcat(dirname,"/");
   strcat(dirname,item->string);
   set_dir(dirname);
}

void display_permissions(char *filename) 
{
   struct stat buffer;
   char link_buf1[MAX_NAMELENGTH];
   char link_buf2[MAX_NAMELENGTH];
   char owner_buf[]="Not owner, not group member";
   char p_buf[]="-rwxrwxrwx";
   uid_t myuid;
   gid_t mygid;

   if ((lstat(filename,&buffer)==-1)) {
      warn_popup(popup,"Error! Couldn't stat file !");
      return;
   }
   myuid=getuid();
   mygid=getgid();
   if (S_ISLNK(buffer.st_mode)) {
      bool statdone=TRUE;
      int lcount;
      
      strcpy(link_buf1,filename);
      while(TRUE) {
	 lcount=readlink(link_buf1,link_buf2,MAX_NAMELENGTH);
	 if (lcount!=-1)  {
	    link_buf2[lcount]=0;
	    if (strcmp(link_buf1,link_buf2)==0) {
	       statdone=FALSE;
	       break;
	    }
	    strcpy(link_buf1,link_buf2);
	    if (lstat(link_buf2,&buffer)==-1)  {
	       statdone=FALSE;
	       break;
	    }
	    if (!S_ISLNK(buffer.st_mode)) {
	       statdone=TRUE;
	       break;
	    }
	 } else statdone=FALSE;
      }
   }
   if (buffer.st_uid != myuid) strcpy(owner_buf,"Not owner, ");
   else strcpy(owner_buf,"Owner, ");
   if (buffer.st_gid != mygid) strcat(owner_buf,"not group member");
   else strcat(owner_buf,"group member");
   strcpy(p_buf,"-");
   if (S_IRUSR & buffer.st_mode) strcat(p_buf,"r"); else strcat(p_buf,"-");
   if (S_IWUSR & buffer.st_mode) strcat(p_buf,"w"); else strcat(p_buf,"-");
   if (S_IXUSR & buffer.st_mode) strcat(p_buf,"x"); else strcat(p_buf,"-");
   if (S_IRGRP & buffer.st_mode) strcat(p_buf,"r"); else strcat(p_buf,"-");
   if (S_IWGRP & buffer.st_mode) strcat(p_buf,"w"); else strcat(p_buf,"-");
   if (S_IXGRP & buffer.st_mode) strcat(p_buf,"x"); else strcat(p_buf,"-");
   if (S_IROTH & buffer.st_mode) strcat(p_buf,"r"); else strcat(p_buf,"-");
   if (S_IWOTH & buffer.st_mode) strcat(p_buf,"w"); else strcat(p_buf,"-");
   if (S_IXOTH & buffer.st_mode) strcat(p_buf,"x"); else strcat(p_buf,"-");
   
   sprintf(link_buf1,"%s\n%s",owner_buf,p_buf);
   XtVaSetValues(wavestat,XtNlabel,link_buf1,NULL);
}

void hilight_file()
{
   XawListReturnStruct *item=XawListShowCurrent(filelist);
   char *filename;
   short af_type;
   char *info;
   
   /* unhilight a possible selected entry in dirlist */
   XawListUnhighlight(dirlist);

   if (browsetype==LOAD) {
      XtVaSetValues(playbutton,XtNsensitive,False,NULL);
      XtVaSetValues(okbutton,XtNsensitive,False,NULL);
   } else {
      XtVaSetValues(playbutton,XtNsensitive,True,NULL);
      XtVaSetValues(okbutton,XtNsensitive,True,NULL);
   }
   
   if (strlen(item->string)==0) {
      strcpy(file_buf,"");
      XtVaSetValues(text,XtNstring,file_buf,NULL);
      return;
   }
   
   strcpy(file_buf,item->string);
   XtVaSetValues(text,XtNstring,file_buf,NULL);

   filename=XtCalloc(MAX_NAMELENGTH,sizeof(char));
   getcwd(filename,MAX_NAMELENGTH);
   strcat(filename,"/");
   strcpy(filename,file_buf);
   
   info=af_info(filename,&af_type);
   switch (af_type) {
    case AF_ERROR: 
      if (info!=NULL) {
	 warn_popup(popup,info);
	 XtFree(info);
      }
      XtFree(filename);
      return;
    case AF_RAW: 
      display_permissions(filename);
      XtFree(info);
      XtFree(filename);
      XtVaSetValues(okbutton,XtNsensitive,True,NULL);
      return;
   }
   
   XtVaSetValues(wavestat,XtNlabel,info,NULL);
   if (af_type!=AF_NOTSUPPORTED) { 
      if (MD->mb->canplay) XtVaSetValues(playbutton,XtNsensitive,True,NULL);
      XtVaSetValues(okbutton,XtNsensitive,True,NULL);
   }
   XtFree(info);
   XtFree(filename);
   return;
}

void hilight_dir()
{
   XawListReturnStruct *item=XawListShowCurrent(dirlist);
   char filename[MAX_NAMELENGTH];

   if (browsetype==LOAD) XtVaSetValues(playbutton,XtNsensitive,FALSE,NULL);
   /* unhilight a possible selected entry in filelist */
   XawListUnhighlight(filelist);
   
   if (strlen(item->string)==0) {
      strcpy(file_buf,"");
      XtVaSetValues(text,XtNstring,file_buf,NULL);
      return;
   }

   getcwd(filename,MAX_NAMELENGTH);
   strcat(filename,"/");
   strcat(filename,item->string);
   display_permissions(filename);
   strcpy(file_buf,item->string);
   XtVaSetValues(text,XtNstring,file_buf,NULL);
}

void tab_expansion()
{
   char *name=NULL,*muster=NULL;
   bool slash_found=False;
   int i,j,k,length;
   
   length=strlen(file_buf);
   if ((last_expansion!=NULL) && (strlen(last_expansion))!=0) {
      if (strcmp(last_expansion,file_buf)==0) return;
      if ((length<=strlen(last_expansion))||
	  ((length>strlen(last_expansion))&&
	   (strncmp(last_expansion,file_buf,strlen(last_expansion))!=0))) {
	 if ((name=XtCalloc(length+1,sizeof(char)))==NULL) return;
	 strcpy(name,file_buf);
	 set_dir(".");
	 strcpy(file_buf,name);
	 XtVaSetValues(text,XtNstring,file_buf,NULL);
	 XawTextSetInsertionPoint(text,strlen(file_buf));
	 XtFree((char *) name);
      }
   }
   if (last_expansion!=NULL) XtFree((char *)last_expansion);
   if ((last_expansion=XtCalloc(length+1,sizeof(char)))==NULL) return;
   strcpy(last_expansion,file_buf);
   
   if (length==0) return;
   if ((name=XtCalloc(length+1,sizeof(char)))==NULL) return;

   for (i=length-1;i>0;i--) {
      if (file_buf[i]=='/') {
	 slash_found=True;
	 break;
      }
   }
   
   if ((muster=XtCalloc(length-i+1,sizeof(char)))==NULL) return;
   
   if (!slash_found) {
      if (file_buf[0]=='/') {
	 i=0;
	 slash_found=True;
      } else {
	 if (chdir(file_buf)!=-1) {
	    set_dir(".");
	    XtFree(name);
	    XtFree(muster);
	    return;
	 } else {
	    /* =^ strcpy(muster,file_buf) */
	    i=-1;
	 }
      }
   }
   
   for (j=0,k=i+1;k<length;k++) {
      muster[j++]=file_buf[k];
   }
   muster[j]='\0';
  
   if (slash_found) {
      strncpy(name,file_buf,i+1);
      if (chdir(name)==-1) {
	 XtFree(name);
	 XtFree(muster);
	 return;
      }
      set_dir(".");
      strcpy(file_buf,muster);
      XtVaSetValues(text,XtNstring,file_buf,NULL);
      XawTextSetInsertionPoint(text,strlen(file_buf));
   }

   if (strlen(muster)!=0) {
      int dir_length,file_length,m_length,no_files,no_dirs;
      int dir_index=-1,file_index=-1;
      String *newdateien=NULL,*newverzeich=NULL;
      
      m_length=strlen(muster);

      XtVaGetValues(dirlist,XtNnumberStrings,&dir_length,NULL);
      XtVaGetValues(filelist,XtNnumberStrings,&file_length,NULL);
      
      for (i=0,j=0;i<dir_length;i++) {
	 if (verzeich[i][0]==muster[0]) {
	    if (strncmp(verzeich[i],muster,m_length)==0) {
	       j++;
	       if (j==1) dir_index=i;
	    }
	 }
      }
      no_dirs=j;

      for (i=0,j=0;i<file_length;i++) {
	 if (dateien[i][0]==muster[0]) {
	    if (strncmp(dateien[i],muster,m_length)==0) {
	       j++;
	       if (j==1) file_index=i;
	    }
	 }
      }
      no_files=j;

      if ((dir_index!=-1)&&(no_dirs==1)&&(file_index==-1)) {
	 XawListHighlight(dirlist,dir_index);
	 XtFree(name);
	 XtFree(muster);
	 hilight_dir();
	 XawTextSetInsertionPoint(text,strlen(file_buf));
	 return;
      }

      if ((file_index!=-1)&&(no_files==1)&&(dir_index==-1)) {
	 XawListHighlight(filelist,file_index);
	 XtFree(name);
	 XtFree(muster);
	 hilight_file();
	 XawTextSetInsertionPoint(text,strlen(file_buf));
	 return;
      }

      if (no_dirs!=0) {
	 newverzeich = (String *)XtCalloc(sizeof(String *),no_dirs);
	 for (i=0,k=0;i<dir_length;i++) {
	    if (verzeich[i][0]==muster[0]) {
	       if (strncmp(verzeich[i],muster,m_length)==0) {
		  newverzeich[k++]=XtNewString(verzeich[i]);
	       }
	    }
	 }
	 free_stringlist(dirlist,verzeich);
	 verzeich=newverzeich;
	 no_dirs=k;
      } else {
	 no_dirs=dir_length;
      }
      
      if (no_files!=0) {
	 newdateien = (String *)XtCalloc(sizeof(String *),no_files);
	 for (i=0,k=0;i<file_length;i++) {
	    if (dateien[i][0]==muster[0]) {
	       if (strncmp(dateien[i],muster,m_length)==0) {
		  newdateien[k++]=XtNewString(dateien[i]);
	       }
	    }
	 }
	 free_stringlist(filelist,dateien);
	 dateien=newdateien;
	 no_files=k;
      } else {
	 no_files=file_length;
      }
      
      if (no_dirs!=dir_length) 
	XawListChange(dirlist,verzeich,no_dirs,0,TRUE);
      if (no_files!=file_length)
	XawListChange(filelist,dateien,no_files,0,TRUE);
    
      if ((no_dirs!=dir_length)||(no_files!=file_length)) {
	 char lbuf[50];
	 sprintf(lbuf,"Files: %i        Directories: %i",no_files,no_dirs);
	 XtVaSetValues(status,XtNlabel,lbuf,NULL);
	 XtVaSetValues(wavestat,XtNlabel,"No file selected !",NULL);
      }
   }

   XtFree(name);
   XtFree(muster);
}

static void mask_menu_cb(Widget w, XtPointer client_data, XtPointer call_data)
{
   char *select=(char *) client_data;
   
   if (strcmp(mask_buf,select)==0) return;
   strcpy(file_buf,"");
   XtVaSetValues(text,XtNstring,file_buf,NULL);
   strcpy(mask_buf,select);
   XtVaSetValues(mtext,XtNstring,mask_buf,NULL);
   ok_action(w,(XtPointer) popup,NULL);
}

void filetype_cb(Widget w, XtPointer client_data, XtPointer call_data)
{
   filetype_dialog(ft_return);
}

void ft_return(int type,int comp)
{
   default_ftype=type;
   default_comp=comp;
   sprintf(ft_label,"%s (%s)",ft2string(default_ftype),
	   comp2string(default_comp));
   XtVaSetValues(playbutton,XtNlabel,ft_label,NULL);
}

void uh_action(Widget w, XtPointer *junk, XKeyPressedEvent *event, 
	       Boolean *flg)
{
/* unhilight a possible selected entry in filelist/dirlist */
/*   printf("%i\n",event->keycode);*/
   if ((event->keycode>=10)&&(event->keycode<=19)) {
      XawListUnhighlight(filelist);
      XawListUnhighlight(dirlist);
   }
   if ((event->keycode>=24)&&(event->keycode<=35)) {
      XawListUnhighlight(filelist);
      XawListUnhighlight(dirlist);
   }
   if ((event->keycode>=38)&&(event->keycode<=61)) {
      XawListUnhighlight(filelist);
      XawListUnhighlight(dirlist);
   }
   if ((event->keycode>=150)&&(event->keycode<=157)) {
      XawListUnhighlight(filelist);
      XawListUnhighlight(dirlist);
   }
   switch (event->keycode) {
    case 22:    
      XawListUnhighlight(filelist);
      XawListUnhighlight(dirlist);
      break;
    case 63:
      XawListUnhighlight(filelist);
      XawListUnhighlight(dirlist);
      break;
    case 82:
      XawListUnhighlight(filelist);
      XawListUnhighlight(dirlist);
      break;
    case 86:
      XawListUnhighlight(filelist);
      XawListUnhighlight(dirlist);
      break;
    case 108: 
      XawListUnhighlight(filelist);
      XawListUnhighlight(dirlist);
      break;
    case 112: 
      XawListUnhighlight(filelist);
      XawListUnhighlight(dirlist);
      break;
   }
}


void set_dir_label(char *directory)
{
   XFontStruct *font;
   Dimension width;
   int dim;

   XtVaGetValues(dirlabel,XtNfont,&font,XtNwidth,&width,NULL);
   dim=XTextWidth(font,directory,strlen(directory));
   if (dim>width) {
      char *fn=XtMalloc(strlen(directory)),*fp;
      int i=strlen(directory);
      int pdim;
      strcpy(fn,"...");
      pdim=XTextWidth(font,fn,strlen(fn));
      while (TRUE) {
         fp=directory+i;
         dim=XTextWidth(font,fp,strlen(fp));
         if (dim+pdim>width) break;
         i--;
      }
      fp+=2;		/* now the directory name fits in the label */
      strcat(fn,fp);
      XtVaSetValues(dirlabel,XtNlabel,fn,NULL);
      XtFree(fn); 
   } else 
      XtVaSetValues(dirlabel,XtNlabel,directory,NULL);
}

void free_stringlist(Widget list,String *strlist)
{
   int length,i;

   if (strlist==NULL) return;
   XtVaGetValues(list,XtNnumberStrings,&length,NULL);

   for (i=0;i<length;i++) XtFree((char *) strlist[i]);
   XtFree((char *) strlist);
}

void set_dir(char *directory)
{
   DIR *ordner;
   struct stat buffer;
   struct dirent *eintrag;
   int no_dirs=0,no_files=0,n=0,i=0,lcount;
   char filename[MAX_NAMELENGTH],*fp;
   char link_buf1[MAX_NAMELENGTH];
   char link_buf2[MAX_NAMELENGTH];
   int pid_status;

   /* kill the bg play process, cause we're making the 		*/
   /* "play / stop" button insensitive				*/

   if (playing) {
      kill((pid_t)pid,SIGKILL);
      waitpid((pid_t)pid, &pid_status, 0);
      XtVaSetValues(playbutton,XtNlabel, "Play", NULL);
      pid=-1;
      playing=FALSE;
   }

   if (chdir(directory)==-1) {
      warn_popup(popup,"Error! Couldn't change directory !");
      return;
   }

   if ((ordner=opendir(directory))==NULL) {
      warn_popup(popup,"Error! Couldn't open directory !");
      return;
   }

   /* set test_buf for test of filemask changed */
   
   strcpy(test_buf,mask_buf);

   /* set playbutton insensitive */

   if (browsetype==LOAD)
     XtVaSetValues(playbutton,XtNsensitive,FALSE,NULL);
   
   getcwd(filename,MAX_NAMELENGTH);
   if (strcmp(filename,"/")!=0) strcat(filename,"/");
   set_dir_label(filename);
   strcpy(file_buf,"");
   XtVaSetValues(text,XtNstring,file_buf,NULL);

   fp=filename + strlen(filename);
   while ((eintrag=readdir(ordner))!=NULL) {
      strcpy(fp,eintrag->d_name);
      lstat(filename,&buffer);
      if (S_ISDIR(buffer.st_mode)) no_dirs++;
      if (S_ISREG(buffer.st_mode)) no_files++;
      if (S_ISLNK(buffer.st_mode)) {
	 bool statdone=TRUE;
	 strcpy(link_buf1,filename);
	 while(statdone) {
	    lcount=readlink(link_buf1,link_buf2,MAX_NAMELENGTH);
	    if (lcount!=-1)  {
	       link_buf2[lcount]=0;
	       if (strcmp(link_buf1,link_buf2)==0) {
		  statdone=FALSE;
		  break;
	       }
	       strcpy(link_buf1,link_buf2);
	       if (lstat(link_buf2,&buffer)==-1)  {
		  statdone=FALSE;
		  break;
	       }
	       if (!S_ISLNK(buffer.st_mode)) {
		  statdone=TRUE;
		  break;
	       }
	    } else statdone=FALSE;
	 }
	 if (S_ISDIR(buffer.st_mode) && statdone) no_dirs++;
	 if (S_ISREG(buffer.st_mode) && statdone) no_files++;
      }
   }
   no_dirs--;					/* delete file "." */
   rewinddir(ordner);

   free_stringlist(dirlist,verzeich);
   free_stringlist(filelist,dateien);

   getcwd(filename,MAX_NAMELENGTH);
   if (strcmp(filename,"/")!=0) strcat(filename,"/");
   else no_dirs--;                             /* no ".." in root directory */
   
   if (no_files==0) {
      /* NULL terminated list if list is empty */
      dateien = (String *)XtCalloc(sizeof(String *), 1);
      dateien[0]=NULL;
   } else {
     dateien = (String *)XtCalloc(sizeof(String *), no_files);
   }
   verzeich = (String *)XtCalloc(sizeof(String *), no_dirs);
   n=0;i=0;
   fp=filename + strlen(filename);

   if ((strlen(mask_buf)==0)||(strcmp(mask_buf,"*")==0)||
       (strcmp(mask_buf,"*.*")==0)) {
      while ((eintrag=readdir(ordner))!=NULL) {
	 char *fn=eintrag->d_name;
	 
	 if (fn[0]=='.') {
	    /* don't show dir "." */
	    if (fn[1]=='\0') continue;
	     /* don't show ".." in root directory */
	    if ((fn[1]=='.') && (fn[2]=='\0')) 
	      if (strcmp(filename,"/")==0) continue;
	 }
	     
	 strcpy(fp,fn);
	 lstat(filename,&buffer);
	 if (S_ISREG(buffer.st_mode)) dateien[n++]=XtNewString(fn);
	 
	 if (S_ISDIR(buffer.st_mode)) { 
	    strcat(fn,"/");
	    verzeich[i++]=XtNewString(fn);
	 }	
	 if (S_ISLNK(buffer.st_mode)) {
	    bool statdone=TRUE;
	    strcpy(link_buf1,filename);
	    while(statdone) {
	       lcount=readlink(link_buf1,link_buf2,MAX_NAMELENGTH);
	       if (lcount!=-1)  {
		  link_buf2[lcount]=0;
		  if (strcmp(link_buf1,link_buf2)==0) {
		     statdone=FALSE;
		     break;
		  }
		  strcpy(link_buf1,link_buf2);
		  if (lstat(link_buf2,&buffer)==-1)  {
		     statdone=FALSE;
		     break;
		  }
		  if (!S_ISLNK(buffer.st_mode)) {
		     statdone=TRUE;
		     break;
		  }
	       } else statdone=FALSE;
	    }
	    if (S_ISREG(buffer.st_mode) && statdone) dateien[n++]=XtNewString(fn);
	    
	    if (S_ISDIR(buffer.st_mode) && statdone) { 
	       strcat(fn,"/");
	       verzeich[i++]=XtNewString(fn);
	    }
	 }
      }
   } else {
      no_files=0;
      while ((eintrag=readdir(ordner))!=NULL) {
	 char *fn=eintrag->d_name;
	 /* don't show dir "." */
	 if (fn[0]=='.') if (fn[1] == '\0') continue;
	 strcpy(fp,fn);
	 lstat(filename,&buffer);
	 if (S_ISREG(buffer.st_mode)) { 
	    if (select_it(fn,mask_buf)==0) {
	       dateien[n++]=XtNewString(fn);
	       no_files++;
	    }
	 }
	 if (S_ISDIR(buffer.st_mode)) { 
	    strcat(fn,"/");		
	    verzeich[i++]=XtNewString(fn);
	 }	
	 if (S_ISLNK(buffer.st_mode)) {
	    bool statdone=TRUE;
	    strcpy(link_buf1,filename);
	    while(TRUE) {
	       lcount=readlink(link_buf1,link_buf2,MAX_NAMELENGTH);
	       if (lcount!=-1) {
		  link_buf2[lcount]=0;
		  if (strcmp(link_buf1,link_buf2)==0) {
		     statdone=FALSE;
		     break;
		  }
		  strcpy(link_buf1,link_buf2);
		  if (lstat(link_buf2,&buffer)==-1)  {
		     statdone=FALSE;
		     break;
		  }
		  if (!S_ISLNK(buffer.st_mode)) {
		     statdone=TRUE;
		     break;
		  }
	       } else statdone=FALSE;
	    }
	    if (S_ISREG(buffer.st_mode) && statdone) {
	       if (select_it(fn,mask_buf)==0) {
		  dateien[n++]=XtNewString(fn);
		  no_files++;
	       }
	    }
	    if (S_ISDIR(buffer.st_mode) && statdone) { 
	       strcat(fn,"/");		
	       verzeich[i++]=XtNewString(fn);
	    }	
	 }
      }
   }
   sprintf(filename,"Files: %i        Directories: %i",no_files,no_dirs);
   XtVaSetValues(status,XtNlabel,filename,NULL);
   XtVaSetValues(wavestat,XtNlabel,"No file selected !",NULL);

    
#ifdef sun
   qsort(dateien, n, sizeof(String),
	 (int (*)(void *,void *))strqsortcmp);
   qsort(verzeich, i, sizeof(String),
	 (int (*)(void *,void *))strqsortcmp);
#else    
   qsort(dateien, n, sizeof(String),
	 (int (*)(_Xconst void *,_Xconst void *))strqsortcmp);
   qsort(verzeich, i, sizeof(String),
	 (int (*)(_Xconst void *,_Xconst void *))strqsortcmp);
#endif
   XawListChange(dirlist,verzeich,i,0,TRUE);
   XawListChange(filelist,dateien,n,0,TRUE);
   closedir(ordner);
}
   
void file_dialog(int (*browse_callback)(char *fname,int type,int comp),
		 String f_mask,int type) 
{
   void abort_play();
   
   /* set the proc to be executed if the ok button is pressed */
   
   browse_return=browse_callback;
   
   default_ftype=app_resources.default_ftype;
   default_comp=app_resources.default_comp;
   
   /* set the type of the browser (create new file, or load existing) */
   
   if ((browsetype==LOAD)&&((type==SAVE)||(type==CREATE))) {
      XtRemoveCallback (playbutton, XtNcallback, play_it, (XtPointer) popup); 
      XtAddCallback (playbutton, XtNcallback, filetype_cb, (XtPointer) popup); 
   }

   if (((browsetype==SAVE)||(browsetype==CREATE))&&(type==LOAD)) {
      XtRemoveCallback (playbutton, XtNcallback, filetype_cb, 
			(XtPointer) popup); 
      XtAddCallback (playbutton, XtNcallback, play_it, (XtPointer) popup);    
   }
   
   browsetype=type;

   if (browsetype==LOAD) {
      XtVaSetValues(playbutton,XtNsensitive,False,NULL);
      XtVaSetValues(playbutton,XtNlabel,"Play",NULL);
   } else {
      XtVaSetValues(playbutton,XtNsensitive,True,NULL);
      sprintf(ft_label,"%s (%s)",ft2string(default_ftype),
	      comp2string(default_comp));
      XtVaSetValues(playbutton,XtNlabel,ft_label,NULL);
   }
   
   /* setting the signal handler */
   
   sig->sa_handler=abort_play;
   sig->sa_flags=SA_RESTART;
   sigaction(SIGUSR1,sig,NULL);    
   
   strcpy(mask_buf,f_mask);
   set_dir(".");

   XtVaSetValues(mtext,XtNstring,mask_buf,NULL);
   popup_centered(popup);
}

void create_file_dialog (Widget w)
{
   Widget form1,form2,viewport,button,mshell;
   Dimension width,height;
   Pixmap pullBitmap;
   int i;
   static XtAccelerators accel;
   static XtTranslations trans_file,trans_dir;
   static XtActionsRec fb_actions[] = {
	{"hilight_file",(XtActionProc)hilight_file},
	{"hilight_dir",(XtActionProc) hilight_dir },
	{"tab_expansion",(XtActionProc)tab_expansion}
   };
   char pull_char[20];

   XtAppAddActions(XtWidgetToApplicationContext(w), 
		   fb_actions, XtNumber(fb_actions));


   popup=XtVaCreatePopupShell("fb_shell",transientShellWidgetClass,w,NULL);

   form  = MW ("fb_main_form",formWidgetClass,popup,NULL);

   form1 = MW ("fb_first_form",formWidgetClass,form,
	       XtNborderWidth,0,
	       NULL);

   button = MW ("fb_file_label",labelWidgetClass,form1,
		XtNborderWidth,0,
		XtNbottom, XtChainTop,XtNtop, XtChainTop,
    		XtNleft,XawRubber,XtNright,XawRubber,		
		XtNresize,FALSE,
		NULL);

   dirlabel = MW ("fb_dir_label",labelWidgetClass,form1,
		XtNfromVert,button,
		XtNborderWidth,0,
		XtNresize,FALSE,
		NULL);

   accel = XtParseAcceleratorTable("#override\n\
		<Key>Return: set() notify() unset()\n");

   text = MW ("fb_fname_text",asciiTextWidgetClass, form1,
	      XtNfromVert,dirlabel,XtNallowVert,TRUE,		
	      XtNwrap, XawtextWrapNever,
	      XtNeditType, XawtextEdit,
	      XtNresize, XawtextResizeWidth,
	      XtNuseStringInPlace, True,
	      XtNlength,MAX_NAMELENGTH,XtNstring,&file_buf,
	      XtNbottom, XtChainTop,XtNtop, XtChainTop,
	      XtNresize,FALSE,
	      NULL);

   XtAddEventHandler(text, KeyPressMask, False,
		     (XtEventHandler)uh_action, (XtPointer) NULL);
   
   status = MW ("fb_count_label",labelWidgetClass,form1,
		XtNfromVert,text,
		XtNborderWidth,0,
		XtNbottom, XtChainTop,XtNtop, XtChainTop,
    		XtNleft,XawRubber,XtNright,XawRubber,		
		XtNresize,FALSE,
		NULL);

   viewport = MW ("fb_file_vp",viewportWidgetClass,form1,
		  XtNfromVert,status,XtNallowVert,TRUE,		
		  XtNforceBars,TRUE,
		  XtNtop,XawRubber,XtNbottom,XawRubber,
		  XtNleft,XawRubber,XtNright,XawRubber,NULL);

   trans_file=XtParseTranslationTable("#override\n\
                  <Btn1Down>,<Btn1Up>: Set() hilight_file()\n\
                   <Btn1Up>(2) : Set() Notify()\n");

   filelist = MW ("fb_file_list",listWidgetClass,viewport,
		  XtNdefaultColumns,1,
		  XtNtranslations, trans_file,
		  NULL);

   if (app_resources.doubleplay && MD->mb->canplay)
     XtAddCallback (filelist, XtNcallback, play_it,(XtPointer) popup);
   else
     XtAddCallback (filelist, XtNcallback, ok_action,(XtPointer) popup);

   viewport = MW ("fb_dir_vp",viewportWidgetClass,form1,
		  XtNfromVert,status,XtNfromHoriz,viewport,
		  XtNallowVert,TRUE,XtNallowHoriz,FALSE,XtNforceBars,TRUE,
		  XtNtop,XawRubber,XtNbottom,XawRubber,
		  XtNleft,XawRubber,XtNright,XawRubber,NULL);

   trans_dir=XtParseTranslationTable("#override\n\
   				  <Btn1Down>,<Btn1Up>: Set() hilight_dir() \n\
   				  <Btn1Up>(2) : Set() Notify()\n");

   dirlist = MW ("fb_dir_list",listWidgetClass,viewport,
		 XtNdefaultColumns,1,
		 XtNtranslations, trans_dir,
		 NULL);
   		
   XtAddCallback (dirlist, XtNcallback, dir_action,0);    

   form2 = MW ("fb_second_form",formWidgetClass,form,
	       XtNfromVert,form1,
	       XtNborderWidth,0,
	       NULL);

   button = MW ("fb_mask_label",labelWidgetClass,form2,
		XtNborderWidth,0,
		XtNbottom, XtChainTop,XtNtop, XtChainTop,
    		XtNleft,XawChainLeft,XtNright,XawChainLeft,		
		XtNresize,FALSE,
		NULL);
   
   strcpy(mask_buf,"*");
   strcpy(test_buf,mask_buf);
   mtext = MW ("fb_mask_text",asciiTextWidgetClass, form2,
	       XtNfromHoriz,button,		
	       XtNeditType, XawtextEdit,
	       XtNwrap, XawtextWrapNever,
	       XtNresize, XawtextResizeWidth,
	       XtNuseStringInPlace, True,
	       XtNlength,MAX_NAMELENGTH,XtNstring,&mask_buf,
	       XtNbottom, XtChainTop,XtNtop, XtChainTop,
	       XtNleft,XawChainLeft,XtNright,XawChainLeft,
	       NULL);

   XtAddEventHandler(mtext, KeyPressMask, False,
		     (XtEventHandler)uh_action, (XtPointer) NULL);

   pullBitmap = XCreateBitmapFromData(XtDisplay(w),
				DefaultRootWindow(XtDisplay(w)),
				(char *)pull_bits,
				pull_width, pull_height);

   button = MW ("fb_pull_btn",menuButtonWidgetClass,form2,
		XtNfromHoriz,mtext,
		XtNbitmap,pullBitmap,
		XtNmenuName,"fb_pull_mshell",
		NULL);

   mshell = XtCreatePopupShell ("fb_pull_mshell",simpleMenuWidgetClass,
				button,NULL,0);
   
   for (i=0;i<XtNumber(mask_strings);i++) {
      sprintf(pull_char,"fb_pull%i_mentry",i);
      button = MW (pull_char,smeBSBObjectClass,mshell,
		   XtNlabel,mask_strings[i],
		   NULL);
      XtAddCallback (button, XtNcallback, mask_menu_cb,
		     (XtPointer)mask_strings[i]);
   }
		   
   wavestat = MW ("fb_status_label",labelWidgetClass,form2,
		  XtNfromVert,mtext,
		  XtNborderWidth,0,
		  XtNbottom, XtChainTop,XtNtop, XtChainTop,
		  XtNleft,XawRubber,XtNright,XawRubber,		
		  XtNresize,FALSE,
		  NULL);
   
   okbutton = MW ("fb_ok_btn",commandWidgetClass,form2,
		  XtNfromVert,wavestat,
		  XtNaccelerators, accel,		
		  XtNtop,XawChainBottom,XtNbottom,XawChainBottom,
		  XtNleft,XawChainLeft,XtNright,XawChainLeft,NULL);
   
   XtSetKeyboardFocus(okbutton ,text);

   XtAddCallback(okbutton,XtNcallback,ok_action,(XtPointer) popup);    

   XtInstallAccelerators(text, okbutton);
   XtInstallAccelerators(mtext, okbutton);

   playbutton = MW ("fb_play_btn",commandWidgetClass,form2,
		    XtNfromVert,wavestat,XtNfromHoriz,okbutton,		
		    XtNtop,XawChainBottom,XtNbottom,XawChainBottom,
		    XtNleft,XawRubber,XtNleft,XawRubber,NULL);

   XtAddCallback (playbutton, XtNcallback, play_it, (XtPointer) popup);    

   button = MW ("fb_ca_btn",commandWidgetClass,form2,
		XtNfromVert,wavestat,XtNfromHoriz,playbutton,		
    		XtNtop,XawChainBottom,XtNbottom,XawChainBottom,
    		XtNleft,XawChainRight,XtNright,XawChainRight,NULL);

   XtAddCallback (button, XtNcallback, cancel_action, (XtPointer) popup);    

   XtSetKeyboardFocus(form1 ,text);
   XtSetKeyboardFocus(form2 ,mtext);
   
   dateien=NULL;
   verzeich=NULL;
   set_dir(".");   

   
   XtRealizeWidget(popup);
   XtVaGetValues(form,XtNwidth,&width,XtNheight,&height, NULL);
   XtVaSetValues(popup,XtNminWidth,width,XtNmaxWidth,width,\
                 XtNminHeight,height,XtNmaxHeight,height,NULL);

   sig=XtNew(struct sigaction);
}
      
void abort_play()
{
   int pid_status;
   waitpid((pid_t)pid, &pid_status, 0);
   XtVaSetValues(playbutton,XtNlabel, "Play", NULL);
   XFlush(XtDisplay(MD->mw->graph));
   playing=FALSE;    /* if we got an signal from bg playing process */
}
        
        
